/* vi:set ts=8 sts=4 sw=4:
 *
 * MzScheme interface by Sergey Khorev <khorev@softlab.ru>
 * Original work by Brent Fulgham <bfulgham@debian.org>
 * (Based on lots of help from Matthew Flatt)
 *
 * This consists of six parts:
 * 1. MzScheme interpreter main program
 * 2. Routines that handle the external interface between MzScheme and
 *    Vim.
 * 3. MzScheme input/output handlers: writes output via [e]msg().
 * 4. Implementation of the Vim Features for MzScheme
 * 5. Vim Window-related Manipulation Functions.
 * 6. Vim Buffer-related Manipulation Functions
 *
 * NOTES
 * 1. Memory, allocated with scheme_malloc*, need not to be freed explicitly,
 *    garbage collector will do it self
 * 2. Requires at least NORMAL features. I can't imagine why one may want
 *    to build with SMALL or TINY features but with MzScheme interface.
 * 3. I don't use K&R-style functions. Anyway, MzScheme headers are ANSI.
 */

#include "vim.h"
#include "if_mzsch.h"

/* Base data structures */
#define SCHEME_VIMBUFFERP(obj)  SAME_TYPE(SCHEME_TYPE(obj), mz_buffer_type)
#define SCHEME_VIMWINDOWP(obj)  SAME_TYPE(SCHEME_TYPE(obj), mz_window_type)

typedef struct
{
    Scheme_Type	    tag;
    Scheme_Env	    *env;
    buf_T	    *buf;
} vim_mz_buffer;

#define INVALID_BUFFER_VALUE ((buf_T *)(-1))

typedef struct
{
    Scheme_Type	    tag;
    struct window   *win;
} vim_mz_window;

#define INVALID_WINDOW_VALUE ((win_T *)(-1))

/*
 * Prims that form MzScheme Vim interface
 */
typedef struct
{
    Scheme_Closed_Prim	*prim;
    char	*name;
    int		mina;	/* arity information */
    int		maxa;
} Vim_Prim;

typedef struct
{
    char	    *name;
    Scheme_Object   *port;
} Port_Info;

/* info for closed prim */
/*
 * data have different means:
 * for do_eval it is char*
 * for do_apply is Apply_Onfo*
 * for do_load is Port_Info*
 */
typedef struct
{
    void	*data;
    Scheme_Env	*env;
} Cmd_Info;

/* info for do_apply */
typedef struct
{
    Scheme_Object   *proc;
    int		    argc;
    Scheme_Object   **argv;
} Apply_Info;

/*
 *========================================================================
 *  Vim-Control Commands
 *========================================================================
 */
/*
 *========================================================================
 *  Utility functions for the vim/mzscheme interface
 *========================================================================
 */
/*  Buffer-related commands */
static Scheme_Object *buffer_new(buf_T *buf);
static Scheme_Object *get_buffer_by_name(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_by_num(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_count(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_line(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_line_list(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_name(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_num(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_size(void *, int, Scheme_Object **);
static Scheme_Object *get_curr_buffer(void *, int, Scheme_Object **);
static Scheme_Object *get_next_buffer(void *, int, Scheme_Object **);
static Scheme_Object *get_prev_buffer(void *, int, Scheme_Object **);
static Scheme_Object *mzscheme_open_buffer(void *, int, Scheme_Object **);
static Scheme_Object *set_buffer_line(void *, int, Scheme_Object **);
static Scheme_Object *set_buffer_line_list(void *, int, Scheme_Object **);
static Scheme_Object *insert_buffer_line_list(void *, int, Scheme_Object **);
static Scheme_Object *get_range_start(void *, int, Scheme_Object **);
static Scheme_Object *get_range_end(void *, int, Scheme_Object **);
static Scheme_Object *get_buffer_namespace(void *, int, Scheme_Object **);
static vim_mz_buffer *get_vim_curr_buffer(void);

/*  Window-related commands */
static Scheme_Object *window_new(win_T *win);
static Scheme_Object *get_curr_win(void *, int, Scheme_Object **);
static Scheme_Object *get_window_count(void *, int, Scheme_Object **);
static Scheme_Object *get_window_by_num(void *, int, Scheme_Object **);
static Scheme_Object *get_window_num(void *, int, Scheme_Object **);
static Scheme_Object *get_window_buffer(void *, int, Scheme_Object **);
static Scheme_Object *get_window_height(void *, int, Scheme_Object **);
static Scheme_Object *set_window_height(void *, int, Scheme_Object **);
#ifdef FEAT_VERTSPLIT
static Scheme_Object *get_window_width(void *, int, Scheme_Object **);
static Scheme_Object *set_window_width(void *, int, Scheme_Object **);
#endif
static Scheme_Object *get_cursor(void *, int, Scheme_Object **);
static Scheme_Object *set_cursor(void *, int, Scheme_Object **);
static Scheme_Object *get_window_list(void *, int, Scheme_Object **);
static vim_mz_window *get_vim_curr_window(void);

/*  Vim-related commands */
static Scheme_Object *mzscheme_beep(void *, int, Scheme_Object **);
static Scheme_Object *get_option(void *, int, Scheme_Object **);
static Scheme_Object *set_option(void *, int, Scheme_Object **);
static Scheme_Object *vim_command(void *, int, Scheme_Object **);
static Scheme_Object *vim_eval(void *, int, Scheme_Object **);
static Scheme_Object *vim_bufferp(void *data, int, Scheme_Object **);
static Scheme_Object *vim_windowp(void *data, int, Scheme_Object **);
static Scheme_Object *vim_buffer_validp(void *data, int, Scheme_Object **);
static Scheme_Object *vim_window_validp(void *data, int, Scheme_Object **);

/*
 *========================================================================
 *  Internal Function Prototypes
 *========================================================================
 */
static int vim_error_check(void);
static int do_mzscheme_command(exarg_T *, void *, Scheme_Closed_Prim *what);
static void startup_mzscheme(void);
static char *string_to_line(Scheme_Object *obj);
static int mzscheme_io_init(void);
static void mzscheme_interface_init(vim_mz_buffer *self);
static void do_output(char *mesg, long len);
static void do_printf(char *format, ...);
static void do_flush(void);
static Scheme_Object *_apply_thunk_catch_exceptions(
	Scheme_Object *, Scheme_Object **);
static Scheme_Object *extract_exn_message(Scheme_Object *v);
static Scheme_Object *do_eval(void *, int noargc, Scheme_Object **noargv);
static Scheme_Object *do_load(void *, int noargc, Scheme_Object **noargv);
static Scheme_Object *do_apply(void *, int noargc, Scheme_Object **noargv);
static void register_vim_exn(Scheme_Env *env);
static vim_mz_buffer *get_buffer_arg(const char *fname, int argnum,
	int argc, Scheme_Object **argv);
static vim_mz_window *get_window_arg(const char *fname, int argnum,
	int argc, Scheme_Object **argv);
static void add_vim_exn(Scheme_Env *env);
static int line_in_range(linenr_T, buf_T *);
static void check_line_range(linenr_T, buf_T *);
static void mz_fix_cursor(int lo, int hi, int extra);

static int eval_in_namespace(void *, Scheme_Closed_Prim *, Scheme_Env *,
		Scheme_Object **ret);
static void make_modules(Scheme_Env *);

/*
 *========================================================================
 *  1. MzScheme interpreter startup
 *========================================================================
 */

static Scheme_Type mz_buffer_type;
static Scheme_Type mz_window_type;

static int initialized = 0;

/* global environment */
static Scheme_Env    *environment = NULL;
/* output/error handlers */
static Scheme_Object *curout = NULL;
static Scheme_Object *curerr = NULL;
/* vim:exn exception */
static Scheme_Object *exn_catching_apply = NULL;
static Scheme_Object *exn_p = NULL;
static Scheme_Object *exn_message = NULL;
static Scheme_Object *vim_exn = NULL; /* Vim Error exception */
 /* values for exn:vim - constructor, predicate, accessors etc */
static Scheme_Object *vim_exn_names = NULL;
static Scheme_Object *vim_exn_values = NULL;

static long range_start;
static long range_end;

/* MzScheme threads scheduling stuff */
static int mz_threads_allow = 0;

#if defined(FEAT_GUI_W32)
static void CALLBACK timer_proc(HWND, UINT, UINT, DWORD);
static UINT timer_id = 0;
#elif defined(FEAT_GUI_GTK)
static gint timer_proc(gpointer);
static guint timer_id = 0;
#elif defined(FEAT_GUI_MOTIF) || defined(FEAT_GUI_ATHENA)
static void timer_proc(XtPointer, XtIntervalId *);
static XtIntervalId timer_id = (XtIntervalId)0;
#elif defined(FEAT_GUI_MAC)
pascal void timer_proc(EventLoopTimerRef, void *);
static EventLoopTimerRef timer_id = NULL;
static EventLoopTimerUPP timerUPP;
#elif defined(FEAT_GUI_KDE)
static int timer_id = 0;
#endif

#ifndef FEAT_GUI_W32 /* Win32 console and Unix */
    void
mzvim_check_threads(void)
{
    /* Last time MzScheme threads were scheduled */
    static time_t mz_last_time = 0;

    if (mz_threads_allow && p_mzq > 0)
    {
	time_t now = time(NULL);

	if ((now - mz_last_time) * 1000 > p_mzq)
	{
	    mz_last_time = now;
	    scheme_check_threads();
	}
    }
}
#endif

#ifdef MZSCHEME_GUI_THREADS
static void setup_timer(void);
static void remove_timer(void);

/* timers are presented in GUI only */
# if defined(FEAT_GUI_W32)
    static void CALLBACK
timer_proc(HWND hwnd, UINT uMsg, UINT idEvent, DWORD dwTime)
# elif defined(FEAT_GUI_GTK)
/*ARGSUSED*/
    static gint
timer_proc(gpointer data)
# elif defined(FEAT_GUI_MOTIF) || defined(FEAT_GUI_ATHENA)
/* ARGSUSED */
    static void
timer_proc(XtPointer timed_out, XtIntervalId *interval_id)
# elif defined(FEAT_GUI_MAC)
    pascal void
timer_proc(EventLoopTimerRef theTimer, void *userData)
#elif defined(FEAT_GUI_KDE)
    void
timer_proc(void)
# endif
{
    scheme_check_threads();
# if defined(FEAT_GUI_GTK)
    return TRUE; /* continue receiving notifications */
# elif defined(FEAT_GUI_MOTIF) || defined(FEAT_GUI_ATHENA)
    /* renew timeout */
    if (mz_threads_allow && p_mzq > 0)
	timer_id = XtAppAddTimeOut(app_context, p_mzq,
		timer_proc, NULL);
# endif
}

    static void
setup_timer(void)
{
# if defined(FEAT_GUI_W32)
    timer_id = SetTimer(NULL, 0, p_mzq, timer_proc);
# elif defined(FEAT_GUI_GTK)
    timer_id = gtk_timeout_add((guint32)p_mzq, (GtkFunction)timer_proc, NULL);
# elif defined(FEAT_GUI_MOTIF) || defined(FEAT_GUI_ATHENA)
    timer_id = XtAppAddTimeOut(app_context, p_mzq, timer_proc, NULL);
# elif defined(FEAT_GUI_MAC)
    timerUPP = NewEventLoopTimerUPP(timer_proc);
    InstallEventLoopTimer(GetMainEventLoop(), p_mzq * kEventDurationMillisecond,
		p_mzq * kEventDurationMillisecond, timerUPP, NULL, &timer_id);
#elif defined(FEAT_GUI_KDE)
    mzscheme_kde_start_timer();
    timer_id = 1;   /* just signal that timer was started */
# endif
}

    static void
remove_timer(void)
{
# if defined(FEAT_GUI_W32)
    KillTimer(NULL, timer_id);
# elif defined(FEAT_GUI_GTK)
    gtk_timeout_remove(timer_id);
# elif defined(FEAT_GUI_MOTIF) || defined(FEAT_GUI_ATHENA)
    XtRemoveTimeOut(timer_id);
# elif defined(FEAT_GUI_MAC)
    RemoveEventLoopTimer(timer_id);
    DisposeEventLoopTimerUPP(timerUPP);
#elif defined(FEAT_GUI_KDE)
    mzscheme_kde_stop_timer();
# endif
    timer_id = 0;
}

    void
mzvim_reset_timer(void)
{
    if (timer_id != 0)
	remove_timer();
    if (mz_threads_allow && p_mzq > 0 && gui.in_use)
	setup_timer();
}

#endif /* MZSCHEME_GUI_THREADS */

    static void
notify_multithread(int on)
{
    mz_threads_allow = on;
#ifdef MZSCHEME_GUI_THREADS
    if (on && timer_id == 0 && p_mzq > 0 && gui.in_use)
	setup_timer();
    if (!on && timer_id != 0)
	remove_timer();
#endif
}

    int
mzscheme_enabled(int verbose)
{
    return initialized;
}

    void
mzscheme_end(void)
{
}

    static void
startup_mzscheme(void)
{
    scheme_set_stack_base(NULL, 1);

    MZ_REGISTER_STATIC(environment);
    MZ_REGISTER_STATIC(curout);
    MZ_REGISTER_STATIC(curerr);
    MZ_REGISTER_STATIC(exn_catching_apply);
    MZ_REGISTER_STATIC(exn_p);
    MZ_REGISTER_STATIC(exn_message);
    MZ_REGISTER_STATIC(vim_exn);
    MZ_REGISTER_STATIC(vim_exn_names);
    MZ_REGISTER_STATIC(vim_exn_values);

    environment = scheme_basic_env();

    /* redirect output */
    scheme_console_output = do_output;
    scheme_console_printf = do_printf;

#ifdef MZSCHEME_COLLECTS
    /* setup 'current-library-collection-paths' parameter */
    scheme_set_param(scheme_config, MZCONFIG_COLLECTION_PATHS,
	    scheme_make_pair(scheme_make_string(MZSCHEME_COLLECTS),
		scheme_null));
#endif

    /* Create buffer and window types for use in Scheme code */
    mz_buffer_type = scheme_make_type("<vim-buffer>");
    mz_window_type = scheme_make_type("<vim-window>");

    register_vim_exn(environment);
    make_modules(environment);

    /*
     * setup callback to receive notifications
     * whether thread scheduling is (or not) required
     */
    scheme_notify_multithread = notify_multithread;
    initialized = 1;
}

/*
 * This routine is called for each new invocation of MzScheme
 * to make sure things are properly initialized.
 */
    static int
mzscheme_init(void)
{
    int do_require = FALSE;

    if (!initialized)
    {
	do_require = TRUE;
        startup_mzscheme();

	if (mzscheme_io_init())
	    return -1;

    }
    /* recreate ports each call effectivelly clearing these ones */
    curout = scheme_make_string_output_port();
    curerr = scheme_make_string_output_port();
    scheme_set_param(scheme_config, MZCONFIG_OUTPUT_PORT, curout);
    scheme_set_param(scheme_config, MZCONFIG_ERROR_PORT, curerr);

    if (do_require)
    {
	/* auto-instantiate in basic env */
	eval_in_namespace("(require (prefix vimext: vimext))", do_eval,
		environment, NULL);
    }

    return 0;
}

/*
 * This routine fills the namespace with various important routines that can
 * be used within MzScheme.
 */
    static void
mzscheme_interface_init(vim_mz_buffer *mzbuff)
{
    Scheme_Object   *attach;

    mzbuff->env = (Scheme_Env *)scheme_make_namespace(0, NULL);

    /*
     * attach instantiated modules from global namespace
     * so they can be easily instantiated in the buffer namespace
     */
    attach = scheme_lookup_global(
	    scheme_intern_symbol("namespace-attach-module"),
	    environment);

    if (attach != NULL)
    {
	Scheme_Object   *ret;
	Scheme_Object	*args[2];

	args[0] = (Scheme_Object *)environment;
	args[1] = scheme_intern_symbol("vimext");

	ret = (Scheme_Object *)mzvim_apply(attach, 2, args);
    }

    add_vim_exn(mzbuff->env);
}

/*
 *========================================================================
 *  2.  External Interface
 *========================================================================
 */

/*
 * Evaluate command in namespace with exception handling
 */
    static int
eval_in_namespace(void *data, Scheme_Closed_Prim *what, Scheme_Env *env,
		Scheme_Object **ret)
{
    Scheme_Object   *value;
    Scheme_Object   *exn;
    Cmd_Info	    info;   /* closure info */

    info.data = data;
    info.env = env;

    scheme_set_param(scheme_config, MZCONFIG_ENV,
	    (Scheme_Object *) env);
    /*
     * ensure all evaluations will be in current buffer namespace,
     * the second argument to scheme_eval_string isn't enough!
     */
    value = _apply_thunk_catch_exceptions(
	    scheme_make_closed_prim_w_arity(what, &info, "mzvim", 0, 0),
	    &exn);

    if (!value)
    {
	value = extract_exn_message(exn);
        /* Got an exn? */
	if (value)
	{
	    scheme_display(value, curerr);  /*  Send to stderr-vim */
	    do_flush();
	}
	/* `raise' was called on some arbitrary value */
	return FAIL;
    }

    if (ret != NULL)	/* if pointer to retval supported give it up */
	*ret = value;
    /* Print any result, as long as it's not a void */
    else if (!SCHEME_VOIDP(value))
	scheme_display(value, curout);  /* Send to stdout-vim */

    do_flush();
    return OK;
}

/* :mzscheme */
    static int
do_mzscheme_command(exarg_T *eap, void *data, Scheme_Closed_Prim *what)
{
    if (mzscheme_init())
	return FAIL;

    range_start = eap->line1;
    range_end = eap->line2;

    return eval_in_namespace(data, what, get_vim_curr_buffer()->env, NULL);
}

/*
 * Routine called by VIM when deleting a buffer
 */
    void
mzscheme_buffer_free(buf_T *buf)
{
    if (buf->mzscheme_ref)
    {
        vim_mz_buffer *bp;
	bp = buf->mzscheme_ref;
	bp->buf = INVALID_BUFFER_VALUE;
	buf->mzscheme_ref = NULL;
	scheme_gc_ptr_ok(bp);
    }
}

/*
 * Routine called by VIM when deleting a Window
 */
    void
mzscheme_window_free(win_T *win)
{
    if (win->mzscheme_ref)
    {
	vim_mz_window *wp;
	wp = win->mzscheme_ref;
	wp->win = INVALID_WINDOW_VALUE;
	win->mzscheme_ref = NULL;
	scheme_gc_ptr_ok(wp);
    }
}

/*
 * ":mzscheme" (or ":mz")
 */
    void
ex_mzscheme(exarg_T *eap)
{
    char_u	*script;

    script = script_get(eap, eap->arg);
    if (!eap->skip)
    {
	if (script == NULL)
	    do_mzscheme_command(eap, eap->arg, do_eval);
	else
	{
	    do_mzscheme_command(eap, script, do_eval);
	    vim_free(script);
	}
    }
}

/* eval MzScheme string */
    void *
mzvim_eval_string(char_u *str)
{
    Scheme_Object *ret = NULL;
    if (mzscheme_init())
	return FAIL;

    eval_in_namespace(str, do_eval, get_vim_curr_buffer()->env, &ret);
    return ret;
}

/*
 * apply MzScheme procedure with arguments,
 * handling errors
 */
    Scheme_Object *
mzvim_apply(Scheme_Object *proc, int argc, Scheme_Object **argv)
{
    Apply_Info	data;
    Scheme_Object *ret = NULL;

    if (mzscheme_init())
	return FAIL;

    data.proc = proc;
    data.argc = argc;
    data.argv = argv;

    eval_in_namespace(&data, do_apply, get_vim_curr_buffer()->env, &ret);
    return ret;
}

    static Scheme_Object *
do_load(void *data, int noargc, Scheme_Object **noargv)
{
    Cmd_Info	    *info = (Cmd_Info *)data;
    Scheme_Object   *result = scheme_void;
    Scheme_Object   *expr;
    char_u	    *file = scheme_malloc_fail_ok(
					  scheme_malloc_atomic, MAXPATHL + 1);
    Port_Info	    *pinfo = (Port_Info *)(info->data);

    /* make Vim expansion */
    expand_env((char_u *)pinfo->name, file, MAXPATHL);
    /* scheme_load looks strange working with namespaces and error handling*/
    pinfo->port = scheme_open_input_file(file, "mzfile");
    scheme_count_lines(pinfo->port); /* to get accurate read error location*/

    /* Like REPL but print only last result */
    while (!SCHEME_EOFP(expr = scheme_read(pinfo->port)))
	result = scheme_eval(expr, info->env);

    /* errors will be caught in do_mzscheme_comamnd and ex_mzfile */
    scheme_close_input_port(pinfo->port);
    pinfo->port = NULL;
    return result;
}

/* :mzfile */
    void
ex_mzfile(exarg_T *eap)
{
    Port_Info	pinfo;

    pinfo.name = (char *)eap->arg;
    pinfo.port = NULL;
    if (do_mzscheme_command(eap, &pinfo, do_load) != OK
	    && pinfo.port != NULL)	/* looks like port was not closed */
	scheme_close_input_port(pinfo.port);
}


/*
 *========================================================================
 * Exception handling code -- cribbed form the MzScheme sources and
 * Matthew Flatt's "Inside PLT MzScheme" document.
 *========================================================================
 */
    static void
init_exn_catching_apply(void)
{
    if (!exn_catching_apply)
    {
	char *e =
	    "(lambda (thunk) "
	        "(with-handlers ([void (lambda (exn) (cons #f exn))]) "
		"(cons #t (thunk))))";

        /* make sure we have a namespace with the standard syntax: */
	Scheme_Env *env = (Scheme_Env *)scheme_make_namespace(0, NULL);
	add_vim_exn(env);

        exn_catching_apply = scheme_eval_string(e, env);
	exn_p = scheme_lookup_global(scheme_intern_symbol("exn?"), env);
        exn_message = scheme_lookup_global(
		scheme_intern_symbol("exn-message"), env);
    }
}

/*
 * This function applies a thunk, returning the Scheme value if there's
 * no exception, otherwise returning NULL and setting *exn to the raised
 * value (usually an exn structure).
 */
    static Scheme_Object *
_apply_thunk_catch_exceptions(Scheme_Object *f, Scheme_Object **exn)
{
    Scheme_Object *v;

    init_exn_catching_apply();

    v = _scheme_apply(exn_catching_apply, 1, &f);
    /* v is a pair: (cons #t value) or (cons #f exn) */

    if (SCHEME_TRUEP(SCHEME_CAR(v)))
	return SCHEME_CDR(v);
    else
    {
	*exn = SCHEME_CDR(v);
	return NULL;
    }
}

    static Scheme_Object *
extract_exn_message(Scheme_Object *v)
{
    init_exn_catching_apply();

    if (SCHEME_TRUEP(_scheme_apply(exn_p, 1, &v)))
	return _scheme_apply(exn_message, 1, &v);
    else
	return NULL; /* Not an exn structure */
}

    static Scheme_Object *
do_eval(void *s, int noargc, Scheme_Object **noargv)
{
    Cmd_Info	*info = (Cmd_Info *)s;

    return scheme_eval_string_all((char *)(info->data), info->env, TRUE);
}

    static Scheme_Object *
do_apply(void *a, int noargc, Scheme_Object **noargv)
{
    Apply_Info	*info = (Apply_Info *)(((Cmd_Info *)a)->data);

    return scheme_apply(info->proc, info->argc, info->argv);
}

/*
 *========================================================================
 *  3.  MzScheme I/O Handlers
 *========================================================================
 */
    static void
do_intrnl_output(char *mesg, long len, int error)
{
    char *p, *prev;

    prev = mesg;
    p = strchr(prev, '\n');
    while (p)
    {
	*p = '\0';
	if (error)
	    EMSG(prev);
	else
	    MSG(prev);
	prev = p + 1;
	p = strchr(prev, '\n');
    }

    if (error)
	EMSG(prev);
    else
	MSG(prev);
}

    static void
do_output(char *mesg, long len)
{
    do_intrnl_output(mesg, len, 0);
}

    static void
do_err_output(char *mesg, long len)
{
    do_intrnl_output(mesg, len, 1);
}

    static void
do_printf(char *format, ...)
{
    do_intrnl_output(format, STRLEN(format), 1);
}

    static void
do_flush(void)
{
    char *buff;
    long length;

    buff = scheme_get_sized_string_output(curerr, &length);
    if (length)
    {
	do_err_output(buff, length);
	return;
    }

    buff = scheme_get_sized_string_output(curout, &length);
    if (length)
	do_output(buff, length);
}

    static int
mzscheme_io_init(void)
{
    /* Nothing needed so far... */
    return 0;
}

/*
 *========================================================================
 *  4. Implementation of the Vim Features for MzScheme
 *========================================================================
 */

/* (command {command-string}) */
    static Scheme_Object *
vim_command(void *data, int argc, Scheme_Object **argv)
{
    Vim_Prim	*prim = (Vim_Prim *)data;
    char	*cmd = SCHEME_STR_VAL(GUARANTEE_STRING(prim->name, 0));

    /* may be use do_cmdline_cmd? */
    do_cmdline((char_u *)cmd, NULL, NULL, DOCMD_NOWAIT|DOCMD_VERBOSE);
    update_screen(VALID);

    raise_if_error();
    return scheme_void;
}

/* (eval {expr-string}) */
    static Scheme_Object *
vim_eval(void *data, int argc, Scheme_Object **argv)
{
#ifdef FEAT_EVAL
    Vim_Prim	    *prim = (Vim_Prim *)data;
    char	    *expr;
    char	    *str;
    Scheme_Object   *result;

    expr = SCHEME_STR_VAL(GUARANTEE_STRING(prim->name, 0));

    str = (char *)eval_to_string((char_u *)expr, NULL);

    if (str == NULL)
	raise_vim_exn(_("invalid expression"));

    result = scheme_make_string(str);

    vim_free(str);

    return result;
#else
    raise_vim_exn(_("expressions disabled at compile time"));
    /* unreachable */
    return scheme_false;
#endif
}

/* (range-start) */
    static Scheme_Object *
get_range_start(void *data, int argc, Scheme_Object **argv)
{
    return scheme_make_integer(range_start);
}

/* (range-end) */
    static Scheme_Object *
get_range_end(void *data, int argc, Scheme_Object **argv)
{
    return scheme_make_integer(range_end);
}

/* (beep) */
    static Scheme_Object *
mzscheme_beep(void *data, int argc, Scheme_Object **argv)
{
    vim_beep();
    return scheme_void;
}

static Scheme_Object *M_global = NULL;

/* (get-option {option-name}) [buffer/window] */
    static Scheme_Object *
get_option(void *data, int argc, Scheme_Object **argv)
{
    Vim_Prim	    *prim = (Vim_Prim *)data;
    char_u	    *name;
    long	    value;
    char_u	    *strval;
    int		    rc;
    Scheme_Object   *rval;
    int		    opt_flags = 0;
    buf_T	    *save_curb = curbuf;
    win_T	    *save_curw = curwin;

    name = (char_u *)SCHEME_STR_VAL(GUARANTEE_STRING(prim->name, 0));

    if (argc > 1)
    {
	if (M_global == NULL)
	{
	    MZ_REGISTER_STATIC(M_global);
	    M_global = scheme_intern_symbol("global");
	}

	if (argv[1] == M_global)
	    opt_flags = OPT_GLOBAL;
	else if (SCHEME_VIMBUFFERP(argv[1]))
	{
	    curbuf = get_valid_buffer(argv[1]);
	    opt_flags = OPT_LOCAL;
	}
	else if (SCHEME_VIMWINDOWP(argv[1]))
	{
	    win_T *win = get_valid_window(argv[1]);

	    curwin = win;
	    curbuf = win->w_buffer;
	    opt_flags = OPT_LOCAL;
	}
	else
	    scheme_wrong_type(prim->name, "vim-buffer/window", 1, argc, argv);
    }

    rc = get_option_value(name, &value, &strval, opt_flags);
    curbuf = save_curb;
    curwin = save_curw;

    switch (rc)
    {
    case 1:
	return scheme_make_integer_value(value);
    case 0:
        rval = scheme_make_string(strval);
	vim_free(strval);
	return rval;
    case -1:
    case -2:
        raise_vim_exn(_("hidden option"));
    case -3:
        raise_vim_exn(_("unknown option"));
    }
    /* unreachable */
    return scheme_void;
}

/* (set-option {option-changing-string} [buffer/window]) */
    static Scheme_Object *
set_option(void *data, int argc, Scheme_Object **argv)
{
    char_u	*cmd;
    int		opt_flags = 0;
    buf_T	*save_curb = curbuf;
    win_T	                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        