/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import org.videolan.vlma.exception.NotFoundException;
import org.videolan.vlma.model.Media;
import org.videolan.vlma.model.Program;
import org.videolan.vlma.model.StreamingStrategy;

/**
 * A set of multicast IP addresses. This set is used to give addresses to start
 * a new programmation.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 */
public class IpBank {

    private static final Logger logger = Logger.getLogger(IpBank.class);

    private HashMap<InetAddress, Boolean> usedIps;

    private VLMaService vlmaService;

    private Configuration configuration;

    /**
     * Initializes IP addresses according to the data interface.
     *
     * @throws UnknownHostException
     *             address is invalid
     */
    synchronized public void initIps() throws UnknownHostException {
        Inet4Address min = (Inet4Address) InetAddress.getByName(configuration.getString("vlma.streaming.udp.ipbank.min"));
        Inet4Address max = (Inet4Address) InetAddress.getByName(configuration.getString("vlma.streaming.udp.ipbank.max"));

        usedIps = new HashMap<InetAddress, Boolean>();
        usedIps.clear();

        for (int ip = ipv42int(min); ip < ipv42int(max); ip++) {
            usedIps.put(int2ipv4(ip), Boolean.FALSE);
        }

        for (Media m : vlmaService.getMedias()) {
            Program p = m.getProgram();
            if (p == null || !p.getStreamingStrategy().getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST)) {
                continue;
            }
            logger.debug("The media " + m.getName() + " has a program");
            if (usedIps.containsKey(p.getIp())) {
                usedIps.put(p.getIp(), Boolean.TRUE);
            }
        }
    }

    synchronized public boolean isUsed(InetAddress ip) {
        return usedIps.containsKey(ip) && usedIps.get(ip);
    }

    /**
     * Gives a new IP address to the programmation, the first which is not used.
     *
     * @return the IP address
     */
    synchronized public InetAddress getIp() {
        for (Map.Entry<InetAddress, Boolean> entry : usedIps.entrySet()) {
            if (!entry.getValue().booleanValue()) {
                entry.setValue(Boolean.TRUE);
                return entry.getKey();
            }
        }
        throw new NotFoundException("No free IP found in the IP bank");
    }

    /**
     * Releases an IP address.
     *
     * @param ip
     */
    synchronized public void releaseIp(InetAddress ip) {
        if(ip == null) return;
        if (usedIps.containsKey(ip)) {
            usedIps.put(ip, Boolean.FALSE);
        } else {
            logger.warn("IP address " + ip.getHostAddress() + " is not in the pool");
        }
    }

    /**
     * Converts an IPv4 address to an integer.
     *
     * @param ip the IPv4 address
     * @return the associated integer
     */
    private int ipv42int(Inet4Address ip) {
        int result = 0;
        byte[] t = ip.getAddress();
        for (int i = 0; i < 4; i++) {
            result += (t[i] & 255) << (8 * (3 - i));
        }
        return result;
    }

    /**
     * Converts an integer to a IPv4 address.
     *
     * @param ip the integer
     * @return the associated IPv4
     * @throws UnknownHostException
     */
    private Inet4Address int2ipv4 (int ip) throws UnknownHostException {
        byte[] result = new byte[4];
        for (int i = 0; i < 4; i++) {
            result[i] = (byte) ((ip >> (8 * (3 - i))) & 255);
        }
        return (Inet4Address) Inet4Address.getByAddress(result);
    }

    /**
     * Sets the VLMa service.
     *
     * @param vlmaService the vlmaService to set
     */
    public void setVlmaService(VLMaService vlmaService) {
        this.vlmaService = vlmaService;
    }

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

}
