/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.daemon;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;


/**
 * This class can get the streams which are received by an Extreme router.
 * 
 * @see IVlStreamWatcher
 * @author sylv
 * 
 */
public class VlExtremeWatcher implements IVlStreamWatcher {
    private static final Logger logger = Logger.getLogger(VlExtremeWatcher.class);
    
    /**
     * Wait step time for telnet connections.
     */
    private static final int TELNET_DEFAULT_WAIT = 100;

    /**
     * Overall wait time for telnet connections.
     */
    private static final int TELNET_DEFAULT_DELAY = 1000;

    /**
     * Router's ip to be asked.
     */
    private static final String ROUTER_HOST = "138.195.131.126";

    /**
     * Router's port on which to connect.
     */
    private static final int ROUTER_PORT = 23;

    /**
     * Username for the connection to the router.
     */
    private static final String ROUTER_USER = "user";

    /**
     * Password for the connection to the router.
     */
    private static final String ROUTER_PASSWORD = "via";

    /**
     * The VLAN to be checked.
     */
    private static final String ROUTER_VLAN = "vlip-videolan";    

    
    public Map<InetAddress, InetAddress> getStreams() {
        String toParse = "";
        // Connexion au routeur en telnet
        logger.log(Level.DEBUG, "Connexion au routeur pour récupération de l'état de flux");
        try {
            // Création du socket et connexion
            Socket socket = new Socket(ROUTER_HOST, ROUTER_PORT);
            PrintWriter out = new PrintWriter(socket.getOutputStream(), true);
            BufferedReader in = new BufferedReader(new InputStreamReader(socket.getInputStream()));
            // Envoi du login et du mot de passe
            out.println(ROUTER_USER);
            out.println(ROUTER_PASSWORD);
            // Envoi de la commande et réception du résultat
            out.println("show igmp snooping " + ROUTER_VLAN);
            out.println("          ");
            Thread.sleep(TELNET_DEFAULT_WAIT);
            int m = TELNET_DEFAULT_DELAY;
            while ((!in.ready()) && ((m -= TELNET_DEFAULT_WAIT) > 0)) {
                Thread.sleep(TELNET_DEFAULT_WAIT);
            }
            while (in.ready()) {
                toParse += (char) in.read();
            }
            // Fermeture de la connexion
            out.close();
            in.close();
            socket.close();
        } catch (UnknownHostException e) {
            logger.log(Level.ERROR, "Erreur lors de l'interrogation du routeur", e);
        } catch (IOException e) {
            logger.log(Level.ERROR, "Erreur lors de l'interrogation du routeur", e);
        } catch (InterruptedException e) {
            logger.log(Level.ERROR, "Le thread a été interrompu", e);
        }

        // Parsing des données du routeur
        HashMap<InetAddress, InetAddress> streams = new HashMap<InetAddress, InetAddress>();
        Matcher m = Pattern.compile("\\s+\\w+\\s+(\\d+\\.\\d+\\.\\d+\\.\\d+)\\s+(\\d+\\.\\d+\\.\\d+\\.\\d+)").matcher(toParse);
        while (m.find()) {
            try {
                InetAddress ip1 = InetAddress.getByName(m.group(1));
                InetAddress ip2 = InetAddress.getByName(m.group(2));
                if ((ip1.isMulticastAddress()) && (!ip2.isMulticastAddress())) {
                    streams.put(ip1, ip2);
                    logger.log(Level.DEBUG, "Le serveur " + ip2.getHostName() + " diffuse " + ip1.getHostName());
                }
            } catch (UnknownHostException e) {
                logger.log(Level.ERROR, "Impossible de résoudre une adresse", e);
            }
        }
        return streams;
    }

}
