# -*- coding: utf-8 -*-
# h-client, a client for an h-source server (such as http://www.h-node.org/)
# Copyright (C) 2011  Antonio Gallo
# Copyright (C) 2012  Michał Masłowski  <mtjm@mtjm.eu>
#
#
# h-client is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# h-client is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with h-client.  If not, see <http://www.gnu.org/licenses/>.


"""Classes providing information about specific device types.

They are used as singletons, don't instantiate them.
"""


from hclient.l10n import _


DEFAULT_INTERFACES = (_("not specified"), _("USB"), _("PCI"), _("PCI-E"),
					_("mini-PCI"), _("mini-PCI-E"), _("ExpressCard"),
					_("PC-Card"))
DEFAULT_INTERFACES_POST = ("not-specified", "USB", "PCI", "PCI-E", "mini-PCI",
						"mini-PCI-E", "ExpressCard", "PC-Card")
MOST_INTERFACES = (_("not specified"), _("USB"), _("PCI"), _("PCI-E"),
				_("mini-PCI"), _("mini-PCI-E"), _("ExpressCard"), _("PC-Card"),
				_("FireWire"), _("Parallel"), _("Serial"))
MOST_INTERFACES_POST = ("not-specified", "USB", "PCI", "PCI-E", "mini-PCI",
						"mini-PCI-E", "ExpressCard", "PC-Card", "Firewire",
						"Parallel", "Serial")


class DeviceType(object):

	"""Base class for device type objects."""

	#: Tuple of options for the "how it works" field.
	how_it_works = (_("yes"), _("no"))
	#: Tuple of protocol names for `how_it_works` entries.
	how_it_works_post = ("yes", "no")
	#: Index of the default `how_it_works` entry.
	default_how_it_works = 0
	#: Name of protocol field for `how_it_works` value, or `None` if
	#: it's not sent.
	how_it_works_post_name = None
	#: Tuple of device subtypes.
	subtypes = ()
	#: Protocol names for subtypes.
	subtypes_post = ()
	#: Tuple of options for the "it tracks users" field.
	tracks_users = ()
	#: Protocol names for "it tracks users".
	tracks_users_post = ()
	#: Tuple of years of commercialization.
	years = tuple([None] + range(2012, 1991, -1))
	#: Tuple of interface types.
	interfaces = ()
	#: Tuple of protocol names for interface types.
	interfaces_post = ()
	#: A sequence of device classes used for this type of devices.
	classes = ()
	# TODO: document what the "type name" is, since it's not the class
	# name used in Python.
	#: Type name used for this device.
	type_name = ""
	#: Controller name at h-source.
	controller = ""
	#: Icon name.
	icon = "unknown.png"

	def __init__(self):
		"""Error, since this class shouldn't be instantiated."""
		raise TypeError("DeviceType instantiated")

	@classmethod
	def post_data(cls, device):
		"""Return additional data to send when reporting a *device* of
		this type."""
		data = {}
		if cls.how_it_works_post_name is not None:
			data[cls.how_it_works_post_name] \
				= cls.how_it_works_post[device.how_it_works]
		if cls.subtypes_post:
			data["subtype"] = cls.subtypes_post[device._subtype]
		if cls.tracks_users_post:
			data["it_tracks_users"] \
				= cls.tracks_users_post[device.tracks_users]
		return data


class VideoCard(DeviceType):

	"""Video card."""

	interfaces = (_("not specified"), _("PCI"), _("AGP"), _("PCI-E"), _("ISA"),
				_("MCA"), _("VLB"))
	interfaces_post = ("not-specified", "PCI", "AGP", "PCI-E", "ISA", "MCA",
					"VLB")
	how_it_works = (_("works with 3D"), _("works without 3D"),
					_("does not work"))
	how_it_works_post = ("works_with_3D", "works_without_3D", "does_not_work")
	how_it_works_post_name = "video_card_works"
	default_how_it_works = 2
	classes = (0x0300,)
	type_name = "videocard"
	controller = "videocards"
	icon = "videocard.png"


class Wifi(DeviceType):

	"""Wi-fi card."""

	interfaces = DEFAULT_INTERFACES
	interfaces_post = DEFAULT_INTERFACES_POST
	how_it_works_post_name = "wifi_works"
	classes = (0x0280,)
	type_name = "wifi"
	controller = "wifi"
	icon = "wifi.png"


class SoundCard(DeviceType):

	"""Sound card."""

	interfaces = (_("not specified"), _("PCI"), _("ISA"), _("USB"),
				_("FireWire"), _("Parallel"), _("PCI-E"), _("PCMCIA"))
	interfaces_post = ("not-specified", "PCI", "ISA", "USB", "Firewire",
					"Parallel", "PCI-E", "PCMCIA")
	how_it_works_post_name = "sound_card_works"
	classes = (0x0403,)
	type_name = "soundcard"
	controller = "soundcards"
	icon = "soundcard.png"


class Printer(DeviceType):

	"""Printer."""

	interfaces = (_("not specified"), _("USB"), _("Serial"), _("Parallel"),
				_("FireWire"), _("SCSI"), _("Ethernet"))
	interfaces_post = ("not-specified", "USB", "Serial", "Parallel", "Firewire",
					"SCSI", "Ethernet")
	how_it_works = (_("A-Full"), _("B-Partial"), _("C-None"))
	how_it_works_post = ("A-Full", "B-Partial", "C-None")
	how_it_works_post_name = "compatibility"
	default_how_it_works = 2
	subtypes = (_("not specified"), _("laser"), _("inkjet"), _("other"))
	subtypes_post = ("not-specified", "laser", "inkjet", "other")
	tracks_users = (_("not specified"), _("no"), _("yes"))
	tracks_users_post = ("not-specified", "no", "yes")
	classes = (0x070100, 0x070101, 0x070102, 0x070103, 0x0701ff)
	type_name = "printer"
	controller = "printers"
	icon = "printer.png"


class Scanner(DeviceType):

	"""Scanner."""

	interfaces = (_("not specified"), _("USB"), _("Serial"), _("Parallel"),
				_("FireWire"), _("SCSI"), _("Ethernet"))
	interfaces_post = ("not-specified", "USB", "Serial", "Parallel", "Firewire",
					"SCSI", "Ethernet")
	how_it_works = (_("A-Full"), _("B-Partial"), _("C-None"))
	how_it_works_post = ("A-Full", "B-Partial", "C-None")
	how_it_works_post_name = "compatibility"
	default_how_it_works = 2
	classes = (0x060101,)
	type_name = "scanner"
	controller = "scanners"
	icon = "scanner.png"


class ThreeGCard(DeviceType):

	"""3G Card."""

	interfaces = DEFAULT_INTERFACES
	interfaces_post = DEFAULT_INTERFACES_POST
	how_it_works_post_name = "wifi_works"
	classes = (0x020205,)
	type_name = "3G-card"
	controller = "threegcards"
	icon = "3G-card.png"


class Webcam(DeviceType):

	"""Web cam."""

	interfaces = (_("not specified"), _("USB"), _("FireWire"), _("Parallel"),
				_("Wifi"), _("Serial"))
	interfaces_post = ("not-specified", "USB", "Firewire", "Parallel", "Wifi",
					"Serial")
	how_it_works_post_name = "webcam_works"
	classes = (0x0e0100, 0xff0100)
	type_name = "webcam"
	controller = "webcams"
	icon = "webcam.png"


class Bluetooth(DeviceType):

	"""Bluetooth."""

	interfaces = DEFAULT_INTERFACES
	interfaces_post = DEFAULT_INTERFACES_POST
	how_it_works_post_name = "bluetooth_works"
	classes = (0xe00101,)
	type_name = "bluetooth"
	controller = "bluetooth"
	icon = "bluetooth.png"


class AcquisitionCard(DeviceType):

	"""Acquisition card."""

	interfaces = MOST_INTERFACES
	interfaces_post = MOST_INTERFACES_POST
	how_it_works_post_name = "compatibility"
	classes = (0x0400, 0x0401, 0x0480)
	type_name = "acquisition-card"
	controller = "acquisitioncards"
	icon = "acquisition-card.png"


class FingerprintReader(DeviceType):

	"""Fingerprint reader."""

	interfaces = MOST_INTERFACES
	interfaces_post = MOST_INTERFACES_POST
	how_it_works_post_name = "fingerprint_works"
	classes = (0x0d0000,)
	type_name = "fingerprint-reader"
	controller = "fingerprintreaders"
	icon = "fingerprint-reader.png"


class EthernetCard(DeviceType):

	"""Ethernet card."""

	interfaces = MOST_INTERFACES
	interfaces_post = MOST_INTERFACES_POST
	how_it_works_post_name = "ethernet_card_works"
	classes = (0x0200,)
	type_name = "ethernet-card"
	controller = "ethernetcards"
	icon = "ethernet-card.png"


class SDCardReader(DeviceType):

	"""SD card reader."""

	interfaces = MOST_INTERFACES
	interfaces_post = MOST_INTERFACES_POST
	how_it_works_post_name = "sd_card_works"
	classes = (0x0805,)
	type_name = "sd-card-reader"
	controller = "sdcardreaders"
	icon = "sd-card-reader.png"


class Unknown(DeviceType):

	interfaces = (_("not specified"), _("USB"))
	interfaces_post = ("not-specified", "USB")
	classes = (0xffffff),
	type_name = "unknown"
	controller = "unknown"
	icon = "unknown.png"


# Local Variables:
# indent-tabs-mode: t
# python-guess-indent: nil
# python-indent: 4
# tab-width: 4
# End:
