!===============================================================================
! Copyright 2004-2022 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!
!   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Cluster Sparse Solver
!             Fortran example complex, double precision, unsymmetric matrix
!
!*******************************************************************************
!----------------------------------------------------------------------
! Example program to show the use of the "CLUSTER_SPARSE_SOLVER" routine
! for solving complex unsymmetric linear systems.
!---------------------------------------------------------------------
program cluster_sparse_solver_complex_unsym
use MKL_CLUSTER_SPARSE_SOLVER
use MKL_SPBLAS
use MPI
use ISO_C_BINDING
implicit none
!.. Internal solver memory pointer for 64-bit architectures
TYPE(MKL_CLUSTER_SPARSE_SOLVER_HANDLE), allocatable ::  pt(:)
!.. All other variables
INTEGER maxfct, mnum, mtype, phase, n, nnz, nrhs, error, msglvl
INTEGER*4 rank, mpi_stat
INTEGER, allocatable ::  iparm(:) ! 64
INTEGER, allocatable :: ia(:) ! 9
INTEGER, allocatable ::  ja(:) ! 20
COMPLEX(kind=8), allocatable :: a(:) ! 20
COMPLEX(kind=8), allocatable :: b(:) ! 8
COMPLEX(kind=8), allocatable :: bs(:) ! 8
COMPLEX(kind=8), allocatable :: x(:) ! 8
INTEGER i, idum(1)
COMPLEX(kind=8) ddum1(1), ddum2(1), ddum3(1)
COMPLEX(kind=8) res, res0
TYPE(SPARSE_MATRIX_T) csrA
TYPE(MATRIX_DESCR) descrA
COMPLEX(kind=8)  alpha, beta
INTEGER info

DATA n /8/, nnz /20/, nrhs /1/, maxfct /1/, mnum /1/

INTEGER*4 mkl_comm

!..
!.. Initialize MPI.
mkl_comm = MPI_COMM_WORLD
call MPI_INIT(mpi_stat)
call MPI_COMM_RANK(mkl_comm, rank, mpi_stat)

!..
!.. Set up Cluster Sparse Solver control parameter
allocate( iparm ( 64 ) )
do i = 1, 64
    iparm(i) = 0
enddo
iparm(1)  = 1 ! no solver default
iparm(2)  = 2 ! fill-in reordering from METIS
iparm(6)  = 0 ! =0 solution on the first n compoments of x
iparm(8)  = 2 ! numbers of iterative refinement steps
iparm(10) = 13 ! perturbe the pivot elements with 1E-13
iparm(11) = 1 ! use nonsymmetric permutation and scaling MPS
iparm(13) = 1 ! maximum weighted matching algorithm is switched-off
iparm(40) = 0 ! Input: matrix/rhs/solution stored on rank 0 MPI process

error  = 0 ! initialize error flag
msglvl = 1 ! print statistical information
mtype  = 13 ! symmetric, indefinite

!..
!.. Initiliaze the internal solver memory pointer. This is only
!.. necessary for the FIRST call of the Cluster Sparse Solver.
allocate( pt ( 64 ) )
do i = 1, 64
   pt(i)%DUMMY = 0
enddo

!..
!.. Since iparm(40) = 0, we fill all arrays containing matrix 
!.. data only on the rank 0 MPI process.
if (rank.eq.0) then
    allocate( ia ( n + 1 ) )
    allocate( ja ( nnz ) )
    allocate( a ( nnz ) )
    allocate( b ( n ) )
    allocate( bs( n ) )
    allocate( x ( n ) )

    ia = (/ 1,5,8,10,12,13,16,18,21 /)

    ja = (/1,          3,                 6,    7,      &
                 2,    3,          5,                   &
                       3,                             8,& 
                            4,                  7,      & 
                 2,                                     & 
                       3,                 6,          8,&
                 2,                             7,      &
                       3,                       7,    8/)

    a = (/(7.d0, 1.d0), (1.d0,1.d0), (2.d0,1.d0), (7.d0,1.d0), &
          (-4.d0,0.d0), (8.d0,1.d0), (2.d0,1.d0),              &
          (1.d0,1.d0),  (5.d0,1.d0),                           &
          (7.d0,0.d0),  (9.d0,1.d0),                           &
          (-4d0,1.d0),                                         &
          (7.d0,1.d0),  (3.d0,1.d0), (8.d0,0.d0),              &
          (1.d0,1.d0),  (11.d0,1.d0),                          &
          (-3.d0,1.d0), (2.d0,1.d0), (5.d0,0.d0)/)

endif

!..
!.. Reordering and Symbolic Factorization, This step also allocates
!.. all memory that is necessary for the factorization
phase = 11 ! only reordering and symbolic factorization
call cluster_sparse_solver (pt, maxfct, mnum, mtype, phase, n, a, ia, &
                            ja, idum, nrhs, iparm, msglvl, ddum1, ddum2,&
                            mkl_comm, error)
if (error.ne.0) then
  if (rank.eq.0) write(*,*) 'ERROR during symbolic factorization: ', error
  goto 999
endif
if (rank.eq.0) write(*,*) 'Reordering completed ... '

!..
!.. Factorization.
phase = 22 ! only factorization
call cluster_sparse_solver (pt, maxfct, mnum, mtype, phase, n, a, ia, &
                            ja, idum, nrhs, iparm, msglvl, ddum1, ddum2,&
                            mkl_comm, error)
if (error.ne.0) then
  if (rank.eq.0) write(*,*) 'ERROR during numerical factorization: ', error
  goto 999
endif
if (rank.EQ.0) write(*,*) 'Factorization completed ... '

!..
!.. Back substitution and iterative refinement
phase = 33 ! only solution
if (rank.eq.0) then
  do i = 1, n
    b(i) = (1.d0,1.d0)
  enddo
endif
call cluster_sparse_solver (pt, maxfct, mnum, mtype, phase, n, a, ia, &
                            ja, idum, nrhs, iparm, msglvl, b, x, &
                            mkl_comm, error)
if (error.ne.0) then
  if (rank.eq.0) write(*,*) 'The following ERROR was detected: ', error
  goto 999
endif

!..  
!.. Validation of solution:  Since iparm(40)=0, the solution lives 
!.. on rank 0 MPI process and should only be evaluated there.
if (rank.eq.0) then
    write(*,*) 'Solve completed ... '
    write(*,*) 'The solution of the system is '
    do i = 1, n
        write(*,'("  x( ", I1, " ) = (", F19.16, ", ", F19.16,")")') i, x(i)
    enddo

    info = mkl_sparse_z_create_csr(csrA, SPARSE_INDEX_BASE_ONE, &
                                   n, n, ia, ia(2), ja, a)

    if (info.NE.SPARSE_STATUS_SUCCESS) then
        print *, 'ERROR in mkl_sparse_z_create_csr:', info
        info =  mkl_sparse_destroy(csrA)
        goto 999
    endif
    descrA % TYPE = SPARSE_MATRIX_TYPE_GENERAL
    alpha = (1.d0, 0.d0)
    beta = (0.d0, 0.d0)
    info = mkl_sparse_z_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, &
                            csrA, descrA, x, beta, bs)
    if (info.NE.SPARSE_STATUS_SUCCESS) then
        print *, 'ERROR in mkl_sparse_z_mv:', info
        info =  mkl_sparse_destroy(csrA)
        goto 999
    endif
    info =  mkl_sparse_destroy(csrA)
    res  = (0.d0,0.d0)
    res0 = (0.d0,0.d0)
    do i=1,n
        res = res + (bs(i)-b(i))*conjg((bs(i)-b(i)))
        res0 = res0 + b(i)*conjg(b(i))
    enddo
    print *, 'Relative residual = ', sqrt(abs(res))/sqrt(abs(res0))
endif

!..
!.. Termination and release of memory
phase = -1 ! release internal memory
call cluster_sparse_solver (pt, maxfct, mnum, mtype, phase, n, ddum1, &
                            idum, idum, idum, nrhs, iparm, msglvl, &
                            ddum2, ddum3, mkl_comm, error)
if (error .NE. 0) then
    if (rank.EQ.0) write(*,*) 'The following ERROR was detected: ', error
    goto 999
endif

999   continue
if (rank.eq.0) then
    if ( allocated( ia ) )      deallocate( ia )
    if ( allocated( ja ) )      deallocate( ja )
    if ( allocated( a ) )       deallocate( a )
    if ( allocated( b ) )       deallocate( b )
    if ( allocated( bs ) )      deallocate( bs )
    if ( allocated( x ) )       deallocate( x )
endif
if ( allocated( pt ) )      deallocate( pt )
if ( allocated( iparm ) )   deallocate( iparm )

if (rank.eq.0) then
    if (sqrt(abs(res))/sqrt(abs(res0)).gt.1.d-10) then
        print *, 'Error: residual is too high!'
        error = 1
    endif
    if (error.ne.0) then
        print *, char(10), 'TEST FAILED'
    else
        print *, char(10), 'TEST PASSED'
    endif
endif
call mpi_finalize(mpi_stat)
if (error.ne.0.and.rank.eq.0) then
    stop 1
endif
end
