;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2014-2025 Ricardo Wurmus <rekado@elephly.net>
;;; Copyright © 2015, 2016, 2017, 2018 Ben Woodcroft <donttrustben@gmail.com>
;;; Copyright © 2015, 2016, 2018, 2019, 2020 Pjotr Prins <pjotr.guix@thebird.nl>
;;; Copyright © 2015 Andreas Enge <andreas@enge.fr>
;;; Copyright © 2016, 2020, 2021 Roel Janssen <roel@gnu.org>
;;; Copyright © 2016-2024 Efraim Flashner <efraim@flashner.co.il>
;;; Copyright © 2016, 2020, 2022 Marius Bakke <marius@gnu.org>
;;; Copyright © 2016, 2018 Raoul Bonnal <ilpuccio.febo@gmail.com>
;;; Copyright © 2017, 2018 Tobias Geerinckx-Rice <me@tobias.gr>
;;; Copyright © 2017, 2021, 2022, 2024, 2025 Arun Isaac <arunisaac@systemreboot.net>
;;; Copyright © 2018 Joshua Sierles, Nextjournal <joshua@nextjournal.com>
;;; Copyright © 2018 Gábor Boskovits <boskovits@gmail.com>
;;; Copyright © 2018-2024 Mădălin Ionel Patrașcu <madalinionel.patrascu@mdc-berlin.de>
;;; Copyright © 2019, 2020, 2021, 2023 Maxim Cournoyer <maxim@guixotic.coop>
;;; Copyright © 2019 Brian Leung <bkleung89@gmail.com>
;;; Copyright © 2019 Brett Gilio <brettg@gnu.org>
;;; Copyright © 2020 Björn Höfling <bjoern.hoefling@bjoernhoefling.de>
;;; Copyright © 2020 Jakub Kądziołka <kuba@kadziolka.net>
;;; Copyright © 2020 Pierre Langlois <pierre.langlois@gmx.com>
;;; Copyright © 2020 Bonface Munyoki Kilyungi <bonfacemunyoki@gmail.com>
;;; Copyright © 2021 Tim Howes <timhowes@lavabit.com>
;;; Copyright © 2021 Hong Li <hli@mdc-berlin.de>
;;; Copyright © 2021, 2022, 2023 Simon Tournier <zimon.toutoune@gmail.com>
;;; Copyright © 2021 Felix Gruber <felgru@posteo.net>
;;; Copyright © 2022-2025 Navid Afkhami <navid.afkhami@mdc-berlin.de>
;;; Copyright © 2022 Antero Mejr <antero@mailbox.org>
;;; Copyright © 2024 Alexis Simon <alexis.simon@runbox.com>
;;; Copyright © 2024 Spencer King <spencer.king@geneoscopy.com>
;;; Copyright © 2025 nomike Postmann <nomike@nomike.com>
;;; Copyright © 2025 Janneke Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2025 Sharlatan Hellseher <sharlatanus@gmail.com>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages bioinformatics)
  #:use-module ((guix licenses) #:prefix license:)
  #:use-module (guix packages)
  #:use-module (guix gexp)
  #:use-module (guix utils)
  #:use-module (guix download)
  #:use-module (guix git-download)
  #:use-module (guix hg-download)
  #:use-module (guix build-system ant)
  #:use-module (guix build-system gnu)
  #:use-module (guix build-system cargo)
  #:use-module (guix build-system cmake)
  #:use-module (guix build-system go)
  #:use-module (guix build-system haskell)
  #:use-module (guix build-system meson)
  #:use-module (guix build-system perl)
  #:use-module (guix build-system pyproject)
  #:use-module (guix build-system python)
  #:use-module (guix build-system qt)
  #:use-module (guix build-system r)
  #:use-module (guix build-system ruby)
  #:use-module (guix build-system trivial)
  #:use-module (guix deprecation)
  #:use-module (gnu packages)
  #:use-module (gnu packages admin)
  #:use-module (gnu packages assembly)
  #:use-module (gnu packages autotools)
  #:use-module (gnu packages algebra)
  #:use-module (gnu packages base)
  #:use-module (gnu packages bash)
  #:use-module (gnu packages bison)
  #:use-module (gnu packages bioconductor)
  #:use-module (gnu packages boost)
  #:use-module (gnu packages certs)
  #:use-module (gnu packages check)
  #:use-module (gnu packages code)
  #:use-module (gnu packages cmake)
  #:use-module (gnu packages compression)
  #:use-module (gnu packages cpp)
  #:use-module (gnu packages cpio)
  #:use-module (gnu packages cran)
  #:use-module (gnu packages curl)
  #:use-module (gnu packages docbook)
  #:use-module (gnu packages documentation)
  #:use-module (gnu packages databases)
  #:use-module (gnu packages datastructures)
  #:use-module (gnu packages dlang)
  #:use-module (gnu packages file)
  #:use-module (gnu packages flex)
  #:use-module (gnu packages gawk)
  #:use-module (gnu packages gcc)
  #:use-module (gnu packages gd)
  #:use-module (gnu packages ghostscript)
  #:use-module (gnu packages golang)
  #:use-module (gnu packages golang-build)
  #:use-module (gnu packages golang-check)
  #:use-module (gnu packages golang-compression)
  #:use-module (gnu packages golang-xyz)
  #:use-module (gnu packages glib)
  #:use-module (gnu packages gnupg)
  #:use-module (gnu packages graph)
  #:use-module (gnu packages graphics)
  #:use-module (gnu packages graphviz)
  #:use-module (gnu packages groff)
  #:use-module (gnu packages gtk)
  #:use-module (gnu packages guile)
  #:use-module (gnu packages guile-xyz)
  #:use-module (gnu packages haskell-check)
  #:use-module (gnu packages haskell-web)
  #:use-module (gnu packages haskell-xyz)
  #:use-module (gnu packages image)
  #:use-module (gnu packages image-processing)
  #:use-module (gnu packages imagemagick)
  #:use-module (gnu packages java)
  #:use-module (gnu packages java-compression)
  #:use-module (gnu packages java-xml)
  #:use-module (gnu packages jemalloc)
  #:use-module (gnu packages jupyter)
  #:use-module (gnu packages libffi)
  #:use-module (gnu packages linux)
  #:use-module (gnu packages llvm)
  #:use-module (gnu packages logging)
  #:use-module (gnu packages lsof)
  #:use-module (gnu packages machine-learning)
  #:use-module (gnu packages man)
  #:use-module (gnu packages maths)
  #:use-module (gnu packages mpi)
  #:use-module (gnu packages ncurses)
  #:use-module (gnu packages node)
  #:use-module (gnu packages nss)
  #:use-module (gnu packages ocaml)
  #:use-module (gnu packages pcre)
  #:use-module (gnu packages package-management)
  #:use-module (gnu packages parallel)
  #:use-module (gnu packages pdf)
  #:use-module (gnu packages perl)
  #:use-module (gnu packages perl-check)
  #:use-module (gnu packages perl-web)
  #:use-module (gnu packages perl6)
  #:use-module (gnu packages pkg-config)
  #:use-module (gnu packages popt)
  #:use-module (gnu packages protobuf)
  #:use-module (gnu packages python)
  #:use-module (gnu packages python-build)
  #:use-module (gnu packages python-check)
  #:use-module (gnu packages python-compression)
  #:use-module (gnu packages python-crypto)
  #:use-module (gnu packages python-science)
  #:use-module (gnu packages python-web)
  #:use-module (gnu packages python-xyz)
  #:use-module (gnu packages qt)
  #:use-module (gnu packages rdf)
  #:use-module (gnu packages readline)
  #:use-module (gnu packages rpc)
  #:use-module (gnu packages rsync)
  #:use-module (gnu packages ruby)
  #:use-module (gnu packages ruby-check)
  #:use-module (gnu packages ruby-xyz)
  #:use-module (gnu packages serialization)
  #:use-module (gnu packages shells)
  #:use-module (gnu packages skribilo)
  #:use-module (gnu packages sphinx)
  #:use-module (gnu packages sqlite)
  #:use-module (gnu packages statistics)
  #:use-module (gnu packages swig)
  #:use-module (gnu packages tbb)
  #:use-module (gnu packages tex)
  #:use-module (gnu packages texinfo)
  #:use-module (gnu packages textutils)
  #:use-module (gnu packages time)
  #:use-module (gnu packages tls)
  #:use-module (gnu packages uglifyjs)
  #:use-module (gnu packages video)
  #:use-module (gnu packages vim)
  #:use-module (gnu packages web)
  #:use-module (gnu packages wget)
  #:use-module (gnu packages xml)
  #:use-module (gnu packages xorg)
  #:use-module (guix platform)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match))

(define-public aragorn
  (package
    (name "aragorn")
    (version "1.2.41")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://www.ansikte.se/ARAGORN/Downloads/"
                                  "aragorn" version ".c"))
              (sha256
               (base32
                "0jkzx7sqiarydvz3bwaxh790fpvpkfc926dhsza1dbdhq32ir8wj"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f ; there are no tests
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (replace 'build
            (lambda _
              (invoke "gcc" "-O3" "-ffast-math" "-finline-functions"
                      "-o" "aragorn" #$source)))
          (replace 'install
            (lambda* (#:key inputs #:allow-other-keys)
              (let ((bin (string-append #$output "/bin"))
                    (man (string-append #$output "/share/man/man1")))
                (install-file "aragorn" bin)
                (mkdir-p man)
                (copy-file (assoc-ref inputs "aragorn.1")
                           (string-append man "/aragorn.1"))))))))
    (native-inputs
     `(("aragorn.1"
        ,(origin
           (method url-fetch)
           (uri "https://www.ansikte.se/ARAGORN/Downloads/aragorn.1")
           (sha256
            (base32
             "0bn9lapa6f0cl07dbn2fjrapirv9d4bk7w248w39fhb4vbczcc3f"))))))
    (home-page "https://www.ansikte.se/ARAGORN/")
    (synopsis "Detect tRNA, mtRNA and tmRNA genes in nucleotide sequences")
    (description
     "Aragorn identifies transfer RNA, mitochondrial RNA and
transfer-messenger RNA from nucleotide sequences, based on homology to known
tRNA consensus sequences and RNA structure.  It also outputs the secondary
structure of the predicted RNA.")
    (license license:gpl3+)))

(define-public bamtools
  (package
    (name "bamtools")
    (version "2.5.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/pezmaster31/bamtools")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14lw571vbks138i0lj66qjdbk8iwa817x2zbpzij61vv1gdgfbn5"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:configure-flags
      '(list "-DBUILD_SHARED_LIBS=ON")))
    (inputs (list jsoncpp zlib))
    (native-inputs (list pkg-config))
    (home-page "https://github.com/pezmaster31/bamtools")
    (synopsis "C++ API and command-line toolkit for working with BAM data")
    (description
     "BamTools provides both a C++ API and a command-line toolkit for handling
BAM files.")
    (license license:expat)))

(define-public bamutils
  (package
    (name "bamutils")
    (version "1.0.15")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/statgen/bamUtil")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pxydf9qsr8667jh525bc2wiqn9nwk8rkg05kbyfmjs8d261fl9y"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f ;fails to link debug libraries
      #:test-target "test"
      #:make-flags
      #~(list (string-append "INSTALLDIR=" #$output "/bin"))
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "src/Makefile"
                (("^DATE=.*") "DATE=\"1970-01-01\"\n"))
              (copy-recursively (assoc-ref inputs "libstatgen")
                                "../libStatGen"))))))
    (inputs
     (list zlib))
    (native-inputs
     `(("libstatgen"
        ,(origin
           (method git-fetch)
           (uri (git-reference
                 (url "https://github.com/statgen/libStatGen/")
                 (commit (string-append "v" version))))
           (file-name (git-file-name "libstatgen" version))
           (sha256
            (base32
             "0spvbpvnpxrgj8kajpkhf1mv7kdyvj723y9zh13jykvnjh8a15j3"))))))
    (home-page "https://genome.sph.umich.edu/wiki/BamUtil")
    (synopsis "Programs for working on SAM/BAM files")
    (description "This package provides several programs that perform
operations on SAM/BAM files.  All of these programs are built into a
single executable called @code{bam}.")
    (license license:gpl3+)))

(define-public bcftools
  (package
    (name "bcftools")
    (version "1.21")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/samtools/bcftools/"
                                  "releases/download/"
                                  version "/bcftools-" version ".tar.bz2"))
              (sha256
               (base32
                "10p2ligd9pqn8wq6szw2s4zqknlm7hmb4057fpdnhlsmsg0lr2jj"))
              (modules '((guix build utils)))
              (snippet '(begin
                          ;; Delete bundled htslib.
                          (delete-file-recursively "htslib-1.21")))))
    (build-system gnu-build-system)
    (arguments
     `(#:configure-flags
       (list "--enable-libgsl")
       #:test-target "test"
       #:phases
       (modify-phases %standard-phases
         (add-before 'check 'patch-tests
           (lambda _
             (substitute* "test/test.pl"
               (("/bin/bash") (which "bash"))))))))
    (native-inputs
     (list htslib perl))
    (inputs
     (list gsl zlib))
    (home-page "https://samtools.github.io/bcftools/")
    (synopsis "Utilities for variant calling and manipulating VCFs and BCFs")
    (description
     "BCFtools is a set of utilities that manipulate variant calls in the
Variant Call Format (VCF) and its binary counterpart BCF.  All commands work
transparently with both VCFs and BCFs, both uncompressed and BGZF-compressed.")
    ;; The sources are dual MIT/GPL, but becomes GPL-only when USE_GPL=1.
    (license (list license:gpl3+ license:expat))))

(define-public bedops
  (package
    (name "bedops")
    (version "2.4.41")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/bedops/bedops")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "046037qdxsn01ln28rbrwnc7wq4a3xahmb2k74l0w75dby5ni42l"))))
    (build-system gnu-build-system)
    (arguments
     (list
      ;; We cannot run the tests because the build system makes strange
      ;; assumptions about where executables are located.
      #:tests? #false
      #:test-target "tests"
      #:make-flags
      #~(list (string-append "CC=" #$(cc-for-target)))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'unpack-tarballs
            (lambda _
              ;; FIXME: Bedops includes tarballs of minimally patched upstream
              ;; libraries jansson, zlib, and bzip2.  We cannot just use stock
              ;; libraries because at least one of the libraries (zlib) is
              ;; patched to add a C++ function definition (deflateInit2cpp).
              ;; Until the Bedops developers offer a way to link against system
              ;; libraries we have to build the in-tree copies of these three
              ;; libraries.

              ;; See upstream discussion:
              ;; https://github.com/bedops/bedops/issues/124

              ;; Unpack the tarballs to benefit from shebang patching.
              (with-directory-excursion "third-party"
                (invoke "tar" "xvf" "jansson-2.6.tar.bz2")
                (invoke "tar" "xvf" "zlib-1.2.7.tar.bz2")
                (invoke "tar" "xvf" "bzip2-1.0.6.tar.bz2"))
              ;; Disable unpacking of tarballs in Makefile.
              (substitute* "system.mk/Makefile.linux"
                (("^\tbzcat .*") "\t@echo \"not unpacking\"\n")
                (("\\./configure") "CONFIG_SHELL=bash ./configure"))
              (substitute* "third-party/zlib-1.2.7/Makefile.in"
                (("^SHELL=.*$") "SHELL=bash\n"))))
          (delete 'configure)
          (replace 'install
            (lambda _
              (invoke "make" "install"
                      (string-append "BINDIR=" #$output "/bin")))))))
    (native-inputs
     (list diffutils perl which))
    (home-page "https://github.com/bedops/bedops")
    (synopsis "Tools for high-performance genomic feature operations")
    (description
     "BEDOPS is a suite of tools to address common questions raised in genomic
studies---mostly with regard to overlap and proximity relationships between
data sets.  It aims to be scalable and flexible, facilitating the efficient
and accurate analysis and management of large-scale genomic data.

BEDOPS provides tools that perform highly efficient and scalable Boolean and
other set operations, statistical calculations, archiving, conversion and
other management of genomic data of arbitrary scale.  Tasks can be easily
split by chromosome for distributing whole-genome analyses across a
computational cluster.")
    (license license:gpl2+)))

(define-public bedtools
  (package
    (name "bedtools")
    (version "2.31.1")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/arq5x/bedtools2/releases/"
                                  "download/v" version "/"
                                  "bedtools-" version ".tar.gz"))
              (sha256
               (base32
                "15rbfg776pcjxqrsa22dk7sdlmx1lijh3jhap04f1cbr4866czpw"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:test-target "test"
      #:make-flags
      #~(list (string-append "prefix=" #$output))
      #:phases
      '(modify-phases %standard-phases
         (delete 'configure))))
    (native-inputs
     (list python-wrapper))
    (inputs
     (list samtools zlib))
    (home-page "https://github.com/arq5x/bedtools2")
    (synopsis "Tools for genome analysis and arithmetic")
    (description
     "Collectively, the bedtools utilities are a swiss-army knife of tools for
a wide-range of genomics analysis tasks.  The most widely-used tools enable
genome arithmetic: that is, set theory on the genome.  For example, bedtools
allows one to intersect, merge, count, complement, and shuffle genomic
intervals from multiple files in widely-used genomic file formats such as BAM,
BED, GFF/GTF, VCF.")
    (license license:expat)))

;; Later releases of bedtools produce files with more columns than
;; what Ribotaper expects.
(define-public bedtools-2.18
  (package (inherit bedtools)
    (name "bedtools")
    (version "2.18.0")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/arq5x/bedtools2/"
                                  "releases/download/v" version
                                  "/bedtools-" version ".tar.gz"))
              (sha256
               (base32
                "11rvca19ncg03kxd0wzlfx5ws7r3nisd0z8s9j9n182d8ksp2pxz"))))
    (arguments
     (list
      #:test-target "test"
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'compatibility
            (lambda _
              (substitute* "src/utils/fileType/FileRecordTypeChecker.h"
                (("static const float PERCENTAGE")
                 "static constexpr float PERCENTAGE"))
              (substitute* "src/utils/general/DualQueue.h"
                (("template <class T, template<class T> class CompareFunc>")
                 "template <class T, template<class U> class CompareFunc>"))
              (substitute* '("src/utils/BamTools/src/api/algorithms/Sort.h"
                             "src/utils/BamTools/src/api/internal/bam/BamMultiMerger_p.h")
                (("(bool operator\\(\\).*) \\{" m pre)
                 (string-append pre " const {")))))
          (delete 'configure)
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin/")))
                (for-each (lambda (file)
                            (install-file file bin))
                          (find-files "bin" ".*"))))))))
    (native-inputs
     (list python-wrapper))
    (inputs
     (list samtools zlib))))

(define-public bitmapperbs
  (package
    (name "bitmapperbs")
    (version "1.0.2.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/chhylp123/BitMapperBS/")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "02ksssfnvmpskld0a2016smfz5nrzm3d90v8974f3cpzywckvp8v"))
              (modules '((guix build utils)))
              ;; This package bundles a modified copy of htslib, so we cannot
              ;; unbundle it.
              (snippet
               '(begin
                  (delete-file-recursively "libdivsufsort-2.0.1")
                  (delete-file-recursively "pSAscan-0.1.0")))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #false
      #:make-flags '(list "bitmapperBS")
      ;; The build system checks for CPU features.  For this reason, we want
      ;; users to build it locally instead of using substitutes.
      #:substitutable? #false
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-build-system
            (lambda _
              (substitute* "Makefile"
                (("make prefix=../htslib_aim install")
                 (string-append "make prefix=" #$output " install-so"))
                (("htslib_aim/include") "htslib")
                (("htslib_aim/lib")
                 (string-append #$output "/lib")))))
          (add-after 'unpack 'patch-references-to-psascan
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "Makefile"
                (("\"(./)?psascan" pre all)
                 (string-append "\"" pre (search-input-file inputs "/bin/psascan"))))))
          (delete 'configure)
          (replace 'install
            (lambda _
              (install-file "bitmapperBS"
                            (string-append #$output "/bin/")))))))
    (inputs
     (list libdivsufsort psascan zlib))
    (home-page "https://github.com/chhylp123/BitMapperBS/")
    (synopsis "Read aligner for whole-genome bisulfite sequencing")
    (description
     "BitMapperBS is memory-efficient aligner that is designed for
whole-genome bisulfite sequencing (WGBS) reads from directional protocol.")
    (supported-systems '("x86_64-linux"))
    (license license:asl2.0)))

(define-public ngs-bits
  (package
    (name "ngs-bits")
    (version "2025_09")
    (home-page "https://github.com/imgag/ngs-bits")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                     (url home-page)
                     (commit version)
                     (recursive? #t))) ;for src/ cppCORE, cppGUI, ccpXML, cppTFW
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18iphn4zxgs8iclk3kcahnb08581a6l01fhrrcql1z20fdh478np"))
              (modules '((guix build utils)))
              ;; This package bundles a modified copy of htslib, so we cannot
              ;; unbundle it.
              (snippet
               '(for-each delete-file-recursively
                          '("htslib"
                            "libxml2")))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:modules '((guix build gnu-build-system)
                  (guix build utils)
                  (srfi srfi-26)
                  (ice-9 match))
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda _
              (setenv "CPLUS_INCLUDE_PATH"
                      (string-append
                       #$(this-package-input "libxml2")
                       "/include/libxml2"
                       ":" (getenv "CPLUS_INCLUDE_PATH")))
              (substitute* "src/base.pri"
                (("QMAKE_LFLAGS [+]=" all)
                 (string-append all " -Wl,-rpath=" #$output "/lib")))))
          (replace 'build
            (lambda _
              (invoke "make" "build_libs_release")
              (invoke "make" "build_tools_release")))
          (add-after 'unpack 'patch-tests
            (lambda _
              ;; Programming exception: Requested key 'liftover_hg19_hg38' not
              ;; found in settings!
              (substitute* "src/tools-TEST/tools-TEST.pro"
                (("BedLiftOver_Test.cpp") ""))
              (let ((coreutils
                     (match (assoc-ref '#$(standard-packages) "coreutils")
                       ((package) package))))
                (substitute* "src/cppCORE-TEST/Helper_Test.cpp"
                  (("/usr/bin/whoami")
                   (string-append coreutils "/bin/whoami"))))))
          (replace 'check
            (lambda _
              ;; Could not create application data path
              ;; '/homeless-shelter/.local/share/cppCORE-TEST'!
              (setenv "HOME" "/tmp")
              (invoke "make" "test_lib")
              (invoke "make" "test_tools")))
          (delete 'install)
          (add-before 'check 'install   ;check creates garbage in bin/
            (lambda _
              (define (install-file* file target)
                (let ((dest (string-append target "/" (basename file))))
                  (format #t "`~a' -> `~a'~%" file dest)
                  (mkdir-p (dirname dest))
                  (let ((stat (lstat file)))
                    (case (stat:type stat)
                      ((symlink)
                       (let ((target (readlink file)))
                         (symlink target dest)))
                      (else
                       (copy-file file dest))))
                  (delete-file file)))
              (copy-recursively "bin" "tmp")
              (let ((doc (string-append #$output "/share/doc/" #$name))
                    (examples (find-files "tmp" "[.]example")))
                (for-each (cute install-file* <> doc) examples))
              ;; FIXME: .INI and .TSV files are read from the bin directory
              ;; (let ((etc (string-append #$output "/etc"))
              ;;       (ini (find-files "tmp" "[.]ini")))
              ;;   (for-each (cute install-file* <> etc) ini))
              (let ((lib (string-append #$output "/lib"))
                    (libraries (find-files "tmp" "^lib.*so")))
                (for-each (cute install-file* <> lib) libraries))
              (let ((bin (string-append #$output "/bin"))
                    (binaries (find-files "tmp")))
                (for-each (cute install-file* <> bin) binaries))
              (rmdir "tmp"))))))
    (inputs (list
             bzip2
             curl
             htslib
             libdeflate
             libxml2
             lzip
             openssl
             qtbase-5
             qtsvg-5
             zlib))
    (native-inputs (list libtool
                         pkg-config
                         python            ;for tests
                         python-matplotlib ;for tests
                         python-numpy))    ;for tests
    (synopsis "Short-read and long-read sequencing tools for diagnostics")
    (description
     "Ngs-bits (Next-Generation Sequencing) is collection of short-read and
long-read sequencing tools for diagnostics.")
    (license license:expat)))

(define-public bustools
  (package
    (name "bustools")
    (version "0.43.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BUStools/bustools")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "068kjlc4d528269nl5mc3j8h2c95r1v545d3fi1iw1ckg8rba0hg"))))
    (build-system cmake-build-system)
    (arguments (list #:tests? #f))          ;no test target
    (inputs (list zlib))
    (home-page "https://bustools.github.io")
    (synopsis "Tools for working with BUS files")
    (description "bustools is a program for manipulating BUS files for single
cell RNA-Seq datasets.  It can be used to error correct barcodes, collapse
UMIs, produce gene count or transcript compatibility count matrices, and is useful
for many other tasks.")
    (license license:bsd-2)))

(define-public cpat
  (package
    (name "cpat")
    (version "3.0.4")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "CPAT" version))
              (sha256
               (base32
                "0dfrwwbhv1n4nh2a903d1qfb30fgxgya89sa70aci3wzf8h2z0vd"))
              (modules '((guix build utils)))
              (snippet
               '(for-each delete-file-recursively
                          (list ".eggs"
                                "lib/__pycache__/"
                                "lib/cpmodule/__pycache__/")))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (with-directory-excursion "test"
                 ;; There is no test4.fa
                 (substitute* "test.sh"
                   ((".*-g test4.fa.*") ""))
                 (invoke "bash" "test.sh"))))))))
    (propagated-inputs
     (list python-numpy python-pysam))
    (inputs
     (list r-minimal))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://wlcb.oit.uci.edu/cpat/")
    (synopsis "Alignment-free distinction between coding and noncoding RNA")
    (description
     "CPAT is a method to distinguish coding and noncoding RNA by using a
logistic regression model based on four pure sequence-based, linguistic
features: ORF size, ORF coverage, Ficket TESTCODE, and Hexamer usage bias.
Linguistic features based method does not require other genomes or protein
databases to perform alignment and is more robust.  Because it is
alignment-free, it runs much faster and also easier to use.")
    (license license:gpl2+)))

(define-public r-anglemania
  (let ((commit "f27399fb947adfa0de6134493e737658ca591af5")
        (revision "1"))
    (package
      (name "r-anglemania")
      (version (git-version "0.99.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/anglemania/")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "05s60wba5jbg81lflifaz9yykl86f3is1y4b63c7rp6vmd5jf5y3"))))
      (properties `((upstream-name . "anglemania")))
      (build-system r-build-system)
      (propagated-inputs (list r-bigparallelr
                               r-bigstatsr
                               r-checkmate
                               r-digest
                               r-dplyr
                               r-magrittr
                               r-matrix
                               r-pbapply
                               r-rcpp
                               r-rmio
                               r-s4vectors
                               r-seurat
                               r-seuratobject
                               r-singlecellexperiment
                               r-tidyr
                               r-withr))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/BIMSBbioinfo/anglemania/")
      (synopsis "Feature extraction for scRNA-seq dataset integration")
      (description
       "Anglemania extracts genes from multi-batch @code{scRNA-seq}
experiments for downstream dataset integration.  It improves conventional
usage of highly-variable genes for integration tasks.")
      (license license:gpl3+))))

(define-public r-anndatar
  (let ((commit "5c3eb7e498d0d9bf1c522ad66f4eb8ad277238b6")
        (revision "1"))
    (package
      (name "r-anndatar")
      (version (git-version "0.99.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/scverse/anndataR")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0sx87i8cb4p08ihgpgflxs0fhkr1kw6lxvky4w766rq7wqy41cgk"))))
      (properties
       '((upstream-name . "anndataR")
         (updater-extra-native-inputs . ("r-vctrs"))))
      (build-system r-build-system)
      (propagated-inputs (list r-matrix r-r6))
      (native-inputs
       (list r-knitr
             r-rhdf5
             r-seuratobject
             r-singlecellexperiment
             r-testthat
             r-vctrs))
      (home-page "https://github.com/scverse/anndataR")
      (synopsis "AnnData interoperability in R")
      (description
       "This package aims to bring the power and flexibility of @code{AnnData}
to the R ecosystem, allowing you to effortlessly manipulate and analyze your
single-cell data.  This package lets you work with backed h5ad and zarr files,
directly access various slots (e.g. X, obs, var), or convert the data into
@code{SingleCellExperiment} and Seurat objects.")
      (license license:expat))))

(define-public r-anpan
  (let ((commit "efd4dc4874c418a43104ed3f418283d2451f0483")
        (revision "1"))
    (package
      (name "r-anpan")
      (version (git-version "0.3.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/biobakery/anpan")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "02w281n489wsn0a935dcj7gmd8fjph5wba3ldlp5qz15brcp0w48"))))
      (properties
       '((upstream-name . "anpan")
         (updater-extra-native-inputs . ("r-testthat"))))
      (build-system r-build-system)
      ;; XXX: We don't actually provide cmdstan itself, just the R interface,
      ;; so we cannot run the tests.
      (arguments (list #:tests? #false))
      (propagated-inputs (list r-ape
                               r-cli
                               r-cmdstanr
                               r-data-table
                               r-dplyr
                               r-fastglm
                               r-furrr
                               r-future
                               r-ggdendro
                               r-ggnewscale
                               r-ggplot2
                               r-loo
                               r-patchwork
                               r-phylogram
                               r-posterior
                               r-progressr
                               r-purrr
                               r-r-utils
                               r-stringr
                               r-tibble
                               r-tidyselect))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/biobakery/anpan")
      (synopsis "Quantifying microbial strain-host associations")
      (description
       "The goal of anpan is to consolidate statistical methods for strain
analysis.  This includes automated filtering of metagenomic functional
profiles, testing genetic elements for association with outcomes, phylogenetic
association testing, and pathway-level random effects models.")
      (license license:expat))))

(define-public r-bedtorch
  (let ((commit "f5ff4f83b94f59eac660333c64e4b2f296b35cea")
        (revision "1"))
    (package
      (name "r-bedtorch")
      (version (git-version "0.1.12.12" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/haizi-zh/bedtorch/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "08l04iqf54b5995gc7rvqqd7w327fwqs7qjyhc9y5cqfj8yv4c48"))))
      (properties `((upstream-name . "bedtorch")))
      (build-system r-build-system)
      ;; Some of the tests attempt to download files off the internet.
      (arguments (list #:tests? #false))
      (propagated-inputs
       (list r-assertthat
             r-curl
             r-data-table
             r-dplyr
             r-genomeinfodb
             r-genomicranges
             r-purrr
             r-r-utils
             r-rcpp
             r-rcurl
             r-readr
             r-rhtslib12
             r-s4vectors
             r-stringr
             r-tidyr))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/haizi-zh/bedtorch/")
      (synopsis "R package for fast BED-file manipulation")
      (description
       "The goal of bedtorch is to provide a fast BED file manipulation tool
suite native in R.")
      (license license:expat))))

(define-public r-bpcells
  (package
    (name "r-bpcells")
    (version "0.3.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/bnprks/BPCells")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "12h1di45fg06nhsliii7v0h1z567pkpis1xm4ar1qr8ns1n8iljw"))))
    (properties `((upstream-name . "BPCells")))
    (build-system r-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'chdir
           (lambda _ (chdir "r"))))))
    (propagated-inputs
     (list r-dplyr
           r-ggplot2
           r-ggrepel
           r-hexbin
           r-lifecycle
           r-magrittr
           r-matrix
           r-patchwork
           r-rcolorbrewer
           r-rcpp
           r-rcppeigen
           r-readr
           r-rlang
           r-scales
           r-scattermore
           r-stringr
           r-tibble
           r-tidyr
           r-vctrs

           ;; Suggested packages
           r-genomicranges
           r-igraph
           r-iranges
           r-matrixstats))
    (inputs (list hdf5 zlib))
    (native-inputs (list pkg-config))
    (home-page "https://github.com/bnprks/BPCells")
    (synopsis "Single cell counts matrices to PCA")
    (description
     "This is a package providing efficient operations for single cell
ATAC-seq fragments and RNA counts matrices.  It is interoperable with standard
file formats, and introduces efficient bit-packed formats that allow large
storage savings and increased read speeds.")
    ;; Either license
    (license (list license:asl2.0 license:expat))))

(define-public r-btools
  (let ((commit "fa21d4ca01d37ea4d98b45582453f3bf95cbc2b5")
        (revision "1"))
    (package
      (name "r-btools")
      (version (git-version "0.0.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/twbattaglia/btools")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0bca593dnxz6xdywpdi0ipli0paln2b3bfxxj0csnmj55ryrz428"))))
      (properties `((upstream-name . "btools")))
      (build-system r-build-system)
      ;; There are no tests.
      (arguments (list #:tests? #false))
      (propagated-inputs
       (list r-biomformat
             r-coin
             r-deseq2
             r-dplyr
             r-genefilter
             r-phyloseq
             r-picante
             r-plotly
             r-reshape2
             r-stringr
             r-vegan))
      (home-page "https://github.com/twbattaglia/btools")
      (synopsis "R functions for microbial diversity analyses")
      (description
       "This package provides an assortment of R functions that is suitable
for all types of microbial diversity analyses.")
      (license license:expat))))

(define-public r-codeandroll2
  (let ((commit "d58e258851a5c0b430e8620d34dbeefb597c548f")
        (revision "1"))
    (package
      (name "r-codeandroll2")
      (version (git-version "2.3.6" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/CodeAndRoll2")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0sy88mfgw6qqhpnlc5020qzr1jllkcrxfhl2lw42bkl5nb56is71"))))
      (properties `((upstream-name . "CodeAndRoll2")))
      (build-system r-build-system)
      (propagated-inputs (list r-colorramps
                               r-dplyr
                               r-gplots
                               r-gtools
                               r-plyr
                               r-rcolorbrewer
                               r-sessioninfo
                               r-sm
                               r-stringendo
                               r-stringr))
      (home-page "https://github.com/vertesy/CodeAndRoll2")
      (synopsis "CodeAndRoll2 for vector, matrix and list manipulations")
      (description
       "CodeAndRoll2 is a set of more than 130 productivity functions.
These functions are used by MarkdownReports, ggExpress, and SeuratUtils.")
      (license license:gpl3))))

(define-public r-conospanel
  (let ((commit "39e76b201a783b4e92fd615010a735a61746fbb9")
        (revision "1"))
    (package
      (name "r-conospanel")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/kharchenkolab/conosPanel")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1zf0aj5d4iaxc3ghvjnaja5qby1avlmljzh94bpyvxbd359z9snn"))))
      (properties `((upstream-name . "conosPanel")))
      (build-system r-build-system)
      (home-page "https://github.com/kharchenkolab/conosPanel")
      (synopsis "Data for the conos package")
      (description "The data within this package is a panel of four samples,
each with 3000 cells.  There are two samples which are bone marrow (BM), and
two samples which are cord blood (CB).")
      (license license:gpl3))))

(define-public r-conqur
  (let ((commit "c7a88794efd4ecfe4d96988dceeec3b410222e48")
        (revision "1"))
    (package
      (name "r-conqur")
      (version (git-version "2.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/wdl2459/ConQuR")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "19a7p2l67mgjy99i5ksjxlhzaqmrnyi1vzvwnhgnx2jrr6crj7rq"))))
      (properties `((upstream-name . "ConQuR")))
      (build-system r-build-system)
      (propagated-inputs (list r-ade4
                               r-ape
                               r-compositions
                               r-cqrreg
                               r-doparallel
                               r-dplyr
                               r-fastdummies
                               r-glmnet
                               r-gplots
                               r-gunifrac
                               r-quantreg
                               r-randomforest
                               r-rocr
                               r-vegan))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/wdl2459/ConQuR")
      (synopsis "Batch effects removal for microbiome data")
      (description
       "This package conducts batch effects removal from a taxa read count
table by a conditional quantile regression method.  The distributional
attributes of microbiome data - zero-inflation and over-dispersion, are
simultaneously considered.")
      (license license:gpl3))))

(define-public r-ewastools
  (let ((commit "f7646cacd73266708479b3fea5d625054d179f95")
        (revision "1"))
    (package
      (name "r-ewastools")
      (version (git-version "1.7.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/hhhh5/ewastools/")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0irarlnxfnasa755adxsn67rxsy01zwhjhw18g4cag08cqiyyw41"))))
      (properties `((upstream-name . "ewastools")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'disable-bad-tests
             (lambda _
               ;; Test data are missing.
               (substitute* "tests/testthat/test-minfi-pipeline.R"
                 ((".*compatibility with minfi.*" m)
                  (string-append m "skip('guix')\n"))))))))
      (propagated-inputs
       (list r-data-table
             r-igraph
             r-illuminaio
             r-mblm
             r-quadprog))
      (native-inputs
       (list r-enmix
             r-illuminahumanmethylationepicanno-ilm10b4-hg19
             r-knitr
             r-minfi
             r-stringi
             r-testthat))
      (home-page "https://github.com/hhhh5/ewastools/")
      (synopsis
       "Quality control toolset for the Illumina Infinium DNA methylation")
      (description
       "This package provides a collection of useful functions for working
with DNA methylation micro-array data.")
      (license license:unlicense))))

(define-public r-hdwgcna
  (let ((commit "950b8c74c5d74b30b4a562820e70ca00ee29e2c6")
        (revision "1"))
    (package
      (name "r-hdwgcna")
      (version (git-version "0.4.00" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/smorabit/hdWGCNA")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0d9hr2ly7k6kq6yp5spjlp6lp9yx593xmggajyzjxlhxi2lbyqgn"))
         (snippet
          '(for-each delete-file
                     (list
                      "docs/deps/bootstrap-5.1.0/bootstrap.bundle.min.js"
                      "docs/deps/bootstrap-5.3.1/bootstrap.bundle.min.js"
                      "docs/deps/bootstrap-toc-1.0.1/bootstrap-toc.min.js"
                      "docs/deps/clipboard.js-2.0.11/clipboard.min.js"
                      "docs/deps/headroom-0.11.0/headroom.min.js"
                      "docs/deps/headroom-0.11.0/jQuery.headroom.min.js"
                      "docs/deps/jquery-3.6.0/jquery-3.6.0.min.js"
                      "docs/deps/search-1.0.0/autocomplete.jquery.min.js"
                      "docs/deps/search-1.0.0/fuse.min.js"
                      "docs/deps/search-1.0.0/mark.min.js")))))
      (properties `((upstream-name . "hdWGCNA")))
      (build-system r-build-system)
      (arguments
       (list
        #:modules
        '((guix build r-build-system)
          (guix build minify-build-system)
          (guix build utils)
          (ice-9 match))
        #:imported-modules
        `(,@%r-build-system-modules (guix build minify-build-system))
        #:phases
        '(modify-phases (@ (guix build r-build-system) %standard-phases)
           (add-after 'unpack 'process-javascript
             (lambda* (#:key inputs #:allow-other-keys)
               (with-directory-excursion "docs/deps/"
                 (for-each (match-lambda
                             ((source . target)
                              (minify source #:target target)))
                           `((,(string-append (assoc-ref inputs "js-bootstrap-5.1")
                                              "/dist/js/bootstrap.bundle.js")
                              . "bootstrap-5.1.0/bootstrap.bundle.min.js")
                             (,(string-append (assoc-ref inputs "js-bootstrap-5.3")
                                              "/dist/js/bootstrap.bundle.js")
                              . "bootstrap-5.3.1/bootstrap.bundle.min.js")
                             (,(search-input-file inputs "/dist/bootstrap-toc.js")
                              . "bootstrap-toc-1.0.1/bootstrap-toc.min.js")
                             (,(search-input-file inputs "/dist/clipboard.js")
                              . "clipboard.js-2.0.11/clipboard.min.js")
                             ("jquery-3.6.0/jquery-3.6.0.js"
                              . "jquery-3.6.0/jquery-3.6.0.min.js")
                             (,(search-input-file inputs "/dist/autocomplete.jquery.js")
                              . "search-1.0.0/autocomplete.jquery.min.js")
                             (,(search-input-file inputs "/dist/fuse.js")
                              . "search-1.0.0/fuse.min.js")
                             (,(search-input-file inputs "/dist/mark.js")
                              . "search-1.0.0/mark.min.js")
                             (,(assoc-ref inputs "js-headroom")
                              . "headroom-0.11.0/headroom.min.js")
                             (,(assoc-ref inputs "js-headroom-jquery")
                              . "docs/deps/headroom-0.11.0/jQuery.headroom.min.js")))))))))
      (propagated-inputs (list r-dplyr
                               r-enrichr
                               r-geneoverlap
                               r-genomicranges
                               r-ggplot2
                               r-ggraph
                               r-ggrepel
                               r-harmony
                               r-igraph
                               r-matrix
                               r-proxy
                               r-seurat
                               r-tester
                               r-tidygraph
                               r-ucell
                               r-wgcna))
      (native-inputs
       `(("esbuild" ,esbuild)
         ("js-autocomplete"
          ,(let ((version "0.38.0"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/algolia/autocomplete")
                     (commit (string-append "v" version))))
               (file-name (git-file-name "js-autocomplete" version))
               (sha256
                (base32
                 "04r1qmd5cxcf8cf471kx51y6wq86rh54y42y7xl51z95ag5csswl")))))
         ("js-bootstrap-5.1"
          ,(let ((version "5.1.0"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/twbs/bootstrap")
                     (commit (string-append "v" version))))
               (file-name (git-file-name "js-bootstrap" version))
               (sha256
                (base32
                 "0wjyi497wy2lbypkxwa72z294zbh827lb2s8pzgi40hvjsw00ns4")))))
         ("js-bootstrap-5.3"
          ,(let ((version "5.3.1"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/twbs/bootstrap")
                     (commit (string-append "v" version))))
               (file-name (git-file-name "js-bootstrap" version))
               (sha256
                (base32
                 "045majqayd4wrpq3r3fisi56grwizykwj5705jlzsg7bwln8v3qr")))))
         ("js-bootstrap-toc"
          ,(let ((version "1.0.1"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/afeld/bootstrap-toc")
                     (commit (string-append "v" version))))
               (file-name (git-file-name "js-bootstrap-toc" version))
               (sha256
                (base32
                 "1d4rqvbkwmmld0d8nz2is1rwc55rlb4b0v2anylxj63335cs0mk0")))))
         ("js-clipboard"
          ,(let ((version "2.0.11"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/zenorocha/clipboard.js")
                     (commit (string-append "v" version))))
               (file-name (git-file-name "js-clipboard" version))
               (sha256
                (base32
                 "0nn9cry7ddf47vkl3vnpc4z86da46wq7m3qf3awcs05197bjn1vj")))))
         ("js-fuse"
          ,(let ((version "6.4.6"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/krisk/Fuse")
                     (commit (string-append "v" version))))
               (file-name (git-file-name "js-fuse" version))
               (sha256
                (base32
                 "1k0l2vxw4nsmwrz48kmgv41ymphrdwbdbckiv9w011jwbnfsqvn5")))))
         ("js-mark"
          ,(let ((version "8.11.1"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/julkue/mark.js")
                     (commit version)))
               (file-name (git-file-name "js-mark" version))
               (sha256
                (base32
                 "108cw4732wjlwwjwpc86fqlr8yfrlagqdn7126qxa70c0ldfma93")))))
         ("js-headroom"
          ,(let ((version "0.11.0"))
             (origin
               (method url-fetch)
               (uri (string-append "https://unpkg.com/headroom.js@"
                                   version "/dist/headroom.js"))
               (file-name (string-append "js-headroom-" version ".js"))
               (sha256
                (base32
                 "0d76fpajxzwsj4847zgrv05daq8ri8hcww5dcldq1q2iyyxsmrva")))))
         ("js-headroom-jquery"
          ,(let ((version "0.11.0"))
             (origin
               (method url-fetch)
               (uri (string-append "https://unpkg.com/headroom.js@"
                                   version "/dist/jQuery.headroom.js"))
               (file-name (string-append "js-headroom-jquery-" version ".js"))
               (sha256
                (base32
                 "1jz1ampd9k8p32m6bid4yrr66bkngd8blj2q0774hl91rszhjqrv")))))
         ("r-devtools" ,r-devtools)
         ("r-testthat" ,r-testthat)))
      (home-page "https://github.com/smorabit/hdWGCNA")
      (synopsis "Weighted gene co-expression network analysis")
      (description
       "@code{hdWGCNA} is an R package for performing weighted gene
co-expression network analysis in high dimensional -omics such as single-cell
RNA-seq or spatial transcriptomics.")
      (license license:gpl3))))

(define-public r-netid
  (let ((commit "6ad1ffdd64a6584cc1d392524dad8e248d4590a8")
        (revision "1"))
    (package
      (name "r-netid")
      (version (git-version "0.1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/WWXkenmo/NetID_package")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1blm5ljql91xmbyxxwm3vw8iz8d4ir9gm7wfnapqhfdbjvg4l21q"))))
      (properties `((upstream-name . "NetID")))
      (build-system r-build-system)
      (propagated-inputs (list r-doparallel
                               r-dorng
                               r-glmnet
                               r-hmisc
                               r-igraph
                               r-irlba
                               r-lmtest
                               r-matrix
                               r-mclust
                               r-pracma
                               r-raceid
                               r-rarpack
                               r-reticulate
                               r-robustrankaggreg
                               r-rsvd
                               r-seurat

                               ;; Needed for use with reticulate
                               python-anndata
                               python-geosketch
                               python-scanpy
                               scvelo))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/WWXkenmo/NetID_package")
      (synopsis
       "Scalable method to infer fate-specific networks from single-cell data")
      (description
       "This package provides a method to sample cells from single-cell data.
It also generates an aggregate profile on a pruned K-Nearest Neighbor graph.
This approach leads to an improved gene expression profile for quantifying
gene regulations.")
      (license license:expat))))

(define-public r-p2data
  (let ((commit "7d4c0e17d7899f9d9b08ab2bf455abe150912f4c")
        (revision "1"))
    (package
      (name "r-p2data")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/kharchenkolab/p2data")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1hadrldldxvhqs43aqs3c88bqfgql3wcfkbll3jz7fh6z7p3x324"))))
      (properties `((upstream-name . "p2data")))
      (build-system r-build-system)
      (home-page "https://github.com/kharchenkolab/p2data")
      (synopsis "Data for pagoda2")
      (description "This package contains data used by pagoda2.  The data
within this package are the 3000 bone marrow cells used for vignettes.")
      (license license:gpl3))))

(define-public r-pizzarr
  (let ((commit "7b3fd727a4ce7cb0fe8e443b971c5f63e18df991")
        (revision "1"))
    (package
      (name "r-pizzarr")
      (version (git-version "0.1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/keller-mark/pizzarr")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0xwb7cl52nzwgh0xiz8cda27qy5q45cqna5k40fngwj4mrbfm383"))))
      (properties `((upstream-name . "pizzarr")))
      (build-system r-build-system)
      ;; Some tests require Internet access.
      (arguments (list #:tests? #false))
      (propagated-inputs (list r-jsonlite r-memoise r-qs r-r6 r-stringr))
      (native-inputs (list r-pbapply r-testthat r-vcr))
      (home-page "https://github.com/keller-mark/pizzarr")
      (synopsis "Slice into Zarr arrays in R")
      (description
       "This package provides an implementation of chunked, compressed,
N-dimensional arrays for R, Zarr specification version 2 (2024)
<doi:10.5281/zenodo.11320255>.")
      (license license:expat))))

(define-public r-rhtslib12
  (let ((commit "ee186daf04876969c7f31c16a0e0fda8e7c16a30")
        (revision "1"))
    (package
      (name "r-rhtslib12")
      (version (git-version "1.23.2" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/haizi-zh/Rhtslib12")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0a3kkp0phi2fq6ip8p9vfj3axn7l15f2mb51a6v3ai4nlkhqqawj"))))
      (properties `((upstream-name . "Rhtslib12")))
      (build-system r-build-system)
      (propagated-inputs (list curl zlib r-zlibbioc))
      (native-inputs (list pkg-config r-knitr))
      (home-page "https://github.com/haizi-zh/Rhtslib12")
      (synopsis "HTSlib high-throughput sequencing library as an R package")
      (description
       "This package provides version 1.12 of the HTSlib C library for
high-throughput sequence analysis.  The package is primarily useful to
developers of other R packages who wish to make use of HTSlib.")
      (license license:lgpl2.0+))))

(define-public r-scdesign2
  (let ((commit "554f2c4b1a7ee6cc04969a287df9b3b77d7bb2fe")
        (revision "1"))
    (package
      (name "r-scdesign2")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/JSB-UCLA/scDesign2")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "09zkz6qdcx54092ab6761ja3pclax9xzana8dk8qvxrdbjshb8p7"))))
      (properties `((upstream-name . "scDesign2")))
      (build-system r-build-system)
      (propagated-inputs (list r-mass r-pscl))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/JSB-UCLA/scDesign2")
      (synopsis "Statistical simulator for scRNA-seq with gene correlation")
      (description
       "This package offers a flexible statistical simulator for scRNA-seq data.
It can generate data that captures gene correlation.  Additionally, it allows
for varying the number of cells and sequencing depth.")
      (license license:expat))))

(define-public r-scenic
  (let ((commit "cedf8490a634da550cea2c831544e5f7f14467d2")
        (revision "1"))
    (package
      (name "r-scenic")
      (version (git-version "1.3.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/aertslab/SCENIC")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "17ai0q260hdqbvm1km1s5dw93pgz4f546ycfii57jyy9m9jka7r0"))))
      (properties `((upstream-name . "SCENIC")))
      (build-system r-build-system)
      (propagated-inputs (list r-aucell
                               r-data-table
                               r-dynamictreecut
                               r-genie3
                               r-ggrepel
                               r-mixtools
                               r-nmf
                               r-rcistarget
                               r-rtsne))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/aertslab/SCENIC")
      (synopsis
       "SCENIC (Single Cell rEgulatory Network Inference and Clustering)")
      (description "SCENIC (Single-cell regulatory network inference and
clustering) is an R package to infer Gene Regulatory Networks and cell types
from single-cell RNA-seq data.")
      ;; As of commit cedf8490a634da550cea2c831544e5f7f14467d2 the license is
      ;; GPLv3.
      (license license:gpl3))))

(define-public r-scent
  (let ((commit "f01f18ac30f8a9bcf85b738c6c7815017e2c8ee5")
        (revision "1"))
    (package
      (name "r-scent")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/immunogenomics/SCENT")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "18krksy8ac7yy8hghzaxscj91c61j195yg7j60zswdq97islvfzi"))))
      (properties `((upstream-name . "SCENT")))
      (build-system r-build-system)
      (inputs (list bedtools))
      (propagated-inputs (list r-boot
                               r-data-table
                               r-hmisc
                               r-lme4
                               r-mass
                               r-matrix
                               r-r-utils
                               r-stringr))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/immunogenomics/SCENT")
      (synopsis
       "Single-Cell enhancer target gene mapping for multimodal single-cell data")
      (description
       "This package contains functions for the SCENT algorithm.
SCENT uses single-cell multimodal data and links ATAC-seq peaks
to their target genes by modeling association between chromatin
accessibility and gene expression across individual single cells.")
      (license license:expat))))

(define-public r-saige
  (let ((commit "44c8dd4d7641fcb4b591a49e10e7ecf501943202")
        (revision "2"))
    (package
      (name "r-saige")
      (version (git-version "1.3.4" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/saigegit/SAIGE")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0n5gaah0fdhd6affmaml3kfmk7knxkr26knyg6hjmwx836fbbyif"))))
      (properties `((upstream-name . "SAIGE")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'do-not-download-packages
             (lambda _
               ;; Pretend to be a Conda build to avoid having to install
               ;; things with pip and cget.
               (setenv "CONDA_BUILD" "1")))
           (add-after 'unpack 'add-missing-include
             (lambda _
               (substitute* "src/SAIGE_fitGLMM_fast.cpp"
                 (("#include <omp.h>")
                  "#include <omp.h>\n#include <tbb/concurrent_vector.h>"))))
           (add-after 'unpack 'link-with-openblas
             (lambda* (#:key inputs #:allow-other-keys)
               (substitute* "src/Makevars"
                 (("-llapack")
                  (search-input-file inputs
                                     "/lib/libopenblas.so")))))
           (add-after 'unpack 'link-with-plink2
             (lambda* (#:key inputs #:allow-other-keys)
               (substitute* "src/Makevars"
                 (("-l:plink2_includes.a")
                  (search-input-file inputs "/lib/libplink2.so"))))))))
      (inputs (list openblas plink-ng savvy superlu zlib zstd))
      (propagated-inputs (list r-bh
                               r-data-table
                               r-dplyr
                               r-lintools
                               r-matrix
                               r-metaskat
                               r-optparse
                               r-qlcmatrix
                               r-rcpp
                               r-rcpparmadillo
                               r-rcppeigen
                               r-rcppparallel
                               r-rhpcblasctl
                               r-rsqlite
                               r-skat
                               r-spatest
                               r-survival))
      (home-page "https://github.com/saigegit/SAIGE")
      (synopsis "Genome-wide association tests in large-scale data sets")
      (description "SAIGE is a package for efficiently controlling for
case-control imbalance and sample relatedness in single-variant assoc
tests (SAIGE) and controlling for sample relatedness in region-based assoc
tests in large cohorts and biobanks (SAIGE-GENE+).")
      (license license:gpl2+))))

(define-public r-sigfit
  (package
    (name "r-sigfit")
    (version "2.2.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/kgori/sigfit")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "16r6m8p90rnb4hxl86fz3kbals3626232i8fj0zhhd23p89z3w4v"))))
    (properties `((upstream-name . "sigfit")))
    (build-system r-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'gcc-14-compatibility
            (lambda _
              (substitute* "configure"
                ;; Modifying src/Makevars directly is overruled.
                (("config\\(\\)\"")
                 "config()\"
echo \"PKG_CXXFLAGS+=-g -O2 -Wno-error=changes-meaning\" >> src/Makevars")))))))
    (native-inputs (list r-codetools))
    (propagated-inputs (list r-rcpp
                             r-rstan
                             r-rstantools
                             r-coda
                             r-clue
                             r-knitr
                             r-rmarkdown
                             r-bh
                             r-rcppeigen
                             r-stanheaders))
    (home-page "https://github.com/kgori/sigfit")
    (synopsis "Flexible Bayesian inference of mutational signatures")
    (description
     "This R package lets you estimate signatures of mutational processes and
their activities on mutation count data.  Starting from a set of
@dfn{single-nucleotide variants} (SNVs), it allows both estimation of the
exposure of samples to predefined mutational signatures (including whether the
signatures are present at all), and identification of signatures de novo from
the mutation counts.")
    (license license:gpl3)))

(define-public r-singlet
  (let ((commit "765a6c45081807a1522f0e8983e2417822a36f36")
        (revision "1"))
    (package
      (name "r-singlet")
      (version (git-version "0.99.26" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/zdebruine/singlet")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "040v8wzl9qr8ribr6qss61fz4698d14cqs8nxbc8hqwiqlpy3vs4"))))
      (properties `((upstream-name . "singlet")))
      (build-system r-build-system)
      (propagated-inputs (list r-dplyr
                               r-fgsea
                               r-ggplot2
                               r-knitr
                               r-limma
                               r-matrix
                               r-msigdbr
                               r-rcpp
                               r-rcppml/devel
                               r-reshape2
                               r-scuttle
                               r-seurat))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/zdebruine/singlet")
      (synopsis "Non-negative Matrix Factorization for single-cell analysis")
      (description
       "This is a package for fast @dfn{Non-negative Matrix
Factorization} (NMF) with automatic rank-determination for dimension reduction
of single-cell data using Seurat, RcppML nmf, SingleCellExperiments and
similar.")
      (license license:gpl2+))))

(define-public r-stacas
  (package
    (name "r-stacas")
    (version "2.2.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/carmonalab/STACAS")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "13i0h5i6vlbrb8ndq9gr81560z9d74b2c7m3rjfzls01irjza9hm"))))
    (properties `((upstream-name . "STACAS")))
    (build-system r-build-system)
    (propagated-inputs
     (list r-biocneighbors
           r-biocparallel
           r-ggplot2
           r-ggridges
           r-pbapply
           r-r-utils
           r-seurat))
    (home-page "https://github.com/carmonalab/STACAS")
    (synopsis "Sub-type anchoring correction for alignment in Seurat")
    (description
     "This package implements methods for batch correction and integration of
scRNA-seq datasets, based on the Seurat anchor-based integration framework.
In particular, STACAS is optimized for the integration of heterogeneous
datasets with only limited overlap between cell sub-types (e.g. TIL sets of
CD8 from tumor with CD8/CD4 T cells from lymphnode), for which the default
Seurat alignment methods would tend to over-correct biological differences.
The 2.0 version of the package allows the users to incorporate explicit
information about cell-types in order to assist the integration process.")
    (license license:gpl3)))

(define-public r-stringendo
  (let ((commit "15594b1bba11048a812874bafec0eea1dcc8618a")
        (revision "1"))
    (package
      (name "r-stringendo")
      (version (git-version "0.6.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/Stringendo")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "15ij4zf2j9c8m9n4bqhmxkchjh2bhddwjfxngfpwv7c5wjqyi6ir"))))
      (properties `((upstream-name . "Stringendo")))
      (build-system r-build-system)
      (propagated-inputs (list r-clipr))
      (home-page "https://github.com/vertesy/Stringendo")
      (synopsis "Stringendo is a string parsing library")
      (description
       "This package provides string parsing functionalities for generating
plotnames, filenames and paths.")
      (license license:gpl3))))

(define-public r-readwriter
  (let ((commit "91373c44641014a1ce8e1c3e928747608aae8f54")
        (revision "1"))
    (package
      (name "r-readwriter")
      (version (git-version "1.5.3" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/ReadWriter")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "156kvmplrip0w1zhs9yl5r0ayjipa0blhy614l65hbsjn1lwbskr"))))
      (properties `((upstream-name . "ReadWriter")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-gtools r-openxlsx r-readr r-stringendo))
      (home-page "https://github.com/vertesy/ReadWriter")
      (synopsis "Functions to read and write files conveniently")
      (description
       "ReadWriter is a set of R functions to read and write files
conveniently.")
      (license license:gpl3))))

(define-public r-streamgraph
  (let ((commit "76f7173ec89d456ace5943a512e20b1f6810bbcb")
        (revision "1"))
    (package
      (name "r-streamgraph")
      (version (git-version "0.9.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/hrbrmstr/streamgraph")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "010rhnby5a9dg08jvlkr65b3p9iipdxi2f5m1k6j53s80p25yvig"))))
      (properties `((upstream-name . "streamgraph")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-dplyr
             r-htmltools
             r-htmlwidgets
             r-magrittr
             r-tidyr
             r-xts))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/hrbrmstr/streamgraph")
      (synopsis "Htmlwidget for building streamgraph visualizations")
      (description
       "A streamgraph is a type of stacked area chart.  It represents the
evolution of a numeric variable for several groups.  Areas are usually
displayed around a central axis, and edges are rounded to give a flowing
shape.  This package provides an @code{htmlwidget} for building streamgraph
visualizations.")
      (license license:expat))))

(define-public r-vitesscer
  (let ((commit "0096880c1b7b7afb42a26277e34508b4cce815bd")
        (revision "1"))
    (package
      (name "r-vitesscer")
      (version (git-version "0.99.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/vitessce/vitessceR")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0axragpfl9xz7a10wx7n2j88lcv64kzi4ixkr6s69x3xa2fmlvay"))))
      (properties `((upstream-name . "vitessceR")))
      (build-system r-build-system)
      (propagated-inputs (list r-delayedarray
                               r-future
                               r-htmlwidgets
                               r-httpuv
                               r-jsonlite
                               r-matrix
                               r-mime
                               r-plumber
                               r-r6
                               r-s4vectors
                               r-stringi
                               r-stringr))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/vitessce/vitessceR")
      (synopsis "Create interactive Single-cell visualizations")
      (description
       "This package provides an R API and @code{htmlwidget} facilitating
interactive visualization of spatial single-cell data with Vitessce.  The R
API contains classes and functions for loading single-cell data stored in
compatible on-disk formats.  The @code{htmlwidget} is a wrapper around the
Vitessce JavaScript library and can be used in the Viewer tab of RStudio or
Shiny apps.")
      (license license:expat))))

(define-public r-wasabi
  (let ((commit "8c33cabde8d18c2657cd6e38e7cb834f87cf9846")
        (revision "1"))
    (package
      (name "r-wasabi")
      (version (git-version "1.0.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/COMBINE-lab/wasabi")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0rpdj6n4cnx8n2zl60dzgl638474sg49dknwi9x3qb4g56dpphfa"))))
      (properties `((upstream-name . "wasabi")))
      (build-system r-build-system)
      (propagated-inputs (list r-data-table r-rhdf5 r-rjson))
      (home-page "https://github.com/COMBINE-lab/wasabi")
      (synopsis "Use Sailfish and Salmon with Sleuth")
      (description
       "This package converts the output of the Sailfish and Salmon RNA-seq
quantification tools so that it can be used with the Sleuth differential
analysis package.")
      (license license:bsd-3))))

(define-public pbgzip
  (let ((commit "2b09f97b5f20b6d83c63a5c6b408d152e3982974"))
    (package
      (name "pbgzip")
      (version (git-version "0.0.0" "0" commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/nh13/pbgzip")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1mlmq0v96irbz71bgw5zcc43g1x32zwnxx21a5p1f1ch4cikw1yd"))))
      (build-system gnu-build-system)
      (native-inputs
       (list autoconf automake))
      (inputs
       (list zlib))
      (home-page "https://github.com/nh13/pbgzip")
      (synopsis "Parallel Block GZIP")
      (description "This package implements parallel block gzip.  For many
formats, in particular genomics data formats, data are compressed in
fixed-length blocks such that they can be easily indexed based on a (genomic)
coordinate order, since typically each block is sorted according to this order.
This allows for each block to be individually compressed (deflated), or more
importantly, decompressed (inflated), with the latter enabling random retrieval
of data in large files (gigabytes to terabytes).  @code{pbgzip} is not limited
to any particular format, but certain features are tailored to genomics data
formats when enabled.  Parallel decompression is somewhat faster, but the true
speedup comes during compression.")
      (license license:expat))))

(define-public randfold
  (package
    (name "randfold")
    (version "2.0.1")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "http://bioinformatics.psb.ugent.be/"
                           "supplementary_data/erbon/nov2003/downloads/"
                           "randfold-" version ".tar.gz"))
       (sha256
        (base32
         "0gqixl4ncaibrxmn25d6lm2hrw4ml2fj13nrc9q1kilsxdfi91mj"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;no tests provided
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (replace 'install
            (lambda _
              (install-file "randfold"
                            (string-append #$output "/bin")))))))
    (inputs (list eddylab-squid))
    (home-page
     "http://bioinformatics.psb.ugent.be/supplementary_data/erbon/nov2003/")
    (synopsis "Minimum free energy of folding randomization test software")
    (description "randfold computes the probability that, for a given
sequence, the @dfn{Minimum Free Energy} (MFE) of the secondary structure is
different from MFE computed with random sequences.")
    (license license:gpl2)))

(define-public ribotaper
  (package
    (name "ribotaper")
    (version "1.3.1")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://ohlerlab.mdc-berlin.de/"
                                  "files/RiboTaper/RiboTaper_Version_"
                                  version ".tar.gz"))
              (sha256
               (base32
                "0ykjbps1y3z3085q94npw8i9x5gldc6shy8vlc08v76zljsm07hv"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'install 'wrap-executables
            (lambda _
              (for-each
               (lambda (script)
                 (wrap-program (string-append #$output "/bin/" script)
                   `("R_LIBS_SITE" ":" = (,(getenv "R_LIBS_SITE")))))
               '("create_annotations_files.bash"
                 "create_metaplots.bash"
                 "Ribotaper_ORF_find.sh"
                 "Ribotaper.sh")))))))
    (inputs
     (list bash-minimal
           bedtools-2.18
           samtools-0.1
           r-minimal
           r-foreach
           r-xnomial
           r-domc
           r-multitaper
           r-seqinr))
    (home-page "https://ohlerlab.mdc-berlin.de/software/RiboTaper_126/")
    (synopsis "Define translated ORFs using ribosome profiling data")
    (description
     "Ribotaper is a method for defining translated @dfn{open reading
frames} (ORFs) using ribosome profiling (ribo-seq) data.  This package
provides the Ribotaper pipeline.")
    (license license:gpl3+)))

(define-public bioawk
  (package
    (name "bioawk")
    (version "1.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/lh3/bioawk")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pxc3zdnirxbf9a0az698hd8xdik7qkhypm7v6hn922x8y9qmspm"))))
    (build-system gnu-build-system)
    (inputs (list zlib))
    (native-inputs (list bison))
    (arguments
     (list
      #:tests? #f ; There are no tests to run.
      ;; Bison must generate files, before other targets can build.
      #:parallel-build? #f
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)           ; There is no configure phase.
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin"))
                    (man (string-append #$output "/share/man/man1")))
                (mkdir-p man)
                (copy-file "awk.1" (string-append man "/bioawk.1"))
                (install-file "bioawk" bin)))))))
    (home-page "https://github.com/lh3/bioawk")
    (synopsis "AWK with bioinformatics extensions")
    (description "Bioawk is an extension to Brian Kernighan's awk, adding the
support of several common biological data formats, including optionally gzip'ed
BED, GFF, SAM, VCF, FASTA/Q and TAB-delimited formats with column names.  It
also adds a few built-in functions and a command line option to use TAB as the
input/output delimiter.  When the new functionality is not used, bioawk is
intended to behave exactly the same as the original BWK awk.")
    (license license:x11)))

(define-public python-bcbio-gff
  (package
    (name "python-bcbio-gff")
    ;; python-bcbio-gff can only be refreshed manually, because guix refresh
    ;; does not understand the tags on the github repository.
    (version "0.7.1")
    (source (origin
              ;; No tests in PyPI package.
              (method git-fetch)
              (uri (git-reference
               (url "https://github.com/chapmanb/bcbb")
               (commit (string-append "bcbio-gff-v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0144xxzibq4mrg8a1w2scs120rd9svq07hm5ccs91n3a4nvwjfsd"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'enter-directory
            (lambda _ (chdir "gff"))))))
    (native-inputs
     (list python-setuptools python-pytest))
    (propagated-inputs
     (list python-biopython
           python-six))
    (home-page "https://github.com/chapmanb/bcbb/tree/master/gff")
    (synopsis "Read and write GFF files with Biopython integration")
    (description
     "This package lets you read and write files in Generic Feature
Format (GFF) with Biopython integration.")
    (properties
     '((upstream-name . "bcbio-gff")))
    (license (license:non-copyleft "http://www.biopython.org/DIST/LICENSE"))))

(define-public python-bcbio-gff-for-python-cmseq
  (hidden-package
   (package
     (inherit python-bcbio-gff)
     (name "python-bcbio-gff")
     (version "0.6.9")
     (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/chapmanb/bcbb")
               (commit (string-append "bcbio-gff-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "131hiir94jkm9jj2wfpybwndgzn8k0zc1ji1qjn5cz7w48x3ri13"))))
     (propagated-inputs
      (modify-inputs (package-propagated-inputs python-bcbio-gff)
        (replace "python-biopython" python-biopython-1.73))))))

(define bed-sample-files
  (let* ((name "bed-sample-files")
         (commit "a06dc0450e484090f15656ffd5d317813a5e1e01")
         (revision "0")
         (version (git-version "0.0.0" revision commit)))
    (origin
      (method git-fetch)
      (uri (git-reference
            (url "https://github.com/fastlmm/bed-sample-files")
            (commit commit)))
      (file-name (git-file-name name version))
      (sha256
       (base32 "1ldr2lvgbcykxa9i2s2298mhfh0sz96aaxs5dx217aipa9vsrjwk")))))

(define-public python-bed-reader
  (package
    (name "python-bed-reader")
    (version "1.0.6")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "bed_reader" version))
       (sha256
        (base32 "1c8ibwvz3b069w7ffh9aasz16lfkmx4z0249c2v909a21mrkkd6n"))
       (modules '((guix build utils)))
       ;; Bundled unused javascript & co.
       (snippet #~(delete-file-recursively "_static"))
       (patches
        (search-patches "python-bed-reader-use-store-samples.patch"))))
    (build-system cargo-build-system)
    (arguments
     (list
      #:install-source? #false
      #:features '(list "extension-module")
      #:cargo-test-flags
      '(list "--features=extension-module"
             ;; Skip doc tests.
             "--lib" "--bins" "--tests" "--"
             ;; This test is the only one not matched by our regexp.
             "--skip=http_one"
             ;; These test require a 84 GB file.
             "--skip=http_two"
             "--skip=http_cloud_urls_md_3")
      #:imported-modules
      (append %cargo-build-system-modules
              %pyproject-build-system-modules)
      #:modules
      '((guix build cargo-build-system)
        ((guix build pyproject-build-system) #:prefix py:)
        (guix build utils)
        (ice-9 match)
        (ice-9 rdelim))
      #:phases
      (with-extensions (list (pyproject-guile-json))
      #~(modify-phases %standard-phases
          (add-after 'configure 'set-data-path
            (lambda _
              ;; This var is still necessary despite the patch-data-path phase.
              ;; Otherwise more tests fail with a read-only filesystem error.
              (setenv "BED_READER_DATA_DIR" #+bed-sample-files)))
          (add-after 'unpack 'patch-data-path
            (lambda _
              ;; If BED_READER_DATA_DIR is unset, default to bed-sample-files.
              (substitute* "bed_reader/_sample_data.py"
                (("os\\.environ\\.get\\(\"BED_READER_DATA_DIR\"" all)
                 (format #f "~a, ~s" all #+bed-sample-files)))
              ;; XXX: More work is necessary to use another
              ;; version of sample files with BED_READER_DATA_DIR
              ;; Currently, only the hardcoded Guix version is working.
              (substitute* '("bed_reader/tests/test_open_bed_cloud.py"
                             "src/bed_cloud.rs"
                             "src/lib.rs"
                             "src/supplemental_documents/cloud_urls_etc.md"
                             "tests/tests_api_cloud.rs")
                (("\
https://raw\\.githubusercontent\\.com/fastlmm/bed-sample-files/main")
                 (string-append "file://" #+bed-sample-files)))
              (substitute* "src/tests.rs"
                (("bed_reader/tests/data")
                 #+bed-sample-files))))
          (add-after 'install 'prepare-python-module
            (lambda _
              ;; We don't use maturin. Conveniently, what we want to drop
              ;; from pyproject.toml is at the end of the file.
              (rename-file "pyproject.toml" "pyproject.toml.bak")
              (call-with-input-file "pyproject.toml.bak"
                (lambda (in)
                  (call-with-output-file "pyproject.toml"
                    (lambda (out)
                      (let loop ()
                        (match (read-line in)
                          ((? eof-object? eof)
                           eof)
                          ("[build-system]"
                           (and (format out "\
[build-system]
build-backend = 'setuptools.build_meta'
requires = ['setuptools']

[tool.setuptools.packages.find]
where = [\".\"]
exclude = [\"src\", \"docs\", \"tests\", \"Cargo.toml\"]
")))
                          ("samples = [\"pooch>=1.5.0\"]"
                           (and (format out "samples = []~%")
                                (loop)))
                          ("[project]"
                           (and (format out "\
[project]
version = ~s
" #$version)
                                (loop)))
                          (line
                           (and (format out "~a~%" line)
                                (loop)))))))))))
          (add-after 'prepare-python-module 'enable-bytecode-determinism
            (assoc-ref py:%standard-phases 'enable-bytecode-determinism))
          (add-after 'enable-bytecode-determinism 'build-python-module
            (assoc-ref py:%standard-phases 'build))
          (add-after 'build-python-module 'install-python-module
            (assoc-ref py:%standard-phases 'install))
          (add-after 'install-python-module 'install-python-library
            (lambda* (#:key inputs outputs #:allow-other-keys)
              (let ((site (py:site-packages inputs outputs)))
                (mkdir-p (string-append site "/bed_reader"))
                (copy-file "target/release/libbed_reader.so"
                           (string-append site "/bed_reader/bed_reader.so")))))
          (add-after 'install-python-library 'add-install-to-pythonpath
            (assoc-ref py:%standard-phases 'add-install-to-pythonpath))
          (add-after 'add-install-to-pythonpath 'check-python
            (lambda* (#:key inputs outputs tests? #:allow-other-keys)
              (when tests?
                (let ((site (py:site-packages inputs outputs))
                      (data-dir "bed_reader/tests/data"))
                  (symlink (canonicalize-path data-dir)
                           (string-append site "/" data-dir))
                  (invoke "pytest" "-v" #$output
                          ;; These test require a 84 GB file.
                          "-k" (string-join
                                (list "not test_http_two"
                                      "test_http_cloud_urls_rst_3"
                                      "test_http_cloud_urls_rst_4"
                                      ;; XXX: python-pooch dependency removed
                                      "test_optional_dependencies")
                                " and not "))
                  (delete-file-recursively
                   (string-append site "/bed_reader/tests"))
                  (delete-file-recursively
                   (string-append #$output "/.pytest_cache"))))))))))
    (native-inputs (list python-pytest
                         python-pytest-cov
                         python-pytest-datadir
                         python-pytest-doctestplus
                         python-recommonmark
                         python-sphinx))
    (inputs (cons python-wrapper (cargo-inputs 'python-bed-reader)))
    (propagated-inputs (list python-numpy python-pandas python-scipy))
    (home-page "https://fastlmm.github.io/")
    (synopsis "Read and write the PLINK BED format, simply and efficiently")
    (description
     "This package lets you read and write the PLINK BED format, simply and
efficiently.")
    (license license:asl2.0)))

;; Note: the name on PyPi is "biofluff".
(define-public python-biofluff
  (package
    (name "python-biofluff")
    (version "3.0.4")
    ;; PyPi tarball does not contain test data.
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/simonvh/fluff")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12yvhgp72s2ygf3h07rrc852zd6q8swc41hm28mcczpsyprggxyz"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; Theses tests require internet access
      '(list "--ignore=tests/test_mirror.py"
             "-k" "not test_plots_big")
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'matplotlib-compatibility
           (lambda _
             (substitute* "fluff/plot.py"
               (("beginarrow=False, endarrow=True,") "")))))))
    (propagated-inputs
     (list htseq
           python-matplotlib
           python-numpy
           python-palettable
           python-pybedtools
           python-pybigwig
           python-pysam
           python-scikit-learn
           python-scipy))
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/simonvh/fluff/")
    (synopsis "Analysis and visualization of high-throughput sequencing data")
    (description
     "Fluff is a Python package that contains several scripts to produce
pretty, publication-quality figures for next-generation sequencing
experiments.")
    (license license:expat)))

(define-public python-bulkvis
  (let ((commit "00a82a90c7e748a34af896e779d27e78a2c82b5e")
        (revision "2"))
    (package
      (name "python-bulkvis")
      (version (git-version "2.0.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/LooseLab/bulkVis")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "02blai158xyyqcg0ljzkmfa6ci05m4awrl4njvp9nwfp717xq8n0"))
                (modules '((guix build utils)))
                (snippet
                 '(substitute* '("requirements.txt"
                                 "setup.py")
                    (("tqdm~=4.46.1") "tqdm")
                    (("tornado~=6.0.4") "tornado")
                    (("pandas~=1.0.5") "pandas")
                    (("h5py~=2.10.0") "h5py")
                    ;; See below for com
                    (("bokeh~=2.1.0") "bokeh")))))
      (build-system pyproject-build-system)
      (arguments
       (list #:tests? #f                ;There are no tests
             #:phases
             '(modify-phases %standard-phases
                ;; See https://github.com/LooseLab/bulkvis/issues/58
                (add-after 'unpack 'bokeh-compatibility
                  (lambda _
                    (substitute* "bulkvis/bulkvis.py"
                      (("import importlib" m)
                       (string-append m "
from bokeh.command.subcommand import Argument
from bokeh.util.dataclasses import entries\n"))
                      (("( *)_parser.add_argument" m indent)
                       (string-append
                        (string-join (list "if isinstance(opts, Argument):\n"
                                           "  opts = dict(entries(opts))\n")
                                     indent 'prefix)
                        m))))))))
      (propagated-inputs (list python-bokeh
                               python-dill
                               python-h5py
                               python-joblib
                               python-matplotlib
                               python-numpy
                               python-pandas
                               python-plotly
                               python-readpaf
                               python-scikit-learn
                               python-scikit-image
                               python-scipy
                               python-seaborn
                               python-tornado-6
                               python-tqdm
                               python-umap-learn))
      (native-inputs (list python-pytest python-setuptools python-wheel))
      (home-page "https://github.com/LooseLab/bulkVis")
      (synopsis "Interactive visualization of bulk RNA-seq data")
      (description
       "This is a Python package for the interactive visualization of bulk
RNA-seq data.  It provides a range of plotting functions and interactive tools
to explore and analyze bulk RNA-seq data.")
      (license license:expat))))

(define-public python-cell2cell
  (package
    (name "python-cell2cell")
    (version "0.7.4")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/earmingol/cell2cell")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "02cqc5rm0qkm0np1k7bim1w7f5qjnwf1jcm5albd9cpvfs4bwgdr"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  ;; We remove the dependency on statannotations because it
                  ;; will not work with the current version of seaborn.  See
                  ;; https://github.com/trevismd/statannotations/issues/122
                  (substitute* "cell2cell/plotting/factor_plot.py"
                    (("from statannotations.Annotator import Annotator")
                     "")
                    (("if statistical_test is not None")
                     "if False"))
                  (substitute* "setup.py"
                    (("'statannotations',") "")
                    ;; We provide version 1.0.4, which should be fine.
                    (("'gseapy == 1.0.3'") "'gseapy'")
                    ;; We provide version 0.9.0, which should be fine.
                    (("'tensorly == 0.8.1'") "'tensorly'")
                    ;; Using matplotlib 3.5.2 leads to this bug:
                    ;; https://github.com/earmingol/cell2cell/issues/19 but we
                    ;; can't package a different minor version of matplotlib
                    ;; and limit its use to just this package.
                    (("matplotlib >= 3.2.0,<=3.5.1") ""))))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f                  ;There are no tests
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable dir to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _ (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs
     (list python-gseapy
           python-kneed
           python-matplotlib
           python-networkx
           python-numpy
           python-openpyxl
           python-pandas
           python-scikit-learn
           python-scipy
           python-seaborn
           python-statsmodels
           python-scanpy
           python-seaborn
           python-tensorly
           python-tqdm
           python-umap-learn
           python-xlrd))
    (native-inputs (list python-wheel))
    (home-page "https://github.com/earmingol/cell2cell")
    (synopsis "Python library for cell communication analysis")
    (description
     "Cell2cell is a Python library for cell communication analysis.
This is a method to calculate, visualize and analyze communication between
cell types.  Cell2cell is suitable for single-cell RNA sequencing
(scRNA-seq) data.")
    (license license:bsd-3)))

(define-public python-cellbender
  (package
    (name "python-cellbender")
    (version "0.2.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/broadinstitute/CellBender")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0h9d9pznffdbya631hkk7b7jwjrgx5saqssar1d42qbyvdji3hgy"))))
    (build-system pyproject-build-system)
    (arguments
     (list #:tests? #false)) ;there are none
    (propagated-inputs
     (list python-anndata
           python-matplotlib
           python-numpy
           python-pandas
           python-pyro-ppl
           python-scikit-learn
           python-scipy
           python-sphinx
           python-sphinx-argparse
           python-sphinx-autodoc-typehints
           python-sphinx-rtd-theme
           python-sphinxcontrib-programoutput
           python-tables))
    (home-page "https://cellbender.rtfd.io/")
    (synopsis "Eliminate technical artifacts from single-cell RNA-seq data")
    (description
     "CellBender is a software package for eliminating technical artifacts
from high-throughput single-cell RNA sequencing (scRNA-seq) data.")
    (license license:bsd-3)))

(define-public python-celltypist
  (package
    (name "python-celltypist")
    (version "1.6.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/Teichlab/celltypist")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0c42cx01zkxr0dk5f1d7q71qdi18v2smlc3wpvwyjlzplya7k2iy"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #false ;there are none
      #:phases
      '(modify-phases %standard-phases
         (add-before 'check 'set-home
           ;; The sanity check requires a HOME directory, because celltypist
           ;; wants to write settings.
           (lambda _ (setenv "HOME" "/tmp")))
         ;; Numba needs a writable dir to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _ (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs
     (list python-click
           python-leidenalg
           python-numpy
           python-openpyxl
           python-pandas
           python-scanpy
           python-scikit-learn
           python-requests))
    (native-inputs (list python-wheel))
    (home-page "https://github.com/Teichlab/celltypist")
    (synopsis "Tool for semi-automatic cell type classification")
    (description
     "CellTypist is an automated cell type annotation tool for scRNA-seq
datasets on the basis of logistic regression classifiers optimised by the
stochastic gradient descent algorithm.  CellTypist allows for cell prediction
using either built-in (with a current focus on immune sub-populations) or
custom models, in order to assist in the accurate classification of different
cell types and subtypes.")
    (license license:expat)))

(define-public python-cmseq
  (package
    (name "python-cmseq")
    (version "1.0.4")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "CMSeq" version))
              (sha256
               (base32
                "0p6a99c299m5wi2z57dgqz52m1z3nfr8mv7kdnk2jvl2p9nql0wk"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #false ;there are no tests
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'patch-samtools-reference
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "cmseq/cmseq.py"
               (("'samtools'")
                (string-append "'" (search-input-file inputs "/bin/samtools") "'"))))))))
    (native-inputs
     (list python-setuptools))
    (inputs (list samtools))
    (propagated-inputs
     (list python-bcbio-gff-for-python-cmseq
           python-biopython-1.73
           python-numpy
           python-pandas
           python-pysam
           python-scipy))
    (home-page "https://github.com/SegataLab/cmseq/")
    (synopsis "Set of utilities on sequences and BAM files")
    (description
     "CMSeq is a set of commands to provide an interface to .bam files for coverage
and sequence consensus.")
    (license license:expat)))

(define-public python-cnmf
  (package
    (name "python-cnmf")
    (version "1.6.0")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "cnmf" version))
       (sha256
        (base32 "0aic8cwj6riykcfgl6v2x3si5z04gaknkh5a8lcyv1qh4s1gx3d3"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f ; no tests in git checkout and PyPI archive
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'set-envs
            (lambda _
              (setenv "MPLCONFIGDIR" "/tmp")
              ;; Numba needs a writable dir to cache functions.
              (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (native-inputs
     (list python-setuptools
           python-wheel))
    (propagated-inputs
     (list python-anndata
           python-fastcluster
           python-matplotlib
           python-numba
           python-numpy
           python-palettable
           python-pandas
           python-pyyaml
           python-scanpy
           python-scikit-learn
           python-scipy))
    (home-page "https://github.com/dylkot/cNMF")
    (synopsis "Consensus NMF for scRNA-Seq data")
    (description
     "This tool offers a pipeline for inferring gene expression programs from
scRNA-Seq. It takes a count matrix (N cells X G genes) as input and produces
a (K x G) matrix of gene expression programs (GEPs) and a (N x K) matrix
specifying the usage of each program for each cell in the data.")
    (license license:expat)))

(define-public python-cyvcf2
  (package
    (name "python-cyvcf2")
    (version "0.31.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/brentp/cyvcf2")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "19jv2502644qjv278yjllqqakwhghmfbaa4h08rayak5mp130i70"))
       (modules '((guix build utils)))
       (snippet
        ;; Delete bundled library
        '(delete-file-recursively "htslib"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'check 'build-extensions
            (lambda _
              ;; Cython extensions have to be built before running the tests.
              (invoke "python" "setup.py" "build_ext" "--inplace")))
          (add-after 'unpack 'fix-setup
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "setup.py"
                (("^htslib_include_dirs =.*")
                 (string-append "htslib_include_dirs = [\""
                                #$(this-package-input "htslib") "/include\"]\n"))
                (("lib_name = \"libhts.so\"")
                 (string-append "lib_name = \""
                                (search-input-file inputs "lib/libhts.so.3")
                                "\"\n")))))
          (add-before 'build 'use-system-htslib-package
            (lambda _
              (setenv "CYTHONIZE" "1")
              (setenv "CYVCF2_HTSLIB_MODE" "EXTERNAL"))))))
    (inputs (list curl htslib libdeflate openssl zlib))
    (native-inputs
     (list python-cython
           python-pytest
           python-setuptools))
    (propagated-inputs
     (list python-click
           python-coloredlogs
           python-numpy))
    (home-page "https://github.com/brentp/cyvcf2/")
    (synopsis "Fast vcf file parsing with Cython and htslib")
    (description "Cyvcf2 is a Cython wrapper around htslib built for fast
parsing of Variant Call Format (VCF) files.")
    (license license:expat)))

(define-public python-decoupler
  (package
    (name "python-decoupler")
    (version "2.1.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/scverse/decoupler")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0b15n5sq940sn29jsgmdkkm4fcpzfq1n221scfwhjxb4ybdpsz4v"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k" (string-join
                   ;; Tests requiring internet access to reach out
                   ;; <datasets.cellxgene.cziscience.com>, <ftp.ebi.ac.uk>,
                   ;; <omnipathdb.org>, <raw.githubusercontent.com>,
                   ;; <static.omnipathdb.org>, <www.ensembl.org>,
                   ;; <www.ncbi.nlm.nih.gov>, and <zenodo.org>.
                   (list "not test_collectri"
                         "test_covid5k"
                         "test_dorothea"
                         "test_download"
                         "test_download_anndata"
                         "test_ensmbl_to_symbol"
                         "test_erygast1k"
                         "test_hallmark"
                         "test_hsctgfb"
                         "test_knocktf"
                         "test_msvisium"
                         "test_pbmc3k"
                         "test_progeny"
                         "test_resource"
                         "test_show_resources"
                         "test_translate"
                         ;; XXX: Some precision mismatched.
                         "test_func_gsea"
                         "test_net_corr")
                   " and not "))
      #:phases
      '(modify-phases %standard-phases
         (add-before 'check 'set-home
           ;; Some tests require a home directory to be set.
           (lambda _ (setenv "HOME" "/tmp")))
         ;; Numba needs a writable dir to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _ (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs (list python-adjusttext
                             python-anndata
                             python-dcor
                             python-docrep
                             python-igraph
                             python-ipywidgets
                             python-marsilea
                             python-numba
                             python-requests
                             python-scipy
                             python-session-info2
                             python-tqdm
                             python-xgboost))
    (native-inputs
     (list nss-certs-for-test
           python-hatchling
           python-gseapy
           python-memory-profiler
           python-pytest
           python-scanpy))
    (home-page "https://github.com/scverse/decoupler")
    (synopsis
     "Framework for modeling, analyzing and interpreting single-cell RNA-seq data")
    (description
     "This package provides different statistical methods to extract
biological activities from omics data within a unified framework.")
    (license license:bsd-3)))

;; See: <https://github.com/scverse/decoupler/blob/main/CHANGELOG.md#200>
(define-deprecated-package python-decoupler-py
  python-decoupler)

(define-public python-demuxem
  (package
    (name "python-demuxem")
    (version "0.1.7")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "demuxEM" version))
              (sha256
               (base32
                "1bhyxqjk44bmyd26m1smapf68wyf7252kk65i27k50dd3kswgnd6"))))
    (build-system pyproject-build-system)
    ;; There are no tests.
    (arguments (list #:tests? #false))
    (propagated-inputs
     (list python-docopt
           python-importlib-metadata
           python-numpy
           python-pandas
           python-pegasusio
           python-scikit-learn
           python-scipy
           python-seaborn))
    (native-inputs
     (list python-cython
           python-setuptools
           python-setuptools-scm
           python-wheel))
    (home-page "https://github.com/lilab-bcb/demuxEM")
    (synopsis "Analyze cell-hashing/nucleus-hashing data")
    (description
     "This is a Python module for analyzing cell-hashing/nucleus-hashing data.
It is the demultiplexing module of Pegasus, which is used by Cumulus in the
demultiplexing step.")
    (license license:bsd-3)))

(define-public python-doubletdetection
  (package
    (name "python-doubletdetection")
    (version "4.2")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "doubletdetection" version))
              (sha256
               (base32
                "0v0a19014h4p6x8pyz1s78xn3q5w5166cysvg574z6vw79a3s9vp"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #false ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'use-poetry-core
            (lambda _
              ;; Patch to use the core poetry API.
              (substitute* "pyproject.toml"
                (("poetry.masonry.api")
                 "poetry.core.masonry.api")))))))
    (propagated-inputs
     (list python-anndata
           python-ipywidgets
           python-leidenalg
           python-louvain-igraph
           python-matplotlib
           python-numpy
           python-pandas
           python-phenograph
           python-scanpy
           python-scipy
           python-tqdm))
    (native-inputs
     (list python-black
           python-flake8
           python-poetry-core
           python-pytest))
    (home-page "https://github.com/JonathanShor/DoubletDetection")
    (synopsis
     "This is a package to detect doublets in single-cell RNA-seq count matrices")
    (description
     "This package provides a method to detect and enable removal of doublets
from single-cell RNA-sequencing.")
    (license license:expat)))

(define-public python-harmonypy
  (package
    (name "python-harmonypy")
    (version "0.0.10")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/slowkow/harmonypy")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "15lxgncrnsx1hapfx78pvx4rjx5d48hqixdnacdy55d84myfmrym"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'patch-pyproject
            (lambda _
              ;; License field should not be empty.
              (substitute* "pyproject.toml"
                (("^license = \"\"") "")))))))
    (propagated-inputs (list python-numpy
                             python-pandas
                             python-scikit-learn
                             python-scipy))
    (native-inputs (list python-hatchling python-pytest))
    (home-page "https://github.com/slowkow/harmonypy")
    (synopsis "Data integration algorithm")
    (description
     "Harmony is an algorithm for integrating multiple high-dimensional
datasets with fuzzy k-means and locally linear adjustments.")
    (license license:gpl2)))

(define-public python-hclust2
  (package
    (name "python-hclust2")
    (version "1.0.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "hclust2" version))
              (sha256
               (base32
                "0v89n2g42d7jhgfs8glf06apgxx6aswp3mfisgnhm518cv8z2rwn"))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #f))      ;there are no tests
    (propagated-inputs
     (list python-matplotlib
           python-numpy
           python-pandas
           python-scipy))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/SegataLab/hclust2/")
    (synopsis "Plotting heat-maps for publications")
    (description
     "Hclust2 is a handy tool for plotting heat-maps with several useful options
to produce high quality figures that can be used in publications.")
    (license license:expat)))

(define-public python-hotspotsc
  (package
    (name "python-hotspotsc")
    (version "1.1.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "hotspotsc" version))
       (sha256
        (base32 "1phbd49nb0ivfcgfi6yxd2masgd0v6133mki9vd5pkrhx0a7wb5d"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #false                   ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'use-poetry-core
            (lambda _
              ;; Patch to use the core poetry API.
              (substitute* "pyproject.toml"
                (("poetry.masonry.api")
                 "poetry.core.masonry.api")
                ;; Fix syntax error
                (("sphinx>") "sphinx")))))))
    (propagated-inputs (list python-anndata
                             python-importlib-metadata
                             python-ipython
                             python-matplotlib
                             python-nbsphinx
                             python-numba
                             python-numpy
                             python-pandas
                             python-pynndescent
                             python-scanpy
                             python-scikit-learn
                             python-scipy
                             python-seaborn
                             python-statsmodels
                             python-tqdm))
    (native-inputs
     (list python-poetry-core python-pytest python-sphinx))
    (home-page "https://github.com/yoseflab/hotspot")
    (synopsis
     "Identifying informative genes in a single-cell dataset")
    (description
     "Hotspot is a tool for identifying informative genes (and gene modules)
in a single-cell dataset.  Importantly, \"informative\" is decided based on
how well a gene's variation agrees with some cell metric---some similarity
mapping between cells.  Genes which are informative are those whose expression
varies in similar way among cells which are nearby in the given metric.")
    (license license:expat)))

(define-public python-htsget
  (package
   (name "python-htsget")
   (version "0.2.6")
   (source (origin
            (method url-fetch)
            (uri (pypi-uri "htsget" version))
            (sha256
             (base32
              "111q4pzkav26aa3hkgh948wqlyrq7dq6sjml9z63n3blw8s6b0c4"))))
   (build-system pyproject-build-system)
   (arguments
    (list #:tests? #f)) ;they depend on Nose test runner
   (native-inputs
    (list python-setuptools-scm python-setuptools))
   (propagated-inputs
    (list python-humanize python-requests python-six))
   (home-page "https://pypi.org/project/htsget/")
   (synopsis "Python API and command line interface for the GA4GH htsget API")
   (description "This package is a client implementation of the GA4GH htsget
protocol.  It provides a simple and reliable way to retrieve genomic data from
servers supporting the protocol.")
   (license license:asl2.0)))

(define-public python-liana-py
  (package
    (name "python-liana-py")
    (version "1.6.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/saezlab/liana-py")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1k6l371wd00m95l5pb2jsmzzxh5nc5v21fg2v0cslr9761q151r9"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k"
             ;; These tests require internet access.
             (string-append "not test_generate_lr_resource"
                            " and not test_get_hcop"
                            " and not test_get_metalinks"
                            " and not test_get_metalinks_values"
                            " and not test_describe_metalinks"
                            " and not test_generate_nondefault_lr_resource"
                            " and not test_translate_resource"
                            ;; Minor accuracy difference
                            " and not test_bivar_morans_perms"
                            ;; XXX "local_scores" array has wrong type.
                            ;; See https://github.com/saezlab/liana-py/issues/147
                            " and not test_morans_analytical"
                            " and not test_cosine_permutation"
                            " and not test_jaccard_pval_none_cats"
                            " and not test_large_adata"
                            ;; XXX ligand column differs: the left column
                            ;; contains duplicates.
                            " and not test_liana_pipe_not_defaults"
                            " and not test_liana_pipe_defaults"
                            ;; XXX unclear failure: 'coo_matrix' object is not
                            ;; subscriptable
                            " and not test_bivar_product"
                            ;; XXX unclear failure: large difference in data
                            ;; frames.
                            " and not test_aggregate_res"
                            ;; XXX: ValueError: Only CSR and CSC matrices are
                            ;; supported.
                            " and not test_bivar_nondefault"
                            " and not test_masked_spearman"
                            " and not test_vectorized_spearman"
                            " and not test_basic_interpolation"
                            " and not test_different_methods"
                            " and not test_fill_value"
                            " and not test_use_raw_layer_parameters")
             ;; These need the optional squidpy, which we don't have yet.
             "--ignore=tests/test_misty.py"
             ;; These need the optional corneto.
             "--ignore=tests/test_causalnet.py"
             ;; Needs internet access.
             "--ignore=tests/test_orthology.py")
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable directory to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _ (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs (list python-anndata
                             python-cell2cell
                             python-decoupler
                             python-hypothesis
                             python-ipykernel
                             python-ipython
                             python-mudata
                             python-nbconvert
                             python-nbsphinx
                             python-numpy
                             python-numpydoc
                             python-omnipath
                             python-pandas
                             python-plotnine
                             python-pypandoc
                             python-scipy
                             python-requests
                             python-scanpy
                             python-statsmodels
                             python-tqdm
                             tzdata))
    (native-inputs
     (list python-hatchling
           python-pytest))
    (home-page "https://github.com/saezlab/liana-py")
    (synopsis "LIANA is a ligand-receptor analysis framework")
    (description "This is a Ligand-Receptor inference framework.  The
framework enables the use of any LR method with any resources.")
    (license license:gpl3+)))

(define-public python-logomaker
  (package
    (name "python-logomaker")
    (version "0.8.7")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "logomaker" version))
              (sha256
               (base32
                "00ikzh7cv3yi1aj5v3hrpia6w7xrw559rqhwy3qxcja4wbk3qy33"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; All tests are marked as skipped in logomaker/tests, this is taken
      ;; form GitHub Actions file .github/workflows/daily-tests.yml.
      #:test-backend #~'custom
      #:test-flags #~(list "-c" "import logomaker")))
    (propagated-inputs
     (list python-matplotlib python-numpy python-pandas))
    (native-inputs (list python-hatchling))
    (home-page "https://github.com/jbkinney/logomaker")
    (synopsis "Package for making Sequence Logos")
    (description "Logomaker is a Python package for generating
publication-quality sequence logos.  Logomaker can generate both standard and
highly customized logos illustrating the properties of DNA, RNA, or protein
sequences.  Logos are rendered as vector graphics embedded within native
matplotlib Axes objects, making them easy to style and incorporate into
multi-panel figures.")
    (license license:expat)))

(define-public python-weblogo
  (package
    (name "python-weblogo")
    (version "3.7.12")
    (home-page "https://github.com/gecrooks/weblogo")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "weblogo" version))
       (sha256
        (base32
         "0mw6aa0dq3kk9k1nakdvm64icz9504spqbvq4v2h0rb1cb52frkw"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list ghostscript
                             python-numpy
                             python-pluggy
                             python-scipy))
    (native-inputs (list ghostscript
                         python-importlib-metadata
                         python-importlib-resources
                         python-pytest
                         python-setuptools
                         python-setuptools-scm
                         python-wheel))
    (arguments
     (list
      #:test-flags
      '(list "-k"
             ;; These tests fail because of circular imports
             (string-append "not test_cli.py"
                            " and not test_transformseq.py"))))
    (synopsis "Sequence Logo Generator")
    (description "WebLogo is a web based application designed to make the
generation of sequence logos as easy and painless as possible.

WebLogo can create output in several common graphics' formats, including the
bitmap formats GIF and PNG, suitable for on-screen display, and the vector
formats EPS and PDF, more suitable for printing, publication, and further
editing.  Additional graphics options include bitmap resolution, titles,
optional axis, and axis labels, antialiasing, error bars, and alternative
symbol formats.

A sequence logo is a graphical representation of an amino acid or nucleic acid
multiple sequence alignment.  Each logo consists of stacks of symbols, one
stack for each position in the sequence.  The overall height of the stack
indicates the sequence conservation at that position, while the height of
symbols within the stack indicates the relative frequency of each amino or
nucleic acid at that position.  The width of the stack is proportional to the
fraction of valid symbols in that position.")
    (license license:expat)))

(define-public python-magic-impute
  (package
    (name "python-magic-impute")
    (version "1.2.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/KrishnaswamyLab/MAGIC")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1yjs16vg87lcg9g16bnblg1v9sk73j6dm229lkcz0bfjlzxjhv8w"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #false ;there are none
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'chdir
           (lambda _ (chdir "python"))))))
    (propagated-inputs
     (list python-future
           python-graphtools
           python-matplotlib
           python-numpy
           python-pandas
           python-scikit-learn
           python-scipy
           python-tasklogger))
    (home-page "https://github.com/KrishnaswamyLab/MAGIC")
    (synopsis "Markov affinity-based graph imputation of cells")
    (description "MAGIC is an interactive tool to impute missing values in
single-cell sequencing data and to restore the structure of the data.  It also
provides data pre-processing functionality such as dimensionality reduction
and gene expression visualization.")
    (license license:gpl2+)))

(define-public python-metacells
  (package
    (name "python-metacells")
    (version "0.9.5")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "metacells" version))
       (sha256
        (base32 "0rb1r8l1ypfmvd3lwlx7ksfbjvgpnybn1s84a8n9gcpz9yx2rhbs"))))
    #;
    (properties '((tunable? . #t)))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          ;; The package "python-igraph" has been deprecated in favor of
          ;; just "igraph".
          (add-after 'unpack 'rename-igraph
            (lambda _
              (substitute* "requirements.txt"
                (("python-igraph") "igraph"))))
          (add-after 'unpack 'do-not-tune
            (lambda _
              ;; Without this they pass -march=native to the compiler.
              (setenv "WHEEL" "1")))
          ;; Numba needs a writable dir to cache functions.
          (add-before 'check 'set-numba-cache-dir
            (lambda _ (setenv "NUMBA_CACHE_DIR" "/tmp")))
          (add-before 'build 'build-extensions
            (lambda _
              (invoke "python" "setup.py" "build_ext" "--inplace"))))))
    (propagated-inputs (list python-anndata
                             python-cvxpy
                             python-fastcluster
                             python-igraph
                             python-importlib-metadata
                             python-numpy
                             python-pandas
                             python-psutil
                             python-pyyaml
                             python-scanpy
                             python-scipy
                             python-threadpoolctl
                             python-umap-learn))
    (native-inputs (list python-pytest python-setuptools))
    (home-page "https://github.com/tanaylab/metacells.git")
    (synopsis "Single-cell RNA Sequencing Analysis")
    (description "The metacells package implements the improved metacell
algorithm for single-cell RNA sequencing (scRNA-seq) data analysis within the
scipy framework, and projection algorithm based on it.  The original metacell
algorithm was implemented in R.  The Python package contains various
algorithmic improvements and is scalable for larger data sets (millions of
cells).")
    (license license:expat)))

(define-public python-ngesh
  (package
    (name "python-ngesh")
    (version "1.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/tresoldi/ngesh")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "07sj4x95b5hvx57pw24f80sk4ag4hkg1z6wzym3pzi8n5gn85n1z"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list python-ete3 python-numpy python-six))
    (native-inputs (list python-pytest))
    (home-page "https://github.com/tresoldi/ngesh")
    (synopsis "Library for phylogenetic tree simulation")
    (description
     "Ngesh is a Python library and CLI tool for simulating phylogenetic trees
and data.  It is intended for benchmarking phylogenetic methods, especially in
historical linguistics andstemmatology.  The generation of stochastic
phylogenetic trees also goes by the name simulationmethods for phylogenetic
trees, synthetic data generation, or just phylogenetic tree simulation.")
    (license license:expat)))

(define-public python-parabam
  (package
    (name "python-parabam")
    ;; XXX: Upstream works on modernization of the project, use the latest
    ;; commit providing fixes.
    ;; See: <https://github.com/cancerit/parabam/issues/10>.
    (properties '((commit . "be5bd35012d37df8cfa88771325a0273519c8c98")
                  (revision . "0")))
    (version (git-version "3.0.1"
                          (assoc-ref properties 'revision)
                          (assoc-ref properties 'commit)))
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/cancerit/parabam")
              (commit (assoc-ref properties 'commit))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1x0c1zhlfplhm4n07vibvh4jprjsdlypnlig87a8r07d26d4qphh"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f        ;no tests
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'cythonize
            (lambda _
              (with-directory-excursion "parabam"
                (for-each (lambda (file)
                            (invoke "cython" "-3" file "-I" "."))
                          (find-files "." ".*\\.pyx$"))))))))
    (propagated-inputs (list python-numpy python-pysam))
    (native-inputs (list python-cython python-setuptools))
    (home-page "https://github.com/cancerit/parabam")
    (synopsis "Parallel BAM File Analysis")
    (description "Parabam is a tool for processing sequencing files in
parallel.  It uses Python's native multiprocessing framework to apply a user
defined rule on an input file.")
    (license license:gpl3)))

(define-public python-pdbfixer
  (package
    (name "python-pdbfixer")
    (version "1.9")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/openmm/pdbfixer")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1zjhb19q5dclkwvzh8n29p31n1vzkhlmmzwqllimi89jsis1cx35"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k"
             ;; These tests fail because they require internet access.
             (string-append "not test_build_and_simulate.py"
                            " and not test_cli.py"
                            " and not test_mutate.py"))))
    (propagated-inputs (list openmm python-numpy))
    (native-inputs (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/openmm/pdbfixer")
    (synopsis "Application for fixing problems in Protein Data Bank")
    (description
     "PDBFixer is designed to rectify issues in Protein Data Bank files.
Its intuitive interface simplifies the process of resolving problems
encountered in PDB files prior to simulation tasks.")
    (license license:expat)))

(define-public python-peaks2utr
  (package
    (name "python-peaks2utr")
    (version "1.4.1")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "peaks2utr" version))
              (sha256
               (base32
                "104il0kk61q07b58g9xrss7xflwlbx4kzsmw9iih99lhfsii0wzg"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; These tests fail because file names are not URLs.
      '(list "-k" (string-join
                   (list "not test_forward_strand_annotations"
                         "test_matching_chr"
                         "test_reverse_strand_annotations")
                   " and not "))
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'relax-requirements
           (lambda _
             (substitute* "setup.cfg"
               (("==") ">=")))))))
    (propagated-inputs
     (list python-asgiref
           python-gffutils
           python-importlib-resources
           macs
           python-numpy
           python-psutil
           python-pybedtools
           python-pysam
           python-requests
           python-tqdm
           python-typing-extensions
           python-zipp))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/haessar/peaks2utr")
    (synopsis "Python CLI for annotating three prime UTR")
    (description
     "This package provides a robust, parallelized Python CLI for annotating
three prime UTR.")
    (license license:gpl3+)))

(define-public python-pegasusio
  (package
    (name "python-pegasusio")
    (version "0.9.1")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "pegasusio" version))
              (sha256
               (base32
                "1vxi6z7jlznp7sgzlbjsl1dzf1h7ypinllnls0lb2773a8vavg5x"))))
    (build-system pyproject-build-system)
    ;; There are no tests.
    (arguments (list #:tests? #false))
    (propagated-inputs
     (list python-anndata
           python-cython
           python-docopt
           python-h5py
           python-loompy
           python-natsort
           python-numpy
           python-pandas
           python-pillow
           python-scipy
           python-setuptools
           python-zarr))
    (native-inputs (list python-cython python-setuptools python-setuptools-scm
                         python-wheel))
    (home-page "https://github.com/lilab-bcb/pegasusio")
    (synopsis "Read or write single-cell genomics data")
    (description
     "Pegasusio is a Python package for reading or writing single-cell
genomics data.")
    (license license:bsd-3)))

(define-public python-phenograph
  (package
    (name "python-phenograph")
    (version "1.5.7")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "PhenoGraph" version))
              (sha256
               (base32
                "0nji449mzwgp1f87iknl5fmnjdkrhkfkapxvafxdw01s0jg8zcj6"))
              (modules '((guix build utils)))
              ;; Remove bundled binaries
              (snippet
               '(delete-file-recursively "phenograph/louvain"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          ;; This test can never succeed because Q_leiden is never set to
          ;; anything other than None.
          (add-after 'unpack 'disable-leiden-test
            (lambda _
              (substitute* "tests/test_cluster.py"
                (("def test_run_leiden") "def _test_run_leiden"))))
          (add-after 'unpack 'patch-louvain
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "phenograph/core.py"
                (("lpath = os.path.*")
                 (string-append "lpath = \""
                                (dirname (search-input-file inputs "/bin/community"))
                                "\"\n"))
                (("linux-(community|hierarchy|convert)" _ thing) thing)
                ;; Do not write binaries, because the unmodified "convert"
                ;; from louvain only knows how to process plain text files.
                (("with open\\(filename \\+ \".bin\", \"w\\+b\"\\) as f:")
                 "with open(filename + \".bin\", \"w+\") as f:")
                (("f.writelines\\(\\[e for t in zip\\(ij, s\\) for e in t\\]\\)")
                 "for [src, dest], weight in zip(ij, s): \
f.write(src.astype(\"str\") + ' ' + \
dest.astype(\"str\") + ' ' + \
weight.astype(\"str\") + '\\n')")))))))
    (inputs
     (list louvain))
    (propagated-inputs
     (list python-leidenalg
           python-numpy
           python-psutil
           python-scikit-learn
           python-scipy))
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/dpeerlab/PhenoGraph.git")
    (synopsis "Graph-based clustering for high-dimensional single-cell data")
    (description
     "PhenoGraph is a clustering method designed for high-dimensional
single-cell data.  It works by creating a graph representing phenotypic
similarities between cells and then identifying communities in this graph.")
    (license license:expat)))

(define-public python-phylophlan
  (package
    (name "python-phylophlan")
    (version "3.0.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/biobakery/phylophlan")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wz70xzxqx2sf5flmf45m15jq027dqijfaj1r51pl50w5x6dkawx"))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #f))      ;there are no tests
    (propagated-inputs
     (list python-biopython
           python-dendropy
           python-matplotlib
           python-numpy
           python-pandas
           python-seaborn))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/biobakery/phylophlan")
    (synopsis
     "Phylogenetic analysis of microbial isolates and genomes from metagenomes")
    (description
     "This package is an integrated pipeline for large-scale phylogenetic
profiling of genomes and metagenomes.  PhyloPhlAn is an accurate, rapid, and
easy-to-use method for large-scale microbial genome characterization and
phylogenetic analysis at multiple levels of resolution.  This software package
can assign both genomes and @acronym{MAGs, metagenome-assembled genomes} to
@acronym{SGBs, species-level genome bins}.  PhyloPhlAn can reconstruct
strain-level phylogenies using clade- specific maximally informative
phylogenetic markers, and can also scale to very large phylogenies comprising
>17,000 microbial species.")
    (license license:expat)))

(define-public python-pybedtools
  (package
    (name "python-pybedtools")
    (version "0.10.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "pybedtools" version))
              (sha256
               (base32
                "0q8if5bd8zgv5xvr5zs4pj8y60yzl8i5jz8xfk6bw4xh4fnvlvqs"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:modules '((srfi srfi-26)
                  (guix build utils)
                  (guix build python-build-system)
                  (guix build pyproject-build-system))
      #:test-flags
      ;; Requires internet access.
      '(list "-k" "not test_chromsizes")
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'fix-references
           (lambda _
             (substitute* "pybedtools/test/test_issues.py"
               (("'/bin/bash'") (string-append "'" (which "bash") "'")))))
         ;; Force the Cythonization of C++ files to guard against compilation
         ;; problems.
         (add-after 'unpack 'remove-cython-generated-files
           (lambda _
             (let ((cython-sources (map (cut string-drop-right <> 4)
                                        (find-files "." "\\.pyx$")))
                   (c/c++-files (find-files "." "\\.(c|cpp|cxx)$")))
               (define (strip-extension filename)
                 (string-take filename (string-index-right filename #\.)))
               (define (cythonized? c/c++-file)
                 (member (strip-extension c/c++-file) cython-sources))
               (for-each delete-file (filter cythonized? c/c++-files)))))
         (add-after 'remove-cython-generated-files 'generate-cython-extensions
           (lambda _
             (invoke "python" "setup.py" "cythonize")))
         (add-before 'check 'build-extensions
           (lambda _
             ;; Cython extensions have to be built before running the tests.
             (invoke "python" "setup.py" "build_ext" "--inplace"))))))
    (propagated-inputs
     (list bedtools samtools
           kentutils ;for bedGraphToBigWig
           python-numpy
           python-pandas
           python-psutil
           python-pysam
           python-pyyaml))
    (inputs
     (list zlib))
    (native-inputs
     (list python-cython
           python-pytest
           python-setuptools
           python-wheel))
    (home-page "https://pythonhosted.org/pybedtools/")
    (synopsis "Python wrapper for BEDtools programs")
    (description
     "This package is a Python wrapper for Aaron Quinlan's BEDtools programs,
which are widely used for genomic interval manipulation or \"genome algebra\".
pybedtools extends BEDTools by offering feature-level manipulations from with
Python.")
    ;; pypi lists GPLv2 in the PKG-INFO and website, but was relicensed in
    ;; version 0.9.0 and the LICENSE.txt is consistent with the source code.
    ;;
    ;; pybedtools/include/gzstream.cpp and pybedtools/include/gzstream.h are
    ;; licensed lgpl2.1+
    (license (list license:expat license:lgpl2.1+))))

(define-public python-pybio
  (let ((commit "c91fddc483da535d5097364405d76ad9b1bde07f")
        (revision "1"))
    (package
      (name "python-pybio")
      (version (git-version "0.3.12" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/grexor/pybio")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0s3n1bqp25zf5pzfsj0x1kqr6i2a6iffpb8hkjwhyvjqrxf5d9rk"))))
      (build-system pyproject-build-system)
      (arguments
       (list
        #:tests? #false                 ;There are no automated tests
        #:phases
        #~(modify-phases %standard-phases
            (add-before 'check 'set-home
              ;; A HOME directory is required when importing the module during
              ;; the sanity check.
              (lambda _ (setenv "HOME" "/tmp")))
            (add-after 'unpack 'adjust-requirements
              (lambda _
                ;; bs4 is an alternative name for beautifulsoup4, only used to
                ;; avoid name squatting on pypi.
                (substitute* "setup.py"
                  (("bs4") "beautifulsoup4"))))
            (add-after 'unpack 'avoid-internet-while-building
              (lambda _
                (setenv "GUIX_BUILD" "yes")
                (substitute* "pybio/__init__.py"
                  (("^pybio.core.genomes.init\\(\\)" m)
                   (string-append m
                                  "\
 if not os.getenv('GUIX_BUILD') else None"))))))))
      (propagated-inputs
       (list python-beautifulsoup4 python-numpy
             python-psutil python-pysam python-requests))
      (native-inputs (list python-pytest python-setuptools python-wheel))
      (home-page "https://github.com/grexor/pybio")
      (synopsis "Basic genomics toolset")
      (description
       "This tool provides a Python framework to streamline genomics
operations.  It offers a direct interface to Ensembl genome assemblies and
annotations, while also accommodating custom genomes via FASTA/GTF inputs.
The primary objective of pybio is to simplify genome management.  It achieves
this by providing automatic download of Ensembl genome assemblies and
annotation, provides Python genomic feature search and sequence retrieval from
the managed genomes, STAR indexing and mapping and more.")
      (license license:gpl3+))))

(define-public python-pybiomart
  (package
    (name "python-pybiomart")
    (version "0.2.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/jrderuiter/pybiomart")
              (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1x5rd33sa0gwj4nizn02lsc675bbzsspz74f38mp0fbxdp7qnxv8"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; Attempts to access the web.
      #~(list "--ignore=tests/test_dataset.py")))
    (propagated-inputs
     (list python-future
           python-pandas
           python-requests
           python-requests-cache))
    (native-inputs
     (list python-pytest
           python-pytest-helpers-namespace
           python-pytest-mock
           python-setuptools
           python-sphinx
           python-sphinx-rtd-theme))
    (home-page "https://github.com/jrderuiter/pybiomart")
    (synopsis "A simple pythonic interface to biomart")
    (description
     "Pybiomart provides a simple pythonic interface to biomart.")
    (license license:expat)))

(define-public python-pygam
  (package
    (name "python-pygam")
    (version "0.9.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/dswah/pyGAM")
              (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1bv404idswsm2ay3yziq1i2cbydq4f3vjav5s4i15bgd13k7zvim"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'patch-pyproject
            (lambda _
              (substitute* "pyproject.toml"
                ;; Change build backend
                (("build-backend = .*")
                 "build-backend = \"poetry.core.masonry.api\"\n")
                ;; Modify version field
                (("^version = \"0.0.0\"")
                 (string-append "version = \"" #$version "\""))))))))
    (propagated-inputs (list python-black
                             python-flake8
                             python-ipython
                             python-numpy
                             python-pandas
                             python-poetry-core
                             python-progressbar2
                             python-scipy))
    (native-inputs (list python-mock python-pytest python-pytest-cov))
    (home-page "https://github.com/dswah/pyGAM")
    (synopsis "Generalized additive models in Python")
    (description
     "This tool is for building Generalized Additive Models in Python.
It emphasizes modularity and performance.  The API will be immediately
familiar to anyone with experience of scikit-learn or scipy.")
    (license license:asl2.0)))

(define bgen-sample-files
  (let* ((name "bgen-sample-files")
         (commit "7b1bc74f58b326ca19606fa5f3c6093d48367993")
         (revision "0")
         (version (git-version "0.0.0" revision commit)))
    (origin
      (method git-fetch)
      (uri (git-reference
            (url "https://github.com/fastlmm/bgen-sample-files")
            (commit commit)))
      (file-name (git-file-name name version))
      (sha256
       (base32 "1b8jkscccyspfr3y1b66xvwnljcha08r2i24v9d6hcm5zii0l54w")))))

(define pynsptools-examples-files
  (let* ((name "pysnptools-examples-files")
         ;; taken from pysnptools/util/pysnptools.hashdown.json
         (commit "ed14e050b2b75e7f4ddb73d512fbe928bbdb2b85")
         (revision "0")
         (version (git-version "0.0.0" revision commit)))
    (origin
      (method git-fetch)
      (uri (git-reference
            (url "https://github.com/fastlmm/PySnpTools")
            (commit commit)))
      (file-name (git-file-name name version))
      (sha256
       (base32 "0hznpj15kx2sla16wlmcqz21n2vi2qb1493v30vz75hnm1m4iwm1"))
      (modules '((guix build utils)
                 (ice-9 ftw)))
      (snippet
       #~(begin
           ;; Delete everything except for examples directory:
           (define (delete-except exceptions)
             (lambda (file)
               (unless (member file `("." ".." ,@exceptions))
                 (delete-file-recursively file))))
           (for-each (delete-except '("pysnptools" "tests")) (scandir "."))
           (with-directory-excursion "pysnptools"
             (for-each (delete-except '("examples")) (scandir ".")))
           (with-directory-excursion "tests"
             (for-each (delete-except '("datasets")) (scandir "."))))))))

(define-public python-pysnptools
  (package
    (name "python-pysnptools")
    (version "0.5.14")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "pysnptools" version))
       (sha256
        (base32 "1babnyky5fk93as1ybdvpz9x3x5099gkgscxflngzfswin23mspk"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list
         ;; These tests require the bgen feature and an additional input.
         "--ignore" "pysnptools/distreader/bgen.py"
         "--ignore" "pysnptools/distreader/test_bgen2.py"
         "--ignore" "pysnptools/distreader/distreader.py"
         "-k" "not pysnptools.distreader.distdata.DistData.val \
and not pysnptools.distreader.disthdf5.DistHdf5.write \
and not pysnptools.distreader.distmemmap.DistMemMap.write"
         ;; These tests require network connection.
         "--ignore" "pysnptools/util/_example_file.py"
         "--ignore" "pysnptools/util/filecache/hashdown.py")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'patch-data-path
            (lambda _
              (substitute* "pysnptools/util/pysnptools.hashdown.json"
                (("https://github\\.com/fastlmm/PySnpTools/\
raw/ed14e050b2b75e7f4ddb73d512fbe928bbdb2b85")
                 (string-append "file://" #+pynsptools-examples-files)))
              (substitute* "pysnptools/util/bgen.hashdown.json"
                (("https://raw\\.githubusercontent\\.com\
/fastlmm/bgen-sample-files/7b1bc74f58b326ca19606fa5f3c6093d48367993")
                 (string-append "file://" #+bgen-sample-files)))))
          (add-after 'unpack 'loosen-requirements
            (lambda _
              (substitute* "pyproject.toml"
                (("(h5py|psutil|wheel)>=[0-9.]*" all target)
                 target)))))))
    (propagated-inputs
     (list python-bed-reader
           python-cloudpickle
           python-h5py
           python-more-itertools
           python-numpy
           python-pandas
           python-psutil
           python-scipy))
    (native-inputs
     (list python-pytest
           python-pytest-cov
           python-pytest-datadir
           python-pytest-doctestplus
           python-setuptools
           python-sphinx
           python-sphinx-rtd-theme
           python-wheel))
    (home-page "http://microsoftgenomics.github.io/PySnpTools/")
    (synopsis "Library for reading and manipulating genetic data")
    (description
     "PySnpTools is a library for reading and manipulating genetic data.  It
can, for example, efficiently read whole PLINK *.bed/bim/fam files or parts of
those files.  It can also efficiently manipulate ranges of integers using set
operators such as union, intersection, and difference.")
    (license license:asl2.0)))

(define-public python-ega-download-client
  (package
    (name "python-ega-download-client")
    (version "5.2.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/EGA-archive/ega-download-client")
              (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0wz36ii6sb65nknh56vr5ss6b446zy9iibz7jv3irgfas955jlrc"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; tests: 78 passed, 2 deselected, 1 warning
      #:test-flags
      ;; Test suite does a lot of fake os mocking and extensively hash
      ;; checking on a large amount of generated files (up to 150GiB each)
      ;; which would take some extra compute resources, therefor functional
      ;; and some of unit tests are excluded to relax the load.
      #~(list #$@(map (lambda (file) (string-append "--ignore="
                                                    "tests/"
                                                    file))
                      (list "functional/"
                            "unit/test_save_to_with_dataset.py"
                            "unit/test_save_to_with_file.py"
                            "unit/test_verify_output_dir.py"))
              #$@(map (lambda (test) (string-append "--deselect="
                                                    "tests/unit/"
                                                    "test_commands.py::"
                                                    test))
                      ;; requests.exceptions.RetryError: None: Max retries
                      ;; exceeded with url:
                      ;; https://test.data.server/metadata/datasets (Caused by
                      ;; ResponseError('too many 503 error responses'))
                      (list "test_error_5xx"
                            "test_error_too_many_requests")))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'relax-requirements
            (lambda _
              (substitute* "setup.py"
                (("==") ">=")))))))
    (propagated-inputs (list python-htsget python-psutil python-requests
                             python-tqdm python-urllib3))
    (native-inputs
     (list nss-certs-for-test
           python-mock
           python-pyfakefs
           python-pytest
           python-responses
           python-setuptools))
    (home-page "https://github.com/EGA-archive/ega-download-client")
    (synopsis "EGA download client")
    (description "PyEGA3 is a tool for viewing and downloading files from
authorized EGA datasets.  It uses the EGA data API and has several key
features:

@itemize
@item Files are transferred over secure https connections and received
  unencrypted, so no need for decryption after download.
@item Downloads resume from where they left off in the event that the
  connection is interrupted.
@item Supports file segmenting and parallelized download of segments,
  improving overall performance.
@item After download completes, file integrity is verified using checksums.
@item Implements the GA4GH-compliant htsget protocol for download of genomic
  ranges for data files with accompanying index files.
@end itemize\n")
    (license license:asl2.0)))

(define-public python-scdamandtools
  (package
    (name "python-scdamandtools")
    (version "1.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/KindLab/scDamAndTools")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mblw6cn5jqik6ky8cv7ry99z6jm1i4r71pzdfl398vrwbda65gd"))))
    (build-system pyproject-build-system)
    (arguments
     (list #:tests? #f))                ;there are none
    (propagated-inputs (list python-h5py
                             python-numpy
                             python-sortedcontainers
                             python-pandas
                             python-pysam
                             python-tqdm))
    (native-inputs (list python-cython python-pytest python-setuptools
                         python-wheel))
    (home-page "https://github.com/KindLab/scDamAndTools")
    (synopsis "Functions for processing raw scDam&T-seq data")
    (description
     "This is a set of functions for processing raw scDam&T-seq data.
scDam&T-seq is a method to simultaneously measure protein-DNA interactions and
transcription from single cells (Rooijers et al., 2019).  It combines a
DamID-based method to measure protein-DNA interactions and an adaptation of
CEL-Seq to measure transcription.  The starting point of the workflow is raw
sequencing data and the end result are tables of UMI-unique DamID and CEL-Seq
counts.")
    (license license:expat)))

(define-public python-snaptools
  (package
    (name "python-snaptools")
    (version "1.4.8")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "snaptools" version))
       (sha256
        (base32
         "1s5373g5jjbshh3q39zy7dlxr7nda6ksxq9d1gw46h82c4fsmfbn"))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #false)) ;there are none
    (propagated-inputs
     (list python-future
           python-h5py
           python-louvain
           python-numpy
           python-pybedtools
           python-pysam))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/r3fang/SnapTools")
    (synopsis "Tools for processing snap files" )
    (description
     "@code{SnapTools} can operate on snap files the following types of
operations:

@itemize
@item index the reference genome before alignment;
@item align reads to the corresponding reference genome;
@item pre-process by convert pair-end reads into fragments, checking the
  mapping quality score, alignment and filtration;
@item create the cell-by-bin matrix.
@end itemize")
    (license license:asl2.0)))

(define-public python-telomerecat
  (package
    (name "python-telomerecat")
    (version "4.0.2")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "telomerecat" version))
       (sha256
        (base32 "16mfdqmp0j6g3h26h59334w9lqb4qihqrlzwvgznj0fiqs1rkxn2"))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #false)) ;there are none
    (propagated-inputs (list python-click python-numpy python-pandas
                             python-parabam python-pysam))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/cancerit/telomerecat")
    (synopsis "Telomere computational analysis tool")
    (description "Telomerecat is a tool for estimating the average telomere
length (TL) for a paired end, whole genome sequencing (WGS) sample.

Telomerecat is adaptable, accurate and fast.  The algorithm accounts for
sequencing amplification artifacts, anneouploidy (common in cancer samples)
and noise generated by WGS.  For a high coverage WGS BAM file of around 100GB
telomerecat can produce an estimate in ~1 hour.")
    (license license:gpl3)))

(define-public python-bioframe
  (package
    (name "python-bioframe")
    (version "0.6.4")
    (source
     (origin
       (method git-fetch)
       ;; pypi version does not contain tests and requirements.txt
       (uri (git-reference
             (url "https://github.com/open2c/bioframe")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1m99hgxw4cb2x4qszb2lhp1isz57sdkqbmcgisnbqxqxkv4gba7v"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k" (string-append "not test_fetch_chromsizes"
                                 " and not test_fetch_chromsizes_local_vs_ucsc"
                                 " and not test_fetch_centromeres"))
      #:phases
      '(modify-phases %standard-phases
         (add-before 'check 'pre-check
           (lambda _ (setenv "MPLCONFIGDIR" "/tmp"))))))
    (native-inputs
     (list python-biopython
           python-hatchling
           python-pysam
           python-pytest
           python-wheel))
    (propagated-inputs
     (list python-matplotlib
           python-numpy
           python-pandas
           python-pyyaml
           python-requests))
    (home-page "https://github.com/open2c/bioframe")
    (synopsis "Pandas utilities for tab-delimited and other genomic files")
    (description
     "This package is a library to enable flexible and scalable operations on
genomic interval dataframes in Python.  Bioframe enables access to a rich set
of dataframe operations.  Working in Python enables rapid visualization and
iteration of genomic analyses.  The philosophy underlying bioframe is to
enable flexible operations.  Instead of creating a function for every possible
use-case, we encourage users to compose functions to achieve their goals.")
    (license license:expat)))

(define-public python-biom-format
  (package
    (name "python-biom-format")
    (version "2.1.16")
    (source
     (origin
       (method git-fetch)
       ;; Use GitHub as source because PyPI distribution does not contain
       ;; test data: https://github.com/biocore/biom-format/issues/693
       (uri (git-reference
             (url "https://github.com/biocore/biom-format")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1npxjsi7r0w8diq6s37q21cqgrqifl7f483lfn72bn7qrvkvbpyz"))
       (modules '((guix build utils)))
       ;; Delete generated C files.
       (snippet
        '(for-each delete-file (find-files "." "\\.c")))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k"
             (string-append ;; Unclear why this one fails.  There is no backtrace.
                            "not test_to_dataframe_is_sparse"
                            ;; These need skbio, but that needs biom-format.
                            " and not test_align_tree_intersect_obs"
                            " and not test_align_tree_intersect_tips"
                            " and not test_align_tree_sample"))
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'use-cython
           (lambda _ (setenv "USE_CYTHON" "1")))
         (add-before 'check 'build-extensions
           (lambda _
             ;; Cython extensions have to be built before running the tests.
             (invoke "python" "setup.py" "build_ext" "--inplace")))
         (add-after 'unpack 'pandas-compatibility
           (lambda _
             (substitute* "biom/tests/test_table.py"
               (("import pandas.util.testing")
                "import pandas.testing")))))))
    (propagated-inputs
     (list python-anndata
           python-click
           python-flake8
           python-future
           python-h5py
           python-numpy
           python-pandas
           ;;python-scikit-bio ;mutually recursive dependency
           python-scipy))
    (native-inputs
     (list python-cython python-pytest python-pytest-cov
           python-wheel))
    (home-page "https://www.biom-format.org")
    (synopsis "Biological Observation Matrix (BIOM) format utilities")
    (description
     "The BIOM file format is designed to be a general-use format for
representing counts of observations e.g. operational taxonomic units, KEGG
orthology groups or lipid types, in one or more biological samples
e.g. microbiome samples, genomes, metagenomes.")
    (license license:bsd-3)))

(define-public python-pairtools
  (package
    (name "python-pairtools")
    (version "1.1.0-fix")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/open2c/pairtools")
                    (commit (string-append "pairtools-v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0983vw4kb6frjncsnml4ahw3l7sixg1paz80s119iah2i086cw06"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'patch-setup.py
           (lambda _
             ;; __NUMPY_SETUP__ is undefined.
             (substitute* "setup.py"
               ((".*__builtins__.__NUMPY_SETUP.*") ""))))
         (add-after 'unpack 'fix-references
           (lambda _
             (substitute* '("pairtools/cli/header.py"
                            "pairtools/cli/merge.py"
                            "pairtools/cli/sort.py")
               (("/bin/bash") (which "bash")))))
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (with-directory-excursion "/tmp"
                 (invoke "pytest" "-v"))))))))
    (native-inputs
     (list python-cython
           python-pytest
           python-pytest-cov
           python-setuptools
           python-wheel))
    (propagated-inputs
     (list htslib ; for bgzip, looked up in PATH
           samtools ; looked up in PATH
           lz4 ; for lz4c
           python-bioframe
           python-click
           python-numpy
           python-pandas
           python-pysam
           python-pyyaml
           python-scipy))
    (home-page "https://github.com/open2c/pairtools")
    (synopsis "Process mapped Hi-C data")
    (description "Pairtools is a simple and fast command-line framework to
process sequencing data from a Hi-C experiment.  Process pair-end sequence
alignments and perform the following operations:

@itemize
@item detect ligation junctions (a.k.a. Hi-C pairs) in aligned paired-end
  sequences of Hi-C DNA molecules
@item sort @code{.pairs} files for downstream analyses
@item detect, tag and remove PCR/optical duplicates
@item generate extensive statistics of Hi-C datasets
@item select Hi-C pairs given flexibly defined criteria
@item restore @code{.sam} alignments from Hi-C pairs.
@end itemize
")
    (license license:expat)))

(define-public python-readpaf
  (package
    (name "python-readpaf")
    (version "0.0.10")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "readpaf" version))
              (sha256
               (base32
                "15m6ffks4zwpp1ycwk6n02py6mw2yh7qr0vhpc178b91gldr97ia"))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #false))  ;there are none
    (propagated-inputs (list python-pandas))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/alexomics/read-paf")
    (synopsis "Minimap2 PAF file reader")
    (description
     "This is a fast parser for minimap2 PAF (Pairwise mApping Format)
files.")
    (license license:expat)))

(define-public python-whatshap
  (let ((revision "0")
        (commit "5722d69404532d3172758acd04e77fce734711c7")
        (base-version "2.3"))
    (package
      (name "python-whatshap")
      (version (git-version base-version revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/whatshap/whatshap")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1f9jgwb2kzhn190lxzhc14ji4flmrrdqh39d0g42qzvr1i19yv7p"))))
      (build-system pyproject-build-system)
      (arguments
       (list
        #:test-flags
        '(list "-k"
               (string-append
                ;; We have no solver for pulp.
                "not test_blockcut_sensitivities3 "
                "and not test_get_optimal_permutations2 "
                ;; Unknown error with the test file.
                "and not test_vcf_with_missing_headers "
                ;; Missing test file.
                "and not test_matrix"))
        #:phases
        #~(modify-phases %standard-phases
            (add-after 'unpack 'pretend-version
              (lambda _
                (setenv "SETUPTOOLS_SCM_PRETEND_VERSION" #$base-version)))
            (replace 'check
              (lambda* (#:key tests? test-flags #:allow-other-keys)
                (when tests?
                  ;; Step out of source dir so python does not import from CWD.
                  (let ((tests (string-append (getcwd) "/tests")))
                    (with-directory-excursion "/tmp"
                      (setenv "HOME" "/tmp")
                      ;; Test data files are looked up relative to the current
                      ;; directory.
                      (mkdir-p "tests/")
                      (copy-recursively (string-append tests "/data")
                                        "tests/data")
                      (apply invoke "pytest" "-vv" tests test-flags)))))))))
      (propagated-inputs (list python-biopython
                               python-matplotlib
                               python-networkx
                               python-pulp
                               python-pyfaidx
                               python-scipy
                               python-xopen))
      (native-inputs (list python-cython
                           python-pytest
                           python-pysam
                           python-setuptools
                           python-setuptools-scm
                           python-sphinx
                           python-sphinx-issues
                           python-wheel))
      (home-page "https://whatshap.readthedocs.io/")
      (synopsis "Read-based phasing of genomic variants")
      (description
       "WhatsHap is software for phasing genomic variants using DNA sequencing
reads, also called read-based phasing or haplotype assembly.  It is especially
suitable for long reads, but works also well with short reads.")
      (license license:expat))))

(define-public bioperl-minimal
  (package
    (name "bioperl-minimal")
    (version "1.7.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/bioperl/bioperl-live")
             (commit (string-append "release-"
                                    (string-map (lambda (c)
                                                  (if (char=? c #\.)
                                                      #\- c)) version)))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0wl8yvzcls59pwwk6m8ahy87pwg6nnibzy5cldbvmcwg2x2w7783"))))
    (build-system perl-build-system)
    (arguments
     (let ((transitive-inputs
            (map (compose package-name cadr)
                 (delete-duplicates
                  (concatenate
                   (map (compose package-transitive-target-inputs cadr)
                        (package-inputs this-package)))))))
       `(#:phases
         (modify-phases %standard-phases
           (add-after 'install 'wrap-programs
             (lambda* (#:key outputs #:allow-other-keys)
               ;; Make sure all executables in "bin" find the required Perl
               ;; modules at runtime.  As the PERL5LIB variable contains also
               ;; the paths of native inputs, we pick the transitive target
               ;; inputs from %build-inputs.
               (let* ((out  (assoc-ref outputs "out"))
                      (bin  (string-append out "/bin/"))
                      (path (string-join
                             (cons (string-append out "/lib/perl5/site_perl")
                                   (map (lambda (name)
                                          (assoc-ref %build-inputs name))
                                        ',transitive-inputs))
                             ":")))
                 (for-each (lambda (file)
                             (wrap-program file
                               `("PERL5LIB" ":" prefix (,path))))
                           (find-files bin "\\.pl$")))))))))
    (inputs
     (list bash-minimal perl-module-build perl-data-stag perl-libwww perl-uri))
    (native-inputs
     (list perl-test-most))
    (home-page "https://metacpan.org/release/BioPerl")
    (synopsis "Bioinformatics toolkit")
    (description
     "BioPerl is the product of a community effort to produce Perl code which
is useful in biology.  Examples include Sequence objects, Alignment objects
and database searching objects.  These objects not only do what they are
advertised to do in the documentation, but they also interact - Alignment
objects are made from the Sequence objects, Sequence objects have access to
Annotation and SeqFeature objects and databases, Blast objects can be
converted to Alignment objects, and so on.  This means that the objects
provide a coordinated and extensible framework to do computational biology.")
    (license license:perl-license)))

(define-public perl-bio-db-hts
  (package
    (name "perl-bio-db-hts")
    (version "3.01")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "mirror://cpan/authors/id/A/AV/AVULLO/Bio-DB-HTS-"
                           version ".tar.gz"))
       (sha256
        (base32
         "0hjg0igfkpvh27zdkdr6pa7cqm9n6r7cwz0np74cl4wmawgvr9hj"))))
    (build-system perl-build-system)
    (native-inputs
     (list perl-module-build pkg-config))
    (propagated-inputs
     (list bioperl-minimal htslib))
    (home-page "https://metacpan.org/release/Bio-DB-HTS")
    (synopsis "Perl interface to HTS library for DNA sequencing")
    (description "This is a Perl interface to the HTS library for DNA
sequencing.")
    (license license:asl2.0)))

(define-public python-biopython
  (package
    (name "python-biopython")
    (version "1.85")
    (source (origin
              (method url-fetch)
              ;; use PyPi rather than biopython.org to ease updating
              (uri (pypi-uri "biopython" version))
              (sha256
               (base32
                "19m03s5rwcyiq5cs1kq9dzj7qvmfvm76idgn967ygr4x0msapbsx"))))
    (build-system pyproject-build-system)
    (arguments
     '(#:phases
       (modify-phases %standard-phases
         (add-before 'check 'set-home
           ;; Some tests require a home directory to be set.
           (lambda _ (setenv "HOME" "/tmp")))
         (add-after 'unpack 'numpy-compatibility
           (lambda _
             (substitute* "Bio/Cluster/__init__.py"
               (("np.True_") "True"))))
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (with-directory-excursion "Tests"
                 (invoke "python3" "run_tests.py" "--offline"))))))))
    (propagated-inputs
     (list python-numpy))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://biopython.org/")
    (synopsis "Tools for biological computation in Python")
    (description
     "Biopython is a set of tools for biological computation including parsers
for bioinformatics files into Python data structures; interfaces to common
bioinformatics programs; a standard sequence class and tools for performing
common operations on them; code to perform data classification; code for
dealing with alignments; code making it easy to split up parallelizable tasks
into separate processes; and more.")
    (license (license:non-copyleft "http://www.biopython.org/DIST/LICENSE"))))

(define-public python-biopython-1.73
  (package
    (inherit python-biopython)
    (version "1.73")
    (source (origin
              (method url-fetch)
              ;; use PyPi rather than biopython.org to ease updating
              (uri (pypi-uri "biopython" version))
              (sha256
               (base32
                "1q55jhf76z3k6is3psis0ckbki7df26x7dikpcc3vhk1vhkwribh"))))
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'python3.11-compatibility
            (lambda _
              ;; Py_TYPE was changed to an inline static function in Python
              ;; 3.11, so it cannot be used on the left-hand side.
              (substitute* "Bio/triemodule.c"
                (("Py_TYPE\\(&Trie_Type\\) = &PyType_Type;")
                 "Py_SET_TYPE(&Trie_Type, &PyType_Type);"))

              ;; PY_SSIZE_T_CLEAN must be defined if s# notation is used.
              ;; Note that this is no longer needed in Python 3.13.
              (substitute* '("Bio/triemodule.c" "Bio/motifs/_pwm.c")
                (("#include <Python.h>")
                 "#define PY_SSIZE_T_CLEAN\n#include <Python.h>")
                (("int s;") "Py_ssize_t s;"))

              ;; The U mode is implicit in Python 3.11.
              (substitute* '("Tests/test_KGML_nographics.py"
                             "Tests/test_PDB.py")
                (("'rU'") "'r'"))
              (substitute* "Bio/SCOP/Cla.py"
                (("\"rU\"") "\"r\""))

              ;; ExactPosition class needs to define __str__.
              (substitute* "Bio/SeqFeature.py"
                (("class UncertainPosition" m)
                 (string-append "\
    # Must define this on Python 3.8 onwards because we redefine __repr__
    def __str__(self):
        return str(int(self))

    def __add__(self, offset):
        # By default preserve any subclass
        return self.__class__(int(self) + offset)
" m)))
              ;; This was removed in Python 3.9 already
              (substitute* "Bio/KEGG/KGML/KGML_parser.py"
                (("self.entry.getchildren\\(\\)")
                 "list(self.entry)")
                (("element.getchildren\\(\\)")
                 "list(element)"))
              (substitute* "Bio/Phylo/NeXMLIO.py"
                (("node.getchildren\\(\\)")
                 "list(node)")
                (("edge.getchildren\\(\\)")
                 "list(edge)"))
              (substitute* "Bio/Entrez/Parser.py"
                (("child.getiterator\\(\\)")
                 "list(child.iter())"))
              (substitute* "Bio/SeqIO/UniprotIO.py"
                (("alt_element.getiterator\\(NS \\+ ('.*')\\)" m something)
                 (string-append "list(alt_element.iter(NS + " something "))"))
                (("link_element.getiterator\\(NS \\+ ('.*')\\)" m something)
                 (string-append "list(link_element.iter(NS + " something "))"))
                (("loc_element.getiterator\\(NS \\+ ('.*')\\)" m something)
                 (string-append "list(loc_element.iter(NS + " something "))"))
                (("element.getiterator\\(NS \\+ ('.*')\\)" m something)
                 (string-append "list(element.iter(NS + " something "))")))))
          (add-after 'unpack 'numpy-compatibility
            (lambda _
              (substitute* '("Bio/Statistics/lowess.py"
                             "Tests/test_Cluster.py")
                (("numpy.float32") "float")
                (("numpy.float") "float"))))
          (add-before 'check 'set-home
            ;; Some tests require a home directory to be set.
            (lambda _ (setenv "HOME" "/tmp")))
          (add-before 'check 'build-extensions
            (lambda _
              ;; Cython extensions have to be built before running the tests.
              (invoke "python3" "setup.py" "build_ext" "--inplace")))
          (add-after 'unpack 'patch-tests
            (lambda _
              (substitute* "Tests/run_tests.py"
                ;; Do not run doctests.
                (("self.tests.append\\(\"doctest\"\\)") "")
                (("opt == \"--offline\"") "True")))))))))

(define-public python-scanrbp
  (package
    (name "python-scanrbp")
    (version "0.3")
    (home-page "https://github.com/grexor/scanrbp")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url home-page)
              (commit "cdc3587cfdb3bc1c68154ce8538b8c985099eb9f")))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1bv25qhr1dwym2j7llsd3ggnjb9l3h4bchng7bp7cq57s9g0bnjz"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f                       ; There are none.
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'check 'set-HOME
            (lambda _ (setenv "HOME" "/tmp")))
          (add-before 'check 'copy-data
            (lambda _
              (let ((data-dir (string-append (getenv "HOME") "/scanRBP_data"))
                    (data-file "data/data.tar.gz"))
                (mkdir-p data-dir)
                (copy-file
                 data-file
                 (string-append data-dir "/" (basename data-file)))))))))
    (propagated-inputs
     (list python-biopython
           python-matplotlib
           python-pybio
           python-scipy
           python-seaborn))
    (native-inputs
     (list python-setuptools))
    (synopsis "Tool for creating a RNA RBP heatmap in Python")
    (description "python-scanrbp is a Python package that provides the scanRBP
tool that loads RNA-protein binding motif PWM and computes the log-odds scores
for all the loaded RBPs across a given genomic sequence and draws a heatmap of
the scores.")
    (license license:gpl3)))

(define-public python-splicekit
  (package
    (name "python-splicekit")
    (version "0.7")
    (home-page "https://github.com/bedapub/splicekit")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url home-page)
              (commit "ded5dbec16b45e0df44750d9ae021ae2416ff921")))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "04l244qagbplksqp81w2s60pkymyhjq389xmqwsyc0n4q6b054h2"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list python-beautifulsoup4
           python-dateutil
           python-levenshtein
           python-logomaker
           python-numpy
           python-pandas
           python-plotly
           python-psutil
           python-pybio
           python-pysam
           python-rangehttpserver
           python-requests
           python-scanrbp))
    (native-inputs
     (list python-setuptools
           python-wheel
           snakemake))
    (arguments
     (list
      #:tests? #false                   ;There are no tests.
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'adjust-requirements
            (lambda _
              (substitute* "setup.py"
                ;; bs4 is an alternative name for beautifulsoup4, only used to
                ;; avoid name squatting on pypi.
                (("bs4") "beautifulsoup4"))))
          ;; fireducks seems to be a binary-only python-panda replacement
          (add-after 'unpack 'remove-fireducks
            (lambda _
              (substitute* '("Snakefile"
                             "splicekit/core/delta_dar.py"
                             "splicekit/core/juan.py"
                             "splicekit/core/motifs.py"
                             "splicekit/judge/__init__.py")
                (("import fireducks.pandas as pd") "import pandas as pd"))
              (substitute* "splicekit.yaml"
                ((".*fireducks.*") "")))))))
    (synopsis "Python toolkit for splicing analysis from short-read RNA-seq")
    (description "Splicekit is a modular platform for splicing analysis from
short-read RNA-seq datasets.  The platform also integrates pybio for genomic
operations and scanRBP for RNA-protein binding studies.  The whole analysis is
self-contained (one single directory) and the platform is written in Python,
in a modular way.")
    (license license:gpl3)))

(define-public python-presto
  (package
    (name "python-presto")
    (version "0.7.6")
    (home-page "https://github.com/immcantation/presto")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url home-page)
              (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1mcngwrxiw8r1j26r5crf7j0dscvhg3b8g1is3j3vq5jpnyn8jmz"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list muscle
           python-biopython
           python-numpy
           python-pandas
           python-pyyaml
           python-scipy
           vsearch))
    (native-inputs
     (list python-pytest
           python-setuptools
           python-wheel))
    (arguments
     (list
      #:build-backend "setuptools.build_meta"
      #:test-flags
      ;; FileNotFoundError: [Errno 2] No such file or directory: 'cd-hit-est'
      '(list "--ignore=tests/test_ClusterSets.py")))
    (synopsis "The REpertoire Sequencing TOolkit")
    (description "Presto is a python toolkit for processing raw reads from
high-throughput sequencing of B cell and T cell repertoires.")
    (license license:agpl3)))

(define-public python-changeo
  (package
    (name "python-changeo")
    (version "1.3.4")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "changeo" version))
       (sha256
        (base32 "0jm4chddpl929a8daicp9mv14nwzf2hajpjfhzckg6x8cpgc1087"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:build-backend "setuptools.build_meta"))
    (native-inputs
     (list python-pytest
           python-setuptools))
    (propagated-inputs
     (list python-airr
           python-biopython
           python-importlib-resources
           python-numpy
           python-packaging
           python-pandas
           python-presto
           python-pyyaml
           python-scipy))
    (home-page "http://changeo.readthedocs.io")
    (synopsis "Repertoire clonal assignment toolkit")
    (description "Change-O is a collection of tools for processing the output
of V(D)J alignment tools, assigning clonal clusters to immunoglobulin (Ig)
sequences, and reconstructing germline sequences.")
    (license license:gpl3)))

(define-public python-fastalite
  (package
    (name "python-fastalite")
    (version "0.4.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/nhoffman/fastalite")
              (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "19sps1l8k2vp7sj943gccabfkr2h3fn1n93sn28h9gay2ywc41ly"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-backend #~'unittest
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'set-version
            (lambda _
              ;; This version file is expected to be created with git describe.
              (mkdir-p "fastalite/data")
              (with-output-to-file "fastalite/data/ver"
                (lambda () (display #$version))))))))
    (native-inputs
     (list python-setuptools))
    (home-page "https://github.com/nhoffman/fastalite")
    (synopsis "Simplest possible FASTA parser")
    (description "This library implements a FASTA and a FASTQ parser without
relying on a complex dependency tree.")
    (license license:expat)))

(define-public biosoup
  (package
    (name "biosoup")
    (version "0.10.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/rvaser/biosoup")
             ;; Corresponds to version 0.10.0
             (commit "38181f09854ff42cbd9632200a2ec9fb37a4b7b6")))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "02hvyka703zagx0nvv2yx3dkc748zc8g6qbrpya7r8kfkcl7y8hw"))))
    (build-system cmake-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (invoke "./bin/biosoup_test")))))))
    (native-inputs
     (list googletest))
    (home-page "https://github.com/rvaser/biosoup")
    (synopsis "C++ support library for bioinformatics tools")
    (description "Biosoup is a C++ collection of header-only data structures
used for storage and logging in bioinformatics tools.")
    (license license:expat)))

(define-public bioparser
  (package
    (name "bioparser")
    (version "3.0.13")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/rvaser/bioparser")
             ;; Corresponds to tag 3.0.13
             (commit "13341e6e0855c6b358ffcea6dad216e1009e1287")))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0c5p2dl8jb12ci9f427jzrmmm9cgvc1k4fxsn2ggkfsin6r1r82i"))))
    (build-system cmake-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (invoke "./bin/bioparser_test")))))))
    (inputs
     (list biosoup))
    (propagated-inputs
     (list zlib))
    (native-inputs
     (list googletest))
    (home-page "https://github.com/rvaser/bioparser")
    (synopsis "C++ library for parsing several formats in bioinformatics")
    (description "Bioparser is a C++ header only parsing library for several
bioinformatics formats (FASTA/Q, MHAP/PAF/SAM), with support for zlib
compressed files.")
    (license license:expat)))

(define-public circtools
  (package
    (name "circtools")
    (version "1.0.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/Kevinzjy/circtools")
             ;; Corresponds to tag v1.0.0
             (commit "79380de59013601021ca3b1352d6f64d2fb89646")
             (recursive? #t)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0wg1s927g32k25j967kfr8l30nmr4c0p4zvy5igvy7cs6chd60lh"))))
    (build-system cargo-build-system)
    (arguments
     `(#:tests? #f
       #:install-source? #f
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'make-writable
           (lambda _
             (for-each make-file-writable (find-files "."))))
         (add-after 'unpack 'prepare-spoa-dependencies
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "vendor/spoa/CMakeLists.txt"
               (("find_package\\(bioparser 3.0.13 QUIET\\)")
                "find_package(bioparser 3.0.13 CONFIG)")
               (("find_package\\(biosoup 0.10.0 QUIET\\)")
                "find_package(biosoup 0.10.0 CONFIG)")
               (("GTest_FOUND") "TRUE")))))))
    (inputs
     (cons* bioparser biosoup (cargo-inputs 'circtools)))
    (native-inputs
     (list cmake-minimal pkg-config googletest))
    (home-page "https://github.com/Kevinzjy/circtools")
    (synopsis "Accelerating functions in CIRI toolkit")
    (description "This package provides accelerated functions for the CIRI
toolkit.  It also provides the @code{ccs} executable to scan for circular
consensus sequences.")
    (license license:expat)))

(define-public ciri-long
  (package
    (name "ciri-long")
    (version "1.0.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/bioinfo-biols/CIRI-long")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "10k88i1fcqchrrjv82rmylwvbwqfba0n51palhig9hsg71xs0dbi"))
       ;; Delete bundled binary
       (snippet '(delete-file "libs/ccs"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'relax-requirements
            (lambda _
              (substitute* "setup.py"
                (("'argparse[^']*',")
                 "") ;only for python2
                (("==")
                 ">=")
                ;; This package changed names.
                (("python-Levenshtein")
                 "levenshtein"))))
          (add-before 'build 'build-libssw
            (lambda _
              (with-directory-excursion "libs/striped_smith_waterman"
                (invoke "make" "libssw.so"))))
          (add-before 'build 'fix-reference-to-ccs
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "CIRI_long/pipeline.py"
                (("'ccs -i")
                 (string-append "'"
                                (assoc-ref inputs "circtools") "/bin/ccs"
                                " -i")))
              ;; yuck!
              (substitute* "CIRI_long/main.py"
                (("os.chmod\\(lib_path.*")
                 "")))))))
    (inputs (list circtools
                  python-biopython
                  python-bwapy
                  python-levenshtein
                  python-mappy
                  python-numpy
                  python-pandas
                  python-pysam
                  python-pyspoa
                  python-scikit-learn
                  python-scipy))
    (native-inputs (list python-cython python-pynose python-setuptools))
    (home-page "https://ciri-cookbook.readthedocs.io/")
    (synopsis "Circular RNA identification for Nanopore sequencing")
    (description "CIRI-long is a package for circular RNA identification using
long-read sequencing data.")
    (license license:expat)))

(define-public python-circe
  (package
    (name "python-circe")
    (version "0.3.8")
    (home-page "https://github.com/cantinilab/circe")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url home-page)
              (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "050zwg6qvd6sa4xd41sn4gigwmyfn0v6r93j5kvpbxklq4144q15"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list python-anndata
           python-dask
           python-distributed
           python-joblib
           python-numpy
           python-pandas
           python-rich
           python-scanpy
           python-scikit-learn))
    (inputs
     (list lapack openblas))
    (native-inputs
     (list python-cython
           python-pybiomart
           python-setuptools))
    (arguments
     (list
      #:tests? #f       ;XXX: tests hangs during collection
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable dir to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _ (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (synopsis "Cis-regulatory interactions between chromatin regions")
    (description "Circe is a Python package for inferring co-accessibility
networks from single-cell ATAC-seq data, using skggm for the graphical lasso
and python-scanpy for data processing.")
    (license license:gpl3)))

(define-public qtltools
  (package
    (name "qtltools")
    (version "1.3.1")
    (source (origin
              (method url-fetch/tarbomb)
              (uri (string-append "https://qtltools.github.io/qtltools/"
                                  "binaries/QTLtools_" version
                                  "_source.tar.gz"))
              (sha256
               (base32
                "13gdry5l43abn3464fmk8qzrxgxnxah2612r66p9dzhhl92j30cd"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; no tests included
       #:make-flags
       ,#~(list (string-append "BOOST_INC="
                               #$(this-package-input "boost") "/include")
                (string-append "BOOST_LIB="
                               #$(this-package-input "boost") "/lib")
                (string-append "HTSLD_INC="
                               #$(this-package-input "htslib") "/include")
                (string-append "HTSLD_LIB="
                               #$(this-package-input "htslib") "/lib")
                (string-append "RMATH_INC="
                               #$(this-package-input "rmath-standalone")
                               "/include")
                (string-append "RMATH_LIB="
                               #$(this-package-input "rmath-standalone")
                               "/lib"))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'fix-linkage
           (lambda _
             (substitute* "qtltools/Makefile"
               (("libboost_iostreams.a")
                "libboost_iostreams.so")
               (("libboost_program_options.a")
                "libboost_program_options.so")
               (("-lblas") "-lopenblas"))))
         (add-before 'build 'chdir
           (lambda _ (chdir "qtltools")))
         (replace 'configure
           (lambda _
             (substitute* "qtltools/Makefile"
               (("LIB_FLAGS=-lz")
                "LIB_FLAGS=-lz -lcrypto -lssl")
               (("LIB_FILES=\\$\\(RMATH_LIB\\)/libRmath.a \
\\$\\(HTSLD_LIB\\)/libhts.a \
\\$\\(BOOST_LIB\\)/libboost_iostreams.a \
\\$\\(BOOST_LIB\\)/libboost_program_options.a")
                "LIB_FILES=$(RMATH_LIB)/libRmath.so \
$(HTSLD_LIB)/libhts.so \
$(BOOST_LIB)/libboost_iostreams.so \
$(BOOST_LIB)/libboost_program_options.so"))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
               (mkdir-p bin)
               (install-file "bin/QTLtools" bin)))))))
    (inputs
     (list curl
           gsl
           boost
           rmath-standalone
           htslib-1.3
           openssl
           openblas
           zlib))
    (home-page "https://qtltools.github.io/qtltools/")
    (synopsis "Tool set for molecular QTL discovery and analysis")
    (description "QTLtools is a tool set for molecular QTL discovery
and analysis.  It allows going from the raw genetic sequence data to
collection of molecular @dfn{Quantitative Trait Loci} (QTLs) in few
easy-to-perform steps.")
    (license license:gpl3+)))

(define-public bpp-core
  (package
    (name "bpp-core")
    (version "2.4.1")
    (source
      (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/BioPP/bpp-core")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0ma2cl677l7s0n5sffh66cy9lxp5wycm50f121g8rx85p95vkgwv"))))
    (build-system cmake-build-system)
    ;; XXX: does not build with later GCC:
    ;; error: ‘numeric_limits’ was not declared in this scope
    (native-inputs (list gcc-10))
    (home-page "https://pbil.univ-lyon1.fr/bpp-doc/bpp-core/html/index.html")
    (synopsis "C++ libraries for Bioinformatics")
    (description
     "Bio++ is a set of C++ libraries for Bioinformatics, including sequence
analysis, phylogenetics, molecular evolution and population genetics.  It is
Object Oriented and is designed to be both easy to use and computer efficient.
Bio++ intends to help programmers to write computer expensive programs, by
providing them a set of re-usable tools.")
      (license license:cecill-c)))

(define-public bpp-phyl
  (package
    (name "bpp-phyl")
    (version "2.4.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BioPP/bpp-phyl")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "192zks6wyk903n06c2lbsscdhkjnfwms8p7jblsmk3lvjhdipb20"))))
    (build-system cmake-build-system)
    (inputs
     (list bpp-core bpp-seq))
    (home-page "https://pbil.univ-lyon1.fr/bpp-doc/bpp-phyl/html/")
    (synopsis "Bio++ phylogenetic library")
    (description
     "Bio++ is a set of C++ libraries for Bioinformatics, including sequence
analysis, phylogenetics, molecular evolution and population genetics.  This
library provides phylogenetics-related modules.")
    (license license:cecill-c)))

(define-public bpp-phyl-omics
  (package
    (name "bpp-phyl-omics")
    (version "2.4.1")
    (source
      (origin
       (method git-fetch)
       (uri (git-reference
            (url "https://github.com/BioPP/bpp-phyl-omics")
            (commit (string-append "v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "172psb8njkjwg3cd6gdy5w0mq8f0817v635yw4bk7146aggjzl1h"))))
    (build-system cmake-build-system)
    (arguments
     (list #:tests? #f)) ; No test provided.
    (inputs
     (list bpp-core
           bpp-phyl
           bpp-seq
           bpp-seq-omics))
    (home-page "https://github.com/BioPP/bpp-phyl-omics")
    (synopsis "Bio++ phylogenetic library genomics components")
    (description
     "This library contains the genomics components of the Bio++ phylogenetics
library.  It is part of the Bio++ project.")
    (license license:cecill)))

(define-public bpp-popgen
  (package
    (name "bpp-popgen")
    (version "2.4.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BioPP/bpp-popgen")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bz0fhrq3dri6a0hvfc3zlvrns8mrzzlnicw5pyfa812gc1qwfvh"))))
    (build-system cmake-build-system)
    (arguments
     (list #:tests? #f)) ; There are no tests.
    (inputs
     (list bpp-core bpp-seq))
    (home-page "https://pbil.univ-lyon1.fr/bpp-doc/bpp-popgen/html/")
    (synopsis "Bio++ population genetics library")
    (description
     "Bio++ is a set of C++ libraries for Bioinformatics, including sequence
analysis, phylogenetics, molecular evolution and population genetics.  This
library provides population genetics-related modules.")
    (license license:cecill-c)))

(define-public bpp-seq
  (package
    (name "bpp-seq")
    (version "2.4.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BioPP/bpp-seq")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32 "1mc09g8jswzsa4wgrfv59jxn15ys3q8s0227p1j838wkphlwn2qk"))))
    (build-system cmake-build-system)
    (inputs
     (list bpp-core))
    (home-page "https://pbil.univ-lyon1.fr/bpp-doc/bpp-seq/html/")
    (synopsis "Bio++ sequence library")
    (description
     "Bio++ is a set of C++ libraries for Bioinformatics, including sequence
analysis, phylogenetics, molecular evolution and population genetics.  This
library provides sequence-related modules.")
    (license license:cecill-c)))

(define-public bpp-seq-omics
  (package
    (name "bpp-seq-omics")
    (version "2.4.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BioPP/bpp-seq-omics")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32 "1sc2xdfnfp5a6qihplp49rgrqmj89898avfy9bqaq1g2fajppgjj"))))
    (build-system cmake-build-system)
    (inputs
     (list bpp-core bpp-seq))
    (home-page "https://github.com/BioPP/bpp-seq-omics")
    (synopsis "Bio++ sequence library genomics components")
    (description
     "This library contains the genomics components of the Bio++ sequence library.
It is part of the Bio++ project.")
    (license license:cecill)))

(define-public bppsuite
  (package
    (name "bppsuite")
    (version "2.4.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BioPP/bppsuite")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wdwcgczqbc3m116vakvi0129wm3acln3cfc7ivqnalwvi6lrpds"))))
    (build-system cmake-build-system)
    (arguments
     (list #:tests? #f)) ; There are no tests.
    (native-inputs
     (list groff man-db texinfo))
    (inputs
     (list bpp-core bpp-seq bpp-phyl bpp-popgen))
    (home-page "https://github.com/BioPP")
    (synopsis "Bioinformatics tools written with the Bio++ libraries")
    (description
     "Bio++ is a set of C++ libraries for Bioinformatics, including sequence
analysis, phylogenetics, molecular evolution and population genetics.  This
package provides command line tools using the Bio++ library.")
    (license license:cecill-c)))

(define-public blast+
  (package
    (name "blast+")
    (version "2.17.0")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://ftp.ncbi.nlm.nih.gov/blast/executables/blast+/"
                    version "/ncbi-blast-" version "+-src.tar.gz"))
              (sha256
               (base32
                "0ix1bqgaywi3ggis4qvaipb0mk3llhgf52vmc97f744risl5f82h"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  ;; Remove bundled bzip2, zlib and pcre.
                  (delete-file-recursively "c++/src/util/compress/bzip2")
                  (delete-file-recursively "c++/src/util/compress/zlib")
                  (delete-file-recursively "c++/src/util/regexp")
                  (substitute* "c++/src/util/compress/Makefile.in"
                    (("bzip2 zlib api") "api"))
                  ;; Remove useless msbuild directory
                  (delete-file-recursively
                   "c++/src/build-system/project_tree_builder/msbuild")

                  ;; Build reproducibly.
                  ;; Do not record the kernel version
                  (substitute* "c++/src/build-system/configure"
                    (("kver=.*") "kver=\"\""))
                  ;; Do not generate random numbers.
                  (substitute* "c++/scripts/common/impl/define_random_macros.sh"
                    (("#define NCBI_RANDOM_VALUE_MAX  0xffffffffu" m)
                     (string-append m "
#define NCBI_RANDOM_VALUE_0    2845495105u
#define NCBI_RANDOM_VALUE_1    2158634051u
#define NCBI_RANDOM_VALUE_2    4072202242u
#define NCBI_RANDOM_VALUE_3    902228395u
#define NCBI_RANDOM_VALUE_4    1353323915u
#define NCBI_RANDOM_VALUE_5    574823513u
#define NCBI_RANDOM_VALUE_6    4119501261u
#define NCBI_RANDOM_VALUE_7    2477640938u
#define NCBI_RANDOM_VALUE_8    2776595395u
#define NCBI_RANDOM_VALUE_9    270550684u
"))
                    (("cksum") "cksum >/dev/null"))))))
    (build-system gnu-build-system)
    (arguments
     (list
      ;; There are two(!) tests for this massive library, and both fail with
      ;; "unparsable timing stats".
      ;; ERR [127] --  [serial/datatool] datatool.sh     (unparsable timing stats)
      ;; ERR [127] --  [serial/datatool] datatool_xml.sh     (unparsable timing stats)
      #:tests? #f
      #:out-of-source? #t
      #:parallel-build? #f              ;not supported
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'configure 'set-HOME
            ;; $HOME needs to be set at some point during the configure phase
            (lambda _ (setenv "HOME" "/tmp")))
          (add-after 'unpack 'enter-dir
            (lambda _ (chdir "c++")))
          (add-after 'enter-dir 'fix-build-system
            (lambda _
              (define (which* cmd)
                (cond ((string=? cmd "date")
                       ;; make call to "date" deterministic
                       "date -d @0")
                      ((which cmd)
                       => identity)
                      (else
                       (format (current-error-port)
                               "WARNING: Unable to find absolute path for ~s~%"
                               cmd)
                       #false)))

              ;; Rewrite hardcoded paths to various tools
              (substitute* (append '("src/build-system/configure.ac"
                                     "src/build-system/configure"
                                     "src/build-system/helpers/run_with_lock.c"
                                     "scripts/common/impl/if_diff.sh"
                                     "scripts/common/impl/run_with_lock.sh"
                                     "src/build-system/Makefile.configurables.real"
                                     "src/build-system/Makefile.in.top"
                                     "src/build-system/Makefile.meta.gmake=no"
                                     "src/build-system/Makefile.meta.in"
                                     "src/build-system/Makefile.meta_l"
                                     "src/build-system/Makefile.meta_p"
                                     "src/build-system/Makefile.meta_r"
                                     "src/build-system/Makefile.mk.in"
                                     "src/build-system/Makefile.requirements"
                                     "src/build-system/Makefile.rules_with_autodep.in")
                                   (find-files "scripts/common/check" "\\.sh$"))
                (("(/usr/bin/|/bin/)([a-z][-_.a-z]*(\\+\\+)?)" all dir cmd)
                 (or (which* cmd) all)))

              (substitute* (find-files "src/build-system" "^config.*")
                (("LN_S=/bin/\\$LN_S") (string-append "LN_S=" (which "ln")))
                (("^PATH=.*") ""))

              ;; rewrite "/var/tmp" in check script
              (substitute* "scripts/common/check/check_make_unix.sh"
                (("/var/tmp") "/tmp"))

              ;; do not reset PATH
              (substitute* (find-files "scripts/common/impl/" "\\.sh$")
                (("^ *PATH=.*") "")
                (("action=/bin/") "action=")
                (("export PATH") ":"))))
          (replace 'configure
            (lambda _
              (let ((lib     (string-append #$output:lib "/lib"))
                    (include (string-append #$output:include
                                            "/include/ncbi-tools++")))
                ;; The 'configure' script doesn't recognize things like
                ;; '--enable-fast-install'.
                (invoke "./configure.orig"
                        (string-append "--with-build-root=" (getcwd) "/build")
                        (string-append "--prefix=" #$output)
                        (string-append "--libdir=" lib)
                        (string-append "--includedir=" include)
                        (string-append "--with-bz2="
                                       #$(this-package-input "bzip2"))
                        (string-append "--with-z="
                                       #$(this-package-input "zlib"))
                        (string-append "--with-pcre="
                                       #$(this-package-input "pcre"))
                        ;; Each library is built twice by default, once
                        ;; with "-static" in its name, and again
                        ;; without.
                        "--without-static"
                        "--with-dll")))))))
    (outputs '("out"       ;  21 MB
               "lib"       ; 226 MB
               "include")) ;  33 MB
    (inputs
     (list bzip2
           lmdb
           zlib
           pcre
           perl
           python-wrapper))
    (native-inputs
     (list cpio))
    (home-page "https://blast.ncbi.nlm.nih.gov")
    (synopsis "Basic local alignment search tool")
    (description
     "BLAST is a popular method of performing a DNA or protein sequence
similarity search, using heuristics to produce results quickly.  It also
calculates an “expect value” that estimates how many matches would have
occurred at a given score by chance, which can aid a user in judging how much
confidence to have in an alignment.")
    ;; Most of the sources are in the public domain, with the following
    ;; exceptions:
    ;;   * Expat:
    ;;     * ./c++/include/util/bitset/
    ;;     * ./c++/src/html/ncbi_menu*.js
    ;;   * Boost license:
    ;;     * ./c++/include/util/impl/floating_point_comparison.hpp
    ;;   * LGPL 2+:
    ;;     * ./c++/include/dbapi/driver/odbc/unix_odbc/
    ;;   * ASL 2.0:
    ;;     * ./c++/src/corelib/teamcity_*
    (license (list license:public-domain
                   license:expat
                   license:boost1.0
                   license:lgpl2.0+
                   license:asl2.0))))

(define-public bless
  (package
    (name "bless")
    (version "1p02")
    (source (origin
              (method url-fetch)
              (uri (string-append "mirror://sourceforge/bless-ec/bless.v"
                                  version ".tgz"))
              (sha256
               (base32
                "0rm0gw2s18dqwzzpl3c2x1z05ni2v0xz5dmfk3d33j6g4cgrlrdd"))
              (modules '((guix build utils)))
              (snippet
               `(begin
                  ;; Remove bundled boost, pigz, zlib, and .git directory
                  ;; FIXME: also remove bundled sources for murmurhash3 and
                  ;; kmc once packaged.
                  (delete-file-recursively "boost")
                  (delete-file-recursively "pigz")
                  (delete-file-recursively "google-sparsehash")
                  (delete-file-recursively "zlib")
                  (delete-file-recursively ".git")))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no "check" target
       #:make-flags
       ,#~(list (string-append "ZLIB="
                               #$(this-package-input "zlib")
                               "/lib/libz.so")
                (string-append "LDFLAGS="
                               (string-join '("-lboost_filesystem"
                                              "-lboost_system"
                                              "-lboost_iostreams"
                                              "-lz"
                                              "-fopenmp"))))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'do-not-build-bundled-pigz
          (lambda* (#:key inputs outputs #:allow-other-keys)
            (substitute* "Makefile"
              (("cd pigz/pigz-2.3.3; make") ""))))
         (add-after 'unpack 'patch-paths-to-executables
          (lambda* (#:key inputs outputs #:allow-other-keys)
            (substitute* "parse_args.cpp"
              (("kmc_binary = .*")
               (string-append "kmc_binary = \""
                              (assoc-ref outputs "out")
                              "/bin/kmc\";"))
              (("pigz_binary = .*")
               (string-append "pigz_binary = \""
                              (assoc-ref inputs "pigz")
                              "/bin/pigz\";")))))
         (replace 'install
          (lambda* (#:key outputs #:allow-other-keys)
            (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
              (for-each (lambda (file)
                          (install-file file bin))
                        '("bless" "kmc/bin/kmc")))))
         (delete 'configure))))
    (native-inputs
     (list perl))
    (inputs
     (list openmpi boost-1.83 sparsehash pigz zlib))
    (supported-systems '("x86_64-linux"))
    (home-page "https://sourceforge.net/p/bless-ec/wiki/Home/")
    (synopsis "Bloom-filter-based error correction tool for NGS reads")
    (description
     "@dfn{Bloom-filter-based error correction solution for high-throughput
sequencing reads} (BLESS) uses a single minimum-sized bloom filter is a
correction tool for genomic reads produced by @dfn{Next-generation
sequencing} (NGS).  BLESS produces accurate correction results with much less
memory compared with previous solutions and is also able to tolerate a higher
false-positive rate.  BLESS can extend reads like DNA assemblers to correct
errors at the end of reads.")
    (license license:gpl3+)))

(define-public bowtie
  (package
    (name "bowtie")
    (version "2.3.4.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BenLangmead/bowtie2")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zl3cf327y2p7p03cavymbh7b00djc7lncfaqih33n96iy9q8ibp"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  (substitute* "Makefile"
                    ;; replace BUILD_HOST and BUILD_TIME for deterministic build
                    (("-DBUILD_HOST=.*") "-DBUILD_HOST=\"\\\"guix\\\"\"")
                    (("-DBUILD_TIME=.*") "-DBUILD_TIME=\"\\\"0\\\"\""))))))
    (build-system gnu-build-system)
    (arguments
     `(#:make-flags
       ,#~(list "allall"
                "WITH_TBB=1"
                (string-append "prefix=" #$output))
       #:phases
       (modify-phases %standard-phases
         (replace 'configure
           (lambda _
             ;; This "extended character" is not considered valid.
             (substitute* "processor_support.h"
               (("“") "\"")
               (("”") "\""))))
         (replace 'check
           (lambda _
             (invoke "perl"
                     "scripts/test/simple_tests.pl"
                     "--bowtie2=./bowtie2"
                     "--bowtie2-build=./bowtie2-build"))))))
    (inputs
     `(("tbb" ,tbb-2020)
       ("zlib" ,zlib)
       ("python" ,python-wrapper)))
    (native-inputs
     (list perl perl-clone perl-test-deep perl-test-simple))
    (home-page "https://bowtie-bio.sourceforge.net/bowtie2/index.shtml")
    (synopsis "Fast and sensitive nucleotide sequence read aligner")
    (description
     "Bowtie 2 is a fast and memory-efficient tool for aligning sequencing
reads to long reference sequences.  It is particularly good at aligning reads
of about 50 up to 100s or 1,000s of characters, and particularly good at
aligning to relatively long (e.g. mammalian) genomes.  Bowtie 2 indexes the
genome with an FM Index to keep its memory footprint small: for the human
genome, its memory footprint is typically around 3.2 GB.  Bowtie 2 supports
gapped, local, and paired-end alignment modes.")
    (supported-systems '("x86_64-linux"))
    (license license:gpl3+)))

(define-public bowtie1
  (package
    (name "bowtie1")
    (version "1.3.1")
    (source (origin
              (method url-fetch)
              (uri (string-append "mirror://sourceforge/bowtie-bio/bowtie/"
                                  version "/bowtie-" version "-src.zip"))
              (sha256
               (base32
                "0q87nhgj9wrnbazcpvqp4594hmyh1isi3s9b2wlghvl4afm1fdg2"))
              (modules '((guix build utils)))
              (snippet
               '(substitute* "Makefile"
                  ;; replace BUILD_HOST and BUILD_TIME for deterministic build
                  (("-DBUILD_HOST=.*") "-DBUILD_HOST=\"\\\"guix\\\"\"")
                  (("-DBUILD_TIME=.*") "-DBUILD_TIME=\"\\\"0\\\"\"")))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; Tests need various perl modules
       #:test-target "simple-test"
       #:make-flags
       ,#~(append #$(if (not (target-x86?))
                        #~'("POPCNT_CAPABILITY=0")
                        #~'())
                  (list (string-append "CC=" #$(cc-for-target))
                        (string-append "CXX=" #$(cxx-for-target))
                        "all"
                        (string-append "prefix=" #$output)))
       #:phases
       (modify-phases %standard-phases
         (delete 'configure))))
    (inputs
     (list python-wrapper tbb zlib))
    (supported-systems %64bit-supported-systems)
    (home-page "https://bowtie-bio.sourceforge.net/index.shtml")
    (synopsis "Fast aligner for short nucleotide sequence reads")
    (description
     "Bowtie is a fast, memory-efficient short read aligner.  It aligns short
DNA sequences (reads) to the human genome at a rate of over 25 million 35-bp
reads per hour.  Bowtie indexes the genome with a Burrows-Wheeler index to
keep its memory footprint small: typically about 2.2 GB for the human
genome (2.9 GB for paired-end).")
    (license license:artistic2.0)))

(define-public tophat
  (package
    (name "tophat")
    (version "2.1.1")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "http://ccb.jhu.edu/software/tophat/downloads/tophat-"
                    version ".tar.gz"))
              (sha256
               (base32
                "19add02kv2xhd6ihd779dr7x35ggym3jqr0m5c4315i1yfb0p11p"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  ;; Remove bundled SeqAn and samtools
                  (delete-file-recursively "src/SeqAn-1.4.2")
                  (delete-file-recursively "src/samtools-0.1.18")))))
    (build-system gnu-build-system)
    (arguments
     '(#:parallel-build? #f             ; not supported
       #:phases
       (modify-phases %standard-phases
         (add-after 'set-paths 'hide-default-gcc
           (lambda* (#:key inputs #:allow-other-keys)
             (let ((gcc (assoc-ref inputs "gcc")))
               ;; Remove the default GCC from CPLUS_INCLUDE_PATH to prevent
               ;; conflicts with the GCC 5 input.
               (setenv "CPLUS_INCLUDE_PATH"
                       (string-join
                        (delete (string-append gcc "/include/c++")
                                (string-split (getenv "CPLUS_INCLUDE_PATH") #\:))
                        ":")))))
         (add-after 'unpack 'use-system-samtools
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "src/Makefile.in"
               (("(noinst_LIBRARIES = )\\$\\(SAMLIB\\)" _ prefix) prefix)
               (("\\$\\(SAMPROG\\): \\$\\(SAMLIB\\)") "")
               (("SAMPROG = samtools_0\\.1\\.18") "")
               (("\\$\\(samtools_0_1_18_SOURCES\\)") "")
               (("am__EXEEXT_1 = samtools_0\\.1\\.18\\$\\(EXEEXT\\)") ""))
             (substitute* '("src/common.cpp"
                            "src/tophat.py")
               (("samtools_0.1.18") (which "samtools")))
             (substitute* '("src/common.h"
                            "src/bam2fastx.cpp")
               (("#include \"bam.h\"") "#include <samtools/bam.h>")
               (("#include \"sam.h\"") "#include <samtools/sam.h>"))
             (substitute* '("src/bwt_map.h"
                            "src/map2gtf.h"
                            "src/align_status.h")
               (("#include <bam.h>") "#include <samtools/bam.h>")
               (("#include <sam.h>") "#include <samtools/sam.h>")))))))
    (native-inputs
     `(("gcc@5" ,gcc-5))) ;; doesn't build with later versions
    (inputs
     `(("boost" ,boost)
       ("bowtie" ,bowtie)
       ("ncurses" ,ncurses)
       ("perl" ,perl)
       ("python" ,python-2)
       ("samtools" ,samtools-0.1)
       ("seqan" ,seqan-1)
       ("zlib" ,zlib)))
    (home-page "https://ccb.jhu.edu/software/tophat/index.shtml")
    (synopsis "Spliced read mapper for RNA-Seq data")
    (description
     "TopHat is a fast splice junction mapper for nucleotide sequence
reads produced by the RNA-Seq method.  It aligns RNA-Seq reads to
mammalian-sized genomes using the ultra high-throughput short read
aligner Bowtie, and then analyzes the mapping results to identify
splice junctions between exons.")
    ;; TopHat is released under the Boost Software License, Version 1.0
    ;; See https://github.com/infphilo/tophat/issues/11#issuecomment-121589893
    (license license:boost1.0)))

(define-public bwa
  (package
    (name "bwa")
    (version "0.7.18")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/lh3/bwa")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1vf3iwkzxqkzhcfz2q3qyvcv3jrvbb012qy21pfgjl8lv20ywfr1"))))
    (build-system gnu-build-system)
    (arguments
     (list #:tests? #f ;no "check" target
           #:make-flags #~(list "CFLAGS=-fcommon"
                                (string-append "CC=" #$(cc-for-target)))
           #:phases
           #~(modify-phases %standard-phases
               (replace 'install
                 (lambda* (#:key outputs #:allow-other-keys)
                   (let* ((out (assoc-ref outputs "out"))
                          (bin (string-append out "/bin"))
                          (lib (string-append out "/lib"))
                          (doc (string-append out "/share/doc/bwa"))
                          (man (string-append out "/share/man/man1")))
                     (install-file "bwa" bin)
                     (install-file "libbwa.a" lib)
                     (install-file "README.md" doc)
                     (install-file "bwa.1" man))))
               ;; no "configure" script
               (delete 'configure))))
    (inputs (list zlib))
    (supported-systems '("x86_64-linux" "aarch64-linux"))
    (home-page "https://bio-bwa.sourceforge.net/")
    (synopsis "Burrows-Wheeler sequence aligner")
    (description
     "BWA is a software package for mapping low-divergent sequences against a
large reference genome, such as the human genome.  It consists of three
algorithms: BWA-backtrack, BWA-SW and BWA-MEM.  The first algorithm is
designed for Illumina sequence reads up to 100bp, while the rest two for
longer sequences ranged from 70bp to 1Mbp.  BWA-MEM and BWA-SW share similar
features such as long-read support and split alignment, but BWA-MEM, which is
the latest, is generally recommended for high-quality queries as it is faster
and more accurate.  BWA-MEM also has better performance than BWA-backtrack for
70-100bp Illumina reads.")
    (license license:gpl3+)))

(define-public bwa-pssm
  (package
    (inherit bwa)
    (name "bwa-pssm")
    (version "0.5.11")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/pkerpedjiev/bwa-pssm")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "076c4q0cdqz8jgylb067y9zmvxglppnzi3qiscn0xiypgc6lgb5r"))))
    (build-system gnu-build-system)
    (arguments
     (substitute-keyword-arguments (package-arguments bwa)
       ((#:phases phases #~%standard-phases)
        #~(modify-phases #$phases
            (add-after 'unpack 'patch-C-error
              (lambda _
                (substitute* "pssm.c"
                  (("inline int map") "int map"))))))))
    (inputs
     (list gdsl zlib perl))
    ;; https://bwa-pssm.binf.ku.dk is down, and all Web Archived copies are
    ;; blank (they actually have "display:none" for some nefarious reason).
    (home-page "https://github.com/pkerpedjiev/bwa-pssm")
    (synopsis "Burrows-Wheeler transform-based probabilistic short read mapper")
    (description
     "BWA-PSSM is a probabilistic short genomic sequence read aligner based on
the use of @dfn{position specific scoring matrices} (PSSM).  Like many of the
existing aligners it is fast and sensitive.  Unlike most other aligners,
however, it is also adaptible in the sense that one can direct the alignment
based on known biases within the data set.  It is coded as a modification of
the original BWA alignment program and shares the genome index structure as
well as many of the command line options.")
    (license license:gpl3+)))

(define-public bwa-meth
  (package
    (name "bwa-meth")
    (version "0.2.9")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/brentp/bwa-meth")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0192h6rdaxa1rx16hgkanwsp3qv9knsmnghy07ya1231qia8h67p"))))
    (build-system pyproject-build-system)
    (arguments
     `(#:tests? #f      ;no tests
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'keep-references-to-bwa
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "bwameth.py"
               (("bwa (mem|index)" _ command)
                (string-append (which "bwa") " " command))
               ;; There's an ill-advised check for "samtools" on PATH.
               (("^checkX.*") "")))))))
    (native-inputs
     (list python-setuptools))
    (inputs
     (list bwa
           python-toolshed))
    (home-page "https://github.com/brentp/bwa-meth")
    (synopsis "Fast and accurante alignment of BS-Seq reads")
    (description
     "BWA-Meth works for single-end reads and for paired-end reads from the
directional protocol (most common).  It uses the method employed by
methylcoder and Bismark of in silico conversion of all C's to T's in both
reference and reads.  It recovers the original read (needed to tabulate
methylation) by attaching it as a comment which BWA appends as a tag to the
read.  It performs favorably to existing aligners gauged by number of on and
off-target reads for a capture method that targets CpG-rich region.")
    (license license:expat)))

(define-public python-bx-python
  (package
    (name "python-bx-python")
    (version "0.14.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/bxlab/bx-python")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1031ddndi5anxk76h1irdwa6nwr5sy20p816zn5jz50010zc562r"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-before 'check 'build-extensions
           (lambda _
             ;; Cython extensions have to be built before running the tests.
             (invoke "python" "setup.py" "build_ext" "--inplace")))
         ;; Unrecognized argument.
         (add-after 'unpack 'disable-cython-doctests
           (lambda _ (substitute* "pytest.ini" (("--doctest-cython") "")))))))
    (propagated-inputs
     (list python-numpy
           python-pyparsing))
    (inputs
     (list zlib))
    (native-inputs
     (list python-cython python-lzo python-pytest python-setuptools))
    (home-page "https://github.com/bxlab/bx-python")
    (synopsis "Tools for manipulating biological data")
    (description
     "bx-python provides tools for manipulating biological data, particularly
multiple sequence alignments.")
    (license license:expat)))

(define-public python-mofax
  (let ((commit "e4821fef8b1f9f50afd84a6acb96fcc49445c967")
        (revision "1"))
    (package
      (name "python-mofax")
      (version (git-version "0.3.7" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/bioFAM/mofax")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "07lkz51n7p7inmmh4bpr6fcr6b6jcjvkfws06zn8kpxfsv5z9kxa"))))
      (build-system pyproject-build-system)
      (propagated-inputs (list python-h5py
                               python-matplotlib
                               python-pandas
                               python-scipy
                               python-seaborn))
      (native-inputs (list python-poetry-core
                           python-numpy python-pytest))
      (home-page "https://github.com/bioFAM/mofax")
      (synopsis
       "Motif activity finder for transcription factor motif analysis")
      (description
       "MoFax is a Python package for transcription factor motif analysis.
It provides convenience functions to load and visualize factor models trained
with MOFA+ in Python.")
      (license license:expat))))

(define-public python-mudata
  (package
    (name "python-mudata")
    (version "0.3.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/scverse/mudata")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17s1w3746d35pcwr97ynhr7s5hfk76vsfcinwyqynx9k3xxi9br4"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'pretend-version
            (lambda _
              (setenv "SETUPTOOLS_SCM_PRETEND_VERSION" #$version))))))
    (propagated-inputs
     (list python-anndata python-h5py python-pandas))
    (native-inputs
     (list python-hatch-vcs
           python-hatchling
           python-numpy
           python-pytest
           python-zarr))
    (home-page "https://github.com/scverse/mudata")
    (synopsis "Python package for multi-omics data analysis")
    (description
     "Mudata is a Python package for multi-omics data analysis.
It is designed to provide functionality to load, process, and store multimodal
omics data.")
    (license license:bsd-3)))

(define-public python-mofapy2
  (package
    (name "python-mofapy2")
    (version "0.7.1")
    (source
     (origin
       ;; The tarball from PyPi doesn't include tests.
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/bioFAM/mofapy2")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0ahhnqk6gjrhyq286mrd5n7mxcv8l6040ffsawbjx9maqx8wbam0"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; cupy is an optional dependency, which
      ;; itself has nonfree dependencies (CUDA)
      '(list "--ignore=mofapy2/notebooks/test_cupy.py")))
    (propagated-inputs (list python-anndata
                             python-h5py
                             python-numpy
                             python-pandas
                             python-scikit-learn
                             python-scipy))
    (native-inputs (list python-poetry-core
                         python-pytest))
    (home-page "https://biofam.github.io/MOFA2/")
    (synopsis "Multi-omics factor analysis")
    (description "MOFA is a factor analysis model that provides a general
framework for the integration of multi-omic data sets in an unsupervised
fashion.  Intuitively, MOFA can be viewed as a versatile and statistically
rigorous generalization of principal component analysis to multi-omics data.
Given several data matrices with measurements of multiple -omics data types on
the same or on overlapping sets of samples, MOFA infers an interpretable
low-dimensional representation in terms of a few latent factors.  These learnt
factors represent the driving sources of variation across data modalities,
thus facilitating the identification of cellular states or disease
subgroups.")
    (license license:lgpl3)))

(define-public python-muon
  (package
    (name "python-muon")
    (version "0.1.7")
    (source
     (origin
       ;; The tarball from PyPi doesn't include tests.
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/scverse/muon")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0lahc3r0sqqkifibjm090dpkwila3jfh967hldw2nm8wgj81ak82"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k" (string-join
                   ;; Even providing a random seed, scipy.sparse.rand produces
                   ;; inconsistent results across scipy versions.
                   (list"not test_tfidf"
                        ;; Tests fails with error: FileNotFoundError: [Errno
                        ;; 2] No such file or directory.
                        "test_filter_obs_with_obsm_obsp"
                        "test_filter_var_with_varm_varp")
                   " and not "))
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable dir to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _
             (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs (list python-anndata
                             python-h5py
                             python-matplotlib
                             python-mofapy2
                             python-mudata
                             python-numba
                             python-numpy
                             python-pandas
                             python-protobuf
                             python-pybedtools
                             python-pysam
                             python-scanpy
                             python-scikit-learn
                             python-seaborn
                             python-tqdm
                             python-umap-learn))
    (native-inputs (list python-flit-core
                         python-pytest
                         python-pytest-flake8))
    (home-page "https://github.com/scverse/muon")
    (synopsis "Multimodal omics analysis framework")
    (description "muon is a multimodal omics Python framework.")
    (license license:bsd-3)))

(define-deprecated-package python-pyega3
  python-ega-download-client)

(define-public python-pysam
  (package
    (name "python-pysam")
    (version "0.23.0")
    (source (origin
              (method git-fetch)
              ;; Test data is missing on PyPi.
              (uri (git-reference
                    (url "https://github.com/pysam-developers/pysam")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hk0ks6kqsm8252d0v1lw2d22x1awmxcr165nnhyacwbqh246skl"))
              (modules '((guix build utils)))
              (snippet '(begin
                          ;; FIXME: Unbundle samtools and bcftools.
                          (delete-file-recursively "htslib")))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; This test requires network access.
      '(list "-k" "not FileHTTP" "tests")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'patch-build-system
            (lambda _
              (substitute* "pyproject.toml"
                ((":__legacy__") ""))))
          (add-before 'build 'set-flags
            (lambda* (#:key inputs #:allow-other-keys)
              (setenv "HTSLIB_MODE" "external")
              (setenv "HTSLIB_LIBRARY_DIR"
                      (string-append (assoc-ref inputs "htslib") "/lib"))
              (setenv "HTSLIB_INCLUDE_DIR"
                      (string-append (assoc-ref inputs "htslib") "/include"))
              (setenv "LDFLAGS" "-lncurses")
              (setenv "CFLAGS" "-D_CURSES_LIB=1")))
          (add-before 'check 'pre-check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (with-directory-excursion "tests"
                  (setenv "HOME" "/tmp")
                  (invoke "make" "-C" "pysam_data")
                  (invoke "make" "-C" "cbcf_data"))))))))
    (propagated-inputs
     (list htslib))                    ; Included from installed header files.
    (inputs
     (list ncurses curl zlib))
    (native-inputs
     (list python-cython-0
           python-pytest
           python-setuptools
           python-wheel
           ;; Dependencies below are are for tests only.
           samtools
           bcftools))
    (home-page "https://github.com/pysam-developers/pysam")
    (synopsis "Python bindings to the SAMtools C API")
    (description
     "Pysam is a Python module for reading and manipulating files in the
SAM/BAM format.  Pysam is a lightweight wrapper of the SAMtools C API.  It
also includes an interface for tabix.")
    (license license:expat)))

(define-public python-twobitreader
  (package
    (name "python-twobitreader")
    (version "3.1.9")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/benjschiller/twobitreader")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yk86vgdh43wpg8wv4riiff3qv0nrmizxzv774ghjkgiss4g1l2q"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-pytest
           python-setuptools))
    (home-page "https://github.com/benjschiller/twobitreader")
    (synopsis "Python library for reading .2bit files")
    (description
     "twobitreader is a Python library for reading .2bit files as used by the
UCSC genome browser.")
    (license license:artistic2.0)))

(define-public tetoolkit
  (package
    (name "tetoolkit")
    (version "2.2.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/mhammell-laboratory/TEtranscripts")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1jgwnm1kmj2mgvc3gib6d73rvyy3l54icjcj2gw0p1ay6kqkzs0p"))))
    (build-system pyproject-build-system)
    (arguments
     `(#:tests? #f ;no tests
       #:phases (modify-phases %standard-phases
                  (add-after 'unpack 'adjust-requirements
                    (lambda _
                      (substitute* "setup.py"
                        ;; This defunct dependency isn't required for Python 3 (see:
                        ;; https://github.com/mhammell-laboratory/TEtranscripts/issues/111).
                        ((".*'argparse'.*")
                         ""))))
                  (add-after 'unpack 'patch-invocations
                    (lambda* (#:key inputs #:allow-other-keys)
                      (substitute* '("bin/TEtranscripts" "bin/TEcount")
                        (("'sort ")
                         (string-append "'"
                                        (search-input-file inputs "bin/sort")
                                        " "))
                        (("'rm -f ")
                         (string-append "'"
                                        (search-input-file inputs "bin/rm")
                                        " -f "))
                        (("'Rscript'")
                         (string-append "'"
                                        (search-input-file inputs
                                                           "bin/Rscript") "'")))
                      (substitute* "TEToolkit/IO/ReadInputs.py"
                        (("BamToBED")
                         (search-input-file inputs "bin/bamToBed")))
                      (substitute* "TEToolkit/Normalization.py"
                        (("\"Rscript\"")
                         (string-append "\""
                                        (search-input-file inputs
                                                           "bin/Rscript") "\"")))))
                  (add-after 'install 'wrap-program
                    (lambda* (#:key outputs #:allow-other-keys)
                      ;; Make sure the executables find R packages.
                      (for-each (lambda (script)
                                  (wrap-program script
                                    `("R_LIBS_SITE" ":" =
                                      (,(getenv "R_LIBS_SITE")))))
                                (list (search-input-file outputs
                                                         "bin/TEtranscripts")
                                      (search-input-file outputs "bin/TEcount"))))))))
    (inputs (list bash-minimal
                  coreutils
                  bedtools
                  python-pysam
                  r-minimal
                  r-deseq2))
    (native-inputs (list python-setuptools))
    (home-page "https://github.com/mhammell-laboratory/TEtranscripts")
    (synopsis "Transposable elements in differential enrichment analysis")
    (description
     "This is package for including transposable elements in differential
enrichment analysis of sequencing datasets.  TEtranscripts and TEcount take
RNA-seq (and similar data) and annotates reads to both genes and transposable
elements.  TEtranscripts then performs differential analysis using DESeq2.
Note that TEtranscripts and TEcount rely on specially curated GTF files, which
are not included due to their size.")
    (license license:gpl3+)))

(define-public cd-hit
  (package
    (name "cd-hit")
    (version "4.8.1")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/weizhongli/cdhit"
                                  "/releases/download/V" version
                                  "/cd-hit-v" version
                                  "-2019-0228.tar.gz"))
              (sha256
               (base32
                "1phmfhgcpyfd6kj7jwzw976613lcpv1wc2pzfdfaxla062x2s5r6"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ; there are no tests
      #:make-flags
      ;; Executables are copied directly to the PREFIX.
      #~(list (string-append "PREFIX=" #$output "/bin")
              ;; Support longer sequences (e.g. Pacbio sequences)
              "MAX_SEQ=60000000")
      #:phases
      '(modify-phases %standard-phases
         ;; No "configure" script
         (delete 'configure)
         ;; Remove sources of non-determinism
         (add-after 'unpack 'be-timeless
           (lambda _
             (substitute* "cdhit-utility.c++"
               ((" \\(built on \" __DATE__ \"\\)") ""))
             (substitute* "cdhit-common.c++"
               (("__DATE__") "\"0\"")
               (("\", %s, \" __TIME__ \"\\\\n\", date") ""))))
         ;; The "install" target does not create the target directory.
         (add-before 'install 'create-target-dir
           (lambda* (#:key outputs #:allow-other-keys)
             (mkdir-p (string-append (assoc-ref outputs "out") "/bin")))))))
    (inputs
     (list perl zlib))
    (home-page "http://weizhongli-lab.org/cd-hit/")
    (synopsis "Cluster and compare protein or nucleotide sequences")
    (description
     "CD-HIT is a program for clustering and comparing protein or nucleotide
sequences.  CD-HIT is designed to be fast and handle extremely large
databases.")
    ;; The manual says: "It can be copied under the GNU General Public License
    ;; version 2 (GPLv2)."
    (license license:gpl2)))

(define-public cd-hit-auxtools
  (package
    (inherit cd-hit)
    (name "cd-hit-auxtools")
    (arguments
     (list
      #:tests? #f                       ; there are no tests
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'chdir (lambda _ (chdir "cd-hit-auxtools")))
          ;; No "configure" script
          (delete 'configure)
          ;; There is no install target.
          (replace 'install
            (lambda _
              (for-each (lambda (file)
                          (install-file file (string-append #$output "/bin")))
                        '("cd-hit-dup" "cd-hit-lap" "read-linker")))))))
    (inputs '())))

(define-public clipper-peak
  (package
    (name "clipper-peak")
    (version "2.0.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/YeoLab/clipper")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0508rgnfjk5ar5d1mjbjyrnarv4kw9ksq0m3jw2bmgabmb5v6ikk"))
       (modules '((guix build utils)))
       (snippet '(begin
                   ;; Delete pre-compiled files.
                   (delete-file "clipper/src/peaks.so")))))
    (build-system pyproject-build-system)
    (arguments
     `(#:tests? #f
       #:phases (modify-phases %standard-phases
                  (add-after 'unpack 'use-python3-for-cython
                    (lambda _
                      (substitute* "setup.py"
                        (("^setup")
                         "peaks.cython_directives = {'language_level': '3'}
readsToWiggle.cython_directives = {'language_level': '3'}
setup"))))
                  (add-after 'unpack 'disable-nondeterministic-test
                    (lambda _
                      ;; This test fails/succeeds non-deterministically.
                      (substitute* "clipper/test/test_call_peak.py"
                        (("test_get_FDR_cutoff_mean")
                         "_test_get_FDR_cutoff_mean"))))
                  ;; This doesn't work because "usage" is executed, and that calls
                  ;; exit(8).
                  (replace 'check
                    (lambda* (#:key tests? inputs outputs #:allow-other-keys)
                      (when tests?
                        (add-installed-pythonpath inputs outputs)
                        (with-directory-excursion "clipper/test"
                          (invoke "python" "-m" "unittest")))))
                  ;; This is not a library
                  (delete 'sanity-check))))
    (inputs (list htseq
                  python-pybedtools
                  python-cython
                  python-scikit-learn
                  python-matplotlib
                  python-pandas
                  python-pysam
                  python-numpy
                  python-scipy))
    (native-inputs (list python-mock
                         python-pynose
                         python-pytz
                         python-setuptools
                         python-setuptools-git))
    (home-page "https://github.com/YeoLab/clipper")
    (synopsis "CLIP peak enrichment recognition")
    (description "CLIPper is a tool to define peaks in CLIP-seq datasets.")
    (license license:gpl2)))

(define-public codingquarry
  (package
    (name "codingquarry")
    (version "2.0")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "mirror://sourceforge/codingquarry/CodingQuarry_v"
                    version ".tar.gz"))
              (sha256
               (base32
                "0115hkjflsnfzn36xppwf9h9avfxlavr43djqmshkkzbgjzsz60i"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;no "check" target
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin"))
                    (doc (string-append #$output "/share/doc/codingquarry")))
                (install-file "INSTRUCTIONS.pdf" doc)
                (copy-recursively "QuarryFiles"
                                  (string-append #$output
                                                 "/share/codingquarry/QuarryFiles"))
                (install-file "CodingQuarry" bin)
                (install-file "CufflinksGTF_to_CodingQuarryGFF3.py" bin)))))))
    ;; TODO: This package also needs a Python 2 variant of biopython
    (inputs (list openmpi python-2)) ;Only Python 2 is supported
    (native-search-paths
     (list (search-path-specification
            (variable "QUARRY_PATH")
            (files '("share/codingquarry/QuarryFiles")))))
    (home-page "https://sourceforge.net/projects/codingquarry/")
    (synopsis "Fungal gene predictor")
    (description "CodingQuarry is a highly accurate, self-training GHMM fungal
gene predictor designed to work with assembled, aligned RNA-seq transcripts.")
    (license license:gpl3+)))

(define-public crossmap
  (package
    (name "crossmap")
    (version "0.6.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "CrossMap" version))
       (sha256
        (base32 "0hqminh5wn1p3x481jbyc7gmncp5xc196hpvki7k25vzbryhwcix"))
       (modules '((guix build utils)))
       (snippet '(begin
                   ;; Delete compiled Python files.
                   (for-each delete-file
                             (find-files "." "\\.pyc$"))
                   (delete-file-recursively ".eggs")))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #f))      ; No tests in PyPI.
    (inputs (list python-bx-python python-numpy python-pybigwig python-pysam
                  zlib))
    (native-inputs (list python-cython python-pyparsing python-setuptools))
    (home-page "https://crossmap.sourceforge.net/")
    (synopsis "Convert genome coordinates between assemblies")
    (description
     "CrossMap is a program for conversion of genome coordinates or annotation
files between different genome assemblies.  It supports most commonly used
file formats including SAM/BAM, Wiggle/BigWig, BED, GFF/GTF, VCF.")
    (license license:gpl2+)))

(define-public python-dnaio
  (package
    (name "python-dnaio")
    (version "0.10.0")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "dnaio" version))
       (sha256
        (base32
         "064xc4j8plb4fpkm8mw55715mvpvi2sxsknpjx18c2zh904salfy"))))
    (build-system pyproject-build-system)
    (arguments
     (list #:tests? #false)) ;there are none
    (native-inputs
     (list python-cython python-pytest python-setuptools-scm
           python-setuptools python-wheel))
    (propagated-inputs
     (list python-xopen))
    (home-page "https://github.com/marcelm/dnaio/")
    (synopsis "Read FASTA and FASTQ files efficiently")
    (description
     "dnaio is a Python library for fast parsing of FASTQ and also FASTA
files.  The code was previously part of the cutadapt tool.")
    (license license:expat)))

(define-public python-deeptoolsintervals
  (package
    (name "python-deeptoolsintervals")
    (version "0.1.9")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "deeptoolsintervals" version))
              (sha256
               (base32
                "1xnl80nblysj6dylj4683wgrfa425rkx4dp5k65hvwdns9pw753x"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list "--pyargs" "deeptoolsintervals"
              "--doctest-modules")))
    (native-inputs
     (list python-pytest
           python-setuptools))
    (inputs
     (list zlib))
    (home-page "https://github.com/deeptools/deeptools_intervals")
    (synopsis "Create GTF-based interval trees with associated meta-data")
    (description
     "This package provides a Python module creating/accessing GTF-based
interval trees with associated meta-data.  It is primarily used by the
@code{deeptools} package.")
    (license license:expat)))

(define-public python-deeptools
  (package
    (name "python-deeptools")
    (version "3.5.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/deeptools/deepTools")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mgcs03amrd5157drbm6ikdg0m0szrn9xbflariz2zrrnqpsai6s"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-test
            (lambda _
              (substitute* "deeptools/test/test_tools.py"
                (("e_ver = _p")
                 "e_ver = \".\" + _p + \"-real\""))
              (substitute* "deeptools/multiBigwigSummary.py"
                (("version='multiBigwigSummary")
                 "version='%(prog)s"))
              (substitute* "deeptools/plotCoverage.py"
                (("version='plotCoverage")
                 "version='%(prog)s")))))))
    (native-inputs
     (list python-mock python-pytest python-setuptools python-wheel))
    (propagated-inputs
     (list python-matplotlib
           python-numpy
           python-numpydoc
           python-py2bit
           python-pybigwig
           python-pysam
           python-scipy
           python-deeptoolsintervals
           python-plotly))
    (home-page "https://pypi.org/project/deepTools/")
    (synopsis "Useful tools for exploring deep sequencing data")
    (description "This package addresses the challenge of handling large amounts
of data that are now routinely generated from DNA sequencing centers.
@code{deepTools} contains useful modules to process the mapped reads data for
multiple quality checks, creating normalized coverage files in standard bedGraph
and bigWig file formats, that allow comparison between different files.  Finally,
using such normalized and standardized files, deepTools can create many
publication-ready visualizations to identify enrichments and for functional
annotations of the genome.")
    ;; The file deeptools/cm.py is licensed under the BSD license.  The
    ;; remainder of the code is licensed under the MIT license.
    (license (list license:bsd-3 license:expat))))

(define-public cutadapt
  (package
    (name "cutadapt")
    (version "4.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "cutadapt" version))
              (sha256
               (base32
                "0xgsv88mrlw2b1radmd1104y7bg8hvv54ay7xfdpnjiw2jgkrha9"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k" "not test_no_read_only_comment_fasta_input")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-test
            (lambda _
              (let ((site (string-append
                           #$output "/lib/python"
                           #$(version-major+minor
                              (package-version python))
                           "/site-packages")))
                (substitute* "tests/test_command.py"
                  (("env=\\{\"LC_CTYPE\": \"C\"\\},")
                   (string-append "env={\"LC_CTYPE\": \"C\", \"GUIX_PYTHONPATH\": \""
                                  (getenv "GUIX_PYTHONPATH") ":" site
                                  "\"},")))))))))
    (inputs
     (list python-dnaio python-xopen))
    (native-inputs
     (list python-cython
           python-pytest
           python-pytest-mock
           python-pytest-timeout
           python-setuptools
           python-setuptools-scm
           python-wheel))
    (home-page "https://cutadapt.readthedocs.io/en/stable/")
    (synopsis "Remove adapter sequences from nucleotide sequencing reads")
    (description
     "Cutadapt finds and removes adapter sequences, primers, poly-A tails and
other types of unwanted sequence from high-throughput sequencing reads.")
    (license license:expat)))

(define-public lammps
  (let ((commit "stable_2Aug2023_update2"))
    (package
      (name "lammps")
      (version (string-append "0." commit))
      (source
       (origin
	 (method git-fetch)
	 (uri (git-reference
	       (url "https://github.com/lammps/lammps.git")
	       (commit commit)))
	 (file-name (git-file-name name version))
	 (sha256
	  (base32
	   "11xagacgxgldkx34qdzyjrjvn8x3hpl0kgzhh9zh7skpq79pwycz"))))
      (build-system gnu-build-system)
      (arguments
       (list
        #:tests? #f                     ; no check target
	#:make-flags
        '(list "CC=mpicc" "mpi"
	       "LMP_INC=-DLAMMPS_GZIP \
-DLAMMPS_JPEG -DLAMMPS_PNG -DLAMMPS_FFMPEG -DLAMMPS_MEMALIGN=64"
	       "LIB=-gz -ljpeg -lpng -lavcodec")
	#:phases
	#~(modify-phases %standard-phases
            (add-after 'unpack 'chdir
	      (lambda _ (chdir "src")))
	    (replace 'configure
	      (lambda _
		(substitute* "MAKE/Makefile.mpi"
		  (("SHELL =.*")
		   (string-append "SHELL=" (which "bash") "\n"))
		  (("cc ") "mpicc "))
		(substitute* "Makefile"
		  (("SHELL =.*")
		   (string-append "SHELL=" (which "bash") "\n")))))
	    (add-after 'configure 'configure-modules
	      (lambda _
		(invoke "make"
			"yes-molecule"
			"yes-misc"
			"yes-granular"
			(string-append "HDF5_PATH="
				       #$(this-package-input "hdf5")))))
	    (replace 'install
	      (lambda _
		(let ((bin (string-append #$output "/bin")))
		  (mkdir-p bin)
		  (install-file "lmp_mpi" bin)))))))
      (inputs
       (list ffmpeg
	     gfortran
	     gzip
	     hdf5
	     libjpeg-turbo
	     libpng
	     openmpi
             python-wrapper))
      (native-inputs (list bc))
      (home-page "https://www.lammps.org/")
      (synopsis "Classical molecular dynamics simulator")
      (description "LAMMPS is a classical molecular dynamics simulator
designed to run efficiently on parallel computers.  LAMMPS has potentials for
solid-state materials (metals, semiconductors), soft matter (biomolecules,
polymers), and coarse-grained or mesoscopic systems.  It can be used to model
atoms or, more generically, as a parallel particle simulator at the atomic,
meso, or continuum scale.")
      (license license:gpl2+))))

(define-public lammps-serial
  (package
    (inherit lammps)
    (name "lammps-serial")
    (arguments
     (substitute-keyword-arguments (package-arguments lammps)
       ((#:make-flags flags)
        '(list "CC=gcc" "serial"
               "LMP_INC=-DLAMMPS_GZIP \
-DLAMMPS_JPEG -DLAMMPS_PNG -DLAMMPS_FFMPEG -DLAMMPS_MEMALIGN=64"
               "LIB=-gz -ljpeg -lpng -lavcodec"))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'configure
              (lambda _
                (substitute* "MAKE/Makefile.serial"
                  (("SHELL =.*")
                   (string-append "SHELL=" (which "bash") "\n"))
                  (("cc ") "gcc "))
                (substitute* "Makefile"
                  (("SHELL =.*")
                   (string-append "SHELL=" (which "bash") "\n")))))
            (replace 'install
	      (lambda _
		(let ((bin (string-append #$output "/bin")))
		  (mkdir-p bin)
		  (install-file "lmp_serial" bin))))))))
    (inputs
     (modify-inputs (package-inputs lammps)
       (delete "openmpi")))))

(define-public libbigwig
  (package
    (name "libbigwig")
    (version "0.4.4")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/dpryan79/libBigWig")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09693dmf1scdac5pyq6qyn8b4mcipvnmc370k9a5z41z81m3dcsj"))))
    (build-system gnu-build-system)
    (arguments
     `(#:test-target "test"
       #:tests? #f ; tests require access to the web
       #:make-flags
       ,#~(list "CC=gcc"
                (string-append "prefix=" #$output))
       #:phases
       (modify-phases %standard-phases
         (delete 'configure))))
    (inputs
     (list zlib curl))
    (native-inputs
     `(("doxygen" ,doxygen)
       ;; Need for tests
       ("python" ,python-2)))
    (home-page "https://github.com/dpryan79/libBigWig")
    (synopsis "C library for handling bigWig files")
    (description
     "This package provides a C library for parsing local and remote BigWig
files.")
    (license license:expat)))

(define-public lsgkm
  (package
    (name "lsgkm")
    (version "0.1.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/Dongwon-Lee/lsgkm.git")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0b3m94kndvimdfjaf1q2yhmsn7lm5s9v81c5xgfjcp6ig7mh3sa5"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:make-flags '(list "-C" "src")
      #:tests? #false                   ;there are no executable tests
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin")))
                (for-each (lambda (file)
                            (install-file file bin))
                          '("src/gkmtrain"
                            "src/gkmpredict"))))))))
    (home-page "https://github.com/Dongwon-Lee/lsgkm")
    (synopsis "Predict regulatory DNA elements in large-scale data")
    (description "gkm-SVM, a sequence-based method for predicting regulatory
DNA elements, is a useful tool for studying gene regulatory mechanisms.
LS-GKM is an effort to improve the method.  It offers much better scalability
and provides further advanced gapped k-mer based kernel functions.  As a
result, LS-GKM achieves considerably higher accuracy than the original
gkm-SVM.")
    (license license:gpl3+)))

(define-public python-fcsparser
  (package
    (name "python-fcsparser")
    (version "0.2.8")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "fcsparser" version))
       (sha256
        (base32 "1skk1k8phq9sj4ar0cnq8px89y3kcyh5zrbl6anz9wcdcyzkc16z"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list python-numpy python-pandas))
    (native-inputs (list python-poetry-core python-pytest))
    (home-page "https://github.com/eyurtsev/fcsparser")
    (synopsis "Package for reading raw fcs files")
    (description
     "This package provides a Python package for reading raw fcs files")
    (license license:expat)))

(define-public python-pybigwig
  (package
    (name "python-pybigwig")
    (version "0.3.22")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "pyBigWig" version))
              (sha256
               (base32
                "0hr25lkp26mk0fp7irdjdrdsd5lann9kyv0xq9npyyxxakvjci2x"))
              (modules '((guix build utils)))
              (snippet
               ;; Delete bundled libBigWig sources
               '(delete-file-recursively "libBigWig"))))
    (build-system pyproject-build-system)
    (arguments
     '(#:tests? #false      ;only one test exists and it needs internet access
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'link-with-libBigWig
           (lambda _
             (substitute* "setup.py"
               (("libs=\\[") "libs=[\"BigWig\", ")))))))
    (propagated-inputs
     (list python-numpy))
    (inputs
     (list libbigwig zlib curl))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/dpryan79/pyBigWig")
    (synopsis "Access bigWig files in Python using libBigWig")
    (description
     "This package provides Python bindings to the libBigWig library for
accessing bigWig files.")
    (license license:expat)))

(define-public python-schema-salad
  (package
    (name "python-schema-salad")
    (version "8.9.20250723145140")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/common-workflow-language/schema_salad")
               (commit version)))
        (file-name (git-file-name name version))
        (sha256
         (base32
          "1bqsbxx1275129j08aqz7qpzk1nlk4h9psvkm7hzb4liag8nyiql"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list
         ;; Skip failing test, probably innocent.
         ;; TODO: Remove when upgrading because updated upstream.
         "--deselect=schema_salad/tests/test_makedoc.py::test_detect_changes_in_html")
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'set-version
            (lambda _
              (setenv "SETUPTOOLS_SCM_PRETEND_VERSION" #$version)))
          (add-after 'unpack 'relax-requirements
            (lambda _
              ;; Mistune dependency is too strict mistune>=3,<3.1 .
              ;; TODO: Remove when upgrading, because updated upstream.
              (substitute* "requirements.txt"
                (("mistune.*") "mistune"))
              (substitute* "setup.py"
                (("mistune[^\"]*") "mistune"))))
          (add-before 'check 'skip-failing-tests
            (lambda _
              (let ((skip-test
                     (lambda (test-pattern)
                       (string-append "@pytest.mark.skip(reason="
                                      "\"test requires network access\")\n"
                                      test-pattern))))
                (substitute* "schema_salad/tests/test_cg.py"
                  (("^def test_load_by_yaml_metaschema\\(" all)
                   (skip-test all)))
                (substitute* "schema_salad/tests/test_cwl11.py"
                  (("^def test_(secondaryFiles|outputBinding|yaml_tab_error)\\(" all)
                   (skip-test all)))
                (substitute* "schema_salad/tests/test_examples.py"
                  (("^def test_bad_schemas\\(" all)
                   (skip-test all)))))))))
    (propagated-inputs
     (list python-cachecontrol
           python-mistune
           python-mypy-extensions
           python-rdflib
           python-requests
           python-ruamel.yaml))
    (native-inputs
     (list python-black ;black is actually used in the tests.
           python-cachecontrol
           python-pytest
           python-pytest-xdist ;because tests are ran with "-n auto" in tox.ini
           python-setuptools
           python-setuptools-scm))
    (home-page "https://github.com/common-workflow-language/schema_salad")
    (synopsis "Schema Annotations for Linked Avro Data (SALAD)")
    (description
     "Salad is a schema language for describing JSON or YAML structured linked
data documents.  Salad schema describes rules for preprocessing, structural
validation, and hyperlink checking for documents described by a Salad schema.
Salad supports rich data modeling with inheritance, template specialization,
object identifiers, object references, documentation generation, code
generation, and transformation to RDF.  Salad provides a bridge between document
and record oriented data modeling and the Semantic Web.")
    (license license:asl2.0)))

(define-public python-scikit-bio
  (package
    (name "python-scikit-bio")
    (version "0.6.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "scikit-bio" version))
              (sha256
               (base32
                "03y1n91p6m44hhxm3rpb355j6ddalydz49s94h85kbhm7iy5l40h"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; Accuracy problem
      '(list "-k" (string-append "not test_fisher_alpha"
                                 ;; UNEXPECTED EXCEPTION: ValueError("could
                                 ;; not convert string to float: 'gut'")
                                 " and not skbio.diversity"))
      #:phases
      '(modify-phases %standard-phases
         (add-before 'check 'build-extensions
           (lambda _
             ;; Cython extensions have to be built before running the tests.
             (invoke "python3" "setup.py" "build_ext" "--inplace")))
         (replace 'check
           (lambda* (#:key tests? test-flags #:allow-other-keys)
             (when tests?
               (apply invoke "python3" "-m" "skbio.test" test-flags)))))))
    (propagated-inputs
     (list python-biom-format
           python-decorator
           python-h5py
           python-hdmedians
           python-natsort
           python-numpy
           python-pandas
           python-requests
           python-scipy))
    (native-inputs
     (list python-coverage python-pytest))
    (home-page "https://scikit-bio.org")
    (synopsis "Data structures, algorithms and educational resources for bioinformatics")
    (description
     "This package provides data structures, algorithms and educational
resources for bioinformatics.")
    (license license:bsd-3)))

(define-public python-scrublet
  (package
    (name "python-scrublet")
    (version "0.2.3")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "scrublet" version))
              (sha256
               (base32
                "0fk5pwk12yz9wpvwkl6j2l2g28f3x35b9r9n5bw6f0i9f0qgd191"))))
    (build-system pyproject-build-system)
    (arguments '(#:tests? #false)) ;there are none
    (propagated-inputs
     (list python-annoy
           python-cython
           python-matplotlib
           python-numba
           python-numpy
           python-pandas
           python-scikit-image
           python-scikit-learn
           python-scipy
           python-umap-learn))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/swolock/scrublet")
    (synopsis "Tool to identify and remove doublets in single-cell data")
    (description "This package provides a tool for identifying and removing
doublets in single-cell RNA-seq data.")
    (license license:expat)))

(define-public python-cwlformat
  (package
    (name "python-cwlformat")
    (version "2022.02.18")
    (source
     ;; The PyPI tarball is missing Readme.md. Readme.md is required for the
     ;; build.
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/rabix/cwl-format")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0agkz2w86k91rc9m5vx5hsqi5nm6fcmzkng6j99hjapz0r9233ql"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; Assert fails to compare large files.
      #~(list "--deselect=tests/test_battery.py::test_formatting_battery")))
    (native-inputs
     (list python-pytest
           python-setuptools))
    (propagated-inputs
     (list python-importlib-resources
           python-ruamel.yaml))
    (home-page "https://github.com/rabix/cwl-format")
    (synopsis "Prettifier for CWL code")
    (description "@code{python-cwlformat} is a specification and a reference
implementation for a very opinionated @acronym{CWL, Common Workflow Language}
code formatter.  It outputs CWL in a standardized YAML format.")
    (license license:asl2.0)))

(define-public python-cwl-upgrader
  (package
    (name "python-cwl-upgrader")
    (version "1.2.11")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "cwl-upgrader" version))
       (sha256
        (base32
         "12j6z8nvwnzjjyypz59hwj5hmrcri2r6aknw52n9dbj6lbzbdd2p"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (propagated-inputs
     (list python-ruamel.yaml
           python-schema-salad))
    (home-page "https://github.com/common-workflow-language/cwl-upgrader")
    (synopsis "CWL document upgrader")
    (description "@code{python-cwl-upgrader} is a standalone upgrader for
@acronym{CWL, Common Workflow Language} documents from version draft-3, v1.0,
and v1.1 to v1.2.")
    (license license:asl2.0)))

(define-public python-cwl-utils
  (package
    (name "python-cwl-utils")
    (version "0.32")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "cwl-utils" version))
       (sha256
        (base32
         "06wkw8d8cqm3hnz8xwnysz874gwaym36c358cr7frw5iglhvsj98"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list "-k"
              (string-append "not test_graph_split"
                             " and not test_load_document_with_remote_uri"
                             " and not test_remote_packing"
                             " and not test_remote_packing_github_soft_links"
                             " and not test_value_from_two_concatenated_expressions"))))
    (inputs
     (list node-lts))
    (native-inputs
     (list python-mypy-extensions
           python-pytest
           python-pytest-mock
           python-pytest-runner
           python-setuptools
           python-wheel))
    (propagated-inputs
     (list python-cwl-upgrader
           python-cwlformat
           python-packaging
           python-rdflib
           python-requests
           python-ruamel.yaml
           python-schema-salad))
    (home-page "https://github.com/common-workflow-language/cwl-utils")
    (synopsis "Python utilities for CWL")
    (description "@code{python-cwl-utils} provides python utilities and
autogenerated classes for loading and parsing CWL v1.0, CWL v1.1, and CWL v1.2
documents.")
    (license license:asl2.0)))

(define-public cwltool
  (package
    (name "cwltool")
    (version "3.1.20250925164626")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/common-workflow-language/cwltool")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13mv7qcl64gng8bq0y9garp0vvn9851n98vzi75ppl16pjkkziks"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; These tests try to connect to the internet.
      #~(list "--ignore=tests/test_content_type.py"
              "--ignore=tests/test_udocker.py"
              "--ignore=tests/test_http_input.py"
              "-k"
              (string-append
               ;; Causes
               ;;   INTERNALERROR> RuntimeError: Unexpectedly no active
               ;;   workers available
               "not test_env_filtering"
               ;; Tries to write to /tmp/guix-build-cwltool-3.fastq
               " and not test_iwdr_writable_secondaryfiles"
               ;; Tries to use cwl-runners.
               " and not test_v1_0_arg_empty_prefix_separate_false"))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'loosen-version-restrictions
            (lambda _
              (substitute* "setup.py"
                (("== 1.5.1") "> 1.5.1")))) ; prov
          (add-after 'unpack 'set-version
            (lambda _
              ;; Set exact version.
              (setenv "SETUPTOOLS_SCM_PRETEND_VERSION" #$version)))
          (add-after 'unpack 'patch-tests
            (lambda _
              (substitute* '("tests/subgraph/env-tool2.cwl"
                             "tests/subgraph/env-tool2_req.cwl"
                             "tests/subgraph/env-wf2_subwf-packed.cwl"
                             "tests/subgraph/env-tool2_no_env.cwl")
                (("\"/bin/sh\"") (string-append "\"" (which "sh") "\""))))))))
    (inputs
     (list python-argcomplete
           python-bagit
           python-coloredlogs
           python-cwl-utils
           python-mypy-extensions
           python-prov
           python-pydot
           python-psutil
           python-rdflib-6
           python-requests
           python-rich-argparse
           python-ruamel.yaml
           python-schema-salad
           python-setuptools
           python-spython
           ;; Not listed as needed but still necessary:
           node-lts))
    (native-inputs
     (list python-arcp
           python-mock
           python-pytest
           python-pytest-mock
           python-pytest-xdist))
    (home-page
     "https://github.com/common-workflow-language/common-workflow-language")
    (synopsis "Common Workflow Language reference implementation")
    (description
     "This is the reference implementation of the @acronym{CWL, Common Workflow
Language} standards.  The CWL open standards are for describing analysis
workflows and tools in a way that makes them portable and scalable across a
variety of software and hardware environments, from workstations to cluster,
cloud, and high performance computing (HPC) environments.  CWL is designed to
meet the needs of data-intensive science, such as Bioinformatics, Medical
Imaging, Astronomy, Physics, and Chemistry.  The @acronym{cwltool, CWL reference
implementation} is intended to be feature complete and to provide comprehensive
validation of CWL files as well as provide other tools related to working with
CWL descriptions.")
    (license license:asl2.0)))

(define-public ravanan
  (package
    (name "ravanan")
    (version "0.2.0")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://github.com/arunisaac/ravanan/releases/download/v"
                    version "/ravanan-" version ".tar.lz"))
              (sha256
               (base32
                "00ayvxsgvma3231ja1759f5fr99a4nw3j23i8qz616wfxa54745s"))))
    (arguments
     (list #:make-flags
           #~(list (string-append "prefix=" #$output)
                   (string-append "NODE=" (search-input-file %build-inputs "bin/node")))
           #:modules `(((guix build guile-build-system)
                        #:select (target-guile-effective-version))
                       ,@%default-gnu-imported-modules)
           #:phases
           (with-imported-modules `((guix build guile-build-system)
                                    ,@%default-gnu-imported-modules)
             #~(modify-phases %standard-phases
                 (replace 'patch-source-shebangs
                   (lambda* (#:key inputs #:allow-other-keys)
                     (substitute* "bin/ravanan"
                       (("^exec guile")
                        (string-append "exec "
                                       (search-input-file inputs "/bin/guile"))))))
                 (delete 'configure)
                 (add-after 'install 'wrap
                   (lambda* (#:key inputs outputs #:allow-other-keys)
                     (let ((out (assoc-ref outputs "out"))
                           (effective-version (target-guile-effective-version)))
                       (wrap-program (string-append out "/bin/ravanan")
                         `("GUILE_LOAD_PATH" prefix
                           (,(string-append out "/share/guile/site/" effective-version)
                            ,(getenv "GUILE_LOAD_PATH")))
                         `("GUILE_LOAD_COMPILED_PATH" prefix
                           (,(string-append out "/lib/guile/"
                                            effective-version "/site-ccache")
                            ,(getenv "GUILE_LOAD_COMPILED_PATH")))))))))))
    (inputs
     (list bash-minimal
           node
           guile-3.0
           guile-filesystem
           guile-gcrypt
           guile-json-4
           guile-libyaml
           guix))
    (native-inputs
     (list guile-run64
           lzip))
    (build-system gnu-build-system)
    (home-page "https://forge.systemreboot.net/ravanan/")
    (synopsis "High-reproducibility CWL runner powered by Guix")
    (description "ravanan is a @acronym{CWL, Common Workflow Language}
implementation that is powered by GNU Guix and provides strong reproducibility
guarantees.  ravanan provides strong caching of intermediate results so the
same step of a workflow is never run twice.  ravanan captures logs from every
step of the workflow for easy tracing back in case of job failures.  ravanan
currently runs on single machines and on slurm via its API.")
    (license license:gpl3+)))

(define-public python-dendropy
  (package
    (name "python-dendropy")
    (version "4.5.1")
    (source
     (origin
       (method git-fetch)
       ;; Source from GitHub so that tests are included.
       (uri (git-reference
             (url "https://github.com/jeetsukumaran/DendroPy")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0lrfzjqzbpk1rrra9vd7z2j7q09jy9w1ss7wn2rd85i4k5y3xz8l"))))
    (build-system pyproject-build-system)
    (arguments
     (list
       #:test-flags
       '(list "-k"
              (string-join
               ;; These tests fail because we have no "paup" executable.
               (list "not test_group1"
                     "test_basic_split_counting_under_different_rootings"
                     "test_basic_split_count_with_incorrect_weight_treatment_raises_error"
                     "test_basic_split_count_with_incorrect_rootings_raises_error"

                     ;; Assert error for unknown reasons
                     "test_by_num_lineages")
               " and not "))
       #:phases
       '(modify-phases %standard-phases
          (add-after 'unpack 'python-compatibility
            (lambda _
              (substitute* "tests/test_datamodel_taxon.py"
                (("collections.Iterable")
                 "collections.abc.Iterable")))))))
    (native-inputs (list python-pytest python-wheel))
    (propagated-inputs (list python-setuptools))
    (home-page "https://dendropy.org/")
    (synopsis "Library for phylogenetics and phylogenetic computing")
    (description
     "DendroPy is a library for phylogenetics and phylogenetic computing: reading,
writing, simulation, processing and manipulation of phylogenetic
trees (phylogenies) and characters.")
    (license license:bsd-3)))

(define-public python-py2bit
  (package
    (name "python-py2bit")
    (version "0.3.3")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "py2bit" version))
       (sha256
        (base32
         "0w1p4l1jwg9kkifm0jsg33a212ps0jn61islmnng2afp77y5nkr6"))))
    (build-system pyproject-build-system)
    (arguments
     (list #:test-flags #~(list "py2bitTest/test.py")))
    (native-inputs (list python-pytest python-setuptools python-setuptools-scm))
    (home-page "https://github.com/dpryan79/py2bit")
    (synopsis "Access 2bit files using lib2bit")
    (description
     "This package provides Python bindings for lib2bit to access 2bit files
with Python.")
    (license license:expat)))

(define-public delly
  (package
    (name "delly")
    (version "0.8.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/dellytools/delly")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32 "1ibnplgfzj96w8glkx17v7sld3pm402fr5ybmf3h0rlcryabxrqy"))
              (modules '((guix build utils)))
              (snippet
               '(delete-file-recursively "src/htslib"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;There are no tests to run.
      #:make-flags
      #~(list "PARALLEL=1"             ; Allow parallel execution at run-time.
              (string-append "prefix=" #$output))
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)           ; There is no configure phase.
          (add-after 'install 'install-templates
            (lambda _
              (let ((templates (string-append #$output
                                              "/share/delly/templates")))
                (mkdir-p templates)
                (copy-recursively "excludeTemplates" templates)))))))
    (inputs
     (list boost bzip2 htslib zlib))
    (home-page "https://github.com/dellytools/delly")
    (synopsis "Integrated structural variant prediction method")
    (description "Delly is an integrated structural variant prediction method
that can discover and genotype deletions, tandem duplications, inversions and
translocations at single-nucleotide resolution in short-read massively parallel
sequencing data.  It uses paired-ends and split-reads to sensitively and
accurately delineate genomic rearrangements throughout the genome.")
    (license license:gpl3+)))

(define-public transanno
  (package
    (name "transanno")
    (version "0.4.5")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/informationsea/transanno")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0x62v8qhnpw8579kcqpr9k5ldv2y3awjp7a32f1j8qky0i1jgxp1"))
       (snippet
        '(with-output-to-file "liftover-rs/build.rs"
           (lambda _
             (format #true
                     "fn main() {~@
                        println!(\"cargo:rustc-link-lib=lzma\");~@
                        }~%"))))))
    (build-system cargo-build-system)
    (arguments
     (list
      #:install-source? #false          ;fails
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'prepare-test-files
            (lambda _
              (delete-file "Cargo.lock")
              (substitute* "liftover-rs/prepare-test.sh"
                (("/bin/bash")
                 (string-append #$(this-package-native-input "bash")
                                "/bin/bash")))
              (invoke "bash" "prepare-test-files.sh")))
          (add-before 'install 'chdir
            (lambda _ (chdir "transanno"))))))
    (native-inputs (list bash pkg-config))
    (inputs (cons* xz `(,zstd "lib") (cargo-inputs 'transanno)))
    (home-page "https://github.com/informationsea/transanno")
    (synopsis "LiftOver tool for new genome assemblies")
    (description "This package provides an accurate VCF/GFF3/GTF LiftOver tool
for new genome assemblies.")
    (license license:gpl3+)))

(define-public trf
  (package
    (name "trf")
    (version "4.09.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/Benson-Genomics-Lab/TRF")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32 "0fhwr4s1mf8nw8fr5imwjvjr42b59p97zr961ifm8xl1bajz4wpg"))))
    (build-system gnu-build-system)
    (home-page "https://github.com/Benson-Genomics-Lab/TRF")
    (synopsis "Tandem Repeats Finder: a program to analyze DNA sequences")
    (description "A tandem repeat in DNA is two or more adjacent, approximate
copies of a pattern of nucleotides.  Tandem Repeats Finder is a program to
locate and display tandem repeats in DNA sequences.  In order to use the
program, the user submits a sequence in FASTA format.  The output consists of
two files: a repeat table file and an alignment file.  Submitted sequences may
be of arbitrary length.  Repeats with pattern size in the range from 1 to 2000
bases are detected.")
    (license license:agpl3+)))

(define-public trinityrnaseq
  (package
    (name "trinityrnaseq")
    (version "2.15.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/trinityrnaseq/trinityrnaseq.git")
                    (commit (string-append "Trinity-v" version))
                    (recursive? #true)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06qvxy6wvvjpn1mkg8m78syyn98xj76569gpl2jbc9q036i56kh8"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:test-target "test"
      #:modules
      '((guix build gnu-build-system)
        (guix build utils)
        (ice-9 match)
        (srfi srfi-1))
      #:make-flags
      #~(list (string-append "CC=" #$(cc-for-target)))
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda _
              (setenv "SHELL" (which "sh"))
              (setenv "CONFIG_SHELL" (which "sh"))
              ;; Do not require version.h, which triggers a local build of a
              ;; vendored htslib.
              (substitute* "trinity-plugins/bamsifter/Makefile"
                (("sift_bam_max_cov.cpp htslib/version.h")
                 "sift_bam_max_cov.cpp"))))
          (add-after 'build 'build-plugins
            (lambda _
              ;; Run this in the subdirectory to avoid running the
              ;; tests right here.
              (with-directory-excursion "trinity-plugins"
                (invoke "make" "plugins"))))
          ;; The install script uses rsync, provides no overrides for the
          ;; default location at /usr/local/bin, and patching it would change
          ;; all lines that do something.
          (replace 'install
            (lambda* (#:key inputs #:allow-other-keys)
              (let ((share (string-append #$output "/share/trinity/"))
                    (bin   (string-append #$output "/bin/")))
                (mkdir-p bin)
                (copy-recursively "." share)

                (wrap-program (string-append share "Trinity")
                  `("R_LIBS_SITE" ":" = (,(getenv "R_LIBS_SITE")))
                  `("PERL5LIB"    ":" = (,(getenv "PERL5LIB")))
                  `("PYTHONPATH"  ":" = (,(getenv "GUIX_PYTHONPATH")))
                  `("PATH"        ":" =
                    ,(cons (string-append share "/trinity-plugins/BIN")
                           (filter-map (match-lambda
                                         ((name . dir)
                                          (string-append dir "/bin")))
                                       inputs))))
                (symlink (string-append share "Trinity")
                         (string-append bin "Trinity"))))))))
    (inputs
     (list blast+
           bowtie
           fastqc
           hisat
           htslib
           icedtea-8
           jellyfish
           kallisto
           multiqc
           perl
           perl-uri-escape
           python-numpy
           python-wrapper
           r-ape
           r-argparse
           r-biobase
           r-ctc
           r-deseq2
           r-edger
           r-fastcluster
           r-glimma
           r-goplot
           r-goseq
           r-gplots
           r-minimal
           r-qvalue
           r-rots
           r-sm
           r-tidyverse
           rsem
           salmon
           samtools
           sra-tools
           star
           zlib))
    (propagated-inputs
     (list coreutils
           gzip
           which))
    (native-inputs (list cmake-minimal))
    (home-page "https://github.com/trinityrnaseq/trinityrnaseq/wiki")
    (synopsis "Trinity RNA-Seq de novo transcriptome assembly")
    (description "Trinity assembles transcript sequences from Illumina RNA-Seq
data.  Trinity represents a novel method for the efficient and robust de novo
reconstruction of transcriptomes from RNA-seq data.  Trinity combines three
independent software modules: Inchworm, Chrysalis, and Butterfly, applied
sequentially to process large volumes of RNA-seq reads.  Trinity partitions
the sequence data into many individual de Bruijn graphs, each representing the
transcriptional complexity at a given gene or locus, and then processes each
graph independently to extract full-length splicing isoforms and to tease
apart transcripts derived from paralogous genes.")
    (license license:bsd-3)))

(define-public repeat-masker
  (package
    (name "repeat-masker")
    (version "4.1.2-p1")
    (source (origin
              (method url-fetch)
              (uri (string-append "http://www.repeatmasker.org/"
                                  "RepeatMasker/RepeatMasker-"
                                  version ".tar.gz"))
              (sha256
               (base32 "15hfdfpzmdjcx7ng7rjfid69bmvgn3z9g9r43qhjnhjhq3v4prab"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #false ; there are none
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'build
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let ((share (string-append (assoc-ref outputs "out")
                                         "/share/RepeatMasker")))
               (mkdir-p share)
               (copy-recursively "." share)
               (with-directory-excursion share
                 (invoke "perl" "configure"
                         "--trf_prgm" (which "trf")
                         "--hmmer_dir"
                         (string-append (assoc-ref inputs "hmmer")
                                        "/bin"))))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out   (assoc-ref outputs "out"))
                    (share (string-append out "/share/RepeatMasker"))
                    (bin   (string-append out "/bin"))
                    (path  (getenv "PERL5LIB")))
               (install-file (string-append share "/RepeatMasker") bin)
               (wrap-program (string-append bin "/RepeatMasker")
                 `("PERL5LIB" ":" prefix (,path ,share)))))))))
    (inputs
     (list bash-minimal
           perl
           perl-text-soundex
           python
           python-h5py
           hmmer
           trf))
    (home-page "https://github.com/Benson-Genomics-Lab/TRF")
    (synopsis "Tandem Repeats Finder: a program to analyze DNA sequences")
    (description "A tandem repeat in DNA is two or more adjacent, approximate
copies of a pattern of nucleotides.  Tandem Repeats Finder is a program to
locate and display tandem repeats in DNA sequences.  In order to use the
program, the user submits a sequence in FASTA format.  The output consists of
two files: a repeat table file and an alignment file.  Submitted sequences may
be of arbitrary length. Repeats with pattern size in the range from 1 to 2000
bases are detected.")
    (license license:osl2.1)))

(define-public trust4
  (package
    (name "trust4")
    (version "1.1.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/liulab-dfci/TRUST4")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "170k4rrchg7f2jyn3v4y4fxxq1d49n1vnvhx5xcnnr6jql8q3h08"))
       (modules '((guix build utils)))
       (snippet '(begin
                   ;; Remove bundled samtools
                   (delete-file-recursively "samtools-0.1.19")))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f ;there are no tests
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'use-samtools-headers
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "Makefile"
                (("LINKPATH= -I./samtools-0.1.19 -L./samtools-0.1.19")
                 (string-append "LINKPATH= -I."
                                #$(this-package-native-input "samtools")
                                "/include/samtools"
                                " -L."
                                #$(this-package-native-input "samtools")
                                "/lib"))
                (("./samtools-0.1.19/")
                 (string-append #$(this-package-native-input
                                   "samtools") "/lib/")))
              (substitute* "alignments.hpp"
                (("samtools-0.1.19")
                 (string-append #$(this-package-native-input
                                   "samtools") "/include/samtools")))))
          (delete 'configure) ; No configure.
          (replace 'install
            (lambda* (#:key outputs #:allow-other-keys)
              (let ((bin (string-append (assoc-ref outputs "out") "/bin"))
                    (scripts (string-append #$output
                                            "/share/trust4/scripts")))
                (install-file "annotator" bin)
                (install-file "bam-extractor" bin)
                (install-file "fastq-extractor" bin)
                (install-file "run-trust4" bin)
                (install-file "trust4" bin)
                ;; install scripts stored in the scrips dir
                (for-each (lambda (file)
                            (chmod file #o555))
                          (find-files "scripts" "\\.p(y|l)"))
                (copy-recursively "scripts" scripts)
                (delete-file-recursively "scripts")
                ;; install the rest of the scripts that are in the main dir
                (for-each (lambda (file)
                            (chmod file #o555)
                            (install-file file bin))
                          (find-files "." "\\.(pl|py|sh)"))))))))
    (native-inputs (list automake samtools-0.1))
    (inputs (list perl python-wrapper zlib))
    (home-page "https://github.com/liulab-dfci/TRUST4")
    (synopsis "TCR and BCR assembly from RNA-seq data")
    (description "This package is analyzing @acronym{TCR, T cell receptor} and
@acronym{BCR, B cell receptor} sequences using unselected RNA sequencing data,
profiled from fluid and solid tissues, including tumors.  TRUST4 performs de
novo assembly on V, J, C genes including the hypervariable @acronym{CDR3,
complementarity-determining region 3} and reports consensus contigs of BCR/TCR
sequences.  TRUST4 then realigns the contigs to IMGT reference gene sequences to
identify the corresponding gene and CDR3 details.  TRUST4 supports both single-end
and paired-end bulk or single-cell sequencing data with any read length.")
    (license license:gpl3)))

(define-public python-airr
  (package
    (name "python-airr")
    (version "1.5.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "airr" version))
       (sha256
        (base32 "0jbigfdwa23xv5riw0ljdfq2qwg1b2fav2kfi81zxd1g1jprxy3i"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-jsondiff
           python-pytest
           python-setuptools))
    (propagated-inputs
     (list python-pandas
           python-pyyaml
           python-yamlordereddictloader))
    (home-page "http://docs.airr-community.org")
    (synopsis "Data Representation Standard library for antibody and TCR sequences")
    (description "Python-airr provides a library by the AIRR community to for
describing, reporting, storing, and sharing adaptive immune receptor
repertoire (AIRR) data, such as sequences of antibodies and T cell
receptors (TCRs).")
    (license license:cc-by4.0)))

(define-public diamond
  (package
    (name "diamond")
    (version "2.1.6")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/bbuchfink/diamond")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kb17jwlsrvgswfahzznrffv1i6ybwwmq99qs7iga5yzbx64jp6q"))))
    (build-system cmake-build-system)
    (arguments
     (list #:configure-flags '(list "-DX86=OFF"))) ;avoid SSE4 and AVX2
    (inputs (list zlib))
    (home-page "https://github.com/bbuchfink/diamond")
    (synopsis "Accelerated BLAST compatible local sequence aligner")
    (description
     "DIAMOND is a BLAST-compatible local aligner for mapping protein and
translated DNA query sequences against a protein reference database (BLASTP
and BLASTX alignment mode).  The speedup over BLAST is up to 20,000 on short
reads at a typical sensitivity of 90-99% relative to BLAST depending on the
data and settings.")
    (license license:gpl3+)))

(define-public discrover
  (package
    (name "discrover")
    (version "1.6.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/maaskola/discrover")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "173fwi2vb6a5kp406hm3jj6j7v4whww796f2qcygp4rpvamh307y"))))
    (build-system cmake-build-system)
    (arguments
     `(#:tests? #f                      ; there are no tests
       #:phases
       (modify-phases %standard-phases
         (add-before 'build 'set-force-source-date
           ;; for reproducible dates, texlive needs this to respect
           ;; SOURCE_DATE_EPOCH
           (lambda _
             (setenv "FORCE_SOURCE_DATE" "1")))
         (add-before 'build 'use-writable-texmfvar
           ;; Use writable TEXMFVAR to generate fonts.
           (lambda _
             (setenv "TEXMFVAR" "/tmp")))
         (add-after 'unpack 'fix-latex-errors
           (lambda _
             (with-fluids ((%default-port-encoding #f))
               (substitute* "doc/references.bib"
                 (("\\{S\\}illanp[^,]+,")
                  "{S}illanp{\\\"a}{\\\"a},")))
             ;; XXX: I just can't get pdflatex to not complain about these
             ;; characters.  They end up in the manual via the generated
             ;; discrover-cli-help.txt.
             (substitute* "src/hmm/cli.cpp"
               (("µ") "mu")
               (("η") "eta")
               (("≤") "<="))
             ;; This seems to be a syntax error.
             (substitute* "doc/discrover-manual.tex"
               (("theverbbox\\[t\\]") "theverbbox"))))
         (add-after 'unpack 'add-missing-includes
           (lambda _
             (substitute* "src/executioninformation.hpp"
               (("#define EXECUTIONINFORMATION_HPP" line)
                (string-append line "\n#include <random>")))
             (substitute* "src/plasma/fasta.hpp"
               (("#define FASTA_HPP" line)
                (string-append line "\n#include <random>"))))))))
    (inputs
     (list boost cairo rmath-standalone))
    (native-inputs
     (list (texlive-local-tree
            (list texlive-doi
                  texlive-etoolbox
                  texlive-examplep
                  texlive-forloop
                  texlive-listofitems
                  texlive-natbib
                  texlive-pgf           ;tikz
                  texlive-readarray
                  texlive-verbatimbox))
           imagemagick))
    (home-page "https://dorina.mdc-berlin.de/public/rajewsky/discrover/")
    (synopsis "Discover discriminative nucleotide sequence motifs")
    (description "Discrover is a motif discovery method to find binding sites
of nucleic acid binding proteins.")
    (license license:gpl3+)))

(define-public eddylab-squid
  (package
    (name "eddylab-squid")
    (version "1.9g")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "http://eddylab.org/software/squid/squid-"
                           version ".tar.gz"))
       (sha256
        (base32
         "19ywv1h581a84yyjnp64gwww99vhgbxi8v4rl37xp92ag7l44brh"))))
    (build-system gnu-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-before 'check 'set-perl-search-path
           (lambda _
             ;; Work around "dotless @INC" build failure.
             (setenv "PERL5LIB"
                     (string-append (getcwd) "/Testsuite:"
                                    (getenv "PERL5LIB"))))))))
    (inputs (list perl))
    (home-page "http://eddylab.org/software.html")
    (synopsis "C function library for sequence analysis")
    (description "SQUID is Sean Eddy's personal library of C functions
and utility programs for sequence analysis.")
    (license license:gpl2)))

(define-public edirect
  (package
    (name "edirect")
    (version "13.3.20200128")
    (source (origin
              (method url-fetch)
              (uri (string-append "ftp://ftp.ncbi.nlm.nih.gov/entrez/entrezdirect"
                                  "/versions/" version
                                  "/edirect-" version ".tar.gz"))
              (sha256
               (base32
                "093zp7klv81ph0y8mm8d78a9hnpfxbv2kdym70gzdf3vz176rw33"))
              (modules '((guix build utils)))
              (snippet
               '(begin (delete-file "Mozilla-CA.tar.gz")
                       (substitute* "rchive.go"
                         ;; This go library does not have any license.
                         (("github.com/fiam/gounidecode/unidecode")
                          "golang.org/rainycape/unidecode"))))))
    (build-system perl-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (delete 'build)
         (delete 'check)                ; simple check after install
         (add-after 'unpack 'patch-programs
           (lambda* (#:key inputs #:allow-other-keys)
             ;; Ignore errors about missing xtract.Linux and rchive.Linux.
              (substitute* "pm-refresh"
                (("cat \\\"\\$target")
                 "grep ^[[:digit:]] \"$target"))))
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin"))
                   (edirect-go (assoc-ref inputs "edirect-go-programs")))
               (for-each
                 (lambda (file)
                   (install-file file bin))
                 '("archive-pubmed" "asp-cp" "asp-ls" "download-ncbi-data"
                   "download-pubmed" "edirect.pl" "efetch" "epost" "esearch"
                   "fetch-pubmed" "ftp-cp" "ftp-ls" "has-asp" "index-pubmed"
                   "pm-prepare" "pm-refresh" "pm-stash" "pm-collect"
                   "pm-index" "pm-invert" "pm-merge" "pm-promote"))
               (symlink (string-append edirect-go "/bin/xtract.Linux")
                        (string-append bin "/xtract"))
               (symlink (string-append edirect-go "/bin/rchive.Linux")
                        (string-append bin "/rchive")))))
         (add-after 'install 'wrap-program
           (lambda* (#:key outputs #:allow-other-keys)
              ;; Make sure everything can run in a pure environment.
              (let ((out (assoc-ref outputs "out"))
                    (path (getenv "PERL5LIB")))
                (for-each
                  (lambda (file)
                    (wrap-program file
                      `("PERL5LIB" ":" prefix (,path)))
                    (wrap-program file
                      `("PATH" ":" prefix (,(string-append out "/bin")
                                           ,(dirname (which "sed"))
                                           ,(dirname (which "gzip"))
                                           ,(dirname (which "grep"))
                                           ,(dirname (which "perl"))
                                           ,(dirname (which "uname"))))))
                  (find-files out ".")))))
         (add-after 'wrap-program 'check
           (lambda* (#:key outputs #:allow-other-keys)
             (invoke (string-append (assoc-ref outputs "out")
                                    "/bin/edirect.pl")
                     "-filter" "-help"))))))
    (inputs
     (list bash-minimal
           edirect-go-programs
           perl-html-parser
           perl-encode-locale
           perl-file-listing
           perl-html-tagset
           perl-html-tree
           perl-http-cookies
           perl-http-date
           perl-http-message
           perl-http-negotiate
           perl-lwp-mediatypes
           perl-lwp-protocol-https
           perl-net-http
           perl-uri
           perl-www-robotrules
           perl-xml-simple
           perl))
    (home-page "https://www.ncbi.nlm.nih.gov/books/NBK179288/")
    (synopsis "Tools for accessing the NCBI's set of databases")
    (description
     "Entrez Direct (EDirect) is a method for accessing the National Center
for Biotechnology Information's (NCBI) set of interconnected
databases (publication, sequence, structure, gene, variation, expression,
etc.) from a terminal.  Functions take search terms from command-line
arguments.  Individual operations are combined to build multi-step queries.
Record retrieval and formatting normally complete the process.

EDirect also provides an argument-driven function that simplifies the
extraction of data from document summaries or other results that are returned
in structured XML format.  This can eliminate the need for writing custom
software to answer ad hoc questions.")
    (native-search-paths
     ;; Ideally this should be set for LWP somewhere.
     (list (search-path-specification
            (variable "PERL_LWP_SSL_CA_FILE")
            (file-type 'regular)
            (separator #f)
            (files '("/etc/ssl/certs/ca-certificates.crt")))))
    (license license:public-domain)))

(define-public edirect-go-programs
  (package
    (inherit edirect)
    (name "edirect-go-programs")
    (build-system go-build-system)
    (arguments
     `(#:install-source? #f
       #:tests? #f      ; No tests.
       #:import-path "ncbi.nlm.nih.gov/entrez/edirect"
       #:phases
       (modify-phases %standard-phases
         (replace 'build
           (lambda* (#:key import-path #:allow-other-keys)
             (with-directory-excursion (string-append "src/" import-path)
               (invoke "go" "build" "-v" "-x" "j2x.go")
               (invoke "go" "build" "-v" "-x" "t2x.go")
               (invoke "go" "build" "-v" "-x" "-o"
                       "xtract.Linux" "xtract.go" "common.go")
               (invoke "go" "build" "-v" "-x" "-o"
                       "rchive.Linux" "rchive.go" "common.go")
               (invoke "go" "build" "-v" "-x" "-o" "symbols.Linux" "s2p.go"))))
         (replace 'install
           (lambda* (#:key outputs import-path #:allow-other-keys)
             (let ((dest    (string-append (assoc-ref outputs "out") "/bin"))
                   (source  (string-append "src/" import-path "/")))
               (for-each (lambda (file)
                           (format #t "installing ~a~%" file)
                           (install-file (string-append source file) dest))
                         '("j2x" "t2x" "symbols.Linux" "xtract.Linux" "rchive.Linux"))
               #t))))))
    (native-inputs '())
    (propagated-inputs '())
    (inputs
     (list go-github-com-fatih-color
           go-github-com-fogleman-gg
           go-github-com-gedex-inflector
           go-github-com-golang-freetype
           go-github-com-klauspost-cpuid
           go-github-com-pbnjay-memory
           go-github-com-surgebase-porter2
           go-golang-org-rainycape-unidecode
           go-golang-org-x-image
           go-golang-org-x-text))))

(define-public python-baltica
  (package
    (name "python-baltica")
    (version "1.1.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/dieterich-lab/Baltica")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "001ac03v9pbqqzf9pv7v8gf0296ksa4f0v3wdmpa6m9701skqi4r"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               ;; The tests need to be run from elsewhere...
               (mkdir-p "/tmp/test")
               (copy-recursively ".tests" "/tmp/test")
               (with-directory-excursion "/tmp/test"
                 (invoke "pytest" "-v" "--doctest-modules"))))))))
    (propagated-inputs
     (list gunicorn
           python-anndata
           python-click
           python-flask
           python-flask-wtf
           python-h5py
           python-numpy
           python-psutil
           python-pysam
           python-pyyaml
           python-scipy
           r-biocmanager
           r-dplyr
           r-genomicranges
           r-here
           r-openxlsx
           r-optparse
           r-readr
           r-rsamtools
           r-stringr
           r-tidyr
           r-upsetr
           r-yaml
           snakemake-7))
    (native-inputs (list python-cython python-pyfakefs python-pytest))
    (home-page "https://github.com/dieterich-lab/Baltica")
    (synopsis "Integrated splice junction usage analysis")
    (description
     "This framework facilitates the execution of @dfn{differential junction
usage} (DJU) methods.  Additionally, it enables the integration of results from
multiple DJU methods.")
    (license license:expat)))

(define-public python-bamnostic
  (package
    (name "python-bamnostic")
    (version "1.1.8")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "bamnostic" version))
              (sha256
               (base32
                "0cjpzyqz6r4lmiwry2gcxdczwpkhl3lyyjg4s8addln17691ysxk"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list python-pytest python-setuptools))
    (native-inputs (list python-wheel))
    (home-page "https://github.com/betteridiot/bamnostic/")
    (synopsis "Tool for binary alignment map, random access and parsing")
    (description
     "Bamnostic is a pure Python @dfn{Binary Alignment Map} (BAM) file parser
and random access tool.")
    (license license:bsd-3)))

(define-public python-episcanpy
  (package
    (name "python-episcanpy")
    (version "0.4.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "episcanpy" version))
              (sha256
               (base32
                "1qqcyhyzy6idpjmidfdpzwk02hbxm25rymz42h51nlk0vd4r2wwm"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; The tests import episcanpy.api, but this is known to not work.
      ;; See https://github.com/colomemaria/epiScanpy/issues/133
      #:tests? #false
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable dir to cache functions.
         (add-before 'build 'set-numba-cache-dir
           (lambda _
             (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs
     (list python-anndata-0.11
           python-bamnostic
           python-h5py
           python-intervaltree
           python-joblib
           python-kneed
           python-legacy-api-wrap
           python-matplotlib
           python-natsort
           python-networkx
           python-numba
           python-numpy
           python-packaging
           python-pandas
           python-scanpy
           python-scikit-learn
           python-scipy
           python-seaborn
           python-statsmodels
           python-tqdm
           python-pysam
           python-tbb
           python-umap-learn))
    (native-inputs (list python-pytest python-setuptools-scm python-wheel))
    (home-page "https://github.com/colomemaria/epiScanpy")
    (synopsis "Tool for epigenomics single cell analysis")
    (description
     "EpiScanpy is a toolkit to analyse single-cell open
chromatin (scATAC-seq) and single-cell DNA methylation (for example scBS-seq)
data.  EpiScanpy is the epigenomic extension of the very popular scRNA-seq
analysis tool Scanpy (Genome Biology, 2018).")
    (license license:bsd-3)))

(define-public python-modbedtools
  (package
    (name "python-modbedtools")
    (version "0.1.6")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "modbedtools" version))
       (sha256
        (base32 "0c2f8fl0mi3wcvv4fygkf6jj1d0zavhc7v9wwbqchpyb7m23cmp9"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list python-pysam))
    (native-inputs (list python-pytest
                         python-setuptools
                         python-setuptools-scm
                         python-wheel))
    (arguments (list #:tests? #f)) ;No tests
    (home-page "https://github.com/lidaof/modbedtools")
    (synopsis
     "Generate modbed track files for visualization on WashU Epigenome Browser")
    (description
     "modbedtools is a python command line tool to generate modbed files for
visualization on the WashU Epigenome Browser.")
    (license license:expat)))

(define-public python-ete3
  (package
    (name "python-ete3")
    (version "3.1.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/etetoolkit/ete")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1i6533wsm06mz0sdrisqai929j744cnczwjgsmxl847q5k16kngd"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k"
             ;; This test crashes Python in the build container
             (string-append "not test_renderer"
                            ;; These all need internet access
                            " and not test_00_update_database"
                            " and not test_01tree_annotation"
                            " and not test_get_topology"
                            " and not test_merged_id"
                            " and not test_ncbi_compare"
                            " and not test_ncbiquery")
             "ete3/test/test_api.py")))
    (propagated-inputs
     (list python-lxml
           python-numpy
           python-pyqt
           python-scipy))
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (home-page "http://etetoolkit.org")
    (synopsis "Python environment for phylogenetic tree exploration")
    (description
     "This package provides a Python environment for phylogenetic tree
exploration.")
    (license license:gpl3+)))

(define-public python-illumina-utils
  (package
    (name "python-illumina-utils")
    (version "2.13")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/meren/illumina-utils")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "15cyb5slw07va5siq8nzc0nwcgnvx1hmqqrgwk2v0fxy250fp9v4"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f                       ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'adjust-requirements
            (lambda _
              (substitute* "requirements.txt"
                (("python-Levenshtein")
                 "levenshtein")))))))
    (propagated-inputs (list python-levenshtein python-matplotlib python-numpy))
    (native-inputs (list python-pip python-setuptools))
    (home-page "https://github.com/meren/illumina-utils")
    (synopsis "Library and scripts to work with Illumina paired-end data")
    (description
     "This package provides a library and collection of scripts to work with
Illumina paired-end data (for CASAVA 1.8+).")
    (license license:gpl2+)))

(define-public python-pyani
  (package
    (name "python-pyani")
    (version "0.2.12")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "pyani" version))
              (sha256
               (base32
                "124kdg7168nbh4a5wisfws1fgkd89dd4js9v6dml2lvgclbv4mjg"))))
    (build-system pyproject-build-system)
    ;; This contains two test data files but no way to run the tests.
    (arguments (list #:tests? #false))
    (propagated-inputs
     (list python-biopython python-matplotlib python-pandas
           python-scipy python-seaborn))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://widdowquinn.github.io/pyani/")
    (synopsis "Calculate genome-scale average nucleotide identity")
    (description
     "Pyani provides a package and script for calculation of genome-scale
average nucleotide identity.")
    (license license:expat)))

(define-public python-pyahocorasick
  (package
    (name "python-pyahocorasick")
    (version "2.2.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/WojciechMula/pyahocorasick")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0r9n8awy80dg8dmgza4kpgwbpkvjf4s85cyswnq04h3x6cf62lll"))))
    (build-system pyproject-build-system)
    (native-inputs (list python-pytest python-setuptools))
    (home-page "https://github.com/WojciechMula/pyahocorasick")
    (synopsis "Library for finding multiple key strings in text")
    (description
     "Pyahocorasick is a fast, memory-efficient library for multi-pattern string search.
This means that you can find multiple key strings occurrences
at once in some input text.")
    (license license:bsd-3)))

(define-public exonerate
  (package
    (name "exonerate")
    (version "2.4.0")
    (source
     (origin
       (method url-fetch)
       (uri
        (string-append
         "http://ftp.ebi.ac.uk/pub/software/vertebrategenomics/exonerate/"
         "exonerate-" version ".tar.gz"))
       (sha256
        (base32
         "0hj0m9xygiqsdxvbg79wq579kbrx1mdrabi2bzqz2zn9qwfjcjgq"))))
    (build-system gnu-build-system)
    (arguments
     `(#:parallel-build? #f)) ; Building in parallel fails on some machines.
    (native-inputs
     (list pkg-config))
    (inputs
     (list glib))
    (home-page
     "https://www.ebi.ac.uk/about/vertebrate-genomics/software/exonerate")
    (synopsis "Generic tool for biological sequence alignment")
    (description
     "Exonerate is a generic tool for pairwise sequence comparison.  It allows
the alignment of sequences using a many alignment models, either exhaustive
dynamic programming or a variety of heuristics.")
    (license license:gpl3)))

(define-public express
  (package
    (name "express")
    (version "1.5.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/adarob/eXpress")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18nb22n7x820fzjngf4qgyb3mspqkw7xyk7v7s5ps6wfrd8qwscb"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:tests? #f                     ;no "check" target
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'use-shared-boost-libs-and-set-bamtools-paths
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "CMakeLists.txt"
                (("set\\(Boost_USE_STATIC_LIBS ON\\)")
                 "set(Boost_USE_STATIC_LIBS OFF)")
                (("\\$\\{CMAKE_CURRENT_SOURCE_DIR\\}/bamtools/include")
                 (search-input-directory inputs "/include/bamtools"))
                (("\\$\\{PROJECT_SOURCE_DIR\\}/bamtools/lib/libbamtools.a")
                 (search-input-file inputs "/lib/libbamtools.so"))
                (("libbamtools.a") "libbamtools.so"))
              (substitute* "src/CMakeLists.txt"
                (("libprotobuf.a") "libprotobuf.so"))))
          (add-after 'unpack 'remove-update-check
            (lambda _
              (substitute* "src/main.cpp"
                (("#include \"update_check.h\"") "")
                (("check_version\\(PACKAGE_VERSION\\);") "")))))))
    (inputs
     (list boost-1.83 bamtools protobuf zlib))
    (home-page "http://bio.math.berkeley.edu/eXpress")
    (synopsis "Streaming quantification for high-throughput genomic sequencing")
    (description
     "eXpress is a streaming tool for quantifying the abundances of a set of
target sequences from sampled subsequences.  Example applications include
transcript-level RNA-Seq quantification, allele-specific/haplotype expression
analysis (from RNA-Seq), transcription factor binding quantification in
ChIP-Seq, and analysis of metagenomic data.")
    (properties `((lint-hidden-cpe-vendors . ("openjsf" "qs_project"))))
    (license license:artistic2.0)))

(define-public express-beta-diversity
  (package
   (name "express-beta-diversity")
   (version "1.0.8")
   (source (origin
             (method git-fetch)
             (uri (git-reference
                   (url "https://github.com/dparks1134/ExpressBetaDiversity")
                   (commit (string-append "v" version))))
             (file-name (git-file-name name version))
             (sha256
              (base32
               "0s0yzg5c21349rh7x4w9266jsvnp7j1hp9cf8sk32hz8nvrj745x"))))
   (build-system gnu-build-system)
   (arguments
    `(#:phases
      (modify-phases %standard-phases
        (delete 'configure)
        (add-before 'build 'enter-source (lambda _ (chdir "source") #t))
        (replace 'check
          (lambda _ (invoke "../bin/ExpressBetaDiversity" "-u") #t))
        (replace 'install
          (lambda* (#:key outputs #:allow-other-keys)
            (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
              (install-file "../scripts/convertToEBD.py" bin)
              (install-file "../bin/ExpressBetaDiversity" bin)
              #t))))))
   (inputs
    `(("python" ,python-2)))
   (home-page "https://github.com/dparks1134/ExpressBetaDiversity")
   (synopsis "Taxon- and phylogenetic-based beta diversity measures")
   (description
    "Express Beta Diversity (EBD) calculates ecological beta diversity
(dissimilarity) measures between biological communities.  EBD implements a
variety of diversity measures including those that make use of phylogenetic
similarity of community members.")
   (license license:gpl3+)))

(define-public fasttree
  (package
   (name "fasttree")
   (version "2.1.10")
   (source (origin
             (method url-fetch)
             (uri (string-append
                   "http://www.microbesonline.org/fasttree/FastTree-"
                   version ".c"))
             (sha256
              (base32
               "0vcjdvy1j4m702vmak4svbfkrpcw63k7wymfksjp9a982zy8kjsl"))))
   (build-system gnu-build-system)
   (arguments
    `(#:tests? #f ; no "check" target
      #:phases
      (modify-phases %standard-phases
        (delete 'unpack)
        (delete 'configure)
        (replace 'build
          (lambda* (#:key source #:allow-other-keys)
            (invoke "gcc"
                    "-O3"
                    "-finline-functions"
                    "-funroll-loops"
                    "-Wall"
                    "-o"
                    "FastTree"
                    source
                    "-lm")
            (invoke "gcc"
                    "-DOPENMP"
                    "-fopenmp"
                    "-O3"
                    "-finline-functions"
                    "-funroll-loops"
                    "-Wall"
                    "-o"
                    "FastTreeMP"
                    source
                    "-lm")
            #t))
        (replace 'install
          (lambda* (#:key outputs #:allow-other-keys)
            (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
              (install-file "FastTree" bin)
              (install-file "FastTreeMP" bin)
              #t))))))
   (home-page "http://www.microbesonline.org/fasttree")
   (synopsis "Infers approximately-maximum-likelihood phylogenetic trees")
   (description
    "FastTree can handle alignments with up to a million of sequences in a
reasonable amount of time and memory.  For large alignments, FastTree is
100-1,000 times faster than PhyML 3.0 or RAxML 7.")
   (license license:gpl2+)))

(define-public fastx-toolkit
  (package
    (name "fastx-toolkit")
    (version "0.0.14")
    (source (origin
              (method url-fetch)
              (uri
               (string-append
                "https://github.com/agordon/fastx_toolkit/releases/download/"
                version "/fastx_toolkit-" version ".tar.bz2"))
              (sha256
               (base32
                "01jqzw386873sr0pjp1wr4rn8fsga2vxs1qfmicvx1pjr72007wy"))))
    (build-system gnu-build-system)
    (inputs
     (list libgtextutils))
    (native-inputs
     (list gcc-6 ;; doesn't build with later versions
           pkg-config))
    (home-page "http://hannonlab.cshl.edu/fastx_toolkit/")
    (synopsis "Tools for FASTA/FASTQ file preprocessing")
    (description
     "The FASTX-Toolkit is a collection of command line tools for Short-Reads
FASTA/FASTQ files preprocessing.

Next-Generation sequencing machines usually produce FASTA or FASTQ files,
containing multiple short-reads sequences.  The main processing of such
FASTA/FASTQ files is mapping the sequences to reference genomes.  However, it
is sometimes more productive to preprocess the files before mapping the
sequences to the genome---manipulating the sequences to produce better mapping
results.  The FASTX-Toolkit tools perform some of these preprocessing tasks.")
    (license license:agpl3+)))

(define-public flash
  (package
    (name "flash")
    (version "1.2.11")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "mirror://sourceforge/flashpage/FLASH-"
                           version ".tar.gz"))
       (sha256
        (base32
         "1b1ns9ghbcxy92xwa2a53ikqacvnyhvca0zfv0s7986xzvvscp38"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:make-flags #~(list (string-append "CC=" #$(cc-for-target)))
      #:tests? #f                       ;no tests
      #:phases
      #~(modify-phases %standard-phases
          ;; No configure script
          (delete 'configure)
          ;; No install target
          (replace 'install
            (lambda _
              (install-file "flash"
                            (string-append #$output "/bin")))))))
    (inputs (list zlib))
    (home-page "http://ccb.jhu.edu/software/FLASH/")
    (synopsis "Merge paired-end nucleotide reads from NGS experiments")
    (description "FLASH (Fast Length Adjustment of SHort reads) is a tool to
merge paired-end reads from next-generation sequencing experiments.  FLASH is
designed to merge pairs of reads when the original DNA fragments are shorter
than twice the length of reads.  The resulting longer reads can significantly
improve genome assemblies.  They can also improve transcriptome assembly when
FLASH is used to merge RNA-seq data.")
    (license license:gpl3+)))

(define-public flexbar
  (package
    (name "flexbar")
    (version "3.4.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/seqan/flexbar")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pq9sxvdnldl14libk234m72dqhwgzs3acgl943wchwdqlcsi5r2"))))
    (build-system cmake-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'do-not-tune-to-CPU
           (lambda _
             (substitute* "src/CMakeLists.txt"
               ((" -march=native") ""))))
         (replace 'check
           (lambda* (#:key outputs #:allow-other-keys)
             (setenv "PATH" (string-append (getcwd) ":" (getenv "PATH")))
             (with-directory-excursion "../source/test"
               (invoke "bash" "flexbar_test.sh"))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (string-append (assoc-ref outputs "out")))
                    (bin (string-append out "/bin/")))
               (install-file "flexbar" bin)))))))
    (inputs
     (list tbb-2020 zlib))
    (native-inputs
     (list pkg-config seqan-2))
    (home-page "https://github.com/seqan/flexbar")
    (synopsis "Barcode and adapter removal tool for sequencing platforms")
    (description
     "Flexbar preprocesses high-throughput nucleotide sequencing data
efficiently.  It demultiplexes barcoded runs and removes adapter sequences.
Moreover, trimming and filtering features are provided.  Flexbar increases
read mapping rates and improves genome and transcriptome assemblies.  It
supports next-generation sequencing data in fasta/q and csfasta/q format from
Illumina, Roche 454, and the SOLiD platform.")
    (license license:bsd-3)))

(define-public fxtract
  (let ((util-commit "776ca85a18a47492af3794745efcb4a905113115"))
    (package
      (name "fxtract")
      (version "2.3")
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/ctSkennerton/fxtract")
               (commit version)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0hab3gpwf4w9s87qlbswq6ws1qqybh4dcqk79q1ahyldzai5fgp5"))))
      (build-system gnu-build-system)
      (arguments
       `(#:make-flags ,#~(list
                          (string-append "PREFIX=" #$output)
                          "CC=gcc")
         #:test-target "fxtract_test"
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)
           (add-before 'build 'copy-util
             (lambda* (#:key inputs #:allow-other-keys)
               (rmdir "util")
               (copy-recursively (assoc-ref inputs "ctskennerton-util") "util")
               #t))
           ;; Do not use make install as this requires additional dependencies.
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (let* ((out (assoc-ref outputs "out"))
                      (bin (string-append out"/bin")))
                 (install-file "fxtract" bin)
                 #t))))))
      (inputs
       (list pcre zlib))
      (native-inputs
       ;; ctskennerton-util is licensed under GPL2.
       `(("ctskennerton-util"
          ,(origin
             (method git-fetch)
             (uri (git-reference
                   (url "https://github.com/ctSkennerton/util")
                   (commit util-commit)))
             (file-name (string-append
                         "ctstennerton-util-" util-commit "-checkout"))
             (sha256
              (base32
               "0cls1hd4vgj3f36fpzzg4xc77d6f3hpc60cbpfmn2gdr7ykzzad7"))))))
      (home-page "https://github.com/ctSkennerton/fxtract")
      (synopsis "Extract sequences from FASTA and FASTQ files")
      (description
       "Fxtract extracts sequences from a protein or nucleotide fastx (FASTA
or FASTQ) file given a subsequence.  It uses a simple substring search for
basic tasks but can change to using POSIX regular expressions, PCRE, hash
lookups or multi-pattern searching as required.  By default fxtract looks in
the sequence of each record but can also be told to look in the header,
comment or quality sections.")
      ;; 'util' requires SSE instructions.
      (supported-systems '("x86_64-linux"))
      (license license:expat))))

(define-public gemma
  (package
    (name "gemma")
    (version "0.98.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                     (url "https://github.com/genetics-statistics/GEMMA")
                     (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1dm8pf1fbdmv2yiz5aybcvk3050m5350gq8xlr4j6swzm3wwhydn"))
              (modules '((guix build utils)))
              (snippet
               #~(begin
                   (delete-file-recursively "contrib")))))
    (build-system gnu-build-system)
    (inputs
     (list gsl openblas zlib))
    (native-inputs
     (list catch2-1
           perl
           shunit2
           which))
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (add-after 'unpack 'prepare-build
            (lambda _
              (mkdir-p "bin")
              (substitute* "Makefile"
                (("/usr/local/opt/openblas")
                 #$(this-package-input "openblas")))))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                ;; 'make slow-check' expects shunit2-2.0.3.
                (with-directory-excursion "test"
                  (invoke "./test_suite.sh")))))
          (replace 'install
            (lambda _
              (install-file "bin/gemma" (string-append #$output "/bin")))))))
    (home-page "https://github.com/genetics-statistics/GEMMA")
    (synopsis "Tool for genome-wide efficient mixed model association")
    (description
     "@acronym{GEMMA, Genome-wide Efficient Mixed Model Association} provides a
standard linear mixed model resolver with application in @acronym{GWAS,
genome-wide association studies}.")
    (license license:gpl3)))

(define-public hisat
  (package
    (name "hisat")
    (version "0.1.6")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "http://ccb.jhu.edu/software/hisat/downloads/hisat-"
                    version "-beta-source.zip"))
              (sha256
               (base32
                "177z85vqp9b30vgxl5py5hz4mm37ila37nzhfam23ci9iyfxgyv9"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no check target
       #:make-flags '("allall"
                      ;; Disable unsupported `popcnt' instructions on
                      ;; architectures other than x86_64
                      ,@(if (string-prefix? "x86_64"
                                            (or (%current-target-system)
                                                (%current-system)))
                            '()
                            '("POPCNT_CAPABILITY=0")))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'patch-sources
           (lambda _
             ;; XXX Cannot use snippet because zip files are not supported
             (substitute* "Makefile"
               (("^CC = .*$") "CC = gcc")
               (("^CPP = .*$") "CPP = g++")
               ;; replace BUILD_HOST and BUILD_TIME for deterministic build
               (("-DBUILD_HOST=.*") "-DBUILD_HOST=\"\\\"guix\\\"\"")
               (("-DBUILD_TIME=.*") "-DBUILD_TIME=\"\\\"0\\\"\""))
             (substitute* '("hisat-build" "hisat-inspect")
               (("/usr/bin/env") (which "env")))
             ;; This "extended character" is not considered valid.
             (substitute* "processor_support.h"
               (("“") "\"")
               (("”") "\""))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
               (for-each (lambda (file)
                           (install-file file bin))
                         (find-files
                          "."
                          "hisat(-(build|align|inspect)(-(s|l)(-debug)*)*)*$")))))
         (delete 'configure))))
    (native-inputs
     (list gcc-10 unzip))
    (inputs
     (list perl python zlib))
    ;; Non-portable SSE instructions are used so building fails on platforms
    ;; other than x86_64.
    (supported-systems '("x86_64-linux"))
    (home-page "https://ccb.jhu.edu/software/hisat/index.shtml")
    (synopsis "Hierarchical indexing for spliced alignment of transcripts")
    (description
     "HISAT is a fast and sensitive spliced alignment program for mapping
RNA-seq reads.  In addition to one global FM index that represents a whole
genome, HISAT uses a large set of small FM indexes that collectively cover the
whole genome.  These small indexes (called local indexes) combined with
several alignment strategies enable effective alignment of RNA-seq reads, in
particular, reads spanning multiple exons.")
    (license license:gpl3+)))

(define-public hisat2
  (package
    (name "hisat2")
    (version "2.2.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/DaehwanKimLab/hisat2/")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0lmzdhzjkvxw7n5w40pbv5fgzd4cz0f9pxczswn3d4cr0k10k754"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; no check target
       #:make-flags (list "CC=gcc" "CXX=g++" "allall")
       #:modules ((guix build gnu-build-system)
                  (guix build utils)
                  (srfi srfi-26))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'make-deterministic
           (lambda _
             (substitute* "Makefile"
               (("`date`") "0"))))
         (delete 'configure)
         (add-before 'build 'build-manual
           (lambda _
             (mkdir-p "doc")
             (invoke "make" "doc")))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin/"))
                    (doc (string-append out "/share/doc/hisat2/")))
               (for-each
                (cut install-file <> bin)
                (find-files "."
                            "hisat2(-(build|align|inspect)(-(s|l)(-debug)*)*)*$"))
               (mkdir-p doc)
               (install-file "doc/manual.inc.html" doc)))))))
    (native-inputs
     (list perl pandoc))             ; for documentation
    (inputs
     `(("python" ,python-wrapper)))
    (home-page "https://daehwankimlab.github.io/hisat2/")
    (synopsis "Graph-based alignment of genomic sequencing reads")
    (description "HISAT2 is a fast and sensitive alignment program for mapping
next-generation sequencing reads (both DNA and RNA) to a population of human
genomes (as well as to a single reference genome).  In addition to using one
global @dfn{graph FM} (GFM) index that represents a population of human
genomes, HISAT2 uses a large set of small GFM indexes that collectively cover
the whole genome.  These small indexes, combined with several alignment
strategies, enable rapid and accurate alignment of sequencing reads.  This new
indexing scheme is called a @dfn{Hierarchical Graph FM index} (HGFM).")
    ;; HISAT2 contains files from Bowtie2, which is released under
    ;; GPLv2 or later.  The HISAT2 source files are released under
    ;; GPLv3 or later.
    (license license:gpl3+)))

(define-public homer
  (package
    (name "homer")
    (version "4.11.1")
    (source
     (origin
       (method url-fetch/zipbomb)
       (uri (string-append "http://homer.ucsd.edu/homer/data/software/homer.v"
                           version ".zip"))
       (sha256
        (base32
         "0ay802swzq6ix9d8fkinpplcvyc1xyi3cjmj2x08jab7c40cvlc0"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda* (#:key inputs #:allow-other-keys)
              (let ((share (string-append #$output "/share/homer")))
                (mkdir-p share)
                (copy-file (assoc-ref inputs "config.txt")
                           (string-append share "/config.txt"))
                (substitute* "configureHomer.pl"
                  (("my \\$homeDir = \\$1;")
                   (string-append "my $homeDir = \"" share "\";"))))))
          (replace 'build
            (lambda _
              (let ((share (string-append #$output "/share/homer")))
                (copy-recursively "." share))
              (invoke "perl" "configureHomer.pl" "-local")))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (invoke (string-append #$output "/share/homer/bin/homer2")))))
          (replace 'install
            (lambda _
              (mkdir-p (string-append #$output "/bin"))
              (symlink (string-append #$output "/share/homer/bin/homer2")
                       (string-append #$output "/bin/homer2"))
              (symlink (string-append #$output "/share/homer/bin/homer")
                       (string-append #$output "/bin/homer"))
              (for-each patch-shebang
                        (find-files (string-append #$output "/share/homer/bin")
                                    "\\.pl$"))
              ;; Also load config file from user's home directory.
              (substitute* (string-append #$output "/share/homer/bin/HomerConfig.pm")
                (("#parseConfigFile") "parseConfigFile")))))))
    (inputs
     (list perl))
    (native-inputs
     `(("perl" ,perl)
       ("unzip" ,unzip)
       ("config.txt"
        ,(origin
           (method url-fetch)
           (uri (string-append "https://web.archive.org/web/20200531014112id_/"
                               "http://homer.ucsd.edu/homer/update.txt"))
           (sha256
            (base32
             "1hf17pk8r6b297ysd27bvxzyn8pxdhxd8wj8g0lqlifbid9fw04h"))))))
    (home-page "http://homer.ucsd.edu/homer")
    (synopsis "Motif discovery and next generation sequencing analysis")
    (description
     "HOMER (Hypergeometric Optimization of Motif EnRichment) is a suite of
tools for Motif Discovery and next-gen sequencing analysis.  It is a
collection of command line programs written in Perl and C++.  HOMER was
primarily written as a de novo motif discovery algorithm and is well suited
for finding 8-20 bp motifs in large scale genomics data.  HOMER contains many
useful tools for analyzing ChIP-Seq, GRO-Seq, RNA-Seq, DNase-Seq, Hi-C and
numerous other types of functional genomics sequencing data sets.")
    (license license:gpl3+)))

(define-public hmmer
  (package
    (name "hmmer")
    (version "3.3.2")
    (source
     (origin
       (method url-fetch)
       (uri (string-append
             "http://eddylab.org/software/hmmer/hmmer-" version ".tar.gz"))
       (sha256
        (base32
         "0s9wf6n0qanbx8qs6igfl3vyjikwbrvh4d9d6mv54yp3xysykzlj"))))
    (build-system gnu-build-system)
    (native-inputs (list perl python)) ; for tests
    (home-page "http://hmmer.org/")
    (synopsis "Biosequence analysis using profile hidden Markov models")
    (description
     "HMMER is used for searching sequence databases for homologs of protein
sequences, and for making protein sequence alignments.  It implements methods
using probabilistic models called profile hidden Markov models (profile
HMMs).")
    ;; hmmer uses non-portable SSE intrinsics so building fails on other
    ;; platforms.
    (supported-systems '("x86_64-linux" "i686-linux"))
    (license license:bsd-3)))

(define-public htseq
  (package
    (name "htseq")
    (version "2.0.2")
    ;; Sources on pypi do not include everything needed to run the tests.
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/htseq/htseq")
                    (commit (string-append "release_" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1kbr4ydjjhizz6r5m3xd4f0wj7qnn8zs0vnzghhgaa0yhbya5r19"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list python-matplotlib
           python-numpy
           python-pysam))
    (native-inputs
     (list python-cython
           python-pandas
           python-pytest
           python-scipy
           python-setuptools
           python-wheel
           swig))
    (home-page "https://github.com/htseq")
    (synopsis "Framework for analyzing high-throughput sequencing data")
    (description
     "This package provides a framework to process and analyze data from
high-throughput sequencing (HTS) assays")
    (license license:gpl3+)))

(define-public python-genomepy
  (package
    (name "python-genomepy")
    (version "0.15.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "genomepy" version))
              (sha256
               (base32
                "0jmyvnsn6w0djjmxh4fjspy1346337jzihxb276v3s275r6zjmln"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; The tests require internet access.
      #:tests? #false
      #:phases
      '(modify-phases %standard-phases
         ;; Needed by tests
         (add-after 'unpack 'set-HOME
           (lambda _ (setenv "HOME" "/tmp"))))))
    (propagated-inputs (list python-appdirs
                             python-biopython
                             python-click
                             python-colorama
                             python-diskcache
                             python-filelock
                             python-loguru
                             python-mygene
                             python-mysql-connector-python
                             python-norns
                             python-numpy
                             python-pandas
                             python-pyfaidx
                             python-requests
                             python-tqdm))
    (native-inputs (list python-hatchling python-pytest))
    (home-page "https://vanheeringen-lab.github.io/genomepy/")
    (synopsis "Genes and genomes at your fingertips")
    (description "genomepy is designed to provide a simple and straightforward
way to download and use genomic data.  This includes

@enumerate
@item searching available data,
@item showing the available metadata,
@item automatically downloading, preprocessing and matching data, and
@item generating optional aligner indexes.
@end enumerate

All with sensible, yet controllable defaults.")
    (license license:expat)))

(define-public java-htsjdk
  (package
    (name "java-htsjdk")
    (version "2.3.0") ; last version without build dependency on gradle
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/samtools/htsjdk")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1b178ixcabanm834ydjl3jiakpyxdmki32hqfv2abrzn3rcwa28i"))
              (modules '((guix build utils)))
              (snippet
               ;; Delete pre-built binaries
               '(begin
                  (delete-file-recursively "lib")
                  (mkdir-p "lib")
                  #t))))
    (build-system ant-build-system)
    (arguments
     `(#:tests? #f ; test require Internet access
       #:jdk ,icedtea-8
       #:make-flags
       ,#~(list (string-append "-Ddist=" #$output "/share/java/htsjdk/"))
       #:build-target "all"
       #:phases
       (modify-phases %standard-phases
         ;; The build phase also installs the jars
         (delete 'install))))
    (inputs
     `(("java-ngs" ,java-ngs)
       ("java-snappy-1" ,java-snappy-1)
       ("java-commons-compress" ,java-commons-compress)
       ("java-commons-logging-minimal" ,java-commons-logging-minimal)
       ("java-commons-jexl-2" ,java-commons-jexl-2)
       ("java-xz" ,java-xz)))
    (native-inputs
     (list java-testng))
    (home-page "https://samtools.github.io/htsjdk/")
    (synopsis "Java API for high-throughput sequencing data (HTS) formats")
    (description
     "HTSJDK is an implementation of a unified Java library for accessing
common file formats, such as SAM and VCF, used for high-throughput
sequencing (HTS) data.  There are also an number of useful utilities for
manipulating HTS data.")
    (license license:expat)))

(define-public java-htsjdk-latest
  (package
    (name "java-htsjdk")
    (version "2.14.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/samtools/htsjdk")
                    (commit version)))
              (file-name (string-append name "-" version "-checkout"))
              (sha256
               (base32
                "1lmya1fdjy03mz6zmdmd86j9v9vfhqb3952mqq075navx1i6g4bc"))))
    (build-system ant-build-system)
    (arguments
     `(#:tests? #f                      ; test require Scala
       #:jdk ,icedtea-8
       #:jar-name "htsjdk.jar"
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'remove-useless-build.xml
           (lambda _ (delete-file "build.xml") #t))
         ;; The tests require the scalatest package.
         (add-after 'unpack 'remove-tests
           (lambda _ (delete-file-recursively "src/test") #t)))))
    (inputs
     `(("java-ngs" ,java-ngs)
       ("java-snappy-1" ,java-snappy-1)
       ("java-commons-compress" ,java-commons-compress)
       ("java-commons-logging-minimal" ,java-commons-logging-minimal)
       ("java-commons-jexl-2" ,java-commons-jexl-2)
       ("java-xz" ,java-xz)))
    (native-inputs
     (list java-junit))
    (home-page "https://samtools.github.io/htsjdk/")
    (synopsis "Java API for high-throughput sequencing data (HTS) formats")
    (description
     "HTSJDK is an implementation of a unified Java library for accessing
common file formats, such as SAM and VCF, used for high-throughput
sequencing (HTS) data.  There are also an number of useful utilities for
manipulating HTS data.")
    (license license:expat)))

;; This is needed for picard 2.10.3
(define-public java-htsjdk-2.10.1
  (package (inherit java-htsjdk-latest)
    (name "java-htsjdk")
    (version "2.10.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/samtools/htsjdk")
                    (commit version)))
              (file-name (string-append name "-" version "-checkout"))
              (sha256
               (base32
                "1kxh7slm2pm3x9p6jxa1wqsq9a31dhiiflhxnxqcisan4k3rwia2"))))
    (build-system ant-build-system)
    (arguments
     `(#:tests? #f                      ; tests require Scala
       #:jdk ,icedtea-8
       #:jar-name "htsjdk.jar"
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'remove-useless-build.xml
           (lambda _ (delete-file "build.xml") #t))
         ;; The tests require the scalatest package.
         (add-after 'unpack 'remove-tests
           (lambda _ (delete-file-recursively "src/test") #t)))))))

(define-public java-maxent
  (package
    (name "java-maxent")
    (version "3.4.4")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/mrmaxent/Maxent")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12q7hhly76l77vm8w8v9icga2gn6xs0bw33a7wb7zikcmvizcyp0"))))
    (build-system ant-build-system)
    (arguments
     (list
      #:tests? #false                   ;there are none
      #:jdk icedtea-8
      #:phases
      #~(modify-phases %standard-phases
          (replace 'build
            (lambda* (#:key inputs #:allow-other-keys)
              ;; See https://github.com/mrmaxent/Maxent/pull/11
              (substitute* "density/Extractor.java"
                (("float") "double"))
              (with-directory-excursion "gnu/getopt/"
                (invoke "make"))
              (mkdir-p "ptii")
              (with-directory-excursion "ptii"
                (invoke "tar" "xvf" (assoc-ref inputs "ptii")
                        "--strip-components=1")
                (copy-recursively "com" "../com"))
              (delete-file-recursively "ptii")
              (apply invoke "javac" "-cp" (getcwd) "-g"
                     (find-files "com/microstar/xml" "\\.java$"))
              (apply invoke "javac" "-cp" (getcwd) "-g"
                     (find-files "density" "\\.java$"))

              ;; This needs the proprietary com.sun.image.codec.jpeg module.
              (delete-file "ptolemy/plot/servlet/PlotServlet.java")
              (apply invoke "javac" "-cp"
                     (string-append (getcwd) ":" (getenv "CLASSPATH")) "-g"
                     (find-files "ptolemy/plot" "\\.java$"))
              (apply invoke "javac" "-cp" (getcwd) "-g"
                     (find-files "com" "\\.java$"))
              (apply invoke "javac" "-cp" (getcwd) "-g"
                     (find-files "gui" "\\.java$"))
              (apply invoke "jar" "cvfm" "maxent.jar"
                     (cons* "density/mc.mf"
                            "density/parameters.csv"
                            (append (find-files "density" "\\.class$")
                                    (find-files "density" "\\.html$")
                                    (find-files "gnu/getopt" ".*")
                                    (find-files "gui/layouts" "\\.class$")
                                    (find-files "com/macfaq/io" "\\.class$")
                                    (find-files "density/tools" "\\.class$")
                                    (find-files "ptolemy/plot" "\\.class$"))))))
          (replace 'install
            (lambda _
              (install-file "maxent.jar"
                            (string-append #$output "/share/java/maxent/")))))))
    (inputs
     (list java-classpathx-servletapi))
    (native-inputs
     `(("make" ,gnu-make)
       ;; For com.microstar.xml
       ("ptii"
        ,(let ((version "4.0.1"))
           (origin
             (method url-fetch)
             (uri (string-append "https://ptolemy.berkeley.edu/ptolemyII/ptII"
                                 (version-major+minor version)
                                 "/ptII" version ".src.tar.gz"))
             (sha256
              (base32
               "0ifmmvrcipcnd4b9im1g379ffrs7g7k99sw5vv9d9h3hzq6hqv21")))))))
    (home-page "http://biodiversityinformatics.amnh.org/open_source/maxent")
    (synopsis "Model species geographic distributions")
    (description
     "Maxent is a stand-alone Java application for modelling species
geographic distributions.")
    (license license:expat)))

;; This version matches java-htsjdk 2.3.0.  Later versions also require a more
;; recent version of java-htsjdk, which depends on gradle.
(define-public java-picard
  (package
    (name "java-picard")
    (version "2.3.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/broadinstitute/picard")
                    (commit version)))
              (file-name (string-append "java-picard-" version "-checkout"))
              (sha256
               (base32
                "1ll7mf4r3by92w2nhlmpa591xd1f46xlkwh59mq6fvbb5pdwzvx6"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  ;; Delete pre-built binaries.
                  (delete-file-recursively "lib")
                  (mkdir-p "lib")
                  (substitute* "build.xml"
                    ;; Remove build-time dependency on git.
                    (("failifexecutionfails=\"true\"")
                     "failifexecutionfails=\"false\"")
                    ;; Use our htsjdk.
                    (("depends=\"compile-htsjdk, ")
                     "depends=\"")
                    (("depends=\"compile-htsjdk-tests, ")
                     "depends=\"")
                    ;; Build picard-lib.jar before building picard.jar
                    (("name=\"picard-jar\" depends=\"" line)
                     (string-append line "picard-lib-jar, ")))
                  #t))))
    (build-system ant-build-system)
    (arguments
     `(#:build-target "picard-jar"
       #:test-target "test"
       ;; Tests require jacoco:coverage.
       #:tests? #f
       #:make-flags
       ,#~(list (string-append "-Dhtsjdk_lib_dir="
                               #$(this-package-input "java-htsjdk")
                               "/share/java/htsjdk/")
                "-Dhtsjdk-classes=dist/tmp"
                (string-append "-Dhtsjdk-version="
                               #$(package-version java-htsjdk)))
       #:jdk ,icedtea-8
       #:phases
       (modify-phases %standard-phases
         ;; FIXME: this phase fails with "duplicate entry: htsjdk/samtools/AbstractBAMFileIndex$1.class"
         (delete 'generate-jar-indices)
         (add-after 'unpack 'fix-guava
           (lambda _
             (substitute* "src/java/picard/cmdline/CommandLineParser.java"
               (("CharMatcher.ASCII") "CharMatcher.ascii()"))))
         (add-after 'unpack 'use-our-htsjdk
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "build.xml"
               (("\\$\\{htsjdk\\}/lib")
                (search-input-directory inputs
                                        "share/java/htsjdk")))))
         (add-after 'unpack 'make-test-target-independent
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "build.xml"
               (("name=\"test\" depends=\"compile, ")
                "name=\"test\" depends=\""))
             #t))
         (replace 'install (install-jars "dist")))))
    (inputs
     (list java-htsjdk java-guava))
    (native-inputs
     (list java-testng))
    (home-page "https://broadinstitute.github.io/picard/")
    (synopsis "Tools for manipulating high-throughput sequencing data and formats")
    (description "Picard is a set of Java command line tools for manipulating
high-throughput sequencing (HTS) data and formats.  Picard is implemented
using the HTSJDK Java library to support accessing file formats that are
commonly used for high-throughput sequencing data such as SAM, BAM, CRAM and
VCF.")
    (license license:expat)))

;; This is needed for dropseq-tools
(define-public java-picard-2.10.3
  (package
    (name "java-picard")
    (version "2.10.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/broadinstitute/picard")
                    (commit version)))
              (file-name (string-append "java-picard-" version "-checkout"))
              (sha256
               (base32
                "1ajlx31l6i1k3y2rhnmgq07sz99g2czqfqgkr9mihmdjp3gwjhvi"))))
    (build-system ant-build-system)
    (arguments
     `(#:jar-name "picard.jar"
       ;; Tests require jacoco:coverage.
       #:tests? #f
       #:jdk ,icedtea-8
       #:main-class "picard.cmdline.PicardCommandLine"
       #:modules ((guix build ant-build-system)
                  (guix build utils)
                  (guix build java-utils)
                  (sxml simple)
                  (sxml transform)
                  (sxml xpath))
       #:phases
       (modify-phases %standard-phases
         ;; FIXME: this phase fails with "duplicate entry: htsjdk/samtools/AbstractBAMFileIndex$1.class"
         (delete 'generate-jar-indices)
         (add-after 'unpack 'remove-useless-build.xml
           (lambda _ (delete-file "build.xml") #t))
         (add-after 'unpack 'fix-guava
           (lambda _
             (substitute* "src/main/java/picard/cmdline/CommandLineParser.java"
               (("CharMatcher.ASCII") "CharMatcher.ascii()"))))
         ;; This is necessary to ensure that htsjdk is found when using
         ;; picard.jar as an executable.
         (add-before 'build 'edit-classpath-in-manifest
           (lambda* (#:key inputs #:allow-other-keys)
             (chmod "build.xml" #o664)
             (call-with-output-file "build.xml.new"
               (lambda (port)
                 (sxml->xml
                  (pre-post-order
                   (with-input-from-file "build.xml"
                     (lambda _ (xml->sxml #:trim-whitespace? #t)))
                   `((target    . ,(lambda (tag . kids)
                                     (let ((name ((sxpath '(name *text*))
                                                  (car kids)))
                                           ;; FIXME: We're breaking the line
                                           ;; early with a dummy path to
                                           ;; ensure that the store reference
                                           ;; isn't broken apart and can still
                                           ;; be found by the reference
                                           ;; scanner.
                                           (msg (format #f
                                                        "\
Class-Path: /~a \
 ~a/share/java/htsjdk.jar${line.separator}${line.separator}"
                                                        ;; maximum line length is 70
                                                        (string-tabulate (const #\b) 57)
                                                        (assoc-ref inputs "java-htsjdk"))))
                                       (if (member "manifest" name)
                                           `(,tag ,@kids
                                                  (replaceregexp
                                                   (@ (file "${manifest.file}")
                                                      (match "\\r\\n\\r\\n")
                                                      (replace "${line.separator}")))
                                                  (echo
                                                   (@ (message ,msg)
                                                      (file "${manifest.file}")
                                                      (append "true"))))
                                           `(,tag ,@kids)))))
                     (*default* . ,(lambda (tag . kids) `(,tag ,@kids)))
                     (*text*    . ,(lambda (_ txt) txt))))
                  port)))
             (rename-file "build.xml.new" "build.xml")
             #t)))))
    (propagated-inputs
     (list java-htsjdk-2.10.1))
    (native-inputs
     (list java-testng java-guava))
    (home-page "https://broadinstitute.github.io/picard/")
    (synopsis "Tools for manipulating high-throughput sequencing data and formats")
    (description "Picard is a set of Java command line tools for manipulating
high-throughput sequencing (HTS) data and formats.  Picard is implemented
using the HTSJDK Java library to support accessing file formats that are
commonly used for high-throughput sequencing data such as SAM, BAM, CRAM and
VCF.")
    (license license:expat)))

;; This is the last version of Picard to provide net.sf.samtools
(define-public java-picard-1.113
  (package (inherit java-picard)
    (name "java-picard")
    (version "1.113")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/broadinstitute/picard")
                    (commit version)))
              (file-name (string-append "java-picard-" version "-checkout"))
              (sha256
               (base32
                "0lkpvin2fz3hhly4l02kk56fqy8lmlgyzr9kmvljk6ry6l1hw973"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  ;; Delete pre-built binaries.
                  (delete-file-recursively "lib")
                  (mkdir-p "lib")))))
    (build-system ant-build-system)
    (arguments
     `(#:build-target "picard-jar"
       #:test-target "test"
       ;; FIXME: the class path at test time is wrong.
       ;; [testng] Error: A JNI error has occurred, please check your installation and try again
       ;; [testng] Exception in thread "main" java.lang.NoClassDefFoundError: com/beust/jcommander/ParameterException
       #:tests? #f
       #:jdk ,icedtea-8
       #:ant ,ant/java8
       ;; This is only used for tests.
       #:make-flags
       (list "-Dsamjdk.intel_deflater_so_path=lib/jni/libIntelDeflater.so")
       #:phases
       (modify-phases %standard-phases
         ;; FIXME: This phase fails.
         (delete 'generate-jar-indices)
         ;; Do not use bundled ant bzip2.
         (add-after 'unpack 'use-ant-bzip
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "build.xml"
               (("\\$\\{lib\\}/apache-ant-1.8.2-bzip2.jar")
                (search-input-file inputs "/lib/ant.jar")))))
         (add-after 'unpack 'make-test-target-independent
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "build.xml"
               (("name=\"test\" depends=\"compile, ")
                "name=\"test\" depends=\"compile-tests, ")
               (("name=\"compile\" depends=\"compile-src, compile-tests\"")
                "name=\"compile\" depends=\"compile-src\""))))
         (add-after 'unpack 'fix-deflater-path
           (lambda* (#:key outputs #:allow-other-keys)
             (substitute* "src/java/net/sf/samtools/Defaults.java"
               (("getStringProperty\\(\"intel_deflater_so_path\", null\\)")
                (string-append "getStringProperty(\"intel_deflater_so_path\", \""
                               (assoc-ref outputs "out")
                               "/lib/jni/libIntelDeflater.so"
                               "\")")))))
         ;; Build the deflater library, because we've previously deleted the
         ;; pre-built one.  This can only be built with access to the JDK
         ;; sources.
         (add-after 'build 'build-jni
           (lambda* (#:key inputs #:allow-other-keys)
             (mkdir-p "lib/jni")
             (mkdir-p "jdk-src")
             (invoke "tar" "--strip-components=1" "-C" "jdk-src"
                     "-xf" (assoc-ref inputs "jdk-src"))
             (invoke "javah" "-jni"
                     "-classpath" "classes"
                     "-d" "lib/"
                     "net.sf.samtools.util.zip.IntelDeflater")
             (with-directory-excursion "src/c/inteldeflater"
               (invoke "gcc" "-I../../../lib" "-I."
                       (string-append "-I" (assoc-ref inputs "jdk")
                                      "/include/linux")
                       "-I../../../jdk-src/src/share/native/common/"
                       "-I../../../jdk-src/src/solaris/native/common/"
                       "-c" "-O3" "-fPIC" "IntelDeflater.c")
               (invoke "gcc" "-shared"
                       "-o" "../../../lib/jni/libIntelDeflater.so"
                       "IntelDeflater.o" "-lz" "-lstdc++"))))
         ;; We can only build everything else after building the JNI library.
         (add-after 'build-jni 'build-rest
           (lambda* (#:key make-flags #:allow-other-keys)
             (apply invoke `("ant" "all" ,@make-flags))))
         (add-before 'build 'set-JAVA6_HOME
           (lambda _
             (setenv "JAVA6_HOME" (getenv "JAVA_HOME"))))
         (replace 'install (install-jars "dist"))
         (add-after 'install 'install-jni-lib
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((jni (string-append (assoc-ref outputs "out")
                                       "/lib/jni")))
               (mkdir-p jni)
               (install-file "lib/jni/libIntelDeflater.so" jni)))))))
    (inputs
     (list java-snappy-1
           java-commons-jexl-2
           java-cofoja
           ant/java8                    ;for bzip2 support at runtime
           zlib))
    (native-inputs
     `(("ant-apache-bcel" ,ant-apache-bcel)
       ("ant-junit" ,ant-junit)
       ("java-testng" ,java-testng)
       ("java-commons-bcel" ,java-commons-bcel)
       ("java-jcommander" ,java-jcommander)
       ("jdk" ,icedtea-8 "jdk")
       ("jdk-src" ,(car (assoc-ref (package-native-inputs icedtea-8) "jdk-drop")))))))

(define-public fastqc
  (package
    (name "fastqc")
    (version "0.11.9")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/s-andrews/FastQC")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "00y9drm0bkpxw8xfl8ysss18jmnhj8blgqgr6fpa58rkpfcbg8qk"))
       (snippet
        '(for-each delete-file
                   '("cisd-jhdf5.jar"
                     "sam-1.103.jar"
                     "jbzip2-0.9.jar")))))
    (build-system ant-build-system)
    (arguments
     (list
      #:tests? #f                       ;there are no tests
      #:build-target "build"
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-dependencies
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* '("build.xml" ".classpath")
                (("jbzip2-0.9.jar")
                 (search-input-file inputs "/share/java/jbzip2.jar"))
                (("sam-1.103.jar")
                 (search-input-file inputs "/share/java/sam-1.112.jar"))
                (("cisd-jhdf5.jar")
                 (search-input-file inputs "/share/java/sis-jhdf5.jar")))))
          ;; There is no installation target
          (replace 'install
            (lambda* (#:key inputs #:allow-other-keys)
              (let* ((bin   (string-append #$output "/bin"))
                     (share (string-append #$output "/share/fastqc/"))
                     (exe   (string-append share "/fastqc")))
                (for-each mkdir-p (list bin share))
                (copy-recursively "bin" share)
                (substitute* exe
                  (("my \\$java_bin = \"java\";")
                   (string-append "my $java_bin = \""
                                  ;; Use java from the JRE, not the JDK
                                  #$(this-package-input "icedtea") "/bin/java"
                                  "\";"))
                  (("\\$RealBin\\$delimiter\\$RealBin.*")
                   (string-append
                    (string-join
                     (list
                      share
                      (search-input-file inputs "/share/java/sam-1.112.jar")
                      (search-input-file inputs "/share/java/jbzip2.jar")
                      (search-input-file inputs "/share/java/sis-jhdf5.jar"))
                     "$delimiter")
                    "\";\n")))
                (chmod exe #o555)
                (symlink exe (string-append bin "/fastqc"))))))))
    (inputs
     (list icedtea
           java-cisd-jhdf5
           java-picard-1.113
           java-jbzip2
           perl))                       ;needed for the wrapper script
    (native-inputs
     (list unzip))
    (home-page "https://www.bioinformatics.babraham.ac.uk/projects/fastqc/")
    (synopsis "Quality control tool for high throughput sequence data")
    (description
     "FastQC aims to provide a simple way to do some quality control
checks on raw sequence data coming from high throughput sequencing
pipelines.  It provides a modular set of analyses which you can use to
give a quick impression of whether your data has any problems of which
you should be aware before doing any further analysis.

The main functions of FastQC are:

@itemize
@item Import of data from BAM, SAM or FastQ files (any variant);
@item Providing a quick overview to tell you in which areas there may
  be problems;
@item Summary graphs and tables to quickly assess your data;
@item Export of results to an HTML based permanent report;
@item Offline operation to allow automated generation of reports
  without running the interactive application.
@end itemize\n")
    (license license:gpl3+)))

(define-public fastp
  (package
    (name "fastp")
    (version "0.23.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/OpenGene/fastp")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "04nmrqpjc3qni0cbazlwvpg8rk1mkfmfma0n4q3zivs3zi6rfnav"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #false                   ;there are none
      #:make-flags
      #~(list (string-append "PREFIX=" #$output))
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (add-before 'install 'create-target-dir
            (lambda _
              (mkdir-p (string-append #$output "/bin")))))))
    (inputs
     (list isa-l libdeflate))
    (home-page "https://github.com/OpenGene/fastp/")
    (synopsis "All-in-one FastQ preprocessor")
    (description
     "Fastp is a tool designed to provide fast all-in-one preprocessing for
FastQ files.  This tool has multi-threading support to afford high
performance.")
    (license license:expat)))

(define-public htscodecs
  (package
    (name "htscodecs")
    (version "1.6.1")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/samtools/htscodecs/"
                                  "releases/download/v"
                                  version "/htscodecs-" version ".tar.gz"))
              (sha256
               (base32
                "0n962938clim15d5mmggsm7ki2ycyjwqvjgddihjlpdsr5qv28aq"))))
    (build-system gnu-build-system)
    (inputs (list bzip2 zlib))
    (home-page "https://github.com/samtools/htscodecs")
    (synopsis "Custom compression for CRAM and others")
    (description
     "This package implements the custom CRAM codecs used for \"EXTERNAL\"
block types.  These consist of two variants of the rANS codec (8-bit and
16-bit renormalisation, with run-length encoding and bit-packing also
supported in the latter), a dynamic arithmetic coder, and custom codecs for
name/ID compression and quality score compression derived from fqzcomp.")
    (license
     (list license:public-domain ;c_range_coder.h, rANS_byte.h, and rANS_word.h
           license:bsd-3))))     ;all the rest

(define-public htslib
  (package
    (name "htslib")
    (version "1.21")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://github.com/samtools/htslib/releases/download/"
                    version "/htslib-" version ".tar.bz2"))
              (sha256
               (base32
                "08qq1yn6lqqnww532s11nr6gz0gfpn58rn3gy90kd5pl6pki1dc4"))
              (snippet
               #~(begin
                   (use-modules (guix build utils))
                   (delete-file-recursively "htscodecs")))))
    (build-system gnu-build-system)
    ;; Let htslib translate "gs://" and "s3://" to regular https links with
    ;; "--enable-gcs" and "--enable-s3". For these options to work, we also
    ;; need to set "--enable-libcurl".
    (arguments
     `(#:configure-flags '("--enable-gcs"
                           "--enable-libcurl"
                           "--enable-s3"
                           "--with-external-htscodecs")))
    (inputs
     (list bzip2 curl openssl xz))
    ;; This is referred to in the pkg-config file as a required library.
    (propagated-inputs
     (list htscodecs zlib))
    (native-inputs
     (list perl))
    (home-page "https://www.htslib.org")
    (synopsis "C library for reading/writing high-throughput sequencing data")
    (description
     "HTSlib is a C library for reading/writing high-throughput sequencing
data.  It also provides the @command{bgzip}, @command{htsfile}, and
@command{tabix} utilities.")
    ;; Files under cram/ are released under the modified BSD license;
    ;; the rest is released under the Expat license
    (license (list license:expat license:bsd-3))))

;; This package should be removed once no packages rely upon it.
(define htslib-1.3
  (package/inherit htslib
    (version "1.3.1")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://github.com/samtools/htslib/releases/download/"
                    version "/htslib-" version ".tar.bz2"))
              (sha256
               (base32
                "1rja282fwdc25ql6izkhdyh8ppw8x2fs0w0js78zgkmqjlikmma9"))))))

(define-public idr
  (package
    (name "idr")
    (version "2.0.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/nboley/idr")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04j876h6z444v2q79drxx283d3k5snd72kj895wbalnl42206x9g"))
              ;; Delete generated C code.
              (snippet
               '(begin (delete-file "idr/inv_cdf.c") #t))))
    (build-system pyproject-build-system)
    ;; There is only one test ("test_inv_cdf.py") and it tests features that
    ;; are no longer part of this package.  It also asserts False, which
    ;; causes the tests to always fail.
    (arguments `(#:tests? #f))
    (propagated-inputs
     (list python-scipy python-sympy python-numpy python-matplotlib))
    (native-inputs
     (list python-cython
           python-setuptools))
    (home-page "https://github.com/nboley/idr")
    (synopsis "Tool to measure the irreproducible discovery rate (IDR)")
    (description
     "The IDR (Irreproducible Discovery Rate) framework is a unified approach
to measure the reproducibility of findings identified from replicate
experiments and provide highly stable thresholds based on reproducibility.")
    (license license:gpl2+)))

(define-public isolator
  (let ((commit "24bafc0a102dce213bfc2b5b9744136ceadaba03")
        (revision "1"))
    (package
      (name "isolator")
      (version (git-version "0.0.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/dcjones/isolator.git")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "12mbcfqhiggcjvzizf2ff7b05z31i47njcyzcivpw5j74pfbr3dv"))))
      (build-system cmake-build-system)
      (arguments
       (list
        #:tests? #f                     ;no check target
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'fix-std
             (lambda _
               (substitute* '("src/summarize.cpp"
                              "src/shredder.cpp")
                 (("isnan") "std::isnan")
                 (("isinf") "std::isinf")))))))
      (inputs
       (list boost-1.83 hdf5 zlib))
      (home-page "https://github.com/dcjones/isolator")
      (synopsis "Tools for the analysis of RNA-Seq experiments")
      (description "Isolator analyzes RNA-Seq experiments.  Isolator has a
particular focus on producing stable, consistent estimates.  It implements a
full hierarchical Bayesian model of an entire RNA-Seq experiment.  It saves
all the samples generated by the sampler, which can be processed to compute
posterior probabilities for arbitrarily complex questions, far beyond the
confines of pairwise tests.  It aggressively corrects for technical effects,
such as random priming bias, GC-bias, 3' bias, and fragmentation effects.
Compared to other MCMC approaches, it is exceedingly efficient, though
generally slower than modern maximum likelihood approaches.")
      (license license:expat))))

(define-public jellyfish
  (package
    (name "jellyfish")
    (version "2.3.0")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/gmarcais/Jellyfish/"
                                  "releases/download/v" version
                                  "/jellyfish-" version ".tar.gz"))
              (sha256
               (base32
                "0npa62wzasdibas5zp3n8j3armsci4kyvh0jw7jr0am4gg7vg5g1"))))
    (build-system gnu-build-system)
    (outputs '("out"      ;for library
               "python")) ;for Python bindings
    (arguments
     (list
      #:configure-flags
      #~(list "--without-sse" ; configure script probes for CPU features when SSE is enabled.
              (string-append "--enable-python-binding=" #$output:python))
      #:phases
      '(modify-phases %standard-phases
         (add-before 'check 'set-SHELL-variable
           (lambda _
             ;; generator_manager.hpp either uses /bin/sh or $SHELL
             ;; to run tests.
             (setenv "SHELL" (which "bash")))))))
    (native-inputs
     (list bc
           time
           python-wrapper
           pkg-config))
    (inputs
     (list htslib))
    (synopsis "Tool for fast counting of k-mers in DNA")
    (description
     "Jellyfish is a tool for fast, memory-efficient counting of k-mers in
DNA.  A k-mer is a substring of length k, and counting the occurrences of all
such substrings is a central step in many analyses of DNA sequence.  Jellyfish
is a command-line program that reads FASTA and multi-FASTA files containing
DNA sequences.  It outputs its k-mer counts in a binary format, which can be
translated into a human-readable text format using the @code{jellyfish dump}
command, or queried for specific k-mers with @code{jellyfish query}.")
    (home-page "http://www.genome.umd.edu/jellyfish.html")
    ;; JELLYFISH seems to be 64-bit only.
    (supported-systems %64bit-supported-systems)
    ;; One of these licenses may be picked
    (license (list license:gpl3+ license:bsd-3))))

(define-public khmer
  (package
    (name "khmer")
    (version "3.0.0a3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/dib-lab/khmer")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "01l4jczglkl7yfhgvzx8j0df7k54bk1r8sli9ll16i1mis0d8f37"))
       (modules '((guix build utils)))
       (snippet
        '(begin
           ;; Delete bundled libraries.  We do not replace the bundled seqan
           ;; as it is a modified subset of the old version 1.4.1.
           ;;
           ;; We do not replace the bundled MurmurHash as the canonical
           ;; repository for this code 'SMHasher' is unsuitable for providing
           ;; a library.  See
           ;; https://lists.gnu.org/archive/html/guix-devel/2016-06/msg00977.html
           (delete-file-recursively "third-party/zlib")
           (delete-file-recursively "third-party/bzip2")
           (delete-file-recursively "third-party/seqan")
           (substitute* "setup.cfg"
             (("# libraries = z,bz2")
              "libraries = z,bz2")
             (("include:third-party/zlib:third-party/bzip2")
              "include:"))
           ;; Delete generated Cython CPP files.
           (for-each delete-file (find-files "khmer/_oxli/" "\\.cpp$"))))))
    (build-system python-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'set-cc
           (lambda _ (setenv "CC" "gcc")))
         (add-after 'unpack 'python-3.8-compatibility
           (lambda _
             ;; Python 3.8 removed time.clock().
             (substitute* "sandbox/sweep-reads.py"
               (("time\\.clock")
                "time.process_time"))))
         (add-after 'unpack 'do-use-cython
           (lambda _
             (substitute* "setup.py"
               (("from setuptools import Extension as CyExtension")
                "from Cython.Distutils import Extension as CyExtension")
               (("from setuptools.command.build_ext import build_ext as _build_ext")
                "from Cython.Distutils import build_ext as _build_ext")
               (("HAS_CYTHON = False") "HAS_CYTHON = True")
               (("cy_ext = 'cpp'") "cy_ext = 'pyx'"))))
         (add-before 'build 'build-extensions
           (lambda _
             ;; Cython extensions have to be built before running the tests.
             (invoke "python" "setup.py" "build_ext" "--inplace")))
         (replace 'check
           (lambda* (#:key tests? inputs outputs #:allow-other-keys)
             (when tests?
               (add-installed-pythonpath inputs outputs)
               (invoke "pytest" "-v")))))))
    (native-inputs
     (list python-cython python-pytest python-pytest-runner))
    (inputs
     (list zlib bzip2 seqan-1 python-screed python-bz2file))
    (home-page "https://khmer.readthedocs.org/")
    (synopsis "K-mer counting, filtering and graph traversal library")
    (description "The khmer software is a set of command-line tools for
working with DNA shotgun sequencing data from genomes, transcriptomes,
metagenomes and single cells.  Khmer can make de novo assemblies faster, and
sometimes better.  Khmer can also identify and fix problems with shotgun
data.")
    ;; When building on i686, armhf and mips64el, we get the following error:
    ;; error: ['khmer', 'khmer.tests', 'oxli'] require 64-bit operating system
    (supported-systems '("x86_64-linux" "aarch64-linux"))
    (license license:bsd-3)))

(define-public kaiju
  (package
    (name "kaiju")
    (version "1.9.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/bioinformatics-centre/kaiju")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1hfmadkfs6jjd7l3byly5xxb0ifm3dm1wis11sjbqfcv6l89snmg"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ; There are no tests.
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (add-before 'build 'move-to-src-dir
            (lambda _ (chdir "src")))
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin")))
                (mkdir-p bin)
                (copy-recursively "../bin" bin)
                (let ((path (search-path-as-list '("bin")
                                                 '#$(match (package-inputs this-package)
                                                      (((_ pkg) ...) pkg)))))
                  (for-each (lambda (script)
                              (let ((exe (string-append bin "/" script)))
                                (chmod exe #o555)
                                (wrap-script exe
                                  #:guile #$(file-append guile-3.0 "/bin/guile")
                                  `("PATH" ":" prefix ,path))))
                            (list "kaiju-convertMAR.py"
                                  "kaiju-gbk2faa.pl"
                                  "kaiju-makedb")))))))))
    (inputs
     (list bzip2
           coreutils
           curl
           gawk
           guile-3.0 ;for wrap-script
           gzip
           perl
           python-wrapper
           tar
           wget
           zlib))
    (home-page "https://kaiju.binf.ku.dk/")
    (synopsis "Fast and sensitive taxonomic classification for metagenomics")
    (description "Kaiju is a program for sensitive taxonomic classification
of high-throughput sequencing reads from metagenomic whole genome sequencing
experiments.")
    (license license:gpl3+)))

(define-public macs
  (package
    (name "macs")
    (version "2.2.9.1")
    (source (origin
              ;; The PyPi tarball does not contain tests.
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/macs3-project/MACS")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10vwc09fq4nvbd39hax1949mvq2wvkgz0k3p2zqmqwq8hv9r5l0j"))
        (modules '((guix build utils)))
        ;; Remove files generated by Cython
        (snippet
         '(begin
            (for-each (lambda (file)
                        (let ((generated-file
                                (string-append (string-drop-right file 3) "c")))
                          (when (file-exists? generated-file)
                            (delete-file generated-file))))
                      (find-files "." "\\.pyx$"))
            (delete-file "MACS2/IO/CallPeakUnitPrecompiled.c")
            ;; Python 3.10 is in fact more recent than 3.6.
            (substitute* "setup.py"
              (("float\\(sys.version\\[:3\\]\\)<3.6") "False"))))))
    (build-system pyproject-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-before 'build 'set-HOME
           (lambda _ (setenv "HOME" "/tmp"))))))
    ;; Propagate these for use of macs as a library.
    (propagated-inputs
     (list python-cython-0 python-numpy))
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/macs3-project/MACS")
    (synopsis "Model based analysis for ChIP-Seq data")
    (description
     "MACS is an implementation of a ChIP-Seq analysis algorithm for
identifying transcript factor binding sites named Model-based Analysis of
ChIP-Seq (MACS).  MACS captures the influence of genome complexity to evaluate
the significance of enriched ChIP regions and it improves the spatial
resolution of binding sites through combining the information of both
sequencing tag position and orientation.")
    (license license:bsd-3)))

(define-public macs-3
  (package
    (name "macs")
    (version "3.0.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/macs3-project/MACS")
                    (commit (string-append "v" version))
                    (recursive? #true)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0x5iz6iq694z3m9zx7zdw0js2l2l40lf1as9k3jy0q4mvz02a3aw"))))
    (properties
     '((updater-extra-inputs . ("zlib"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         ;; FIXME: our version of numpy is a little too old.
         (add-after 'unpack 'relax-requirements
           (lambda _
             (substitute* '("pyproject.toml" "requirements.txt" "setup.py")
               (("numpy.*=1.25") "numpy>=1.23")))))))
    (propagated-inputs
     (list python-cykhash
           python-hmmlearn
           python-numpy
           python-scikit-learn
           python-scipy))
    (native-inputs
     (list python-cython
           python-pytest
           python-setuptools
           python-wheel
           zlib))
    (home-page "https://github.com/macs3-project/MACS")
    (synopsis "Model based analysis for ChIP-Seq data")
    (description
     "MACS is an implementation of a ChIP-Seq analysis algorithm for
identifying transcript factor binding sites named Model-based Analysis of
ChIP-Seq (MACS).  MACS captures the influence of genome complexity to evaluate
the significance of enriched ChIP regions and it improves the spatial
resolution of binding sites through combining the information of both
sequencing tag position and orientation.")
    (license license:bsd-3)))

(define-public maffilter
  (package
    (name "maffilter")
    (version "1.3.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/jydu/maffilter")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "196c16qw82niqqyzi7j1ga1n0zmir73bm26kg04m0i5aq2cpa0ml"))))
    (build-system cmake-build-system)
    (arguments (list #:tests? #false)) ;there are none
    (inputs
     (list boost
           bpp-core
           bpp-phyl
           bpp-phyl-omics
           bpp-seq
           bpp-seq-omics
           zlib))
    (home-page "https://jydu.github.io/maffilter/")
    (synopsis "Multiple alignment format file processor")
    (description
     "MafFilter is a program dedicated to the analysis of genome alignments.
It parses and manipulates @acronym{MAF, multiple alignment format} files as
well as more simple fasta files.  This package can be used to design a
pipeline as a series of consecutive filters, each performing a dedicated
analysis.  Many of the filters are available, from alignment cleaning to
phylogeny reconstruction and population genetics analysis.  Despite various
filtering options and format conversion tools, MafFilter can compute a wide
range of statistics (phylogenetic trees, nucleotide diversity, inference of
selection, etc.).")
    (license license:gpl3+)))

(define-public mafft
  (package
    (name "mafft")
    (version "7.475")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://mafft.cbrc.jp/alignment/software/mafft-" version
                    "-without-extensions-src.tgz"))
              (file-name (string-append name "-" version ".tgz"))
              (sha256
               (base32
                "0i2i2m3blh2xkbkdk48hxfssks30ny0v381gdl7zwhcvp0axs26r"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; no automated tests, though there are tests in the read me
       #:make-flags ,#~(list (string-append "PREFIX=" #$output)
                             (string-append "BINDIR="
                                            (string-append #$output "/bin")))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'enter-dir
           (lambda _ (chdir "core") #t))
         (add-after 'enter-dir 'patch-makefile
           (lambda _
             ;; on advice from the MAFFT authors, there is no need to
             ;; distribute mafft-profile, mafft-distance, or
             ;; mafft-homologs.rb as they are too "specialised".
             (substitute* "Makefile"
               ;; remove mafft-homologs.rb from SCRIPTS
               (("^SCRIPTS = mafft mafft-homologs.rb")
                "SCRIPTS = mafft")
               ;; remove mafft-homologs from MANPAGES
               (("^MANPAGES = mafft.1 mafft-homologs.1")
                "MANPAGES = mafft.1")
               ;; remove mafft-distance from PROGS
               (("^PROGS = dvtditr dndfast7 dndblast sextet5 mafft-distance")
                "PROGS = dvtditr dndfast7 dndblast sextet5")
               ;; remove mafft-profile from PROGS
               (("splittbfast disttbfast tbfast mafft-profile 2cl mccaskillwrap")
                "splittbfast disttbfast tbfast f2cl mccaskillwrap")
               (("^rm -f mafft-profile mafft-profile.exe") "#")
               (("^rm -f mafft-distance mafft-distance.exe") ")#")
               ;; do not install MAN pages in libexec folder
               (("^\t\\$\\(INSTALL\\) -m 644 \\$\\(MANPAGES\\) \
\\$\\(DESTDIR\\)\\$\\(LIBDIR\\)") "#"))))
         (add-after 'enter-dir 'patch-paths
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* '("pairash.c"
                            "mafft.tmpl")
               (("perl") (which "perl"))
               (("([\"`| ])awk" _ prefix)
                (string-append prefix (which "awk")))
               (("grep") (which "grep")))))
         (delete 'configure)
         (add-after 'install 'wrap-programs
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (path (string-append
                           (assoc-ref inputs "coreutils") "/bin:")))
               (for-each (lambda (file)
                           (wrap-program file
                             `("PATH" ":" prefix (,path))))
                         (find-files bin))))))))
    (inputs
     (list bash-minimal perl ruby gawk grep coreutils))
    (home-page "https://mafft.cbrc.jp/alignment/software/")
    (synopsis "Multiple sequence alignment program")
    (description
     "MAFFT offers a range of multiple alignment methods for nucleotide and
protein sequences.  For instance, it offers L-INS-i (accurate; for alignment
of <~200 sequences) and FFT-NS-2 (fast; for alignment of <~30,000
sequences).")
    (license (license:non-copyleft
              "https://mafft.cbrc.jp/alignment/software/license.txt"
              "BSD-3 with different formatting"))))

(define-public mageck
  (package
    (name "mageck")
    (version "0.5.9.5")
    (source (origin
              (method url-fetch)
              (uri (string-append "mirror://sourceforge/mageck/"
                                  (version-major+minor version)
                                  "/mageck-" version ".tar.gz"))
              (sha256
               (base32
                "0fnry7d3ngiw0jarvmwd3pxy5vvsf931m4aifz6mjfd6dl1ihsmh"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:modules '((guix build pyproject-build-system)
                  (guix build utils)
                  (srfi srfi-1)
                  (ice-9 match))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'use-python3
            (lambda _
              (substitute* "bin/mageck"
                (("python2") "python"))))
         (add-after 'unpack 'numpy-compatibility
           (lambda _
              (substitute* "mageck/cnv_normalization.py"
                (("np.float") "float"))))
          ;; The build system declares executables both in "scripts" and
          ;; "data", which all are supposed to end up in #$output/bin.  The
          ;; pyproject-build-system gets confused by this, though, so we have
          ;; to manually install the mageck executable after the 'install
          ;; phase.
          (add-after 'unpack 'patch-build-system
            (lambda _
              (substitute* "setup.py"
                (("scripts=\\['bin/mageck'\\],") ""))))
          (add-after 'install 'install-mageck
            (lambda _
              (install-file "bin/mageck" (string-append #$output "/bin"))))
          (add-after 'install-mageck 'make-excutable
            (lambda _
              (for-each (lambda (file) (chmod file #o555))
                        (find-files (string-append #$output "/bin")))))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (let ((tests '(("demo1" "run.sh")
                               ("demo2" "runmageck.sh")
                               ("demo3" "run.sh")
                               ("demo4" "run.sh"))))
                  (setenv "PATH"
                          (string-append #$output "/bin:"
                                         (getenv "PATH")))
                  (for-each (match-lambda
                              ((dir script)
                               (with-directory-excursion (string-append "demo/" dir)
                                 (invoke "bash" script))))
                            tests))))))))
    (native-inputs
     (list python-wheel))
    (inputs
     (list python-numpy
           python-scipy
           python-matplotlib
           python-statsmodels
           python-pyqt
           r-minimal
           r-xtable
           r-gplots))
    (home-page "https://sourceforge.net/projects/mageck/")
    (synopsis "Model-based analysis of genome-wide CRISPR-Cas9 Knockout")
    (description
     "Model-based Analysis of Genome-wide CRISPR-Cas9
Knockout (MAGeCK) is a computational tool to identify important genes
from the recent genome-scale CRISPR-Cas9 knockout screens
technology.  Its features include:

@enumerate
@item Simple, easy to use pipeline to screen genes in Genome-wide
   CRISPR-Cas9 Knockout experiments;
@item High sensitivity and low false discovery rate;
@item Fully utilize the screening data by performing both positive and
   negative screening in one dataset;
@item Provide statistical evaluation in genes, sgRNAs and pathways;
@item Require as few as 2 samples;
@item Identify cell-type specific targets;
@item A set of visualization features that generate publication
   standard figures.
@end enumerate\n")
    (license license:bsd-3)))

(define-public mash
  (package
    (name "mash")
    (version "2.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/marbl/mash")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00x4pvxwp3isf0qign1qmxwxc9rwzn5b3igjw9hyn3vx17bsx92q"))
              (modules '((guix build utils)))
              (snippet
               ;; Delete bundled kseq.
               ;; TODO: Also delete bundled murmurhash and open bloom filter.
               '(delete-file "src/mash/kseq.h"))
              (patches (search-patches "mash-add-missing-headers.patch"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; No tests.
       #:configure-flags
       ,#~(list
           (string-append "--with-capnp=" #$(this-package-input "capnproto"))
           (string-append "--with-gsl=" #$(this-package-input "gsl")))
       #:make-flags (list "CC=gcc")
       #:phases
       (modify-phases %standard-phases
         (replace 'bootstrap
           ;; The bootstrap script is missing a shebang.
           (lambda _
             (invoke "bash" "./bootstrap.sh")))
         (add-after 'unpack 'fix-includes
           (lambda _
             (substitute* '("src/mash/Sketch.cpp"
                            "src/mash/CommandFind.cpp"
                            "src/mash/CommandScreen.cpp"
                            "src/mash/CommandTaxScreen.cpp")
               (("^#include \"kseq\\.h\"")
                "#include \"htslib/kseq.h\""))))
         (add-after 'fix-includes 'use-c++14
           (lambda _
             ;; capnproto 1.0 requires c++14 to build.
             (substitute* (list "configure.ac" "Makefile.in")
               (("c\\+\\+11") "c++14")))))))
    (native-inputs
     (list autoconf))
    (inputs
     (list ;; Capnproto and htslib are statically embedded in the final
           ;; application. Therefore we also list their licenses, below.
           capnproto
           htslib
           gsl
           zlib))
    (supported-systems '("x86_64-linux"))
    (home-page "https://mash.readthedocs.io")
    (synopsis "Fast genome and metagenome distance estimation using MinHash")
    (description "Mash is a fast sequence distance estimator that uses the
MinHash algorithm and is designed to work with genomes and metagenomes in the
form of assemblies or reads.")
    (license (list license:bsd-3          ; Mash
                   license:expat          ; HTSlib and capnproto
                   license:public-domain  ; MurmurHash 3
                   license:cpl1.0))))     ; Open Bloom Filter

(define-public metabat
  (package
    (name "metabat")
    (version "2.15")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://bitbucket.org/berkeleylab/metabat.git")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0v3gsps0ypani14102z2y1a2wignhpf7s1h45mxmj5f783rkhqd9"))))
    (build-system cmake-build-system)
    (arguments
     `(#:configure-flags
       ,#~(list (string-append "-Dzlib_LIB=" #$(this-package-input "zlib")
                               "/lib/libz.so")
                (string-append "-Dhtslib_LIB=" #$(this-package-input "htslib")
                               "/lib/libhts.so")
                (string-append "-DBOOST_ROOT=" #$(this-package-input "boost")))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'configure-version-file
           (lambda _
             (copy-file "metabat_version.h.in" "metabat_version.h")
             (substitute* "metabat_version.h"
               (("@_time_stamp@") "19700101")
               (("@GIT_IS_DIRTY@") "0")
               (("@GIT_RETRIEVED_STATE@") "0")
               (("@GIT_HEAD_SHA1@") (string-append "v" ,version)))))
         (add-after 'unpack 'do-not-use-bundled-libraries
           (lambda _
             (substitute* "CMakeLists.txt"
               (("include\\(cmake.*") ""))
             (substitute* "src/CMakeLists.txt"
               (("set\\(Boost.*") "")
               (("add_dependencies.*") "")))))))
    (inputs
     (list zlib perl samtools htslib boost-1.83))
    (home-page "https://bitbucket.org/berkeleylab/metabat")
    (synopsis
     "Reconstruction of single genomes from complex microbial communities")
    (description
     "Grouping large genomic fragments assembled from shotgun metagenomic
sequences to deconvolute complex microbial communities, or metagenome binning,
enables the study of individual organisms and their interactions.  MetaBAT is
an automated metagenome binning software, which integrates empirical
probabilistic distances of genome abundance and tetranucleotide frequency.")
    ;; The source code contains inline assembly.
    (supported-systems '("x86_64-linux" "i686-linux"))
    (license (license:non-copyleft "file://license.txt"
                                   "See license.txt in the distribution."))))

(define-public metal
  (package
    (name "metal")
    (version "2011-03-25")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "http://csg.sph.umich.edu/abecasis/Metal/"
                           "download/generic-metal-" version ".tar.gz"))
       (sha256
        (base32
         "1bk00hc0xagmq0mabmbb8bykl75qd4kfyirba869h4x6hmn4a0f3"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f
      #:make-flags
      #~(list (string-append "INSTALLDIR=" #$output "/bin") "all")
      #:phases
      '(modify-phases %standard-phases
         (replace 'configure
           (lambda _
             (substitute* "Makefile"
               (("^CFLAGS=") "CFLAGS=-std=c++11 ")))))))
    (inputs (list zlib `(,zlib "static")))
    (home-page "http://csg.sph.umich.edu/abecasis/Metal/")
    (synopsis "Facilitate meta-analysis of large datasets")
    (description "METAL is a tool for meta-analysis genomewide association
scans.  METAL can combine either test statistics and standard errors or
p-values across studies (taking sample size and direction of effect into
account).  METAL analysis is a convenient alternative to a direct analysis of
merged data from multiple studies.  It is especially appropriate when data
from the individual studies cannot be analyzed together because of differences
in ethnicity, phenotype distribution, gender or constraints in sharing of
individual level data imposed.  Meta-analysis results in little or no loss of
efficiency compared to analysis of a combined dataset including data from all
individual studies.")
    (license license:bsd-3)))

(define-public minced
  (package
    (name "minced")
    (version "0.3.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/ctSkennerton/minced")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1f5h9him0gd355cnx7p6pnxpknhckd4g0v62mg8zyhfbx9as25fv"))))
    (build-system gnu-build-system)
    (arguments
     `(#:test-target "test"
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (add-before 'check 'fix-test
           (lambda _
             ;; Fix test for latest version.
             (substitute* "t/Aquifex_aeolicus_VF5.expected"
               (("minced:0.1.6") "minced:0.2.0"))
             #t))
         (replace 'install ; No install target.
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (wrapper (string-append bin "/minced")))
               ;; Minced comes with a wrapper script that tries to figure out where
               ;; it is located before running the JAR. Since these paths are known
               ;; to us, we build our own wrapper to avoid coreutils dependency.
               (install-file "minced.jar" bin)
               (with-output-to-file wrapper
                 (lambda _
                   (display
                    (string-append
                     "#!" (assoc-ref inputs "bash") "/bin/sh\n\n"
                     (assoc-ref inputs "jre") "/bin/java -jar "
                     bin "/minced.jar \"$@\"\n"))))
               (chmod wrapper #o555))
             #t)))))
    (native-inputs
     `(("jdk" ,icedtea "jdk")))
    (inputs
     `(("bash" ,bash)
       ("jre" ,icedtea "out")))
    (home-page "https://github.com/ctSkennerton/minced")
    (synopsis "Mining CRISPRs in Environmental Datasets")
    (description
     "MinCED is a program to find Clustered Regularly Interspaced Short
Palindromic Repeats (CRISPRs) in DNA sequences.  It can be used for
unassembled metagenomic reads, but is mainly designed for full genomes and
assembled metagenomic sequence.")
    (license license:gpl3+)))

(define-public muscle
  (package
    (name "muscle")
    (version "3.8.1551")
    (source (origin
              (method url-fetch/tarbomb)
              (uri (string-append
                    "http://www.drive5.com/muscle/muscle_src_"
                    version ".tar.gz"))
              (sha256
               (base32
                "0bj8kj7sdizy3987zx6w7axihk40fk8rn76mpbqqjcnd64i5a367"))))
    (build-system gnu-build-system)
    (arguments
     `(#:make-flags (list "LDLIBS = -lm")
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'check
           ;; There are no tests, so just test if it runs.
           (lambda _ (invoke "./muscle" "-version") #t))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin")))
               (install-file "muscle" bin)
               #t))))))
    (home-page "http://www.drive5.com/muscle")
    (synopsis "Multiple sequence alignment program")
    (description
     "MUSCLE aims to be a fast and accurate multiple sequence alignment
program for nucleotide and protein sequences.")
    ;; License information found in 'muscle -h' and usage.cpp.
    (license license:public-domain)))

(define-public music
  (let ((commit "b1caecdb164b1ab80acccb9463abe2526a56f69f")
        (revision "1"))
    (package
      (name "music")
      (version (git-version "0.0.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/gersteinlab/MUSIC.git")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0arj300h8cpbya7y98g066xsxcg2a65h3y0qs250rlj072f1b4ia"))))
      (build-system gnu-build-system)
      (arguments
       (list
        #:tests? #f                     ; no "check" target
        #:phases
        #~(modify-phases %standard-phases
            (delete 'configure)
            ;; There is no "install" target.
            (replace 'install
              (lambda _
                (let ((bin (string-append #$output "/bin")))
                  (install-file "bin/MUSIC" bin)))))))
      (home-page "https://github.com/gersteinlab/MUSIC/")
      (synopsis "Multiscale enrichment calling for ChIP-Seq datasets")
      (description
       "MUSIC is an algorithm for identification of enriched regions at
multiple scales in the read depth signals from ChIP-Seq experiments.")
      ;; See https://github.com/gersteinlab/MUSIC/issues/6
      (license license:gpl2+)
      (properties '((lint-hidden-cpe-vendors . ("apple")))))))

(define-public newick-utils
  ;; There are no recent releases so we package from git.
  (let ((commit "da121155a977197cab9fbb15953ca1b40b11eb87"))
    (package
      (name "newick-utils")
      (version (string-append "1.6-1." (string-take commit 8)))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/tjunier/newick_utils")
                      (commit commit)))
                (file-name (string-append name "-" version "-checkout"))
                (sha256
                 (base32
                  "1hkw21rq1mwf7xp0rmbb2gqc0i6p11108m69i7mr7xcjl268pxnb"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? ,(not (or (target-riscv64?)   ;XXX: stuck on riscv64-linux
                            (%current-target-system)))
         #:make-flags (list "CFLAGS=-O2 -g -fcommon")))
      (inputs
       ;; XXX: TODO: Enable Lua and Guile bindings.
       ;; https://github.com/tjunier/newick_utils/issues/13
       (list libxml2 flex bison))
      (native-inputs
       (list autoconf automake libtool))
      (synopsis "Programs for working with newick format phylogenetic trees")
      (description
       "Newick-utils is a suite of utilities for processing phylogenetic trees
in Newick format.  Functions include re-rooting, extracting subtrees,
trimming, pruning, condensing, drawing (ASCII graphics or SVG).")
      (home-page "https://github.com/tjunier/newick_utils")
      (license license:bsd-3))))

(define-public orfm
  (package
    (name "orfm")
    (version "0.7.1")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://github.com/wwood/OrfM/releases/download/v"
                    version "/orfm-" version ".tar.gz"))
              (sha256
               (base32
                "16iigyr2gd8x0imzkk1dr3k5xsds9bpmwg31ayvjg0f4pir9rwqr"))))
    (build-system gnu-build-system)
    (inputs (list zlib))
    (native-inputs
     (list ruby-bio-commandeer ruby-rspec ruby))
    (synopsis "Simple and not slow open reading frame (ORF) caller")
    (description
     "An ORF caller finds stretches of DNA that, when translated, are not
interrupted by stop codons.  OrfM finds and prints these ORFs.")
    (home-page "https://github.com/wwood/OrfM")
    (license license:lgpl3+)))

(define-public prank
  (package
    (name "prank")
    (version "170427")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "http://wasabiapp.org/download/prank/prank.source."
                    version ".tgz"))
              (sha256
               (base32
                "0nc8g9c5rkdxcir46s0in9ci1sxwzbjibxrvkksf22ybnplvagk2"))))
    (build-system gnu-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'enter-src-dir
           (lambda _ (chdir "src")))
         (add-after 'unpack 'remove-m64-flag
           ;; Prank will build with the correct 'bit-ness' without this flag
           ;; and this allows building on 32-bit machines.
           (lambda _
             (substitute* "src/Makefile"
               (("-m64") ""))))
         (delete 'configure)
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (man (string-append out "/share/man/man1"))
                    (path (string-append
                           (assoc-ref inputs "mafft") "/bin:"
                           (assoc-ref inputs "exonerate") "/bin:"
                           (assoc-ref inputs "bppsuite") "/bin")))
               (install-file "prank" bin)
               (wrap-program (string-append bin "/prank")
                 `("PATH" ":" prefix (,path)))
               (install-file "prank.1" man)))))))
    (inputs
     (list mafft exonerate bppsuite))
    (home-page "http://wasabiapp.org/software/prank/")
    (synopsis "Probabilistic multiple sequence alignment program")
    (description
     "PRANK is a probabilistic multiple sequence alignment program for DNA,
codon and amino-acid sequences.  It is based on a novel algorithm that treats
insertions correctly and avoids over-estimation of the number of deletion
events.  In addition, PRANK borrows ideas from maximum likelihood methods used
in phylogenetics and correctly takes into account the evolutionary distances
between sequences.  Lastly, PRANK allows for defining a potential structure
for sequences to be aligned and then, simultaneously with the alignment,
predicts the locations of structural units in the sequences.")
    (license license:gpl2+)))

(define-public proteinortho
  (package
    (name "proteinortho")
    (version "6.3.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://gitlab.com/paulklemm_PHD/proteinortho.git")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0p8iaxq193fh67hw3cydvdah1vz1c3f18227gj1mhkww0ms7g6xa"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  ;; Remove pre-built scripts and source tarballs.
                  (delete-file-recursively "src/BUILD/")
                  (delete-file "src/lapack-3.8.0.tar.gz")))))
    (build-system gnu-build-system)
    (arguments
     (list
       #:test-target "test"
       #:parallel-tests? #f
       #:make-flags
       #~(list (string-append "CC=" #$(cc-for-target))
               (string-append "CXX=" #$(cxx-for-target))
               (string-append "PREFIX=" #$output)
               (string-append "INSTALLDIR=" #$output "/bin"))
       #:phases
       #~(modify-phases %standard-phases
           (delete 'configure)
           (add-before 'install 'make-install-directory
             ;; The install directory is not created during 'make install'.
             (lambda* (#:key outputs #:allow-other-keys)
               (mkdir-p (string-append (assoc-ref outputs "out") "/bin"))))
           (add-after 'install 'wrap-programs
             (lambda* (#:key inputs outputs #:allow-other-keys)
               (let ((path (getenv "PATH"))
                     (out (assoc-ref outputs "out"))
                     (guile (search-input-file inputs "bin/guile")))
                 (for-each (lambda (script)
                             (wrap-script script #:guile guile
                                          `("PATH" ":" prefix (,path))))
                           (cons (string-append out "/bin/proteinortho")
                                 (find-files out "\\.(pl|py)$")))))))))
    (inputs
     (list bash-minimal
           guile-3.0         ; for wrap-script
           diamond
           perl
           python-wrapper
           blast+
           openblas))
    (native-inputs
     (list which))
    (home-page "https://www.bioinf.uni-leipzig.de/Software/proteinortho")
    (synopsis "Detect orthologous genes across species")
    (description
     "Proteinortho is a tool to detect orthologous genes across different
species.  For doing so, it compares similarities of given gene sequences and
clusters them to find significant groups.  The algorithm was designed to handle
large-scale data and can be applied to hundreds of species at once.")
    (properties `((tunable? . #t)))
    (license license:gpl3+)))

(define-public prodigal
  (package
    (name "prodigal")
    ;; Check for a new home page when updating this package:
    ;; https://github.com/hyattpd/Prodigal/issues/36#issuecomment-536617588
    (version "2.6.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/hyattpd/Prodigal")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fs1hqk83qjbjhrvhw6ni75zakx5ki1ayy3v6wwkn3xvahc9hi5s"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; no check target
       #:make-flags
       ,#~(list (string-append "INSTALLDIR=" #$output "/bin"))
       #:phases
       (modify-phases %standard-phases
         (delete 'configure))))
    (home-page "https://github.com/hyattpd/Prodigal")
    (synopsis "Protein-coding gene prediction for Archaea and Bacteria")
    (description
     "Prodigal runs smoothly on finished genomes, draft genomes, and
metagenomes, providing gene predictions in GFF3, Genbank, or Sequin table
format.  It runs quickly, in an unsupervised fashion, handles gaps, handles
partial genes, and identifies translation initiation sites.")
    (license license:gpl3+)))

(define-public roary
  (package
    (name "roary")
    (version "3.12.0")
    (source
     (origin
       (method url-fetch)
       (uri (string-append
             "mirror://cpan/authors/id/A/AJ/AJPAGE/Bio-Roary-"
             version ".tar.gz"))
       (sha256
        (base32
         "0qxrds9wx7cfhlkihrp6697kx0flhhxymap9fwan0b3rbdhcnmff"))))
    (build-system perl-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (delete 'build)
         (replace 'check
           (lambda _
             ;; The tests are not run by default, so we run each test file
             ;; directly.
             (setenv "PATH" (string-append (getcwd) "/bin" ":"
                                           (getenv "PATH")))
             (setenv "PERL5LIB" (string-append (getcwd) "/lib" ":"
                                               (getenv "PERL5LIB")))
             (for-each (lambda (file)
                         (display file)(display "\n")
                         (invoke "perl" file))
                       (find-files "t" ".*\\.t$"))))
         (replace 'install
           ;; There is no 'install' target in the Makefile.
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (perl (string-append out "/lib/perl5/site_perl"))
                    (roary-plots "contrib/roary_plots"))
               (mkdir-p bin)
               (mkdir-p perl)
               (copy-recursively "bin" bin)
               (copy-recursively "lib" perl))))
         (add-after 'install 'wrap-programs
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (perl5lib (getenv "PERL5LIB"))
                    (path (getenv "PATH")))
               (for-each (lambda (prog)
                           (let ((binary (string-append out "/" prog)))
                             (wrap-program binary
                               `("PERL5LIB" ":" prefix
                                 (,(string-append perl5lib ":" out
                                                  "/lib/perl5/site_perl"))))
                             (wrap-program binary
                               `("PATH" ":" prefix
                                 (,(string-append path ":" out "/bin"))))))
                         (find-files "bin" ".*[^R]$"))
               (let ((file
                      (string-append out "/bin/roary-create_pan_genome_plots.R"))
                     (r-site-lib (getenv "R_LIBS_SITE"))
                     (coreutils-path
                      (dirname (search-input-file inputs "bin/chmod"))))
                 (wrap-program file
                   `("R_LIBS_SITE" ":" prefix
                     (,(string-append r-site-lib ":" out "/site-library/"))))
                 (wrap-program file
                   `("PATH" ":" prefix
                     (,(string-append coreutils-path ":" out "/bin")))))))))))
    (native-inputs
     (list perl-env-path perl-test-files perl-test-most perl-test-output))
    (inputs
     (list bash-minimal
           perl-array-utils
           bioperl-minimal
           perl-digest-md5-file
           perl-exception-class
           perl-file-find-rule
           perl-file-grep
           perl-file-slurper
           perl-file-which
           perl-graph
           perl-graph-readwrite
           perl-log-log4perl
           perl-moose
           perl-perlio-utf8_strict
           perl-text-csv
           bedtools
           cd-hit
           blast+
           mcl
           parallel
           prank
           mafft
           fasttree
           grep
           sed
           gawk
           r-minimal
           r-ggplot2
           coreutils))
    (home-page "https://sanger-pathogens.github.io/Roary/")
    (synopsis "High speed stand-alone pan genome pipeline")
    (description
     "Roary is a high speed stand alone pan genome pipeline, which takes
annotated assemblies in GFF3 format (produced by the Prokka program) and
calculates the pan genome.  Using a standard desktop PC, it can analyse
datasets with thousands of samples, without compromising the quality of the
results.  128 samples can be analysed in under 1 hour using 1 GB of RAM and a
single processor.  Roary is not intended for metagenomics or for comparing
extremely diverse sets of genomes.")
    (license license:gpl3)))

(define-public raxml
  (package
    (name "raxml")
    (version "8.2.12")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/stamatak/standard-RAxML")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1jqjzhch0rips0vp04prvb8vmc20c5pdmsqn8knadcf91yy859fh"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; There are no tests.
       ;; Use 'standard' Makefile rather than SSE or AVX ones.
       #:make-flags (list "-f" "Makefile.HYBRID.gcc")
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (executable "raxmlHPC-HYBRID"))
               (install-file executable bin)
               (symlink (string-append bin "/" executable) "raxml"))
             #t)))))
    (inputs
     (list openmpi))
    (home-page "https://cme.h-its.org/exelixis/web/software/raxml/index.html")
    (synopsis "Randomized Axelerated Maximum Likelihood phylogenetic trees")
    (description
     "RAxML is a tool for phylogenetic analysis and post-analysis of large
phylogenies.")
    ;; The source includes x86 specific code
    (supported-systems '("x86_64-linux" "i686-linux"))
    (license license:gpl2+)))

(define-public rsem
  (package
    (name "rsem")
    (version "1.3.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/deweylab/RSEM")
             (commit (string-append "v" version))))
       (sha256
        (base32 "1yl4i7z20n2p84j1lmk15aiak3yqc6fiw0q5a4pndw7pxfiq3rzp"))
       (file-name (git-file-name name version))
       (modules '((guix build utils)))
       (snippet
        '(begin
           ;; remove bundled copy of boost and samtools
           (delete-file-recursively "boost")
           (delete-file-recursively "samtools-1.3")))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no "check" target
       #:make-flags
       ,#~(list (string-append "BOOST="
                               #$(this-package-input "boost")
                               "/include/")
                (string-append "SAMHEADERS="
                               #$(this-package-input "htslib")
                               "/include/htslib/sam.h")
                (string-append "SAMLIBS="
                               #$(this-package-input "htslib")
                               "/lib/libhts.so"))
       #:phases
       (modify-phases %standard-phases
         ;; No "configure" script.
         (replace 'configure
           (lambda _
             (substitute* "Makefile"
               (("^all : \\$\\(PROGRAMS\\).*") "all: $(PROGRAMS)\n")
               ;; Do not build bundled samtools library.
               (("^\\$\\(SAMLIBS\\).*") "")
               ;; Needed for Boost
               (("gnu\\+\\+98") "gnu++11"))
             ;; C++11 compatibility
             (substitute* "buildReadIndex.cpp"
               (("success = \\(getline")
                "success = (bool)(getline"))
             (substitute* '("PairedEndHit.h"
                            "SingleHit.h")
               (("return \\(in>>sid>>pos")
                "return (bool)(in>>sid>>pos"))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (string-append (assoc-ref outputs "out")))
                    (bin (string-append out "/bin/"))
                    (perl (string-append out "/lib/perl5/site_perl")))
               (mkdir-p bin)
               (mkdir-p perl)
               (for-each (lambda (file)
                           (install-file file bin))
                         (find-files "." "rsem-.*"))
               (install-file "rsem_perl_utils.pm" perl))))
         (add-after 'install 'wrap-program
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((out (assoc-ref outputs "out")))
               (for-each (lambda (prog)
                           (wrap-program (string-append out "/bin/" prog)
                             `("PERL5LIB" ":" prefix
                               (,(string-append out "/lib/perl5/site_perl")))))
                         '("rsem-calculate-expression"
                           "rsem-control-fdr"
                           "rsem-generate-data-matrix"
                           "rsem-generate-ngvector"
                           "rsem-plot-transcript-wiggles"
                           "rsem-prepare-reference"
                           "rsem-run-ebseq"
                           "rsem-run-prsem-testing-procedure"))))))))
    (inputs
     (list bash-minimal boost-1.83 r-minimal perl htslib-1.3 zlib))
    (home-page "https://deweylab.biostat.wisc.edu/rsem/")
    (synopsis "Estimate gene expression levels from RNA-Seq data")
    (description
     "RSEM is a software package for estimating gene and isoform expression
levels from RNA-Seq data.  The RSEM package provides a user-friendly
interface, supports threads for parallel computation of the EM algorithm,
single-end and paired-end read data, quality scores, variable-length reads and
RSPD estimation.  In addition, it provides posterior mean and 95% credibility
interval estimates for expression levels.  For visualization, it can generate
BAM and Wiggle files in both transcript-coordinate and genomic-coordinate.")
    (license license:gpl3+)))

(define-public rseqc
  (package
    (name "rseqc")
    (version "3.0.1")
    (source
     (origin
       (method url-fetch)
       (uri
        (string-append "mirror://sourceforge/rseqc/"
                       "RSeQC-" version ".tar.gz"))
       (sha256
        (base32
         "0gbb9iyb7swiv5455fm5rg98r7l6qn27v564yllqjd574hncpx6m"))))
    (build-system pyproject-build-system)
    (arguments (list #:tests? #f))      ; No tests.
    (inputs
     (list python-bx-python
           python-cython
           python-numpy
           python-pybigwig
           python-pyparsing
           python-pysam
           python-setuptools
           zlib))
    (home-page "https://rseqc.sourceforge.net/")
    (synopsis "RNA-seq quality control package")
    (description
     "RSeQC provides a number of modules that can comprehensively evaluate
high throughput sequence data, especially RNA-seq data.  Some basic modules
inspect sequence quality, nucleotide composition bias, PCR bias and GC bias,
while RNA-seq specific modules evaluate sequencing saturation, mapped reads
distribution, coverage uniformity, strand specificity, etc.")
    (license license:gpl3+)))

(define-public seek
  ;; There are no release tarballs.  And the installation instructions at
  ;; http://seek.princeton.edu/installation.jsp only mention a mercurial
  ;; changeset ID.  This is a git repository, though.  So we just take the
  ;; most recent commit.
  (let ((commit "196ed4c7633246e9c628e4330d77577ccfd7f1e5")
        (revision "1"))
    (package
      (name "seek")
      (version (git-version "1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/FunctionLab/sleipnir.git")
                      (commit commit)
                      (recursive? #true)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0c658n8nz563a96dsi4gl2685vxph0yfmmqq5yjc6i4xin1jy1ab"))))
      (build-system cmake-build-system)
      (arguments
       `(#:configure-flags
         ,#~(list (string-append "-DSVM_LIBRARY="
                                 #$(this-package-input "libsvm")
                                 "/lib/libsvm.so")
                  (string-append "-DSVM_INCLUDE="
                                 #$(this-package-input "libsvm")
                                 "/include"))
         #:tests? #false ; tests only fail in the build container
         #:phases
         (modify-phases %standard-phases
           ;; The check phase expects to find the unit_tests executable in the
           ;; "build/bin" directory, but it is actually in "build/tests".
           (replace 'check
             (lambda* (#:key tests? #:allow-other-keys)
               (when tests?
                 (invoke "tests/unit_tests")))))))
      (inputs
       `(("apache-thrift:include" ,apache-thrift "include")
         ("apache-thrift:lib" ,apache-thrift "lib")
         ("gsl" ,gsl)
         ("boost" ,boost)
         ("gengetopt" ,gengetopt)
         ("libsvm" ,libsvm)
         ("log4cpp" ,log4cpp)
         ("python" ,python)
         ("readline" ,readline)))
      (native-inputs
       (list pkg-config))
      (home-page "https://seek.princeton.edu")
      (synopsis "Gene co-expression search engine")
      (description
       "SEEK is a computational gene co-expression search engine.  SEEK provides
biologists with a way to navigate the massive human expression compendium that
now contains thousands of expression datasets.  SEEK returns a robust ranking
of co-expressed genes in the biological area of interest defined by the user's
query genes.  It also prioritizes thousands of expression datasets according
to the user's query of interest.")
      (license license:cc-by3.0))))

(define-public samtools
  (package
    (name "samtools")
    (version "1.19")
    (source
     (origin
       (method url-fetch)
       (uri
        (string-append "https://github.com/samtools/samtools"
                       "/releases/download/" version
                       "/samtools-" version ".tar.bz2"))
       (sha256
        (base32
         "10wby07w33rfypy4kf73v9wwnbyh0lrazbsmrgrvcl88w8c3nszs"))
       (modules '((guix build utils)))
       ;; Delete bundled htslib.
       (snippet '(delete-file-recursively "htslib-1.19"))))
    (build-system gnu-build-system)
    (arguments
     `(#:configure-flags (list "--with-ncurses")
       ;; The samtools test suite (and software) expects SSE-based math, even on
       ;; i686-linux, and not 387-based math.  Adjust the CPPFLAGS accordingly.
       ,@(if (target-x86-32?)
             `(#:make-flags (list "CPPFLAGS = -msse -mfpmath=sse"))
             '())
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'patch-tests
           (lambda _
             (substitute* "test/test.pl"
               ;; The test script calls out to /bin/bash
               (("/bin/bash") (which "bash"))))))))
    (native-inputs (list pkg-config))
    (inputs
     (list htslib ncurses perl python zlib))
    (home-page "https://samtools.sourceforge.net")
    (synopsis "Utilities to efficiently manipulate nucleotide sequence alignments")
    (description
     "Samtools implements various utilities for post-processing nucleotide
sequence alignments in the SAM, BAM, and CRAM formats, including indexing,
variant calling (in conjunction with bcftools), and a simple alignment
viewer.")
    (license license:expat)))

(define-public samtools-0.1
  ;; This is the most recent version of the 0.1 line of samtools.  The input
  ;; and output formats differ greatly from that used and produced by samtools
  ;; 1.x and is still used in many bioinformatics pipelines.
  (package (inherit samtools)
    (version "0.1.19")
    (source
     (origin
       (method url-fetch)
       (uri
        (string-append "mirror://sourceforge/samtools/samtools/"
                       version "/samtools-" version ".tar.bz2"))
       (sha256
        (base32 "1m33xsfwz0s8qi45lylagfllqg7fphf4dr0780rsvw75av9wk06h"))))
    (arguments
     `(#:tests? #f ;no "check" target
       #:make-flags
       (list "LIBCURSES=-lncurses")
       ,@(substitute-keyword-arguments (package-arguments samtools)
           ((#:modules _ #f)
            '((ice-9 ftw)
              (ice-9 regex)
              (guix build gnu-build-system)
              (guix build utils)))
           ((#:phases phases)
            `(modify-phases ,phases
               (replace 'install
                 (lambda* (#:key outputs #:allow-other-keys)
                   (let ((bin (string-append
                               (assoc-ref outputs "out") "/bin")))
                     (mkdir-p bin)
                     (install-file "samtools" bin)
                     #t)))
               (add-after 'install 'install-library
                 (lambda* (#:key outputs #:allow-other-keys)
                   (let ((lib (string-append (assoc-ref outputs "out") "/lib")))
                     (install-file "libbam.a" lib))))
               (add-after 'install 'install-headers
                 (lambda* (#:key outputs #:allow-other-keys)
                   (let ((include (string-append (assoc-ref outputs "out")
                                                 "/include/samtools/")))
                     (for-each (lambda (file)
                                 (install-file file include))
                               (scandir "." (lambda (name)
                                              (string-match "\\.h$" name)))))))
               (delete 'patch-tests)
               (delete 'configure))))))))

(define-public savvy
  (package
    (name "savvy")
    (version "2.1.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/statgen/savvy")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "03jh89gl67adnpkwx8yrdn62pd9sg69k21gxh15my2vvpfl1pxlx"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:configure-flags
      '(list "-DBUILD_TESTS=ON"
             "-DBUILD_EVAL=ON")
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'use-shared-libraries
           (lambda _
             ;; Do not prefer static libraries.
             (substitute* "CMakeLists.txt"
               (("set\\(CMAKE_FIND_LIBRARY_SUFFIXES \".a;.*") "")))))))
    (native-inputs (list pkg-config))
    (inputs (list htslib))
    (propagated-inputs (list streambuf-shrinkwrap))
    (home-page "https://github.com/statgen/savvy/")
    (synopsis "Interface to various variant calling formats")
    (description
     "Savvy is the official C++ interface for the SAV file format and offers
seamless support for BCF and VCF files.")
    (license license:mpl2.0)))

(define-public morpheus
  (package
    (name "morpheus")
    (version "2.3.6")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://gitlab.com/morpheus.lab/morpheus")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1jyzbkz8d39kjicrk3ihcx7yvq5wsynvnlcw922bqqsw8nwnn12c"))
       (modules '((guix build utils)))
       (snippet
        '(begin
           (delete-file-recursively "3rdparty/eigen")
           (substitute* '("morpheus/core/cpm_shape_tracker.cpp"
                          "morpheus/core/membranemapper.h"
                          "morpheus/testing/components/motility/directed_motion_test.cpp"
                          "morpheus/testing/components/interaction/generator_cell_sorting.cpp"
                          "morpheus/testing/components/interaction/test_cell_sorting.cpp"
                          "morpheus/testing/core/cpm/generator_csm_plane.cpp"
                          "morpheus/testing/test_operators.h")
             (("#include \"eigen/") "#include \"eigen3/"))))))
    ;; This is for a different Morpheus.
    (properties '((lint-hidden-cve "CVE-2022-31261")))
    (build-system cmake-build-system)
    (arguments
     (list
      #:configure-flags
      '(list "-DMORPHEUS_GUI=OFF"
             "-DDOWNLOAD_XTENSOR=OFF")
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'disable-gtest-download
           (lambda _
             (substitute* "3rdparty/CMakeLists.txt"
               (("add_subdirectory\\(GTest\\)") ""))))
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (invoke "ctest" "--output-junit" "test_results.xml")))))))
    (inputs (list boost
                  eigen
                  file
                  gnuplot
                  libtiff
                  libxslt
                  xsimd
                  xtensor
                  xtl
                  zlib))
    (native-inputs
     (list doxygen googletest xxd))
    (home-page "https://gitlab.com/morpheus.lab/morpheus")
    (synopsis "Multicellular simulation")
    (description
     "Morpheus is a modeling and simulation environment for the study of
multi-scale and multicellular systems.")
    (license license:bsd-3)))

(define-public mosaik
  (let ((commit "5c25216d3522d6a33e53875cd76a6d65001e4e67"))
    (package
      (name "mosaik")
      (version "2.2.30")
      (source (origin
                ;; There are no release tarballs nor tags.
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/wanpinglee/MOSAIK")
                      (commit commit)))
                (file-name (string-append name "-" version))
                (sha256
                 (base32
                  "17gj3s07cm77r41z92awh0bim7w7q7fbn0sf5nkqmcm1vw052qgw"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f ; no tests
         #:make-flags (list "CC=gcc")
         #:phases
         (modify-phases %standard-phases
           (replace 'configure
                    (lambda _ (chdir "src") #t))
           (replace 'install
                    (lambda* (#:key outputs #:allow-other-keys)
                      (let ((bin (string-append (assoc-ref outputs "out")
                                                "/bin")))
                        (mkdir-p bin)
                        (copy-recursively "../bin" bin)
                        #t))))))
      (inputs
       `(("perl" ,perl)
         ("zlib:static" ,zlib "static")
         ("zlib" ,zlib)))
      (supported-systems '("x86_64-linux"))
      (home-page "https://github.com/wanpinglee/MOSAIK")
      (synopsis "Map nucleotide sequence reads to reference genomes")
      (description
       "MOSAIK is a program for mapping second and third-generation sequencing
reads to a reference genome.  MOSAIK can align reads generated by all the
major sequencing technologies, including Illumina, Applied Biosystems SOLiD,
Roche 454, Ion Torrent and Pacific BioSciences SMRT.")
      ;; MOSAIK is released under the GPLv2+ with the exception of third-party
      ;; code released into the public domain:
      ;; 1. fastlz by Ariya Hidayat - http://www.fastlz.org/
      ;; 2. MD5 implementation - RSA Data Security, RFC 1321
      (license (list license:gpl2+ license:public-domain)))))

(define-public mosaicatcher
  (package
    (name "mosaicatcher")
    (version "0.3.1")
    (source (origin
              ;; There are no release tarballs nor tags.
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/friendsofstrandseq/mosaicatcher")
                    (commit (string-append version "-dev"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1n2s5wvvj2y0vfgjkg1q11xahpbagxz7h2vf5q7qyy25s12kbzbd"))
              (patches (search-patches "mosaicatcher-unbundle-htslib.patch"))))
    (build-system cmake-build-system)
    (arguments
     `(#:tests? #false ; there are no tests
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'chdir
           (lambda _ (chdir "src")))
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((target (assoc-ref outputs "out"))
                    (bin (string-append target "/bin"))
                    (share (string-append target "/share/mosaicatcher")))
               (install-file "mosaic" bin)
               (mkdir-p share)
               (copy-recursively "../R" share)))))))
    (inputs
     (list boost htslib))
    (home-page "https://github.com/friendsofstrandseq/mosaicatcher")
    (synopsis "Count and classify Strand-seq reads")
    (description
     "Mosaicatcher counts Strand-seq reads and classifies strand states of
each chromosome in each cell using a Hidden Markov Model.")
    (license license:expat)))

(define-public ngs-sdk
  (package
    (name "ngs-sdk")
    (version "2.10.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/ncbi/ngs")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ix51c25hjn57w93qmwzw80xh2i34wx8j2hn7szh8p6w8i3az5qa"))))
    (build-system gnu-build-system)
    (arguments
     `(#:parallel-build? #f ; not supported
       #:tests? #f ; no "check" target
       #:phases
       (modify-phases %standard-phases
         (replace 'configure
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((out (assoc-ref outputs "out")))
               ;; Allow 'konfigure.perl' to find 'package.prl'.
               (setenv "PERL5LIB"
                       (string-append ".:" (getenv "PERL5LIB")))

               ;; The 'configure' script doesn't recognize things like
               ;; '--enable-fast-install'.
               (invoke "./configure"
                       (string-append "--build-prefix=" (getcwd) "/build")
                       (string-append "--prefix=" out))
               #t)))
         (add-after 'unpack 'enter-dir
           (lambda _ (chdir "ngs-sdk") #t)))))
    (native-inputs (list perl))
    ;; According to the test
    ;;   unless ($MARCH =~ /x86_64/i || $MARCH =~ /i?86/i)
    ;; in ngs-sdk/setup/konfigure.perl
    (supported-systems '("i686-linux" "x86_64-linux"))
    (home-page "https://github.com/ncbi/ngs")
    (synopsis "API for accessing Next Generation Sequencing data")
    (description
     "NGS is a domain-specific API for accessing reads, alignments and pileups
produced from Next Generation Sequencing.  The API itself is independent from
any particular back-end implementation, and supports use of multiple back-ends
simultaneously.")
    (license license:public-domain)))

(define-public java-ngs
  (package (inherit ngs-sdk)
    (name "java-ngs")
    (arguments
     `(,@(substitute-keyword-arguments
             `(#:modules ((guix build gnu-build-system)
                          (guix build utils)
                          (srfi srfi-1)
                          (srfi srfi-26))
                         ,@(package-arguments ngs-sdk))
           ((#:phases phases)
            `(modify-phases ,phases
               (replace 'enter-dir (lambda _ (chdir "ngs-java") #t)))))))
    (inputs
     `(("jdk" ,icedtea "jdk")
       ("ngs-sdk" ,ngs-sdk)))
    (synopsis "Java bindings for NGS SDK")))

(define-public ncbi-vdb
  (package
    (name "ncbi-vdb")
    (version "2.10.6")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/ncbi/ncbi-vdb")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0m8hlxscidsfqm9x9fyi62q6lpf1dv5115kgjjgnrkl49q9c27m6"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:parallel-build? #f              ; not supported
      #:tests? #f                       ; no "check" target
      #:make-flags '(list "HAVE_HDF5=1")
      #:phases
      #~(modify-phases %standard-phases
          #;
          (add-after 'unpack 'make-files-writable
            (lambda _ (for-each make-file-writable (find-files "." ".*"))))
          (add-before 'configure 'set-perl-search-path
            (lambda _
              ;; Work around "dotless @INC" build failure.
              (setenv "PERL5LIB"
                      (string-append (getcwd) "/setup:"
                                     (getenv "PERL5LIB")))))
          ;; See https://github.com/ncbi/ncbi-vdb/issues/14
          (add-after 'unpack 'patch-krypto-flags
            (lambda _
              (substitute* "libs/krypto/Makefile"
                (("-Wa,-march=generic64\\+aes") "")
                (("-Wa,-march=generic64\\+sse4") ""))))
          (replace 'configure
            (lambda _
              ;; Override include path for libmagic
              (substitute* "setup/package.prl"
                (("name => 'magic', Include => '/usr/include'")
                 (string-append "name=> 'magic', Include => '"
                                #$(this-package-input "file")
                                "/include" "'")))

              ;; Install kdf5 library (needed by sra-tools)
              (substitute* "build/Makefile.install"
                (("LIBRARIES_TO_INSTALL =")
                 "LIBRARIES_TO_INSTALL = kdf5.$(VERSION_LIBX) kdf5.$(VERSION_SHLX)"))

              (substitute* "build/Makefile.env"
                (("CFLAGS	=" prefix)
                 (string-append prefix "-msse2 ")))

              ;; Override search path for ngs-java
              (substitute* "setup/package.prl"
                (("/usr/local/ngs/ngs-java")
                 #$(this-package-input "java-ngs")))

              ;; The 'configure' script doesn't recognize things like
              ;; '--enable-fast-install'.
              (invoke "./configure"
                      (string-append "--build-prefix=" (getcwd) "/build")
                      (string-append "--prefix=" #$output)
                      (string-append "--debug")
                      (string-append "--with-xml2-prefix="
                                     #$(this-package-input "libxml2"))
                      (string-append "--with-ngs-sdk-prefix="
                                     #$(this-package-input "ngs-sdk"))
                      (string-append "--with-hdf5-prefix="
                                     #$(this-package-input "hdf5")))))
          (add-after 'install 'install-interfaces
            (lambda _
              ;; Install interface libraries.  On i686 the interface libraries
              ;; are installed to "linux/gcc/i386", so we need to use the Linux
              ;; architecture name ("i386") instead of the target system prefix
              ;; ("i686").
              (mkdir (string-append #$output "/ilib"))
              (copy-recursively (string-append
                                 "build/ncbi-vdb/linux/gcc/"
                                 #$(platform-linux-architecture
                                    (lookup-platform-by-target-or-system
                                     (or (%current-target-system)
                                         (%current-system))))
                                 "/rel/ilib")
                                (string-append #$output "/ilib"))
              ;; Install interface headers
              (copy-recursively "interfaces"
                                (string-append #$output "/include"))))
          (add-after 'install-interfaces 'install-libs
            (lambda _
              (copy-recursively (string-append
                                 "build/ncbi-vdb/linux/gcc/"
                                 #$(platform-linux-architecture
                                    (lookup-platform-by-target-or-system
                                     (or (%current-target-system)
                                         (%current-system))))
                                 "/rel/lib")
                                (string-append #$output "/lib"))))
          ;; These files are needed by sra-tools.
          (add-after 'install 'install-configuration-files
            (lambda _
              (let ((target (string-append #$output "/kfg")))
                (mkdir target)
                (install-file "libs/kfg/default.kfg" target)
                (install-file "libs/kfg/certs.kfg" target)))))))
    (inputs
     (list file hdf5-1.10 java-ngs libxml2 ngs-sdk ))
    (native-inputs (list perl))
    ;; NCBI-VDB requires SSE capability.
    (supported-systems '("i686-linux" "x86_64-linux"))
    (home-page "https://github.com/ncbi/ncbi-vdb")
    (synopsis "Database engine for genetic information")
    (description
     "The NCBI-VDB library implements a highly compressed columnar data
warehousing engine that is most often used to store genetic information.
Databases are stored in a portable image within the file system, and can be
accessed/downloaded on demand across HTTP.")
    (license license:public-domain)))

(define-public plink
  (package
    (name "plink")
    (version "1.07")
    (source
     (origin
       (method url-fetch)
       (uri (string-append
             "https://zzz.bwh.harvard.edu/plink/dist/plink-"
             version "-src.zip"))
       (sha256
        (base32 "0as8gxm4pjyc8dxmm1sl873rrd7wn5qs0l29nqfnl31x8i467xaa"))
       (patches (search-patches "plink-1.07-unclobber-i.patch"
                                "plink-endian-detection.patch"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no "check" target
       #:make-flags ,#~(list (string-append "LIB_LAPACK="
                                            #$(this-package-input "openblas")
                                            "/lib/libopenblas.so")
                             "WITH_LAPACK=1"
                             "FORCE_DYNAMIC=1"
                             ;; disable phoning home
                             "WITH_WEBCHECK=")
       #:phases
       (modify-phases %standard-phases
         ;; no "configure" script
         (delete 'configure)
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out")
                                       "/bin/")))
               (install-file "plink" bin)))))))
    (inputs
     (list zlib openblas))
    (native-inputs
     (list unzip gcc-8))
    (home-page "http://pngu.mgh.harvard.edu/~purcell/plink/")
    (synopsis "Whole genome association analysis toolset")
    (description
     "PLINK is a whole genome association analysis toolset, designed to
perform a range of basic, large-scale analyses in a computationally efficient
manner.  The focus of PLINK is purely on analysis of genotype/phenotype data,
so there is no support for steps prior to this (e.g. study design and
planning, generating genotype or CNV calls from raw data).  Through
integration with gPLINK and Haploview, there is some support for the
subsequent visualization, annotation and storage of results.")
    ;; Code is released under GPLv2, except for fisher.h, which is under
    ;; LGPLv2.1+
    (license (list license:gpl2 license:lgpl2.1+))))

(define-public plink-ng
  (package (inherit plink)
    (name "plink-ng")
    (version "2.0.0-a.6.16")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/chrchang/plink-ng")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1v020l8b5njc458silaqfwl8jg7vn46wg8hh4iiq3ip6p31g2dm7"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #false ;TEST_EXTRACT_CHR doesn't produce expected files
      #:make-flags
      #~(list "BLASFLAGS=-lopenblas"
              "NO_SSE42=1"
              "NO_AVX2=1"
              "STATIC_ZSTD="
              (string-append "CC=" #$(cc-for-target))
              (string-append "PREFIX=" #$output)
              "DESTDIR=")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'chdir
            (lambda _ (chdir "2.0/build_dynamic")))
          (add-after 'chdir 'patch-headers
            (lambda _
              (substitute* '("../include/pvar_ffi_support.h"
                             "../include/pgenlib_ffi_support.h")
                (("#include \"include/pgenlib_misc.h\"")
                 "#include <pgenlib_misc.h>"))))
          (delete 'configure)           ; no "configure" script
          (replace 'check
            (lambda* (#:key tests? inputs #:allow-other-keys)
              (when tests?
                (setenv "PATH" (string-append (getcwd) ":" (getenv "PATH")))
                (with-directory-excursion "../Tests"
                  (substitute* "run_tests.sh"
                    (("^./run_tests" m)
                     (string-append (which "bash") " " m)))
                  (invoke "bash" "run_tests.sh")))))
          (add-after 'build 'build-library
            (lambda _
              ;; Inspired by what r-saige does to the plink2 sources.
              (apply invoke "gcc" "-std=c++14" "-fPIC" "-O3" "-o" "libplink2.so"
                     (append (find-files "../include" "\\.cc$")
                             (list "-shared" "-lz" "-lzstd" "-lpthread" "-lm" "-ldeflate")))))
          (replace 'install
            (lambda _
              (install-file "plink2" (string-append #$output "/bin"))
              (install-file "libplink2.so" (string-append #$output "/lib"))
              (copy-recursively "../include" (string-append #$output "/include"))
              (for-each (lambda (header)
                          (install-file header (string-append #$output "/include")))
                        (find-files "." "\\.h$")))))))
    (inputs
     (list libdeflate openblas zlib `(,zstd "lib")))
    (native-inputs
     (list diffutils python simde)) ; for tests
    (home-page "https://www.cog-genomics.org/plink/")
    (license license:gpl3+)))

(define-public smithlab-cpp
  (let ((revision "1")
        (commit "728a097bec88c6f4b8528b685932049e660eff2e"))
    (package
      (name "smithlab-cpp")
      (version (string-append "0." revision "." (string-take commit 7)))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/smithlabcode/smithlab_cpp")
                      (commit commit)))
                (file-name (string-append name "-" version "-checkout"))
                (sha256
                 (base32
                  "0d476lmj312xk77kr9fzrv7z1bv96yfyx0w7y62ycmnfbx32ll74"))))
      (build-system gnu-build-system)
      (arguments
       `(#:modules ((guix build gnu-build-system)
                    (guix build utils)
                    (srfi srfi-26))
         #:tests? #f ;no "check" target
         #:phases
         (modify-phases %standard-phases
           (add-after 'unpack 'use-samtools-headers
            (lambda _
              (substitute* '("SAM.cpp"
                             "SAM.hpp")
                (("sam.h") "samtools/sam.h"))
              #t))
           (replace 'install
            (lambda* (#:key outputs #:allow-other-keys)
              (let* ((out     (assoc-ref outputs "out"))
                     (lib     (string-append out "/lib"))
                     (include (string-append out "/include/smithlab-cpp")))
                (mkdir-p lib)
                (mkdir-p include)
                (for-each (cut install-file <> lib)
                          (find-files "." "\\.o$"))
                (for-each (cut install-file <> include)
                          (find-files "." "\\.hpp$")))
              #t))
           (delete 'configure))))
      (inputs
       (list samtools-0.1 zlib))
      (home-page "https://github.com/smithlabcode/smithlab_cpp")
      (synopsis "C++ helper library for functions used in Smith lab projects")
      (description
       "Smithlab CPP is a C++ library that includes functions used in many of
the Smith lab bioinformatics projects, such as a wrapper around Samtools data
structures, classes for genomic regions, mapped sequencing reads, etc.")
      (license license:gpl3+))))

(define-public preseq
  (package
    (name "preseq")
    (version "2.0.3")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/smithlabcode/preseq/"
                                  "releases/download/v" version
                                  "/preseq_v" version ".tar.bz2"))
              (sha256
               (base32 "149x9xmk1wy1gff85325yfzqc0qk4sgp1w6gbyj9cnji4x1dszbl"))
              (modules '((guix build utils)))
              ;; Remove bundled samtools.
              (snippet '(delete-file-recursively "samtools"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no "check" target
       #:phases
       (modify-phases %standard-phases
         (delete 'configure))
       #:make-flags
       ,#~(list (string-append "PREFIX=" #$output)
                (string-append "LIBBAM="
                               #$(this-package-input "samtools")
                               "/lib/libbam.a")
                (string-append "SMITHLAB_CPP="
                               #$(this-package-input "smithlab-cpp")
                               "/lib")
                "PROGS=preseq"
                "INCLUDEDIRS=$(SMITHLAB_CPP)/../include/smithlab-cpp $(SAMTOOLS_DIR)")))
    (inputs
     (list gsl samtools-0.1 smithlab-cpp zlib))
    (home-page "http://smithlabresearch.org/software/preseq/")
    (synopsis "Program for analyzing library complexity")
    (description
     "The preseq package is aimed at predicting and estimating the complexity
of a genomic sequencing library, equivalent to predicting and estimating the
number of redundant reads from a given sequencing depth and how many will be
expected from additional sequencing using an initial sequencing experiment.
The estimates can then be used to examine the utility of further sequencing,
optimize the sequencing depth, or to screen multiple libraries to avoid low
complexity samples.")
    (license license:gpl3+)))

(define-public python-screed
  (package
    (name "python-screed")
    (version "1.1.3")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "screed" version))
       (sha256
        (base32 "1f3jw3g8d9cdgdggqpfgw42rfcpzdalbbral6l2mmafvqybids1p"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-importlib-resources
           python-pytest
           python-setuptools
           python-setuptools-scm))
    (home-page "https://github.com/dib-lab/screed/")
    (synopsis "Short read sequence database utilities")
    (description "Screed parses FASTA and FASTQ files and generates databases.
Values such as sequence name, sequence description, sequence quality and the
sequence itself can be retrieved from these databases.")
    (license license:bsd-3)))

(define-public python-slamdunk
  (package
    (name "python-slamdunk")
    (version "0.4.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/t-neumann/slamdunk")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0lv3h5k2pn1pz35kz0wk5xmricxzy8qscs2y7nwh0k6x4pn0m0s5"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list python-biopython
                             python-intervaltree
                             python-joblib
                             python-pandas
                             python-pybedtools
                             python-pysam))
    (native-inputs (list python-cython python-pytest python-setuptools
                         python-wheel))
    (home-page "https://t-neumann.github.io/slamdunk/")
    (synopsis "Streamline SLAM-seq analysis with high sensitivity")
    (description
     "SlamDunk is a fully automated tool for automated, robust,
scalable and reproducible SLAMseq data analysis.  Diagnostic plotting features
and a MultiQC plugin will make your SLAMseq data ready for immediate QA and
interpretation.")
    (license license:agpl3+)))

(define-public python-taggd
  (package
    (name "python-taggd")
    (version "0.4.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/jfnavarro/taggd")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17hi1vs1qwhxx8jnradnl9k471li6fjb6w5sljkpzjxy7rkxwb85"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'check 'remove-local-taggd
            (lambda _
              ;; This would otherwise interfere with finding the installed
              ;; taggd when running tests.
              (delete-file-recursively "taggd"))))))
    (propagated-inputs
     (list python-numpy
           python-pysam
           python-tqdm
           python-aiofiles
           python-dnaio
           python-types-aiofiles
           python-types-tqdm))
    (native-inputs
     (list python-cython python-pytest python-setuptools))
    (home-page "https://github.com/jfnavarro/taggd")
    (synopsis "Genetic barcode demultiplexing")
    (description "This package provides TagGD barcode demultiplexing utilities
for Spatial Transcriptomics data.")
    (license license:bsd-3)))

(define-public stpipeline
  (package
    (name "stpipeline")
    (version "2.0.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/jfnavarro/st_pipeline")
              (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1qah9sa7wy9ywf0si2ngqg0qyr9jjp5gxmjx3y65i78bxyq8pfyx"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases '(modify-phases %standard-phases
                  ;; requirements.txt and pyproject.toml have all versions
                  ;; of the dependencies hardcoded. All tests pass, so it should
                  ;; be good enough.
                  ;; However, the sanity-check of any Python package that has
                  ;; stpipelines a dependency, would fail too.
                  (delete 'sanity-check))))
    (propagated-inputs (list htseq
                             python-distance
                             python-dnaio
                             python-numpy
                             python-pandas
                             python-pysam
                             python-regex
                             python-scikit-learn
                             python-scipy
                             python-seaborn
                             python-taggd
                             python-types-regex
                             samtools
                             star))
    (native-inputs (list
                    python-cython
                    python-pytest
                    python-poetry-core))
    (home-page "https://github.com/jfnavarro/st_pipeline")
    (synopsis "Pipeline for spatial mapping of unique transcripts")
    (description
     "This package provides an automated pipeline for spatial mapping of
unique transcripts.")
    (license license:expat)))

(define-public sra-tools
  (package
    (name "sra-tools")
    (version "2.10.6")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/ncbi/sra-tools")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1cr2mijkfs5sm35ffjs6861qsd1qkgnhnbavdv65zg5d655abbjf"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:parallel-build? #f             ; not supported
      #:tests? #f                      ; no "check" target
      #:make-flags
      #~(list (string-append "DEFAULT_CRT="
                             #$(this-package-input "ncbi-vdb")
                             "/kfg/certs.kfg")
              (string-append "DEFAULT_KFG="
                             #$(this-package-input "ncbi-vdb")
                             "/kfg/default.kfg")
              (string-append "VDB_LIBDIR="
                             #$(this-package-input "ncbi-vdb")
                             #$(if (string-prefix? "x86_64"
                                                   (or (%current-target-system)
                                                       (%current-system)))
                                   "/lib64"
                                   "/lib32")))
       #:phases
       #~(modify-phases %standard-phases
           (add-before 'configure 'set-perl-search-path
             (lambda _
               ;; Work around "dotless @INC" build failure.
               (setenv "PERL5LIB"
                       (string-append (getcwd) "/setup:"
                                      (getenv "PERL5LIB")))))
           (replace 'configure
             (lambda* (#:key inputs outputs #:allow-other-keys)
               ;; The build system expects a directory containing the sources and
               ;; raw build output of ncbi-vdb, including files that are not
               ;; installed.  Since we are building against an installed version of
               ;; ncbi-vdb, the following modifications are needed.
               (substitute* "setup/konfigure.perl"
                 ;; Make the configure script look for the "ilib" directory of
                 ;; "ncbi-vdb" without first checking for the existence of a
                 ;; matching library in its "lib" directory.
                 (("^            my \\$f = File::Spec->catdir\\(\\$libdir, \\$lib\\);")
                  "my $f = File::Spec->catdir($ilibdir, $ilib);")
                 ;; Look for interface libraries in ncbi-vdb's "ilib" directory.
                 (("my \\$ilibdir = File::Spec->catdir\\(\\$builddir, 'ilib'\\);")
                  "my $ilibdir = File::Spec->catdir($dir, 'ilib');"))

               ;; Dynamic linking
               (substitute* "tools/copycat/Makefile"
                 (("smagic-static") "lmagic"))
               (substitute* "tools/driver-tool/utf8proc/Makefile"
                 (("CC\\?=gcc") "myCC=gcc")
                 (("\\(CC\\)") "(myCC)"))

               ;; Don't link libxml2 statically
               (substitute* "build/ld.linux.exe.sh"
                 (("grep -q 'OS_DISTRIBUTOR = Ubuntu.*") "true\n"))

               ;; The 'configure' script doesn't recognize things like
               ;; '--enable-fast-install'.
               (invoke "./configure"
                       (string-append "--build-prefix=" (getcwd) "/build")
                       (string-append "--prefix=" #$output)
                       (string-append "--debug")
                       (string-append "--with-fuse-prefix="
                                      #$(this-package-input "fuse"))
                       (string-append "--with-magic-prefix="
                                      #$(this-package-input "file"))
                       (string-append "--with-xml2-prefix="
                                      #$(this-package-input "libxml2"))
                       (string-append "--with-ncbi-vdb-sources="
                                      #$(this-package-input "ncbi-vdb"))
                       (string-append "--with-ncbi-vdb-build="
                                      #$(this-package-input "ncbi-vdb"))
                       (string-append "--with-ngs-sdk-prefix="
                                      #$(this-package-input "ngs-sdk"))
                       (string-append "--with-hdf5-prefix="
                                      #$(this-package-input "hdf5"))))))))
    (native-inputs (list perl))
    (inputs
     (list ngs-sdk
           ncbi-vdb
           file
           fuse-2
           hdf5-1.10
           libxml2
           zlib
           python-wrapper))
    (home-page
     "https://trace.ncbi.nlm.nih.gov/Traces/sra/sra.cgi?view=software")
    (synopsis "Tools and libraries for reading and writing sequencing data")
    (description
     "The SRA Toolkit from NCBI is a collection of tools and libraries for
reading of sequencing files from the Sequence Read Archive (SRA) database and
writing files into the .sra format.")
    (license license:public-domain)))

(define-public seqan
  (package
    (name "seqan")
    (version "3.0.3")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/seqan/seqan3/releases/"
                                  "download/" version "/seqan3-"
                                  version "-Source.tar.xz"))
              (sha256
               (base32
                "1h2z0cvgidhkmh5xsbw75waqbrqbbv6kkrvb0b92xfh3gqpaiz22"))))
    (build-system cmake-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (replace 'check
           (lambda _
             (invoke "ctest" "test" "--output-on-failure"))))))
    (native-inputs
     (list bzip2 cereal zlib))
    (home-page "https://www.seqan.de")
    (synopsis "Library for nucleotide sequence analysis")
    (description
     "SeqAn is a C++ library of efficient algorithms and data structures for
the analysis of sequences with the focus on biological data.  It contains
algorithms and data structures for string representation and their
manipulation, online and indexed string search, efficient I/O of
bioinformatics file formats, sequence alignment, and more.")
    (license license:bsd-3)))

(define-public seqan-2
  (package
    (inherit seqan)
    (version "2.4.0")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/seqan/seqan/releases/"
                                  "download/seqan-v" version
                                  "/seqan-library-" version ".tar.xz"))
              (sha256
               (base32
                "19a1rlxx03qy1i1iriicly68w64yjxbv24g9gdywnfmq998v35yx"))))
    ;; The documentation is 7.8MB and the includes are 3.6MB heavy, so it
    ;; makes sense to split the outputs.
    (outputs '("out" "doc"))
    (build-system trivial-build-system)
    (arguments
     `(#:modules ((guix build utils))
       #:builder
       ,#~(begin
            (use-modules (guix build utils))
            (let ((tar #$(this-package-native-input "tar"))
                  (xz  #$(this-package-native-input "xz"))
                  (out #$output)
                  (doc #$output:doc))
              (setenv "PATH" (string-append tar "/bin:" xz "/bin"))
              (invoke "tar" "xvf" #$(this-package-native-input "source"))
              (chdir (string-append "seqan-library-" #$version))
              (copy-recursively "include" (string-append out "/include"))
              (copy-recursively "share"  (string-append doc "/share"))))))
    (native-inputs
     `(("source" ,source)
       ("tar" ,tar)
       ("xz" ,xz)))))

(define-public seqan-1
  (package (inherit seqan)
    (name "seqan")
    (version "1.4.2")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://packages.seqan.de/seqan-library/"
                                  "seqan-library-" version ".tar.bz2"))
              (sha256
               (base32
                "05s3wrrwn50f81aklfm65i4a749zag1vr8z03k21xm0pdxy47yvp"))))
    ;; The documentation is 7.8MB and the includes are 3.6MB heavy, so it
    ;; makes sense to split the outputs.
    (outputs '("out" "doc"))
    (build-system trivial-build-system)
    (arguments
     `(#:modules ((guix build utils))
       #:builder
       ,#~(begin
            (use-modules (guix build utils))
            (let ((tar  #$(this-package-native-input "tar"))
                  (bzip #$(this-package-native-input "bzip2"))
                  (out #$output)
                  (doc #$output:doc))
              (setenv "PATH" (string-append tar "/bin:" bzip "/bin"))
              (invoke "tar" "xvf" #$(this-package-native-input "source"))
              (chdir (string-append "seqan-library-" #$version))
              (copy-recursively "include" (string-append out "/include"))
              (copy-recursively "share"  (string-append doc "/share"))))))
    (native-inputs
     `(("source" ,source)
       ("tar" ,tar)
       ("bzip2" ,bzip2)))))

(define-public seqmagick
  (let ((commit "dee6ab9839cff317142e55678bb12c4c9ab2150f")
        (revision "0"))
    (package
      (name "seqmagick")
      (version (git-version "0.8.6" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
                (url "https://github.com/fhcrc/seqmagick")
                (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0syipb7m44s5bqrhs17bwr28svy2s83j8d93kbazav92jzszzsw4"))))
      (build-system pyproject-build-system)
      (arguments
       (list
        #:phases
        #~(modify-phases %standard-phases
            (add-after 'unpack 'set-version
              ;; Alternative of setup.py version check:
              ;;
              ;; subprocess.call(
              ;; ('mkdir -p seqmagick/data && '
              ;;  'git describe --tags --dirty > seqmagick/data/ver.tmp '
              ;;  '&& mv seqmagick/data/ver.tmp seqmagick/data/ver '
              ;;  '|| rm -f seqmagick/data/ver.tmp'),
              ;; shell=True, stderr=open(os.devnull, "w"))
              ;;
              (lambda _
                (mkdir "seqmagick/data")
                (with-output-to-file "seqmagick/data/ver"
                  (lambda _ (display #$version))))))))
      (native-inputs
       (list python-pynose
             python-setuptools))
      (inputs (list python-biopython python-pygtrie))
      (home-page "https://github.com/fhcrc/seqmagick")
      (synopsis "Tools for converting and modifying sequence files")
      (description
       "Bioinformaticians often have to convert sequence files between formats
and do little manipulations on them, and it's not worth writing scripts for
that.  Seqmagick is a utility to expose the file format conversion in
BioPython in a convenient way.  Instead of having a big mess of scripts, there
is one that takes arguments.")
      (license license:gpl3))))

(define-public seqtk
  (package
    (name "seqtk")
    (version "1.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/lh3/seqtk")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bfzlqa84b5s1qi22blmmw2s8xdyp9h9ydcq22pfjhh5gab3yz6l"))))
    (build-system gnu-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'check
           ;; There are no tests, so we just run a sanity check.
           (lambda _ (invoke "./seqtk" "seq") #t))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
               (install-file "seqtk" bin)
               #t))))))
    (inputs
     (list zlib))
    (home-page "https://github.com/lh3/seqtk")
    (synopsis "Toolkit for processing biological sequences in FASTA/Q format")
    (description
     "Seqtk is a fast and lightweight tool for processing sequences in the
FASTA or FASTQ format.  It parses both FASTA and FASTQ files which can be
optionally compressed by gzip.")
      (license license:expat)))

(define-public snap-aligner
  (package
    (name "snap-aligner")
    (version "2.0.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/amplab/snap")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0yag3jcazdqfxgmw0vbi91i98kj9sr0aqx83qqj6m5b45wxs7jms"))))
    (build-system gnu-build-system)
    (arguments
     '(#:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests? (invoke "./unit_tests"))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin")))
               (install-file "snap-aligner" bin)
               (install-file "SNAPCommand" bin)))))))
    (native-inputs
     (list zlib))
    (home-page "https://snap.cs.berkeley.edu/")
    (synopsis "Short read DNA sequence aligner")
    (description
     "SNAP is a fast and accurate aligner for short DNA reads.  It is
optimized for modern read lengths of 100 bases or higher, and takes advantage
of these reads to align data quickly through a hash-based indexing scheme.")
    ;; 32-bit systems are not supported by the unpatched code.
    ;; Following the bug reports https://github.com/amplab/snap/issues/68 and
    ;; https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=812378 we see that
    ;; systems without a lot of memory cannot make good use of this program.
    (supported-systems '("x86_64-linux"))
    (license license:asl2.0)))

(define-public sortmerna
  (package
    (name "sortmerna")
    (version "4.3.7")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/biocore/sortmerna")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "14p3zi6ysnrs2l33r5mc0fr0la2jnnxmqdcif02fppln8w31j753"))))
    (build-system cmake-build-system)
    (outputs '("out"      ;for binaries
               "db"))     ;for sequence databases
    (arguments
     (list
      #:tests? #false ;unclear how to run them
      #:configure-flags
      #~(list "-DWITH_TESTS=ON"
              "-DCMAKE_CXX_FLAGS=-pthread"
              "-DZLIB_STATIC=OFF"
              "-DROCKSDB_STATIC=OFF"
              "-DPORTABLE=OFF" ;do not use static linking
              (string-append "-DROCKSDB_HOME="
                             #$(this-package-input "rocksdb"))
              (string-append "-DRAPIDJSON_HOME="
                             #$(this-package-input "rapidjson"))
              (string-append "-DRapidJson_DIR="
                             #$(this-package-input "rapidjson")
                             "/lib/cmake/RapidJSON")
              (string-append "-DRapidJSON_INCLUDE_DIR="
                             #$(this-package-input "rapidjson")
                             "/include"))
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'fix-includes
           (lambda _
             (substitute* "src/sortmerna/kseq_load.cpp"
               (("#include <iostream>" all)
                (string-append all "\n#include <cstdint>")))))
         (add-after 'unpack 'find-concurrentqueue-headers
           (lambda* (#:key inputs #:allow-other-keys)
             ;; Ensure that headers can be found
             (setenv "CPLUS_INCLUDE_PATH"
                     (string-append (search-input-directory
                                     inputs "/include/concurrentqueue")
                                    ":"
                                    (or (getenv "CPLUS_INCLUDE_PATH") "")))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out   (assoc-ref outputs "out"))
                    (bin   (string-append out "/bin"))
                    (db    (assoc-ref outputs "db"))
                    (share
                     (string-append db "/share/sortmerna/rRNA_databases")))
               (install-file "src/sortmerna" bin)
               (for-each (lambda (file)
                           (install-file file share))
                         (find-files "../source/data/rRNA_databases" ".*fasta"))))))))
    (inputs
     (list concurrentqueue
           gflags ; because of rocksdb
           rapidjson rocksdb zlib))
    (native-inputs
     (list pkg-config))
    (home-page "https://bioinfo.lifl.fr/RNA/sortmerna/")
    (synopsis "Biological sequence analysis tool for NGS reads")
    (description
     "SortMeRNA is a biological sequence analysis tool for filtering, mapping
and @acronym{OTU, operational taxonomic unit} picking of @acronym{NGS, next
generation sequencing} reads.  The core algorithm is based on approximate seeds
and allows for fast and sensitive analyses of nucleotide sequences.  The main
application of SortMeRNA is filtering rRNA from metatranscriptomic data.")
    ;; The source includes x86 specific code
    (supported-systems '("x86_64-linux" "i686-linux"))
    (license license:lgpl3)))

(define-public star
  (package
    (name "star")
    (version "2.7.8a")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/alexdobin/STAR")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zc5biymja9zml9yizcj1h68fq9c6sxfcav8a0lbgvgsm44rvans"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  (substitute* "source/Makefile"
                    (("/bin/rm") "rm"))
                  ;; Remove pre-built binaries and bundled htslib sources.
                  (delete-file-recursively "bin/MacOSX_x86_64")
                  (delete-file-recursively "bin/Linux_x86_64")
                  (delete-file-recursively "bin/Linux_x86_64_static")
                  (delete-file-recursively "source/htslib")
                  #t))))
    (build-system gnu-build-system)
    (arguments
     '(#:tests? #f ;no check target
       #:make-flags '("STAR")
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'enter-source-dir
           (lambda _ (chdir "source") #t))
         (add-after 'enter-source-dir 'make-reproducible
           (lambda _
             (substitute* "Makefile"
               (("(COMPILATION_TIME_PLACE=\")(.*)(\")" _ pre mid post)
                (string-append pre "Built with Guix" post)))
             #t))
         ;; See https://github.com/alexdobin/STAR/pull/562
         (add-after 'enter-source-dir 'add-missing-header
           (lambda _
             (substitute* "SoloReadFeature_inputRecords.cpp"
               (("#include \"binarySearch2.h\"" h)
                (string-append h "\n#include <math.h>")))
             #t))
         (add-after 'enter-source-dir 'do-not-use-bundled-htslib
           (lambda _
             (substitute* "Makefile"
               (("(Depend.list: \\$\\(SOURCES\\) parametersDefault\\.xxd) htslib"
                 _ prefix) prefix))
             (substitute* '("BAMfunctions.cpp"
                            "signalFromBAM.h"
                            "bam_cat.h"
                            "bam_cat.c"
                            "STAR.cpp"
                            "bamRemoveDuplicates.cpp")
               (("#include \"htslib/([^\"]+\\.h)\"" _ header)
                (string-append "#include <" header ">")))
             (substitute* "IncludeDefine.h"
               (("\"htslib/(htslib/[^\"]+.h)\"" _ header)
                (string-append "<" header ">")))
             #t))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
               (install-file "STAR" bin))
             #t))
         (delete 'configure))))
    (native-inputs
     (list xxd))
    (inputs
     (list htslib zlib))
    (home-page "https://github.com/alexdobin/STAR")
    (synopsis "Universal RNA-seq aligner")
    (description
     "The Spliced Transcripts Alignment to a Reference (STAR) software is
based on a previously undescribed RNA-seq alignment algorithm that uses
sequential maximum mappable seed search in uncompressed suffix arrays followed
by seed clustering and stitching procedure.  In addition to unbiased de novo
detection of canonical junctions, STAR can discover non-canonical splices and
chimeric (fusion) transcripts, and is also capable of mapping full-length RNA
sequences.")
    ;; Only 64-bit systems are supported according to the README.
    (supported-systems '("x86_64-linux" "mips64el-linux"))
    ;; STAR is licensed under GPLv3 or later; htslib is MIT-licensed.
    (license license:gpl3+)))

(define-public star-for-pigx
  (package
    (inherit star)
    (name "star")
    (version "2.7.3a")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/alexdobin/STAR")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1hgiqw5qhs0pc1xazzihcfd92na02xyq2kb469z04y1v51kpvvjq"))
              (modules '((guix build utils)))
              (snippet
               '(begin
                  (substitute* "source/Makefile"
                    (("/bin/rm") "rm"))
                  ;; Remove pre-built binaries and bundled htslib sources.
                  (delete-file-recursively "bin/MacOSX_x86_64")
                  (delete-file-recursively "bin/Linux_x86_64")
                  (delete-file-recursively "bin/Linux_x86_64_static")
                  (delete-file-recursively "source/htslib")
                  #t))))))

(define-public starlong
  (package (inherit star)
    (name "starlong")
    (arguments
     (substitute-keyword-arguments (package-arguments star)
       ((#:make-flags flags)
        `(list "STARlong"))
       ((#:phases phases)
        `(modify-phases ,phases
           ;; Allow extra long sequence reads.
           (add-after 'unpack 'make-extra-long
             (lambda _
               (substitute* "source/IncludeDefine.h"
                 (("(#define DEF_readNameLengthMax ).*" _ match)
                  (string-append match "900000\n")))
               #t))
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
                 (install-file "STARlong" bin))
               #t))))))))

(define-public subread
  (package
    (name "subread")
    (version "2.0.3")
    (source (origin
              (method url-fetch)
              (uri (string-append "mirror://sourceforge/subread/subread-"
                                  version "/subread-" version "-source.tar.gz"))
              (sha256
               (base32
                "0szmllia7jl0annk5568xjhw6cc8yj1c5mb961qk5m0lz6ig7kjn"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no "check" target
      ;; The CC and CCFLAGS variables are set to contain a lot of x86_64
      ;; optimizations by default, so we override these flags such that x86_64
      ;; flags are only added when the build target is an x86_64 system.
       #:make-flags
       (list (let ((system ,(or (%current-target-system)
                                (%current-system)))
                   (flags '("-ggdb" "-fomit-frame-pointer"
                            "-ffast-math" "-funroll-loops"
                            "-fmessage-length=0" "-fcommon"
                            "-O9" "-Wall" "-DMAKE_FOR_EXON"
                            "-DMAKE_STANDALONE"
                            "-DSUBREAD_VERSION=\\\"${SUBREAD_VERSION}\\\""))
                   (flags64 '("-mmmx" "-msse" "-msse2" "-msse3")))
               (if (string-prefix? "x86_64" system)
                   (string-append "CCFLAGS=" (string-join (append flags flags64)))
                   (string-append "CCFLAGS=" (string-join flags))))
             "-f" "Makefile.Linux"
             "CC=gcc ${CCFLAGS}")
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'enter-dir
           (lambda _ (chdir "src")))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
               (mkdir-p bin)
               (copy-recursively "../bin" bin))))
         ;; no "configure" script
         (delete 'configure))))
    (inputs (list zlib))
    (home-page "https://subread.sourceforge.net/")
    (synopsis "Tool kit for processing next-gen sequencing data")
    (description
     "The subread package contains the following tools: subread aligner, a
general-purpose read aligner; subjunc aligner: detecting exon-exon junctions
and mapping RNA-seq reads; featureCounts: counting mapped reads for genomic
features; exactSNP: a SNP caller that discovers SNPs by testing signals
against local background noises.")
    (license license:gpl3+)))

(define-public stringtie
  (package
    (name "stringtie")
    (version "3.0.2")
    (source (origin
              (method url-fetch)
              (uri (string-append "http://ccb.jhu.edu/software/stringtie/dl/"
                                  "stringtie-" version ".tar.gz"))
              (sha256
               (base32
                "1mrqcvnhxbidyhf521naywb5ci579160ydqnl5kb4hqjqvm01mn2"))
              ;; This package bundles an annoying amount of third party source
              ;; code.
              (modules '((guix build utils)))
              (snippet
               '(delete-file-recursively "htslib"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ;no test suite
       #:make-flags '("LIBDEFLATE=-ldeflate"
                      "LIBBZ2=-lbz2"
                      "LIBLZMA=-llzma")
       #:phases
       (modify-phases %standard-phases
         ;; no configure script
         (delete 'configure)
         (add-before 'build 'use-system-samtools
           (lambda _
             (substitute* "Makefile"
               ((" -lm") " -lm -lhts")
               ((" \\$\\{HTSLIB\\}/libhts\\.a") " "))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin/")))
               (install-file "stringtie" bin)))))))
    (inputs
     (list bzip2 htslib libdeflate zlib))
    (home-page "https://ccb.jhu.edu/software/stringtie/")
    (synopsis "Transcript assembly and quantification for RNA-Seq data")
    (description
     "StringTie is a fast and efficient assembler of RNA-Seq sequence
alignments into potential transcripts.  It uses a novel network flow algorithm
as well as an optional de novo assembly step to assemble and quantitate
full-length transcripts representing multiple splice variants for each gene
locus.  Its input can include not only the alignments of raw reads used by
other transcript assemblers, but also alignments of longer sequences that have
been assembled from those reads.  To identify differentially expressed genes
between experiments, StringTie's output can be processed either by the
Cuffdiff or Ballgown programs.")
    (license license:expat)))

(define-public taxtastic
  (package
    (name "taxtastic")
    (version "0.11.1")
    (source (origin
              ;; The Pypi version does not include tests.
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/fhcrc/taxtastic")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18h3vlyx9qp7xymd8ra6jn5h0vzlpzgcd75kslqyv2qcg1a7scc4"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags #~(list "-k" "not test_new_nodes02")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'prepare-directory
            (lambda _
              ;; This package is unknown to pypi.
              (substitute* '("requirements.txt" "setup.py")
                ((".*psycopg-binary.*") ""))
              ;; This test fails, but the error is not caught by the test
              ;; framework, so the tests fail...
              (substitute* "tests/test_taxit.py"
                (("self.cmd_fails\\(''\\)")
                 "self.cmd_fails('nothing')"))
              ;; This version file is expected to be created with git describe.
              (mkdir-p "taxtastic/data")
              (with-output-to-file "taxtastic/data/ver"
                (lambda () (display #$version)))))
          (replace 'check
            ;; Note, this fails to run with "-v" as it tries to write to a
            ;; closed output stream.
            (lambda* (#:key tests? test-flags #:allow-other-keys)
              (when tests?
                (apply invoke "python" "-m" "unittest" test-flags)))))))
    (native-inputs
     (list python-setuptools python-wheel))
    (propagated-inputs
     (list python-biopython
           python-decorator
           python-dendropy
           python-fastalite
           python-jinja2
           python-pandas
           python-psycopg
           python-psycopg2-binary
           python-pyyaml
           python-sqlalchemy-2
           python-sqlparse))
    (home-page "https://github.com/fhcrc/taxtastic")
    (synopsis "Tools for taxonomic naming and annotation")
    (description
     "Taxtastic is software written in python used to build and maintain
reference packages i.e. collections of reference trees, reference alignments,
profiles, and associated taxonomic information.")
    (license license:gpl3+)))

(define-public vcftools
  (package
    (name "vcftools")
    (version "0.1.16")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "https://github.com/vcftools/vcftools/releases/download/v"
                    version "/vcftools-" version ".tar.gz"))
              (sha256
               (base32
                "1qqlx7flfv7axrjwkaz6njkscsl1d0jw98ns8d8bh1n1hd1pgz6v"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; no "check" target
       #:make-flags
       ,#~(list
           "CFLAGS=-O2"                 ; override "-m64" flag
           (string-append "PREFIX=" #$output)
           (string-append "MANDIR=" #$output "/share/man/man1"))))
    (native-inputs
     (list pkg-config))
    (inputs
     (list perl zlib))
    (home-page "https://vcftools.github.io/")
    (synopsis "Tools for working with VCF files")
    (description
     "VCFtools is a program package designed for working with VCF files, such
as those generated by the 1000 Genomes Project.  The aim of VCFtools is to
provide easily accessible methods for working with complex genetic variation
data in the form of VCF files.")
    ;; The license is declared as LGPLv3 in the README and
    ;; at https://vcftools.github.io/license.html
    (license license:lgpl3)))

(define-public infernal
  (package
    (name "infernal")
    (version "1.1.4")
    (source (origin
              (method url-fetch)
              (uri (string-append "http://eddylab.org/software/infernal/"
                                  "infernal-" version ".tar.gz"))
              (sha256
               (base32
                "1z4mgwqg1j4n5ika08ai8mg9yjyjhf4821jp83v2bgwzxrykqjgr"))))
    (build-system gnu-build-system)
    (native-inputs
     (list perl python)) ; for tests
    (home-page "http://eddylab.org/infernal/")
    (synopsis "Inference of RNA alignments")
    (description "Infernal (\"INFERence of RNA ALignment\") is a tool for
searching DNA sequence databases for RNA structure and sequence similarities.
It is an implementation of a special case of profile stochastic context-free
grammars called @dfn{covariance models} (CMs).  A CM is like a sequence
profile, but it scores a combination of sequence consensus and RNA secondary
structure consensus, so in many cases, it is more capable of identifying RNA
homologs that conserve their secondary structure more than their primary
sequence.")
    ;; Infernal 1.1.3 requires VMX or SSE capability for parallel instructions.
    (supported-systems '("i686-linux" "x86_64-linux"))
    (license license:bsd-3)))

(define-public r-genenmf
  (let ((commit "7ac475261123232d3e691ef851936e05055becd4")
        (revision "1"))
    (package
      (name "r-genenmf")
      (version (git-version "0.9.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
                (url "https://github.com/carmonalab/GeneNMF")
                (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0kv419nyyddpj2j2dvhxhwk5m26p39f3jc2r7rz98j86x6hgwgpl"))))
      (properties `((upstream-name . "GeneNMF")))
      (build-system r-build-system)
      (propagated-inputs (list r-cluster
                               r-colorspace
                               r-dendextend
                               r-irlba
                               r-lsa
                               r-matrix
                               r-pheatmap
                               r-rcppml
                               r-seurat
                               r-viridis))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/carmonalab/GeneNMF")
      (synopsis "Non-Negative matrix factorization for single-cell omics")
      (description
       "This package provides a collection of methods to extract gene programs
from single-cell gene expression data using @dfn{non-negative matrix
factorization} (NMF).  @code{GeneNMF} contains functions to directly interact
with the Seurat toolkit and derive interpretable gene program signatures.")
      (license license:gpl3))))

(define-public r-ggexpress
  (let ((commit "82f169385f87af328ff971195c2f64ff3c573a8a")
        (revision "1"))
    (package
      (name "r-ggexpress")
      (version (git-version "0.6.6" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/ggExpress")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "122hnw8xv33ngcd4fd7hmf817a06ih1knrxxi0cgklj1fwinm0z4"))))
      (properties `((upstream-name . "ggExpress")))
      (build-system r-build-system)
      (propagated-inputs (list r-cowplot
                               r-ggplot2
                               r-ggpubr
                               r-markdownhelpers
                               r-markdownreports
                               r-rcolorbrewer
                               r-sessioninfo
                               r-seurat
                               r-sm
                               r-stringendo
                               r-tidyverse))
      (home-page "https://github.com/vertesy/ggExpress")
      (synopsis
       "This is a fast tool to create, annotate and export plots in R")
      (description
       "This package is a set of R functions for generating precise figures.
This tool helps you to create clean markdown reports about what you just
discovered with your analysis script.")
      (license license:gpl3))))

(define-public r-gg3d
  (let ((commit "ffdd837d30c1671cd0895db94bdd7b1594dbfcb0")
        (revision "1"))
    (package
      (name "r-gg3d")
      (version (git-version "0.0.0.9000" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/AckerDWM/gg3D")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0vz5laj5vz9m1jzb7h7q9f37zjmxwhylhaidlsg61r57wpxvjr3p"))))
      (properties `((upstream-name . "gg3D")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-dplyr
             r-ggplot2
             r-magrittr
             r-plot3d
             r-plyr
             r-scales))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/AckerDWM/gg3D")
      (synopsis "3D perspective plots for ggplot2")
      (description
       "This package adds 3D perspective plotting of points, paths, and line,
3D perspective axes, 3D perspective annotations, and wireframe plots.")
      (license license:gpl3+))))

(define-public r-ggsankey
  (let ((commit "be08dd0f86eaee9f9ff9e7ff95d47930660a3c36")
        (revision "1"))
    (package
      (name "r-ggsankey")
      (version (git-version "0.0.99999" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/davidsjoberg/ggsankey")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0acpmydqqc91pq5p9wpkpmgqp3nhiljabd7d3i00kwhjxgm2bvba"))))
      (properties `((upstream-name . "ggsankey")))
      (build-system r-build-system)
      (propagated-inputs (list r-dplyr
                               r-ggplot2
                               r-magrittr
                               r-purrr
                               r-stringr
                               r-tidyr))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/davidsjoberg/ggsankey")
      (synopsis "Sankey, Alluvial and Sankey bump plots")
      (description
       "This package provides a package that makes it easy to implement
sankey, alluvial and sankey bump plots in @code{ggplot2}.")
      (license license:expat))))

(define-public r-gutils
  (let ((commit "fc24db6f73898c3c25e400e7d627e3e5229003f2")
        (revision "2"))
    (package
      (name "r-gutils")
      (version (git-version "0.2.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/mskilab/gUtils")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "11ijzy1zyjv2wgxrfcaan7g82jl27skd41hw4s2xh9lijkn8ilwh"))))
      (properties `((upstream-name . "gUtils")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           ;; These deprecated procedures have been removed in testthat.
           (add-after 'unpack 'testthat-compatibility
             (lambda _
               (substitute* "tests/testthat/test_rangeops.R"
                 (("is_true\\(\\)") "expect_true")))))))
      (propagated-inputs
       (list r-biocgenerics
             r-data-table
             r-genomeinfodb
             r-genomicranges
             r-iranges
             r-matrix
             r-s4vectors
             r-stringr))
      (native-inputs (list r-testthat r-xvector))
      (home-page "https://github.com/mskilab/gUtils")
      (synopsis "Additional capabilities and speed for GenomicRanges operations")
      (description
       "This is an R package providing additional capabilities and speed for
@code{GenomicRanges} operations.")
      (license license:gpl2))))

(define-public r-bamutils
  (let ((commit "0a7d2d90cc720aac3fe44f964abf294f1b827f32")
        (revision "2"))
    (package
      (name "r-bamutils")
      (version (git-version "0.0.0.9000" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/mskilab/bamutils/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1r3y8775y2j9w92pdp42iaif7nbh260xljrhaswkz6lp71phf1yw"))))
      (properties `((upstream-name . "bamUtils")))
      (build-system r-build-system)
      ;; See https://github.com/mskilab-org/bamUtils/issues/12
      (arguments (list #:tests? #false))
      (propagated-inputs
       (list r-abind
             r-biocgenerics
             r-data-table
             r-genomicalignments
             r-genomicranges
             r-gutils
             r-rsamtools
             r-variantannotation))
      (native-inputs (list r-testthat samtools))
      (home-page "https://github.com/mskilab/bamutils/")
      (synopsis "Utility functions for manipulating BAMs")
      (description "This package provides utility functions for manipulating
BAM files.")
      (license license:gpl2))))

(define-public r-gtrack
  (let ((commit "a694fa36cedafca2658da79fc8e5b673535b15e5")
        (revision "1"))
    (package
      (name "r-gtrack")
      (version (git-version "0.1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/mskilab/gTrack/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "070qlrbqsbj9max2vx740zigqh0ymvnw2pm1ia5la3wb4dbfwh2b"))))
      (properties `((upstream-name . "gTrack")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biocgenerics
             r-data-table
             r-genomeinfodb
             r-genomicranges
             r-gutils
             r-iranges
             r-matrix
             r-rcolorbrewer
             r-rcpp
             r-rcurl
             r-rtracklayer
             r-s4vectors))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/mskilab/gTrack/")
      (synopsis "Plot tracks of complex genomic data across multiple genomic windows")
      (description
       "This package provides an object for plotting GRanges, RleList, UCSC
file formats, and ffTrack objects in multi-track panels.")
      (license license:gpl2))))

(define-public r-gchain
  (let ((commit "19f8bb924fdadca136b9827b8538574b278e86a1")
        (revision "2"))
    (package
      (name "r-gchain")
      (version (git-version "0.2.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/mskilab/gChain/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0cg7cy97g2w2dcd980jz5fawjjrfl590hdx3rsblavcdlnzcs3x4"))))
      (properties `((upstream-name . "gChain")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'biostrings-compatibility
             (lambda _
               (substitute* "R/gChain.R"
                 ((" pairwiseAlignment") " pwalign::pairwiseAlignment")
                 (("alignedPattern") "pwalign::alignedPattern")
                 (("deletion\\(") "pwalign::deletion(")
                 (("insertion\\(") "pwalign::insertion(")
                 (("Biostrings::pattern") "pwalign::pattern"))))
           (add-after 'unpack 'skip-bad-tests
             (lambda _
               (substitute* "tests/testthat/test_gChain.R"
                 ;; C stack usage  7973568 is too close to the limit
                 ((".*'testing lift\\(\\) works'.*" m)
                  (string-append m "skip('guix')"))
                 ;; C stack usage  7973568 is too close to the limit
                 ((".*'testing \"\\*\" works'.*" m)
                  (string-append m "skip('guix')"))))))))
      (propagated-inputs
       (list r-bamutils
             r-biostrings
             r-data-table
             r-genomicalignments
             r-genomicranges
             r-gtrack
             r-gutils
             r-matrix
             r-rtracklayer))
      (native-inputs (list r-pwalign r-testthat))
      (home-page "https://github.com/mskilab/gChain/")
      (synopsis "Additional capabilities and speed for GenomicRanges operations")
      (description
       "This R package provides additional capabilities and speed for
GenomicRanges operations.")
      (license license:gpl2))))

(define-public r-skitools
  (let ((commit "ba322dcc2c0ca24a7f3dabecfb37f95e0e2186d1")
        (revision "2"))
    (package
      (name "r-skitools")
      (version (git-version "0.0.0.9000" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/mskilab/skitools/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1za371x1cmkmlmicbhzxwycg7g1z9q3d5j7ia2rzayf30s9rcc47"))))
      (properties `((upstream-name . "skitools")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'disable-bad-test
             (lambda _
               ;; See https://github.com/mskilab-org/skitools/issues/11
               (substitute* "tests/testthat/test_skitools.R"
                 ((".*test ra.merge.*" m)
                  (string-append m "skip('guix')\n"))))))))
      (propagated-inputs
       (list r-biostrings
             r-complexheatmap
             r-data-table
             r-devtools
             r-dt
             r-gchain
             r-genomeinfodb
             r-genomicranges
             r-ggplot2
             r-gplots
             r-gutils
             r-htmlwidgets
             r-hwriter
             r-igraph
             r-iranges
             r-plotly
             r-rcolorbrewer
             r-reshape2
             r-s4vectors
             r-stringr
             r-variantannotation))
      (native-inputs (list r-bsgenome-hsapiens-ucsc-hg19))
      (home-page "https://github.com/mskilab/skitools/")
      (synopsis "Various mskilab R utilities")
      (description
       "This package provides R miscellaneous utilities for basic data
manipulation, debugging, visualization, lsf management, and common mskilab
tasks.")
      (license license:expat))))

(define-public r-chromunity
  (let ((commit "101e30aafef23b4d8a4fba40375bf7d13a7c0b11")
        (revision "2"))
    (package
      (name "r-chromunity")
      (version (git-version "0.0.2" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/mskilab/chromunity")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1fq6l9xn6zl1nqr6ny2575r3iqmhf8dgsd7rxaglm6a7fy3r8zx7"))))
      (properties `((upstream-name . "chromunity")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'disable-bad-tests
             (lambda _
               ;; This test operates on a NULL object.
               (substitute* "tests/testthat/test_chromunity.R"
                 ((".*synergy.*" m)
                  (string-append m "skip('guix')\n"))))))))
      (propagated-inputs
       (list r-arrow
             r-biocgenerics
             r-bsgenome-hsapiens-ucsc-hg38
             r-data-table
             r-gchain
             r-genomicranges
             r-gutils
             r-igraph
             r-magrittr
             r-mass
             r-matrix
             r-pbmcapply
             r-plyr
             r-r6
             r-zoo))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/mskilab/chromunity")
      (synopsis "Discovery of communities in Pore-C concatemers")
      (description "This is a package for the discovery of communities in
Pore-C concatemers.")
      (license license:gpl3))))

(define-public r-cmdstanr
  (let ((commit "a45d4f7d686aa6b57ce25f342a71eea79507f01c")
        (revision "1"))
    (package
      (name "r-cmdstanr")
      (version (git-version "0.8.1.9000" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/stan-dev/cmdstanr")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "01vyh3sy704rs7yvaka2hp6vld0mdl7hbrs20dpbwidgm4ksrvsi"))))
      (properties `((upstream-name . "cmdstanr")))
      (build-system r-build-system)
      (propagated-inputs (list r-checkmate
                               r-data-table
                               r-jsonlite
                               r-posterior
                               r-processx
                               r-r6
                               r-rlang
                               r-withr))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/stan-dev/cmdstanr")
      (synopsis "R interface to 'CmdStan'")
      (description
       "This package provides a lightweight interface to Stan
<https://mc-stan.org>.  The @code{CmdStanR} interface is an alternative
to RStan that calls the command line interface for compilation and
running algorithms instead of interfacing with C++ via Rcpp'.")
      (license license:bsd-3))))

(define-public r-dnamcrosshyb
  ;; There aren't any releases.
  (let ((commit "fe8acb33667e81f00dcb84e0fa75c87ab2db5d8f")
        (revision "1"))
    (package
      (name "r-dnamcrosshyb")
      (version (git-version "0.0.0.9000" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/pjhop/DNAmCrosshyb")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "12j1xsiqpvny5rp23z1az0k4cj5ajbcwkg65z00s16vywi2rx6nb"))))
      (properties `((upstream-name . "DNAmCrosshyb")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biocgenerics
             r-biocparallel
             r-biostrings
             r-bsgenome-hsapiens-ucsc-hg19-masked
             r-bsgenome-hsapiens-ucsc-hg38-masked
             r-dplyr
             r-genomicranges
             r-ggplot2
             r-iranges
             r-magrittr
             r-minfi
             r-purrr
             r-s4vectors
             r-shiny
             r-stringi
             r-stringr
             r-tibble
             r-tidyr
             r-watermelon))
      (home-page "https://github.com/pjhop/DNAmCrosshyb")
      (synopsis "DNAmCrosshyb")
      (description
       "This package provides helper functions to detect cross-hybridization
on Illumina DNAm arrays.")
      (license license:gpl3))))

(define-public r-doubletcollection
  (let ((commit "c0d62f1853942ee6a087eaf7b000d9e4261e2dfd")
        (revision "1"))
    (package
      (name "r-doubletcollection")
      (version (git-version "1.1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/xnnba1984/DoubletCollection")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "02cvibyc2nwc4037ramm5cskjwyrb9ib9hkrfhmvhbslkn5ixz1v"))))
      (properties `((upstream-name . "DoubletCollection")))
      (build-system r-build-system)
      ;; There are no tests.
      (arguments (list #:tests? #false))
      (propagated-inputs (list r-biocgenerics
                               r-doubletfinder
                               r-gam
                               r-ggplot2
                               r-ggthemes
                               r-mast
                               r-mclust
                               r-prroc
                               r-reticulate
                               r-scales
                               r-scdblfinder
                               r-scds
                               r-seurat
                               r-singlecellexperiment
                               r-slingshot
                               r-summarizedexperiment
                               python
                               python-scrublet
                               python-doubletdetection))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/xnnba1984/DoubletCollection")
      (synopsis "Tool for finding doublets in scRNA-seq data")
      (description
       "This is an R package that integrates the installation of
doublet-detection methods.  In addition, this tool is used for execution and
benchmark of those eight mentioned methods.")
      (license license:gpl3+))))

(define-public r-plsdabatch
  (let ((commit "4aadf3a99709afae462db310386b6cf5db20088c")
        (revision "1"))
    (package
      (name "r-plsdabatch")
      (version (git-version "0.2.3" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/EvaYiwenWang/PLSDAbatch")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "047l923lq2ji7rwybh9b9zkblzvvhkpli5gb2x8g2q9f2n5022nr"))))
      (properties `((upstream-name . "PLSDAbatch")))
      (build-system r-build-system)
      (propagated-inputs (list r-ggplot2
                               r-ggpubr
                               r-gridextra
                               r-lmertest
                               r-mixomics
                               r-mvtnorm
                               r-performance
                               r-rdpack
                               r-scales))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/EvaYiwenWang/PLSDAbatch")
      (synopsis "PLSDA-batch")
      (description
       "This package provides a new batch effect correction method based on
Projection to Latent Structures Discriminant Analysis named “PLSDA-batch” to
correct data prior to any downstream analysis.  PLSDA-batch estimates latent
components related to treatment and batch effects to remove batch variation.
The method is multivariate, non-parametric and performs dimension reduction.
Combined with centered log ratio transformation for addressing uneven library
sizes and compositional structure, PLSDA-batch addresses all characteristics
of microbiome data that existing correction methods have ignored so far.")
      (license license:gpl3))))

(define-public r-psupertime
  (let ((commit "73825a28d3bd9bc881c15ee0c4c218eec1c9c207")
        (revision "1"))
    (package
      (name "r-psupertime")
      (version (git-version "0.2.6" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/wmacnair/psupertime")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "00h1r3ffz6m9dwcgkvyki8405b059qn6mnjsd8d76a1rabaf2vfh"))))
      (properties `((upstream-name . "psupertime")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-cowplot
             r-data-table
             r-fastcluster
             r-forcats
             r-ggplot2
             r-glmnet
             r-knitr
             r-matrix
             r-rcolorbrewer
             r-scales
             r-scran
             r-singlecellexperiment
             r-stringr
             r-summarizedexperiment
             r-topgo))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/wmacnair/psupertime")
      (synopsis
       "Psupertime is supervised pseudotime for single cell RNAseq data")
      (description
       "Psupertime is supervised pseudotime for single cell RNAseq data.  It
uses single cell RNAseq data, where the cells have a known ordering.  This
ordering helps to identify a small number of genes which place cells in that
known order.  It can be used for discovery of relevant genes, for
identification of subpopulations, and characterization of further unknown or
differently labelled data.")
      (license license:gpl3))))

(define-public r-pando
  (package
    (name "r-pando")
    (version "1.0.5")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/quadbiolab/Pando")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "04kr1b28p5j7h48g32cldkg87xcmxnmd4kspygkfs7a4amihpi66"))))
    (properties `((upstream-name . "Pando")))
    (build-system r-build-system)
    (arguments
     (list
      ;; The test file data/test_seurat.rds is not included.
      #:tests? #false
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'loosen-requirements
           (lambda _
             (substitute* "DESCRIPTION"
               ((" \\(==.*,") ",")))))))
    (propagated-inputs
     (list r-bayestestr
           r-foreach
           r-genomicranges
           r-ggplot2
           r-ggpointdensity
           r-ggraph
           r-glmnetutils
           r-grr
           r-iranges
           r-irlba
           r-matrix
           r-matrixgenerics
           r-motifmatchr
           r-pals
           r-patchwork
           r-seurat
           r-signac
           r-sparsematrixstats
           r-tfbstools
           r-tidygraph
           r-tidyverse
           r-uwot))
    (native-inputs
     (list r-bsgenome-hsapiens-ucsc-hg38
           r-devtools
           r-doparallel
           r-ensdb-hsapiens-v86
           r-knitr))
    (home-page "https://github.com/quadbiolab/Pando")
    (synopsis "Infer regulomes from multi-modal single-cell genomics data")
    (description
     "Pando leverages multi-modal single-cell measurements to infer gene
regulatory networks using a flexible linear model-based framework.  By
modeling the relationship between TF-binding site pairs with the expression of
target genes, Pando simultaneously infers gene modules and sets of regulatory
regions for each transcription factor.")
    (license license:expat)))

(define-public r-premessa
  (let ((commit "68b42bb984637d0f3ad6a0ecc83e9278994afc85")
        (revision "1"))
    (package
      (name "r-premessa")
      (version (git-version "0.3.4" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/ParkerICI/premessa")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1l0q431zk0lvg22130nx84gdqi7cpl05yah4am63lbx6m4c769pb"))
                (snippet
                 '(delete-file "inst/normalizer_shinyGUI/www/d3.min.js"))))
      (properties `((upstream-name . "premessa")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'process-javascript
             (lambda* (#:key inputs #:allow-other-keys)
               (with-directory-excursion "inst/normalizer_shinyGUI/www/"
                 (invoke "esbuild" (assoc-ref inputs "d3.v4.js")
                         "--minify" "--outfile=d3.min.js")))))))
      (propagated-inputs
       (list r-data-table
             r-flowcore
             r-ggplot2
             r-gridextra
             r-hexbin
             r-jsonlite
             r-reshape
             r-rhandsontable
             r-shiny
             r-shinyjqui))
      (native-inputs
       `(("esbuild" ,esbuild)
         ("d3.v4.js"
          ,(origin
             (method url-fetch)
             (uri "https://d3js.org/d3.v4.js")
             (sha256
              (base32
               "0y7byf6kcinfz9ac59jxc4v6kppdazmnyqfav0dm4h550fzfqqlg"))))))
      (home-page "https://github.com/ParkerICI/premessa")
      (synopsis "Pre-processing of flow and mass cytometry data")
      (description
       "This is an R package for pre-processing of flow and mass cytometry
data.  This package includes panel editing or renaming for FCS files,
bead-based normalization and debarcoding.")
      (license license:gpl3))))

(define-public r-projectils
  (let ((commit "cc73b97471b4b6eea11ce779b5c4a7dc5c3e1709")
        (revision "1"))
    (package
      (name "r-projectils")
      (version (git-version "3.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/carmonalab/ProjecTILs")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0dpzvbhhb9andnj7angpj32cgkwd6rs6qgpl6i21pqzcn6vqqhqw"))))
      (properties `((upstream-name . "ProjecTILs")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biocneighbors
             r-biocparallel
             r-dplyr
             r-ggplot2
             r-matrix
             r-patchwork
             r-pheatmap
             r-pracma
             r-purrr
             r-rcolorbrewer
             r-reshape2
             r-scales
             r-scgate
             r-seurat
             r-seuratobject
             r-stacas
             r-ucell
             r-umap
             r-uwot))
      (home-page "https://github.com/carmonalab/ProjecTILs")
      (synopsis "Reference-based analysis of scRNA-seq data")
      (description
       "This package implements methods to project single-cell RNA-seq data
onto a reference atlas, enabling interpretation of unknown cell transcriptomic
states in the the context of known, reference states.")
      (license license:gpl3))))

(define-public r-presto
  (let ((commit "7636b3d0465c468c35853f82f1717d3a64b3c8f6")
        (revision "1"))
    (package
      (name "r-presto")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/immunogenomics/presto")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "07giligbadv3ap0p4hzsviqpgiipkqcq7x6p0g8lmb8xxpjz3y29"))))
      (properties
       '((upstream-name . "presto")
         (updater-extra-native-inputs . ("r-broom"))))
      (build-system r-build-system)
      (propagated-inputs
       (list r-data-table
             r-dplyr
             r-matrix
             r-purrr
             r-rcpp
             r-rcpparmadillo
             r-rlang
             r-tibble
             r-tidyr))
      (native-inputs (list r-broom r-knitr r-testthat))
      (home-page "https://github.com/immunogenomics/presto")
      (synopsis "Fast Functions for Differential Expression using Wilcox and AUC")
      (description "This package performs a fast Wilcoxon rank sum test and
auROC analysis.")
      (license license:gpl3))))

(define-public r-sccustomize
  (let ((commit "397374590dae2ccc0c560897dcd1ce4382c18798")
        (revision "1"))
    (package
      (name "r-sccustomize")
      (version (git-version "2.0.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/samuel-marsh/scCustomize")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "11bafm0mlck27fqd8brz80pxb8dc5q0aqbp8zv0s9sx97njp7wsl"))))
      (properties `((upstream-name . "scCustomize")))
      (build-system r-build-system)
      (propagated-inputs (list r-circlize
                               r-colorway
                               r-cowplot
                               r-data-table
                               r-dittoseq
                               r-dplyr
                               r-forcats
                               r-ggbeeswarm
                               r-ggplot2
                               r-ggprism
                               r-ggpubr
                               r-ggrastr
                               r-ggrepel
                               r-glue
                               r-janitor
                               r-magrittr
                               r-matrix
                               r-paletteer
                               r-patchwork
                               r-pbapply
                               r-purrr
                               r-remotes
                               r-scales
                               r-scattermore
                               r-seurat
                               r-seuratobject
                               r-stringi
                               r-stringr
                               r-tibble
                               r-tidyr
                               r-tidyselect
                               r-viridis))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/samuel-marsh/scCustomize")
      (synopsis "Custom visualization and analyses of single-cell sequencing")
      (description
       "This is a collection of functions created and/or curated to aid in the
visualization and analysis of single-cell data using R.")
      (license license:gpl3+))))

(define-public r-markdownhelpers
  (let ((commit "793372d28ebed607cc1d35f909a1caedb2b41ffe")
        (revision "1"))
    (package
      (name "r-markdownhelpers")
      (version (git-version "0.2.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/MarkdownHelpers")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1d18s2ydhfjm3hjkxz42dirhwrrv792m6mvkmypallaa2qnwrmkg"))))
      (properties `((upstream-name . "MarkdownHelpers")))
      (build-system r-build-system)
      (propagated-inputs (list r-devtools r-stringendo r-usethis))
      (home-page "https://github.com/vertesy/MarkdownHelpers")
      (synopsis "Helper functions for MarkdownReports and ggExpress")
      (description
       "This package provides a set of R functions to parse markdown and other
generic helpers.")
      (license license:gpl3))))

(define-public r-markdownreports
  (let ((commit "3ba1103e3ddc6df3a0c090eb884f5e65c461eb31")
        (revision "1"))
    (package
      (name "r-markdownreports")
      (version (git-version "4.5.9" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/MarkdownReports")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1xmasdb630b6nvxi5m1i8pyxiy49nxpzyxf9h8spdppx92rhdkc8"))))
      (properties `((upstream-name . "MarkdownReports")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-clipr
             r-codeandroll2
             r-colorramps
             r-devtools
             r-gplots
             r-markdownhelpers
             r-rcolorbrewer
             r-readwriter
             r-sessioninfo
             r-sm
             r-stringendo
             r-venndiagram
             r-vioplot))
      (home-page "https://github.com/vertesy/MarkdownReports")
      (synopsis "Tool for generating cientific figures and reports")
      (description
       "This is a set of R functions that allows you to generate precise
figures.  This tool will create clean markdown reports about what you just
discovered.")
      (license license:gpl3))))

(define-public r-metacell
  (let ((commit "d6a6926d103ee0cb34a611c753572429c94a53d9")
        (revision "1"))
    (package
      (name "r-metacell")
      (version (git-version "0.3.41" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/tanaylab/metacell/")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0zrsckr3y35x37mj3ibm6scyqx925s84wzrz1i42fnm9n3msc265"))))
      (properties `((upstream-name . "metacell")))
      (build-system r-build-system)
      (propagated-inputs (list r-cluster
                               r-cowplot
                               r-data-table
                               r-dbscan
                               r-domc
                               r-dplyr
                               r-entropy
                               r-ggplot2
                               r-graph
                               r-igraph
                               r-kernsmooth
                               r-magrittr
                               r-matrix
                               r-matrixstats
                               r-pdist
                               r-pheatmap
                               r-plyr
                               r-rcolorbrewer
                               r-rcurl
                               r-rgraphviz
                               r-slam
                               r-singlecellexperiment
                               r-svglite
                               r-tgconfig
                               r-tgstat
                               r-tgutil
                               r-tidyr
                               r-umap
                               r-umap
                               r-zoo))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/tanaylab/metacell/")
      (synopsis "Meta cell analysis for single cell RNA-seq data")
      (description
       "This package facilitates the analysis of single-cell RNA-seq UMI matrices.
It does this by computing partitions of a cell similarity graph into
small homogeneous groups of cells, which are defined as metacells (MCs).
The derived MCs are then used for building different representations of
the data, allowing matrix or 2D graph visualization forming a basis for
analysis of cell types, subtypes, transcriptional gradients,cell-cycle
variation, gene modules and their regulatory models and more.")
      (license license:expat))))

(define-public r-metadeconfoundr
  ;; There are some relevant updates after the release of version 0.3.0.
  (let ((commit "90aec0226c5128bfcbbc08903452eff460d21424")
        (revision "1"))
    (package
      (name "r-metadeconfoundr")
      (version (git-version "0.3.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/TillBirkner/metadeconfoundR")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0zkqar27p5qwq46xbxsw5x1pl50xbkgqiizw5bydlyhwb0ga2f3h"))))
      (properties `((upstream-name . "metadeconfoundR")))
      (build-system r-build-system)
      (propagated-inputs (list r-bigmemory
                               r-detectseparation
                               r-doparallel
                               r-dosnow
                               r-foreach
                               r-futile-logger
                               r-ggplot2
                               r-lme4
                               r-lmtest
                               r-reshape2
                               r-snow))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/TillBirkner/metadeconfoundR")
      (synopsis "Check multiple covariates for potential confounding effects")
      (description
       "This package detects naive associations between omics features and
metadata in cross-sectional data-sets using non-parametric tests.  In a second
step, confounding effects between metadata associated to the same omics
feature are detected and labeled using nested post-hoc model comparison tests.
The generated output can be graphically summarized using the built-in plotting
function.")
      (license license:gpl2))))

(define-public r-music
  (let ((commit "f21fe67f5670d5e9fca0ad7550abaae3423eb59c")
        (revision "2"))
    (package
      (name "r-music")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/xuranw/MuSiC")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "05q72sml35sw3rd0lyg7r9r3470q47x5dsjn4bpgzl99s5d76llx"))))
      (properties `((upstream-name . "MuSiC")))
      (build-system r-build-system)
      (propagated-inputs (list r-biobase
                               r-ggplot2
                               r-matrix
                               r-mcmcpack
                               r-nnls
                               r-singlecellexperiment
                               r-toast))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/xuranw/MuSiC")
      (synopsis "Multi-subject single cell deconvolution")
      (description
       "MuSiC is a deconvolution method that utilizes cross-subject scRNA-seq
to estimate cell type proportions in bulk RNA-seq data.")
      (license license:gpl3+))))

(define-public r-sleuth
  (package
    (name "r-sleuth")
    (version "0.30.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/pachterlab/sleuth")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "09xgc7r6iisjkk0c0wn0q56zy0aph386kphwixfzq4422y7vlqci"))))
    (properties `((upstream-name . "sleuth")))
    (build-system r-build-system)
    ;; FIXME: there are 10 unknown failures.
    (arguments (list #:tests? #false))
    (propagated-inputs (list r-aggregation
                             r-data-table
                             r-dplyr
                             r-ggplot2
                             r-lazyeval
                             r-matrixstats
                             r-pheatmap
                             r-reshape2
                             r-rhdf5
                             r-shiny
                             r-tidyr))
    (native-inputs (list r-knitr r-testthat))
    (home-page "https://github.com/pachterlab/sleuth")
    (synopsis "Tools for investigating RNA-Seq")
    (description
     "Sleuth is a program for differential analysis of RNA-Seq data.
It makes use of quantification uncertainty estimates obtained via Kallisto for
accurate differential analysis of isoforms or genes, allows testing in the
context of experiments with complex designs, and supports interactive
exploratory data analysis via sleuth live.")
    (license license:gpl3)))

(define-public r-snapatac
  (package
    (name "r-snapatac")
    (version "2.0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/r3fang/SnapATAC")
               (commit (string-append "v" version))))
        (file-name (git-file-name name version))
        (sha256
          (base32 "037jzlbl436fi7lkpq7d83i2vd1crnrik3vac2x6xj75dbikb2av"))))
    (properties `((upstream-name . "SnapATAC")))
    (build-system r-build-system)
    (propagated-inputs
      (list r-bigmemory
            r-doparallel
            r-dosnow
            r-edger
            r-foreach
            r-genomicranges
            r-igraph
            r-iranges
            r-irlba
            r-matrix
            r-plyr
            r-plot3d
            r-rann
            r-raster
            r-rcolorbrewer
            r-rhdf5
            r-rtsne
            r-scales
            r-viridis))
    (home-page "https://github.com/r3fang/SnapATAC")
    (synopsis "Single nucleus analysis package for ATAC-Seq")
    (description
      "This package provides a fast and accurate analysis toolkit for single
cell ATAC-seq (Assay for transposase-accessible chromatin using sequencing).
Single cell ATAC-seq can resolve the heterogeneity of a complex tissue and
reveal cell-type specific regulatory landscapes.  However, the exceeding data
sparsity has posed unique challenges for the data analysis.  This package
@code{r-snapatac} is an end-to-end bioinformatics pipeline for analyzing large-
scale single cell ATAC-seq data which includes quality control, normalization,
clustering analysis, differential analysis, motif inference and exploration of
single cell ATAC-seq sequencing data.")
    (license license:gpl3)))

(define-public r-tsis
  (let ((commit "24460298fbe1d26e4da390f6e4f3d4d9d62334dc")
        (revision "1"))
    (package
      (name "r-tsis")
      (version (git-version "0.2.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/wyguo/TSIS")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "17c8i25iwhldvs3c51m0wny40iffm8szzijalpwrxhzbv0xa94rb"))))
      (properties `((upstream-name . "TSIS")))
      (build-system r-build-system)
      (native-inputs (list r-knitr))
      (home-page "https://github.com/wyguo/TSIS")
      (synopsis "Time-series isoform switch of alternative splicing")
      (description
       "TSIS is used for detecting transcript isoform switches in time-series
data.  Transcript isoform switches occur when a pair of alternatively spliced
isoforms reverse the order of their relative expression levels.  TSIS
characterizes the transcript switch by defining the isoform switch time-points
for any pair of transcript isoforms within a gene.  In addition, this tool
describes the switch using five different features or metrics.  Also it
filters the results with user’s specifications and visualizes the results
using different plots for the user to examine further details of the
switches.")
      (license license:gpl3))))

(define-public r-umi4cpackage
  (let ((commit "88b07d896a137418ba6c31c2474b9dbe1d86fc20")
        (revision "1"))
    (package
      (name "r-umi4cpackage")
      (version (git-version "0.0.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/tanaylab/umi4cpackage")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0bjzamdw2lcfhlbzc0vdva87c3wwnij8jsvnrpx4wyyxvpcz13m5"))))
      (properties `((upstream-name . "umi4cPackage")))
      (build-system r-build-system)
      (arguments
       `(#:phases
         (modify-phases %standard-phases
           (add-after 'unpack 'fix-references
             (lambda _
               (substitute* "inst/conf/paths.conf"
                 (("TG3C\\.bowtie2_bin=.*")
                  (string-append "TG3C.bowtie2_bin="
                                 (which "bowtie2") "\n")))
               (substitute* "inst/perl/map3c/TG3C/import3C.pl"
                 (("\"perl")
                  (string-append "\"" (which "perl")))))))))
      (inputs
       (list perl bowtie))
      (propagated-inputs
       (list r-misha r-zoo))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/tanaylab/umi4cpackage")
      (synopsis "Processing and analysis of UMI-4C contact profiles")
      (description "This is a package that lets you process UMI-4C data from
scratch to produce nice plots.")
      (license license:expat))))

(define-public r-shinycell
  (let ((commit
         "aecbd56e66802f28e397f5ae1f19403aadd12163")
        (revision "1"))
    (package
      (name "r-shinycell")
      (version (git-version "2.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/SGDDNB/ShinyCell")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "13jn2ikmvljnzayk485g1mmq5abcp9m1b8n1djdb1agmn83zaki5"))))
      (properties `((upstream-name . "ShinyCell")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-data-table
             r-ggplot2
             r-glue
             r-gridextra
             r-hdf5r
             r-matrix
             r-r-utils
             r-rcolorbrewer
             r-readr
             r-reticulate))
      (home-page "https://github.com/SGDDNB/ShinyCell")
      (synopsis "Shiny interactive web apps for single-cell data")
      (description
       "This package provides Shiny apps for interactive exploration of
single-cell data.")
      (license license:gpl3))))

(define-public r-archr
  (let ((commit "c61b0645d1482f80dcc24e25fbd915128c1b2500")
        (revision "1"))
    (package
      (name "r-archr")
      (version (git-version "1.0.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/GreenleafLab/ArchR")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0sgdfd8iwgj8cssj2zr3gmshg8nv54q6dd8asjf99i39qkni7p9i"))))
      (properties `((upstream-name . "ArchR")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biocgenerics
             r-biostrings
             r-chromvar
             r-complexheatmap
             r-data-table
             r-genomicranges
             r-ggplot2
             r-ggrepel
             r-gridextra
             r-gtable
             r-gtools
             r-magrittr
             r-matrix
             r-matrixstats
             r-motifmatchr
             r-nabor
             r-plyr
             r-rcpp
             r-rhdf5
             r-rsamtools
             r-s4vectors
             r-stringr
             r-summarizedexperiment
             r-uwot))
      (home-page "https://github.com/GreenleafLab/ArchR")
      (synopsis "Analyze single-cell regulatory chromatin in R")
      (description
       "This package is designed to streamline scATAC analyses in R.")
      (license license:gpl2+))))

(define-public r-azimuth
  (let ((commit "243ee5db80fcbffa3452c944254a325a3da2ef9e")
        (revision "1"))
    (package
      (name "r-azimuth")
      (version (git-version "0.5.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/satijalab/azimuth")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0vs4813kf2cv2j1z800ihwk62bw4dgmwgkd47rln58581chv39zm"))))
      (properties `((upstream-name . "Azimuth")))
      (build-system r-build-system)
      (propagated-inputs (list r-bsgenome-hsapiens-ucsc-hg38
                               r-dt
                               r-ensdb-hsapiens-v86
                               r-future
                               r-ggplot2
                               r-glmgampoi
                               r-googlesheets4
                               r-hdf5r
                               r-htmltools
                               r-httr
                               r-jaspar2020
                               r-jsonlite
                               r-matrix
                               r-patchwork
                               r-plotly
                               r-presto
                               r-rcpp
                               r-rlang
                               r-scales
                               r-seurat
                               r-seuratdata
                               r-seuratdisk
                               r-seuratobject
                               r-shiny
                               r-shinybs
                               r-shinydashboard
                               r-shinyjs
                               r-signac
                               r-stringr
                               r-tfbstools
                               r-withr))
      (home-page "https://github.com/satijalab/azimuth")
      (synopsis
       "Shiny app showcasing a single-cell data query-reference mapping algorithm")
      (description "Azimuth utilizes an annotated reference dataset.  It
automates the processing, analysis, and interpretation.  This applies
specifically to new single-cell RNA-seq or ATAC-seq experiments.  Azimuth
leverages a reference-based mapping pipeline that inputs accounts matrix and
performs normalization, visualization, cell annotation, and differential
expression.")
      (license license:gpl3))))

(define-public r-icellnet
  ;; This is an arbitrary commit because there is no tag for 2.2.1.
  (let ((commit "e10ee4ae5feaf8130764dcb0736957d2a90b828c")
        (revision "1"))
    (package
      (name "r-icellnet")
      (version (git-version "2.2.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/soumelis-lab/ICELLNET")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0bszxw9zcsd1gxhg9q89jfzi95266wfjn74hp7z8yb99cr3j53k4"))))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'enter-dir
             (lambda _ (chdir "icellnet"))))))
      (propagated-inputs
       (list r-annotationdbi
             r-complexheatmap
             r-circlize
             r-data-table
             r-dplyr
             r-ggplot2
             r-hgu133plus2-db
             r-jetset
             r-psych
             r-reshape2
             r-rlist))
      (home-page "https://github.com/soumelis-lab/ICELLNET")
      (synopsis "Transcriptomic-based framework to dissect cell communication")
      (description "This package provides a a transcriptomic-based framework
to dissect cell communication in a global manner.  It integrates an original
expert-curated database of ligand-receptor interactions taking into account
multiple subunits expression.  Based on transcriptomic profiles (gene
expression), this package computes communication scores between cells
and provides several visualization modes that can be helpful to dig into
cell-cell interaction mechanism and extend biological knowledge.")
      (license license:gpl3))))

(define-public r-scde
  (package
    (name "r-scde")
    (version "1.99.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/hms-dbmi/scde")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10na2gyka24mszdxf92wz9h2c13hdf1ww30c68gfsw53lvvhhhxb"))))
    (build-system r-build-system)
    (propagated-inputs
     (list r-rcpp
           r-rcpparmadillo
           r-mgcv
           r-rook
           r-rjson
           r-cairo
           r-rcolorbrewer
           r-edger
           r-quantreg
           r-nnet
           r-rmtstat
           r-extremes
           r-pcamethods
           r-biocparallel
           r-flexmix))
    (home-page "https://hms-dbmi.github.io/scde/")
    (synopsis "R package for analyzing single-cell RNA-seq data")
    (description "The SCDE package implements a set of statistical methods for
analyzing single-cell RNA-seq data.  SCDE fits individual error models for
single-cell RNA-seq measurements.  These models can then be used for
assessment of differential expression between groups of cells, as well as
other types of analysis.  The SCDE package also contains the pagoda framework
which applies pathway and gene set overdispersion analysis to identify aspects
of transcriptional heterogeneity among single cells.")
    ;; See https://github.com/hms-dbmi/scde/issues/38
    (license license:gpl2)))

(define-public r-miamiplot
  (let ((commit "beede9c5d6431b4d822aa42e064e01baeb5dd4a0")
        (revision "1"))
    (package
      (name "r-miamiplot")
      (version (git-version "1.1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/juliedwhite/miamiplot")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0wxxk1lk9jbf0imf59qp302ffasvs84idinkvzirs3dw9w3589n9"))))
      (properties `((upstream-name . "miamiplot")))
      (build-system r-build-system)
      (propagated-inputs (list r-checkmate
                               r-dplyr
                               r-ggplot2
                               r-ggrepel
                               r-gridextra
                               r-magrittr
                               r-rlang))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/juliedwhite/miamiplot")
      (synopsis "Create a ggplot2 miami plot")
      (description
       "This package generates a Miami plot with centered chromosome labels.
The output is a ggplot2 object.  Users can specify which data they want
plotted on top vs. bottom, whether to display significance line(s), what
colors to give chromosomes, and what points to label.")
      (license license:gpl2))))

(define-public r-millefy
  (package
    (name "r-millefy")
    (version "0.1.9-beta")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/yuifu/millefy")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0z2y0x99f761pxvg6n37cmnyrnj699jhjk43pvk05sa86iykgizl"))))
    (properties `((upstream-name . "millefy")))
    (build-system r-build-system)
    (propagated-inputs
     (list r-data-table
           r-destiny
           r-dplyr
           r-genomicranges
           r-iranges
           r-magrittr
           r-rsamtools
           r-rtracklayer
           r-tidyr))
    (native-inputs (list r-testthat))
    (home-page "https://github.com/yuifu/millefy")
    (synopsis "Make millefy plot with single-cell RNA-seq data")
    (description "@code{Millefy} is a tool for visualizing read coverage of
@dfn{scRNA-seq}(single-cell RNA sequencing) datasets in genomic contexts.  By
dynamically and automatically reorder single cells based on locus-specific
pseudo time, @code{Millefy} highlights cell-to-cell heterogeneity in read coverage
of scRNA-seq data.")
    (license license:expat)))

(define-public r-misha
  (package
    (name "r-misha")
    (version "4.1.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/tanaylab/misha")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0bgivx3lzjh3173jsfrhb5kvhjsn53br0n4hmyx7i3dwy2cnnp2p"))
       ;; Delete bundled executable.
       (snippet
        '(delete-file "exec/bigWigToWig"))))
    (build-system r-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'do-not-use-bundled-bigWigToWig
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "R/misha.R"
               (("get\\(\".GLIBDIR\"\\), \"/exec/bigWigToWig")
                (string-append "\""
                               (assoc-ref inputs "kentutils")
                               "/bin/bigWigToWig"))))))))
    (inputs
     (list kentutils))
    (home-page "https://github.com/tanaylab/misha")
    (synopsis "Toolkit for analysis of genomic data")
    (description "This package is intended to help users to efficiently
analyze genomic data resulting from various experiments.")
    (license license:gpl2)))

(define-public r-scseqcomm
  (let ((commit "01076e703999f1a5aa76419d821b50aebe2b777a")
        (revision "0"))
    (package
      (name "r-scseqcomm")
      (version (git-version "0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://gitlab.com/sysbiobig/scseqcomm")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "1fw5si47d6agnz5fibmp2b1sv08pbpwv1j71w57xbav9044i032q"))
         ;; Delete bundled dependency.
         (modules '((guix build utils)))
         (snippet
          '(delete-file-recursively "other_deps"))))
      (build-system r-build-system)
      (inputs
       (list r-add2ggplot
             r-chorddiag
             r-doparallel
             r-dplyr
             r-foreach
             r-gridextra
             r-ggplot2
             r-gtable
             r-htmlwidgets
             r-igraph
             r-matrix
             r-org-hs-eg-db
             r-psych
             r-rcolorbrewer
             r-rlang
             r-scico
             r-tidygraph
             r-topgo))
      (native-inputs
       (list r-knitr))
      (home-page "https://gitlab.com/sysbiobig/scseqcomm")
      (synopsis "Inter- and intra- cellular signaling from single cell RNA-seq")
      (description "This package is tools for analysing intercellular and
intracellular signaling from single cell RNA-seq (scRNA-seq) data.")
      (license license:gpl3))))

(define-public r-shaman
  (let ((commit "d6944e8ac7bd1dbd5c6cec646eafc1d19d0ca96f")
        (release "2.0")
        (revision "2"))
    (package
      (name "r-shaman")
      (version (git-version release revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/tanaylab/shaman")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "03sx138dzpfiq23j49z0m0s4j79855mrg64hpj9c83408wzphxi6"))
                (snippet
                 ;; This file will be generated.
                 '(delete-file "inst/doc/shaman-package.R"))))
      (build-system r-build-system)
      (propagated-inputs
       (list r-data-table
             r-domc
             r-ggplot2
             r-gviz
             r-misha
             r-plyr
             r-rann
             r-rcpp
             r-reshape2
             ;; For vignettes
             r-rmarkdown
             r-knitr))
      (home-page "https://github.com/tanaylab/shaman")
      (synopsis "Sampling HiC contact matrices for a-parametric normalization")
      (description "The Shaman package implements functions for
resampling Hi-C matrices in order to generate expected contact
distributions given constraints on marginal coverage and
contact-distance probability distributions.  The package also provides
support for visualizing normalized matrices and statistical analysis
of contact distributions around selected landmarks.")
      ;; Any version of the GPL
      (license license:gpl3+))))

(define-public r-centipede
  (package
    (name "r-centipede")
    (version "1.2")
    (source (origin
              (method url-fetch)
              (uri (string-append "http://download.r-forge.r-project.org/"
                                  "src/contrib/CENTIPEDE_" version ".tar.gz"))
              (sha256
               (base32
                "1hsx6qgwr0i67fhy9257zj7s0ppncph2hjgbia5nn6nfmj0ax6l9"))))
    (build-system r-build-system)
    (home-page "https://centipede.uchicago.edu/")
    (synopsis "Predict transcription factor binding sites")
    (description
     "CENTIPEDE applies a hierarchical Bayesian mixture model to infer regions
of the genome that are bound by particular transcription factors.  It starts
by identifying a set of candidate binding sites, and then aims to classify the
sites according to whether each site is bound or not bound by a transcription
factor.  CENTIPEDE is an unsupervised learning algorithm that discriminates
between two different types of motif instances using as much relevant
information as possible.")
    (license (list license:gpl2+ license:gpl3+))))

(define-public r-demultiplex
  (let ((commit "6e2a1422c8e6f418cfb271997eebc91f9195f299")
        (revision "1"))
    (package
      (name "r-demultiplex")
      (version (git-version "1.0.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/chris-mcginnis-ucsf/MULTI-seq")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "01kv88wp8vdaq07sjk0d3d1cb553mq1xqg0war81pgmg63bgi38w"))))
      (properties `((upstream-name . "deMULTIplex")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-kernsmooth r-reshape2 r-rtsne r-shortread r-stringdist))
      (home-page "https://github.com/chris-mcginnis-ucsf/MULTI-seq")
      (synopsis "MULTI-seq pre-processing and classification tools")
      (description
       "deMULTIplex is an R package for analyzing single-cell RNA sequencing
data generated with the MULTI-seq sample multiplexing method.  The package
includes software to

@enumerate
@item Convert raw MULTI-seq sample barcode library FASTQs into a sample
  barcode UMI count matrix, and
@item Classify cell barcodes into sample barcode groups.
@end enumerate
")
      (license license:cc0))))

(define-public r-demultiplex2
  (let ((commit "c1ce09e8b5784b485680cf96b90bbc12eed96907")
        (revision "2"))
    (package
      (name "r-demultiplex2")
      (version (git-version "1.0.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/Gartner-Lab/deMULTIplex2")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0fid5yh0hqg477h5xxq6c8yh8zs5jawjskzns05rndxhbzm51lpr"))))
      (properties `((upstream-name . "deMULTIplex2")))
      (build-system r-build-system)
      (propagated-inputs (list r-data-table
                               r-devtools
                               r-dplyr
                               r-ggextra
                               r-ggplot2
                               r-gridextra
                               r-magrittr
                               r-mass
                               r-matrix
                               r-rcolorbrewer
                               r-shortread
                               r-stringdist
                               r-tidyr
                               r-uwot
                               r-viridis
                               r-xvector))
      (home-page "https://github.com/Gartner-Lab/deMULTIplex2")
      (synopsis "Robust sample demultiplexing for scRNA-seq")
      (description
       "This package implements a bioinformatics algorithm for demultiplexing
multiplexed single cell datasets.  It is built on a statistical model of tag
read counts derived from the physical mechanism of tag cross-contamination.")
      (license license:cc0))))

(define-public r-demuxmix
  (let ((commit "09a7918ca6e0cd23e6bbaed2b97388bc499e248e")
        (revision "1"))
    (package
      (name "r-demuxmix")
      (version (git-version "1.1.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/huklein/demuxmix")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "03kfnns7m2447jsc3xplyv9qx8hvvdjmii7j837g3bb6smyxss96"))))
      (properties `((upstream-name . "demuxmix")))
      (build-system r-build-system)
      (propagated-inputs (list r-ggplot2 r-gridextra r-mass r-matrix))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/huklein/demuxmix")
      (synopsis
       "Demultiplexing oligo-barcoded scRNA-seq data using regression mixture models")
      (description
       "This package is used for demultiplexing single-cell sequencing
experiments of pooled cells.  These cells are labeled with barcode
oligonucleotides.  The package implements methods to fit regression mixture
models for a probabilistic classification of cells, including multiplet
detection.  Demultiplexing error rates can be estimated, and methods for
quality control are provided.")
      (license license:artistic2.0))))

(define-public gdc-client
  (package
    (name "gdc-client")
    (version "2.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/NCI-GDC/gdc-client.git")
              (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1yzd8j3p7w9x00qj6mx8kvlv253063pdnf7ixpkqmmwzbjhv072s"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'relax-requirements
            (lambda _
              (substitute* "setup.cfg"
                (("intervaltree~=3.0.2") "intervaltree")
                (("termcolor~=1.1.0") "termcolor"))))
          (add-after 'unpack 'fix-pytest-config
            (lambda _
              (substitute* "pyproject.toml"
                ((" --cov=gdc_client --cov-report term") "")))))))
    (native-inputs
     (list python-flask
           python-httmock
           python-moto
           python-pytest
           python-requests-mock
           python-setuptools))
    (inputs
     (list python-importlib-metadata
           python-intervaltree
           python-jsonschema
           python-lxml
           python-progressbar2
           python-pyyaml
           python-requests
           python-termcolor))
    (home-page "https://gdc.nci.nih.gov/access-data/gdc-data-transfer-tool")
    (synopsis "GDC data transfer tool")
    (description
     "The gdc-client provides several convenience functions over
the GDC API which provides general download/upload via HTTPS.")
    (license license:asl2.0)))

(define-public vsearch
  (package
    (name "vsearch")
    (version "2.9.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/torognes/vsearch")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0vhrpjfdf75ba04b24xknp41790cvcgwl0vgpy7qbzj5xh2521ss"))
       (patches (search-patches "vsearch-unbundle-cityhash.patch"))
       (snippet
        '(begin
           ;; Remove bundled cityhash sources.  The vsearch source is adjusted
           ;; for this in the patch.
           (delete-file "src/city.h")
           (delete-file "src/citycrc.h")
           (delete-file "src/city.cc")
           #t))))
    (build-system gnu-build-system)
    (inputs
     (list zlib bzip2 cityhash))
    (native-inputs
     (list autoconf automake))
    (synopsis "Sequence search tools for metagenomics")
    (description
     "VSEARCH supports DNA sequence searching, clustering, chimera detection,
dereplication, pairwise alignment, shuffling, subsampling, sorting and
masking.  The tool takes advantage of parallelism in the form of SIMD
vectorization as well as multiple threads to perform accurate alignments at
high speed.  VSEARCH uses an optimal global aligner (full dynamic programming
Needleman-Wunsch).")
    (home-page "https://github.com/torognes/vsearch")
    ;; vsearch uses non-portable SSE intrinsics so building fails on other
    ;; platforms.
    (supported-systems '("x86_64-linux"))
    ;; Dual licensed; also includes public domain source.
    (license (list license:gpl3 license:bsd-2))))

(define-public pairadise
  (package
    (name "pairadise")
    (version "1.0.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/Xinglab/PAIRADISE")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0ycwcqabm4zdng0a7j593g35d5yzvvwm7dyi3b8s19zdi4rjzrwd"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'chdir
           (lambda _ (chdir "pairadise")))
         (add-before 'build '2to3
           (lambda _ (invoke "2to3" "--write" "--nobackups" "."))))))
    (native-inputs (list python-setuptools python-wheel))
    (inputs (list star))
    (propagated-inputs (list python-pysam))
    (home-page "https://github.com/Xinglab/PAIRADISE")
    (synopsis "Paired replicate analysis of allelic differential splicing events")
    (description
     "PAIRADISE is a method for detecting @dfn{allele-specific alternative
splicing} (ASAS) from RNA-seq data.  Unlike conventional approaches that
detect ASAS events one sample at a time, PAIRADISE aggregates ASAS signals
across multiple individuals in a population.  By treating the two alleles of
an individual as paired, and multiple individuals sharing a heterozygous SNP
as replicates, PAIRADISE formulates ASAS detection as a statistical problem
for identifying differential alternative splicing from RNA-seq data with
paired replicates.")
    (license license:gpl3+)))

(define-public r-pairadise
  (package
    (inherit pairadise)
    (name "r-pairadise")
    (build-system r-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'chdir
           (lambda _ (chdir "pairadise/src/pairadise_model"))))))
    (inputs '())
    (propagated-inputs (list r-doparallel r-foreach r-iterators r-nloptr))
    (license license:expat)))

(define-public r-pairwiseadonis
  ;; There is no tag for version 0.4.1, nor is there a release archive.
  (let ((commit "cb190f7668a0c82c0b0853927db239e7b9ec3e83")
        (revision "1"))
    (package
      (name "r-pairwiseadonis")
      ;; The versioning scheme of this package is inconsistent, with versions
      ;; progressing from 0.21 to 0.3 and then to 0.4.1, which does not follow
      ;; a standard numerical order or convention (e.g., semantic versioning).
      (version (git-version "0.4.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/pmartinezarbizu/pairwiseAdonis")
               (commit commit)))
          (file-name (git-file-name name version))
          (sha256
           (base32 "11hl6qqmr5vns476921802y0qmb46i1jf0rf7xfkyswlh6xkcl99"))))
      (properties `((upstream-name . "pairwiseAdonis")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        ;; Move into the subdirectory containing the R package files
        '(modify-phases %standard-phases
           (add-after 'unpack 'move-to-subdir
             (lambda _ (chdir "pairwiseAdonis"))))))
      (propagated-inputs
       (list r-cluster
             r-permute
             r-vegan))
      (synopsis "Pairwise multilevel comparison using adonis")
      (description
       "This package implements two functions:
@itemize
@item @code{pairwise.adonis} is a wrapper function for multilevel pairwise
comparison using adonis2 from package vegan.  The function returns adjusted
p-values using @code{p.adjust()}.  It does not accept interaction between factors
neither strata.
@item @code{pairwise.adonis2} accepts a model formula like in adonis from vegan.
You can use interactions between factors and define strata to constrain
permutations.  For pairwise comparison a list of unique pairwise combination of
factors is produced.
@end itemize")
      (home-page "https://github.com/pmartinezarbizu/pairwiseAdonis")
      (license license:gpl2+))))

(define-public pardre
  (package
    (name "pardre")
    (version "2.2.5")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "mirror://sourceforge/pardre/ParDRe-rel"
                           version ".tar.gz"))
       (sha256
        (base32
         "105s4f8zs8hh0sc32r9p725n7idza9cj5jvp5z1m5pljjhgk3if5"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; tests require "prove"
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
               (install-file "ParDRe" bin)))))))
    (inputs
     (list openmpi-c++ zlib))
    (synopsis "Parallel tool to remove duplicate DNA reads")
    (description
     "ParDRe is a parallel tool to remove duplicate genetic sequence reads.
Duplicate reads can be seen as identical or nearly identical sequences with
some mismatches.  This tool lets users avoid the analysis of unnecessary
reads, reducing the time of subsequent procedures with the
dataset (e.g. assemblies, mappings, etc.).  The tool is implemented with MPI
in order to exploit the parallel capabilities of multicore clusters.  It is
faster than multithreaded counterparts (end of 2015) for the same number of
cores and, thanks to the message-passing technology, it can be executed on
clusters.")
    (home-page "https://sourceforge.net/projects/pardre/")
    (license license:gpl3+)))

(define-public ngshmmalign
  (package
    (name "ngshmmalign")
    (version "0.1.1")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "https://github.com/cbg-ethz/ngshmmalign/"
                           "releases/download/" version
                           "/ngshmmalign-" version ".tar.bz2"))
       (sha256
        (base32
         "0jryvlssi2r2ii1dxnx39yk6bh4yqgq010fnxrgfgbaj3ykijlzv"))))
    (build-system cmake-build-system)
    (arguments '(#:tests? #false))      ; there are none
    (inputs
     (list boost))
    (home-page "https://github.com/cbg-ethz/ngshmmalign/")
    (synopsis "Profile HMM aligner for NGS reads")
    (description
     "ngshmmalign is a profile HMM aligner for NGS reads designed particularly
for small genomes (such as those of RNA viruses like HIV-1 and HCV) that
experience substantial biological insertions and deletions.")
    (license license:gpl2+)))

(define-public prinseq
  (package
    (name "prinseq")
    (version "0.20.4")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "mirror://sourceforge/prinseq/standalone/"
                           "prinseq-lite-" version ".tar.gz"))
       (sha256
        (base32
         "0vxmzvmm67whxrqdaaamwgjk7cf0fzfs5s673jgg00kz7g70splv"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #false                  ; no check target
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (delete 'build)
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (scripts (find-files "." "prinseq.*.pl"))
                    (guile (search-input-file inputs "bin/guile")))
               (substitute* scripts
                 (("\"perl -pe")
                  (string-append "\"" (which "perl") " -pe")))
               (for-each (lambda (file)
                           (chmod file #o555)
                           (install-file file bin)
                           (wrap-script (string-append bin "/" (basename file))
                                        #:guile guile
                                        `("PERL5LIB" ":" prefix
                                          (,(getenv "PERL5LIB")))))
                         scripts)))))))
    (inputs
     (list guile-3.0 ; for wrapper scripts
           perl
           perl-cairo
           perl-data-dumper
           perl-digest-md5
           perl-getopt-long
           perl-json
           perl-statistics-pca))
    (home-page "https://prinseq.sourceforge.net/")
    (synopsis "Preprocess sequence data in FASTA or FASTQ formats")
    (description
     "PRINSEQ is a bioinformatics tool to help you preprocess your genomic or
metagenomic sequence data in FASTA or FASTQ formats.  The tool is written in
Perl and can be helpful if you want to filter, reformat, or trim your sequence
data.  It also generates basic statistics for your sequences.")
    (license license:gpl3+)))

(define-public shorah
  (package
    (name "shorah")
    (version "1.99.2")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "https://github.com/cbg-ethz/shorah"
                           "/releases/download/v" version
                           "/shorah-" version ".tar.xz"))
       (sha256
        (base32
         "158dir9qcqspknlnyfr9zwk41x48nrh5wcg10k2grh9cidp9daiq"))))
    (build-system gnu-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'fix-test-wrapper
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
               (substitute* "examples/run_end2end_test"
                 (("\\$\\{interpreter\\} ../\\$\\{testscript\\}")
                  (string-append bin "/${testscript}"))))))
         (delete 'check)
         (add-after 'install 'wrap-programs
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (site (string-append
                           out "/lib/python"
                           ,(version-major+minor
                             (package-version python))
                           "/site-packages"))
                    (pythonpath (getenv "GUIX_PYTHONPATH"))
                    (script (string-append out "/bin/shorah")))
               (chmod script #o555)
               (wrap-program script `("GUIX_PYTHONPATH" ":" prefix (,site ,pythonpath))))))
         (add-after 'wrap-programs 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (invoke "make" "check")))))))
    (inputs
     (list bash-minimal
           boost
           htslib
           python
           python-biopython
           python-numpy
           zlib))
    (native-inputs
     (list pkg-config))
    (home-page "")
    (synopsis "Short reads assembly into haplotypes")
    (description
     "ShoRAH is a project for the analysis of next generation sequencing data.
It is designed to analyse genetically heterogeneous samples.  Its tools
provide error correction, haplotype reconstruction and estimation of the
frequency of the different genetic variants present in a mixed sample.")
    (license license:gpl3+)))

(define-public sickle
  (package
    (name "sickle")
    (version "1.33")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/najoshi/sickle")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1bnq480lpylq9sfsa1y71b4qz0ipi2zjnp8ds48fh10ijlwmwmdc"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;There are no tests
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (replace 'install
            (lambda _
              (install-file "sickle" (string-append #$output "/bin")))))))
    (propagated-inputs (list zlib))
    (home-page "https://github.com/najoshi/sickle")
    (synopsis "Adaptive trimming tool for FASTQ files using quality")
    (description
     "Sickle is a tool that trims reads based on quality and length
thresholds.  It uses sliding windows to detect low-quality bases at the 3'-end
and high-quality bases at the 5'-end.  Additionally, it discards reads based
on the length threshold.")
    (license license:expat)))

(define-public ruby-bio-kseq
  (package
    (name "ruby-bio-kseq")
    (version "0.0.2")
    (source
     (origin
       (method url-fetch)
       (uri (rubygems-uri "bio-kseq" version))
       (sha256
        (base32
         "1xyaha46khb5jc6wzkbf7040jagac49jbimn0vcrzid0j8jdikrz"))))
    (build-system ruby-build-system)
    (arguments
     `(#:test-target "spec"))
    (native-inputs
     (list bundler ruby-rspec ruby-rake-compiler))
    (inputs
     (list zlib))
    (synopsis "Ruby bindings for the kseq.h FASTA/Q parser")
    (description
     "@code{Bio::Kseq} provides ruby bindings to the @code{kseq.h} FASTA and
FASTQ parsing code.  It provides a fast iterator over sequences and their
quality scores.")
    (home-page "https://github.com/gusevfe/bio-kseq")
    (license license:expat)))

(define-public bio-locus
  (package
    (name "bio-locus")
    (version "0.0.7")
    (source
     (origin
       (method url-fetch)
       (uri (rubygems-uri "bio-locus" version))
       (sha256
        (base32
         "02vmrxyimkj9sahsp4zhfhnmbvz6dbbqz1y01vglf8cbwvkajfl0"))))
    (build-system ruby-build-system)
    (native-inputs
     (list ruby-rspec))
    (synopsis "Tool for fast querying of genome locations")
    (description
     "Bio-locus is a tabix-like tool for fast querying of genome
locations.  Many file formats in bioinformatics contain records that
start with a chromosome name and a position for a SNP, or a start-end
position for indels.  Bio-locus allows users to store this chr+pos or
chr+pos+alt information in a database.")
    (home-page "https://github.com/pjotrp/bio-locus")
    (license license:expat)))

(define-public bio-blastxmlparser
  (package
    (name "bio-blastxmlparser")
    (version "2.0.4")
    (source (origin
              (method url-fetch)
              (uri (rubygems-uri "bio-blastxmlparser" version))
              (sha256
               (base32
                "1wf4qygcmdjgcqm6flmvsagfr1gs9lf63mj32qv3z1f481zc5692"))))
    (build-system ruby-build-system)
    (propagated-inputs
     (list ruby-bio-logger ruby-nokogiri))
    (inputs
     (list ruby-rspec))
    (synopsis "Fast big data BLAST XML parser and library")
    (description
     "Very fast parallel big-data BLAST XML file parser which can be used as
command line utility.  Use blastxmlparser to: Parse BLAST XML; filter output;
generate FASTA, JSON, YAML, RDF, JSON-LD, HTML, CSV, tabular output etc.")
    (home-page "https://github.com/pjotrp/blastxmlparser")
    (license license:expat)))

(define-public bioruby
  (package
    (name "bioruby")
    (version "2.0.4")
    (source
     (origin
       (method url-fetch)
       (uri (rubygems-uri "bio" version))
       (sha256
        (base32
         "08aknxk2ingwscwfqpw5vnax6jpk1sxfaialravladb63hcl8dx9"))))
    (build-system ruby-build-system)
    (propagated-inputs
     (list ruby-libxml))
    (native-inputs
     (list which))  ; required for test phase
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'patch-test-command
            (lambda _
              (substitute* '("test/functional/bio/test_command.rb")
                (("/bin/sh") (which "sh")))
              (substitute* '("test/functional/bio/test_command.rb")
                (("/bin/ls") (which "ls")))
              (substitute* '("test/functional/bio/test_command.rb")
                (("which") (which "which")))
              (substitute* '("test/functional/bio/test_command.rb"
                             "test/data/command/echoarg2.sh")
                (("/bin/echo") (which "echo"))))))))
    (synopsis "Ruby library, shell and utilities for bioinformatics")
    (description "BioRuby comes with a comprehensive set of Ruby development
tools and libraries for bioinformatics and molecular biology.  BioRuby has
components for sequence analysis, pathway analysis, protein modelling and
phylogenetic analysis; it supports many widely used data formats and provides
easy access to databases, external programs and public web services, including
BLAST, KEGG, GenBank, MEDLINE and GO.")
    (home-page "http://bioruby.org/")
    ;; Code is released under Ruby license, except for setup
    ;; (LGPLv2.1+) and scripts in samples (which have GPL2 and GPL2+)
    (license (list license:ruby license:lgpl2.1+ license:gpl2+ ))))

(define-public centrifuge
  (package
    (name "centrifuge")
    (version "1.0.4")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/DaehwanKimLab/centrifuge.git")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "167610gbz1rrh6ir3j7jcmhzg3x5msn7x7a3dpv7wmwdndnnqvg0"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #false                   ; no check target
      #:make-flags
      #~(list (string-append "prefix=" #$output))
      #:phases
      '(modify-phases %standard-phases
         (delete 'configure))))
    (inputs (list python-wrapper))
    (native-inputs
     (list pandoc perl                  ;for documentation
           which))
    (home-page "https://github.com/DaehwanKimLab/centrifuge/")
    (synopsis "Classifier for metagenomic sequences")
    (description "Centrifuge is a microbial classification engine that enables
rapid, accurate and sensitive labeling of reads and quantification of species
on desktop computers.  The system uses an indexing scheme based on the
@dfn{Burrows-Wheeler transform} (BWT) and the @dfn{Ferragina-Manzini} (FM)
index, optimized specifically for the metagenomic classification problem.
Centrifuge requires a relatively small index (4.7 GB for all complete
bacterial and viral genomes plus the human genome) and classifies sequences at
very high speed, allowing it to process the millions of reads from a typical
high-throughput DNA sequencing run within a few minutes.")
    (license license:gpl3+)))

(define-public bio-vcf
  (package
    (name "bio-vcf")
    (version "0.9.5")
    (source
      (origin
        (method url-fetch)
        (uri (rubygems-uri "bio-vcf" version))
        (sha256
         (base32
          "1glw5pn9s8z13spxk6yyfqaz80n9lga67f33w35nkpq9dwi2vg6g"))))
    (build-system ruby-build-system)
    (native-inputs
     (list ruby-cucumber))
    (synopsis "Smart VCF parser DSL")
    (description
     "Bio-vcf provides a @acronym{DSL, domain specific language} for processing
the VCF format.  Record named fields can be queried with regular expressions.
Bio-vcf is a new generation VCF parser, filter and converter.  Bio-vcf is not
only very fast for genome-wide (WGS) data, it also comes with a filtering,
evaluation and rewrite language and can output any type of textual data,
including VCF header and contents in RDF and JSON.")
    (home-page "https://github.com/vcflib/bio-vcf")
    (license license:expat)))

(define-public r-phantompeakqualtools
  (let ((commit "8d2b2d18c686d894ef5908b37da7adf72a07ef42")
        (revision "1"))
    (package
      (name "r-phantompeakqualtools")
      (version (git-version "1.2.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/kundajelab/phantompeakqualtools")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "00anrvcwsp02d98qhj1xpj85644h2pp4kfzq6dgbmwmdr6jvy7p4"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f ; There are no tests.
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)
           (delete 'build)
           (replace 'install
             (lambda* (#:key inputs outputs #:allow-other-keys)
               (let ((script (string-append (assoc-ref outputs "out")
                                            "/share/scripts")))
                 (install-file "run_spp.R" script)))))))
      (inputs
       `(("r" ,r-minimal)))
      (propagated-inputs
       (list r-catools
             r-snow
             r-snowfall
             r-bitops
             r-rsamtools
             r-spp
             gawk
             samtools
             boost
             gzip))
      (home-page "https://github.com/kundajelab/phantompeakqualtools")
      (synopsis "Informative enrichment for ChIP-seq data")
      (description "This package computes informative enrichment and quality
measures for ChIP-seq/DNase-seq/FAIRE-seq/MNase-seq data.  It can also be
used to obtain robust estimates of the predominant fragment length or
characteristic tag shift values in these assays.")
      (license license:bsd-3))))

(define-public rcas-web
  (let ((commit "71c93e3835653beb4eaa6e89b860bee3779729b8")
        (revision "2"))
    (package
      (name "rcas-web")
      (version (git-version "0.1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/rcas-web")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0232g0f7g0w5cgaib462zbfssvfq8i0iqv5b5wfmbrbn0sw99l9a"))))
      (build-system gnu-build-system)
      (arguments
       (list
        #:phases
        #~(modify-phases %standard-phases
            (add-after 'install 'wrap-executable
              (lambda* (#:key inputs outputs #:allow-other-keys)
                (let* ((json   #$(this-package-input "guile-json"))
                       (redis  #$(this-package-input "guile-redis"))
                       (path   (string-append
                                json  "/share/guile/site/3.0:"
                                redis "/share/guile/site/3.0")))
                  (wrap-program (string-append #$output "/bin/rcas-web")
                    `("GUILE_LOAD_PATH" ":" = (,path))
                    `("GUILE_LOAD_COMPILED_PATH" ":" = (,path))
                    `("R_LIBS_SITE" ":" = (,(getenv "R_LIBS_SITE"))))))))))
      (inputs
       (list bash-minimal
             r-minimal
             r-rcas
             guile-3.0
             guile-json-4
             guile-redis))
      (native-inputs
       (list autoconf
             automake
             pkg-config))
      (home-page "https://github.com/BIMSBbioinfo/rcas-web")
      (synopsis "Web interface for RNA-centric annotation system (RCAS)")
      (description "This package provides a simple web interface for the
@dfn{RNA-centric annotation system} (RCAS).")
      (license license:agpl3+))))

(define-public r-chipkernels
  (let ((commit "c9cfcacb626b1221094fb3490ea7bac0fd625372")
        (revision "1"))
    (package
      (name "r-chipkernels")
      (version (git-version "1.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/ManuSetty/ChIPKernels")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "14bj5qhjm1hsm9ay561nfbqi9wxsa7y487df2idsaaf6z10nw4v0"))))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        #~(modify-phases %standard-phases
            (add-before 'install 'fix-r-4.5.0
            ;; Changes in R 4.5.0: C-Level Facilities.
            ;; Strict R headers are now the default. This removes the legacy
            ;; definitions of PI, Calloc, Realloc and Free: use M_PI,
            ;; R_Calloc, R_Realloc or R_Free instead.
            ;; https://cran.r-project.org/doc/manuals/r-release/NEWS.html
            (lambda _
              (substitute* "src/hamming_distance.c"
                (("Calloc") "R_Calloc")
                (("Free") "R_Free")
                (("Realloc") "R_Realloc")))))))
      (propagated-inputs
       (list r-iranges
             r-xvector
             r-biostrings
             r-bsgenome
             r-gtools
             r-genomicranges
             r-sfsmisc
             r-kernlab
             r-s4vectors
             r-biocgenerics))
      (home-page "https://github.com/ManuSetty/ChIPKernels")
      (synopsis "Build string kernels for DNA Sequence analysis")
      (description "ChIPKernels is an R package for building different string
kernels used for DNA Sequence analysis.  A dictionary of the desired kernel
must be built and this dictionary can be used for determining kernels for DNA
Sequences.")
      (license license:gpl2+))))

(define-public r-seqgl
  (package
    (name "r-seqgl")
    (version "1.1.4")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/ManuSetty/SeqGL")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1r6ywvhxl3ffv48lgj7sbd582mcc6dha3ksgc2qjlvjrnkbj3799"))))
    (build-system r-build-system)
    (propagated-inputs
     (list r-biostrings
           r-chipkernels
           r-genomicranges
           r-spams
           r-wgcna
           r-fastcluster))
    (home-page "https://github.com/ManuSetty/SeqGL")
    (synopsis "Group lasso for Dnase/ChIP-seq data")
    (description "SeqGL is a group lasso based algorithm to extract
transcription factor sequence signals from ChIP, DNase and ATAC-seq profiles.
This package presents a method which uses group lasso to discriminate between
bound and non bound genomic regions to accurately identify transcription
factors bound at the specific regions.")
    (license license:gpl2+)))

(define-public r-seraster
  ;; There are no tags or releases.
  (let ((commit "4fdc1ffe5d3feb65de9880329d221cf276b393a1")
        (revision "1"))
    (package
      (name "r-seraster")
      (version (git-version "0.99.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/JEFworks-Lab/SEraster")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0y33lk8q0h9nfzmf7slxxvw3l0djassp4l63nsjpm3p6z4pah5s4"))))
      (properties `((upstream-name . "SEraster")))
      (build-system r-build-system)
      (propagated-inputs (list r-biocparallel
                               r-ggplot2
                               r-matrix
                               r-rearrr
                               r-sf
                               r-spatialexperiment
                               r-summarizedexperiment))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/JEFworks-Lab/SEraster")
      (synopsis
       "Rasterization framework for scalable spatial omics data analysis")
      (description
       "This package is a rasterization preprocessing framework that aggregates
cellular information into spatial pixels to reduce resource requirements for
spatial omics data analysis.  SEraster reduces the number of points in spatial
omics datasets for downstream analysis through a process of rasterization where
single cells gene expression or cell-type labels are aggregated into equally
sized pixels based on a user-defined resolution.  SEraster can be incorporated
with other packages to conduct downstream analyses for spatial omics datasets,
such as detecting spatially variable genes.")
      (license license:gpl3))))

(define-public emboss
  (package
    (name "emboss")
    (version "6.5.7")
    (source (origin
              (method url-fetch)
              (uri (string-append "ftp://emboss.open-bio.org/pub/EMBOSS/old/"
                                  (version-major+minor version) ".0/"
                                  "EMBOSS-" version ".tar.gz"))
              (sha256
               (base32
                "0vsmz96gc411yj2iyzdrsmg4l2n1nhgmp7vrgzlxx3xixv9xbf0q"))))
    (build-system gnu-build-system)
    (arguments
     `(#:configure-flags
       ,#~(list (string-append "--with-hpdf="
                               #$(this-package-input "libharu")))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'fix-checks
           (lambda _
             ;; The PNGDRIVER tests check for the presence of libgd, libpng
             ;; and zlib, but assume that they are all found at the same
             ;; prefix.
             (substitute* "configure.in"
               (("CHECK_PNGDRIVER")
                "LIBS=\"$LIBS -lgd -lpng -lz -lm\"
AC_DEFINE([PLD_png], [1], [Define to 1 if PNG support is available])
AM_CONDITIONAL(AMPNG, true)"))
             #t))
         (add-after 'fix-checks 'disable-update-check
           (lambda _
             ;; At build time there is no connection to the Internet, so
             ;; looking for updates will not work.
             (substitute* "Makefile.am"
               (("\\$\\(bindir\\)/embossupdate") ""))
             #t))
         (add-after 'disable-update-check 'autogen
           (lambda _ (invoke "autoreconf" "-vif") #t)))))
    (inputs
     (list perl
           libpng
           gd
           libx11
           libharu
           zlib))
    (native-inputs
     (list autoconf automake libtool pkg-config))
    (home-page "https://emboss.sourceforge.net")
    (synopsis "Molecular biology analysis suite")
    (description "EMBOSS is the \"European Molecular Biology Open Software
Suite\".  EMBOSS is an analysis package specially developed for the needs of
the molecular biology (e.g. EMBnet) user community.  The software
automatically copes with data in a variety of formats and even allows
transparent retrieval of sequence data from the web.  It also provides a
number of libraries for the development of software in the field of molecular
biology.  EMBOSS also integrates a range of currently available packages and
tools for sequence analysis into a seamless whole.")
    (license license:gpl2+)))

(define-public bits
  (let ((revision "1")
        (commit "3cc4567896d9d6442923da944beb704750a08d2d"))
    (package
      (name "bits")
      ;; The version is 2.13.0 even though no release archives have been
      ;; published as yet.
      (version (git-version "2.13.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/arq5x/bits")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "17n2kffk4kmhivd8c98g2vr6y1s23vbg4sxlxs689wni66797hbs"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f ;no tests included
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)
           (add-after 'unpack 'remove-cuda
             (lambda _
               (substitute* "Makefile"
                 ((".*_cuda") "")
                 (("(bits_test_intersections) \\\\" _ match) match))
               #t))
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (copy-recursively
                "bin" (string-append (assoc-ref outputs "out") "/bin"))
               #t)))))
      (inputs
       (list gsl zlib))
      (home-page "https://github.com/arq5x/bits")
      (synopsis "Implementation of binary interval search algorithm")
      (description "This package provides an implementation of the
BITS (Binary Interval Search) algorithm, an approach to interval set
intersection.  It is especially suited for the comparison of diverse genomic
datasets and the exploration of large datasets of genome
intervals (e.g. genes, sequence alignments).")
      (license license:gpl2))))

(define-public piranha
  ;; There is no release tarball for the latest version.  The latest commit is
  ;; older than one year at the time of this writing.
  (let ((revision "1")
        (commit   "0466d364b71117d01e4471b74c514436cc281233"))
    (package
      (name "piranha")
      (version (git-version "1.2.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/smithlabcode/piranha")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "117dc0zf20c61jam69sk4abl57ah6yi6i7qra7d7y5zrbgk12q5n"))))
      (build-system gnu-build-system)
      (arguments
       `(#:test-target "test"
         #:phases
         (modify-phases %standard-phases
           (add-after 'unpack 'copy-smithlab-cpp
             (lambda* (#:key inputs #:allow-other-keys)
               (for-each (lambda (file)
                           (install-file file "./src/smithlab_cpp/"))
                         (find-files (assoc-ref inputs "smithlab-cpp")))))
           (add-after 'install 'install-to-store
             (lambda* (#:key outputs #:allow-other-keys)
               (let* ((out (assoc-ref outputs "out"))
                      (bin (string-append out "/bin")))
                 (for-each (lambda (file)
                             (install-file file bin))
                           (find-files "bin" ".*"))))))
         #:configure-flags
         ,#~(list (string-append "--with-bam_tools_headers="
                                 #$(this-package-input "bamtools") "/include/bamtools")
                  (string-append "--with-bam_tools_library="
                                 #$(this-package-input "bamtools") "/lib/bamtools"))))
      (inputs
       `(("bamtools" ,bamtools)
         ("samtools" ,samtools-0.1)
         ("gsl" ,gsl)
         ("smithlab-cpp"
          ,(let ((commit "3723e2db438c51501d0423429ff396c3035ba46a"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/smithlabcode/smithlab_cpp")
                     (commit commit)))
               (file-name (string-append "smithlab_cpp-" commit "-checkout"))
               (sha256
                (base32
                 "0l4gvbwslw5ngziskja41c00x1r06l3yidv7y0xw9djibhykzy0g")))))))
      (native-inputs
       `(("python" ,python-2)))
      (home-page "https://github.com/smithlabcode/piranha")
      (synopsis "Peak-caller for CLIP-seq and RIP-seq data")
      (description
       "Piranha is a peak-caller for genomic data produced by CLIP-seq and
RIP-seq experiments.  It takes input in BED or BAM format and identifies
regions of statistically significant read enrichment.  Additional covariates
may optionally be provided to further inform the peak-calling process.")
      (license license:gpl3+))))

(define-public filevercmp
  (let ((commit "1a9b779b93d0b244040274794d402106907b71b7")
        (revision "1"))
    (package
      (name "filevercmp")
      (version (git-version "0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/ekg/filevercmp")
                      (commit commit)))
                (file-name (git-file-name name commit))
                (sha256
                 (base32
                  "1j9vxsy0y050v59h0q1d6501fcw1kjvj0d18l1xk2zyg0jzj247c"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f ; There are no tests to run.
         #:phases
         (modify-phases %standard-phases
           (delete 'configure) ; There is no configure phase.
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (let ((out (assoc-ref outputs "out")))
                 (install-file "filevercmp" (string-append out "/bin"))
                 (install-file "filevercmp.h" (string-append out "/include"))
                 #t))))))
      (home-page "https://github.com/ekg/filevercmp")
      (synopsis "This program compares version strings")
      (description "This program compares version strings.  It intends to be a
replacement for strverscmp.")
      (license license:gpl3+))))

(define-public multiqc
  (package
    (name "multiqc")
    (version "1.14")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "multiqc" version))
       (sha256
        (base32
         "0qlk2h0765hxp1w9hqf925rh8hq642lzgs6ppg91wln9y02s9fyw"))))
    (build-system pyproject-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (replace 'check
           (lambda* (#:key inputs tests? #:allow-other-keys)
             (when tests?
               (setenv "HOME" "/tmp")
               (let ((here (getcwd)))
                 (copy-recursively (assoc-ref inputs "tests") "/tmp/tests")
                 (with-directory-excursion "/tmp/tests"
                   (invoke "multiqc" "data" "--ignore" "data/modules")))))))))
    (inputs
     (list python-click
           python-coloredlogs
           python-future
           python-jinja2
           python-lzstring
           python-markdown
           python-matplotlib
           python-networkx
           python-numpy
           python-pyyaml
           python-requests
           python-rich
           python-rich-click
           python-setuptools ; For pkg_resources.
           python-simplejson
           python-spectra))
    (native-inputs
     `(("python-pytest" ,python-pytest)
       ("python-wheel" ,python-wheel)
       ("tests"
        ,(let ((commit "c3e7400affe3f3ca996973805797af61b93070ba"))
           (origin
             (method git-fetch)
             (uri (git-reference
                   (url "https://github.com/ewels/MultiQC_TestData")
                   (commit commit)))
             (file-name (git-file-name "multiqc-test-data" commit))
             (sha256
              (base32
               "0wwqbkbfnqj5ax6l0da4csbmv1sw7ya7mwvdwryckaiqmkxy80va")))))))
    (home-page "https://multiqc.info")
    (synopsis "Aggregate bioinformatics analysis reports")
    (description
     "MultiQC is a tool to aggregate bioinformatics results across many
samples into a single report.  It contains modules for a large number of
common bioinformatics tools.")
    (license license:gpl3+)))

(define-public r-chromvarmotifs
  (let ((commit "38bed559c1f4770b6c91c80bf3f8ea965da26076")
        (revision "1"))
    (package
      (name "r-chromvarmotifs")
      (version (git-version "0.2.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/GreenleafLab/chromVARmotifs")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0i9v1m1hrg1lkd2pnkj5nnrpks6vhhhpbdhsfl2lmjak4npxxr5q"))))
      (properties `((upstream-name . "chromVARmotifs")))
      (build-system r-build-system)
      (propagated-inputs
       `(("r-tfbstools" ,r-tfbstools)))
      (home-page "https://github.com/GreenleafLab/chromVARmotifs")
      (synopsis "Stores motif collections for use with motifmatchr or chromVAR")
      (description
       "This package stores motif collections as lists of @dfn{position
frequency matrix} (PWMatrixList) objects provided by the @code{TFBSTools}
package for use in R with packages like @code{motifmatchr} or
@code{chromVAR}.")
      (license license:expat))))

(define-public r-raremetals2
  (package
    (name "r-raremetals2")
    (version "0.1")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "http://genome.sph.umich.edu/w/images/"
                           "b/b7/RareMETALS2_" version ".tar.gz"))
       (sha256
        (base32
         "0z5ljcgvnm06ja9lm85a3cniq7slxcy37aqqkxrdidr79an5fs4s"))))
    (properties `((upstream-name . "RareMETALS2")))
    (build-system r-build-system)
    (propagated-inputs
     (list r-seqminer r-mvtnorm r-mass r-compquadform r-getopt))
    (home-page "http://genome.sph.umich.edu/wiki/RareMETALS2")
    (synopsis "Analyze gene-level association tests for binary trait")
    (description
     "The R package rareMETALS2 is an extension of the R package rareMETALS.
It was designed to meta-analyze gene-level association tests for binary trait.
While rareMETALS offers a near-complete solution for meta-analysis of
gene-level tests for quantitative trait, it does not offer the optimal
solution for binary trait.  The package rareMETALS2 offers improved features
for analyzing gene-level association tests in meta-analyses for binary
trait.")
    (license license:gpl3)))

(define-public r-rnacrosslinkoo
  (let ((commit "a317e0fa6ddf34c309529d57390769e2b2b5bfb7")
        (revision "1"))
    (package
      (name "r-rnacrosslinkoo")
      (version (git-version "0.1.3" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/cran/rnaCrosslinkOO")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0csv9924z0ish960k034qzv3gxh1yabnxni8hsrn5j6xl1r3cdpl"))))
      (properties `((upstream-name . "rnaCrosslinkOO")))
      (build-system r-build-system)
      (propagated-inputs (list r-classdiscovery
                               r-doparallel
                               r-foreach
                               r-genomicranges
                               r-ggplot2
                               r-ggrepel
                               r-heatmap3
                               r-igraph
                               r-iranges
                               r-mass
                               r-mixtools
                               r-patchwork
                               r-r4rna
                               r-rcolorbrewer
                               r-reshape2
                               r-rrna
                               r-s4vectors
                               r-seqinr
                               r-tidyverse
                               r-topdom))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/cran/rnaCrosslinkOO")
      (synopsis "Analysis of RNA crosslinking data")
      (description
       "The package is ideal for analyzing RNA structure and chemical probing data.")
      (license license:gpl3))))

(define-public r-rnaseqdtu
  (let ((commit "5bee1e769d2e1dc6a3f1cecb78078050eeb5b9ac")
        (revision "1"))
    (package
      (name "r-rnaseqdtu")
      (version (git-version "2.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/mikelove/rnaseqDTU/")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0jfi1ydsk8m5nadwnih48v87nnxdc7s3f0pny4axmnj40dd42as0"))))
      (properties `((upstream-name . "rnaseqDTU")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-deseq2
             r-devtools
             r-dexseq
             r-drimseq
             r-edger
             r-rafalib
             r-stager))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/mikelove/rnaseqDTU/")
      (synopsis "RNA-seq workflow for differential transcript usage")
      (description
       "This package provides an RNA-seq workflow for differential transcript
usage (DTU) following Salmon quantification.  This workflow performs a DTU
analysis on simulated data.  It also shows how to use stageR to perform
two-stage testing of DTU, a statistical framework to screen at the gene level
and then confirm which transcripts within the significant genes show evidence
of DTU.")
      (license license:artistic2.0))))

(define-public r-dropbead
  (let ((commit "d746c6f3b32110428ea56d6a0001ce52a251c247")
        (revision "2"))
    (package
      (name "r-dropbead")
      (version (string-append "0-" revision "." (string-take commit 7)))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/rajewsky-lab/dropbead")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0sbzma49aiiyw8b0jpr7fnhzys9nsqmp4hy4hdz1gzyg1lhnca26"))))
      (build-system r-build-system)
      ;; Tests make invalid assumption about test files and their locations.
      (arguments (list #:tests? #false))
      (propagated-inputs
       (list r-ggplot2 r-rcolorbrewer r-gridextra r-gplots r-plyr))
      (native-inputs (list r-data-table))
      (home-page "https://github.com/rajewsky-lab/dropbead")
      (synopsis "Basic exploration and analysis of Drop-seq data")
      (description "This package offers a quick and straight-forward way to
explore and perform basic analysis of single cell sequencing data coming from
droplet sequencing.  It has been particularly tailored for Drop-seq.")
      (license license:gpl3))))

(define-public r-cellchat
  (let ((commit
         "21edd226ca408e4c413408f98562d71ee0b54e5d")
        (revision "1"))
    (package
      (name "r-cellchat")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/sqjin/CellChat")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0cvzl9mi8jjznpql2gv67swnk1dndn3a2h22z5l84h7lwpwjmh53"))
         (snippet
          '(for-each delete-file '("src/CellChat.so"
                                   "src/CellChat_Rcpp.o"
                                   "src/RcppExports.o")))))
      (properties `((upstream-name . "CellChat")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biocgenerics
             r-circlize
             r-colorspace
             r-complexheatmap
             r-cowplot
             r-dplyr
             r-expm
             r-fnn
             r-forcats
             r-future
             r-future-apply
             r-gg-gap
             r-ggalluvial
             r-ggplot2
             r-ggrepel
             r-igraph
             r-irlba
             r-magrittr
             r-matrix
             r-nmf
             r-patchwork
             r-pbapply
             r-rcolorbrewer
             r-rcpp
             r-rcppeigen
             r-reshape2
             r-reticulate
             r-rspectra
             r-rtsne
             r-scales
             r-shape
             r-sna
             r-stringr
             r-svglite))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/sqjin/CellChat")
      (synopsis "Analysis of cell-cell communication from single-cell transcriptomics data")
      (description
       "This package infers, visualizes and analyzes the cell-cell
communication networks from scRNA-seq data.")
      (license license:gpl3))))

(define-public r-cellchat-2
  (let ((commit "b05405af0f4f2cac99f2211e888d42de4c5a9d59")
        (revision "1"))
    (package
      (name "r-cellchat")
      (version (git-version "2.1.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/jinworks/CellChat")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "14s7bl70xi21ivqr1c3qx6kjf1ndlyxag880xldqkfsqpksvyvkc"))
         (snippet '(for-each delete-file
                             '("src/CellChat.so" "src/CellChat_Rcpp.o"
                               "src/RcppExports.o")))))
      (properties `((upstream-name . "CellChat")))
      (build-system r-build-system)
      (propagated-inputs (list r-biocgenerics
                               r-biocneighbors
                               r-bslib
                               r-circlize
                               r-colorspace
                               r-complexheatmap
                               r-cowplot
                               r-dplyr
                               r-fnn
                               r-future
                               r-future-apply
                               r-ggalluvial
                               r-ggnetwork
                               r-ggplot2
                               r-ggpubr
                               r-ggrepel
                               r-igraph
                               r-irlba
                               r-magrittr
                               r-matrix
                               r-nmf
                               r-patchwork
                               r-pbapply
                               r-plotly
                               r-plyr
                               r-rcolorbrewer
                               r-rcpp
                               r-rcppeigen
                               r-reshape2
                               r-reticulate
                               r-rspectra
                               r-scales
                               r-shape
                               r-shiny
                               r-sna
                               r-stringr
                               r-svglite))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/jinworks/CellChat")
      (synopsis
       "Inference of cell-cell communication from single-cell and transcriptomics")
      (description
       "This R tool infers, visualizes, and analyzes cell-cell communication networks.
It supports scRNA-seq and spatially resolved transcriptomics data.")
      (license license:gpl3))))

(define-public r-copykat
  (let ((commit                         ;no tag
         "256de33dfc1b80a1a0ac9e098c5557f95a4e0d53")
        (revision "0"))
    (package
      (name "r-copykat")
      (version (git-version "1.0.8" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/navinlabcode/copykat")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0ckyqnial3imcqlgd6xfgwk5w977l1i87sx4kdbwdvg40w0vh1j8"))))
      (properties `((upstream-name . "copykat")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-cluster
             r-dlm
             r-gplots
             r-mcmcpack
             r-mixtools
             r-paralleldist
             r-rcolorbrewer))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/navinlabcode/copykat")
      (synopsis "Inference of genomic copy number from single cell RNAseq data")
      (description
       "This package Copynumber KAryotyping of Tumors infers genomic copy
number and subclonal structure of human tumors using integrative Bayesian
approaches to identify genome-wide aneuploidy at 5MB resolution in single
cells data.  It separates tumor cells and tumor subclones from normal cells
using high-throughput sc-RNAseq data.")
      (license license:gpl2))))

(define-public sambamba
  (package
    (name "sambamba")
    (version "0.8.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/biod/sambamba")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1zdkd1md5wk4la71p82pbclqqcm55abk23fk087da6186i1bsihl"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; there is no test target
       #:parallel-build? #f             ; not supported
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (add-after 'unpack 'prepare-build-tools
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "Makefile"
               (("\\$\\(shell which ldmd2\\)") (which "ldmd2")))
             (setenv "CC" "gcc")
             (setenv "D_LD" (which "ld.gold"))))
         (add-after 'unpack 'unbundle-prerequisites
           (lambda _
             (substitute* "Makefile"
               (("= lz4/lib/liblz4.a") "= -L-llz4")
               (("ldc_version_info lz4-static") "ldc_version_info"))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
               (mkdir-p bin)
               (copy-file (string-append "bin/sambamba-" ,version)
                          (string-append bin "/sambamba"))))))))
    (native-inputs
     (list python))
    (inputs
     (list ldc lz4 zlib))
    (home-page "https://github.com/biod/sambamba")
    (synopsis "Tools for working with SAM/BAM data")
    (description "Sambamba is a high performance modern robust and
fast tool (and library), written in the D programming language, for
working with SAM and BAM files.  Current parallelised functionality is
an important subset of samtools functionality, including view, index,
sort, markdup, and depth.")
    (license license:gpl2+)))

(define-public r-rphyloxml
  (let ((commit "a30e39249239b2de01d6964ae2a2205a6c48b475")
        (revision "1"))
    (package
      (name "r-rphyloxml")
      (version (git-version "0.0-9000" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/USCbiostats/rphyloxml")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "15ijzqvjxx6vqyqlg5asdbqlhw1g0ix6palf1rism3si0qapddgw"))
         (snippet
          '(delete-file "docs/jquery.sticky-kit.min.js"))))
      (properties `((upstream-name . "rphyloxml")))
      (build-system r-build-system)
      (arguments
       (list
        #:modules
        '((guix build r-build-system)
          (guix build minify-build-system)
          (guix build utils))
        #:imported-modules
        `(,@%r-build-system-modules (guix build minify-build-system))
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'process-javascript
             (lambda* (#:key inputs #:allow-other-keys)
               (with-directory-excursion "inst/"
                 (minify (assoc-ref inputs "js-jquery-sticky-kit")
                         #:target "docs/jquery.sticky-kit.min.js")))))))
      (propagated-inputs (list r-ape r-xml2))
      (native-inputs
       `(("esbuild" ,esbuild)
         ("js-jquery-sticky-kit"
          ,(origin
             (method url-fetch)
             (uri "https://raw.githubusercontent.com/leafo/sticky-kit/\
v1.1.2/jquery.sticky-kit.js")
             (sha256
              (base32
               "17c3a1hqc3ybwj7hpw8prazajp2x98aq7nyfn71h6lzjvblq297g"))))))
      (home-page "https://github.com/USCbiostats/rphyloxml")
      (synopsis "Read and write phyloXML files in R")
      (description
       "The package reads phylogenetic data in the @code{phyloXML} format.
It also includes functions for writing data in this format.")
      (license license:expat))))

(define-public ritornello
  (package
    (name "ritornello")
    (version "2.0.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/KlugerLab/Ritornello")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xahvq215qld7x1w8vpa5zbrsj6p9crb9shqa2x89sb0aaxa02jk"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; there are no tests
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'patch-samtools-references
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* '("src/SamStream.h"
                            "src/FLD.cpp")
               (("<sam.h>") "<samtools/sam.h>"))
             #t))
         (delete 'configure)
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out    (assoc-ref outputs "out"))
                    (bin    (string-append out "/bin/")))
               (mkdir-p bin)
               (install-file "bin/Ritornello" bin)
               #t))))))
    (inputs
     (list samtools-0.1 fftw boost zlib))
    (home-page "https://github.com/KlugerLab/Ritornello")
    (synopsis "Control-free peak caller for ChIP-seq data")
    (description "Ritornello is a ChIP-seq peak calling algorithm based on
signal processing that can accurately call binding events without the need to
do a pair total DNA input or IgG control sample.  It has been tested for use
with narrow binding events such as transcription factor ChIP-seq.")
    (license license:gpl3+)))

(define-public trim-galore
  (package
    (name "trim-galore")
    (version "0.6.6")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/FelixKrueger/TrimGalore")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0yrwg6325j4sb9vnplvl3jplzab0qdhp92wl480qjinpfq88j4rs"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; no tests
       #:phases
       (modify-phases %standard-phases
         (replace 'configure
           (lambda _
             ;; Trim Galore tries to figure out what version of Python
             ;; cutadapt is using by looking at the shebang.  Of course that
             ;; doesn't work, because cutadapt is wrapped in a shell script.
             (substitute* "trim_galore"
               (("my \\$python_return.*")
                "my $python_return = \"Python 3.999\";\n"))
             #t))
         (delete 'build)
         (add-after 'unpack 'hardcode-tool-references
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "trim_galore"
               (("\\$path_to_cutadapt = 'cutadapt'")
                (string-append "$path_to_cutadapt = '"
                               (assoc-ref inputs "cutadapt")
                               "/bin/cutadapt'"))
               (("\\$compression_path = \"gzip\"")
                (string-append "$compression_path = \""
                               (assoc-ref inputs "gzip")
                               "/bin/gzip\""))
               (("\"gunzip")
                (string-append "\""
                               (assoc-ref inputs "gzip")
                               "/bin/gunzip"))
               (("\"pigz")
                (string-append "\""
                               (assoc-ref inputs "pigz")
                               "/bin/pigz")))
             #t))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out")
                                       "/bin")))
               (mkdir-p bin)
               (install-file "trim_galore" bin)
               #t))))))
    (inputs
     (list gzip perl pigz cutadapt))
    (native-inputs
     (list unzip))
    (home-page "https://www.bioinformatics.babraham.ac.uk/projects/trim_galore/")
    (synopsis "Wrapper around Cutadapt and FastQC")
    (description "Trim Galore! is a wrapper script to automate quality and
adapter trimming as well as quality control, with some added functionality to
remove biased methylation positions for RRBS sequence files.")
    (license license:gpl3+)))

(define-public phylip
  (package
    (name "phylip")
    (version "3.697")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "http://evolution.gs.washington.edu/phylip/"
                           "download/phylip-" version ".tar.gz"))
       (sha256
        (base32
         "1h8h0nafnlbqryswxgplx80k2044yhfz97jh13vsgzlaifqdh9ls"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; no check target
       #:make-flags (list "-f" "Makefile.unx" "CFLAGS=-fcommon" "install")
       #:parallel-build? #f             ; not supported
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'enter-dir
           (lambda _ (chdir "src")))
         (delete 'configure)
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let ((target (string-append (assoc-ref outputs "out")
                                          "/bin")))
               (mkdir-p target)
               (for-each (lambda (file)
                           (install-file file target))
                         (find-files "../exe" ".*"))))))))
    (home-page "https://evolution.genetics.washington.edu/phylip/")
    (synopsis "Tools for inferring phylogenies")
    (description "PHYLIP (the PHYLogeny Inference Package) is a package of
programs for inferring phylogenies (evolutionary trees).")
    (license license:bsd-2)))

(define-public phyml
  (package
    (name "phyml")
    (version "3.3.20220408")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/stephaneguindon/phyml")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03hdqmnsgnzkcrp9r9ajdfkj33jgq4b86kra8ssjlrph65y344sa"))
              (snippet
               '(delete-file "doc/phyml-manual.pdf"))))
    (build-system gnu-build-system)
    (supported-systems '("x86_64-linux"))
    (arguments
     (let ((default-flags (list "--disable-native")))
       `(#:phases
         (let ((build (lambda (what)
                         (lambda args
                           (apply (assoc-ref %standard-phases 'configure)
                                  (append args
                                          (list #:configure-flags
                                                (cons (format #false "--enable-~a" what)
                                                      '() #;,default-flags))))
                           (apply (assoc-ref %standard-phases 'build) args)
                           (apply (assoc-ref %standard-phases 'install) args)))))
           (modify-phases %standard-phases
             ;; We cannot use --disable-native; see
             ;; https://github.com/stephaneguindon/phyml/issues/173 Instead we
             ;; patch the code to at least get rid of -march=native.
             (add-after 'unpack 'remove-march-native
               (lambda _
                 (substitute* "configure.ac"
                   (("DEFAULT_VECTOR_FLAG=\"-march=native\"")
                    "DEFAULT_VECTOR_FLAG=\"-march=athlon64-sse3\"\n"))))
             (add-after 'build 'build-manual
               (lambda _
                 (with-directory-excursion "doc"
                   (invoke "make" "phyml-manual.pdf"))))
             (add-after 'build-manual 'install-manual
               (lambda* (#:key outputs #:allow-other-keys)
                 (with-directory-excursion "doc"
                   (install-file "phyml-manual.pdf"
                                 (string-append (assoc-ref outputs "out")
                                                "/share/doc/phyml")))))
             (add-after 'install 'build-phyml-mpi
               (build "phyml-mpi"))
             (add-after 'build-phyml-mpi 'build-rf
               (build "rf"))
             (add-after 'build-rf 'build-phyrex
               (build "phyrex")))))))
    (native-inputs
     (list automake
           autoconf
           openmpi
           (texlive-local-tree
            (list texlive-caption
                  texlive-cite
                  texlive-fancyvrb
                  texlive-infwarerr
                  texlive-kvoptions
                  texlive-pdftexcmds
                  texlive-psfrag
                  texlive-xcolor))))
    (home-page "https://github.com/stephaneguindon/phyml")
    (synopsis "Programs for working on SAM/BAM files")
    (description
     "@code{PhyML} is a software package that uses modern statistical
approaches to analyse alignments of nucleotide or amino acid sequences in a
phylogenetic framework.  The main tool in this package builds phylogenies
under the maximum likelihood criterion.  It implements a large number of
substitution models coupled with efficient options to search the space of
phylogenetic tree topologies.  code{PhyREX} fits the
spatial-Lambda-Fleming-Viot model to geo-referenced genetic data.  This model
is similar to the structured coalescent but assumes that individuals are
distributed along a spatial continuum rather than discrete demes.
@code{PhyREX} can be used to estimate population densities and rates of
dispersal.  Its output can be processed by treeannotator (from the
@code{BEAST} package) as well as @code{SPREAD}.")
    (license license:gpl3)))

(define-public imp
  (package
    (name "imp")
    (version "2.23.0")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "https://integrativemodeling.org/"
                           version "/download/imp-" version ".tar.gz"))
       (sha256
        (base32
         "080z50iq1f3nmkccy2i5hj0i34j26sbwcaqizcbljji9mvp0nc0q"))))
    (build-system cmake-build-system)
    (arguments
     `(#:tests? #false ; The test suite is notoriously fickle
       #:configure-flags
       (let ((disabled-tests
              '("expensive"                 ;exclude expensive tests
                "IMP.modeller"              ;fail to import its own modules
                "IMP.parallel-test_sge.py"  ;fail in build container
                ;; The following test fails non-reproducibly on
                ;; an inexact numbers assertion.
                "IMP.em-medium_test_local_fitting.py"
                ;; The following test fails for unknown reasons
                "IMP.foxs-add-missing-residues.py")))
         (list
          (string-append
           "-DCMAKE_CTEST_ARGUMENTS="
           (string-join
            (list "-L" "-tests?-"       ;select only tests
                  "-E" (format #f "'(~a)'" (string-join disabled-tests "|")))
            ";"))))))
    (native-inputs
     (list python-wrapper swig))
    (inputs
     (list boost-for-mysql
           cereal
           cgal
           gsl
           hdf5
           fftw
           eigen
           ;; Enabling MPI causes the build to use all the available memory and
           ;; fail (tested on a machine with 32 GiB of RAM).
           ;;("mpi" ,openmpi)
           opencv))
    (propagated-inputs
     (list python-numpy python-scipy python-pandas python-scikit-learn
           python-networkx))
    (home-page "https://integrativemodeling.org")
    (synopsis "Integrative modeling platform")
    (description "IMP's broad goal is to contribute to a comprehensive
structural characterization of biomolecules ranging in size and complexity
from small peptides to large macromolecular assemblies, by integrating data
from diverse biochemical and biophysical experiments.  IMP provides a C++ and
Python toolbox for solving complex modeling problems, and a number of
applications for tackling some common problems in a user-friendly way.")
    ;; IMP is largely available under the GNU Lesser GPL; see the file
    ;; COPYING.LGPL for the full text of this license. Some IMP modules are
    ;; available under the GNU GPL (see the file COPYING.GPL).
    (license (list license:lgpl2.1+
                   license:gpl3+))))

;; We use this seemingly arbitrary commit because of
;; https://github.com/3DGenomes/TADbit/issues/371
(define-public tadbit
  (let ((commit "283812901a00078c725bd9f0ee33366af6783969")
        (revision "2"))
    (package
      (name "tadbit")
      (version (git-version "1.0.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/3DGenomes/TADbit")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "07q0alxah6xl1hibaj1kj7pdzq3pg6csyapsihbwlbmqyfmllbgz"))))
      (build-system python-build-system)
      (arguments
       (list
        #:phases
        #~(modify-phases %standard-phases
            ;; Scipy 1.12 no longer has binom_test or polyfit
            (add-after 'unpack 'scipy-compatibility
              (lambda _
                (substitute* '("_pytadbit/modelling/imp_modelling.py"
                               "_pytadbit/modelling/restraints.py")
                  (("from scipy .*import polyfit")
                   "from numpy import polyfit"))
                (substitute* "_pytadbit/mapping/restriction_enzymes.py"
                  (("from scipy.stats import binom_test")
                   "from scipy.stats import binomtest")
                  (("binom_test\\(pats[k]['count'], nreads, 0.25\\*\\*len\\(k\\), alternative='greater'\\)")
                   "binomtest(pats[k]['count'], nreads, 0.25**len(k), alternative='greater').pval"))))
            (add-after 'unpack 'fix-problems-with-setup.py
              (lambda _
                (substitute* "src/test/Makefile"
                  (("^CFLAGS=") "CFLAGS= -fcommon"))

                ;; Don't attempt to install the bash completions to
                ;; the home directory.
                (rename-file "extras/.bash_completion"
                             "extras/tadbit")
                (substitute* "setup.py"
                  (("\\(path.expanduser\\('~'\\)")
                   (string-append "(\"" #$output
                                  "/etc/bash_completion.d\""))
                  (("extras/\\.bash_completion")
                   "extras/tadbit"))))
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (when tests?
                  (invoke "python3" "test/test_all.py")))))))
      (native-inputs
       (list `(,glib "bin") ;for gtester
             glib
             pkg-config))
      (inputs
       ;; TODO: add Chimera for visualization
       (list imp
             mcl
             python-future
             python-h5py
             python-scipy
             python-numpy
             python-matplotlib
             python-pysam
             r-minimal))
      (home-page "https://3dgenomes.github.io/TADbit/")
      (synopsis "Analyze, model, and explore 3C-based data")
      (description
       "TADbit is a complete Python library to deal with all steps to analyze,
model, and explore 3C-based data.  With TADbit the user can map FASTQ files to
obtain raw interaction binned matrices (Hi-C like matrices), normalize and
correct interaction matrices, identify and compare the so-called
@dfn{Topologically Associating Domains} (TADs), build 3D models from the
interaction matrices, and finally, extract structural properties from the
models.  TADbit is complemented by TADkit for visualizing 3D models.")
      (license license:gpl3+))))

(define-public kentutils
  (package
    (name "kentutils")
    ;; 302.1.0 is out, but the only difference is the inclusion of
    ;; pre-built binaries.
    (version "302.0.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/ENCODE-DCC/kentUtils")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0n1wbyjpzii2b9qhyp9r1q76j623cggpg3y8fmw78ld3z4y7ivha"))
       (modules '((guix build utils)
                  (srfi srfi-26)
                  (ice-9 ftw)))
       (snippet
        '(begin
           ;; Only the contents of the specified directories are free
           ;; for all uses, so we remove the rest.  "hg/autoSql" and
           ;; "hg/autoXml" are nominally free, but they depend on a
           ;; library that is built from the sources in "hg/lib",
           ;; which is nonfree.
           (let ((free (list "." ".."
                             "utils" "lib" "inc" "tagStorm"
                             "parasol" "htslib"))
                 (directory? (lambda (file)
                               (eq? 'directory (stat:type (stat file))))))
             (for-each (lambda (file)
                         (and (directory? file)
                              (delete-file-recursively file)))
                       (map (cut string-append "src/" <>)
                            (scandir "src"
                                     (lambda (file)
                                       (not (member file free)))))))
           ;; Only make the utils target, not the userApps target,
           ;; because that requires libraries we won't build.
           (substitute* "Makefile"
             ((" userApps") " utils"))
           ;; Only build libraries that are free.
           (substitute* "src/makefile"
             (("DIRS =.*") "DIRS =\n")
             (("cd jkOwnLib.*") "")
             ((" hgLib") "")
             (("cd hg.*") ""))
           (substitute* "src/utils/makefile"
             ;; These tools depend on "jkhgap.a", which is part of the
             ;; nonfree "src/hg/lib" directory.
             (("raSqlQuery") "")
             (("pslLiftSubrangeBlat") "")

             ;; Do not build UCSC tools, which may require nonfree
             ;; components.
             (("ALL_APPS =.*") "ALL_APPS = $(UTILS_APPLIST)\n"))
           #t))))
    (build-system gnu-build-system)
    (arguments
     `( ;; There is no global test target and the test target for
       ;; individual tools depends on input files that are not
       ;; included.
       #:tests? #f
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'fix-permissions
           (lambda _ (make-file-writable "src/inc/localEnvironment.mk") #t))
         (add-after 'unpack 'fix-paths
           (lambda _
             (substitute* "Makefile"
               (("/bin/echo") (which "echo")))
             #t))
         (add-after 'unpack 'prepare-samtabix
           (lambda* (#:key inputs #:allow-other-keys)
             (copy-recursively (assoc-ref inputs "samtabix")
                               "samtabix")
             #t))
         (delete 'configure)
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out")
                                       "/bin")))
               (copy-recursively "bin" bin))
             #t)))))
    (native-inputs
     `(("samtabix"
        ,(let ((commit "10fd107909c1ac4d679299908be4262a012965ba"))
           (origin
             (method git-fetch)
             (uri (git-reference
                   (url "http://genome-source.cse.ucsc.edu/samtabix.git")
                   (commit commit)))
             (file-name (git-file-name "samtabix" (string-take commit 7)))
             (sha256
              (base32
               "0c1nj64l42v395sa84n7az43xiap4i6f9n9dfz4058aqiwkhkmma")))))))
    (inputs
     `(("zlib" ,zlib)
       ("tcsh" ,tcsh)
       ("perl" ,perl)
       ("libpng" ,libpng)
       ("mariadb-dev" ,mariadb "dev")
       ("openssl" ,openssl)))
    (home-page "https://genome.cse.ucsc.edu/index.html")
    (synopsis "Assorted bioinformatics utilities")
    (description "This package provides the kentUtils, a selection of
bioinformatics utilities used in combination with the UCSC genome
browser.")
    ;; Only a subset of the sources are released under a non-copyleft
    ;; free software license.  All other sources are removed in a
    ;; snippet.  See this bug report for an explanation of how the
    ;; license statements apply:
    ;; https://github.com/ENCODE-DCC/kentUtils/issues/12
    (license (license:non-copyleft
              "http://genome.ucsc.edu/license/"
              "The contents of this package are free for all uses."))))

(define-public skewer
  (let ((commit "978e8e46cba44595ece623e5387749284a86b74b")
        (revision "1"))
    (package
      (name "skewer")
      (version (git-version "0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/relipmoc/skewer")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "1b3skfnj3zmd2065fiyxjrg4k6xsxfy6nxz80vmajd830mk6vr69"))))
      (build-system gnu-build-system)
      (arguments
       (list
        #:tests? #f
        #:phases
        #~(modify-phases %standard-phases
            (delete 'configure)
            (replace 'install
              (lambda _
                (install-file "skewer" (string-append #$output "/bin")))))))
      ;; XXX Fails to build with GCC 11.
      (native-inputs (list gcc-10))
      (home-page "https://github.com/relipmoc/skewer")
      (synopsis "Bit-masked k-difference matching")
      (description "Skewer implements the bit-masked k-difference matching
algorithm dedicated to the task of adapter trimming and it is specially
designed for processing next-generation sequencing (NGS) paired-end
sequences.")
      (license license:expat))))

(define-public f-seq
  (let ((commit "6ccded34cff38cf432deed8503648b4a66953f9b")
        (revision "1"))
    (package
      (name "f-seq")
      (version (git-version "1.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/aboyle/F-seq")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1nk33k0yajg2id4g59bc4szr58r2q6pdq42vgcw054m8ip9wv26h"))
                (modules '((guix build utils)))
                ;; Remove bundled Java library archives.
                (snippet
                 '(for-each delete-file (find-files "lib" ".*")))))
      (build-system ant-build-system)
      (arguments
       `(#:tests? #f ; no tests included
         #:phases
         (modify-phases %standard-phases
           (replace 'install
             (lambda* (#:key inputs outputs #:allow-other-keys)
               (let* ((target (assoc-ref outputs "out"))
                      (bin (string-append target "/bin"))
                      (doc (string-append target "/share/doc/f-seq"))
                      (lib (string-append target "/lib")))
                 (mkdir-p target)
                 (mkdir-p doc)
                 (substitute* "bin/linux/fseq"
                   (("java") (which "java"))
                   (("\\$REALDIR/../lib/commons-cli-1.1.jar")
                    (search-input-file inputs
                                       (string-append "/lib/m2/commons-cli/commons-cli/"
                                                      ,(package-version java-commons-cli)
                                                      "/commons-cli-"
                                                      ,(package-version java-commons-cli)
                                                      ".jar")))
                   (("REALDIR=.*")
                    (string-append "REALDIR=" bin "\n")))
                 (install-file "README.txt" doc)
                 (install-file "bin/linux/fseq" bin)
                 (install-file "build~/fseq.jar" lib)
                 (copy-recursively "lib" lib)))))))
      (inputs
       (list perl java-commons-cli))
      (home-page "https://fureylab.web.unc.edu/software/fseq/")
      (synopsis "Feature density estimator for high-throughput sequence tags")
      (description
       "F-Seq is a software package that generates a continuous tag sequence
density estimation allowing identification of biologically meaningful sites
such as transcription factor binding sites (ChIP-seq) or regions of open
chromatin (DNase-seq).  Output can be displayed directly in the UCSC Genome
Browser.")
      (license license:gpl3+))))

(define-public bismark
  (package
    (name "bismark")
    (version "0.24.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/FelixKrueger/Bismark")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0j4dy33769f0jr2w1brb710zxwpg3zwjlnvlcpi5pr6mqc8dkg8n"))
       (snippet '(delete-file "plotly/plot.ly"))))
    (build-system perl-build-system)
    (arguments
     (list
      #:tests? #f                       ; there are no tests
      #:modules '((guix build utils)
                  (ice-9 popen)
                  (srfi srfi-26)
                  (guix build perl-build-system))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'replace-plotly.js
            (lambda* (#:key inputs #:allow-other-keys)
              (let* ((share (string-append #$output "/share/bismark"))
                     (file (assoc-ref inputs "plotly.js"))
                     (installed "plotly/plotly.js"))
                ;; The bundled plotly.js is minified.
                (let ((minified (open-pipe* OPEN_READ "uglifyjs" file)))
                  (call-with-output-file installed
                    (cut dump-port minified <>)))
                (substitute* "bismark2report"
                  (("plotly_template.tpl")
                   (string-append share "/plotly/plotly_template.tpl"))
                  (("my \\$plotly_code = read_report_template\\('plot.ly'\\);")
                   (string-append "\
my $plotly_code = read_report_template('" share "/plotly/plotly.js');
$plotly_code = \"<script>\" . $plotly_code . \"</script>\";"))))))
          (replace 'configure
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "bismark"
                (("\\(\\!system \"which samtools >/dev/null 2>&1\"\\)")
                 "(\"true\")")
                (("\\$samtools_path = `which samtools`;")
                 (string-append "$samtools_path = '"
                                (search-input-file inputs "/bin/samtools")
                                "';"))
                (("\\$path_to_bowtie2 = 'bowtie2'")
                 (string-append "$path_to_bowtie2 = '"
                                (search-input-file inputs "/bin/bowtie2")
                                "'"))
                (("\\$path_to_hisat2 = 'hisat2'")
                 (string-append "$path_to_hisat2 = '"
                                (search-input-file inputs "/bin/hisat2")
                                "'"))
                (("\\$path_to_minimap2 = 'minimap2'")
                 (string-append "$path_to_minimap2 = '"
                                (search-input-file inputs "/bin/minimap2")
                                "'")))))
          (delete 'build)
          (replace 'install
            (lambda* (#:key inputs outputs #:allow-other-keys)
              (let* ((bin (string-append #$output "/bin"))
                     (share   (string-append #$output "/share/bismark"))
                     (docdir  (string-append #$output "/share/doc/bismark"))
                     (scripts '("bismark"
                                "bismark_genome_preparation"
                                "bismark_methylation_extractor"
                                "bismark2bedGraph"
                                "bismark2report"
                                "coverage2cytosine"
                                "deduplicate_bismark"
                                "filter_non_conversion"
                                "bam2nuc"
                                "bismark2summary"
                                "NOMe_filtering")))
                (mkdir-p share)
                (mkdir-p docdir)
                (mkdir-p bin)
                (for-each (lambda (file) (install-file file bin))
                          scripts)
                (copy-recursively "docs" docdir)
                (copy-recursively "plotly"
                                  (string-append share "/plotly"))

                ;; Fix references to gunzip
                (substitute* (map (lambda (file)
                                    (string-append bin "/" file))
                                  scripts)
                  (("\"gunzip -c")
                   (string-append "\"" (assoc-ref inputs "gzip")
                                  "/bin/gunzip -c")))))))))
    (inputs
     (list bowtie gzip hisat2 minimap2 perl-carp perl-getopt-long samtools))
    (native-inputs
     `(("plotly.js"
        ,(origin
           (method url-fetch)
           (uri (string-append "https://raw.githubusercontent.com/plotly/plotly.js/"
                               "v1.39.4/dist/plotly.js"))
           (sha256
            (base32 "138mwsr4nf5qif4mrxx286mpnagxd1xwl6k8aidrjgknaqg88zyr"))))
       ("uglifyjs" ,node-uglify-js)))
    (home-page "https://www.bioinformatics.babraham.ac.uk/projects/bismark/")
    (synopsis "Map bisulfite treated sequence reads and analyze methylation")
    (description "Bismark is a program to map bisulfite treated sequencing
reads to a genome of interest and perform methylation calls in a single step.
The output can be easily imported into a genome viewer, such as SeqMonk, and
enables a researcher to analyse the methylation levels of their samples
straight away.  Its main features are:

@itemize
@item Bisulfite mapping and methylation calling in one single step
@item Supports single-end and paired-end read alignments
@item Supports ungapped and gapped alignments
@item Alignment seed length, number of mismatches etc are adjustable
@item Output discriminates between cytosine methylation in CpG, CHG
  and CHH context
@end itemize\n")
    (license license:gpl3+)))

(define-public paml
  (package
    (name "paml")
    (version "4.9e")
    (source (origin
              (method url-fetch)
              (uri (string-append "http://abacus.gene.ucl.ac.uk/software/"
                                  "paml" version ".tgz"))
              (sha256
               (base32
                "13zf6h9fiqghwhch2h06x1zdr6s42plsnqahflp5g7myr3han3s6"))
              (modules '((guix build utils)))
              ;; Remove Windows binaries
              (snippet
               '(begin
                  (for-each delete-file (find-files "." "\\.exe$"))
                  ;; Some files in the original tarball have restrictive
                  ;; permissions, which makes repackaging fail
                  (for-each (lambda (file) (chmod file #o644)) (find-files "."))
                  #t))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; there are no tests
       #:make-flags '("CC=gcc" "CFLAGS=-fcommon -O3")
       #:phases
       (modify-phases %standard-phases
         (replace 'configure
           (lambda _
             (substitute* "src/BFdriver.c"
               (("/bin/bash") (which "bash")))
             (chdir "src")))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((tools '("baseml" "basemlg" "codeml"
                            "pamp" "evolver" "yn00" "chi2"))
                   (bin    (string-append (assoc-ref outputs "out") "/bin"))
                   (docdir (string-append (assoc-ref outputs "out")
                                           "/share/doc/paml")))
               (mkdir-p bin)
               (for-each (lambda (file) (install-file file bin)) tools)
               (copy-recursively "../doc" docdir)))))))
    (home-page "http://abacus.gene.ucl.ac.uk/software/paml.html")
    (synopsis "Phylogentic analysis by maximum likelihood")
    (description "PAML (for Phylogentic Analysis by Maximum Likelihood)
contains a few programs for model fitting and phylogenetic tree reconstruction
using nucleotide or amino-acid sequence data.")
    ;; GPLv3 only
    (license license:gpl3)))

(define-public segemehl
  (package
    (name "segemehl")
    (version "0.3.4")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://www.bioinf.uni-leipzig.de/Software"
                                  "/segemehl/downloads/segemehl-"
                                  version ".tar.gz"))
              (sha256
               (base32
                "0lbzbb7i8zadsn9b99plairhq6s2h1z8qdn6n7djclfis01nycz4"))))
    (build-system gnu-build-system)
    (arguments
     `(#:make-flags
       (list (string-append "CC=" ,(cc-for-target))
             "all")
       #:tests? #false ; there are none
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         ;; There is no installation target
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out  (assoc-ref outputs "out"))
                    (bin  (string-append out "/bin"))
                    (exes (list "segemehl.x" "haarz.x")))
               (mkdir-p bin)
               (for-each (lambda (exe)
                           (install-file exe bin))
                         exes)))))))
    (inputs
     (list htslib ncurses zlib))
    (native-inputs
     (list pkg-config))
    (home-page "https://www.bioinf.uni-leipzig.de/Software/segemehl")
    (synopsis "Map short sequencer reads to reference genomes")
    (description "Segemehl is software to map short sequencer reads to
reference genomes.  Segemehl implements a matching strategy based on enhanced
suffix arrays (ESA).  It accepts fasta and fastq queries (gzip'ed and
bgzip'ed).  In addition to the alignment of reads from standard DNA- and
RNA-seq protocols, it also allows the mapping of bisulfite converted
reads (Lister and Cokus) and implements a split read mapping strategy.  The
output of segemehl is a SAM or BAM formatted alignment file.")
    (license license:gpl3+)))

(define-public kallisto
  (package
    (name "kallisto")
    (version "0.50.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/pachterlab/kallisto")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zfs79mv75599cf9d7d3c5a3s8idgz9qvl4qfzhvhbd87y3dv7p0"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:tests? #false          ;no "check" target
      ;; The build system attempts to link libbifrost.a into the kallisto
      ;; executable before the library has been built.
      #:parallel-build? #false))
    (inputs (list hdf5 zlib))
    (home-page "https://pachterlab.github.io/kallisto/")
    (synopsis "Near-optimal RNA-Seq quantification")
    (description
     "Kallisto is a program for quantifying abundances of transcripts from
RNA-Seq data, or more generally of target sequences using high-throughput
sequencing reads.  It is based on the novel idea of pseudoalignment for
rapidly determining the compatibility of reads with targets, without the need
for alignment.  Pseudoalignment of reads preserves the key information needed
for quantification, and kallisto is therefore not only fast, but also as
accurate as existing quantification tools.")
    (license license:bsd-2)))

(define-public libgff
  (package
    (name "libgff")
    (version "2.0.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/COMBINE-lab/libgff")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ds9r22y8bl1rj7bhl0003kgmm6aam7g8l41mnjfrzw15d9zf9k4"))))
    (build-system cmake-build-system)
    (arguments `(#:tests? #f))          ; no tests included
    (home-page "https://github.com/COMBINE-lab/libgff")
    (synopsis "Parser library for reading/writing GFF files")
    (description "This is a simple \"libraryfication\" of the GFF/GTF parsing
code that is used in the Cufflinks codebase.  The goal of this library is to
provide this functionality without the necessity of drawing in a heavy-weight
dependency like SeqAn.")
    (license (license:x11-style "https://www.boost.org/LICENSE_1_0.txt"))))

(define-public sailfish
  (package
    (name "sailfish")
    (version "0.10.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/kingsfordgroup/sailfish")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1amcc5hqvsl42hg4x19bi9vy47cl874s0lw1fmi0hwsdk9i8c03v"))
              (modules '((guix build utils)))
              ;; Delete bundled headers for eigen3.
              (snippet
               '(delete-file-recursively "include/eigen3/"))))
    (build-system cmake-build-system)
    (arguments
     `(#:configure-flags
       ,#~(list (string-append "-DBOOST_INCLUDEDIR="
                               #$(this-package-input "boost")
                               "/include/")
                (string-append "-DBOOST_LIBRARYDIR="
                               #$(this-package-input "boost")
                               "/lib/")
                (string-append "-DBoost_LIBRARIES="
                               "-lboost_iostreams "
                               "-lboost_filesystem "
                               "-lboost_system "
                               "-lboost_thread "
                               "-lboost_timer "
                               "-lboost_chrono "
                               "-lboost_program_options")
                "-DBoost_FOUND=TRUE"
                ;; Don't download RapMap---we already have it!
                "-DFETCHED_RAPMAP=1")
       ;; Tests must be run after installation and the location of the test
       ;; data file must be overridden.  But the tests fail.  It looks like
       ;; they are not really meant to be run.
       #:tests? #f
       #:phases
       (modify-phases %standard-phases
         ;; Boost cannot be found, even though it's right there.
         (add-after 'unpack 'do-not-look-for-boost
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "CMakeLists.txt"
               (("find_package\\(Boost 1\\.53\\.0") "#"))))
         (add-after 'unpack 'do-not-assign-to-macro
           (lambda _
             (substitute* "include/spdlog/details/format.cc"
               (("const unsigned CHAR_WIDTH = 1;") ""))))
         (add-after 'unpack 'prepare-rapmap
           (lambda* (#:key inputs #:allow-other-keys)
             (let ((src "external/install/src/rapmap/")
                   (include "external/install/include/rapmap/")
                   (rapmap (assoc-ref inputs "rapmap")))
               (mkdir-p src)
               (mkdir-p include)
               (for-each (lambda (file)
                           (install-file file src))
                         (find-files (string-append rapmap "/src") "\\.(c|cpp)"))
               (copy-recursively (string-append rapmap "/include") include))))
         (add-after 'unpack 'use-system-libraries
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* '("src/SailfishIndexer.cpp"
                            "src/SailfishUtils.cpp"
                            "src/SailfishQuantify.cpp"
                            "src/FASTAParser.cpp"
                            "include/PCA.hpp"
                            "include/SailfishUtils.hpp"
                            "include/SailfishIndex.hpp"
                            "include/CollapsedEMOptimizer.hpp"
                            "src/CollapsedEMOptimizer.cpp")
               (("#include \"jellyfish/config.h\"") ""))
             (substitute* "src/CMakeLists.txt"
               (("\\$\\{GAT_SOURCE_DIR\\}/external/install/include/jellyfish-2.2..")
                (search-input-directory
                 inputs
                 (string-append "/include/jellyfish-" ,(package-version jellyfish))))
               (("\\$\\{GAT_SOURCE_DIR\\}/external/install/lib/libjellyfish-2.0.a")
                (search-input-file inputs
                                   "/lib/libjellyfish-2.0.a"))
               (("\\$\\{GAT_SOURCE_DIR\\}/external/install/lib/libdivsufsort.a")
                (search-input-file inputs
                                   "/lib/libdivsufsort.so"))
               (("\\$\\{GAT_SOURCE_DIR\\}/external/install/lib/libdivsufsort64.a")
                (search-input-file inputs
                                   "/lib/libdivsufsort64.so")))
             (substitute* "CMakeLists.txt"
               ;; Don't prefer static libs
               (("SET\\(CMAKE_FIND_LIBRARY_SUFFIXES.*") "")
               (("find_package\\(Jellyfish.*") "")
               (("ExternalProject_Add\\(libjellyfish") "message(")
               (("ExternalProject_Add\\(libgff") "message(")
               (("ExternalProject_Add\\(libsparsehash") "message(")
               (("ExternalProject_Add\\(libdivsufsort") "message("))

             ;; Ensure that Eigen headers can be found
             (setenv "CPLUS_INCLUDE_PATH"
                     (string-append (search-input-directory
                                     inputs "/include/eigen3")
                                    ":"
                                    (or (getenv "CPLUS_INCLUDE_PATH") ""))))))))
    (inputs
     `(("boost" ,boost-1.83)
       ("eigen" ,eigen)
       ("jemalloc" ,jemalloc)
       ("jellyfish" ,jellyfish)
       ("sparsehash" ,sparsehash)
       ("rapmap" ,(origin
                    (method git-fetch)
                    (uri (git-reference
                          (url "https://github.com/COMBINE-lab/RapMap")
                          (commit (string-append "sf-v" version))))
                    (file-name (string-append "rapmap-sf-v" version "-checkout"))
                    (sha256
                     (base32
                      "1hv79l5i576ykv5a1srj2p0q36yvyl5966m0fcy2lbi169ipjakf"))
                    (modules '((guix build utils)))
                    ;; These files are expected to be excluded.
                    (snippet
                     '(begin (delete-file-recursively "include/spdlog")
                             (for-each delete-file '("include/xxhash.h"
                                                     "src/xxhash.c"))))))
       ("libdivsufsort" ,libdivsufsort)
       ("libgff" ,libgff)
       ("tbb" ,tbb-2020)
       ("zlib" ,zlib)))
    (native-inputs
     (list pkg-config))
    (home-page "https://www.cs.cmu.edu/~ckingsf/software/sailfish/")
    (synopsis "Mapping-based isoform quantification from RNA-Seq reads")
    (description "Sailfish is a tool for genomic transcript quantification
from RNA-seq data.  It requires a set of target transcripts (either from a
reference or de-novo assembly) to quantify.  All you need to run sailfish is a
fasta file containing your reference transcripts and a (set of) fasta/fastq
file(s) containing your reads.")
    (license license:gpl3+)))

(define libstadenio-for-salmon
  (package
    (name "libstadenio")
    (version "1.14.8")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/COMBINE-lab/staden-io_lib")
                    (commit (string-append "v" version))))
              (file-name (string-append name "-" version "-checkout"))
              (sha256
               (base32
                "1x8kxxqxl892vwfbprlbyfwkkv7c34ggkc94892x9x0g37x5nbwx"))))
    (build-system gnu-build-system)
    (arguments '(#:parallel-tests? #f)) ; not supported
    (inputs
     `(("zlib" ,zlib)))
    (native-inputs
     `(("perl" ,perl)))                 ; for tests
    (home-page "https://github.com/COMBINE-lab/staden-io_lib")
    (synopsis "General purpose trace and experiment file library")
    (description "This package provides a library of file reading and writing
code to provide a general purpose Trace file (and Experiment File) reading
interface.

The following file formats are supported:

@enumerate
@item SCF trace files
@item ABI trace files
@item ALF trace files
@item ZTR trace files
@item SFF trace archives
@item SRF trace archives
@item Experiment files
@item Plain text files
@item SAM/BAM sequence files
@item CRAM sequence files
@end enumerate\n")
    (license license:bsd-3)))

(define-public salmon
  (package
    (name "salmon")
    (version "1.10.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/COMBINE-lab/salmon")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vii7nlx0ddn62qniwx4abfj6ajwl3w6mizavpa0mkhpxnm06rqw"))
              (modules '((guix build utils)))
              (snippet
               ;; Delete bundled headers for eigen3.
               '(delete-file-recursively "include/eigen3/"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:configure-flags
      #~(list (string-append "-Dlibgff_DIR="
                             #$(this-package-input "libgff") "/lib")
              "-DCMAKE_CXX_FLAGS=\"-DHAVE_NUMERIC_LIMITS128=1\""
              "-Dlibgff_FOUND=TRUE"
              "-DTBB_FOUND=TRUE"
              #$(string-append "-DTBB_VERSION=" (package-version tbb))
              "-DFETCHED_PUFFERFISH=TRUE"
              "-DUSE_SHARED_LIBS=TRUE")
       #:phases
       '(modify-phases %standard-phases
          (add-after 'unpack 'prepare-pufferfish
            (lambda* (#:key inputs #:allow-other-keys)
              (copy-recursively (assoc-ref inputs "pufferfish")
                                "external/pufferfish")
              ;; This test isn't working correctly, so compilation aborts.
              (substitute* "external/pufferfish/include/string_view.hpp"
                (("#if __has_include\\(<string_view>\\)")
                 "#if 0"))
              (let ((headers "external/install/pufferfish/include/pufferfish")
                    (source "external/install/src/pufferfish"))
                (mkdir-p headers)
                (mkdir-p source)
                (for-each (lambda (file)
                            (install-file (string-append "external/pufferfish/include/" file)
                                          headers))
                          (list "ProgOpts.hpp" "BooPHF.hpp" "SpinLock.hpp"
                                "Kmer.hpp" "CanonicalKmer.hpp" "string_view.hpp"
                                "CanonicalKmerIterator.hpp"
                                "PufferfishBaseIndex.hpp"
                                "PufferfishIndex.hpp"
                                "PufferfishSparseIndex.hpp"
                                "PufferfishLossyIndex.hpp"
                                "PufferfishTypes.hpp"
                                "rank9b.hpp" "rank9sel.hpp" "macros.hpp"
                                "select.hpp" "Util.hpp"
                                "PairedAlignmentFormatter.hpp"
                                "SelectiveAlignmentUtils.hpp"
                                "PuffAligner.hpp" "MemCollector.hpp"
                                "MemChainer.hpp" "CommonTypes.hpp"
                                "SAMWriter.hpp" "PufferfishConfig.hpp"
                                "BulkChunk.hpp" "BinWriter.hpp"))

                (for-each (lambda (dir)
                            (copy-recursively
                             (string-append "external/pufferfish/include/" dir)
                             (string-append headers "/" dir)))
                          (list "libdivide"
                                "ksw2pp"
                                "compact_vector"
                                "itlib"
                                "metro"
                                "chobo"
                                "sparsepp"
                                "simde"
                                "tsl"))
                (copy-recursively
                 (string-append "external/pufferfish/src/metro/")
                 (string-append source "/metro"))
                (install-file
                 (string-append "external/pufferfish/src/rank9b.cpp")
                 source)

                ;; Do not complain about not having built libtbb
                (substitute* "external/pufferfish/external/twopaco/CMakeLists.txt"
                  (("add_dependencies.*") "")))))
          (add-after 'unpack 'do-not-phone-home
            (lambda _
              (substitute* "src/Salmon.cpp"
                (("getVersionMessage\\(\\)") "\"\""))))
          (add-after 'unpack 'use-system-libraries
            (lambda* (#:key inputs #:allow-other-keys)
              ;; Ensure that all headers can be found
              (setenv "CPLUS_INCLUDE_PATH"
                      (string-append (or (getenv "CPLUS_INCLUDE_PATH") "")
                                     ":"
                                     (getcwd) "/external/install/pufferfish/include:"
                                     (assoc-ref inputs "eigen")
                                     "/include/eigen3"))))
          (add-after 'unpack 'fix-error-message-in-tests
            (lambda _
              (substitute* "cmake/TestSalmonQuasi.cmake"
                (("SALMON_QUASI_INDEX_COMMAND")
                 "SALMON_QUASI_INDEX_CMD")))))))
    (inputs
     (list boost-1.83
           bzip2
           cereal
           curl
           eigen
           htscodecs
           jemalloc
           libgff
           tbb
           libstadenio-for-salmon
           xz
           zlib))
    (native-inputs
     `(("pkg-config" ,pkg-config)
       ("pufferfish" ,(origin
                        (method git-fetch)
                        (uri (git-reference
                              (url "https://github.com/COMBINE-lab/pufferfish")
                              (commit (string-append "salmon-v" version))))
                        (file-name (git-file-name "pufferfish" version))
                        (sha256
                         (base32
                          "0h12h4wv7k3if1m6as7wvrbd7djm0zsg1vs0xn84bnrswwv5z2l3"))))))
    (home-page "https://github.com/COMBINE-lab/salmon")
    (synopsis "Quantification from RNA-seq reads using lightweight alignments")
    (description "Salmon is a program to produce highly-accurate,
transcript-level quantification estimates from RNA-seq data.  Salmon achieves
its accuracy and speed via a number of different innovations, including the
use of lightweight alignments (accurate but fast-to-compute proxies for
traditional read alignments) and massively-parallel stochastic collapsed
variational inference.")
    (license license:gpl3+)))

(define-public scallop
  (package
    (name "scallop")
    (version "0.10.5")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/Kingsford-Group/scallop"
                                  "/releases/download/v" version
                                  "/scallop-" version ".tar.gz"))
              (sha256
               (base32
                "0ylkhr5hsmw7bsdszrjz9yqknsijh2fl5n6rjqmdmcdky5hkr7mh"))))
    (build-system gnu-build-system)
    (inputs
     (list boost htslib))
    (home-page "https://github.com/Kingsford-Group/scallop")
    (synopsis "Reference-based transcriptome assembler for RNA-seq")
    (description "Scallop is a reference-based transcript assembler.  Scallop
features its high accuracy in assembling multi-exon transcripts as well as
lowly expressed transcripts.")
    (license license:bsd-3)))

(define-public python-fanc
  (package
    (name "python-fanc")
    (version "0.9.28")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/vaquerizaslab/fanc")
                    ;; There are no tags.  This commit corresponds to
                    ;; version 0.9.28.
                    (commit "d5d86085c920a4dca6e5f6be4857129d718243cc")))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0pfl35ancd0izw9w2dp85lznq9rsyv2nv5f5la6dnbvj20d1m06d"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-m" "not longrunning"
             ;; XXX: some of the tests here just take forever
             "--ignore=fanc/test/test_matrix.py"
             "--ignore=fanc/test/test_pairs.py")))
    (propagated-inputs
     (list python-biopython
           python-cooler
           python-deprecated
           python-future
           python-genomic-regions
           python-gridmap
           python-h5py
           python-imageio
           python-intervaltree
           python-matplotlib
           python-msgpack
           python-msgpack-numpy
           python-numpy
           python-pandas
           python-pillow
           python-progressbar2
           python-pybedtools
           python-pybigwig
           python-pysam
           python-pywavelets
           python-pyyaml
           python-scikit-image
           python-scikit-learn
           python-scipy
           python-seaborn
           python-tables
           python-tifffile))
    (native-inputs
     (list python-cython python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/vaquerizaslab/fanc")
    (synopsis "Framework for the analysis of C-data")
    (description
     "FAN-C provides a pipeline for analysing Hi-C data starting at
mapped paired-end sequencing reads.")
    (license license:gpl3+)))

(define-deprecated-package fanc
  python-fanc)

(define-public python-genomic-regions
  (package
    (name "python-genomic-regions")
    (version "0.0.10")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "genomic_regions" version))
              (sha256
               (base32
                "0hz811iyd1prml1r90qyzimmwyjwycwkjqw4vnl12bxy61rfzjz5"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list python-future
           python-intervaltree
           python-numpy
           python-pandas
           python-pybedtools
           python-pybigwig
           python-pytest
           python-msgpack-numpy
           python-cython
           python-msgpack
           python-pysam))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://pypi.org/project/genomic-regions/")
    (synopsis "Consistently handle genomic regions")
    (description "This package aims to simplify working with genomic region /
interval data by providing a common interface that lets you access a wide
selection of file types and formats for handling genomic region data---all
using the same syntax.")
    (license license:expat)))

(define-public python-goatools
  (package
    (name "python-goatools")
    (version "1.3.1")
    ;; Pypi tarball doesn't include test files.
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/tanghaibao/goatools")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0a295zk9jc8kny5vnka63q3gbksins42kcmgvskf8xy7hkca7cmq"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; Almost all tests require access to the internet.
      #:tests? #false
      #:test-flags
      ;; These have syntax errors.
      '(list "--ignore=tests/test_i195_sgd_gaf.py"
             "--ignore=tests/test_i206.py"
             "--ignore=tests/test_setup_dirs.py")))
    (propagated-inputs (list python-docopt
                             python-numpy
                             python-openpyxl
                             python-pandas
                             python-pydot
                             python-requests
                             python-scipy
                             python-setuptools
                             python-statsmodels
                             python-xlsxwriter))
    (native-inputs (list python-pytest python-wheel))
    (home-page "https://github.com/tanghaibao/goatools")
    (synopsis "Python scripts to find enrichment of GO terms")
    (description "Python scripts to find enrichment of GO terms.  In addition,
this package is used for processing the obo-formatted file from Gene Ontology
website.  The data structure is a directed acyclic graph that allows easy
traversal from leaf to root.")
    (license license:bsd-2)))

(define-public python-loompy
  (package
    (name "python-loompy")
    (version "3.0.7")
    ;; The tarball on Pypi does not include the tests.
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/linnarsson-lab/loompy")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0xmw2yv1y3y7vh5jcbrmlkn43nmfs0pf6z78k1yxqs3qy248m9b0"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         ;; See https://github.com/linnarsson-lab/loompy/issues/169
         (add-after 'unpack 'fix-h5py-error
           (lambda _
             (substitute* "tests/test_file_attribute_manager.py"
               (("h5py.File\\(f.name\\)")
                "h5py.File(f.name, 'a')"))))
         (add-after 'unpack 'numpy-compatibility
           (lambda _
             (substitute* "tests/test_connection.py"
               (("np.int") "int"))))
         ;; Numba needs a writable dir to cache functions.
         (add-before 'check 'set-numba-cache-dir
           (lambda _
             (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs
     (list python-click
           python-h5py
           python-numba
           python-numpy
           python-numpy-groupies
           python-pandas
           python-scipy))
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/linnarsson-lab/loompy")
    (synopsis "Work with .loom files for single-cell RNA-seq data")
    (description "The loom file format is an efficient format for very large
omics datasets, consisting of a main matrix, optional additional layers, a
variable number of row and column annotations.  Loom also supports sparse
graphs.  This library makes it easy to work with @file{.loom} files for
single-cell RNA-seq data.")
    (license license:bsd-3)))

(define-public python-biothings-client
  (package
    (name "python-biothings-client")
    (version "0.2.6")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "biothings_client" version))
       (sha256
        (base32 "0bccs37d5saxn5xsd2rfpkrnc5a120xs3ibizai66fgvp1vxbnc4"))))
    (build-system python-build-system)
    (arguments `(#:tests? #false)) ; require internet access
    (propagated-inputs (list python-requests))
    (home-page "https://github.com/biothings/biothings_client.py")
    (synopsis "Python client for BioThings API services")
    (description "This package provides a Python client for BioThings
API services.")
    (license license:bsd-3)))

(define-public python-mgatk
  (package
    (name "python-mgatk")
    (version "0.7.0")
    (source
     (origin
       (method git-fetch) ; no tests in PyPI archive
       (uri (git-reference
             (url "https://github.com/caleblareau/mgatk")
             ;; There is no tag for 0.7.0, but this is the commit
             ;; corresponding to the version bump, see
             ;; <https://github.com/caleblareau/mgatk/issues/101>.
             (commit "8ffeac8476564049ef51b4d4d40eed452ae2fc38")))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1qspzglj487bpyg8wpc29fjr8mj993q8w3jrdhylggiqpjx4l607"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list python-biopython
           python-click
           python-numpy
           python-optparse-pretty
           python-pandas
           python-pysam
           python-regex
           python-ruamel.yaml
           python-setuptools
           snakemake))
    (native-inputs
     (list python-pytest python-wheel))
    (home-page "https://github.com/caleblareau/mgatk")
    (synopsis "Mitochondrial genome analysis toolkit")
    (description "This package is a Python-based command line interface for
processing .bam files with mitochondrial reads and generating high-quality
heteroplasmy estimation from sequencing data.  The mgatk package places a
special emphasis on mitochondrial genotypes generated from single-cell
genomics data, primarily @acronym{mtscATAC-seq, mitochondrial single-cell
ATAC-sequence}, but is generally applicable across other assays.")
    (license license:expat)))

(define-public python-multicore-tsne
  (let ((commit "c1dbf84eb550980876d8ed822af4e9dfd21c5e05")
        (revision "1"))
    (package
      (name "python-multicore-tsne")
      (version (git-version "0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/DmitryUlyanov/Multicore-TSNE.git")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0dj805mmd35rfkj7jpkhvnj47x51lqpvascjwyaqxj7pwk6sbkzv"))))
      (build-system pyproject-build-system)
      ;; Tests require an old version of scikit-learn.  See
      ;; https://github.com/DmitryUlyanov/Multicore-TSNE/issues/97.
      (arguments (list #:tests? #false))
      (propagated-inputs
       (list python-cffi python-numpy python-packaging))
      (native-inputs
       (list cmake-minimal
             python-scipy
             python-scikit-learn
             python-setuptools
             python-wheel))
      (home-page "https://github.com/DmitryUlyanov/Multicore-TSNE")
      (synopsis "Parallel t-SNE implementation with Python and Torch wrappers")
      (description
       "This package contains a multicore Barnes-Hut implementation of the
t-SNE algorithm.  The implementation is described here:
@url{http://lvdmaaten.github.io/publications/papers/JMLR_2014.pdf}.")
      (license license:bsd-3))))

(define-public python-multivelo
  (package
    (name "python-multivelo")
    (version "0.1.2")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "multivelo" version))
              (sha256
               (base32
                "1b4qyngwagh5sc2ygyfqyirg63myzh1g1glk03a1ykxfii32cjlp"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f                       ;pypi source does not contain tests
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'sanity-check 'set-env
            (lambda _
              ;; numba RuntimeError: cannot cache function 'rdist'
              (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (native-inputs (list python-pypa-build))
    (propagated-inputs
     (list python-anndata
           python-h5py
           python-ipywidgets
           python-joblib
           python-loompy
           python-matplotlib
           python-numba
           python-numpy
           python-pandas
           python-scanpy
           python-scikit-learn
           python-scipy
           python-seaborn
           python-tqdm
           python-umap-learn
           scvelo))
    (home-page "https://github.com/welch-lab/MultiVelo")
    (synopsis "Velocity inference from single-cell multi-omic data")
    (description "MultiVelo uses a probabilistic latent variable model to
estimate the switch time and rate parameters of gene regulation, providing a
quantitative summary of the temporal relationship between epigenomic and
transcriptomic changes.")
    (license license:bsd-3)))

(define-public python-mygene
  (package
    (name "python-mygene")
    (version "3.2.2")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "mygene" version))
       (sha256
        (base32 "1snszwdgfygchxshcbry3b5pbcw3g1isp8dw46razxccqaxwlag7"))))
    (build-system python-build-system)
    (propagated-inputs
     (list python-biothings-client))
    (home-page "https://github.com/biothings/mygene.py")
    (synopsis "Python Client for MyGene.Info services")
    (description "MyGene.Info provides simple-to-use REST web services
to query/retrieve gene annotation data.  It's designed with simplicity
and performance emphasized.  Mygene is a Python wrapper to access
MyGene.Info services.")
    (license license:bsd-3)))

;; We cannot use the latest commit because it requires Java 9.
(define-public java-forester
  (let ((commit "86b07efe302d5094b42deed9260f719a4c4ac2e6")
        (revision "1"))
    (package
      (name "java-forester")
      (version (string-append "0-" revision "." (string-take commit 7)))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/cmzmasek/forester")
                      (commit commit)))
                (file-name (string-append name "-" version "-checkout"))
                (sha256
                 (base32
                  "0vxavc1yrf84yrnf20dq26hi0lglidk8d382xrxsy4qmlbjd276z"))
                (modules '((guix build utils)))
                (snippet
                 '(begin
                    ;; Delete bundled jars and pre-built classes
                    (delete-file-recursively "forester/java/resources")
                    (delete-file-recursively "forester/java/classes")
                    (for-each delete-file (find-files "forester/java/" "\\.jar$"))
                    ;; Delete bundled applications
                    (delete-file-recursively "forester_applications")
                    #t))))
      (build-system ant-build-system)
      (arguments
       `(#:tests? #f ; there are none
         #:jdk ,icedtea-8
         #:modules ((guix build ant-build-system)
                    (guix build utils)
                    (guix build java-utils)
                    (sxml simple)
                    (sxml transform))
         #:phases
         (modify-phases %standard-phases
           (add-after 'unpack 'chdir
             (lambda _ (chdir "forester/java") #t))
           (add-after 'chdir 'fix-dependencies
             (lambda _
               (chmod "build.xml" #o664)
               (call-with-output-file "build.xml.new"
                 (lambda (port)
                   (sxml->xml
                    (pre-post-order
                     (with-input-from-file "build.xml"
                       (lambda _ (xml->sxml #:trim-whitespace? #t)))
                     `(;; Remove all unjar tags to avoid repacking classes.
                       (unjar     . ,(lambda _ '()))
                       (*default* . ,(lambda (tag . kids) `(,tag ,@kids)))
                       (*text*    . ,(lambda (_ txt) txt))))
                    port)))
               (rename-file "build.xml.new" "build.xml")
               #t))
           ;; FIXME: itext is difficult to package as it depends on a few
           ;; unpackaged libraries.
           (add-after 'chdir 'remove-dependency-on-unpackaged-itext
             (lambda _
               (delete-file "src/org/forester/archaeopteryx/PdfExporter.java")
               (substitute* "src/org/forester/archaeopteryx/MainFrame.java"
                 (("pdf_written_to = PdfExporter.*")
                  "throw new IOException(\"PDF export is not available.\");"))
               #t))
           ;; There is no install target
           (replace 'install (install-jars ".")))))
      (propagated-inputs
       (list java-commons-codec java-openchart2))
      (home-page "https://sites.google.com/site/cmzmasek/home/software/forester")
      (synopsis "Phylogenomics libraries for Java")
      (description "Forester is a collection of Java libraries for
phylogenomics and evolutionary biology research.  It includes support for
reading, writing, and exporting phylogenetic trees.")
      (license license:lgpl2.1+))))

(define-public java-forester-1.005
  (package
    (name "java-forester")
    (version "1.005")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://repo1.maven.org/maven2/"
                                  "org/biojava/thirdparty/forester/"
                                  version "/forester-" version "-sources.jar"))
              (file-name (string-append name "-" version ".jar"))
              (sha256
               (base32
                "04r8qv4rk3p71z4ajrvp11py1z46qrx0047j3zzs79s6lnsm3lcv"))))
    (build-system ant-build-system)
    (arguments
     `(#:tests? #f ; there are none
       #:jdk ,icedtea-8
       #:modules ((guix build ant-build-system)
                  (guix build utils)
                  (guix build java-utils)
                  (sxml simple)
                  (sxml transform))
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'fix-dependencies
           (lambda* (#:key inputs #:allow-other-keys)
             (call-with-output-file "build.xml"
               (lambda (port)
                 (sxml->xml
                  (pre-post-order
                   (with-input-from-file "src/build.xml"
                     (lambda _ (xml->sxml #:trim-whitespace? #t)))
                   `(;; Remove all unjar tags to avoid repacking classes.
                     (unjar     . ,(lambda _ '()))
                     (*default* . ,(lambda (tag . kids) `(,tag ,@kids)))
                     (*text*    . ,(lambda (_ txt) txt))))
                  port)))
             (copy-file (assoc-ref inputs "synth_look_and_feel_1.xml")
                        "synth_look_and_feel_1.xml")
             (copy-file (assoc-ref inputs "phyloxml.xsd")
                        "phyloxml.xsd")
             (substitute* "build.xml"
               (("../resources/synth_laf/synth_look_and_feel_1.xml")
                "synth_look_and_feel_1.xml")
               (("../resources/phyloxml_schema/1.10/phyloxml.xsd")
                "phyloxml.xsd"))
             #t))
         ;; FIXME: itext is difficult to package as it depends on a few
         ;; unpackaged libraries.
         (add-after 'unpack 'remove-dependency-on-unpackaged-itext
           (lambda _
             (delete-file "src/org/forester/archaeopteryx/PdfExporter.java")
             (substitute* '("src/org/forester/archaeopteryx/MainFrame.java"
                            "src/org/forester/archaeopteryx/MainFrameApplication.java")
               (("pdf_written_to = PdfExporter.*")
                "throw new IOException(\"PDF export is not available.\"); /*")
               ((".getPrintSizeX\\(\\), getOptions\\(\\).getPrintSizeY\\(\\) \\);") "*/")
               (("getCurrentTreePanel\\(\\).getHeight\\(\\) \\);") "*/"))
             #t))
         (add-after 'unpack 'delete-pre-built-classes
           (lambda _ (delete-file-recursively "src/classes") #t))
         ;; There is no install target
         (replace 'install (install-jars ".")))))
    (propagated-inputs
     (list java-commons-codec java-openchart2))
    ;; The source archive does not contain the resources.
    (native-inputs
     `(("phyloxml.xsd"
        ,(origin
           (method url-fetch)
           (uri (string-append "https://raw.githubusercontent.com/cmzmasek/forester/"
                               "b61cc2dcede0bede317db362472333115756b8c6/"
                               "forester/resources/phyloxml_schema/1.10/phyloxml.xsd"))
           (file-name (string-append name "-phyloxml-" version ".xsd"))
           (sha256
            (base32
             "1zxc4m8sn4n389nqdnpxa8d0k17qnr3pm2y5y6g6vh4k0zm52npv"))))
       ("synth_look_and_feel_1.xml"
        ,(origin
           (method url-fetch)
           (uri (string-append "https://raw.githubusercontent.com/cmzmasek/forester/"
                               "29e04321615da6b35c1e15c60e52caf3f21d8e6a/"
                               "forester/java/classes/resources/"
                               "synth_look_and_feel_1.xml"))
           (file-name (string-append name "-synth-look-and-feel-" version ".xml"))
           (sha256
            (base32
             "1gv5602gv4k7y7713y75a4jvj7i9s7nildsbdl7n9q10sc2ikg8h"))))))
    (home-page "https://sites.google.com/site/cmzmasek/home/software/forester")
    (synopsis "Phylogenomics libraries for Java")
    (description "Forester is a collection of Java libraries for
phylogenomics and evolutionary biology research.  It includes support for
reading, writing, and exporting phylogenetic trees.")
    (license license:lgpl2.1+)))

(define-public java-biojava-core
  (package
    (name "java-biojava-core")
    (version "4.2.11")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/biojava/biojava")
                    (commit (string-append "biojava-" version))))
              (file-name (string-append name "-" version "-checkout"))
              (sha256
               (base32
                "1bvryh2bpsvash8ln79cmc9sqm8qw72hz4xzwqxcrjm8ssxszhqk"))))
    (build-system ant-build-system)
    (arguments
     `(#:jdk ,icedtea-8
       #:jar-name "biojava-core.jar"
       #:source-dir "biojava-core/src/main/java/"
       #:test-dir "biojava-core/src/test"
       ;; These tests seem to require internet access.
       #:test-exclude (list "**/SearchIOTest.java"
                            "**/BlastXMLParserTest.java"
                            "**/GenbankCookbookTest.java"
                            "**/GenbankProxySequenceReaderTest.java")
       #:phases
       (modify-phases %standard-phases
         (add-before 'build 'copy-resources
           (lambda _
             (copy-recursively "biojava-core/src/main/resources"
                               "build/classes")
             #t))
         (add-before 'check 'copy-test-resources
           (lambda _
             (copy-recursively "biojava-core/src/test/resources"
                               "build/test-classes")
             #t)))))
    (propagated-inputs
     (list java-log4j-api java-log4j-core java-slf4j-api
           java-slf4j-simple))
    (native-inputs
     (list java-junit java-hamcrest-core))
    (home-page "https://biojava.org")
    (synopsis "Core libraries of Java framework for processing biological data")
    (description "BioJava is a project dedicated to providing a Java framework
for processing biological data.  It provides analytical and statistical
routines, parsers for common file formats, reference implementations of
popular algorithms, and allows the manipulation of sequences and 3D
structures.  The goal of the biojava project is to facilitate rapid
application development for bioinformatics.

This package provides the core libraries.")
    (license license:lgpl2.1+)))

(define-public java-biojava-phylo
  (package (inherit java-biojava-core)
    (name "java-biojava-phylo")
    (build-system ant-build-system)
    (arguments
     `(#:jdk ,icedtea-8
       #:jar-name "biojava-phylo.jar"
       #:source-dir "biojava-phylo/src/main/java/"
       #:test-dir "biojava-phylo/src/test"
       #:phases
       (modify-phases %standard-phases
         (add-before 'build 'copy-resources
           (lambda _
             (copy-recursively "biojava-phylo/src/main/resources"
                               "build/classes")
             #t))
         (add-before 'check 'copy-test-resources
           (lambda _
             (copy-recursively "biojava-phylo/src/test/resources"
                               "build/test-classes")
             #t)))))
    (propagated-inputs
     (list java-log4j-api
           java-log4j-core
           java-slf4j-api
           java-slf4j-simple
           java-biojava-core
           java-forester))
    (native-inputs
     (list java-junit java-hamcrest-core))
    (home-page "https://biojava.org")
    (synopsis "Biojava interface to the forester phylogenomics library")
    (description "The phylo module provides a biojava interface layer to the
forester phylogenomics library for constructing phylogenetic trees.")))

(define-public java-biojava-alignment
  (package (inherit java-biojava-core)
    (name "java-biojava-alignment")
    (build-system ant-build-system)
    (arguments
     `(#:jdk ,icedtea-8
       #:jar-name "biojava-alignment.jar"
       #:source-dir "biojava-alignment/src/main/java/"
       #:test-dir "biojava-alignment/src/test"
       #:phases
       (modify-phases %standard-phases
         (add-before 'build 'copy-resources
           (lambda _
             (copy-recursively "biojava-alignment/src/main/resources"
                               "build/classes")
             #t))
         (add-before 'check 'copy-test-resources
           (lambda _
             (copy-recursively "biojava-alignment/src/test/resources"
                               "build/test-classes")
             #t)))))
    (propagated-inputs
     (list java-log4j-api
           java-log4j-core
           java-slf4j-api
           java-slf4j-simple
           java-biojava-core
           java-biojava-phylo
           java-forester))
    (native-inputs
     (list java-junit java-hamcrest-core))
    (home-page "https://biojava.org")
    (synopsis "Biojava API for genetic sequence alignment")
    (description "The alignment module of BioJava provides an API that
contains

@itemize
@item implementations of dynamic programming algorithms for sequence
  alignment;
@item reading and writing of popular alignment file formats;
@item a single-, or multi- threaded multiple sequence alignment algorithm.
@end itemize\n")))

(define-public java-biojava-core-4.0
  (package (inherit java-biojava-core)
    (name "java-biojava-core")
    (version "4.0.0")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/biojava/biojava")
                    (commit (string-append "biojava-" version))))
              (file-name (string-append name "-" version "-checkout"))
              (sha256
               (base32
                "13675f6y9aqi7bi2lk3s1z7a22ynccjiqwa8izh7p97xi9wsfmd8"))))))

(define-public java-biojava-phylo-4.0
  (package (inherit java-biojava-core-4.0)
    (name "java-biojava-phylo")
    (build-system ant-build-system)
    (arguments
     `(#:jdk ,icedtea-8
       #:jar-name "biojava-phylo.jar"
       #:source-dir "biojava-phylo/src/main/java/"
       #:test-dir "biojava-phylo/src/test"
       #:phases
       (modify-phases %standard-phases
         (add-before 'build 'copy-resources
           (lambda _
             (copy-recursively "biojava-phylo/src/main/resources"
                               "build/classes")
             #t))
         (add-before 'check 'copy-test-resources
           (lambda _
             (copy-recursively "biojava-phylo/src/test/resources"
                               "build/test-classes")
             #t)))))
    (propagated-inputs
     (list java-log4j-api
           java-log4j-core
           java-slf4j-api
           java-slf4j-simple
           java-biojava-core-4.0
           java-forester-1.005))
    (native-inputs
     (list java-junit java-hamcrest-core))
    (home-page "https://biojava.org")
    (synopsis "Biojava interface to the forester phylogenomics library")
    (description "The phylo module provides a biojava interface layer to the
forester phylogenomics library for constructing phylogenetic trees.")))

(define-public java-biojava-alignment-4.0
  (package (inherit java-biojava-core-4.0)
    (name "java-biojava-alignment")
    (build-system ant-build-system)
    (arguments
     `(#:jdk ,icedtea-8
       #:jar-name "biojava-alignment.jar"
       #:source-dir "biojava-alignment/src/main/java/"
       #:test-dir "biojava-alignment/src/test"
       #:phases
       (modify-phases %standard-phases
         (add-before 'build 'copy-resources
           (lambda _
             (copy-recursively "biojava-alignment/src/main/resources"
                               "build/classes")
             #t))
         (add-before 'check 'copy-test-resources
           (lambda _
             (copy-recursively "biojava-alignment/src/test/resources"
                               "build/test-classes")
             #t)))))
    (propagated-inputs
     (list java-log4j-api
           java-log4j-core
           java-slf4j-api
           java-slf4j-simple
           java-biojava-core-4.0
           java-biojava-phylo-4.0
           java-forester-1.005))
    (native-inputs
     (list java-junit java-hamcrest-core))
    (home-page "https://biojava.org")
    (synopsis "Biojava API for genetic sequence alignment")
    (description "The alignment module of BioJava provides an API that
contains

@itemize
@item implementations of dynamic programming algorithms for sequence
  alignment;
@item reading and writing of popular alignment file formats;
@item a single-, or multi- threaded multiple sequence alignment algorithm.
@end itemize\n")))

(define-public dropseq-tools
  (package
    (name "dropseq-tools")
    (version "1.13")
    (source
     (origin
       (method url-fetch)
       (uri "http://mccarrolllab.com/download/1276/")
       (file-name (string-append "dropseq-tools-" version ".zip"))
       (sha256
        (base32
         "0yrffckxqk5l8b5xb6z4laq157zd9mdypr2p4b4vq2bhjzi1sj0s"))
       ;; Delete bundled libraries
       (modules '((guix build utils)))
       (snippet
        '(begin
           (for-each delete-file (find-files "jar/lib" "\\.jar$"))
           (delete-file-recursively "3rdParty")))))
    (build-system ant-build-system)
    (arguments
     `(#:tests? #f                      ; test data are not included
       #:test-target "test"
       #:build-target "all"
       #:source-dir "public/src/"
       #:jdk ,icedtea-8
       #:make-flags
       (list ,#~(string-append "-Dpicard.executable.dir="
                               #$(this-package-input "java-picard")
                               "/share/java/"))
       #:modules ((ice-9 match)
                  (srfi srfi-1)
                  (guix build utils)
                  (guix build java-utils)
                  (guix build ant-build-system))
       #:phases
       (modify-phases %standard-phases
         ;; FIXME: fails with "java.io.FileNotFoundException:
         ;; /gnu/store/…-dropseq-tools-1.13/share/java/lib/biojava-alignment.jar"
         (delete 'generate-jar-indices)
         ;; All dependencies must be linked to "lib", because that's where
         ;; they will be searched for when the Class-Path property of the
         ;; manifest is computed.
         (add-after 'unpack 'record-references
	   (lambda* (#:key inputs #:allow-other-keys)
             (mkdir-p "jar/lib")
             (let ((dirs (filter-map (match-lambda
                                       ((name . dir)
                                        (if (and (string-prefix? "java-" name)
                                                 (not (string=? name "java-testng")))
                                            dir #f)))
                                     inputs)))
               (for-each (lambda (jar)
                           (symlink jar (string-append "jar/lib/" (basename jar))))
                         (append-map (lambda (dir) (find-files dir "\\.jar$"))
                                     dirs)))))
         ;; There is no installation target
         (replace 'install
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let* ((out     (assoc-ref outputs "out"))
                    (bin     (string-append out "/bin"))
                    (share   (string-append out "/share/java/"))
                    (lib     (string-append share "/lib/"))
                    (scripts (list "BAMTagHistogram"
                                   "BAMTagofTagCounts"
                                   "BaseDistributionAtReadPosition"
                                   "CollapseBarcodesInPlace"
                                   "CollapseTagWithContext"
                                   "ConvertToRefFlat"
                                   "CreateIntervalsFiles"
                                   "DetectBeadSynthesisErrors"
                                   "DigitalExpression"
                                   "Drop-seq_alignment.sh"
                                   "FilterBAM"
                                   "FilterBAMByTag"
                                   "GatherGeneGCLength"
                                   "GatherMolecularBarcodeDistributionByGene"
                                   "GatherReadQualityMetrics"
                                   "PolyATrimmer"
                                   "ReduceGTF"
                                   "SelectCellsByNumTranscripts"
                                   "SingleCellRnaSeqMetricsCollector"
                                   "TagBamWithReadSequenceExtended"
                                   "TagReadWithGeneExon"
                                   "TagReadWithInterval"
                                   "TrimStartingSequence"
                                   "ValidateReference")))
               (for-each mkdir-p (list bin share lib))
               (install-file "dist/dropseq.jar" share)
               (for-each (lambda (script)
                           (chmod script #o555)
                           (install-file script bin))
                         scripts)
               (substitute* (map (lambda (script)
                                   (string-append bin "/" script))
                                 scripts)
                 (("^java") (which "java"))
                 (("jar_deploy_dir=.*")
                  (string-append "jar_deploy_dir=" share "\n"))))))
         ;; FIXME: We do this after stripping jars because we don't want it to
         ;; copy all these jars and strip them.  We only want to install
         ;; links.  Arguably, this is a problem with the ant-build-system.
         (add-after 'strip-jar-timestamps 'install-links
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out     (assoc-ref outputs "out"))
                    (share   (string-append out "/share/java/"))
                    (lib     (string-append share "/lib/")))
               (for-each (lambda (jar)
                           (symlink (readlink jar)
                                    (string-append lib (basename jar))))
                         (find-files "jar/lib" "\\.jar$"))))))))
    (inputs
     `(("jdk" ,icedtea-8)
       ("java-picard" ,java-picard-2.10.3)
       ("java-log4j-1.2-api" ,java-log4j-1.2-api)
       ("java-commons-math3" ,java-commons-math3)
       ("java-commons-jexl2" ,java-commons-jexl-2)
       ("java-commons-collections4" ,java-commons-collections4)
       ("java-commons-lang2" ,java-commons-lang)
       ("java-commons-io" ,java-commons-io)
       ("java-snappy-1.0.3-rc3" ,java-snappy-1)
       ("java-guava" ,java-guava)
       ("java-la4j" ,java-la4j)
       ("java-biojava-core" ,java-biojava-core-4.0)
       ("java-biojava-alignment" ,java-biojava-alignment-4.0)
       ("java-jdistlib" ,java-jdistlib)
       ("java-simple-xml" ,java-simple-xml)
       ("java-snakeyaml" ,java-snakeyaml)))
    (native-inputs
     (list unzip java-testng))
    (home-page "http://mccarrolllab.com/dropseq/")
    (synopsis "Tools for Drop-seq analyses")
    (description "Drop-seq is a technology to enable biologists to
analyze RNA expression genome-wide in thousands of individual cells at
once.  This package provides tools to perform Drop-seq analyses.")
    (license license:expat)))

(define-public pigx-rnaseq
  (package
    (name "pigx-rnaseq")
    (version "0.1.1")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/BIMSBbioinfo/pigx_rnaseq/"
                                  "releases/download/v" version
                                  "/pigx_rnaseq-" version ".tar.gz"))
              (sha256
               (base32
                "0mlas0srl04mvgsyydm67gg5syijf1k2f6dy7bdqqxc70fywfd08"))))
    (build-system gnu-build-system)
    (arguments
     `(#:parallel-tests? #f             ; not supported
       #:phases
       (modify-phases %standard-phases
         ;; "test.sh" runs the whole pipeline, which takes a long time and
         ;; might fail due to OOM.  The MultiQC is also resource intensive.
         (add-after 'unpack 'disable-resource-intensive-test
           (lambda _
             (substitute* "Makefile.in"
               (("^  tests/test_multiqc/test.sh") "")
               (("^  test.sh") ""))))
         (add-before 'bootstrap 'autoreconf
           (lambda _
             (invoke "autoreconf" "-vif")))
         (add-before 'configure 'set-PYTHONPATH
           (lambda _
             (setenv "PYTHONPATH" (getenv "GUIX_PYTHONPATH"))))
         (add-before 'check 'set-timezone
           ;; The readr package is picky about timezones.
           (lambda* (#:key inputs #:allow-other-keys)
             (setenv "TZ" "UTC+1")
             (setenv "TZDIR"
                     (search-input-directory inputs
                                             "share/zoneinfo"))))
         (add-before 'check 'pre-check
           (lambda _
             ;; Needed for tests
             (setenv "HOME" "/tmp"))))))
    (inputs
     (list coreutils
           sed
           gzip
           snakemake-7
           megadepth
           multiqc
           star-for-pigx
           hisat2
           fastp
           htseq
           samtools
           r-minimal
           r-rmarkdown
           r-genomeinfodb
           r-ggplot2
           r-ggpubr
           r-ggrepel
           r-gprofiler2
           r-deseq2
           r-dt
           r-knitr
           r-pheatmap
           r-corrplot
           r-reshape2
           r-plotly
           r-scales
           r-summarizedexperiment
           r-crosstalk
           r-tximport
           r-rtracklayer
           r-rjson
           salmon
           pandoc
           python-wrapper
           python-deeptools
           python-pyyaml))
    (native-inputs
     (list tzdata automake autoconf))
    (home-page "https://bioinformatics.mdc-berlin.de/pigx/")
    (synopsis "Analysis pipeline for RNA sequencing experiments")
    (description "PiGX RNAseq is an analysis pipeline for preprocessing and
reporting for RNA sequencing experiments.  It is easy to use and produces high
quality reports.  The inputs are reads files from the sequencing experiment,
and a configuration file which describes the experiment.  In addition to
quality control of the experiment, the pipeline produces a differential
expression report comparing samples in an easily configurable manner.")
    (license license:gpl3+)))

(define-public pigx-chipseq
  (package
    (name "pigx-chipseq")
    (version "0.1.0")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/BIMSBbioinfo/pigx_chipseq/"
                                  "releases/download/v" version
                                  "/pigx_chipseq-" version ".tar.gz"))
              (sha256
               (base32
                "008n6drj9q5av86xihxlj4py2c9p3c5z5ld89c3bksrp77zxiy67"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; parts of the tests rely on access to the network
       #:phases
       (modify-phases %standard-phases
         (add-before 'configure 'set-PYTHONPATH
           (lambda _
             (setenv "PYTHONPATH" (getenv "GUIX_PYTHONPATH"))))
         (add-before 'check 'pre-check
           (lambda _
             ;; Needed for tests
             (setenv "HOME" "/tmp"))))))
    (inputs
     (list grep
           coreutils
           r-minimal
           r-argparser
           r-biocparallel
           r-biostrings
           r-chipseq
           r-corrplot
           r-data-table
           r-deseq2
           r-dplyr
           r-dt
           r-genomation
           r-genomicalignments
           r-genomicranges
           r-ggplot2
           r-ggrepel
           r-gprofiler2
           r-heatmaply
           r-hexbin
           r-htmlwidgets
           r-jsonlite
           r-pheatmap
           r-plotly
           r-rmarkdown
           r-rsamtools
           r-rsubread
           r-rtracklayer
           r-s4vectors
           r-stringr
           r-tibble
           r-tidyr
           python-wrapper
           python-pyyaml
           python-magic
           python-xlrd
           trim-galore
           macs
           multiqc
           perl
           pandoc
           fastqc
           bowtie
           idr
           snakemake-7
           samtools
           bedtools
           kentutils))
    (native-inputs
     (list python-pytest))
    (home-page "https://bioinformatics.mdc-berlin.de/pigx/")
    (synopsis "Analysis pipeline for ChIP sequencing experiments")
    (description "PiGX ChIPseq is an analysis pipeline for preprocessing, peak
calling and reporting for ChIP sequencing experiments.  It is easy to use and
produces high quality reports.  The inputs are reads files from the sequencing
experiment, and a configuration file which describes the experiment.  In
addition to quality control of the experiment, the pipeline enables to set up
multiple peak calling analysis and allows the generation of a UCSC track hub
in an easily configurable manner.")
    (license license:gpl3+)))

(define-public pigx-bsseq
  (package
    (name "pigx-bsseq")
    (version "0.1.10")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/BIMSBbioinfo/pigx_bsseq/"
                                  "releases/download/v" version
                                  "/pigx_bsseq-" version ".tar.gz"))
              (sha256
               (base32
                "1vy3mhbrfdnjbhikwg3mgkfnwnzk96a1n27cxrr7gsffpmz9q6wa"))))
    (build-system gnu-build-system)
    (arguments
     (list
      ;; TODO: tests currently require 12+GB of RAM.  See
      ;; https://github.com/BIMSBbioinfo/pigx_bsseq/issues/164
      #:tests? #f
      #:phases
      '(modify-phases %standard-phases
         (add-before 'configure 'set-PYTHONPATH
           (lambda _
             (setenv "PYTHONPATH" (getenv "GUIX_PYTHONPATH"))))
         (add-before 'check 'set-timezone
           ;; The readr package is picky about timezones.
           (lambda* (#:key inputs #:allow-other-keys)
             (setenv "TZ" "UTC+1")
             (setenv "TZDIR"
                     (search-input-directory inputs
                                             "share/zoneinfo"))))
         (add-before 'check 'pre-check
           (lambda _
             ;; Needed for tests
             (setenv "HOME" "/tmp"))))))
    (native-inputs
     (list tzdata-for-tests))
    (inputs
     (list coreutils
           sed
           grep
           r-minimal
           r-annotationhub
           r-dt
           r-genomation
           r-ggbio
           r-ggrepel
           r-matrixstats
           r-methylkit
           r-reshape2
           r-rtracklayer
           r-rmarkdown
           r-bookdown
           r-ggplot2
           r-ggbio
           pandoc
           python-wrapper
           python-pyyaml
           snakemake-7
           bismark
           bowtie
           bwa-meth
           fastqc
           methyldackel
           multiqc
           trim-galore
           cutadapt
           samblaster
           samtools))
    (home-page "https://bioinformatics.mdc-berlin.de/pigx/")
    (synopsis "Bisulfite sequencing pipeline from fastq to methylation reports")
    (description "PiGx BSseq is a data processing pipeline for raw fastq read
data of bisulfite experiments; it produces reports on aggregate methylation
and coverage and can be used to produce information on differential
methylation and segmentation.")
    (license license:gpl3+)))

(define-public pigx-scrnaseq
  (package
    (name "pigx-scrnaseq")
    (version "1.1.10")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/BIMSBbioinfo/pigx_scrnaseq/"
                                  "releases/download/v" version
                                  "/pigx_scrnaseq-" version ".tar.gz"))
              (sha256
               (base32
                "082im1j6qpd2y8ax4msxw7mfc40zxpl29hx92skqdd55745f58z8"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-before 'configure 'set-additional-environment-variables
           (lambda _
             ;; Needed for tests
             (setenv "HOME" "/tmp")
             ;; Needed because of loompy
             (setenv "NUMBA_CACHE_DIR" "/tmp")
             ;; Needed to capture environment
             (setenv "PYTHONPATH" (getenv "GUIX_PYTHONPATH")))))))
    (inputs
     (list coreutils
           perl
           fastqc
           flexbar
           icedtea-8
           jellyfish
           python-wrapper
           python-pyyaml
           python-pandas
           python-magic
           python-numpy
           python-loompy
           pandoc
           samtools
           snakemake-5
           star-for-pigx
           r-minimal
           r-argparser
           r-cowplot
           r-data-table
           r-delayedarray
           r-delayedmatrixstats
           r-dplyr
           r-dropbead
           r-dt
           r-genomicalignments
           r-genomicfiles
           r-genomicranges
           r-ggplot2
           r-hdf5array
           r-pheatmap
           r-rmarkdown
           r-rsamtools
           r-rtracklayer
           r-rtsne
           r-scater
           r-scran
           r-seurat
           r-singlecellexperiment
           r-stringr
           r-yaml))
    (home-page "https://bioinformatics.mdc-berlin.de/pigx/")
    (synopsis "Analysis pipeline for single-cell RNA sequencing experiments")
    (description
     "PiGX scRNAseq is an analysis pipeline for preprocessing and
quality control for single cell RNA sequencing experiments.  The inputs are
read files from the sequencing experiment, and a configuration file which
describes the experiment.  It produces processed files for downstream analysis
and interactive quality reports.  The pipeline is designed to work with UMI
based methods.")
    (license license:gpl3+)))

(define-public pigx-sars-cov-2
  (package
    (name "pigx-sars-cov-2")
    (version "0.0.9")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/BIMSBbioinfo/pigx_sars-cov-2"
                                  "/releases/download/v" version
                                  "/pigx_sars-cov-2-" version ".tar.gz"))
              (sha256
               (base32
                "168hrafgsn165q3hcdn916vlbl0zbzri1lckaqapbrcqk00izxma"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'unpack-databases
           (lambda* (#:key inputs #:allow-other-keys)
             ;; The tests need to be able to write caches to HOME.
             ;; They also default to reading the databases from there.
             (setenv "HOME" "/tmp")
             ;; Unpack the three databases in the expected location.
             (let ((root "/tmp/.local/share/pigx/databases")
                   (use-underscore (lambda (c) (if (equal? c #\-) #\_ c))))
               (for-each (lambda (db)
                           (let ((where (string-append root "/"
                                                       (string-map use-underscore db))))
                             (mkdir-p where)
                             (invoke "tar" "-C" where
                                     "-xf" (assoc-ref inputs db))))
                         '("kraken-db" "krona-db" "vep-db")))))
         (add-before 'configure 'set-PYTHONPATH
           (lambda _
             (setenv "PYTHONPATH" (getenv "GUIX_PYTHONPATH")))))))
    (native-inputs
     (let ((bimsb-origin
            (lambda (name hash)
              (origin
                (method url-fetch)
                (uri
                 (string-append "https://bimsbstatic.mdc-berlin.de/akalin/AAkalin_pathogenomics"
                                "/databases_small-20221006/" name))
                (sha256 (base32 hash))))))
       `(("kraken-db"
          ,(bimsb-origin
            "kraken_db.tar.gz"
            "0sdm4xh5npg6c3y2pz8xgphim4qpglm8wdid6rlaaqsn6iikv0mz"))
         ("krona-db"
          ,(bimsb-origin
            "krona_db.tar.gz"
            "1rwy4gd3vw1gdjldrgf44c1xaa3vq8i3pgisjhrac81yx63x8f2h"))
         ("vep-db"
          ,(bimsb-origin
            "vep_db.tar.gz"
            "0d8hhi43zsw3wqm7gd0z0gpcdsc6h6ra0imn87hifl9a64jxqzxz")))))
    (inputs
     (list bash-minimal
           bedtools
           bwa
           ensembl-vep
           fastp
           fastqc
           ivar
           kraken2
           krona-tools
           lofreq
           multiqc
           python-pyyaml
           python-wrapper
           r-base64url
           r-data-table
           r-deconvr
           r-dplyr
           r-dt
           r-ggplot2
           r-htmltools
           r-jsonlite
           r-knitr
           r-magrittr
           r-mass
           r-minimal
           r-plotly
           r-qpcr
           r-r-utils
           r-reshape2
           r-rmarkdown
           r-stringr
           r-tidyr
           r-viridis
           samtools
           snakemake-7
           wget))
    (home-page "https://bioinformatics.mdc-berlin.de/pigx/")
    (synopsis "Analysis pipeline for wastewater sequencing")
    (description "PiGx SARS-CoV-2 is a pipeline for analysing data from
sequenced wastewater samples and identifying given variants-of-concern of
SARS-CoV-2.  The pipeline can be used for continuous sampling.  The output
report will provide an intuitive visual overview about the development of
variant abundance over time and location.")
    (license license:gpl3+)))

(define-deprecated-package pigx-sars-cov2-ww
  pigx-sars-cov-2)

(define-public pigx
  (package
    (name "pigx")
    (version "0.0.3")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://github.com/BIMSBbioinfo/pigx/"
                                  "releases/download/v" version
                                  "/pigx-" version ".tar.gz"))
              (sha256
               (base32
                "1i5njdy1clj5ncw45d16p7mwmqvb1ilikl9n797pxklc3f4s7mq7"))))
    (build-system gnu-build-system)
    (inputs
     (list python pigx-bsseq pigx-chipseq pigx-rnaseq pigx-scrnaseq))
    (home-page "https://bioinformatics.mdc-berlin.de/pigx/")
    (synopsis "Analysis pipelines for genomics")
    (description "PiGx is a collection of genomics pipelines.  It includes the
following pipelines:

@itemize
@item PiGx BSseq for raw fastq read data of bisulfite experiments
@item PiGx RNAseq for RNAseq samples
@item PiGx scRNAseq for single cell dropseq analysis
@item PiGx ChIPseq for reads from ChIPseq experiments
@end itemize

All pipelines are easily configured with a simple sample sheet and a
descriptive settings file.  The result is a set of comprehensive, interactive
HTML reports with interesting findings about your samples.")
    (license license:gpl3+)))

(define-public genrich
  (package
    (name "genrich")
    (version "0.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/jsh58/Genrich")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0x0q6z0208n3cxzqjla4rgjqpyqgwpmz27852lcvzkzaigymq4zp"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f ; there are none
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (install-file "Genrich" (string-append (assoc-ref outputs "out") "/bin"))
             #t)))))
    (inputs
     (list zlib))
    (home-page "https://github.com/jsh58/Genrich")
    (synopsis "Detecting sites of genomic enrichment")
    (description "Genrich is a peak-caller for genomic enrichment
assays (e.g. ChIP-seq, ATAC-seq).  It analyzes alignment files generated
following the assay and produces a file detailing peaks of significant
enrichment.")
    (license license:expat)))

(define-public mantis
  ;; This is an arbitrary commit as a year has passed since 0.1 was tagged.
  (let ((commit "b6979a269172a45201c8366680d8b889f889432b")
        (revision "2"))
    (package
      (name "mantis")
      (version (git-version "0.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/splatlab/mantis")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0dq8a785hnaxx5kq757m5czs8xpcjpcph1inq2nm8h6zfvqyj8xs"))))
      (build-system cmake-build-system)
      (arguments
       '(#:tests? #f ; there are none
         #:configure-flags (list "-DNH=ON"))) ; do not use SSE4.2 instructions
      (inputs
       (list sdsl-lite openssl zlib))
      (native-inputs
       (list gcc-7))
      (home-page "https://github.com/splatlab/mantis")
      (synopsis "Large-scale sequence-search index data structure")
      (description "Mantis is a space-efficient data structure that can be
used to index thousands of raw-read genomics experiments and facilitate
large-scale sequence searches on those experiments.  Mantis uses counting
quotient filters instead of Bloom filters, enabling rapid index builds and
queries, small indexes, and exact results, i.e., no false positives or
negatives.  Furthermore, Mantis is also a colored de Bruijn graph
representation, so it supports fast graph traversal and other topological
analyses in addition to large-scale sequence-level searches.")
      ;; uses __uint128_t and inline assembly
      (supported-systems '("x86_64-linux"))
      (license license:bsd-3))))

(define-public sjcount
  ;; There is no tag for version 3.2, nor is there a release archive.
  (let ((commit "292d3917cadb3f6834c81e509c30e61cd7ead6e5")
        (revision "1"))
    (package
      (name "sjcount")
      (version (git-version "3.2" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/pervouchine/sjcount-full")
                      (commit commit)))
                (file-name (string-append name "-" version "-checkout"))
                (sha256
                 (base32
                  "0gdgj35j249f04rqgq8ymcc1xg1vi9kzbajnjqpaq2wpbh8bl234"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f                    ; requires a 1.4G test file
         #:make-flags
         ,#~(list (string-append "SAMTOOLS_DIR="
                                 #$(this-package-input "samtools")
                                 "/lib/"))
         #:phases
         (modify-phases %standard-phases
           (replace 'configure
             (lambda* (#:key inputs #:allow-other-keys)
               (substitute* "makefile"
                 (("-I \\$\\{SAMTOOLS_DIR\\}")
                  (string-append "-I" (assoc-ref inputs "samtools")
                                 "/include/samtools"))
                 (("-lz ") "-lz -lpthread "))))
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (for-each (lambda (tool)
                           (install-file tool
                                         (string-append (assoc-ref outputs "out")
                                                        "/bin")))
                         '("j_count" "b_count" "sjcount")))))))
      (inputs
       (list samtools-0.1 zlib))
      (home-page "https://github.com/pervouchine/sjcount-full/")
      (synopsis "Annotation-agnostic splice junction counting pipeline")
      (description "Sjcount is a utility for fast quantification of splice
junctions in RNA-seq data.  It is annotation-agnostic and offset-aware.  This
version does count multisplits.")
      (license license:gpl3+))))

(define-public minimap2
  (package
    (name "minimap2")
    (version "2.28")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "https://github.com/lh3/minimap2/"
                           "releases/download/v" version "/"
                           "minimap2-" version ".tar.bz2"))
       (sha256
        (base32
         "1d50j9fdmmaj7sdf4f49xddc235f7032lwh5ijgi2afj6lkp39gz"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #f                      ; there are none
       #:modules ((guix build utils)
                  (guix build gnu-build-system)
                  (srfi srfi-26))
       #:make-flags
       (list (string-append "CC=" ,(cc-for-target))
             (let ((system ,(or (%current-target-system)
                                (%current-system))))
               (cond
                 ((or (string-prefix? "i586" system)
                      (string-prefix? "i686" system))
                  "sse2only=1")
                 ((string-prefix? "armhf" system)
                  "arm_neon=1")
                 ((string-prefix? "aarch64" system)
                  "aarch64=1")
                 (else "all"))))
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (lib (string-append out "/lib"))
                    (inc (string-append out "/include"))
                    (man (string-append out "/share/man/man1")))
               (install-file "minimap2" bin)
               (install-file "libminimap2.a" lib)
               (install-file "minimap2.1" man)
               (map (cut install-file <> inc)
                    (find-files "." "\\.h$"))
               ;; Not this file.
               (delete-file (string-append inc "/emmintrin.h"))
               (mkdir-p (string-append lib "/pkgconfig"))
               (with-output-to-file (string-append lib "/pkgconfig/minimap2.pc")
                (lambda _
                  (format #t "prefix=~a~@
                          exec_prefix=${prefix}~@
                          libdir=${exec_prefix}/lib~@
                          includedir=${prefix}/include~@
                          ~@
                          Name: libminimap2~@
                          Version: ~a~@
                          Description: A versatile pairwise aligner for genomic and spliced nucleotide sequence~@
                          Libs: -L${libdir} -lminimap2~@
                          Cflags: -I${includedir}~%"
                          out ,version)))))))))
    (inputs
     (list zlib))
    (home-page "https://lh3.github.io/minimap2/")
    (synopsis "Pairwise aligner for genomic and spliced nucleotide sequences")
    (description "Minimap2 is a versatile sequence alignment program that
aligns DNA or mRNA sequences against a large reference database.  Typical use
cases include:

@enumerate
@item mapping PacBio or Oxford Nanopore genomic reads to the human genome;
@item finding overlaps between long reads with error rate up to ~15%;
@item splice-aware alignment of PacBio Iso-Seq or Nanopore cDNA or Direct RNA
  reads against a reference genome;
@item aligning Illumina single- or paired-end reads;
@item assembly-to-assembly alignment;
@item full-genome alignment between two closely related species with
  divergence below ~15%.
@end enumerate\n")
    (license license:expat)))

(define-public python-mappy
  (package
    (name "python-mappy")
    (version "2.24")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/lh3/minimap2")
              (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0k658659ivxk2xnyawrfqdhcji6a3xcqdr5a9r5myzb2ivypnjmh"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-backend #~'custom
      #:test-flags
      #~(list "-c" (string-append "import mappy;"
                                  " print(mappy.revcomp("
                                  "mappy.Aligner('test/MT-human.fa')"
                                  ".seq('MT_human', 100, 200)))"))))
    (native-inputs
     (list python-cython python-setuptools))
    (inputs
     (list zlib))
    (home-page "https://github.com/lh3/minimap2")
    (synopsis "Python binding for minimap2")
    (description "This package provides a convenient interface to minimap2,
a fast and accurate C program to align genomic and transcribe nucleotide
sequences.")
    (license license:expat)))

(define-public miniasm
  (package
   (name "miniasm")
   (version "0.3")
   (source (origin
            (method git-fetch)
            (uri (git-reference
                   (url "https://github.com/lh3/miniasm")
                   (commit (string-append "v" version))))
            (file-name (git-file-name name version))
            (sha256
             (base32
              "04dv5wv8bhsw1imxwyd438bnn9kby7svp44nbcz8lsadzjjci5gs"))))
   (build-system gnu-build-system)
   (inputs
    (list zlib))
   (arguments
    `(#:tests? #f ; There are no tests.
      #:phases
      (modify-phases %standard-phases
        (delete 'configure)
        (replace 'install
          (lambda* (#:key inputs outputs #:allow-other-keys)
            (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
              (install-file "miniasm" bin)
              (install-file "minidot" bin)
              #t))))))
   (home-page "https://github.com/lh3/miniasm")
   (synopsis "Ultrafast de novo assembly for long noisy reads")
   (description "Miniasm is a very fast OLC-based de novo assembler for noisy
long reads.  It takes all-vs-all read self-mappings (typically by minimap) as
input and outputs an assembly graph in the GFA format.  Different from
mainstream assemblers, miniasm does not have a consensus step.  It simply
concatenates pieces of read sequences to generate the final unitig sequences.
Thus the per-base error rate is similar to the raw input reads.")
   (license license:expat)))

(define-public bandage
  (package
    (name "bandage")
    (version "0.8.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/rrwick/Bandage")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1bbsn5f5x8wlspg4pbibqz6m5vin8c19nl224f3z3km0pkc97rwv"))))
    (build-system qt-build-system)
    (arguments
     (list
      #:modules '((guix build qt-build-system)
                  ((guix build gnu-build-system) #:prefix gnu:)
                  (guix build utils))
      ;; TODO: Once <https://issues.guix.gnu.org/47475> is fixed,
      ;; consider uncommenting the following:
      ;;
      ;; Prevent the (rarely updated) imagemagick/stable package from
      ;; ending up in the closure.
      ;; #:disallowed-references (list imagemagick/stable)
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda _
              (invoke "qmake" "Bandage.pro")))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (substitute* "tests/bandage_command_line_tests.sh"
                  (("^bandagepath=.*")
                   (string-append "bandagepath=" (getcwd) "/Bandage\n")))
                (with-directory-excursion "tests"
                  (setenv "XDG_RUNTIME_DIR" (getcwd))
                  (invoke "./bandage_command_line_tests.sh")))))
          (replace 'build (assoc-ref gnu:%standard-phases 'build))
          (replace 'install
            (lambda _
              (install-file "Bandage" (string-append #$output "/bin")))))))
    (inputs
     (list qtbase-5 qtsvg-5))
    (native-inputs
     ;; imagemagick/stable cannot be used here, as it will end up in
     ;; the closure.  See <https://issues.guix.gnu.org/47475>.
     (list imagemagick))
    (home-page "https://rrwick.github.io/Bandage/")
    (synopsis
     "Bioinformatics Application for Navigating De novo Assembly Graphs Easily")
    (description "Bandage is a program for visualising de novo assembly graphs.
It allows users to interact with the assembly graphs made by de novo assemblers
such as Velvet, SPAdes, MEGAHIT and others.  De novo assembly graphs contain not
only assembled contigs but also the connections between those contigs, which
were previously not easily accessible.  Bandage visualises assembly graphs, with
connections, using graph layout algorithms.  Nodes in the drawn graph, which
represent contigs, can be automatically labelled with their ID, length or depth.
Users can interact with the graph by moving, labelling and colouring nodes.
Sequence information can also be extracted directly from the graph viewer.  By
displaying connections between contigs, Bandage opens up new possibilities for
analysing and improving de novo assemblies that are not possible by looking at
contigs alone.")
    (license (list license:gpl2+        ; bundled ogdf
                   license:gpl3+))))

(define-public libmaus2
  (package
    (name "libmaus2")
    (version "2.0.786")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://gitlab.com/german.tischler/libmaus2")
                    (commit (string-append version "-release-20210531143054"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1rxakmwjcx2yq5sjh3v849f7dfw4xzc2fyzf6s28s3p95z84w564"))))
    (build-system gnu-build-system)
    ;; The test suite attempts to execute ../test-driver, which does not exist.
    (arguments '(#:tests? #false))
    (propagated-inputs
     (list zlib))
    (native-inputs
     (list pkg-config))
    (home-page "https://gitlab.com/german.tischler/libmaus2")
    (synopsis "Collection of data structures and algorithms useful for bioinformatics")
    (description "libmaus2 is a collection of data structures and
algorithms.  It contains:

@itemize
@item I/O classes (single byte and UTF-8);
@item @code{bitio} classes (input, output and various forms of bit level
  manipulation);
@item text indexing classes (suffix and LCP array, fulltext and minute (FM),
  etc.);
@item BAM sequence alignment files input/output (simple and collating);
and many lower level support classes.
@end itemize\n")
    ;; The code is explicitly available under the terms of either GPLv2 or
    ;; GPLv3 according to the AUTHORS file, though most files have a GPLv3+
    ;; license header.
    (license (list license:gpl2+ license:gpl3+))))

(define-public biobambam2
  (package
    (name "biobambam2")
    (version "2.0.182")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://gitlab.com/german.tischler/biobambam2")
                    (commit (string-append version "-release-20210412001032"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0b7w7a2a7hpkgrdn0n7hy4pilzrj82zqrh7q4bg1l0cd6bqr60m5"))))
    (build-system gnu-build-system)
    (arguments
     ;; The test suite attempts to execute ../test-driver, which does not exist.
     `(#:tests? #false
       #:configure-flags
       ,#~(list (string-append "--with-libmaus2="
                               #$(this-package-input "libmaus2")))))
    (inputs
     (list libmaus2 xerces-c))
    (native-inputs
     (list pkg-config))
    (home-page "https://gitlab.com/german.tischler/biobambam2")
    (synopsis "Tools for processing BAM files")
    (description "This package contains some tools for processing BAM files
including:

@itemize
@item bamsormadup: parallel sorting and duplicate marking
@item bamcollate2: reads BAM and writes BAM reordered such that alignment or
  collated by query name
@item bammarkduplicates: reads BAM and writes BAM with duplicate alignments
  marked using the BAM flags field
@item bammaskflags: reads BAM and writes BAM while masking (removing) bits
  from the flags column
@item bamrecompress: reads BAM and writes BAM with a defined compression
  setting.  This tool is capable of multi-threading.
@item bamsort: reads BAM and writes BAM resorted by coordinates or query name
@item bamtofastq: reads BAM and writes FastQ; output can be collated or
  uncollated by query name.
@end itemize
")
    ;; The COPYING file states that the code is distributed under version 3 of
    ;; the GPL, but the license headers include the "or later" clause.
    (license license:gpl3+)))

(define-public r-disgenet2r
  (let ((commit "8d8ce37da7384004038b25e784b9f7cfe2353de1")
        (revision "1"))
    (package
      (name "r-disgenet2r")
      (version (git-version "0.99.2" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://bitbucket.org/ibi_group/disgenet2r")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0dvk75s6wqssdyfn5iczldb8krvrz2s0dslxns4571cb2pr09b84"))))
      (properties `((upstream-name . "disgenet2r")))
      (build-system r-build-system)
      (propagated-inputs (list r-data-table
                               r-ggplot2
                               r-gtable
                               r-httr
                               r-igraph
                               r-jsonlite
                               r-purrr
                               r-reshape
                               r-reshape2
                               r-sparql
                               r-stringr
                               r-tidyr
                               r-tidyverse
                               r-venndiagram))
      (native-inputs (list r-knitr))
      (home-page "https://bitbucket.org/ibi_group/disgenet2r")
      (synopsis "Query, visualize, and expand DisGeNET data")
      (description
       "This is an R package to query and expand DisGeNET data, and to
visualize the results within R framework.  The disgenet2r package is designed
to retrieve data from DisGeNET v6.0 (Jan, 2019).")
      (license license:expat))))

(define-public r-dtmm
  (let ((commit "3a553b1e17d27d90a496d2e23e98e5dfe4abc266")
        (revision "1"))
    (package
      (name "r-dtmm")
      (version (git-version "0.1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/MaStatLab/DTMM")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "19279wafqfs9gk9489f8zsp52qcdb0mkxgvqszq4i733ckr2mmkk"))))
      (properties `((upstream-name . "DTMM")))
      (build-system r-build-system)
      (propagated-inputs (list r-ape r-rcpp r-rcpparmadillo))
      (home-page "https://github.com/MaStatLab/DTMM")
      (synopsis "Dirichlet-tree multinomial mixtures")
      (description
       "This package lets you perform unsupervised clustering of amplicon
sequencing data in microbiome studies with the Dirichlet-tree Multinomial
Mixtures.")
      (license license:cc0))))

;; Needed for r-liana
(define-public r-omnipathr/devel
  (package
    (name "r-omnipathr")
    (version "3.15.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/saezlab/omnipathr")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sq32s3v7q5w9zym54fm0ws1d1wnwyhczazyk9g722ydx2zvbls4"))))
    (properties `((upstream-name . "OmnipathR")))
    (build-system r-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'set-HOME
           (lambda _ (setenv "HOME" "/tmp"))))))
    (propagated-inputs
     (list r-checkmate
           r-crayon
           r-curl
           r-digest
           r-dplyr
           r-fs
           r-httr2
           r-igraph
           r-jsonlite
           r-later
           r-logger
           r-lubridate
           r-magrittr
           r-progress
           r-purrr
           r-r-utils
           r-rappdirs
           r-readr
           r-readxl
           r-rlang
           r-rmarkdown
           r-rsqlite
           r-rvest
           r-sessioninfo
           r-stringi
           r-stringr
           r-tibble
           r-tidyr
           r-tidyselect
           r-vctrs
           r-withr
           r-xml
           r-xml2
           r-yaml
           r-zip))
    (native-inputs (list r-knitr r-testthat))
    (home-page "https://github.com/saezlab/omnipathr")
    (synopsis "OmniPath web service client and more")
    (description
     "This package provides a client for the OmniPath web service and many
other resources.  It also includes functions to transform and pretty print
some of the downloaded data, functions to access a number of other resources
such as BioPlex, ConsensusPathDB, EVEX, Gene Ontology, Guide to
Pharmacology (IUPHAR/BPS), Harmonizome, HTRIdb, Human Phenotype Ontology,
InWeb InBioMap, KEGG Pathway, Pathway Commons, Ramilowski et al. 2015,
RegNetwork, ReMap, TF census, TRRUST and Vinayagam et al. 2011.  Furthermore,
OmnipathR features a close integration with the NicheNet method for ligand
activity prediction from transcriptomics data, and its R implementation
@code{nichenetr}.")
    (license license:expat)))

(define-public r-liana
  (let ((commit "6cab46c54234f861ea176c3de77c4b8aa45ecb3d")
        (revision "1"))
    (package
      (name "r-liana")
      (version (git-version "0.1.14" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/saezlab/liana/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1zkvkkvxyhn60b2rribz5fhvm9kvscq8xz6is440080ihpyfyadz"))))
      (properties
       '((upstream-name . "liana")
         (updater-extra-native-inputs
          . ("r-delayedmatrixstats" "r-sparsematrixstats"))))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           ;; This is needed to find ~/.config/OmnipathR/omnipathr.yml
           (add-after 'unpack 'set-HOME
             (lambda _ (setenv "HOME" "/tmp")))
           (add-after 'unpack 'disable-bad-tests
             (lambda _
               ;; These tests attempt to connect to the Internet.
               (substitute* "tests/testthat/test-liana_utils.R"
                 ((".*Test liana pipe.*" m)
                  (string-append m "skip('guix')\n")))
               (substitute* "tests/testthat/test-liana_tensor.R"
                 ((".*Test tensor wrapper.*" m)
                  (string-append m "skip('guix')\n"))
                 ((".*Test decompose_tensor.*" m)
                  (string-append m "skip('guix')\n")))
               ;; XXX: This test returns a data.frame instead of an error.
               (substitute* "tests/testthat/test-liana_cytotalk.R"
                 ((".*Test Cytotalk Wrap.*" m)
                  (string-append m "skip('guix')\n"))))))))
      (propagated-inputs
       (list r-basilisk
             r-basilisk-utils
             r-complexheatmap
             r-dplyr
             r-ggplot2
             r-magrittr
             r-omnipathr/devel
             r-purrr
             r-rcolorbrewer
             r-readr
             r-reticulate
             r-rlang
             r-scater
             r-scran
             r-scuttle
             r-seuratobject
             r-singlecellexperiment
             r-stringr
             r-tibble
             r-tidyr
             r-tidyselect))
      (native-inputs
       (list r-delayedmatrixstats
             r-knitr
             r-seurat
             r-sparsematrixstats
             r-testthat))
      (home-page "https://github.com/saezlab/liana/")
      (synopsis "LIANA: a LIgand-receptor ANalysis frAmework")
      (description
       "LIANA provides a number of methods and resource for ligand-receptor
interaction inference from scRNA-seq data.")
      (license license:gpl3))))

(define-public r-circus
  (package
    (name "r-circus")
    (version "0.1.7")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BIMSBbioinfo/ciRcus")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d1jz7r81zbcy1gkppggkjmgjxyjhva69s3cdb01m3f0790m4fv0"))))
    (properties `((upstream-name . "ciRcus")))
    (build-system r-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'disable-bad-tests
           (lambda _
             ;; These tests need Internet access.
             (substitute* "tests/testthat/test-loadAnnot.R"
               (("test_that.*Annotation.*" m)
                (string-append m "skip('guix')\n"))))))))
    (propagated-inputs
     (list r-annotationdbi
           r-annotationhub
           r-biocgenerics
           r-biomart
           r-data-table
           r-dbi
           r-genomeinfodb
           r-genomicfeatures
           r-genomicranges
           r-ggplot2
           r-hash
           r-iranges
           r-rcolorbrewer
           r-rmysql
           r-rtracklayer
           r-s4vectors
           r-stringr
           r-summarizedexperiment))
    (native-inputs
     (list r-knitr r-testthat))
    (home-page "https://github.com/BIMSBbioinfo/ciRcus")
    (synopsis "Annotation, analysis and visualization of circRNA data")
    (description "Circus is an R package for annotation, analysis and
visualization of circRNA data.  Users can annotate their circRNA candidates
with host genes, gene features they are spliced from, and discriminate between
known and yet unknown splice junctions.  Circular-to-linear ratios of circRNAs
can be calculated, and a number of descriptive plots easily generated.")
    (license license:artistic2.0)))

(define-public r-domultibarheatmap
  (let ((commit "9e65afa0aa69fee631c61b7bf3e7742632c9cb95")
        (revision "1"))
    (package
      (name "r-domultibarheatmap")
      (version (git-version "0.1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/elliefewings/DoMultiBarHeatmap")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0q9mrc6md08aff6hhzlw3igvv3w7pr1wildzm8i0km9xvbi9iyy9"))))
      (properties `((upstream-name . "DoMultiBarHeatmap")))
      (build-system r-build-system)
      (propagated-inputs (list r-ggplot2 r-magrittr r-rlang r-seurat))
      (home-page "https://github.com/elliefewings/DoMultiBarHeatmap")
      (synopsis "Produce heatmap from a Seurat object with multiple annotation bars")
      (description "This package builds on Seurat's @code{Doheatmap} function
code to produce a heatmap from a Seurat object with multiple annotation
bars.")
      (license license:cc0))))

(define-public r-doubletfinder
  (let ((commit "554097ba4e2c0ed7c28dc7f0b5b75277f3a50551")
        (revision "1"))
    (package
      (name "r-doubletfinder")
      (version (git-version "2.0.3" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/chris-mcginnis-ucsf/DoubletFinder")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1q1pnqw7ry4syp04wjmvz5bws6z4vg4c340ky07lk0vp577x2773"))))
      (properties `((upstream-name . "DoubletFinder")))
      (build-system r-build-system)
      (propagated-inputs (list r-fields r-kernsmooth r-rocr))
      (home-page "https://github.com/chris-mcginnis-ucsf/DoubletFinder")
      (synopsis "Identify doublets in single-cell RNA sequencing data")
      (description
       "DoubletFinder identifies doublets by generating artificial doublets
from existing scRNA-seq data and defining which real cells preferentially
co-localize with artificial doublets in gene expression space.  Other
DoubletFinder package functions are used for fitting DoubletFinder to
different scRNA-seq datasets.  For example, ideal DoubletFinder performance in
real-world contexts requires optimal pK selection and homotypic doublet
proportion estimation.  pK selection is achieved using pN-pK parameter sweeps
and maxima identification in mean-variance-normalized bimodality coefficient
distributions.  Homotypic doublet proportion estimation is achieved by finding
the sum of squared cell annotation frequencies.")
      (license license:cc0))))

;; There have been no releases.
(define-public r-cytobackbone
  (let ((commit "4c1a0a35cc5ae1f8f516127cec92351d96fe26e7")
        (revision "1"))
    (package
      (name "r-cytobackbone")
      (version (git-version "1.0.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/tchitchek-lab/CytoBackBone")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0ahiad14zcgdk42xzw5xryic2ibn2l8lkrcdvl2b5sz2js028yb3"))))
      (properties `((upstream-name . "CytoBackBone")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-flowcore
             r-flowutils
             r-fnn
             r-ggplot2
             r-preprocesscore))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/tchitchek-lab/CytoBackBone")
      (synopsis "Merge phenotype information from different cytometric profiles")
      (description
       "This package implements an algorithm which increases the number of
simultaneously measurable markers and in this way helps with study of the
immune responses.  Thus, the present algorithm, named @code{CytoBackBone},
allows combining phenotypic information of cells from different cytometric
profiles obtained from different cytometry panels.  This computational
approach is based on the principle that each cell has its own phenotypic and
functional characteristics that can be used as an identification card.
@code{CytoBackBone} uses a set of predefined markers, that we call the
backbone, to define this identification card.  The phenotypic information of
cells with similar identification cards in the different cytometric profiles
is then merged.")
      (license license:gpl2))))

(define-public r-cytoexplorer
  (let ((commit "0efb1cc19fc701ae03905cf1b8484c1dfeb387df")
        (revision "1"))
    (package
      (name "r-cytoexplorer")
      (version (git-version "1.1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/DillonHammill/CytoExploreR")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "1y7dadsy15i47rjmdq6ns80jzm6p0zmixll37q68ba2c7xn5pq3v"))
         (snippet
          '(delete-file
            "docs/articles/CytoExploreR_files/vis-4.20.1/vis.min.js"))))
      (properties `((upstream-name . "CytoExploreR")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           (add-after 'unpack 'skip-bad-tests
             (lambda _
               (with-directory-excursion "tests/testthat/"
                 ;; All of these fail because with_mock has been removed from
                 ;; r-testthat.
                 (substitute* "test-cyto_channels-helpers.R"
                   (("^test_that.*cyto_channel_select.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_gate_draw.R"
                   (("^test_that.*cyto_gate_draw.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_gate_helpers.R"
                   (("^test_that.*cyto_gate_edit.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_helpers.R"
                   (("^test_that.*cyto_markers_edit.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_details_edit.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_channel_match.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_spillover_compute.R"
                   (("^test_that.*cyto_spillover_compute universal reference.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_spillover_compute internal reference.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-openCyto-plugins.R"
                   (("^test_that.*cyto_gate_manual.*" m)
                    (string-append m "skip('guix')")))

                 ;; These tests fail.  Some fail because of unimportant
                 ;; differences in printed messages.
                 (substitute* "test-cyto_channels-helpers.R"
                   (("^test_that.*cyto_channels_restrict.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_gate_helpers.R"
                   (("^test_that.*cyto_gate_rename.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_gate_remove.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_helpers.R"
                   (("^test_that.*cyto_compensate.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_empty.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_group_by.*" m)
                    (string-append m "skip('guix')")))
                 (substitute* "test-cyto_spillover_spread_compute.R"
                   (("^test_that.*cyto_spillover_spread_compute internal reference.*" m)
                    (string-append m "skip('guix')"))
                   (("^test_that.*cyto_spillover_spread_compute universal reference.*" m)
                    (string-append m "skip('guix')"))))))
           (add-after 'unpack 'process-javascript
             (lambda* (#:key inputs #:allow-other-keys)
               (with-directory-excursion "docs/articles/CytoExploreR_files/"
                 (let ((source (search-input-file inputs "/dist/vis.js"))
                       (target "vis-4.20.1/vis.min.js"))
                   (invoke "esbuild" source "--minify"
                           (string-append "--outfile=" target)))))))))
      (propagated-inputs
       (list r-biocgenerics
             r-bslib
             r-data-table
             r-dplyr
             r-embedsom
             r-flowai
             r-flowcore
             r-flowworkspace
             r-gtools
             r-magrittr
             r-mass
             r-opencyto
             r-purrr
             r-rhandsontable
             r-robustbase
             r-rsvd
             r-rtsne
             r-shiny
             r-superheat
             r-tibble
             r-tidyr
             r-umap
             r-visnetwork))
      (native-inputs
       (list esbuild
             r-cytoexplorerdata
             r-knitr
             r-mockery
             r-testthat
             (let ((version "4.20.1"))
               (origin
                 (method git-fetch)
                 (uri (git-reference
                       (url "https://github.com/almende/vis")
                       (commit (string-append "v" version))))
                 (file-name (git-file-name "js-vis" version))
                 (sha256
                  (base32
                   "09ldcqzzki5c0jlwas5992qjffqxnx6j5sl703qccfw7rg1hn469"))))))
      (home-page "https://github.com/DillonHammill/CytoExploreR")
      (synopsis "Interactive analysis of cytometry data")
      (description
       "This package has been developed under ROpenSci gudelines to integrate
conventional and cutting edge cytometry analysis tools under a unified
framework.  It aims to represent an intuitive and interactive approach to
analysing cytometry data in R.")
      (license license:gpl2))))

(define-public r-cytoexplorerdata
  (let ((commit "488edf083092247ad547172906efe6f8c2aa8700")
        (revision "1"))
    (package
      (name "r-cytoexplorerdata")
      (version (git-version "1.0.3" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/DillonHammill/CytoExploreRData")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0mp3d8cv57psz5zjp7a1f07fxw9dw0w0x9fv32k46a566k1073p3"))))
      (properties `((upstream-name . "CytoExploreRData")))
      (build-system r-build-system)
      (native-inputs (list r-knitr))
      (home-page "https://github.com/DillonHammill/CytoExploreRData")
      (synopsis "Flow Cytometry Data for CytoExploreR")
      (description
       "This package contains @code{gatingTemplates}, example fcs files and compensation
controls for use in @code{CytoExploreR}.")
      (license license:gpl2))))

(define-public r-giotto
  (let ((commit "3e6671a2512484a7b90b421b7e697d1abc2ec760")
        (revision "1"))
    (package
      (name "r-giotto")
      (version (git-version "1.1.2" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/RubD/Giotto/")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "086kyfhwqcd9qkl2yb9g4xgk8xs88ga4slha7rwilxm1sg8fhchf"))))
      (properties `((upstream-name . "Giotto")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-clusterr
             r-complexheatmap
             r-cowplot
             r-data-table
             r-dbscan
             r-deldir
             r-farver
             r-fitdistrplus
             r-ggdendro
             r-ggplot2
             r-ggraph
             r-ggrepel
             r-igraph
             r-irlba
             r-lfa
             r-limma
             r-magick
             r-magrittr
             r-matrix
             r-matrixstats
             r-plotly
             r-qvalue
             r-r-utils
             r-rcolorbrewer
             r-rcpp
             r-reshape2
             r-reticulate
             r-rfast
             r-rlang
             r-rtsne
             r-scales
             r-uwot))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/RubD/Giotto/")
      (synopsis "Spatial single-cell transcriptomics toolbox")
      (description
       "This package provides a toolbox to process, analyze and visualize
spatial single-cell expression data.")
      (license license:expat))))

;; Variant of r-illuminahumanmethylationepicmanifest in the
;; (gnu packages bioconductor) module.
(define-public r-illuminahumanmethylationepicmanifest-latest
  (let ((commit "a9ffbad36f5e496ece6c4c37b80e2f4f7e02d0c3")
        (revision "1"))
    (package
      (name "r-illuminahumanmethylationepicmanifest")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url
                "https://github.com/achilleasNP/IlluminaHumanMethylationEPICmanifest")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0v8f0hl0v8gwi61vgqw56rn5j09h95hj54rb8pzbn0znm162n4fc"))))
      (properties `((upstream-name . "IlluminaHumanMethylationEPICmanifest")))
      (build-system r-build-system)
      (home-page
       "https://github.com/achilleasNP/IlluminaHumanMethylationEPICmanifest")
      (synopsis "Illumina Human Methylation Manifest 1.0 B5 for R and minfi")
      (description
       "This is a drop-in replacement for the
@code{IlluminaHumanMethylationEPIC} package.  It utilizes a Manifest based on
1.0B5 annotation.  As of version 0.3.0, the
@code{IlluminaHumanMethylationEPIC} package still employs the 1.0B2 annotation
manifest.  A corresponding annotation package,
@code{IlluminaHumanMethylationEPICanno.ilm10b5.hg38}, is available to ensure
proper annotation.  The decision to maintain the same name is due to
complications in downstream processing caused by array name lookup in certain
preprocessing options.")
      (license license:artistic2.0))))

(define-public r-illuminahumanmethylationepicanno-ilm10b5-hg38
  (let ((commit "3db06910e27f626e0cc8b335ff45cf9a4050a36a")
        (revision "1"))
    (package
      (name "r-illuminahumanmethylationepicanno-ilm10b5-hg38")
      (version (git-version "0.0.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url
                       "https://github.com/achilleasNP/IlluminaHumanMethylationEPICanno.ilm10b5.hg38")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0y8fhiwmkldi57f8nq64njfgljw68cm6pb1vh44hjwcc12k48sqr"))))
      (properties `((upstream-name . "IlluminaHumanMethylationEPICanno.ilm10b5.hg38")))
      (build-system r-build-system)
      (propagated-inputs (list r-minfi))
      (home-page
       "https://github.com/achilleasNP/IlluminaHumanMethylationEPICanno.ilm10b5.hg38")
      (synopsis "Illumina Human Methylation EPIC Annotation version 1.0B5")
      (description
       "This package provides a companion annotation file to the
@code{IlluminaHumanMethylationEPICmanifest} package based on the same
annotation 1.0B5.")
      (license license:artistic2.0))))

(define-public r-maxprobes
  (let ((commit "c2120dba972e12115280ef274ff80550cee5b264")
        (revision "1"))
    (package
      (name "r-maxprobes")
      (version (git-version "0.0.2" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/markgene/maxprobes")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1kij9bds2l9mwm519bzyd2608563qjlmbrayhva1s0vgml5iq9wh"))))
      (properties `((upstream-name . "maxprobes")))
      (build-system r-build-system)
      (propagated-inputs (list r-minfi r-minfidata))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/markgene/maxprobes")
      (synopsis "Methylation array cross-reactive probes")
      (description
       "The Maxprobes package collects cross-reactive probes of Illumina
methylation array 450K and EPIC/850K.")
      (license license:gpl2+))))

(define-public gffread
  ;; We cannot use the tagged release because it is not in sync with gclib.
  ;; See https://github.com/gpertea/gffread/issues/26
  (let ((commit "ba7535fcb3cea55a6e5a491d916e93b454e87fd0")
        (revision "1"))
    (package
      (name "gffread")
      (version (git-version "0.9.12" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/gpertea/gffread")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "1dl2nbcg96lxpd0drg48ssa8343nf7pw9s9mkrc4mjjmfwsin3ki"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f ; no check target
         #:make-flags
         (list "GCLDIR=gclib")
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)
           (add-after 'unpack 'copy-gclib-source
             (lambda* (#:key inputs #:allow-other-keys)
               (mkdir-p "gclib")
               (copy-recursively (assoc-ref inputs "gclib-source") "gclib")
               #t))
           ;; There is no install target
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (let* ((out (assoc-ref outputs "out"))
                      (bin (string-append out "/bin")))
                 (install-file "gffread" bin))
               #t)))))
      (native-inputs
       `(("gclib-source"
          ,(let ((version "0.10.3")
                 (commit "54917d0849c1e83cfb057b5f712e5cb6a35d948f")
                 (revision "1"))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/gpertea/gclib")
                     (commit commit)))
               (file-name (git-file-name "gclib" version))
               (sha256
                (base32
                 "0b51lc0b8syrv7186fd7n8f15rwnf264qgfmm2palrwks1px24mr")))))))
      (home-page "https://github.com/gpertea/gffread/")
      (synopsis "Parse and convert GFF/GTF files")
      (description
       "This package provides a GFF/GTF file parsing utility providing format
conversions, region filtering, FASTA sequence extraction and more.")
      ;; gffread is under Expat, but gclib is under Artistic 2.0
      (license (list license:expat
                     license:artistic2.0)))))

(define-public fit-sne
  (package
    (name "fit-sne")
    (version "1.2.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/KlugerLab/FIt-SNE")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1imq4577awc226wvygf94kpz156qdfw8xl0w0f7ss4w10lhmpmf5"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #false ; there are none
       #:phases
       ;; There is no build system.
       (modify-phases %standard-phases
         (delete 'configure)
         (replace 'build
           (lambda _
             (invoke "g++" "-std=c++11" "-O3"
                     "src/sptree.cpp"
                     "src/tsne.cpp"
                     "src/nbodyfft.cpp"
                     "-o" "bin/fast_tsne"
                     "-pthread" "-lfftw3" "-lm"
                     "-Wno-address-of-packed-member")))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((out (assoc-ref outputs "out"))
                    (bin (string-append out "/bin"))
                    (share (string-append out "/share/fit-sne")))
               (for-each (lambda (file) (install-file file bin))
                         (find-files "bin"))

               (substitute* "fast_tsne.R"
                 (("^FAST_TSNE_SCRIPT_DIR.*")
                  (string-append "FAST_TSNE_SCRIPT_DIR = \"" out "\"\n")))
               (install-file "fast_tsne.R" share)))))))
    (inputs
     (list fftw))
    (home-page "https://github.com/KlugerLab/FIt-SNE")
    (synopsis "Fast Fourier Transform-accelerated interpolation-based t-SNE")
    (description "@dfn{t-Stochastic Neighborhood Embedding} (t-SNE) is a
method for dimensionality reduction and visualization of high dimensional
datasets.  A popular implementation of t-SNE uses the Barnes-Hut algorithm to
approximate the gradient at each iteration of gradient descent.  This
implementation differs in these ways:

@itemize
@item Instead of approximating the N-body simulation using Barnes-Hut, we
  interpolate onto an equispaced grid and use FFT to perform the convolution.
@item Instead of computing nearest neighbors using vantage-point trees, we
  approximate nearest neighbors using the Annoy library.  The neighbor lookups
  are multithreaded to take advantage of machines with multiple cores.
@end itemize
")
    ;; See LICENSE.txt for details on what license applies to what files.
    (license (list license:bsd-4 license:expat license:asl2.0))))

(define-public python-scanpy
  (package
    (name "python-scanpy")
    (version "1.11.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/theislab/scanpy")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "18ddb3jkyjan87f5kymyq951sa5955z41f10h6z954map8dy2136"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; 736 passed, 95 skipped, 20 xfailed, 148 warnings
      #:test-flags
      '(list "-m" "not gpu"
             "--numprocesses" (number->string (parallel-job-count))
             ;; These tests require Internet access.
             "--ignore-glob=tests/notebooks/*"
             "--ignore=tests/test_clustering.py"
             "--ignore=tests/test_datasets.py"
             "--ignore=tests/test_normalization.py"
             "--ignore=tests/test_score_genes.py"
             "--ignore=tests/test_highly_variable_genes.py"
             "--ignore=tests/test_aggregated.py"
             "--ignore=get/_aggregated.py"
             ;; TODO: I can't get the plotting tests to work, even with Xvfb.
             ;; Some of them also require Internet access because they want to
             ;; download a dataset.
             "--ignore=tests/test_plotting.py"
             "--ignore=tests/test_embedding_plots.py"
             "--ignore=tests/test_preprocessing.py"
             "--ignore=tests/test_read_10x.py"
             "--ignore=plotting/_tools/scatterplots.py"
             ;; Adding additional options does not help to resolve the
             ;; faileur: TypeError: _FlakyPlugin._make_test_flaky() got an
             ;; unexpected keyword argument 'reruns'.
             "--ignore=tests/test_backed.py"
             "-k"
             ;; Plot tests that fail.
             (string-append "not test_clustermap"
                            " and not test_dotplot_matrixplot_stacked_violin"
                            " and not test_paga_compare"
                            " and not test_paga_path"
                            " and not test_paga_pie"
                            " and not test_paga_plots"
                            " and not test_violin"
                            " and not test_scatter_no_basis_per_obs"
                            " and not test_spatial_general"
                            " and not test_visium_empty_img_key"

                            ;; These are doctests that fail because of missing
                            ;; datasets.
                            " and not scanpy.get._aggregated.aggregate"
                            " and not scanpy.plotting._tools.scatterplots.spatial"

                            ;; One difference in a long array.
                            " and not test_cell_demultiplexing"

                            ;; These try to connect to the network
                            " and not test_scrublet_plots"
                            " and not test_plot_rank_genes_groups_gene_symbols"
                            " and not test_pca_n_pcs"
                            " and not test_pca_chunked"
                            " and not test_pca_layer"
                            " and not test_pca_sparse"
                            " and not test_pca_reproducible"
                            " and not test_clip"

                            ;; Missing test data.
                            " and not test_covariance_eigh_impls"
                            " and not test_embedding_colorbar_location"
                            " and not test_sparse_dask_input_errors"
                            " and not test_sparse_dask_input_errors"
                            " and not test_spatial_external_img"

                            ;; Somehow broken tests.
                            " and not test_sim_toggleswitch"
                            " and not scanpy.datasets._datasets.krumsiek11"
                            " and not scanpy.datasets._datasets.toggleswitch"
                            " and not scanpy.external.pp._scanorama_integrate.scanorama_integrate"
                            " and not scanpy.preprocessing._simple.filter_cells"))
       #:phases
       #~(modify-phases %standard-phases
           ;; XXX This should not be necessary, but I noticed while building
           ;; python-metacells that the anndata version check fails, resulting
           ;; in the wrong module name to be loaded from anndata.  I cannot
           ;; reproduce this in an interactive Python session.  We patch this
           ;; here to ensure that the appropriate module name is used.
           (add-after 'unpack 'patch-version-check
             (lambda _
               (substitute* "src/scanpy/_utils/__init__.py"
                 (("Version\\(anndata_version\\) < Version\\(\"0.6.10\"\\):")
                  "False:")
                 (("Version\\(anndata_version\\) >= Version\\(\"0.10.0\"\\):")
                  "True:"))))
           (add-after 'unpack 'pretend-version
             (lambda _
               (setenv "SETUPTOOLS_SCM_PRETEND_VERSION" #$version)))
           (add-after 'unpack 'add-anndata-source
             (lambda _
               (setenv "PYTHONPATH"
                       (string-append (getcwd) ":"
                                      #$(this-package-native-input "python-anndata:source") ":"
                                      (getenv "GUIX_PYTHONPATH")))))
           (add-before 'check 'pre-check
             (lambda _
               ;; Numba needs a writable dir to cache functions.
               (setenv "NUMBA_CACHE_DIR" "/tmp")
               ;; For Matplotlib.
               (setenv "HOME" "/tmp"))))))
    (propagated-inputs
     (list python-anndata
           python-dask
           python-h5py
           python-igraph
           python-joblib
           python-legacy-api-wrap
           python-louvain
           python-matplotlib
           python-natsort
           python-networkx
           python-numba
           python-packaging
           python-pandas
           python-patsy
           python-pytoml
           python-scikit-learn
           python-scipy
           python-seaborn
           python-session-info2
           python-setuptools ; For pkg_resources.
           python-sinfo
           python-statsmodels
           python-tables
           python-tqdm
           python-umap-learn))
    (native-inputs
     `(;; This package needs anndata.tests, which is not installed.
       ("python-anndata:source" ,(package-source python-anndata))
       ("python-flaky" ,python-flaky)
       ("python-flit" ,python-flit)
       ("python-hatch-vcs" ,python-hatch-vcs)
       ("python-hatchling" ,python-hatchling)
       ("python-leidenalg" ,python-leidenalg)
       ("python-pytest" ,python-pytest)
       ("python-pytest-mock" ,python-pytest-mock)
       ("python-pytest-nunit" ,python-pytest-nunit)
       ("python-pytest-xdist" ,python-pytest-xdist)
       ("python-scanorama" ,python-scanorama)
       ("python-setuptools-scm" ,python-setuptools-scm)))
    (home-page "https://github.com/theislab/scanpy")
    (synopsis "Single-Cell Analysis in Python")
    (description "Scanpy is a scalable toolkit for analyzing single-cell gene
expression data.  It includes preprocessing, visualization, clustering,
pseudotime and trajectory inference and differential expression testing.  The
Python-based implementation efficiently deals with datasets of more than one
million cells.")
    (license license:bsd-3)))

(define-public python-bbknn
  (package
    (name "python-bbknn")
    (version "1.6.0")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "bbknn" version))
       (sha256
        (base32
         "06q43cpi7wi6f2d2jqs8f9rbd94pg1hh7978gm92mi9gvzbaj08w"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f                       ;no tests are included
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable dir to cache functions.
         (add-before 'check 'set-numba-cache-dir
           (lambda _
             (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs
     (list python-annoy
           python-cython
           python-numpy
           python-pandas
           python-pynndescent
           python-scikit-learn
           python-scipy
           python-umap-learn))
    (native-inputs (list python-flit-core))
    (home-page "https://github.com/Teichlab/bbknn")
    (synopsis "Batch balanced KNN")
    (description "BBKNN is a batch effect removal tool that can be directly
used in the Scanpy workflow.  It serves as an alternative to
@code{scanpy.api.pp.neighbors()}, with both functions creating a neighbour
graph for subsequent use in clustering, pseudotime and UMAP visualisation.  If
technical artifacts are present in the data, they will make it challenging to
link corresponding cell types across different batches.  BBKNN actively
combats this effect by splitting your data into batches and finding a smaller
number of neighbours for each cell within each of the groups.  This helps
create connections between analogous cells in different batches without
altering the counts or PCA space.")
    (license license:expat)))

(define-public python-drep
  (package
    (name "python-drep")
    (version "3.2.0")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "drep" version))
       (sha256
        (base32
         "08vk0x6v5c5n7afgd5pcjhsvb424absypxy22hw1cm1n9kirbi77"))))
    (build-system python-build-system)
    (propagated-inputs
     (list python-biopython
           python-matplotlib
           python-numpy
           python-pandas
           python-pytest
           python-scikit-learn
           python-seaborn
           python-tqdm))
    (home-page "https://github.com/MrOlm/drep")
    (synopsis "De-replication of microbial genomes assembled from multiple samples")
    (description
     "dRep is a Python program for rapidly comparing large numbers of genomes.
dRep can also \"de-replicate\" a genome set by identifying groups of highly
similar genomes and choosing the best representative genome for each genome
set.")
    (license license:expat)))

(define-public instrain
  (package
    (name "instrain")
    ;; Git repository does not tag releases, use the latest commit from master
    ;; branch.
    (properties '((commit . "6180be7b49a61b7e1ffe9f1489da5c6aa2ff9ac3")
                  (revision . "0")))
    (version (git-version "1.10.0"
                          (assoc-ref properties 'revision)
                          (assoc-ref properties 'commit)))
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/MrOlm/instrain")
              (commit (assoc-ref properties 'commit))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1njsxjf3248121yw3q1ig6asf6b3wa5fgjfyc6dkgk6nd4ih8wni"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; Tests assume that test files exist (they don't) and are located in
      ;; the developer's home directory, see:
      ;; <https://github.com/MrOlm/inStrain/issues/218>.
      #:tests? #false
      #:phases
      #~(modify-phases %standard-phases
         (add-after 'unpack 'patch-relative-imports
           (lambda _
             (substitute* (find-files "test/tests" "test_.*\\.py")
               (("from test_utils import BTO")
                "from .test_utils import BTO")
               (("import test_utils") "from . import test_utils"))
             (substitute* "docker/run_instrain.py"
               (("from s3_utils")
                "from .s3_utils")
               (("from job_utils")
                "from .job_utils"))))
         (add-after 'unpack 'relax-requirements
           (lambda _
             (substitute* "setup.py"
               ((".*pytest.*") "")))))))
    (propagated-inputs
     (list python-biopython-1.73
           python-h5py
           python-lmfit
           python-matplotlib
           python-networkx
           python-numpy
           python-pandas
           python-psutil
           python-pysam
           python-seaborn
           python-tqdm))
    (native-inputs
     (list python-setuptools))
    (home-page "https://github.com/MrOlm/inStrain")
    (synopsis "Calculation of strain-level metrics")
    (description
     "inStrain is a Python program for analysis of co-occurring genome
populations from metagenomes that allows highly accurate genome comparisons,
analysis of coverage, microdiversity, and linkage, and sensitive SNP detection
with gene localization and synonymous non-synonymous identification.")
    ;; The tool itself says that the license is "MIT", but the repository
    ;; contains a LICENSE file with the GPLv3.
    ;; See https://github.com/MrOlm/inStrain/issues/51
    (license license:expat)))

(define-public gffcompare
  (let ((commit "be56ef4349ea3966c12c6397f85e49e047361c41")
        (revision "1"))
    (package
      (name "gffcompare")
      (version (git-version "0.10.15" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/gpertea/gffcompare/")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0cp5qpxdhw4mxpya5dld8wi3jk00zyklm6rcri426wydinrnfmkg"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f                    ; no check target
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)
           (add-before 'build 'copy-gclib-source
             (lambda* (#:key inputs #:allow-other-keys)
               (mkdir "../gclib")
               (copy-recursively
                (assoc-ref inputs "gclib-source") "../gclib")
               #t))
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
                 (install-file "gffcompare" bin)
                 #t))))))
      (native-inputs
       `(("gclib-source" ; see 'README.md' of gffcompare
          ,(let ((commit "54917d0849c1e83cfb057b5f712e5cb6a35d948f")
                 (revision "1")
                 (name "gclib")
                 (version (git-version "0.10.3" revision commit)))
             (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/gpertea/gclib/")
                     (commit commit)))
               (file-name (git-file-name name version))
               (sha256
                (base32 "0b51lc0b8syrv7186fd7n8f15rwnf264qgfmm2palrwks1px24mr")))))))
      (home-page "https://github.com/gpertea/gffcompare/")
      (synopsis "Tool for comparing or classifing transcripts of RNA-Seq")
      (description
       "@code{gffcompare} is a tool that can:
@enumerate
@item compare and evaluate the accuracy of RNA-Seq transcript assemblers
(Cufflinks, Stringtie);
@item collapse (merge) duplicate transcripts from multiple GTF/GFF3 files (e.g.
resulted from assembly of different samples);
@item classify transcripts from one or multiple GTF/GFF3 files as they relate to
reference transcripts provided in a annotation file (also in GTF/GFF3 format).
@end enumerate")
      (license
       (list
        license:expat                   ;license for gffcompare
        license:artistic2.0)))))        ;license for gclib

(define-public intervaltree
  (let ((commit "b90527f9e6d51cd36ecbb50429e4524d3a418ea5"))
    (package
      (name "intervaltree")
      (version (git-version "0.0.0" "1" commit))
      (source
        (origin
          (method git-fetch)
          (uri (git-reference
                 (url "https://github.com/ekg/intervaltree/")
                 (commit commit)))
          (file-name (git-file-name name version))
          (sha256
           (base32 "0rgv6q5fl4x5d74n6p5wvdna6zmbdbqpb4jqqh6vq3670gn08xad"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f                    ; No tests.
         #:make-flags
         ,#~(list (string-append "PREFIX=" #$output) "DESTDIR=\"\"")
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)))) ; There is no configure phase.
      (home-page "https://github.com/ekg/intervaltree")
      (synopsis "Minimal C++ interval tree implementation")
      (description "An interval tree can be used to efficiently find a set of
numeric intervals overlapping or containing another interval.  This library
provides a basic implementation of an interval tree using C++ templates,
allowing the insertion of arbitrary types into the tree.")
      (license license:expat))))

(define-public python-pypairix
  (package
    (name "python-pypairix")
    (version "0.3.9")
    ;; The tarball on pypi does not include the makefile to build the
    ;; programs.
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/4dn-dcic/pairix")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1jlxj3xa67q1i58pmbi6imhvl6f5w9m5qxv0xd45ba86mp8mnmvz"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'relax-gcc-14-strictness
            (lambda _
              (setenv "CFLAGS"
                      (string-append "-g -O2"
                                     " -Wno-error=incompatible-pointer-types"
                                     " -Wno-error=int-conversion"))))
          (add-before 'build 'build-programs
            (lambda _ (invoke "make")))
          (add-after 'install 'install-programs
            (lambda _
              (copy-recursively "bin" (string-append #$output "/bin")))))))
    (native-inputs
     (list python-pytest
           python-setuptools))
    (inputs
     (list zlib))
    (home-page "https://github.com/4dn-dcic/pairix")
    (synopsis "Support for querying pairix-indexed bgzipped text files")
    (description
     "Pypairix is a Python module for fast querying on a pairix-indexed
bgzipped text file that contains a pair of genomic coordinates per line.")
    (license license:expat)))

(define-public python-pyrodigal
  (package
    (name "python-pyrodigal")
    (version "3.6.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/althonos/pyrodigal")
              (commit (string-append "v" version))
              ;; XXX: vendor -> <https://github.com/hyattpd/Prodigal>
              (recursive? #t)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1gcvdrx0q730i0r3lndl7l7h0h8xvzsi09ymf14b498mj03yjdq9"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'patch-pyproject
            (lambda _
              (substitute* "pyproject.toml"
                ;; Extra keys present in "project": 'platform'
                (("platform =.*") ""))))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (invoke "python" "-m" "unittest" "pyrodigal.tests" "-vv")))))))
    (propagated-inputs (list python-archspec))
    (native-inputs
     (list cmake-minimal
           python-cython
           python-scikit-build-core))
    (home-page "https://github.com/althonos/pyrodigal")
    (synopsis "Cython bindings and Python interface for Prodigal")
    (description
     "This package offers Cython bindings and a Python interface for Prodigal.
 Prodigal is an ORF finder designed for both genomes and metagenomes.")
    (license license:gpl3)))

(define-public python-pyfaidx
  (package
    (name "python-pyfaidx")
    (version "0.7.2.1")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "pyfaidx" version))
              (sha256
               (base32
                "182ia2zg026lgphv68agxm9imw7649z9pdhfn8zkalrxkq5d5w1h"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; These tests require the download of large fasta.gz files.
      '(list "--ignore=tests/test_Fasta_bgzip.py")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-version
            (lambda _
              (substitute* "pyproject.toml"
                (("dynamic = \\[\"version\"\\]")
                 (string-append "version = \"" #$version "\""))))))))
    (native-inputs
     (list (libc-utf8-locales-for-target)         ;tests need "en_US.utf8"
           python-fsspec
           python-mock
           python-numpy
           python-pytest
           python-pytest-cov
           python-wheel))
    (propagated-inputs (list python-setuptools))
    (home-page "http://mattshirley.com")
    (synopsis "Random access to fasta subsequences")
    (description
     "This package provides procedures for efficient pythonic random access to
fasta subsequences.")
    (license license:bsd-3)))

(define-public python-cooler
  (package
    (name "python-cooler")
    (version "0.9.3")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "cooler" version))
       (sha256
        (base32
         "0qqb0i5449r6w871klsbjzxsjhdkpsaas3hvs9f1hc7ssrkf1vi1"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      '(list "-k"
             (string-append
              ;; These tests download files from the internet.
              "not test_fetch_chromsizes"
              " and not test_roundtrip"
              ;; This test depends on ipytree, which contains a lot of
              ;; minified JavaScript.
              " and not test_print_trees"))))
    (propagated-inputs
     (list python-asciitree
           python-biopython
           python-click
           python-cytoolz
           python-dask
           python-h5py
           python-multiprocess
           python-numpy
           python-pandas
           python-pyfaidx
           python-pypairix
           python-pysam
           python-pyyaml
           python-scipy
           python-simplejson))
    (native-inputs
     (list python-coverage
           python-hatchling
           python-isort
           python-pytest
           python-pytest-cov))
    ;; Almost all the projects of the Mirnylab are moved under Open2C umbrella
    (home-page "https://github.com/open2c/cooler")
    (synopsis "Sparse binary format for genomic interaction matrices")
    (description
     "Cooler is a support library for a sparse, compressed, binary persistent
storage format, called @code{cool}, used to store genomic interaction data,
such as Hi-C contact matrices.")
    (license license:bsd-3)))

(define-public python-cooltools
  (package
    (name "python-cooltools")
    (version "0.7.0")
    (source (origin
              (method url-fetch)
              (uri (pypi-uri "cooltools" version))
              (sha256
               (base32
                "076fgqzf6453cx5zs32vz0f5yvfg53w8ayq9s79jssy7gj2x89lb"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-cython
           python-pytest))
    (propagated-inputs
     (list python-bioframe
           python-click
           python-cooler
           python-joblib
           python-matplotlib
           python-multiprocess
           python-numba
           python-numpy
           python-pandas
           python-scikit-image
           python-scikit-learn
           python-scipy))
    (home-page "https://github.com/open2c/cooltools")
    (synopsis
     "Analysis tools for genomic interaction data stored in .cool format")
    (description
     "This package provides necessary tools for the analysis of the genomic
interaction data stored in @code{.cool} format.  This collection of tools
includes operations like compartment, insulation or peak calling.")
    (license license:expat)))

(define-public python-hicmatrix
  (package
    (name "python-hicmatrix")
    (version "17.1")
      (source
        (origin
          ;;Pypi sources do not contain any test
          (method git-fetch)
          (uri (git-reference
                 (url "https://github.com/deeptools/HiCMatrix")
                 (commit version)))
          (file-name (git-file-name name version))
          (sha256
            (base32
             "14gq7r9b64ff56l5f8h8zc2i2y3xri646jl0anb74japqxrwvlna"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         ;; TODO: this must be fixed in python-tables
         (add-before 'check 'find-blosc2
           (lambda* (#:key inputs #:allow-other-keys)
             (setenv "LD_LIBRARY_PATH"
                     (dirname (search-input-file
                               inputs "/lib/libblosc2.so"))))))))
    (inputs (list c-blosc2))
    (propagated-inputs
     (list python-cooler
           python-intervaltree
           python-numpy
           python-pandas
           python-scipy
           python-tables))
    (native-inputs
     (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/deeptools/HiCMatrix/")
    (synopsis "HiCMatrix class for HiCExplorer and pyGenomeTracks")
    (description
     "This helper package implements the @code{HiCMatrix} class for
the HiCExplorer and pyGenomeTracks packages.")
    (license license:gpl3+)))

(define-public python-pygenometracks
  (package
    (name "python-pygenometracks")
    (version "3.5")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "pyGenomeTracks" version))
       (sha256
        (base32
         "1l7smg2gc1vm2181lzmdnywb11gp6s6z6j444dbsigv65car8z8p"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f                       ;there are none
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'relax-requirements
           (lambda _
             (substitute* "setup.py"
               (("matplotlib ==3.1.1")
                "matplotlib >=3.1.1"))))
         (add-after 'unpack 'remove-invalid-syntax
           (lambda _
             (substitute* "setup.py"
               ((".\\*,") ",")))))))
    (propagated-inputs
     (list python-future
           python-gffutils
           python-hicmatrix
           python-intervaltree
           python-matplotlib
           python-numpy
           python-pybigwig
           python-pysam
           python-tqdm))
    (native-inputs
     (list python-pytest
           python-wheel))
    (home-page "https://pygenometracks.readthedocs.io")
    (synopsis "Program and library to plot beautiful genome browser tracks")
    (description
     "This package aims to produce high-quality genome browser tracks that
are highly customizable.  Currently, it is possible to plot: bigwig, bed (many
options), bedgraph, links (represented as arcs), and Hi-C matrices.
pyGenomeTracks can make plots with or without Hi-C data.")
    (license license:gpl3+)))

(define-public python-iced
  (package
    (name "python-iced")
    (version "0.5.13")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "iced" version))
       (sha256
        (base32
         "1fg6fxg6qhymr5d8drlvaclxgsfay2qcfsvpxkyy0r6cchscnpj5"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-pytest
           python-setuptools
           python-wheel))
    (propagated-inputs
     (list python-numpy
           python-pandas
           python-scipy
           python-scikit-learn))
    (home-page "https://github.com/hiclib/iced")
    (synopsis "ICE normalization")
    (description "This is a package for normalizing Hi-C contact counts
efficiently.")
    (license license:bsd-3)))

(define-public python-hic2cool
  (package
    (name "python-hic2cool")
    (version "1.0.1")
    ;; pypi sources do not contain the test_data directory and no test can be
    ;; run
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/4dn-dcic/hic2cool")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0k0i43z43rxbpna4hfci406ma906w893frfj3cha1n8drvhdql6c"))))
    (build-system pyproject-build-system)
    (native-inputs
     (list python-poetry-core))
    (propagated-inputs
     (list python-cooler
           python-h5py
           python-numpy
           python-pandas
           python-scipy))
    (home-page "https://github.com/4dn-dcic/hic2cool")
    (synopsis "Converter for .hic and .cool files")
    (description
     "This package provides a converter between @code{.hic} files (from
juicer) and single-resolution or multi-resolution @code{.cool} files (for
cooler).  Both @code{hic} and @code{cool} files describe Hi-C contact
matrices.")
    (license license:expat)))

(define-public python-scanorama
  (package
    (name "python-scanorama")
    (version "1.7.4")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/brianhie/scanorama")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1jpn4kq3qqa40xr0dwa9bw5cgga6h9ww9gfbyj6w3mfs8rv4w9rz"))))
    (build-system pyproject-build-system)
    (propagated-inputs (list python-annoy
                             python-fbpca
                             python-geosketch
                             python-intervaltree
                             python-matplotlib
                             python-numpy
                             python-scikit-learn
                             python-scipy))
    (native-inputs (list python-anndata python-pytest python-setuptools))
    (home-page "https://github.com/brianhie/scanorama")
    (synopsis
     "Panoramic stitching of heterogeneous single cell transcriptomic data")
    (description
     "Scanorama enables batch-correction and integration of heterogeneous
scRNA-seq datasets, which is described in the paper \"Efficient integration of
heterogeneous single-cell transcriptomes using Scanorama\" by Brian Hie, Bryan
Bryson, and Bonnie Berger.")
    (license license:expat)))

(define-public r-pore
  (package
    (name "r-pore")
    (version "0.24")
    (source
     (origin
       (method url-fetch)
       (uri
        (string-append "mirror://sourceforge/rpore/" version
                       "/poRe_" version ".tar.gz"))
       (sha256
        (base32 "0pih9nljbv8g4x8rkk29i7aqq681b782r5s5ynp4nw9yzqnmmksv"))))
    (properties `((upstream-name . "poRe")))
    (build-system r-build-system)
    (propagated-inputs
     (list r-bit64 r-data-table r-rhdf5 r-shiny r-svdialogs))
    (home-page "https://sourceforge.net/projects/rpore/")
    (synopsis "Visualize Nanopore sequencing data")
    (description
     "This package provides graphical user interfaces to organize and visualize Nanopore
sequencing data.")
    ;; This is free software but the license variant is unclear:
    ;; <https://github.com/mw55309/poRe_docs/issues/10>.
    (license license:bsd-3)))

(define-public r-xbioc
  (let ((revision "1")
        (commit "6ff0670a37ab3036aaf1d94aa4b208310946b0b5"))
    (package
      (name "r-xbioc")
      (version (git-version "0.1.16" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/renozao/xbioc")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0w8bsq5myiwkfhh83nm6is5ichiyvwa1axx2szvxnzq39x6knf66"))))
      (build-system r-build-system)
      (propagated-inputs
       (list r-annotationdbi
             r-assertthat
             r-biobase
             r-biocmanager
             r-digest
             r-pkgmaker
             r-plyr
             r-reshape2
             r-stringr))
      (home-page "https://github.com/renozao/xbioc/")
      (synopsis "Extra base functions for Bioconductor")
      (description "This package provides extra utility functions to perform
common tasks in the analysis of omics data, leveraging and enhancing features
provided by Bioconductor packages.")
      (license license:gpl3+))))

(define-public r-cssam
  (let ((revision "1")
        (commit "9ec58c982fa551af0d80b1a266890d92954833f2"))
    (package
      (name "r-cssam")
      (version (git-version "1.4" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/shenorrLab/csSAM")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "128syf9v39gk0z3ip000qpsjbg6l1siyq6c8b0hz41dzg5achyb3"))))
      (build-system r-build-system)
      (propagated-inputs
       (list r-formula
             r-ggplot2
             r-pkgmaker
             r-plyr
             r-rngtools
             r-scales))
      (home-page "https://github.com/shenorrLab/csSAM/")
      (synopsis "Cell type-specific statistical analysis of microarray")
      (description "This package implements the method csSAM that computes
cell-specific differential expression from measured cell proportions using
SAM.")
      ;; Any version
      (license license:lgpl2.1+))))

(define-public r-bseqsc
  (let ((revision "1")
        (commit "fef3f3e38dcf3df37103348b5780937982b43b98"))
    (package
      (name "r-bseqsc")
      (version (git-version "1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/shenorrLab/bseqsc")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1prw13wa20f7wlc3gkkls66n1kxz8d28qrb8icfqdwdnnv8w5qg8"))))
      (build-system r-build-system)
      (propagated-inputs
       (list r-abind
             r-annotationdbi
             r-biobase
             r-cssam
             r-dplyr
             r-e1071
             r-edger
             r-ggplot2
             r-nmf
             r-openxlsx
             r-pkgmaker
             r-plyr
             r-preprocesscore
             r-rngtools
             r-scales
             r-stringr
             r-xbioc))
      (home-page "https://github.com/shenorrLab/bseqsc")
      (synopsis "Deconvolution of bulk sequencing experiments using single cell data")
      (description "BSeq-sc is a bioinformatics analysis pipeline that
leverages single-cell sequencing data to estimate cell type proportion and
cell type-specific gene expression differences from RNA-seq data from bulk
tissue samples.  This is a companion package to the publication \"A
single-cell transcriptomic map of the human and mouse pancreas reveals inter-
and intra-cell population structure.\" Baron et al. Cell Systems (2016)
@url{https://www.ncbi.nlm.nih.gov/pubmed/27667365}.")
      (license license:gpl2+))))

(define-public porechop
  ;; The recommended way to install is to clone the git repository
  ;; https://github.com/rrwick/Porechop#installation
  (let ((commit "289d5dca4a5fc327f97b3f8cecb68ecaf1014861")
        (revision "1"))
    (package
      (name "porechop")
      (version (git-version "0.2.3" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/rrwick/Porechop")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "05ps43gig0d3ia9x5lj84lb00hbsl6ba9n7y7jz927npxbr2ym23"))))
      (build-system pyproject-build-system)
      (native-inputs (list python-setuptools python-wheel python-pytest))
      (home-page "https://github.com/rrwick/porechop")
      (synopsis
       "Finding, trimming or splitting adapters, in Oxford Nanopore reads")
      (description
       "The porechop package is a tool for finding and removing adapters from Oxford
Nanopore reads.  Adapters on the ends of reads are trimmed off, and when a read
has an adapter in its middle, it is treated as chimeric and chopped into
separate reads.  Porechop performs thorough alignments to effectively find
adapters, even at low sequence identity.  Porechop also supports demultiplexing
of Nanopore reads that were barcoded with the Native Barcoding Kit, PCR
Barcoding Kit or Rapid Barcoding Kit.")
      (license license:gpl3+))))

(define-public jamm
  (package
    (name "jamm")
    (version "1.0.7.6")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/mahmoudibrahim/JAMM")
             (commit (string-append "JAMMv" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0bsa5mf9n9q5jz7mmacrra41l7r8rac5vgsn6wv1fb52ya58b970"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (delete 'build)
          (replace 'install
            (lambda _
              (let ((libexec (string-append #$output "/libexec/jamm"))
                    (bin (string-append #$output "/bin")))
                (substitute* '("JAMM.sh"
                               "SignalGenerator.sh")
                  (("^sPath=.*")
                   (string-append "sPath=\"" libexec "\"\n")))
                (for-each (lambda (file)
                            (install-file file libexec))
                          (list "bincalculator.r"
                                "peakfinder.r"
                                "peakhelper.r"
                                "signalmaker.r"
                                "xcorr.r"
                                "xcorrhelper.r"
                                ;; Perl scripts
                                "peakfilter.pl"
                                "readshifter.pl"))

                (for-each
                 (lambda (script)
                   (chmod script #o555)
                   (install-file script bin)
                   (wrap-program (string-append bin "/" script)
                     `("PATH" ":" prefix
                       (,(string-append #$(this-package-input "coreutils") "/bin")
                        ,(string-append #$(this-package-input "gawk") "/bin")
                        ,(string-append #$(this-package-input "perl") "/bin")
                        ,(string-append #$(this-package-input "r-minimal") "/bin")))
                     `("PERL5LIB" ":" prefix (,(getenv "PERL5LIB")))
                     `("R_LIBS_SITE" ":" prefix (,(getenv "R_LIBS_SITE")))))
                 (list "JAMM.sh" "SignalGenerator.sh"))))))))
    (inputs
     (list bash
           coreutils
           gawk
           perl
           r-minimal
           ;;("r-parallel" ,r-parallel)
           r-signal
           r-mclust))
    (home-page "https://github.com/mahmoudibrahim/JAMM")
    (synopsis "Peak finder for NGS datasets")
    (description
     "JAMM is a peak finder for next generation sequencing datasets (ChIP-Seq,
ATAC-Seq, DNase-Seq, etc.) that can integrate replicates and assign peak
boundaries accurately.  JAMM is applicable to both broad and narrow
datasets.")
    (license license:gpl3+)))

(define-public ngless
  (package
    (name "ngless")
    (version "1.5.0")
    (source (origin
              (method url-fetch)
              (uri (hackage-uri "NGLess" version))
              (sha256
               (base32
                "0pljyrlpr9r3cl5311dhgxdl8y40szyi4vprn34i3piy0qrldymi"))))
    (build-system haskell-build-system)
    (arguments
     `(#:haddock? #f
       #:phases
       (modify-phases %standard-phases
         (add-before 'configure 'update-constraints
           (lambda _
             (substitute* "NGLess.cabal"
               (("\\b(base)\\s+[^,]+" all dep)
                dep))))
         (add-after 'unpack 'create-Versions.hs
           (lambda _
             (substitute* "NGLess/Dependencies/Versions.hs"
               (("bwaVersion = .+")
                (string-append "bwaVersion = \""
                               ,(package-version bwa) "\""))
               (("samtoolsVersion = .+")
                (string-append "samtoolsVersion = \""
                               ,(package-version samtools) "\""))
               (("prodigalVersion = .+")
                (string-append "prodigalVersion = \""
                               ,(package-version prodigal) "\""))
               (("megahitVersion = .+")
                (string-append "megahitVersion = \""
                               ,(package-version megahit) "\""))
               (("minimap2Version = .+")
                (string-append "minimap2Version = \""
                               ,(package-version minimap2) "\"")))))
         ;; See NGLess/FileManagement.hs.
         (add-after 'install 'wrap-program
             (lambda* (#:key inputs outputs #:allow-other-keys)
               (let* ((out (assoc-ref outputs "out"))
                      (bwa (search-input-file inputs "/bin/bwa"))
                      (samtools (search-input-file inputs "/bin/samtools"))
                      (prodigal (search-input-file inputs "/bin/prodigal"))
                      (minimap2 (search-input-file inputs "/bin/minimap2"))
                      (megahit (search-input-file inputs "/bin/megahit")))
                 (wrap-program (string-append out "/bin/ngless")
                   `("NGLESS_BWA_BIN" " " = (,bwa))
                   `("NGLESS_SAMTOOLS_BIN" " " = (,samtools))
                   `("NGLESS_PRODIGAL_BIN" " " = (,prodigal))
                   `("NGLESS_MINIMAP2_BIN" " " = (,minimap2))
                   `("NGLESS_MEGAHIT_BIN" " " = (,megahit))))))
          ;; Sanity check.
          (add-after 'wrap-program 'check-install
             (lambda* (#:key outputs #:allow-other-keys)
               (let* ((ngless (string-append (assoc-ref outputs "out") "/bin/ngless")))
                 (invoke ngless "--check-install"))))
          (add-after 'register 'remove-libraries
            (lambda* (#:key outputs #:allow-other-keys)
              (delete-file-recursively (string-append (assoc-ref outputs "out") "/lib")))))))
    (inputs (list bash-minimal
                  prodigal
                  bwa
                  samtools
                  minimap2
                  megahit
                  ghc-missingh
                  ghc-aeson
                  ghc-ansi-terminal
                  ghc-async
                  ghc-atomic-write
                  ghc-bytestring-lexing
                  ghc-conduit
                  ghc-conduit-algorithms
                  ghc-conduit-extra
                  ghc-configurator
                  ghc-convertible
                  ghc-data-default
                  ghc-edit-distance
                  ghc-either
                  ghc-errors
                  ghc-extra
                  ghc-file-embed
                  ghc-filemanip
                  ghc-hashable
                  ghc-hashtables
                  ghc-hostname
                  ghc-http-client
                  ghc-http-conduit
                  ghc-inline-c
                  ghc-inline-c-cpp
                  ghc-int-interval-map
                  ghc-network
                  ghc-optparse-applicative
                  ghc-primitive
                  ghc-random-shuffle
                  ghc-regex
                  ghc-resourcet
                  ghc-safe
                  ghc-stm-chans
                  ghc-stm-conduit
                  ghc-strict
                  ghc-tar
                  ghc-tar-conduit
                  ghc-unix-compat
                  ghc-unliftio
                  ghc-unliftio-core
                  ghc-vector
                  ghc-vector-algorithms
                  ghc-yaml
                  ghc-zlib
                  ghc-bzlib-conduit
                  ghc-double-conversion
                  ghc-safeio))
    (propagated-inputs
     (list r-r6 r-hdf5r r-iterators r-itertools r-matrix))
    (native-inputs (list ghc-hunit
                         ghc-quickcheck
                         ghc-tasty
                         ghc-tasty-hunit
                         ghc-tasty-quickcheck
                         ghc-tasty-th))
    (home-page "https://ngless.embl.de/")
    (synopsis "DSL for processing next-generation sequencing data")
    (description "Ngless is a domain-specific language for
@dfn{next-generation sequencing} (NGS) data processing.")
    (license license:expat)))

(define-public ghc-int-interval-map
  (let ((commit "678763de7fe6d7fa3f1c44b32d18ce58670270f4")
        (revision "1"))
    (package
      (name "ghc-int-interval-map")
      (version "0.0.0.0")
      (source
        (origin
           (method git-fetch)
           (uri (git-reference
                 (url "https://github.com/ngless-toolkit/interval-to-int.git")
                 (commit commit)))
           (file-name (git-file-name name version))
          (sha256 (base32 "0fd728b5if89vj5j4f9y7k0b2xv2ycz5a21iy15wbdcf5bhim7i8"))))
      (build-system haskell-build-system)
      (inputs
        (list ghc-either ghc-primitive ghc-vector ghc-vector-algorithms))
      (native-inputs
        (list ghc-hedgehog
              ghc-tasty
              ghc-tasty-hedgehog
              ghc-tasty-hunit
              ghc-tasty-quickcheck
              ghc-tasty-th))
      (home-page "https://github.com/luispedro/interval-to-int#readme")
      (synopsis "Interval map structure in Haskell")
      (description "An interval map structure that is optimized for low
memory (each interval is represented by about 3 words + whatever the
cargo is) and has semantics that are appropriate for genomic intervals
(namely, intervals can overlap and queries will return all matches
together).  It also designed to be used in two phases: a construction
phase + query phase).")
      (license license:expat))))

(define-public filtlong
  ;; The recommended way to install is to clone the git repository
  ;; https://github.com/rrwick/Filtlong#installation
  ;; and the latest release is more than nine months old
  (let ((commit "d1bb46dfe8bc7efe6257b5ce222c04bfe8aedaab")
        (revision "1"))
    (package
      (name "filtlong")
      (version (git-version "0.2.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/rrwick/Filtlong")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1xr92r820x8qlkcr3b57iw223yq8vjgyi42jr79w2xgw47qzr575"))))
      (build-system gnu-build-system)
      (arguments
       (list
        #:tests? #f                    ; no check target
        #:phases
        #~(modify-phases %standard-phases
            (delete 'configure)
            (replace 'install
              (lambda _
                (let ((bin (string-append #$output "/bin"))
                      (scripts (string-append #$output "/share/filtlong/scripts")))
                  (install-file "bin/filtlong" bin)
                  (install-file "scripts/histogram.py" scripts)
                  (install-file "scripts/read_info_histograms.sh" scripts))))
            (add-after 'install 'wrap-program
              (lambda _
                (let ((path (getenv "GUIX_PYTHONPATH")))
                  (wrap-program (string-append #$output
                                               "/share/filtlong/scripts/histogram.py")
                    `("GUIX_PYTHONPATH" ":" prefix (,path))))))
            (add-before 'check 'patch-tests
              (lambda _
                (substitute* "scripts/read_info_histograms.sh"
                  (("awk") (which "gawk"))))))))
      (inputs
       (list bash-minimal
             gawk                     ;for read_info_histograms.sh
             python-wrapper           ;required for histogram.py
             zlib))
      (home-page "https://github.com/rrwick/Filtlong/")
      (synopsis "Tool for quality filtering of Nanopore and PacBio data")
      (description
       "The Filtlong package is a tool for filtering long reads by quality.
It can take a set of long reads and produce a smaller, better subset.  It uses
both read length (longer is better) and read identity (higher is better) when
choosing which reads pass the filter.")
      (license (list license:gpl3       ;filtlong
                     license:asl2.0))))) ;histogram.py

(define-public nanopolish
  (package
    (name "nanopolish")
    (version "0.14.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/jts/nanopolish")
             (commit (string-append "v" version))
             (recursive? #t)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1ha9v5ia0qrwgav0956cfc7n64wjm9a9w3rvkg21g37a994yixg5"))
       (modules '((guix build utils)))
       (snippet
        ;; TODO: unbundle slow5lib
        '(begin (for-each delete-file-recursively
                          '("htslib" "minimap2"))))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:make-flags
      #~(list "HDF5=noinstall"
              "EIGEN=noinstall"
              "HTS=noinstall"
              "MINIMAP2=noinstall"
              (string-append "CC=" #$(cc-for-target)))
      #:tests? #f                    ; no check target
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'find-eigen
            (lambda* (#:key inputs #:allow-other-keys)
              (setenv "CPATH"
                      (string-append
                       (search-input-directory inputs "/include/eigen3")
                       ":" (or (getenv "CPATH") "")))))
          (delete 'configure)
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin"))
                    (scripts (string-append #$output "/share/nanopolish/scripts")))

                (install-file "nanopolish" bin)
                (for-each (lambda (file) (install-file file scripts))
                          (find-files "scripts" ".*")))))
          (add-after 'install 'wrap-programs
            (lambda* (#:key inputs #:allow-other-keys)
              (let ((pythonpath (getenv "GUIX_PYTHONPATH"))
                    (perl5lib (getenv "PERL5LIB"))
                    (scripts (string-append #$output "/share/nanopolish/scripts"))
                    (guile (search-input-file inputs "bin/guile")))
                (for-each (lambda (file)
                            (wrap-program file `("GUIX_PYTHONPATH" ":" prefix (,pythonpath))))
                          (find-files scripts "\\.py"))
                (for-each (lambda (file)
                            (wrap-script file #:guile guile
                                         `("PERL5LIB" ":" prefix (,perl5lib))))
                          (find-files scripts "\\.pl"))))))))
    (inputs
     (list bash-minimal
           guile-3.0                    ;for wrappers
           eigen
           hdf5
           htslib
           minimap2
           perl
           bioperl-minimal
           perl-getopt-long
           python-wrapper
           python-biopython
           python-numpy
           python-pysam
           python-scikit-learn
           python-scipy
           zlib))
    (native-inputs (list cmake-minimal))
    (home-page "https://github.com/jts/nanopolish")
    (synopsis "Signal-level analysis of Oxford Nanopore sequencing data")
    (description
     "This package analyses the Oxford Nanopore sequencing data at signal-level.
Nanopolish can calculate an improved consensus sequence for a draft genome
assembly, detect base modifications, call SNPs (Single nucleotide
polymorphisms) and indels with respect to a reference genome and more.")
    (license license:expat)))

(define-public cnvkit
  (package
    (name "cnvkit")
    (version "0.9.12")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/etal/cnvkit")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "090yh17symcahddx399kcx0mcw4gdrcc2jil3p8lb92r8c8kglb5"))))
    (build-system pyproject-build-system)
    (propagated-inputs
     (list python-biopython
           python-future
           python-matplotlib
           python-numpy
           python-pandas
           python-pomegranate
           python-pyfaidx
           python-pysam
           python-reportlab
           python-scikit-learn
           python-scipy
           ;; R packages
           r-dnacopy))
    (inputs (list r-minimal)) ;for tests
    (native-inputs (list python-pytest python-setuptools python-wheel))
    (home-page "https://cnvkit.readthedocs.org/")
    (synopsis "Copy number variant detection from targeted DNA sequencing")
    (description
     "CNVkit is a Python library and command-line software toolkit to infer
and visualize copy number from high-throughput DNA sequencing data.  It is
designed for use with hybrid capture, including both whole-exome and custom
target panels, and short-read sequencing platforms such as Illumina and Ion
Torrent.")
    (license license:asl2.0)))

(define-public python-pyfit-sne
  (package
    (name "python-pyfit-sne")
    (version "1.2.1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/KlugerLab/pyFIt-SNE")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0f3n7wcmxbnqiisgimhpa6p5chqpb1hj69i6rpg2hv2671i8nn68"))))
    (build-system pyproject-build-system)
    (arguments
     '(#:tests? #f)) ;there are none
    (propagated-inputs (list python-numpy))
    (inputs (list fftw))
    (native-inputs (list python-cython python-setuptools python-wheel))
    (home-page "https://github.com/KlugerLab/pyFIt-SNE")
    (synopsis "FFT-accelerated Interpolation-based t-SNE")
    (description
     "t-Stochastic Neighborhood Embedding (t-SNE) is a highly successful
method for dimensionality reduction and visualization of high dimensional
datasets.  A popular implementation of t-SNE uses the Barnes-Hut algorithm to
approximate the gradient at each iteration of gradient descent.  This package
is a Cython wrapper for FIt-SNE.")
    (license license:bsd-4)))

(define-public bbmap
  (package
    (name "bbmap")
    (version "39.01")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "mirror://sourceforge/bbmap/BBMap_" version ".tar.gz"))
              (sha256
               (base32
                "1rlkw2mlkn699dk9n5lnigmvwq3zr2f8hnq9plx7zi1h06jqsq4q"))))
    (build-system ant-build-system)
    (arguments
     (list
      #:build-target "dist"
      #:tests? #f ; there are none
      #:make-flags
      #~(list (string-append "-Dmpijar="
                             #$(this-package-input "java-openmpi")
                             "/lib/mpi.jar"))
      #:modules '((guix build ant-build-system)
                  (guix build utils)
                  (guix build java-utils))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'build 'build-jni-library
            (lambda _
              (with-directory-excursion "jni"
                (invoke "make" "-f" "makefile.linux"))))
          ;; There is no install target
          (replace 'install (install-jars "dist"))
          (add-after 'install 'install-scripts-and-documentation
            (lambda _
              (substitute* "calcmem.sh"
                (("\\| awk ") (string-append "| " (which "awk") " ")))
              (let* ((scripts (find-files "." "\\.sh$"))
                     (bin (string-append #$output "/bin"))
                     (doc (string-append #$output "/share/doc/bbmap"))
                     (jni (string-append #$output "/lib/jni")))
                (substitute* scripts
                  (("\\$DIR\"\"docs") doc)
                  (("^CP=.*")
                   (string-append "CP=" #$output "/share/java/BBTools.jar\n"))
                  (("^NATIVELIBDIR.*")
                   (string-append "NATIVELIBDIR=" jni "\n"))
                  (("CMD=\"java")
                   (string-append "CMD=\"" (which "java"))))
                (for-each (lambda (script) (install-file script bin)) scripts)

                ;; Install JNI library
                (install-file "jni/libbbtoolsjni.so" jni)

                ;; Install documentation
                (install-file "docs/readme.txt" doc)
                (copy-recursively "docs/guides" doc)))))
       #:jdk openjdk11))
    (inputs
     (list gawk java-eclipse-jdt-core java-eclipse-jdt-compiler-apt
           java-openmpi))
    (home-page "https://sourceforge.net/projects/bbmap/")
    (synopsis "Aligner and other tools for short sequencing reads")
    (description
     "This package provides bioinformatic tools to align, deduplicate,
reformat, filter and normalize DNA and RNA-seq data.  It includes the
following tools: BBMap, a short read aligner for DNA and RNA-seq data; BBNorm,
a kmer-based error-correction and normalization tool; Dedupe, a tool to
simplify assemblies by removing duplicate or contained subsequences that share
a target percent identity; Reformat, to convert reads between
fasta/fastq/scarf/fasta+qual/sam, interleaved/paired, and ASCII-33/64, at over
500 MB/s; and BBDuk, a tool to filter, trim, or mask reads with kmer matches
to an artifact/contaminant file.")
    (license license:bsd-3)))

(define-public velvet
  (package
    (name "velvet")
    (version "1.2.10")
    (source (origin
              (method url-fetch)
              (uri (string-append "https://www.ebi.ac.uk/~zerbino/velvet/"
                                  "velvet_" version ".tgz"))
              (sha256
               (base32
                "0h3njwy66p6bx14r3ar1byb0ccaxmxka4c65rn4iybyiqa4d8kc8"))
              ;; Delete bundled libraries
              (modules '((guix build utils)))
              (snippet
               '(begin
                  (delete-file "Manual.pdf")
                  (delete-file-recursively "third-party")))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:make-flags #~(list "OPENMP=t")
      #:test-target "test"
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (add-after 'unpack 'fix-zlib-include
            (lambda _
              (substitute* "src/binarySequences.c"
                (("../third-party/zlib-1.2.3/zlib.h") "zlib.h"))))
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin"))
                    (doc (string-append #$output "/share/doc/velvet")))
                (mkdir-p bin)
                (mkdir-p doc)
                (install-file "velveth" bin)
                (install-file "velvetg" bin)
                (install-file "Manual.pdf" doc)
                (install-file "Columbus_manual.pdf" doc)))))))
    (inputs
     (list openmpi zlib))
    (native-inputs
     (list (texlive-local-tree
            (list texlive-infwarerr
                  texlive-kvoptions
                  texlive-pdftexcmds))))
    (home-page "https://www.ebi.ac.uk/~zerbino/velvet/")
    (synopsis "Nucleic acid sequence assembler for very short reads")
    (description
     "Velvet is a de novo genomic assembler specially designed for short read
sequencing technologies, such as Solexa or 454.  Velvet currently takes in
short read sequences, removes errors then produces high quality unique
contigs.  It then uses paired read information, if available, to retrieve the
repeated areas between contigs.")
    (license license:gpl2+)))

(define-public vembrane
  (package
    (name "vembrane")
    (version "1.0.7")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                     (url "https://github.com/vembrane/vembrane")
                     (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "127wmwj0162nfaql68jwxlkz7rbnjya70xrj4j8zwvcnxcj7x5v3"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'use-poetry-core
            (lambda _
              ;; Patch to use the core poetry API.
              (substitute* "pyproject.toml"
                (("poetry.masonry.api") "poetry.core.masonry.api")))))))
    (inputs
     (list python-asttokens
           python-intervaltree
           python-numpy
           python-pysam
           python-pyyaml))
    (native-inputs
     (list python-poetry-core
           python-pytest))
    (home-page "https://github.com/vembrane/vembrane")
    (synopsis "Filter VCF/BCF files with Python expressions")
    (description "Vembrane simultaneously filters variants based on
any INFO or FORMAT field, CHROM, POS, ID, REF, ALT, QUAL, FILTER, and the
annotation field ANN.  When filtering based on ANN, annotation entries are
filtered first.  If no annotation entry remains, the entire variant is
deleted.")
    (license license:expat)))

(define-public arriba
  (package
    (name "arriba")
    (version "1.0.1")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "https://github.com/suhrig/arriba/releases/"
                           "download/v" version "/arriba_v" version ".tar.gz"))
       (sha256
        (base32
         "0jx9656ry766vb8z08m1c3im87b0c82qpnjby9wz4kcz8vn87dx2"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "Makefile"
                (("-I\\$\\(HTSLIB\\)/htslib")
                 (string-append "-I"
                                (search-input-directory inputs "/include/htslib")))
                ((" \\$\\(HTSLIB\\)/libhts.a")
                 (string-append " " (search-input-file inputs "/lib/libhts.so"))))
              (let ((samtools (search-input-file inputs "/bin/samtools")))
                (substitute* "run_arriba.sh"
                  (("^STAR ")
                   (string-append (search-input-file inputs "/bin/STAR") " "))
                  (("samtools --version-only")
                   (string-append samtools " --version-only"))
                  (("samtools index")
                   (string-append samtools " index"))
                  (("samtools sort")
                   (string-append samtools " sort"))))))
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin")))
                (install-file "arriba" bin)
                (install-file "run_arriba.sh" bin)
                (install-file "draw_fusions.R" bin)
                (wrap-program (string-append bin "/draw_fusions.R")
                  `("R_LIBS_SITE" ":" prefix (,(getenv "R_LIBS_SITE"))))))))))
    (inputs
     (list bash-minimal
           htslib
           r-minimal
           r-circlize
           r-genomicalignments
           r-genomicranges
           samtools
           star
           zlib))
    (home-page "https://github.com/suhrig/arriba")
    (synopsis "Gene fusion detection from RNA-Seq data")
    (description
     "Arriba is a command-line tool for the detection of gene fusions from
RNA-Seq data.  It was developed for the use in a clinical research setting.
Therefore, short runtimes and high sensitivity were important design criteria.
It is based on the fast STAR aligner and the post-alignment runtime is
typically just around two minutes.  In contrast to many other fusion detection
tools which build on STAR, Arriba does not require to reduce the
@code{alignIntronMax} parameter of STAR to detect small deletions.")
    ;; All code is under the Expat license with the exception of
    ;; "draw_fusions.R", which is under GPLv3.
    (license (list license:expat license:gpl3))))

(define-public adapterremoval
  (package
    (name "adapterremoval")
    (version "2.3.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/MikkelSchubert/adapterremoval")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0wyr182667wqdmzhwwa6f7dddr8kk0scgzb8lsm23wim544mvn6i"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:make-flags
      #~(list "COLOR_BUILD=no"
              (string-append "PREFIX=" #$output))
      #:test-target "test"
      #:phases
      '(modify-phases %standard-phases
         (delete 'configure))))
    (inputs (list zlib))
    (home-page "https://adapterremoval.readthedocs.io/")
    (synopsis "Rapid sequence adapter trimming, identification, and read merging")
    (description
     "This program searches for and removes remnant adapter sequences from
@dfn{High-Throughput Sequencing} (HTS) data and (optionally) trims low quality
bases from the 3' end of reads following adapter removal.  AdapterRemoval can
analyze both single end and paired end data, and can be used to merge
overlapping paired-ended reads into (longer) consensus sequences.
Additionally, the AdapterRemoval may be used to recover a consensus adapter
sequence for paired-ended data, for which this information is not available.")
    (license license:gpl3+)))

(define-public checkm
  (package
    (name "checkm")
    (version "1.1.3")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "checkm-genome" version))
       (sha256
        (base32 "0i2nnki639hgjag17wlva2x0ymn37b4krqsf6akxddykhfbkdnkz"))))
    (build-system pyproject-build-system)
    (arguments
     `(#:tests? #f ;Some tests fail for unknown reasons.
       #:phases (modify-phases %standard-phases
                  (add-before 'check 'set-HOME
                    (lambda _
                      (setenv "HOME" "/tmp"))))))
    (inputs (list python-dendropy python-matplotlib python-numpy python-pysam
                  python-scipy))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://ecogenomics.github.io/CheckM/")
    (synopsis "Assess the quality of putative genome bins")
    (description
     "CheckM provides a set of tools for assessing the quality of genomes
recovered from isolates, single cells, or metagenomes.  It provides robust
estimates of genome completeness and contamination by using collocated sets of
genes that are ubiquitous and single-copy within a phylogenetic lineage.
Assessment of genome quality can also be examined using plots depicting key
genomic characteristics (e.g., GC, coding density) which highlight sequences
outside the expected distributions of a typical genome.  CheckM also provides
tools for identifying genome bins that are likely candidates for merging based
on marker set compatibility, similarity in genomic characteristics, and
proximity within a reference genome.")
    (license license:gpl3+)))

(define-public chromap
  (package
    (name "chromap")
    (version "0.2.7")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/haowenz/chromap")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "099y3kac6npiqyx5prc9ggigw6248j7kchjznqvd9hxkq24rrsqs"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #false ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (replace 'install
            (lambda _
              (install-file "chromap" (string-append #$output "/bin")))))))
    (inputs (list zlib))
    (home-page "https://github.com/haowenz/chromap")
    (synopsis "Fast alignment and preprocessing of chromatin profiles")
    (description "Chromap is a fast method for aligning and preprocessing high
throughput chromatin profiles.  Typical use cases include:

@itemize
@item trimming sequencing adapters, mapping bulk ATAC-seq or ChIP-seq genomic
  reads to the human genome and removing duplicates;
@item trimming sequencing adapters, mapping single cell ATAC-seq genomic reads
  to the human genome, correcting barcodes, removing duplicates and performing
  Tn5 shift;
@item split alignment of Hi-C reads against a reference genome.
@end itemize")
    (supported-systems '("x86_64-linux")) ;requires CPU with SSE4.1
    (license license:expat)))

(define-public umi-tools
  (package
    (name "umi-tools")
    (version "1.1.6")
    ;; TODO: Delete generated Cython C files.
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
              (url "https://github.com/CGATOxford/UMI-tools")
              (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1liykfj4msvcgk8an5qq802jcxwlijqxrvijipqj1pwpxqzl9qnh"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list "--ignore=tests/test_style.py"
              "--ignore=step1_unit_test.py")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-bash-path
            (lambda _
              (substitute* "tests/test_umi_tools.py"
                (("/bin/bash")
                 (which "sh")))))
          (add-before 'check 'build-extensions
            (lambda _
              (invoke "python" "setup.py" "build_ext" "--inplace"))))))
    (inputs
     (list python-pandas
           python-future
           python-scipy
           python-matplotlib
           python-regex
           python-pybktree
           python-scipy
           python-pysam))
    (native-inputs
     (list python-cython
           python-pyaml
           python-pytest
           python-setuptools))
    (home-page "https://github.com/CGATOxford/UMI-tools")
    (synopsis "Tools for analyzing unique modular identifiers")
    (description "This package provides tools for dealing with @dfn{Unique
Molecular Identifiers} (UMIs) and @dfn{Random Molecular Tags} (RMTs) in
genetic sequences.  There are six tools: the @code{extract} and
@code{whitelist} commands are used to prepare a fastq containing UMIs @code{+/-}
cell barcodes for alignment.  The remaining commands, @code{group},
@code{dedup}, and @code{count}/@code{count_tab}, are used to identify PCR
duplicates using the UMIs and perform different levels of analysis depending
on the needs of the user.")
    (license license:expat)))

(define-public ataqv
  (package
    (name "ataqv")
    (version "1.0.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/ParkerLab/ataqv")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "031xr6jx1aprh26y5b1lv3gzrlmzg4alfl73vvshymx8cq8asrqi"))))
    (build-system gnu-build-system)
    (arguments
     `(#:make-flags
       ,#~(list (string-append "prefix=" #$output)
                (string-append "BOOST_ROOT="
                               #$(this-package-input "boost"))
                (string-append "HTSLIB_ROOT="
                               #$(this-package-input "htslib")))
       #:test-target "test"
       #:phases
       (modify-phases %standard-phases
         (delete 'configure))))
    (inputs
     (list boost-1.83 htslib ncurses zlib))
    (native-inputs
     (list lcov))
    (home-page "https://github.com/ParkerLab/ataqv")
    (synopsis "Toolkit for quality control and visualization of ATAC-seq data")
    (description "This package provides a toolkit for measuring and comparing
ATAC-seq results.  It was written to make it easier to spot differences that
might be caused by ATAC-seq library prep or sequencing.  The main program,
@code{ataqv}, examines aligned reads and reports some basic metrics.")
    (license license:gpl3+)))

(define-public r-psiplot
  (package
    (name "r-psiplot")
    (version "2.3.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/kcha/psiplot")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "08438h16cfry5kqh3y9hs8q1b1a8bxhblsm75knviz5r6q0n1jxh"))))
    (build-system r-build-system)
    (propagated-inputs
     (list r-mass
           r-dplyr
           r-tidyr
           r-purrr
           r-readr
           r-magrittr
           r-ggplot2))
    (native-inputs (list r-testthat))
    (home-page "https://github.com/kcha/psiplot")
    (synopsis "Plot percent spliced-in values of alternatively-spliced exons")
    (description
     "PSIplot is an R package for generating plots of @dfn{percent
spliced-in} (PSI) values of alternatively-spliced exons that were computed by
vast-tools, an RNA-Seq pipeline for alternative splicing analysis.  The plots
are generated using @code{ggplot2}.")
    (license license:expat)))

(define-public r-scopeloomr
  (let ((commit "99726f5f7da794042036b73924b6a10d6e7b4d5d")
        (revision "1"))
    (package
      (name "r-scopeloomr")
      (version (git-version "0.13.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/aertslab/SCopeLoomR")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1ci17ms0c0hf7yfp9ckcg7a2y1s0nm19jj3cifsd55hwc0gdglmz"))))
      (properties `((upstream-name . "SCopeLoomR")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-base64enc r-hdf5r r-igraph r-matrix r-plyr r-rjson r-rlist))
      (home-page "https://github.com/aertslab/SCopeLoomR")
      (synopsis "Build .loom files and extract data from them")
      (description
       "This is an R package to build generic @code{.loom} files aligning with
the default naming convention of the @code{.loom} format and to integrate
other data types e.g.: regulons (SCENIC), clusters from Seurat, trajectory
information...  The package can also be used to extract data from @code{.loom}
files.")
      (license license:expat))))

(define-public rscape
  (package
    (name "rscape")
    (version "2.0.0.q")
    (source (origin
              (method url-fetch)
              (uri (string-append "http://eddylab.org/software/rscape/"
                                  "rscape_v" version ".tar.gz"))
              (sha256
               (base32
                "1jabvm3fzh8iy4803ns12v1fsy28x6wdy8wx4ik8y0mfac4h787q"))))
    (build-system gnu-build-system)
    (propagated-inputs (list gsl openmpi))
    (native-inputs (list automake autoconf))
    (home-page "https://github.com/EddyRivasLab/R-scape")
    (synopsis "RNA structural covariation above phylogenetic expectation")
    (description
     "R-scape discovers RNA secondary structure consensus elements.
These elements include riboswitches and ribozymes.  It utilizes probabilistic
modeling of sequence alignments, explicitly considering folding dependencies.
The tool enables the de novo search for new structural elements and
facilitates comparative analysis of known RNA families.")
    (license license:bsd-3)))

(define-public r-databaselinke-r
  (let ((commit "cf3d6cc3d36f2e1c9a557390232e9a8ed5abb7fd")
        (revision "1"))
    (package
      (name "r-databaselinke-r")
      (version (git-version "1.7.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/vertesy/DatabaseLinke.R")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0hk76sb3w1v8a7c1knpc572ypsbgqlrv0p49c9y55a0dr12n16s1"))))
      (properties `((upstream-name . "DatabaseLinke.R")))
      (build-system r-build-system)
      (propagated-inputs (list r-readwriter))
      (home-page "https://github.com/vertesy/DatabaseLinke.R")
      (synopsis
       "Parse links to databases from your list of gene symbols")
      (description
       "This package provides a set of functions to parse and open (search
query) links to genomics related and other websites for R.  Useful when you
want to explore e.g.: the function of a set of differentially expressed
genes.")
      (license license:gpl3))))

(define-public r-seurat-utils
  (let ((commit "c0374cc9e25ce391ba8013fda0f8c7babbb9201d")
        (revision "1"))
    (package
      (name "r-seurat-utils")
      (version (git-version "2.5.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/vertesy/Seurat.utils")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "15l86b43q245gzz7gsr5rhs4sir74lc14d64yqxfqcb0zrb2bzzd"))))
      (properties `((upstream-name . "Seurat.utils")))
      (build-system r-build-system)
      (propagated-inputs (list r-codeandroll2
                               r-cowplot
                               r-databaselinke-r
                               r-dplyr
                               r-enhancedvolcano
                               r-foreach
                               r-ggcorrplot
                               r-ggexpress
                               r-ggplot2
                               r-ggpubr
                               r-ggrepel
                               r-hgnchelper
                               r-htmlwidgets
                               r-job
                               r-magrittr
                               r-markdownhelpers
                               r-markdownreports
                               r-matrix
                               r-matrixstats
                               r-pheatmap
                               r-plotly
                               r-princurve
                               r-qs
                               r-r-utils
                               r-readr
                               r-readwriter
                               r-reshape2
                               r-rstudioapi
                               r-scales
                               r-seurat
                               r-soupx
                               r-sparsematrixstats
                               r-stringendo
                               r-stringr
                               r-tibble
                               r-tictoc
                               r-tidyverse
                               r-vroom))
      (home-page "https://github.com/vertesy/Seurat.utils")
      (synopsis "Collection of utility functions for Seurat")
      (description
       "This is a collection of utility functions for Seurat.  These functions
allow the automation and multiplexing of plotting, 3D plotting, visualization
of statistics & QC, interaction with the Seurat object.  Some functionalities
require functions from CodeAndRoll and MarkdownReports libraries.")
      (license license:gpl3))))

(define-public r-seuratextend
  (let ((commit "5dacd7c1e1da3bc0e78ab81d60056f6d7da3d386")
        (revision "1"))
    (package
      (name "r-seuratextend")
      (version (git-version "1.2.5" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/huayc09/SeuratExtend")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1fnh1idf871jfkpzmr2py15h5v5d9ykh9p0kvlpx7riiidi8wsl2"))))
      (properties `((upstream-name . "SeuratExtend")))
      (build-system r-build-system)
      (propagated-inputs (list r-biocmanager
                               r-dplyr
                               r-ggplot2
                               r-ggpubr
                               r-glue
                               r-hdf5r
                               r-magrittr
                               r-mosaic
                               r-purrr
                               r-remotes
                               r-reshape2
                               r-reticulate
                               r-rlist
                               r-scales
                               r-seurat
                               r-seuratextenddata
                               r-seuratobject
                               r-tidyr))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/huayc09/SeuratExtend")
      (synopsis "Enhanced toolkit for scRNA-seq analysis")
      (description
       "This package is designed to improve and simplify the analysis of
scRNA-seq data.  It uses the Seurat object for this purpose.  It provides an
array of enhanced visualization tools, an integrated functional and pathway
analysis pipeline, seamless integration with popular Python tools, and a suite
of utility functions to aid in data manipulation and presentation.")
      (license license:gpl3+))))

(define-public r-seuratextenddata
  (let ((commit "e7f17d4bacca436705b06b2cd0c879f83add9271")
        (revision "1"))
    (package
      (name "r-seuratextenddata")
      (version (git-version "0.2.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/huayc09/SeuratExtendData")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1vl2wvpbv5q04mzc6z3f1jw2jc5i8pw5vm8y1rg5b1cihkwppvq5"))))
      (properties `((upstream-name . "SeuratExtendData")))
      (build-system r-build-system)
      (home-page "https://github.com/huayc09/SeuratExtendData")
      (synopsis "Data attached to SeuratExtend package")
      (description "This package provides data for the SeuratExtend tool.")
      (license license:gpl3+))))

(define-public r-seuratwrappers
  ;; There are no releases or tags.
  (let ((commit "d28512f804d5fe05e6d68900ca9221020d52cf1d")
        (revision "1"))
    (package
      (name "r-seuratwrappers")
      (version (git-version "0.3.1" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/satijalab/seurat-wrappers")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0rm74y2fj3cmiqn7jz1ald8jbw53c2qxkj3mgl4pxih9vx39jhgy"))))
      (properties `((upstream-name . "SeuratWrappers")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biocmanager
             r-cowplot
             r-ggplot2
             r-igraph
             r-matrix
             r-remotes
             r-rlang
             r-rsvd
             r-r-utils
             r-seurat))
      (home-page "https://github.com/satijalab/seurat-wrappers")
      (synopsis "Community-provided methods and extensions for the Seurat object")
      (description
       "SeuratWrappers is a collection of community-provided methods and
extensions for Seurat, curated by the Satija Lab at NYGC.  These methods
comprise functionality not presently found in Seurat, and are able to be
updated much more frequently.")
      (license license:gpl3))))

(define-public python-ctxcore
  (package
    (name "python-ctxcore")
    (version "0.2.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/aertslab/ctxcore")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0nv4lc46cnzpg5gcdxrsv7b4srmkq55zl3rcadw5pn3yyz5fzd2k"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'pretend-version
            ;; The version string is usually derived via setuptools-scm, but
            ;; it doesn't work without the .git directory.
            (lambda _
              (setenv "SETUPTOOLS_SCM_PRETEND_VERSION" #$version))))))
    (propagated-inputs
     (list python-cytoolz
           python-frozendict
           python-numba
           python-numpy
           python-pandas
           python-pyarrow
           python-pyyaml
           python-tqdm))
    (native-inputs
     (list python-pytest
           python-setuptools
           python-setuptools-scm
           python-wheel))
    (home-page "https://github.com/aertslab/ctxcore")
    (synopsis "Core functions for pycisTarget and the SCENIC tool suite")
    (description
     "ctxcore is part of the SCENIC suite of tools.  It provides core functions for
pycisTarget and SCENIC.")
    (license license:gpl3+)))

(define-public python-arboreto
  (package
    (name "python-arboreto")
    (version "0.1.6")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/aertslab/arboreto")
                    (commit "2f475dca08f47a60acc2beb8dd897e77b7495ca4")))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l0im8ay7l2d24f7vaha454vsaha9s36bfqhbijg3b8ir8apsd7l"))))
    (build-system pyproject-build-system)
    ;; Lots of tests fail because python-distributed fails to start the
    ;; "Nanny" process.
    (arguments '(#:tests? #false))
    (native-inputs
     (list python-setuptools
           python-wheel))
    (propagated-inputs
     (list python-bokeh
           python-dask
           python-distributed
           python-lz4
           python-numpy
           python-pandas
           python-pyarrow
           python-scikit-learn
           python-scipy
           python-tornado-6))
    (home-page "https://github.com/aertslab/arboreto")
    (synopsis "Gene regulatory network inference using tree-based ensemble regressors")
    (description
     "This package implements scalable gene regulatory network inference using
tree-based ensemble regressors.")
    (license license:bsd-3)))

(define-public pyscenic
  ;; We use this commit because the last release is not compatible with the
  ;; current version of numpy.  See
  ;; https://github.com/aertslab/pySCENIC/issues/579#issuecomment-2405207860
  (let ((commit "eaf23eb1fdcaae79b273de56b374b71aa8afde5a")
        (revision "1"))
    (package
      (name "pyscenic")
      (version (git-version "0.12.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/aertslab/pySCENIC")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "1si2ifmj0cy2yba8lw5m7mg97iqaxi0cwhy3j43rz5bzkp0cah8n"))))
      (build-system pyproject-build-system)
      (arguments
       (list
        #:phases
        '(modify-phases %standard-phases
           ;; The cli modules referenced here have been removed, so this
           ;; breaks the sanity check.
           (add-after 'unpack 'do-not-reference-deleted-modules
             (lambda _
               (substitute* "setup.py"
                 (("\"db2feather = .*\",") "")
                 (("\"csv2loom = .*\",") "")
                 (("\"invertdb = .*\",") "")
                 (("\"gmt2regions = .*\",") ""))))
           ;; Numba needs a writable dir to cache functions.
           (add-before 'check 'set-numba-cache-dir
             (lambda _
               (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
      (propagated-inputs
       (list python-aiohttp
             python-arboreto
             python-attrs
             python-boltons
             python-cloudpickle
             python-ctxcore
             python-cytoolz
             python-dask
             python-distributed
             python-frozendict
             python-fsspec
             python-interlap
             python-llvmlite
             python-loompy
             python-multiprocessing-on-dill
             python-networkx
             python-numba
             python-numexpr
             python-numpy
             python-pandas
             python-pyarrow             ;XXX for dask
             python-pyyaml
             python-requests
             python-scikit-learn
             python-scipy
             python-setuptools
             python-tqdm
             python-umap-learn))
      (native-inputs
       (list python-pytest python-wheel))
      (home-page "https://scenic.aertslab.org/")
      (synopsis "Single-Cell regulatory network inference and clustering")
      (description
       "pySCENIC is a Python implementation of the SCENIC pipeline (Single-Cell
rEgulatory Network Inference and Clustering) which enables biologists to infer
transcription factors, gene regulatory networks and cell types from
single-cell RNA-seq data.")
      (license license:gpl3+))))

(define-public python-ikarus
  (package
    (name "python-ikarus")
    (version "0.0.2")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "ikarus" version))
       (sha256
        (base32 "086czpvj4yafz4vrq5rx2gy0bj2l8nzwnkk0gw8qvy4w133xjysy"))))
    (build-system pyproject-build-system)
    (arguments
     `(#:tests? #f
       #:phases (modify-phases %standard-phases
                  ;; See https://github.com/BIMSBbioinfo/ikarus/issues/12
                  (add-after 'unpack 'fix-issue-12
                    (lambda _
                      (substitute* "ikarus/classifier.py"
                        (("pyscenic.genesig")
                         "ctxcore.genesig"))))
                  ;; Numba needs a writable dir to cache functions.
                  (add-before 'check 'set-numba-cache-dir
                    (lambda _
                      (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs (list python-numpy
                             python-pandas
                             python-scipy
                             python-scanpy
                             python-anndata
                             python-ctxcore ;because of issue 12
                             pyscenic))
    (native-inputs (list python-setuptools python-wheel))
    (home-page "https://github.com/BIMSBbioinfo/ikarus")
    (synopsis "Machine learning classifier of tumor cells")
    (description
     "ikarus is a stepwise machine learning pipeline that tries to cope with a task
of distinguishing tumor cells from normal cells.  Leveraging multiple
annotated single cell datasets it can be used to define a gene set specific to
tumor cells.  First, the latter gene set is used to rank cells and then to
train a logistic classifier for the robust classification of tumor and normal
cells.  Finally, sensitivity is increased by propagating the cell labels based
on a custom cell-cell network.  ikarus is tested on multiple single cell
datasets to ascertain that it achieves high sensitivity and specificity in
multiple experimental contexts.")
    (license license:expat)))

(define-public tbsp
  (let ((commit "dc30c03868233c5504299c9cb0d7b2064ba9cb41")
        (revision "2"))
    (package
      (name "tbsp")
      (version (git-version "1.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/phoenixding/tbsp")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1im0bik2hxkcb7jzkcxp5nqb30hd8lfraxml6i5ik52j6z3qqln1"))))
      (build-system pyproject-build-system)
      (arguments
       '(#:tests? #f ;no tests included
         #:phases (modify-phases %standard-phases
                    (add-after 'unpack 'relax-requirements
                      (lambda _
                        (substitute* "setup.py"
                          ((", <3.0")
                           ""))))))) ;matplotlib
      (inputs (list python-matplotlib
                    python-networkx
                    python-numpy
                    python-pybigwig
                    python-biopython-1.73
                    python-scikit-learn
                    python-scipy))
      (native-inputs (list python-setuptools python-wheel))
      (home-page "https://github.com/phoenixding/tbsp/")
      (synopsis "SNP-based trajectory inference")
      (description
       "Several studies focus on the inference of developmental and response
trajectories from single cell RNA-Seq (scRNA-Seq) data.  A number of
computational methods, often referred to as pseudo-time ordering, have been
developed for this task.  CRISPR has also been used to reconstruct lineage
trees by inserting random mutations.  The tbsp package implements an
alternative method to detect significant, cell type specific sequence
mutations from scRNA-Seq data.")
      (license license:expat))))

(define-public tabixpp
  (package
   (name "tabixpp")
   (version "1.1.2")
   (source (origin
             (method git-fetch)
             (uri (git-reference
                   (url "https://github.com/ekg/tabixpp")
                   (commit (string-append "v" version))))
             (file-name (git-file-name name version))
             (sha256
              (base32 "00aqs147yn8zcvxims5njwxqsbnlbjv7lnmiwqy80bfdcbhljkqf"))
             (modules '((guix build utils)))
             (snippet
              '(delete-file-recursively "htslib"))))
   (build-system gnu-build-system)
   (inputs
    (list bzip2 curl htslib xz zlib))
   (arguments
    (list #:make-flags #~(list (string-append "CC=" #$(cc-for-target))
                               (string-append "CXX=" #$(cxx-for-target))
                               "HTS_HEADERS="
                               (string-append "HTS_LIB="
                                              (search-input-file %build-inputs
                                                                 "/lib/libhts.a"))
                               "INCLUDES=")
          #:tests? #f ; There are no tests to run.
          #:phases
          #~(modify-phases %standard-phases
              (delete 'configure) ; There is no configure phase.
              ;; Build shared and static libraries.
              (add-after 'build 'build-libraries
                (lambda* (#:key inputs #:allow-other-keys)
                  (invoke #$(cxx-for-target)
                          "-shared" "-o" "libtabixpp.so" "tabix.o" "-lhts")
                  (invoke #$(ar-for-target) "rcs" "libtabixpp.a" "tabix.o")))
              (replace 'install
                (lambda _
                  (let ((lib (string-append #$output "/lib"))
                        (bin (string-append #$output "/bin")))
                    (install-file "tabix++" bin)
                    (install-file "libtabixpp.so" lib)
                    (install-file "libtabixpp.a" lib)
                    (install-file "tabix.hpp" (string-append #$output "/include"))
                    (mkdir-p (string-append lib "/pkgconfig"))
                    (with-output-to-file (string-append lib "/pkgconfig/tabixpp.pc")
                      (lambda _
                        (format #t "\
prefix=~a~@
exec_prefix=${prefix}~@
libdir=${exec_prefix}/lib~@
includedir=${prefix}/include~@
~@
~@
Name: libtabixpp~@
Version: ~a~@
Description: C++ wrapper around tabix project~@
Libs: -L${libdir} -ltabixpp~@
Cflags: -I${includedir}~%"
                                #$output #$version)))))))))
   (home-page "https://github.com/ekg/tabixpp")
   (synopsis "C++ wrapper around tabix project")
   (description "This is a C++ wrapper around the Tabix project which abstracts
some of the details of opening and jumping in tabix-indexed files.")
   (license license:expat)))

(define-public smithwaterman
  (let ((commit "2610e259611ae4cde8f03c72499d28f03f6d38a7"))
    (package
      (name "smithwaterman")
      (version (git-version "0.0.0" "2" commit))
      (source (origin
        (method git-fetch)
        (uri (git-reference
              (url "https://github.com/ekg/smithwaterman/")
              (commit commit)))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0i9d8zrxpiracw3mxzd9siybpy62p06rqz9mc2w93arajgbk45bs"))))
      (build-system gnu-build-system)
      (arguments
       (list
         #:tests? #f ; There are no tests to run.
         #:make-flags
         #~(list (string-append "CXX=" #$(cxx-for-target))
                 "libsw.a" "all")
         #:phases
         #~(modify-phases %standard-phases
             (delete 'configure) ; There is no configure phase.
             (add-after 'unpack 'patch-source
               (lambda _
                 (substitute* "Makefile"
                   (("-c ") "-c -fPIC "))
                 #$@(if (%current-target-system)
                     #~((substitute* "Makefile"
                          (("\tld")
                           (string-append "\t" #$(%current-target-system) "-ld"))
                          (("\tar")
                           (string-append "\t" #$(%current-target-system) "-ar"))))
                     '())))
             (add-after 'build 'build-dynamic
               (lambda _
                 (invoke #$(cxx-for-target)
                         "-shared" "-o" "libsmithwaterman.so"
                         "smithwaterman.o" "SmithWatermanGotoh.o"
                         "disorder.o" "BandedSmithWaterman.o"
                         "LeftAlign.o" "Repeats.o" "IndelAllele.o")))
             (replace 'install
               (lambda* (#:key outputs #:allow-other-keys)
                 (let* ((out (assoc-ref outputs "out"))
                        (bin (string-append out "/bin"))
                        (lib (string-append out "/lib")))
                   (install-file "smithwaterman" bin)
                   (for-each
                     (lambda (file)
                       (install-file file (string-append out "/include/smithwaterman")))
                     (find-files "." "\\.h$"))
                   (install-file "libsmithwaterman.so" lib)
                   (install-file "libsw.a" lib)
                   (mkdir-p (string-append lib "/pkgconfig"))
                   (with-output-to-file (string-append lib "/pkgconfig/smithwaterman.pc")
                     (lambda _
                       (format #t "prefix=~a~@
                               exec_prefix=${prefix}~@
                               libdir=${exec_prefix}/lib~@
                               includedir=${prefix}/include/smithwaterman~@
                               ~@
                               ~@
                               Name: smithwaterman~@
                               Version: ~a~@
                               Description: smith-waterman-gotoh alignment algorithm~@
                               Libs: -L${libdir} -lsmithwaterman~@
                               Cflags: -I${includedir}~%"
                               out #$version)))))))))
      (home-page "https://github.com/ekg/smithwaterman")
      (synopsis "Implementation of the Smith-Waterman algorithm")
      (description "Implementation of the Smith-Waterman algorithm.")
      ;; The licensing terms are unclear: https://github.com/ekg/smithwaterman/issues/9.
      (license (list license:gpl2 license:expat)))))

(define-public sylamer
  (package
    (name "sylamer")
    (version "18-131")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/micans/sylamer/")
                    (commit "aa75c3584797c0c15f860addb645f7bc1dd7627d")))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ddiwlrdghhb4574rvfw0brjp9gs5l6nfsy82h0m4mvz1dr3gkj5"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ; no test target
      #:make-flags
      #~(list (string-append "GSLPREFIX=" #$(this-package-input "gsl")))
      #:phases
      '(modify-phases %standard-phases
         (replace 'configure
           (lambda* (#:key outputs #:allow-other-keys)
             (substitute* "Makefile"
               (("cp sylamer \\$\\(HOME\\)/local/bin")
                (string-append "install -D -t " (assoc-ref outputs "out")
                               "/bin sylamer")))
             (install-file "Makefile" "src")
             (chdir "src"))))))
    (inputs (list gsl zlib))
    (home-page "https://www.ebi.ac.uk/research/enright/software/sylamer")
    (synopsis "Asses microRNA binding and siRNA off-target effects")
    (description "Sylamer is a system for finding significantly over or
under-represented words in sequences according to a sorted gene list.
Typically it is used to find significant enrichment or depletion of microRNA
or siRNA seed sequences from microarray expression data.  Sylamer is extremely
fast and can be applied to genome-wide datasets with ease.  Results are
plotted in terms of a significance landscape plot.  These plots show
significance profiles for each word studied across the sorted genelist.")
    (license license:gpl3+)))

(define-public multichoose
  (package
    (name "multichoose")
    (version "1.0.3")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/ekg/multichoose/")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ci5fqvmpamwgxvmyd79ygj6n3bnbl3vc7b6h1sxz58186sm3pfs"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;Tests require node.
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)           ;There is no configure phase.
          (replace 'install
            (lambda _
              (let ((bin (string-append #$output "/bin"))
                    (include (string-append #$output "/include")))
                ;; TODO: There are Python modules for these programs too.
                (install-file "multichoose" bin)
                (install-file "multipermute" bin)
                (install-file "multichoose.h" include)
                (install-file "multipermute.h" include)))))))
    (home-page "https://github.com/ekg/multichoose")
    (synopsis "Efficient loopless multiset combination generation algorithm")
    (description "This library implements an efficient loopless multiset
combination generation algorithm which is (approximately) described in
\"Loopless algorithms for generating permutations, combinations, and other
combinatorial configurations.\", G. Ehrlich - Journal of the ACM (JACM),
1973. (Algorithm 7.)")
    (license license:expat)))

(define-public fsom
  (let ((commit "a6ef318fbd347c53189384aef7f670c0e6ce89a3"))
    (package
      (name "fsom")
      (version (git-version "0.0.0" "1" commit))
      (source (origin
        (method git-fetch)
        (uri (git-reference
              (url "https://github.com/ekg/fsom/")
              (commit commit)))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0gw1lpvr812pywg9y546x0h1hhj261xwls41r6kqhddjlrcjc0pi"))))
      (build-system gnu-build-system)
      (arguments
       `(#:tests? #f ; There are no tests to run.
         #:phases
         (modify-phases %standard-phases
           (delete 'configure) ; There is no configure phase.
           (replace 'install
             (lambda* (#:key outputs #:allow-other-keys)
               (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
                 (install-file "fsom" bin)))))))
      (native-inputs
       (list gcc-6))
      (home-page "https://github.com/ekg/fsom")
      (synopsis "Manage SOM (Self-Organizing Maps) neural networks")
      (description "A tiny C library for managing SOM (Self-Organizing Maps)
neural networks.")
      (license license:gpl3))))

(define-public fastahack
  (package
    (name "fastahack")
    (version "1.0.0")
    (source (origin
      (method git-fetch)
      (uri (git-reference
            (url "https://github.com/ekg/fastahack/")
            (commit (string-append "v" version))))
      (file-name (git-file-name name version))
      (sha256
       (base32 "0rp1blskhzxf7vbh253ibpxbgl9wwgyzf1wbkxndi08d3j4vcss9"))))
    (build-system gnu-build-system)
    (arguments
     (list #:make-flags #~(list (string-append "CXX=" #$(cxx-for-target)))
           ;; Unclear how to run tests: https://github.com/ekg/fastahack/issues/15
           #:tests? #f
           #:phases
           #~(modify-phases %standard-phases
               (delete 'configure) ; There is no configure phase.
               (add-after 'unpack 'patch-source
                 (lambda _
                   (substitute* "Makefile"
                     (("-c ") "-c -fPIC "))))
               (add-after 'build 'build-dynamic
                 (lambda _
                   (invoke #$(cxx-for-target)
                           "-shared" "-o" "libfastahack.so"
                           "Fasta.o" "FastaHack.o" "split.o" "disorder.o")))
               (replace 'install
                 (lambda* (#:key outputs #:allow-other-keys)
                   (let* ((out (assoc-ref outputs "out"))
                          (lib (string-append out "/lib"))
                          (bin (string-append out "/bin")))
                     (mkdir-p (string-append out "/include/fastahack"))
                     (for-each
                      (lambda (file)
                        (install-file file (string-append out "/include/fastahack")))
                      (find-files "." "\\.h$"))
                     (install-file "fastahack" bin)
                     (install-file "libfastahack.so" lib)
                     (mkdir-p (string-append lib "/pkgconfig"))
                     (with-output-to-file (string-append lib "/pkgconfig/fastahack.pc")
                       (lambda _
                         (format #t "prefix=~a~@
                           exec_prefix=${prefix}~@
                           libdir=${exec_prefix}/lib~@
                           includedir=${prefix}/include/fastahack~@
                           ~@
                           ~@
                           Name: fastahack~@
                           Version: ~a~@
                           Description: Indexing and sequence extraction from FASTA files~@
                           Libs: -L${libdir} -lfastahack~@
                           Cflags: -I${includedir}~%"
                                 out #$version)))))))))
    (home-page "https://github.com/ekg/fastahack")
    (synopsis "Indexing and sequence extraction from FASTA files")
    (description "Fastahack is a small application for indexing and
extracting sequences and subsequences from FASTA files.  The included library
provides a FASTA reader and indexer that can be embedded into applications
which would benefit from directly reading subsequences from FASTA files.  The
library automatically handles index file generation and use.")
    (license (list license:expat license:gpl2))))

(define-public wfa2-lib
  (package
    (name "wfa2-lib")
    (version "2.3.3")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/smarco/WFA2-lib")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "110s1s79z8ndjs4qdgmrj708iiaqyagia3gi2fxak101lg263diw"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          ;; This pkg-config file is provided by other distributions.
          (add-after 'install 'install-pkg-config-file
            (lambda _
              (let ((pkgconfig (string-append #$output "/lib/pkgconfig")))
                (mkdir-p pkgconfig)
                (with-output-to-file (string-append pkgconfig "/libwfa2.pc")
                  (lambda _
                    (format #t "\
prefix=~a~@
exec_prefix=${prefix}~@
libdir=${exec_prefix}/lib~@
includedir=${prefix}/include/wfa2lib~@
~@
Name: libwfa2~@
Version: ~a~@
Description: Wavefront alignment algorithm library~@
Libs: -L${libdir} -lwfa2 -lwfa2cpp~@
Cflags: -I${includedir}~%"
                            #$output #$version)))))))))
    (native-inputs
     (list pkg-config))
    (home-page "https://github.com/smarco/WFA2-lib")
    (synopsis "Wavefront alignment algorithm library")
    (description "The wavefront alignment (WFA) algorithm is an exact
gap-affine algorithm that takes advantage of homologous regions between the
sequences to accelerate the alignment process.")
    (properties `((tunable? . #t)))
    (license license:expat)))

(define-public vcflib
  (package
    (name "vcflib")
    (version "1.0.12")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/vcflib/vcflib")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1llqnbgx4ybb77cp5k3cs0y7n4p8ma6b3sj10iarpg5r6a5gps5a"))
       (modules '((guix build utils)))
       (snippet
        #~(begin
            (substitute* "CMakeLists.txt"
              ((".*fastahack.*") "")
              ((".*smithwaterman.*") "")
              (("(pkg_check_modules\\(TABIXPP)" text)
               (string-append
                "pkg_check_modules(WFA2 REQUIRED libwfa2)\n"
                "pkg_check_modules(FASTAHACK REQUIRED fastahack)\n"
                "pkg_check_modules(SMITHWATERMAN REQUIRED smithwaterman)\n"
                text))
              (("\\$\\{TABIXPP_LIBRARIES\\}" text)
               (string-append "${FASTAHACK_LIBRARIES} "
                              "${WFA2_LIBRARIES} "
                              "${SMITHWATERMAN_LIBRARIES} "
                              text))
              (("install\\(TARGETS \\$\\{WFALIB\\}.*") ""))
            (substitute* (find-files "." "\\.(h|c)(pp)?$")
              (("SmithWatermanGotoh.h") "smithwaterman/SmithWatermanGotoh.h")
              (("\"convert.h\"") "<smithwaterman/convert.h>")
              (("\"disorder.h\"") "<smithwaterman/disorder.h>")
              (("\"wavefront/wfa.hpp\"") "<wavefront/wfa.hpp>")
              (("Fasta.h") "fastahack/Fasta.h"))))))
    (build-system cmake-build-system)
    (inputs
     (list bzip2
           curl
           htslib
           fastahack
           perl
           python
           pybind11
           smithwaterman
           tabixpp
           xz
           zlib))
    (propagated-inputs
     (list wfa2-lib))
    (native-inputs
     `(("pkg-config" ,pkg-config)
       ;; Submodules.
       ;; This package builds against the .o files so we need to extract the source.
       ("filevercmp-src" ,(package-source filevercmp))
       ("fsom-src" ,(package-source fsom))
       ("intervaltree-src" ,(package-source intervaltree))))
    (arguments
     (list
      #:configure-flags
      #~(list "-DZIG=OFF"
              "-DTABIXPP_LOCAL=OFF"
              "-DTABIX_FOUND=ON"
              "-DWFA_GITMODULE=OFF"
              (string-append "-DPKG_CONFIG_EXECUTABLE="
                             (search-input-file
                              %build-inputs (string-append
                                             "/bin/" #$(pkg-config-for-target)))))
      #:tests? #f                       ; no tests
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'find-wfa2lib-headers
            (lambda _
              (setenv "CPLUS_INCLUDE_PATH"
                      (string-append
                       #$(this-package-input "wfa2-lib")
                       "/include/wfa2lib:"
                       (or (getenv "CPLUS_INCLUDE_PATH") "")))))
          (add-after 'unpack 'build-shared-library
            (lambda _
              (substitute* "CMakeLists.txt"
                (("vcflib STATIC") "vcflib SHARED"))))
          (add-after 'unpack 'unpack-submodule-sources
            (lambda* (#:key inputs native-inputs #:allow-other-keys)
              (let ((unpack (lambda (source target)
                              (mkdir-p target)
                              (with-directory-excursion target
                                (let ((source (or (assoc-ref inputs source)
                                                  (assoc-ref native-inputs source))))
                                  (if (file-is-directory? source)
                                      (copy-recursively source ".")
                                      (invoke "tar" "xvf"
                                              source
                                              "--strip-components=1")))))))
                (and
                 (unpack "filevercmp-src" "contrib/filevercmp")
                 (unpack "fsom-src" "contrib/fsom")
                 (unpack "intervaltree-src" "contrib/intervaltree"))

                ;; This is needed for downstream packages to allow building
                ;; with GCC 11+.
                (substitute* "contrib/intervaltree/IntervalTree.h"
                  (("#include <vector>" m)
                   (string-append m "
#include <limits> /* std::numeric_limits */"))))))
          ;; This pkg-config file is provided by other distributions.
          (add-after 'install 'install-pkg-config-file
            (lambda _
              (let ((pkgconfig (string-append #$output "/lib/pkgconfig")))
                (mkdir-p pkgconfig)
                (with-output-to-file (string-append pkgconfig "/vcflib.pc")
                  (lambda _
                    (format #t "\
prefix=~a~@
exec_prefix=${prefix}~@
libdir=${exec_prefix}/lib~@
includedir=${prefix}/include~@
~@
Name: vcflib~@
Version: ~a~@
Requires: libwfa2, smithwaterman, fastahack, tabixpp~@
Description: C++ library for parsing and manipulating VCF files~@
Libs: -L${libdir} -lvcflib~@
Cflags: -I${includedir}~%"
                            #$output #$version)))))))))
    (home-page "https://github.com/vcflib/vcflib/")
    (synopsis "Library for parsing and manipulating VCF files")
    (description "Vcflib provides methods to manipulate and interpret
sequence variation as it can be described by VCF.  It is both an API for parsing
and operating on records of genomic variation as it can be described by the VCF
format, and a collection of command-line utilities for executing complex
manipulations on VCF files.")
    (license license:expat)))

(define-public freebayes
  (package
    (name "freebayes")
    (version "1.3.9")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/freebayes/freebayes")
                    (commit (string-append "v" version))
                    (recursive? #true)))
              (file-name (git-file-name name version))
              (sha256
               (base32 "18i8xv5cx4n9924f4gqjk2sg3vi437nh7ssy99ng8ajschgaw1m2"))))
    (build-system meson-build-system)
    (inputs
     (list bzip2 fastahack htslib smithwaterman tabixpp vcflib xz))
    (native-inputs
     (list bc
           grep                         ; Built with perl support.
           parallel
           perl
           perl6-tap-harness
           pkg-config
           python
           samtools
           simde))
    (arguments
     (list
      #:phases
      `(modify-phases %standard-phases
         (add-after 'unpack 'patch-source
           (lambda* (#:key inputs #:allow-other-keys)
             (setenv "CPLUS_INCLUDE_PATH"
                     (string-append (getcwd) "/contrib/multichoose:"
                                    (getenv "CPLUS_INCLUDE_PATH")))
             (substitute* '("src/BedReader.cpp"
                            "src/BedReader.h")
               (("intervaltree/IntervalTree.h") "vcflib/IntervalTree.h")
               (("<IntervalTree.h>") "<vcflib/IntervalTree.h>"))
             (substitute* '("src/AlleleParser.h"
                            "src/ResultData.h")
               (("Variant.h") "vcflib/Variant.h"))
             (substitute* (find-files "." "\\.(h|c)(pp)?$")
               (("\"multichoose.h\"") "<vcflib/multichoose.h>")
               (("\"convert.h\"") "<smithwaterman/convert.h>")
               (("\"split.h\"") "<fastahack/split.h>")
               (("\"join.h\"") "<vcflib/join.h>"))
             (substitute* "meson.build"
               ;; Our pkg-config file is vcflib.pc
               (("libvcflib") "vcflib")
               (("vcflib_inc,") ""))))
         ;; The slow tests take longer than the specified timeout.
         ,@(if (any (cute string=? <> (%current-system))
                    '("armhf-linux" "aarch64-linux"))
               '((replace 'check
                   (lambda* (#:key tests? #:allow-other-keys)
                     (when tests?
                       (invoke "meson" "test" "--timeout-multiplier" "5")))))
               '()))))
    (home-page "https://github.com/freebayes/freebayes")
    (synopsis "Haplotype-based variant detector")
    (description "FreeBayes is a Bayesian genetic variant detector designed to
find small polymorphisms, specifically SNPs (single-nucleotide polymorphisms),
indels (insertions and deletions), MNPs (multi-nucleotide polymorphisms), and
complex events (composite insertion and substitution events) smaller than the
length of a short-read sequencing alignment.")
    (license license:expat)))

(define-public samblaster
  (package
    (name "samblaster")
    (version "0.1.26")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/GregoryFaust/samblaster")
                    (commit (string-append "v." version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0g24fq5hplnfgqkh3xqpg3lgx3wmxwnh9c7m6yw7pbi40lmgl1jv"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #f                       ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)           ;There is no configure phase.
          (replace 'install
            (lambda _
              (install-file "samblaster" (string-append #$output "/bin")))))))
    (home-page "https://github.com/GregoryFaust/samblaster")
    (synopsis "Mark duplicates in paired-end SAM files")
    (description "Samblaster is a fast and flexible program for marking
duplicates in read-id grouped paired-end SAM files.  It can also optionally
output discordant read pairs and/or split read mappings to separate SAM files,
and/or unmapped/clipped reads to a separate FASTQ file.  When marking
duplicates, samblaster will require approximately 20MB of memory per 1M read
pairs.")
    (license license:expat)))

(define-public r-hdf5dataframe
  (let ((commit "1e30e6b82b5599d5d98fbcb8bf7312dd1711ca3c")
        (revision "2"))
    (package
      (name "r-hdf5dataframe")
      (version (git-version "0.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/HDF5DataFrame")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0y6hp31gy17v87ll4rij6pwy6b7k7gnw8iwnvzsaa13ga73g69ak"))))
      (properties `((upstream-name . "HDF5DataFrame")))
      (build-system r-build-system)
      (propagated-inputs (list r-biocgenerics r-delayedarray r-hdf5array
                               r-rhdf5 r-s4vectors))
      (native-inputs (list r-knitr r-testthat))
      (home-page "https://github.com/BIMSBbioinfo/HDF5DataFrame")
      (synopsis "Bioconductor-friendly bindings for Parquet")
      (description
       "This package implements bindings for h5 files that are compatible with
Bioconductor S4 data structures, namely the @code{DataFrame} and
@code{DelayedArray}.  This allows HDF5-backed data to be easily used as data
frames with arbitrary sets of columns.")
      (license license:expat))))

(define-public r-imagearray
  (let ((commit "78b4b18d4326aca8aecb2cf01b019c5809078310")
        (revision "1"))
    (package
      (name "r-imagearray")
      (version (git-version "1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/ImageArray")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0lqcvp0xrvi5c5v31cjvcbjcyrd2wssc948fvndarvg0vhvgqw71"))))
      (properties `((upstream-name . "ImageArray")))
      (build-system r-build-system)
      (propagated-inputs (list r-delayedarray r-hdf5array r-magick r-s4arrays
                               r-zarrarray))
      (home-page "https://github.com/BIMSBbioinfo/ImageArray")
      (synopsis "DelayedArray based image operations")
      (description "@code{DelayedArray} based image operations.")
      (license license:expat))))

(define-public r-velocyto
  (let ((commit "d7790346cb99f49ab9c2b23ba70dcf9d2c9fc350")
        (revision "1"))
    (package
      (name "r-velocyto")
      (version (git-version "0.6" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/velocyto-team/velocyto.R")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "16wqf70j7rd7pay2q513iyz12i8n9vrpg1bisah4lddbcpx5dz1n"))))
      (build-system r-build-system)
      (inputs
       (list boost-1.83))
      (propagated-inputs
       (list r-hdf5r
             r-mass
             r-mgcv
             r-pcamethods
             r-rcpp
             r-rcpparmadillo
             ;; Suggested packages
             r-rtsne
             r-cluster
             r-abind
             r-biocgenerics
             r-genomicalignments
             r-rsamtools
             r-edger
             r-igraph))
      (home-page "https://velocyto.org")
      (synopsis "RNA velocity estimation in R")
      (description
       "This package provides basic routines for estimation of gene-specific
transcriptional derivatives and visualization of the resulting velocity
patterns.")
      (license license:gpl3))))

(define-public r-voltron
  (let ((commit "1c0810b6603010af9afc7bb75eaebbe6de5ace30")
        (revision "1"))
    (package
      (name "r-voltron")
      (version (git-version "0.2.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/VoltRon")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0w8lbk74n0pxz4rscl5vnfaw2xpl107y55cc5xff4km37dw901hv"))))
      (properties `((upstream-name . "VoltRon")))
      (build-system r-build-system)
      (arguments
       (list
        #:phases
        #~(modify-phases %standard-phases
            ;; By default VoltRon will use Basilisk, which uses Conda to set
            ;; up a Python environment.  We override the fallback default
            ;; here.  Users can still override the location of the Python
            ;; interpreter with the option "voltron.python.path".
            (add-after 'unpack 'do-not-use-conda
              (lambda* (#:key inputs #:allow-other-keys)
                (substitute* "R/conversion.R"
                  (("return\\(NULL\\)")
                   (string-append
                    "source(paste0(system.file(package=\"VoltRon\"), \"/guix-refs.R\"));\n"
                    "return(guix_python);")))))
            ;; We do this outside of the source code to ensure that
            ;; references are accessible to Guix.
            (add-after 'install 'record-python-reference
              (lambda* (#:key inputs #:allow-other-keys)
                (mkdir-p (string-append #$output "/site-library/VoltRon/"))
                (call-with-output-file (string-append #$output "/site-library/VoltRon/guix-refs.R")
                  (lambda (port)
                    (format port "\
Sys.setenv(GUIX_PYTHONPATH=\"~a\");
guix_python <- \"~a\";"
                            (getenv "GUIX_PYTHONPATH")
                            (search-input-file inputs "/bin/python3")))))))))
      (inputs
       (list opencv
             ;; These Python inputs are used via reticulate.
             python
             python-numpy
             python-pandas
             python-anndata
             python-h5py
             python-natsort
             python-numcodecs
             python-packaging
             python-scipy
             python-tifffile
             python-zarr
             which       ;tests/testthat/test_conversion.R
             zlib))
      (propagated-inputs (list r-biocsingular
                               r-data-table
                               r-dplyr
                               r-ebimage
                               r-ggplot2
                               r-ggpubr
                               r-ggrepel
                               r-ids
                               r-igraph
                               r-irlba
                               r-magick
                               r-matrix
                               r-pizzarr
                               r-rann
                               r-rcdt
                               r-rcpp
                               r-rcppannoy
                               r-rcpparmadillo
                               r-reshape2
                               r-reticulate
                               r-rhdf5
                               r-rjson
                               r-rlang
                               r-s4arrays
                               r-shiny
                               r-shinyjs
                               r-sp
                               r-stringr
                               r-uwot

                               ;; Suggested packages
                               r-anndata
                               r-anndatar
                               r-arrow
                               r-bpcells
                               r-circlize
                               r-codetools
                               r-complexheatmap
                               r-delayedarray
                               r-deseq2
                               r-geojsonr
                               r-ggforce
                               r-ggnewscale
                               r-giotto
                               r-glmgampoi
                               r-hdf5array
                               r-hdf5dataframe
                               r-imagearray
                               r-music
                               r-rhdf5
                               r-rstudioapi
                               r-s4vectors
                               r-seurat
                               r-seuratobject
                               r-singlecellexperiment
                               r-spacexr
                               r-spatialexperiment
                               r-summarizedexperiment
                               r-viridislite
                               r-vitesscer
                               r-xml
                               r-zarrdataframe))
      (native-inputs (list pkg-config r-testthat))
      (home-page "https://github.com/BIMSBbioinfo/VoltRon")
      (synopsis "VoltRon for spatial data integration and analysis")
      (description
       "@code{VoltRon} is a novel spatial omic analysis toolbox for
multi-omics integration using spatial image registration.  @code{VoltRon} is
capable of analyzing multiple types and modalities of spatially-aware
datasets.  @code{VoltRon} visualizes and analyzes regions of interests (ROIs),
spots, cells and even molecules.")
      (license license:expat))))

(define-public r-zarrdataframe
  (let ((commit "fa89bd272ebc33a90edd4016c6f87f966102a4f5")
        (revision "1"))
    (package
      (name "r-zarrdataframe")
      (version (git-version "0.0.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/ZarrDataFrame")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "121vyqwyg773l77xvx0rvrf0dyn26g3b01v42pz0a8lj4qmk86vn"))))
      (properties `((upstream-name . "ZarrDataFrame")))
      (build-system r-build-system)
      (propagated-inputs (list r-biocgenerics r-delayedarray r-pizzarr
                               r-s4vectors r-zarrarray))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/BIMSBbioinfo/ZarrDataFrame")
      (synopsis "Bioconductor-friendly Bindings for Zarr")
      (description
       "This package implements bindings for zarr store that are compatible
with Bioconductor S4 data structures, namely the @code{DataFrame} and
@code{DelayedArray}.  This allows Zarr-backed data to be easily used as data
frames with arbitrary sets of columns.")
      (license license:expat))))

(define-public r-zarrarray
  (let ((commit "508d87193b20feba8cd24fa96b33e1bc9cc49958")
        (revision "1"))
    (package
      (name "r-zarrarray")
      (version (git-version "1.0" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/BIMSBbioinfo/ZarrArray")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0fz7zj9ixmz6yh709cd55abc5378b549cx454cgk274nhp24xdbr"))))
      (properties `((upstream-name . "ZarrArray")))
      (build-system r-build-system)
      (propagated-inputs (list r-biocgenerics
                               r-delayedarray
                               r-iranges
                               r-matrix
                               r-pizzarr
                               r-s4arrays
                               r-s4vectors
                               r-sparsearray))
      (home-page "https://github.com/BIMSBbioinfo/ZarrArray")
      (synopsis "Zarr backend for DelayedArray objects")
      (description "Zarr backend for @code{DelayedArray} objects.")
      (license license:expat))))

(define-public methyldackel
  (package
    (name "methyldackel")
    (version "0.6.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/dpryan79/MethylDackel")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06kj76pyhzxfcjcpm840a3km3fa9994kfq4asglnb228pwak326z"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:test-target "test"
      #:make-flags
      #~(list "CC=gcc"
              "CFLAGS=-fcommon"
              "LIBBIGWIG=-lBigWig"
              (string-append "prefix=" #$output "/bin/"))
      #:phases
      '(modify-phases %standard-phases
         (replace 'configure
           (lambda _
             (substitute* "Makefile"
               (("install MethylDackel \\$\\(prefix\\)" match)
                (string-append "install -d $(prefix); " match))))))))
    (inputs
     (list curl htslib libbigwig zlib))
    ;; Needed for tests
    (native-inputs
     (list python-wrapper))
    (home-page "https://github.com/dpryan79/MethylDackel")
    (synopsis "Universal methylation extractor for BS-seq experiments")
    (description
     "MethylDackel will process a coordinate-sorted and indexed BAM or CRAM
file containing some form of BS-seq alignments and extract per-base
methylation metrics from them.  MethylDackel requires an indexed fasta file
containing the reference genome as well.")
    ;; See https://github.com/dpryan79/MethylDackel/issues/85
    (license license:expat)))

;; This package bundles PCRE 8.02 and cannot be built with the current
;; version.
(define-public phast
  (package
    (name "phast")
    (version "1.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/CshlSiepelLab/phast")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10lpbllvny923jjbbyrpxahhd1m5h7sbj9gx7rd123rg10mlidki"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:make-flags
      #~(list "CC=gcc" (string-append "DESTDIR=" #$output))
      #:phases
      #~(modify-phases %standard-phases
          (replace 'configure
            (lambda* (#:key inputs outputs #:allow-other-keys)
              ;; Fix syntax
              (substitute* "test/Makefile"
                (("        ") "	"))
              (substitute* "Makefile"
                (("CLAPACKPATH=/usr/lib")
                 (string-append "CLAPACKPATH="
                                #$(this-package-input "clapack") "/lib")))
              ;; Renaming the libraries is not necessary with our version of
              ;; CLAPACK.
              (substitute* "src/lib/Makefile"
                (("ifdef CLAPACKPATH") "ifdef UNNECESSARY"))
              (substitute* "src/make-include.mk"
                (("-lblaswr") "-lblas")
                (("-ltmg") "-ltmglib")
                (("liblapack.a") "liblapack.so")
                (("libblas.a") "libblas.so")
                (("libf2c.a") "libf2c.so"))
              (substitute* "src/Makefile"
                (("/opt") "/share")
                (("/usr/") "/"))))
          (replace 'check
            (lambda _
              (setenv "PATH"
                      (string-append (getcwd) "/bin:" (getenv "PATH")))
              ;; Disable broken test
              (substitute* "test/Makefile"
                ((".*if.*hmrc_summary" m) (string-append "#" m)))
              ;; Only run the msa_view tests because the others fail for
              ;; unknown reasons.
              (invoke "make" "-C" "test" "msa_view"))))))
    (inputs
     (list clapack))
    (native-inputs
     (list perl))
    (home-page "http://compgen.cshl.edu/phast/")
    (synopsis "Phylogenetic analysis with space/time models")
    (description
     "Phylogenetic Analysis with Space/Time models (PHAST) is a collection of
command-line programs and supporting libraries for comparative and
evolutionary genomics.  Best known as the search engine behind the
Conservation tracks in the University of California, Santa Cruz (UCSC) Genome
Browser, PHAST also includes several tools for phylogenetic modeling,
functional element identification, as well as utilities for manipulating
alignments, trees and genomic annotations.")
    (license license:bsd-3)))

(define-public python-gffutils
  (package
    (name "python-gffutils")
    (version "0.13")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/daler/gffutils")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "148i7bk5bawrz19jp3nl0z859wdlldgrw8f0aw9wsprj8s3d713a"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      ;; Failed: 'yield' keyword is allowed in fixtures, but not in tests
      ;; (test_region)
      #:test-flags #~(list "--ignore=gffutils/test/test_1.py")
      #:phases
      #~(modify-phases %standard-phases
          (add-before 'build 'set-HOME
            (lambda _
              ;; FileNotFoundError: [Errno 2] No such file or directory:
              ;; '/homeless-shelter/.gffutils.test'
              (setenv "HOME" "/tmp"))))))
    (native-inputs
     (list python-pytest
           python-setuptools
           python-wheel))
    (propagated-inputs
     (list python-argcomplete
           python-argh
           python-biopython
           python-pybedtools
           python-pyfaidx
           python-simplejson))
    (home-page "https://github.com/daler/gffutils")
    (synopsis "Tool for manipulation of GFF and GTF files")
    (description
     "python-gffutils is a Python package for working with and manipulating
the GFF and GTF format files typically used for genomic annotations.  The
files are loaded into a SQLite database, allowing much more complex
manipulation of hierarchical features (e.g., genes, transcripts, and exons)
than is possible with plain-text methods alone.")
    (license license:expat)))

(define-public indelfixer
  (package
    (name "indelfixer")
    (version "1.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/cbg-ethz/InDelFixer/")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10ak05x8i1bx2p7rriv2rglqg1wr7c8wrhjrqlq1wm7ka99w8i79"))))
    (build-system ant-build-system)
    (arguments
     `(#:jar-name "InDelFixer.jar"
       #:source-dir "src/main/java"
       #:test-dir "src/test"))
    (inputs
     `(("java-commons-lang2" ,java-commons-lang)
       ("java-args4j" ,java-args4j)))
    (native-inputs
     (list java-junit))
    (home-page "https://github.com/cbg-ethz/InDelFixer/")
    (synopsis "Iterative and sensitive NGS sequence aligner")
    (description "InDelFixer is a sensitive aligner for 454, Illumina and
PacBio data, employing a full Smith-Waterman alignment against a reference.
This Java command line application aligns Next-Generation Sequencing (NGS) and
third-generation reads to a set of reference sequences, by a prior fast k-mer
matching and removes indels, causing frame shifts.  In addition, only a
specific region can be considered.  An iterative refinement of the alignment
can be performed, by alignment against the consensus sequence with wobbles.
The output is in SAM format.")
    (license license:gpl3+)))

(define-public libsbml
  (package
    (name "libsbml")
    (version "5.20.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/sbmlteam/libsbml")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bzsgwblh5l15xxy7kpdwiya5kwm26sj6daqr9i93h04manpqmfw"))))
    (build-system cmake-build-system)
    (arguments
     `(#:configure-flags
       ,#~(list "-DWITH_CHECK=ON"
                (string-append "-DLIBXML_LIBRARY="
                               #$(this-package-input "libxml2")
                               "/lib/libxml2.so")
                (string-append "-DLIBXML_INCLUDE_DIR="
                               #$(this-package-input "libxml2")
                               "/include/libxml2"))))
    (propagated-inputs
     (list libxml2))
    (native-inputs
     (list check swig))
    (home-page "https://sbml.org/Software/libSBML")
    (synopsis "Process SBML files and data streams")
    (description "LibSBML is a library to help you read, write, manipulate,
translate, and validate SBML files and data streams.  The @dfn{Systems Biology
Markup Language} (SBML) is an interchange format for computer models of
biological processes.  SBML is useful for models of metabolism, cell
signaling, and more.  It continues to be evolved and expanded by an
international community.")
    (license license:lgpl2.1+)))

(define-public kraken2
  (package
    (name "kraken2")
    (version "2.1.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/DerrickWood/kraken2")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pl6ml1ldg2hnhy8ps56q0fl1wq3g91qkhinj6pb4yjjhv1rxsjf"))))
    (build-system gnu-build-system)
    (arguments
     `(#:tests? #false                  ; there are none
       #:make-flags
       ,#~(list "-C" "src"
                (string-append "KRAKEN2_DIR=" #$output "/bin"))
       #:phases
       (modify-phases %standard-phases
         (delete 'configure)
         (add-before 'install 'install-scripts
           (lambda* (#:key outputs #:allow-other-keys)
             (let* ((bin (string-append (assoc-ref outputs "out") "/bin"))
                    (replacements `(("KRAKEN2_DIR" . ,bin)
                                    ("VERSION" . ,,version))))
               (mkdir-p bin)

               (with-directory-excursion "scripts"
                 (let ((scripts (find-files "." ".*")))
                   (substitute* scripts
                     (("#####=([^=]+)=#####" _ key)
                      (or (assoc-ref replacements key)
                          (error (format #false "unknown key: ~a~%" key)))))
                   (substitute* "kraken2"
                     (("compression_program = \"bzip2\"")
                      (string-append "compression_program = \""
                                     (which "bzip2")
                                     "\""))
                     (("compression_program = \"gzip\"")
                      (string-append "compression_program = \""
                                     (which "gzip")
                                     "\"")))
                   (substitute* '("download_genomic_library.sh"
                                  "download_taxonomy.sh"
                                  "16S_gg_installation.sh"
                                  "16S_silva_installation.sh"
                                  "16S_rdp_installation.sh")
                     (("wget") (which "wget")))
                   (substitute* '("download_taxonomy.sh"
			          "download_genomic_library.sh"
			          "rsync_from_ncbi.pl")
		     (("rsync -")
                      (string-append (which "rsync") " -")))
                   (substitute* "mask_low_complexity.sh"
                     (("which") (which "which")))
                   (substitute* '("mask_low_complexity.sh"
                                  "download_genomic_library.sh"
                                  "16S_silva_installation.sh")
                     (("sed -e ")
                      (string-append (which "sed") " -e ")))
                   (substitute* '("rsync_from_ncbi.pl"
                                  "16S_rdp_installation.sh"
                                  "16S_silva_installation.sh"
                                  "16S_gg_installation.sh"
                                  "download_taxonomy.sh"
                                  "download_genomic_library.sh")
                     (("gunzip") (which "gunzip")))
                   (for-each (lambda (script)
                               (chmod script #o555)
                               (install-file script bin))
                             scripts)))))))))
    (inputs
     (list gzip
           perl
           rsync
           sed
           wget
           which))
  (home-page "https://github.com/DerrickWood/kraken2")
  (synopsis "Taxonomic sequence classification system")
  (description "Kraken is a taxonomic sequence classifier that assigns
taxonomic labels to DNA sequences.  Kraken examines the k-mers within a query
sequence and uses the information within those k-mers to query a
database. That database maps k-mers to the lowest common ancestor (LCA) of all
genomes known to contain a given k-mer.")
  (license license:expat)))

(define-public lofreq
  (package
    (name "lofreq")
    (version "2.1.5")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/CSB5/lofreq")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qssrn3mgjak7df6iqc1rljqd3g3a5syvg0lsv4vds43s3fq23bl"))))
    (build-system gnu-build-system)
    (arguments
     '(#:test-target "bug-tests"
       #:tests? #false ;test data are not included
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'set-source-file-times-to-1980
           (lambda _
             (let ((circa-1980 (* 10 366 24 60 60)))
               (for-each (lambda (file)
                           (let ((s (lstat file)))
                             (unless (eq? (stat:type s) 'symlink)
                               (utime file circa-1980 circa-1980))))
                         (find-files "." #:directories? #t))))))))
    (inputs
     (list htslib python-wrapper zlib))
    (native-inputs
     (list autoconf automake which))
    (home-page "https://csb5.github.io/lofreq/")
    (synopsis "Sensitive variant calling from sequencing data")
    (description "LoFreq is a fast and sensitive variant-caller for inferring
SNVs and indels from next-generation sequencing data.  It makes full use of
base-call qualities and other sources of errors inherent in
sequencing (e.g. mapping or base/indel alignment uncertainty), which are
usually ignored by other methods or only used for filtering.")
    (license license:expat)))

(define-public louvain
  (package
    (name "louvain")
    (version "0.2")
    (source (origin
              (method url-fetch)
              (uri "mirror://sourceforge/louvain/louvain_latest.tar.gz")
              (sha256
               (base32
                "0hqlv5jqc889nbv7j1bchrx4zhh69hgr2mqvfdygc7kwrywn22lb"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:tests? #false                   ;there are none
      #:phases
      #~(modify-phases %standard-phases
          (delete 'configure)
          (add-after 'unpack 'patch-includes
            (lambda _
              (substitute* "main_community.cpp"
                (("using namespace std;" m)
                 (string-append "#include <unistd.h> /* for getpid */\n" m)))))
          (replace 'install
            (lambda _
              (for-each
               (lambda (exe)
                 (install-file exe (string-append #$output "/bin")))
               '("convert" "community" "hierarchy")))))))
    (home-page "https://sourceforge.net/projects/louvain/")
    (synopsis "Multi-criteria community detection")
    (description "This package offers a set of functions to use in order to
compute communities on graphs weighted or unweighted.")
    (license license:gpl3+)))

(define-public ivar
  (package
    (name "ivar")
    (version "1.4.4")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/andersen-lab/ivar")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bwygjv81m4yv5ycxb5gx22k6njgwrzzs7jrsadq6363klcv4irh"))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:parallel-tests? #false          ;not supported
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'patch-CXXFLAGS
           (lambda _
             (substitute* '("src/Makefile.am"
                            "tests/Makefile.am")
               (("-Werror") "")))))))
    (inputs
     (list htslib zlib))
    (native-inputs
     (list autoconf automake))
    (home-page "https://andersen-lab.github.io/ivar/html/")
    (synopsis "Tools for amplicon-based sequencing")
    (description "iVar is a computational package that contains functions
broadly useful for viral amplicon-based sequencing.")
    (license license:gpl3+)))

(define-public python-pyliftover
  (package
    (name "python-pyliftover")
    (version "0.4")
    ;; The version of pypi does not include test data.
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/konstantint/pyliftover")
             (commit version)))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1j8jp9iynv2l3jv5pr0pn0p3azlama1bqg233piglzm6bqh3m2m3"))))
    (build-system pyproject-build-system)
    (arguments
     `(#:tests? #f)) ;the tests access the web
    (native-inputs (list python-pytest python-setuptools python-wheel))
    (home-page "https://github.com/konstantint/pyliftover")
    (synopsis
     "Python implementation of UCSC liftOver genome coordinate conversion")
    (description
     "PyLiftover is a library for quick and easy conversion of genomic (point)
coordinates between different assemblies.")
    (license license:expat)))

(define-public python-cgatcore
  (package
    (name "python-cgatcore")
    (version "0.6.16")
    ;; The version of pypi does not include test data.
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/cgat-developers/cgat-core")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kvfb6fpfncdfb8wjmn7n2vmqk3wd7sdrfw1rhlihfdxbfzb5fa8"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      ;; Failed: 'yield' keyword is allowed in fixtures, but not in tests
      ;; (test_import)
      '(list "--ignore=tests/test_import.py"
             "-k"
             (string-append
              ;; This test actually does what it should, but the check fails with
              ;; TypeError: cannot unpack non-iterable Namespace object
              "not test_start_and_stop_are_logged_with_argparse"
              ;; These all attempt to connect to localhost with SSH
              " and not test_job_should_use_TMPDIR_and_clean_up"
              " and not test_job_should_use_TMPDIR_and_clean_up_after_fail"
              " and not test_job_should_write_to_explicit_temp_and_not_clean_up"))
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'fix-references
           (lambda _
             (substitute* "cgatcore/pipeline/execution.py"
               (("#!/bin/bash") (string-append "#!" (which "bash")))
               (("executable=\"/bin/bash\"")
                (string-append "executable=\"" (which "bash") "\""))
               (("\\\\time") (which "time")))))
         (add-after 'unpack 'sqlite-compatibility
           (lambda _
             ;; Load apsw (and thus newer sqlite3) before importing Python's
             ;; older sqlite3 library.
             (substitute* "cgatcore/pipeline/__init__.py"
               (("import os")
                (string-append "import os\nimport apsw")))
             (substitute* "tests/template_pipeline.py"
               (("import sys" m)
                (string-append "import apsw\n" m))))))))
    (native-inputs
     (list inetutils
           lsof
           openssl
           python-pytest
           python-setuptools))
    (inputs (list time))
    (propagated-inputs
     (list python-apsw
           python-gevent
           python-pandas
           python-paramiko
           python-pyyaml
           python-ruffus
           python-sqlalchemy))
    (home-page "https://github.com/cgat-developers/cgat-core")
    (synopsis "Computational genomics analysis toolkit")
    (description
     "CGAT-core is a set of libraries and helper functions used to enable
researchers to design and build computational workflows for the analysis of
large-scale data-analysis.")
    (license license:expat)))

(define-public perl-cworld-dekker
  (package
    (name "perl-cworld-dekker")
    (version "1.01")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/dekkerlab/cworld-dekker.git")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1dvh23fx52m59y6304xi2j2pl2hiqadlqg8jyv2pm14j1hy71ych"))))
    (build-system perl-build-system)
    (arguments
     (list
      #:modules '((guix build perl-build-system)
                  (guix build utils)
                  (srfi srfi-26))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'hardcode-references
            (lambda* (#:key inputs #:allow-other-keys)
              (let ((bedtools #$(this-package-input "bedtools"))
                    (r #$(this-package-input "r-minimal")))
                (substitute* '("scripts/python/getEigenVectors.py"
                               "scripts/python/matrix2EigenVectors.py")
                  (("bedtools intersect")
                   (string-append bedtools "/bin/bedtools intersect")))
                (substitute* "lib/cworld/dekker.pm"
                  (("bedtools --version")
                   (string-append bedtools "/bin/bedtools --version")))
                (substitute* '("scripts/perl/correlateMatrices.pl"
                               "scripts/perl/matrix2scaling.pl"
                               "scripts/perl/matrix2distance.pl"
                               "scripts/perl/coverageCorrect.pl"
                               "scripts/perl/matrix2anchorPlot.pl"
                               "scripts/python/matrix2EigenVectors.py"
                               "scripts/python/matrix2insulation-lite.py"
                               "scripts/perl/matrix2compartment.pl"
                               "scripts/perl/anchorPurge.pl"
                               "scripts/perl/applyCorrection.pl"
                               "scripts/perl/compareInsulation.pl"
                               "scripts/perl/fillMissingData.pl"
                               "scripts/perl/matrix2loess.pl"
                               "scripts/python/getEigenVectors.py"
                               "scripts/perl/aggregateBED.pl"
                               "scripts/perl/collapseMatrix.pl"
                               "scripts/perl/matrix2direction.pl"
                               "scripts/perl/singletonRemoval.pl"
                               "lib/cworld/dekker.pm"
                               "scripts/perl/matrix2insulation.pl")
                  (("(`|\")Rscript" _ pre)
                   (string-append pre r "/bin/Rscript"))))))
          (add-after 'install 'install-scripts
            (lambda _
              (let ((share (string-append #$output "/share/cworld-dekker")))
                (mkdir-p share)
                (copy-recursively "scripts" share)

                ;; Make all scripts executable and wrap them.
                (let ((r     (find-files share "\\.R$"))
                      (py    (find-files share "\\.py$"))
                      (pl    (find-files share "\\.pl$"))
                      (wrap  (lambda* (script var #:optional (extra ""))
                               (let ((path (string-append (getenv var)
                                                          extra)))
                                 (wrap-program script
                                   `(,var ":" prefix (,path)))))))
                  (for-each (cut chmod <> #o555) (append r py pl))
                  (for-each (cut wrap <> "PERL5LIB"
                                 (string-append ":" #$output
                                                "/lib/perl5/site_perl"))
                            pl)
                  (for-each (cut wrap <> "GUIX_PYTHONPATH") py))))))))
    (inputs
     (list gd
           perl-gd
           bedtools
           python-wrapper
           python-scipy
           python-numpy
           python-matplotlib
           python-h5py
           python-scikit-learn
           r-minimal))
    (native-inputs
     (list perl-module-build))
    (home-page "https://github.com/dekkerlab/cworld-dekker")
    (synopsis "Utility and analysis scripts for 3C, 4C, 5C, and Hi-C data")
    (description "This package is a collection of Perl, Python, and R
scripts for manipulating 3C/4C/5C/Hi-C data.")
    (license license:asl2.0)))

(define-public ensembl-vep
  (let* ((api-version "112")
         (api-module
          (lambda (name hash)
            (origin (method git-fetch)
                    (uri (git-reference
                          (url (string-append "https://github.com/Ensembl/"
                                              name ".git"))
                          (commit (string-append "release/" api-version))))
                    (file-name (string-append name "-" api-version "-checkout"))
                    (sha256 (base32 hash))))))
    (package
      (name "ensembl-vep")
      (version api-version)
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/Ensembl/ensembl-vep.git")
               (commit (string-append "release/" version))))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0nznmiln2q6bzd3xp8d5cw56higa5w3g9rmg66956bkfky53chvw"))))
      (build-system gnu-build-system)
      (arguments
       `(#:modules ((guix build gnu-build-system)
                    (guix build utils)
                    (ice-9 match))
         #:phases
         (modify-phases %standard-phases
           (delete 'configure)
           (delete 'build)
           ;; Tests need to run after installation
           (delete 'check)
           (replace 'install
             (lambda* (#:key inputs outputs #:allow-other-keys)
               (let* ((modules '(("ensembl" "/")
                                 ("ensembl-variation" "/Variation")
                                 ("ensembl-funcgen"   "/Funcgen")
                                 ("ensembl-io"        "/")))
                      (scripts '(("convert_cache.pl" "vep_convert_cache.pl")
                                 ("INSTALL.pl"       "vep_install.pl")
                                 ("haplo"            #f)
                                 ("variant_recoder"  #f)
                                 ("filter_vep"       #f)
                                 ("vep"              #f)))
                      (out  (assoc-ref outputs "out"))
                      (bin  (string-append out "/bin"))
                      (perl (string-append out "/lib/perl5/site_perl")))
                 (for-each
                  (match-lambda
                    ((name path)
                     (let ((dir (string-append perl "/Bio/EnsEMBL" path)))
                       (mkdir-p dir)
                       (copy-recursively
                        (string-append (assoc-ref inputs (string-append "api-module-" name))
                                       "/modules/Bio/EnsEMBL" path)
                        dir))))
                  modules)
                 (copy-recursively "modules/" perl)
                 (mkdir-p bin)
                 (for-each
                  (match-lambda
                    ((script new-name)
                     (let ((location (string-append bin "/"
                                                    (or new-name (basename script)))))
                       (copy-file script location)
                       (chmod location #o555)
                       (wrap-program location
                         `("PERL5LIB" ":" prefix (,(getenv "PERL5LIB")
                                                  ,perl))))))
                  scripts)

                 ;; Fix path to tools
                 (with-directory-excursion (string-append perl "/Bio/EnsEMBL")
                   (substitute* '("Funcgen/RunnableDB/ProbeMapping/PrePipelineChecks.pm"
                                  "VEP/BaseRunner.pm"
                                  "VEP/Utils.pm"
                                  "VEP/AnnotationSource/Cache/VariationTabix.pm"
                                  "VEP/AnnotationSource/Cache/BaseSerialized.pm"
                                  "Variation/Utils/BaseVepTabixPlugin.pm"
                                  "Variation/Utils/VEP.pm"
                                  "Variation/Pipeline/ReleaseDataDumps/PreRunChecks.pm")
                     (("`which")
                      (string-append "`"
                                     (assoc-ref inputs "which")
                                     "/bin/which")))))))
           (add-after 'install 'check
             (lambda* (#:key tests? inputs outputs #:allow-other-keys)
               (when tests?
                 (setenv "PERL5LIB"
                         (string-append (getenv "PERL5LIB")
                                        ":"
                                        (assoc-ref outputs "out")
                                        "/lib/perl5/site_perl"))
                 (copy-recursively (string-append (assoc-ref inputs "source") "/t")
                                   "/tmp/t")
                 (for-each make-file-writable (find-files "/tmp/t"))
                 ;; TODO: haplo needs Set/IntervalTree.pm
                 (invoke "perl" "-e" (string-append "
use Test::Harness; use Test::Exception;
my $dirname = \"/tmp\";
opendir TEST, \"$dirname\\/t\";
my @test_files = map {\"$dirname\\/t\\/\".$_} grep {!/^\\./ && /\\.t$/} readdir TEST; closedir TEST;
@test_files = grep {!/Haplo/} @test_files;
runtests(@test_files);
"))))))))
      (inputs
       (list bioperl-minimal
             perl-bio-db-hts
             perl-dbi
             perl-dbd-mysql
             perl-libwww
             perl-list-moreutils
             perl-http-tiny
             perl-json
             which))
      (propagated-inputs
       (list kentutils))
      (native-inputs
       `(("unzip" ,unzip)
         ("perl" ,perl)
         ("api-module-ensembl"
          ,(api-module "ensembl"
                       "10r576iqrz1p61mqbbfs87w4w3nldi45p51z52mq76f3n00l010h"))
         ("api-module-ensembl-variation"
          ,(api-module "ensembl-variation"
                       "0ad03xnyyqpya1bkg6igq9abqxdmi58j89wn2kb3m4bvy1zs03mf"))
         ("api-module-ensembl-funcgen"
          ,(api-module "ensembl-funcgen"
                       "1m3dxnr6k9x5hmcsxckpxlqx7vbc82lk74r84vcjijgxjkfa5rpp"))
         ("api-module-ensembl-io"
          ,(api-module "ensembl-io"
                       "1xr6kdqqzmbi3m4bls589d1dfjvbs13dxlbf1y172xjyipvmag43"))
         ("perl-test-deep" ,perl-test-deep)
         ("perl-test-exception" ,perl-test-exception)
         ("perl-test-harness" ,perl-test-harness)
         ("perl-test-warnings" ,perl-test-warnings)))
      (home-page "http://www.ensembl.org/vep")
      (synopsis "Predict functional effects of genomic variants")
      (description
       "This package provides a Variant Effect Predictor, which predicts
the functional effects of genomic variants.  It also provides
Haplosaurus, which uses phased genotype data to predict
whole-transcript haplotype sequences, and Variant Recoder, which
translates between different variant encodings.")
      (license license:asl2.0))))

(define-public r-signac
  (let ((commit "8ecdde291676102bb3b503f48926c993354b5471")
        (revision "1"))
    (package
      (name "r-signac")
      (version (git-version "1.12.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/timoast/signac/")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0idkcqb1i13m164nxbdwgs3vflw0cxzm2ir84aw0i811vx9lqz8c"))))
      (properties `((upstream-name . "Signac")))
      (build-system r-build-system)
      (inputs (list zlib))
      (propagated-inputs
       (list r-biocgenerics
             r-data-table
             r-dplyr
             r-fastmatch
             r-future
             r-future-apply
             r-genomeinfodb
             r-genomicranges
             r-ggplot2
             r-iranges
             r-irlba
             r-matrix
             r-patchwork
             r-pbapply
             r-rcpp
             r-rcpproll
             r-rsamtools
             r-s4vectors
             r-scales
             r-seuratobject
             r-stringi
             r-tidyr
             r-tidyselect
             r-vctrs))
      (native-inputs (list r-testthat))
      (home-page "https://github.com/timoast/signac/")
      (synopsis "Analysis of single-cell chromatin data")
      (description
       "This package provides a framework for the analysis and exploration of
single-cell chromatin data.  The Signac package contains functions for
quantifying single-cell chromatin data, computing per-cell quality control
metrics, dimension reduction and normalization, visualization, and DNA
sequence motif analysis.")
      (license license:expat))))

(define-public tombo
  (package
    (name "tombo")
    (version "1.5.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "ont-tombo" version))
       (sha256
        (base32
         "1023hadgcsgi53kz53ql45207hfizf9sw57z0qij3ay1bx68zbpm"))))
    (build-system pyproject-build-system)
    (arguments
     '(#:tests? #false))                ;no tests
    (native-inputs
     (list python-cython-0 python-setuptools))
    ;; The package mainly consists of a command-line tool, but also has a
    ;; Python-API. Thus these must be propagated.
    (propagated-inputs
     (list python-future
           python-h5py
           python-mappy
           python-numpy
           python-scipy
           python-tqdm
           python-rpy2))
    (home-page "https://github.com/nanoporetech/tombo")
    (synopsis "Analysis of raw nanopore sequencing data")
    (description "Tombo is a suite of tools primarily for the identification of
modified nucleotides from nanopore sequencing data.  Tombo also provides tools
for the analysis and visualization of raw nanopore signal.")
    ;; Some parts may be BSD-3-licensed.
    (license license:mpl2.0)))

;; This is a fork of the original unmaintained python-pyvcf.
(define-public python-pyvcf3
  (let ((commit "1fb3789153d1d8e28e2cedf121399f276b5f312a")
        (revision "0"))
    (package
      (name "python-pyvcf3")
      (version (git-version "1.0.3" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/dridk/PyVCF3")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0i4j5bq5q32q216ja7yvg0mpww5j0b9p8ky5bya4d31wqmygal8z"))))
      (build-system pyproject-build-system)
      (propagated-inputs (list python-setuptools))
      (native-inputs (list python-pytest python-setuptools))
      (home-page "https://github.com/dridk/PyVCF3")
      (synopsis "Variant Call Format parser for Python")
      (description "This package provides a @acronym{VCF,Variant Call Format}
parser for Python.")
      (license license:expat))))

(define-deprecated/public-alias python-pyvcf python-pyvcf3)

(define-public nanosv
  (package
    (name "nanosv")
    (version "1.2.4")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "NanoSV" version))
       (sha256
        (base32 "1wl2daj0bwrl8fx5xi8j8hfs3mp3vg3qycy66538n032v1qkc6xg"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #f   ; No tests upstream, even in git.
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'relax-requirements
            (lambda _
              (substitute* "setup.py"
                (("'pyvcf'")
                 "'pyvcf3'")))))))
    (native-inputs (list python-setuptools python-wheel))
    (inputs (list python-configparser python-pysam python-pyvcf3))
    (home-page "https://github.com/mroosmalen/nanosv")
    (synopsis "Structural variation detection tool for Oxford Nanopore data")
    (description
     "NanoSV is a software package that can be used to identify
structural genomic variations in long-read sequencing data, such as data
produced by Oxford Nanopore Technologies’ MinION, GridION or PromethION
instruments, or Pacific Biosciences RSII or Sequel sequencers.")
    (license license:expat)))

(define-public python-strawc
  (package
    (name "python-strawc")
    (version "0.0.2.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "strawC" version))
       (sha256
        (base32
         "1z1gy8n56lhriy6hdkh9r82ndikndipq2cy2wh8q185qig4rimr6"))))
    (build-system pyproject-build-system)
    (native-inputs (list python-setuptools))
    (arguments (list #:tests? #f))      ;No tests on either PyPI or Git.
    (inputs
     (list curl zlib))
    (propagated-inputs
     (list pybind11))
    (home-page "https://github.com/aidenlab/straw")
    (synopsis "Stream data from .hic files")
    (description "Straw is library which allows rapid streaming of contact
data from @file{.hic} files.  This package provides Python bindings.")
    (license license:expat)))

(define-public python-pybbi
  (package
    (name "python-pybbi")
    (version "0.4.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "pybbi" version))
       (sha256
        (base32 "0p1s6y9f33wzmvxdhfg9b37sas2kghnmvkfnb317aiad1p6ks6ba"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list "-k" (string-join
                    ;; Network is required to run these tests.
                    (list "not test_aws_403_redirect"
                          "test_chromsizes"
                          "test_fetch_remote"
                          "test_fetch_remote_https"
                          "test_sigs")
                    " and not "))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'set-cc
            (lambda _ (setenv "CC" #$(cc-for-target))))
          (add-before 'check 'remove-local-bbi
            (lambda _
              ;; This would otherwise interfere with finding the installed bbi
              ;; when running tests.
              (delete-file-recursively "bbi"))))))
    (native-inputs
     (list pkg-config
           python-cython
           python-pandas
           python-pkgconfig
           python-pytest
           python-setuptools))
    (inputs
     (list libpng
           openssl
           zlib))
    (propagated-inputs
     (list python-numpy))
    (home-page "https://github.com/nvictus/pybbi")
    (synopsis "Python bindings to UCSC Big Binary file library")
    (description
     "This package provides Python bindings to the UCSC Big
Binary (bigWig/bigBed) file library.  This provides read-level access to local
and remote bigWig and bigBed files but no write capabilitites.  The main
feature is fast retrieval of range queries into numpy arrays.")
    (license license:expat)))

(define-public python-dna-features-viewer
  (package
    (name "python-dna-features-viewer")
    (version "3.1.1")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "dna_features_viewer" version))
       (sha256
        (base32
         "10a5pmh73spdgfml88zjqg1rn2j2zdxn06ir8a5q3arspyqvyqcb"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:tests? #false ;there are none
      #:phases
      '(modify-phases %standard-phases
         ;; See https://github.com/Edinburgh-Genome-Foundry/DnaFeaturesViewer/issues/73
         ;; and https://github.com/Edinburgh-Genome-Foundry/DnaFeaturesViewer/pull/74
         (add-after 'unpack 'fix-biopython-compatibility
           (lambda _
             (substitute* "dna_features_viewer/biotools.py"
               (("zip\\(aa1 \\+ \"\\*\", aa3 \\+ \\[\"\\*\"\\]\\)")
                "zip(aa1 + ('*',), aa3 + ('*',))")))))))
    (propagated-inputs
     (list python-biopython
           python-matplotlib
           python-packaging))
    (native-inputs (list python-setuptools python-wheel))
    (home-page
     "https://github.com/Edinburgh-Genome-Foundry/DnaFeaturesViewer")
    (synopsis "Plot features from DNA sequences")
    (description
     "DNA Features Viewer is a Python library to visualize DNA features,
e.g. from GenBank or Gff files, or Biopython SeqRecords.")
    (license license:expat)))

(define-public python-coolbox
  (package
    (name "python-coolbox")
    (version "0.3.8")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "coolbox" version))
       (sha256
        (base32
         "0gqp76285w9klswr47y6kxbzwhv033b26jfa179kccfhiaq5p2xa"))))
    (build-system python-build-system)
    (arguments '(#:tests? #false)) ; there are none
    (inputs
     (list pybind11))
    (propagated-inputs
     (list python-cooler
           python-dna-features-viewer
           python-fire
           python-h5py
           python-intervaltree
           python-ipywidgets
           jupyter
           python-matplotlib
           python-nbformat
           python-numpy
           python-numpydoc
           python-pandas
           python-pybbi
           python-pytest
           python-scipy
           python-statsmodels
           python-strawc
           python-svgutils
           python-termcolor
           python-voila))
    (home-page "https://github.com/GangCaoLab/CoolBox")
    (synopsis "Genomic data visualization toolkit")
    (description
     "CoolBox is a toolkit for visual analysis of genomics data.  It aims to
be highly compatible with the Python ecosystem, easy to use and highly
customizable with a well-designed user interface.  It can be used in various
visualization situations, for example, to produce high-quality genome track
plots or fetch common used genomic data files with a Python script or command
line, interactively explore genomic data within Jupyter environment or web
browser.")
    (license license:gpl3+)))

(define-public python-pyspoa
  (package
    (name "python-pyspoa")
    (version "0.0.5")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/nanoporetech/pyspoa")
             (commit (string-append "v" version))
             (recursive? #true)))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1lgf2shzhxkcsircd6vy46h27pjljd5q95fyz1cm3lkk702qbnzx"))))
    (build-system python-build-system)
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-before 'build 'build-libspoa
           (lambda _
             (mkdir-p "src/build")
             (with-directory-excursion "src/build"
               (invoke "cmake"
                       "-Dspoa_optimize_for_portability=ON"
                       "-DCMAKE_BUILD_TYPE=Release"
                       "-DCMAKE_CXX_FLAGS=\"-I ../vendor/cereal/include/\" -fPIC"
                       "..")
               (invoke "make"))))
         (replace 'check
           (lambda* (#:key inputs outputs tests? #:allow-other-keys)
             (when tests?
               (add-installed-pythonpath inputs outputs)
               (invoke "python" "tests/test_pyspoa.py")))))))
    (propagated-inputs
     (list pybind11))
    (native-inputs
     `(("cmake" ,cmake-minimal)))
    (home-page "https://github.com/nanoporetech/pyspoa")
    (synopsis "Python bindings for the SIMD partial order alignment library")
    (description
     "This package provides Python bindings for spoa, a C++ implementation of
the @dfn{partial order alignment} (POA) algorithm (as described in
10.1093/bioinformatics/18.3.452) which is used to generate consensus
sequences")
    (license license:expat)))

(define-public python-bwapy
  (package
    (name "python-bwapy")
    (version "0.1.4")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "bwapy" version))
       (sha256
        (base32 "090qwx3vl729zn3a7sksbviyg04kc71gpbm3nd8dalqp673x1npw"))
       (modules '((guix build utils)))
       (snippet
        '(for-each delete-file (find-files "." "\\.o$")))))
    (build-system pyproject-build-system)
    (arguments
     (list #:tests? #f ;no tests
           #:phases
           #~(modify-phases %standard-phases
               ;; TODO: it's possible that the import error points to a real
               ;; problem with the C sources.
               (delete 'sanity-check))))
    (propagated-inputs
     (list python-cffi python-setuptools))
    (inputs
     (list zlib))
    (home-page "https://github.com/ACEnglish/bwapy")
    (synopsis "Python bindings to bwa aligner")
    (description "This package provides Python bindings to the bwa mem
aligner.")
    ;; These Python bindings are licensed under Mozilla Public License 2.0,
    ;; bwa itself is licenced under GNU General Public License v3.0.
    (license license:mpl2.0)))

(define-public scvelo
  (package
    (name "scvelo")
    (version "0.2.4")
    (source
     (origin
       (method url-fetch)
       (uri (pypi-uri "scvelo" version))
       (sha256
        (base32 "0h5ha1459ljs0qgpnlfsw592i8dxqn6p9bl08l1ikpwk36baxb7z"))))
    (build-system pyproject-build-system)
    (arguments
     (list
       #:test-flags
       '(list "--pyargs" "scvelo/core"
              ;; XXX: these two tests fail for unknown reasons
              "-k"
              "not test_perfect_fit and not test_perfect_fit_2d")
       #:phases
       #~(modify-phases %standard-phases
           (add-after 'unpack 'matplotlib-compatibility
             (lambda _
               (substitute* "scvelo/settings.py"
                 (("warnings.filterwarnings\\(\"ignore\", category=cbook.mplDeprecation\\)")
                  ""))))
           ;; Numba needs a writable dir to cache functions.
           (add-before 'check 'set-numba-cache-dir
             (lambda _
               (setenv "NUMBA_CACHE_DIR" "/tmp"))))))
    (propagated-inputs
     (list pybind11     ;XXX: marked as install requirement
           python-anndata
           python-hnswlib
           python-igraph
           python-loompy
           python-louvain
           python-matplotlib
           python-numba
           python-numpy
           python-pandas
           python-scanpy
           python-scikit-learn
           python-scipy
           python-umap-learn))
    (native-inputs
     (list python-pytest
           python-setuptools
           python-setuptools-scm))
    (home-page "https://scvelo.org")
    (synopsis "RNA velocity generalized through dynamical modeling")
    (description "ScVelo is a scalable toolkit for RNA velocity analysis in
single cells.  RNA velocity enables the recovery of directed dynamic
information by leveraging splicing kinetics. scVelo generalizes the concept of
RNA velocity by relaxing previously made assumptions with a stochastic and a
dynamical model that solves the full transcriptional dynamics.  It thereby
adapts RNA velocity to widely varying specifications such as non-stationary
populations.")
    (license license:bsd-3)))

(define-public scregseg
  ;; 0.1.3 was released in 2023, there are a lot of comparability fixes on
  ;; master branch, use the latest commit for now.
  (let ((commit "78ebff8c3507752c3bfbc4db3f72f7e8a733e92f")
        (revision "0"))
  (package
    (name "scregseg")
    (version (git-version "0.1.3" revision commit))
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/BIMSBbioinfo/scregseg")
                    (commit commit)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19iasx6zh305cn8p390ack78f4iklyk61xmnf99c2b8ibml7jmzj"))
              (snippet
               '(delete-file "src/scregseg/_utils.c"))))
    (build-system pyproject-build-system)
    (arguments
     (list
      #:test-flags
      #~(list "--pyargs" "scregseg")
      #:phases
      '(modify-phases %standard-phases
         ;; Numba needs a writable dir to cache functions.
         (add-before 'check 'set-numba-cache-dir
           (lambda _
             (setenv "NUMBA_CACHE_DIR" "/tmp")))
         ;; NumPy 1.20 deprecated the type wrappers for int and float.
         (add-after 'unpack 'compatibility
           (lambda _
             (substitute* "src/scregseg/_utils.pyx"
               (("np.float") "float")))))))
    (native-inputs
     (list python-cython
           python-setuptools))
    (propagated-inputs
     (list python-scikit-learn-1.6
           python-scipy
           python-numpy
           python-hmmlearn
           python-pandas
           python-numba
           python-anndata
           python-scanpy
           python-pybedtools
           python-pysam
           python-matplotlib
           python-seaborn
           python-coolbox))
    (home-page "https://github.com/BIMSBbioinfo/scregseg")
    (synopsis "Single-cell regulatory landscape segmentation")
    (description "Scregseg (Single-Cell REGulatory landscape SEGmentation) is a
tool that facilitates the analysis of single cell ATAC-seq data by an
HMM-based segmentation algorithm.  Scregseg uses an HMM with
Dirichlet-Multinomial emission probabilities to segment the genome either
according to distinct relative cross-cell accessibility profiles or (after
collapsing the single-cell tracks to pseudo-bulk tracks) to capture distinct
cross-cluster accessibility profiles.")
    (license license:gpl3+))))

(define-public megadepth
  (package
    (name "megadepth")
    (version "1.1.1")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/ChristopherWilks/megadepth")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hj69d2dgmk2zwgazik7xzc04fxxlk93p888kpgc52fmhd95qph7"))))
    (build-system cmake-build-system)
    (arguments
     `(#:tests? #false ; some tests seem to require connection to
                       ; www.ebi.ac.uk; this may be caused by htslib.
       #:phases
       (modify-phases %standard-phases
         (add-after 'unpack 'prepare-CMakeLists.txt
           (lambda _
             (rename-file "CMakeLists.txt.ci" "CMakeLists.txt")
             (substitute* "CMakeLists.txt"
               (("`cat ../VERSION`") ,version)
               (("target_link_libraries\\(megadepth_static") "#")
               (("target_link_libraries\\(megadepth_statlib") "#")
               (("add_executable\\(megadepth_static") "#")
               (("add_executable\\(megadepth_statlib") "#"))

             (substitute* "tests/test.sh"
               ;; Disable remote test
               (("./megadepth http://stingray.cs.jhu.edu/data/temp/test.bam") "#")
               ;; Prior to installation the binary's name differs from what
               ;; the test script assumes.
               (("./megadepth") "../build/megadepth_dynamic"))))
         (replace 'check
           (lambda* (#:key tests? #:allow-other-keys)
             (when tests?
               (with-directory-excursion "../source"
                 (invoke "bash" "tests/test.sh" "use-local-test-data")))))
         (replace 'install
           (lambda* (#:key outputs #:allow-other-keys)
             (let ((bin (string-append (assoc-ref outputs "out") "/bin")))
               (mkdir-p bin)
               (copy-file "megadepth_dynamic"
                          (string-append bin "/megadepth"))))))))
    (native-inputs
     (list diffutils perl grep))
    (inputs
     (list curl htslib libdeflate libbigwig zlib))
    (home-page "https://github.com/ChristopherWilks/megadepth")
    (synopsis "BigWig and BAM/CRAM related utilities")
    (description "Megadepth is an efficient tool for extracting coverage
related information from RNA and DNA-seq BAM and BigWig files.  It supports
reading whole-genome coverage from BAM files and writing either indexed TSV or
BigWig files, as well as efficient region coverage summary over intervals from
both types of files.")
    (license license:expat)))

(define-public megahit
  (package
    (name "megahit")
    (version "1.2.9")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/voutcn/megahit.git")
                    (commit (string-append "v" version))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1r5d9nkdmgjsbrpj43q9hy3s8jwsabaz3ji561v18hy47v58923c"))))
    (build-system cmake-build-system)
    (arguments
     (list
      #:modules '((guix build cmake-build-system)
                  ((guix build gnu-build-system) #:prefix gnu:)
                  (guix build utils))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-headers
            (lambda _
              (substitute* "src/localasm/local_assemble.h"
                (("#include <string>") "#include <string>\n#include <cstdint>"))))
          (add-after 'unpack 'fix-tests
            (lambda _
              (substitute* "src/megahit"
                (("os.path.join\\(script_path, '..'\\)")
                 "os.path.join(script_path, '../source')"))))
          (replace 'check
            (lambda* (#:rest args)
              (apply (assoc-ref gnu:%standard-phases 'check)
                     #:test-target "simple_test" args))))))
    (inputs (list python-wrapper zlib))
    (home-page "https://www.ncbi.nlm.nih.gov/pubmed/25609793")
    (synopsis "Meta-genome assembler")
    (description "Megahit is a fast and memory-efficient NGS assembler.  It is
optimized for metagenomes, but also works well on generic single genome
assembly (small or mammalian size) and single-cell assembly.")
    (license license:gpl3)))

(define-public mudskipper
  (let ((commit "effd3fac03bc09d313e84fa680f18fdc6f3a16a0")
        (revision "1"))
    (package
      (name "mudskipper")
      (version (git-version "0.1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/OceanGenomics/mudskipper")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "17jm39sbrwgjlynxsn4g7lvq5hx6rwyjg86p10v4mc74fmdn5xd5"))))
      (build-system cargo-build-system)
      (arguments
       (list #:install-source? #f
             #:phases
             #~(modify-phases %standard-phases
                 (add-after 'unpack 'skip-doctesting-code-blocks
                   ;; See also https://github.com/rust-lang/rust/issues/63193.
                   (lambda _
                     (substitute* "src/query_bam_records.rs"
                       (((string-append
                          "(pub )?fn ("
                          (string-join
                           '("get_next_query_records"
                             "get_next_query_records_skip"
                             "get_primary_record_of_sa_tag"
                             "get_records_from_sa_tag"
                             "group_records"
                             "group_records_skip"
                             "new")
                           "|")
                          ")")
                         all)
                        (string-append "#[cfg(not(doctest))]\n" all))))))))
      (native-inputs
       (list pkg-config))
      (inputs
       (cons* zlib xz (cargo-inputs 'mudskipper)))
      (home-page "https://github.com/OceanGenomics/mudskipper")
      (synopsis "Convert genomic alignments to transcriptomic BAM/RAD files")
      (description "Mudskipper is a tool for projecting genomic alignments to
transcriptomic coordinates.")
      (license license:bsd-3))))

(define-public r-ascat
  (package
   (name "r-ascat")
   (version "2.5.2")
   (source (origin
            (method git-fetch)
            (uri (git-reference
                  (url "https://github.com/Crick-CancerGenomics/ascat.git")
                  (commit (string-append "v" version))))
            (file-name (git-file-name name version))
            (sha256
             (base32
              "0cc0y3as6cb64iwnx0pgbajiig7m4z723mns9d5i4j09ccid3ccm"))))
   (build-system r-build-system)
   (arguments
    `(#:phases
      (modify-phases %standard-phases
       (add-after 'unpack 'move-to-ascat-dir
         (lambda _
           (chdir "ASCAT"))))))
   (propagated-inputs
    (list r-rcolorbrewer))
   (home-page "https://github.com/VanLoo-lab/ascat/")
   (synopsis "Allele-Specific Copy Number Analysis of Tumors in R")
   (description "This package provides the @acronym{ASCAT,Allele-Specific Copy
Number Analysis of Tumors} R package that can be used to infer tumour purity,
ploidy and allele-specific copy number profiles.")
   (license license:gpl3)))

(define-public r-battenberg
  (package
   (name "r-battenberg")
   (version "2.2.9")
   (source (origin
            (method git-fetch)
            (uri (git-reference
                  (url "https://github.com/Wedge-lab/battenberg.git")
                  (commit (string-append "v" version))))
            (file-name (git-file-name name version))
            (sha256
             (base32
              "0nmcq4c7y5g8h8lxsq9vadz9bj4qgqn118alip520ny6czaxki4h"))))
   (build-system r-build-system)
   (propagated-inputs
    (list r-devtools
          r-readr
          r-doparallel
          r-ggplot2
          r-rcolorbrewer
          r-gridextra
          r-gtools
          r-ascat))
   (home-page "https://github.com/Wedge-lab/battenberg")
   (synopsis "Subclonal copy number estimation in R")
   (description "This package contains the Battenberg R package for subclonal
copy number estimation, as described by
@url{doi:10.1016/j.cell.2012.04.023,Nik-Zainal et al.}")
   (license license:gpl3)))

(define-public r-bayesprism
  (package
    (name "r-bayesprism")
    (version "2.2.2")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/Danko-Lab/BayesPrism")
             ;; The version string in the DESCRIPTION file was changed to
             ;; 2.2.2 in this commit.
             (commit "5d43190d5fdfc900571ae1b05b9dcad9ee6b8b2b")))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0rlq6nv1adc9w1ync8834kv59ksixxzbpf3xlbxi5s8x54gv4q9y"))))
    (properties `((upstream-name . "BayesPrism")))
    (build-system r-build-system)
    (arguments
     (list
      #:phases
      '(modify-phases %standard-phases
         (add-after 'unpack 'chdir
           (lambda _ (chdir "BayesPrism"))))))
    (propagated-inputs (list r-biocparallel
                             r-gplots
                             r-matrix
                             r-nmf
                             r-scran
                             r-snowfall
                             r-r-utils))
    (native-inputs (list r-knitr))
    (home-page "https://github.com/Danko-Lab/BayesPrism")
    (synopsis "Bayesian cell type and gene expression deconvolution")
    (description
     "BayesPrism includes deconvolution and embedding learning modules.
The deconvolution module models a prior from cell type-specific expression
profiles from scRNA-seq to jointly estimate the posterior distribution of cell
type composition and cell type-specific gene expression from bulk RNA-seq
expression of tumor samples.  The embedding learning module uses
@dfn{Expectation-maximization} (EM) to approximate the tumor expression using
a linear combination of malignant gene programs while conditional on the
inferred expression and fraction of non-malignant cells estimated by the
deconvolution module.")
    (license license:gpl3)))

(define-public r-catch
  (let ((commit "196ddd5a51b1a5f5daa01de53fdaad9b7505e084")
        (revision "1"))
    (package
      (name "r-catch")
      (version (git-version "1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/zhanyinx/CaTCH")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "11c7f1fc8f57wnwk1hrgr5y814m80zj8gkz5021vxyxy2v02cqgd"))))
      (build-system r-build-system)
      (arguments
       `(#:phases
         (modify-phases %standard-phases
           (add-after 'unpack 'chdir
             (lambda _ (chdir "CaTCH"))))))
      (home-page "https://github.com/zhanyinx/CaTCH_R")
      (synopsis "Call a hierarchy of domains based on Hi-C data")
      (description "This package allows building the hierarchy of domains
starting from Hi-C data.  Each hierarchical level is identified by a minimum
value of physical insulation between neighboring domains.")
      (license license:gpl2+))))

(define-public r-spacexr
  (let ((commit "0a0861e3d1e16014a20e9b743d0e19d3b42231f3")
        (revision "1"))
    (package
      (name "r-spacexr")
      (version (git-version "2.2.1" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/dmcable/spacexr")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0hqrllzq2hi6dsr8nykn1yp7s3rcbjrv3m624b61n4nxnzza458a"))))
      (properties `((upstream-name . "spacexr")))
      (build-system r-build-system)
      (propagated-inputs (list r-compquadform
                               r-data-table
                               r-doparallel
                               r-dplyr
                               r-fields
                               r-foreach
                               r-ggplot2
                               r-knitr
                               r-locfdr
                               r-matrix
                               r-metafor
                               r-mgcv
                               r-pals
                               r-quadprog
                               r-readr
                               r-reshape2
                               r-rfast
                               r-rmarkdown
                               r-tibble))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/dmcable/spacexr")
      (synopsis
       "Cell type identification and differential expression in spatial transcriptomics")
      (description
       "This package is used for cell type identification in spatial transcriptomics.
It also handles cell type-specific differential expression.")
      (license license:gpl3))))

(define-public r-spectre
  (let ((commit "ff5314fd5b3f29121e52f948d91e222a19dd4d25")
        (revision "1"))
    (package
      (name "r-spectre")
      (version "1.2.0")
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/ImmuneDynamics/Spectre")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32
           "0hziqlaq6mnkxi4bwp5xilxy90yh1vz1j1n117d38vmchvclx9yq"))))
      (properties `((upstream-name . "Spectre")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-biobase
             r-biocmanager
             r-caret
             r-class
             r-colorramps
             r-data-table
             r-dendsort
             r-devtools
             r-dplyr
             r-exactextractr
             r-factoextra
             r-flowcore
             r-flowsom
             r-flowviz
             r-fnn
             r-ggplot2
             r-ggpointdensity
             r-ggpubr
             r-ggraph
             r-ggthemes
             r-gridextra
             r-gridextra
             r-gtools
             r-hdf5array
             r-irlba
             r-patchwork
             r-pheatmap
             r-plyr
             r-qs
             r-raster
             r-rcolorbrewer
             r-rhdf5
             r-rstudioapi
             r-rsvd
             r-rtsne
             r-s2
             r-scales
             r-scattermore
             r-sf
             r-sp
             r-stars
             r-stringr
             r-tidygraph
             r-tidyr
             r-tidyr
             r-tiff
             r-umap
             r-uwot))
      (home-page "https://github.com/ImmuneDynamics/Spectre")
      (synopsis "High-dimensional cytometry and imaging analysis")
      (description
       "This package provides a computational toolkit in R for the
integration, exploration, and analysis of high-dimensional single-cell
cytometry and imaging data.")
      (license license:expat))))

(define-public r-compgenomrdata
  (let ((commit "24484cb77631e1123ead6c329b9d62c160e600c6")
        (revision "1"))
    (package
      (name "r-compgenomrdata")
      (version (git-version "0.1.0" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/compgenomr/compGenomRData")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "06gdvz4m4qlb1ylv10qfx09zv4c95cm7nps4y2s67m946kv8czv8"))))
      (properties `((upstream-name . "compGenomRData")))
      (build-system r-build-system)
      (home-page "https://github.com/compgenomr/compGenomRData")
      (synopsis "Data for Computational Genomics with R book")
      (description "This package provides data for the book \"Computational
Genomics with R\".")
      (license license:gpl3))))

(define-public r-cytonorm
  (let ((commit "166f9ff3d692278241018c2846cb4f86ab16065b")
        (revision "1"))
    (package
      (name "r-cytonorm")
      (version (git-version "0.0.10" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/saeyslab/CytoNorm")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "0vbqy3b26j1zqmwgqx59kkn9pm2m0qwlcppskvahrxcwrj6m1s5y"))))
      (properties `((upstream-name . "CytoNorm")))
      (build-system r-build-system)
      (propagated-inputs
       (list r-cytoml
             r-dplyr
             r-emdist
             r-flowcore
             r-flowsom
             r-flowworkspace
             r-ggplot2
             r-gridextra
             r-pheatmap
             r-stringr))
      (home-page "https://github.com/saeyslab/CytoNorm")
      (synopsis "Normalize cytometry data measured across multiple batches")
      (description
       "This package can be used to normalize cytometry samples when a control
sample is taken along in each of the batches.  This is done by first
identifying multiple clusters/cell types, learning the batch effects from the
control samples and applying quantile normalization on all markers of
interest.")
      (license license:gpl2+))))

(define-public r-kbet
  (let ((commit "f35171dfb04c7951b8a09ac778faf7424c4b6bc0")
        (revision "1"))
    (package
      (name "r-kbet")
      (version (git-version "0.99.6" revision commit))
      (source (origin
                (method git-fetch)
                (uri (git-reference
                      (url "https://github.com/theislab/kBET")
                      (commit commit)))
                (file-name (git-file-name name version))
                (sha256
                 (base32
                  "1r91prl2kki3zk694vhlmxdlqh0ixlhs8jfcqw6wc7cdsa0nv67k"))))
      (properties `((upstream-name . "kBET")))
      (build-system r-build-system)
      (propagated-inputs (list r-cluster r-fnn r-ggplot2 r-mass r-rcolorbrewer))
      (native-inputs (list r-knitr))
      (home-page "https://github.com/theislab/kBET")
      (synopsis "k-nearest neighbour batch effect test")
      (description
       "This tool detects batch effects in high-dimensional data based on chi^2-test.")
      ;; Any version of the GPL
      (license license:gpl3+))))

(define-public python-vireosnp
  (let ((commit "e3654633f7663732572c03c5dcf9fb00ec43b653")
        (revision "0"))
    (package
      (name "python-vireosnp")
      (version (git-version "0.5.9" revision commit))
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
                (url "https://github.com/huangyh09/vireoSNP")
                (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1wd4llm54fvc1pc4nqfdc43g637gfx1f4z4aznvdr3biy9jksqza"))))
      (build-system pyproject-build-system)
      (arguments
       (list
        #:phases
        #~(modify-phases %standard-phases
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (if tests?
                    (with-directory-excursion "examples"
                      (invoke "bash" "demo.sh"))
                    (format #f "test suite not run.~%")))))))
      (propagated-inputs (list python-matplotlib python-numpy python-scipy))
      (native-inputs (list python-setuptools))
      (home-page "https://github.com/huangyh09/vireoSNP")
      (synopsis "Deconvolution based on SNP for multiplexed scRNA-seq data")
      (description
       "This package provides a deconvolution based on Single Nucleotide
Position (SNP) for multiplexed scRNA-seq data.  The name vireo stand for
Variational Inference for Reconstructing Ensemble Origin by expressed SNPs in
multiplexed scRNA-seq data and follows the clone identification from
single-cell data named @url{https://github.com/PMBio/cardelino, cardelino}.")
      (license license:asl2.0))))

(define-public ccwl
  (package
    (name "ccwl")
    (version "0.4.0")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "https://ccwl.systemreboot.net/releases/ccwl-"
                           version ".tar.lz"))
       (sha256
        (base32
         "1vnkj34400kj77xavdr0cry0vm2fmk40vf7cwca61q3jl3h7hl2c"))))
    (build-system gnu-build-system)
    (arguments
     `(#:make-flags '("GUILE_AUTO_COMPILE=0") ; to prevent guild warnings
       #:modules (((guix build guile-build-system)
                   #:select (target-guile-effective-version))
                  ,@%default-gnu-modules)
       #:imported-modules ((guix build guile-build-system)
                           ,@%default-gnu-imported-modules)
       #:phases
       (modify-phases %standard-phases
         (add-after 'patch-source-shebangs 'patch-more-source-shebangs
           (lambda* (#:key inputs #:allow-other-keys)
             (substitute* "scripts/ccwl"
               (("^exec guile")
                (string-append "exec "
                               (search-input-file inputs "/bin/guile"))))))
         (add-after 'install 'wrap
           (lambda* (#:key inputs outputs #:allow-other-keys)
             (let ((out (assoc-ref outputs "out"))
                   (effective-version (target-guile-effective-version)))
               (wrap-program (string-append out "/bin/ccwl")
                 `("GUILE_LOAD_PATH" prefix
                   (,(string-append out "/share/guile/site/" effective-version)
          