#!/bin/bash
###############################################################################
#     parabola-vmbootstrap -- create and start parabola virtual machines      #
#                                                                             #
#     Copyright (C) 2017 - 2019  Andreas Grapentin                            #
#     Copyright (C) 2019 - 2020  bill-auger                                   #
#                                                                             #
#     This program is free software: you can redistribute it and/or modify    #
#     it under the terms of the GNU General Public License as published by    #
#     the Free Software Foundation, either version 3 of the License, or       #
#     (at your option) any later version.                                     #
#                                                                             #
#     This program is distributed in the hope that it will be useful,         #
#     but WITHOUT ANY WARRANTY; without even the implied warranty of          #
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
#     GNU General Public License for more details.                            #
#                                                                             #
#     You should have received a copy of the GNU General Public License       #
#     along with this program.  If not, see <http://www.gnu.org/licenses/>.   #
###############################################################################


usage()
{
  print "USAGE:"
  print "  pvm2tarball [-h] [-o <FILENAME>] <IMG>"
  echo
  prose "Produce a parabola release tarball from <IMG>."
  echo
  prose "<IMG> is expected to be a valid parabola image, ideally freshly bootstrapped
         using pvmbootstrap. If <FILENAME> is not specifed, generate an archive name
         from <IMG> and place it in the current working directory"
  echo
  echo  "Supported options:"
  echo  "  -o <FILENAME>  Write the generated tar archive to FILE instead of"
  echo  "                   generating a name for the archive from IMG"
  echo  "  -h             Display this help and exit"
  echo
  echo  "This script is part of parabola-vmbootstrap. source code available at:"
  echo  "  <https://git.parabola.nu/parabola-vmbootstrap.git>"
}

main()
{
  pvm_check_unprivileged # exits on failure

  # parse options
  local outfile
  while getopts 'ho:' arg; do
    case "$arg" in
      h) usage; return "$EXIT_SUCCESS";;
      o) outfile="$OPTARG";;
      *) error "invalid argument: %s\n" "$arg"; usage >&2; exit "$EXIT_INVALIDARGUMENT";;
    esac
  done
  local shiftlen=$(( OPTIND - 1 ))
  shift $shiftlen
  if [ "$#" -ne 1 ]; then usage >&2; exit "$EXIT_INVALIDARGUMENT"; fi

  local image_filename imagefile="$1"
  image_filename="$(basename "$imagefile")"
  shift

  # determine output file
  [ -n "$outfile" ] || outfile="$(dirname $imagefile)/${image_filename%.img*}.tar.gz"

  # ensure that the image file exists, prompt to clobber existing output file
  pvm_check_file_exists_and_writable "$imagefile" || exit "$EXIT_FAILURE"
  pvm_prompt_clobber_file            "$outfile"   || exit "$EXIT_FAILURE"

  # mount the root filesystem
  local bootdir workdir loopdev
  pvm_mount || exit "$EXIT_FAILURE" # assumes: $imagefile , sets: $loopdev $bootdir $workdir

  # tar the root filesystem, excluding unneeded things
  # HACKING:
  #  to update the exclude list, one can download the latest archlinuxarm
  #  release tarball, and scroll over the diff generated by running both the
  #  archlinuxarm and the generated parabola tarball through:
  #
  # `tar -tf <tarball> | sort`
  msg "imploding tarball"
  sudo tar -c -f "$outfile" -C "$workdir" -X - . << EOF
./boot/lost+found
./etc/.updated
./etc/pacman.d/gnupg
./lost+found
./root/.bash_history
./var/.updated
./var/log/journal/*
./var/log/pacman.log
./var/log/tallylog
EOF

  # give the archive back to the user
  sudo chown "$(id -u)":"$(id -g)" "$outfile"

  # cleanup
  pvm_cleanup
}


if   source "$(dirname "$(readlink -f "${BASH_SOURCE[0]}")")"/pvm-common.sh.inc 2> /dev/null || \
     source /usr/lib/parabola-vmbootstrap/pvm-common.sh.inc                     2> /dev/null
then main "$@"
else echo "can not find pvm-common.sh.inc" && exit 1
fi
