/* pubkey.c  -	pubkey dispatcher
 * Copyright (C) 1998,1999,2000,2002,2003 Free Software Foundation, Inc.
 *
 * This file is part of Libgcrypt.
 *
 * Libgcrypt is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser general Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * Libgcrypt is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>

#include "g10lib.h"
#include "mpi.h"
#include "cipher.h"
#include "ath.h"

static gpg_err_code_t pubkey_decrypt (int algo, gcry_mpi_t *result, gcry_mpi_t *data, gcry_mpi_t *skey, int flags);
static gpg_err_code_t pubkey_sign (int algo, gcry_mpi_t *resarr, gcry_mpi_t hash, gcry_mpi_t *skey);
static gpg_err_code_t pubkey_verify (int algo, gcry_mpi_t hash, gcry_mpi_t *data, gcry_mpi_t *pkey,
				     int (*cmp) (void *, gcry_mpi_t), void *opaque);

/* This is the list of the default public-key ciphers included in
   libgcrypt.  */
static struct
{
  gcry_pubkey_spec_t *pubkey;
} pubkey_table[] =
  {
#if USE_RSA
    { &pubkey_spec_rsa },
#endif
#if USE_ELGAMAL
    { &pubkey_spec_elg },
#endif
#if USE_DSA
    { &pubkey_spec_dsa },
#endif
    { NULL },
  };

/* List of registered ciphers.  */
static gcry_module_t *pubkeys_registered;

/* This is the lock protecting PUBKEYS_REGISTERED.  */
static ath_mutex_t pubkeys_registered_lock;

/* Flag to check wether the default pubkeys have already been
   registered.  */
static int default_pubkeys_registered;

/* Convenient macro for registering the default digests.  */
#define REGISTER_DEFAULT_PUBKEYS                   \
  do                                               \
    {                                              \
      ath_mutex_lock (&pubkeys_registered_lock);   \
      if (! default_pubkeys_registered)            \
        {                                          \
          gcry_pubkey_register_default ();         \
          default_pubkeys_registered = 1;          \
        }                                          \
      ath_mutex_unlock (&pubkeys_registered_lock); \
    }                                              \
  while (0)

/* These dummy functions are used in case a cipher implementation
   refuses to provide it's own functions.  */

static gpg_err_code_t
dummy_generate (int id, unsigned int nbits, unsigned long dummy,
                gcry_mpi_t *skey, gcry_mpi_t **retfactors)
{
  log_bug ("no generate() for %d\n", id);
  return GPG_ERR_PUBKEY_ALGO;
}

static gpg_err_code_t
dummy_check_secret_key (int id, gcry_mpi_t *skey)
{
  log_bug ("no check_secret_key() for %d\n", id);
  return GPG_ERR_PUBKEY_ALGO;
}

static gpg_err_code_t
dummy_encrypt (int id, gcry_mpi_t *resarr, gcry_mpi_t data, gcry_mpi_t *pkey, int flags)
{
  log_bug ("no encrypt() for %d\n", id);
  return GPG_ERR_PUBKEY_ALGO;
}

static gpg_err_code_t
dummy_decrypt (int id, gcry_mpi_t *result, gcry_mpi_t *data, gcry_mpi_t *skey, int flags)
{
  log_bug ("no decrypt() for %d\n", id);
  return GPG_ERR_PUBKEY_ALGO;
}

static gpg_err_code_t
dummy_sign (int id, gcry_mpi_t *resarr, gcry_mpi_t data, gcry_mpi_t *skey)
{
  log_bug ("no sign() for %d\n", id);
  return GPG_ERR_PUBKEY_ALGO;
}

static gpg_err_code_t
dummy_verify (int id, gcry_mpi_t hash, gcry_mpi_t *data, gcry_mpi_t *pkey,
	      int (*cmp) (void *, gcry_mpi_t), void *opaquev)
{
  log_bug ("no verify() for %d\n", id);
  return GPG_ERR_PUBKEY_ALGO;
}

static unsigned
dummy_get_nbits (int id, gcry_mpi_t *pkey)
{
  log_bug ("no get_nbits() for %d\n", id);
  return 0;
}

/* Internal function.  Register all the pubkeys included in
   PUBKEY_TABLE.  Returns zero on success or an error code.  */
static void
gcry_pubkey_register_default (void)
{
  gpg_err_code_t err = 0;
  int i;
  
  for (i = 0; (! err) && pubkey_table[i].pubkey; i++)
    {
#define pubkey_use_dummy(func)                       \
      if (! pubkey_table[i].pubkey->func)            \
	pubkey_table[i].pubkey->func = dummy_##func;

      pubkey_use_dummy (generate);
      pubkey_use_dummy (check_secret_key);
      pubkey_use_dummy (encrypt);
      pubkey_use_dummy (decrypt);
      pubkey_use_dummy (sign);
      pubkey_use_dummy (verify);
      pubkey_use_dummy (get_nbits);

      err = _gcry_module_add (&pubkeys_registered,
			      pubkey_table[i].pubkey->id,
			      (void *) pubkey_table[i].pubkey, NULL);
    }

  if (err)
    BUG ();
}

/* Internal callback function.  Used via _gcry_module_lookup.  */
static int
gcry_pubkey_lookup_func_name (void *spec, void *data)
{
  gcry_pubkey_spec_t *pubkey = (gcry_pubkey_spec_t *) spec;
  char *name = (char *) data;
  char **sexp_names = pubkey->sexp_names;
  int ret = stricmp (name, pubkey->name);

  while (ret && *sexp_names)
    ret = stricmp (name, *sexp_names++);

  return ! ret;
}

/* Internal function.  Lookup a pubkey entry by it's name.  */
static gcry_module_t *
gcry_pubkey_lookup_name (const char *name)
{
  gcry_module_t *pubkey;

  pubkey = _gcry_module_lookup (pubkeys_registered, (void *) name,
				gcry_pubkey_lookup_func_name);

  return pubkey;
}

/* Public function.  Register a provided PUBKEY.  Returns zero on
   success, in which case the chosen pubkey ID has been stored in
   PUBKEY, or an error code.  */
gpg_error_t
gcry_pubkey_register (gcry_pubkey_spec_t *pubkey, gcry_module_t **module)
{
  gpg_err_code_t err = GPG_ERR_NO_ERROR;
  gcry_module_t *mod;

  ath_mutex_lock (&pubkeys_registered_lock);
  err = _gcry_module_add (&pubkeys_registered, 0,
			  (void *) pubkey, &mod);
  ath_mutex_unlock (&pubkeys_registered_lock);

  if (! err)
    {
      *module = mod;
      pubkey->id = mod->id;
    }

  return err;
}

/* Public function.  Unregister the pubkey identified by ID, which
   must have been registered with gcry_pubkey_register.  */
void
gcry_pubkey_unregister (gcry_module_t *module)
{
  ath_mutex_lock (&pubkeys_registered_lock);
  _gcry_module_release (module);
  ath_mutex_unlock (&pubkeys_registered_lock);
}

static void
release_mpi_array (gcry_mpi_t *array)
{
  for (; *array; array++)
    {
      mpi_free(*array);
      *array = NULL;
    }
}

/****************
 * Map a string to the pubkey algo
 */
int
gcry_pk_map_name (const char *string)
{
  gcry_module_t *pubkey;
  int id = 0;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = gcry_pubkey_lookup_name (string);
  if (pubkey)
    {
      id = ((gcry_pubkey_spec_t *) pubkey->spec)->id;
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return id;
}


/****************
 * Map a pubkey algo to a string
 */
const char *
gcry_pk_algo_name (int id)
{
  const char *name = NULL;
  gcry_module_t *pubkey;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      name = ((gcry_pubkey_spec_t *) pubkey->spec)->name;
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return name;
}


static void
disable_pubkey_algo (int id)
{
  gcry_module_t *pubkey;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      if (! (pubkey-> flags & FLAG_MODULE_DISABLED))
	pubkey->flags |= FLAG_MODULE_DISABLED;
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);
}


/****************
 * a use of 0 means: don't care
 */
static gpg_err_code_t
check_pubkey_algo (int id, unsigned use)
{
  gpg_err_code_t err = GPG_ERR_NO_ERROR;
  gcry_pubkey_spec_t *pubkey;
  gcry_module_t *module;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  module = _gcry_module_lookup_id (pubkeys_registered, id);
  if (module)
    {
      pubkey = (gcry_pubkey_spec_t *) module->spec;

      if (((use & GCRY_PK_USAGE_SIGN)
	   && (! (pubkey->use & GCRY_PK_USAGE_SIGN)))
	  || ((use & GCRY_PK_USAGE_ENCR)
	      && (! (pubkey->use & GCRY_PK_USAGE_ENCR))))
	err = GPG_ERR_WRONG_PUBKEY_ALGO;
      else if (module->flags & FLAG_MODULE_DISABLED)
	err = GPG_ERR_PUBKEY_ALGO;
      _gcry_module_release (module);
    }
  else
    err = GPG_ERR_PUBKEY_ALGO;
  ath_mutex_unlock (&pubkeys_registered_lock);

  return err;
}


/****************
 * Return the number of public key material numbers
 */
static int
pubkey_get_npkey (int id)
{
  gcry_module_t *pubkey;
  int npkey = 0;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      npkey = strlen (((gcry_pubkey_spec_t *) pubkey->spec)->elements_pkey);
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return npkey;
}

/****************
 * Return the number of secret key material numbers
 */
static int
pubkey_get_nskey (int id)
{
  gcry_module_t *pubkey;
  int nskey = 0;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      nskey = strlen (((gcry_pubkey_spec_t *) pubkey->spec)->elements_skey);
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return nskey;
}

/****************
 * Return the number of signature material numbers
 */
static int
pubkey_get_nsig (int id)
{
  gcry_module_t *pubkey;
  int nsig = 0;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      nsig = strlen (((gcry_pubkey_spec_t *) pubkey->spec)->elements_sig);
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return nsig;
}

/****************
 * Return the number of encryption material numbers
 */
static int
pubkey_get_nenc (int id)
{
  gcry_module_t *pubkey;
  int nenc = 0;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      nenc = strlen (((gcry_pubkey_spec_t *) pubkey->spec)->elements_enc);
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return nenc;
}


static gpg_err_code_t
pubkey_generate (int id, unsigned int nbits, unsigned long use_e,
                 gcry_mpi_t *skey, gcry_mpi_t **retfactors)
{
  gpg_err_code_t err = GPG_ERR_PUBKEY_ALGO;
  gcry_module_t *pubkey;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      err = (*((gcry_pubkey_spec_t *) pubkey->spec)->generate) (id, nbits, use_e, skey,
							    retfactors);
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return err;
}

static gpg_err_code_t
pubkey_check_secret_key (int id, gcry_mpi_t *skey)
{
  gpg_err_code_t err = GPG_ERR_PUBKEY_ALGO;
  gcry_module_t *pubkey;

  REGISTER_DEFAULT_PUBKEYS;

  ath_mutex_lock (&pubkeys_registered_lock);
  pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
  if (pubkey)
    {
      err = (*((gcry_pubkey_spec_t *) pubkey->spec)->check_secret_key) (id, skey);
      _gcry_module_release (pubkey);
    }
  ath_mutex_unlock (&pubkeys_registered_lock);

  return err;
}


/****************
 * This is the interface to the public key encryption.  Encrypt DATA
 * with PKEY and put it into RESARR which should be an array of MPIs
 * of size PUBKEY_MAX_NENC (or less if the algorithm allows this -
 * check with pubkey_get_nenc() )
 */
static gpg_err_code_t
pubkey_encrypt (int id, gcry_mpi_t *resarr, gcry_mpi_t data, gcry_mpi_t *pkey,
		int flags)
{
  gcry_pubkey_spec_t *pubkey;
  gcry_module_t *module;
  gpg_err_code_t rc;
  int i;

  if (DBG_CIPHER)
    {
      log_debug ("pubkey_encrypt: algo=%d\n", id);
      for(i = 0; i < pubkey_get_npkey (id); i++)
	log_mpidump ("  pkey:", pkey[i]);
      log_mpidump ("  data:", data);
    }

  ath_mutex_lock (&pubkeys_registered_lock);
  module = _gcry_module_lookup_id (pubkeys_registered, id);
  if (module)
    {
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      rc = (*pubkey->encrypt) (id, resarr, data, pkey, flags);
      _gcry_module_release (module);
      goto ready;
    }
  rc = GPG_ERR_PUBKEY_ALGO;

 ready:
  ath_mutex_unlock (&pubkeys_registered_lock);

  if (!rc && DBG_CIPHER)
    {
      for(i = 0; i < pubkey_get_nenc (id); i++)
	log_mpidump("  encr:", resarr[i] );
    }
  return rc;
}


/****************
 * This is the interface to the public key decryption.
 * ALGO gives the algorithm to use and this implicitly determines
 * the size of the arrays.
 * result is a pointer to a mpi variable which will receive a
 * newly allocated mpi or NULL in case of an error.
 */
static gpg_err_code_t
pubkey_decrypt (int id, gcry_mpi_t *result, gcry_mpi_t *data, gcry_mpi_t *skey,
		int flags)
{
  gcry_pubkey_spec_t *pubkey;
  gcry_module_t *module;
  gpg_err_code_t rc;
  int i;

  *result = NULL; /* so the caller can always do a mpi_free */
  if (DBG_CIPHER)
    {
      log_debug ("pubkey_decrypt: algo=%d\n", id);
      for(i = 0; i < pubkey_get_nskey (id); i++)
	log_mpidump ("  skey:", skey[i]);
      for(i = 0; i < pubkey_get_nenc (id); i++)
	log_mpidump ("  data:", data[i]);
    }

  ath_mutex_lock (&pubkeys_registered_lock);
  module = _gcry_module_lookup_id (pubkeys_registered, id);
  if (module)
    {
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      rc = (*pubkey->decrypt) (id, result, data, skey, flags);
      _gcry_module_release (module);
      goto ready;
    }

  rc = GPG_ERR_PUBKEY_ALGO;
  
 ready:
  ath_mutex_unlock (&pubkeys_registered_lock);

  if (! rc && DBG_CIPHER)
    log_mpidump (" plain:", *result);

  return rc;
}


/****************
 * This is the interface to the public key signing.
 * Sign data with skey and put the result into resarr which
 * should be an array of MPIs of size PUBKEY_MAX_NSIG (or less if the
 * algorithm allows this - check with pubkey_get_nsig() )
 */
static gpg_err_code_t
pubkey_sign (int id, gcry_mpi_t *resarr, gcry_mpi_t data, gcry_mpi_t *skey)
{
  gcry_pubkey_spec_t *pubkey;
  gcry_module_t *module;
  gpg_err_code_t rc;
  int i;

  if (DBG_CIPHER)
    {
      log_debug ("pubkey_sign: algo=%d\n", id);
      for(i = 0; i < pubkey_get_nskey (id); i++)
	log_mpidump ("  skey:", skey[i]);
      log_mpidump("  data:", data );
    }

  ath_mutex_lock (&pubkeys_registered_lock);
  module = _gcry_module_lookup_id (pubkeys_registered, id);
  if (module)
    {
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      rc = (*pubkey->sign) (id, resarr, data, skey);
      _gcry_module_release (module);
      goto ready;
    }

  rc = GPG_ERR_PUBKEY_ALGO;

 ready:
  ath_mutex_unlock (&pubkeys_registered_lock);

  if (! rc && DBG_CIPHER)
    for (i = 0; i < pubkey_get_nsig (id); i++)
      log_mpidump ("   sig:", resarr[i]);

  return rc;
}

/****************
 * Verify a public key signature.
 * Return 0 if the signature is good
 */
static gpg_err_code_t
pubkey_verify (int id, gcry_mpi_t hash, gcry_mpi_t *data, gcry_mpi_t *pkey,
	       int (*cmp)(void *, gcry_mpi_t), void *opaquev)
{
  gcry_pubkey_spec_t *pubkey;
  gcry_module_t *module;
  gpg_err_code_t rc;
  int i;

  if (DBG_CIPHER)
    {
      log_debug ("pubkey_verify: algo=%d\n", id);
      for (i = 0; i < pubkey_get_npkey (id); i++)
	log_mpidump ("  pkey:", pkey[i]);
      for (i = 0; i < pubkey_get_nsig (id); i++)
	log_mpidump ("   sig:", data[i]);
      log_mpidump ("  hash:", hash);
    }

  ath_mutex_lock (&pubkeys_registered_lock);
  module = _gcry_module_lookup_id (pubkeys_registered, id);
  if (module)
    {
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      rc = (*pubkey->verify) (id, hash, data, pkey, cmp, opaquev);
      _gcry_module_release (module);
      goto ready;
    }

  rc = GPG_ERR_PUBKEY_ALGO;

 ready:
  ath_mutex_unlock (&pubkeys_registered_lock);
  return rc;
}

/* Internal function.   */
static gpg_err_code_t
sexp_elements_extract (gcry_sexp_t key_sexp, const char *element_names,
		       gcry_mpi_t *elements)
{
  gpg_err_code_t err = GPG_ERR_NO_ERROR;
  int i, index;
  const char *name;
  gcry_sexp_t list;

  for (name = element_names, index = 0; *name && (! err); name++, index++)
    {
      list = gcry_sexp_find_token (key_sexp, name, 1);
      if (! list)
	err = GPG_ERR_NO_OBJ;
      else
	{
	  elements[index] = gcry_sexp_nth_mpi (list, 1, GCRYMPI_FMT_USG);
	  gcry_sexp_release (list);
	  if (! elements[index])
	    err = GPG_ERR_INV_OBJ;
	}
    }

  if (err)
    for (i = 0; i < index; i++)
      if (elements[i])
	gcry_free (elements[i]);

  return err;
}

/****************
 * Convert a S-Exp with either a private or a public key to our
 * internal format. Currently we do only support the following
 * algorithms:
 *    dsa
 *    rsa
 *    openpgp-dsa
 *    openpgp-rsa
 *    openpgp-elg
 *    openpgp-elg-sig
 * Provide a SE with the first element be either "private-key" or
 * or "public-key". the followed by a list with its first element
 * be one of the above algorithm identifiers and the following
 * elements are pairs with parameter-id and value.
 * NOTE: we look through the list to find a list beginning with
 * "private-key" or "public-key" - the first one found is used.
 *
 * FIXME: Allow for encrypted secret keys here.
 *
 * Returns: A pointer to an allocated array of MPIs if the return value is
 *	    zero; the caller has to release this array.
 *
 * Example of a DSA public key:
 *  (private-key
 *    (dsa
 *	(p <mpi>)
 *	(g <mpi>)
 *	(y <mpi>)
 *	(x <mpi>)
 *    )
 *  )
 * The <mpi> are expected to be in GCRYMPI_FMT_USG
 */
static gpg_err_code_t
sexp_to_key (gcry_sexp_t sexp, int want_private, gcry_mpi_t **retarray,
             gcry_module_t **retalgo)
{
    gcry_sexp_t list, l2;
    const char *name;
    size_t n;
    int algo;
    const char *elems;
    gcry_mpi_t *array;
    gpg_err_code_t err = GPG_ERR_NO_ERROR;
    gcry_module_t *module;
    gcry_pubkey_spec_t *pubkey;

    /* check that the first element is valid */
    list = gcry_sexp_find_token( sexp, want_private? "private-key"
						    :"public-key", 0 );
    if( !list )
	return GPG_ERR_INV_OBJ; /* Does not contain a public- or private-key object */
    l2 = gcry_sexp_cadr( list );
    gcry_sexp_release ( list );
    list = l2;
    name = gcry_sexp_nth_data( list, 0, &n );
    if( !name ) {
	gcry_sexp_release ( list );
	return GPG_ERR_INV_OBJ; /* invalid structure of object */
    }

    {
      char *name_terminated = gcry_xmalloc (n + 1);
      strncpy (name_terminated, name, n);
      name_terminated[n] = 0;

      ath_mutex_lock (&pubkeys_registered_lock);
      module = gcry_pubkey_lookup_name (name_terminated);
      ath_mutex_unlock (&pubkeys_registered_lock);

      free (name_terminated);
    }

    if (! module)
      {
	gcry_sexp_release (list);
	return GPG_ERR_PUBKEY_ALGO; /* unknown algorithm */
      }
    else
      pubkey = (gcry_pubkey_spec_t *) module->spec;

    algo = pubkey->id;
    elems = want_private ? pubkey->elements_skey : pubkey->elements_pkey;
    array = gcry_calloc (strlen (elems) + 1, sizeof (*array));
    if (! array)
      err = gpg_err_code_from_errno (errno);
    if (! err)
      err = sexp_elements_extract (list, elems, array);

    if (list)
      gcry_sexp_release (list);

    if (err)
      {
	if (array)
	  gcry_free (array);

	ath_mutex_lock (&pubkeys_registered_lock);
	_gcry_module_release (module);
	ath_mutex_unlock (&pubkeys_registered_lock);
      }
    else
      {
	*retarray = array;
	*retalgo = module;
      }

    return err;
}

static gpg_err_code_t
sexp_to_sig (gcry_sexp_t sexp, gcry_mpi_t **retarray,
	     gcry_module_t **retalgo)
{
    gcry_sexp_t list, l2;
    const char *name;
    size_t n;
    int algo;
    const char *elems;
    gcry_mpi_t *array;
    gpg_err_code_t err = GPG_ERR_NO_ERROR;
    gcry_module_t *module;
    gcry_pubkey_spec_t *pubkey;

    /* check that the first element is valid */
    list = gcry_sexp_find_token( sexp, "sig-val" , 0 );
    if( !list )
	return GPG_ERR_INV_OBJ; /* Does not contain a signature value object */
    l2 = gcry_sexp_cadr( list );
    gcry_sexp_release ( list );
    list = l2;
    if( !list )
	return GPG_ERR_NO_OBJ; /* no cadr for the sig object */
    name = gcry_sexp_nth_data( list, 0, &n );
    if( !name ) {
	gcry_sexp_release ( list );
	return GPG_ERR_INV_OBJ; /* invalid structure of object */
    }

    {
      char *name_terminated = gcry_xmalloc (n + 1);
      strncpy (name_terminated, name, n);
      name_terminated[n] = 0;
      
      ath_mutex_lock (&pubkeys_registered_lock);
      module = gcry_pubkey_lookup_name (name_terminated);
      ath_mutex_unlock (&pubkeys_registered_lock);

      free (name_terminated);
    }

    if (! module)
      {
	gcry_sexp_release (list);
	return GPG_ERR_PUBKEY_ALGO; /* unknown algorithm */
      }
    else
      pubkey = (gcry_pubkey_spec_t *) module->spec;

    algo = pubkey->id;
    elems = pubkey->elements_sig;
    array = gcry_calloc (strlen (elems) + 1 , sizeof (*array));
    if (! array)
      err = gpg_err_code_from_errno (errno);

    if (! err)
      err = sexp_elements_extract (list, elems, array);

    if (list)
      gcry_sexp_release (list);

    if (err)
      {
	ath_mutex_lock (&pubkeys_registered_lock);
	_gcry_module_release (module);
	ath_mutex_unlock (&pubkeys_registered_lock);

	if (array)
	  gcry_free (array);
      }
    else
      {
	*retarray = array;
	*retalgo = module;
      }

    return err;
}


/****************
 * Take sexp and return an array of MPI as used for our internal decrypt
 * function.
 * s_data = (enc-val
 *           [(flags [pkcs1])
 *	      (<algo>
 *		(<param_name1> <mpi>)
 *		...
 *		(<param_namen> <mpi>)
 *	      ))
 * RET_MODERN is set to true when at least an empty flags list has been found.
 */
static gpg_err_code_t
sexp_to_enc (gcry_sexp_t sexp, gcry_mpi_t **retarray, gcry_module_t **retalgo,
             int *ret_modern, int *ret_want_pkcs1, int *flags)
{
  gcry_sexp_t list = NULL, l2 = NULL;
  gcry_pubkey_spec_t *pubkey = NULL;
  gcry_module_t *module = NULL;
  const char *name;
  size_t n;
  int parsed_flags = 0;
  const char *elems;
  gcry_mpi_t *array = NULL;
  gpg_err_code_t err = GPG_ERR_NO_ERROR;

  *ret_want_pkcs1 = 0;
  *ret_modern = 0;

  /* check that the first element is valid */
  list = gcry_sexp_find_token (sexp, "enc-val" , 0);
  if (! list)
    err = GPG_ERR_INV_OBJ; /* Does not contain an encrypted value object */

  if (! err)
    {
      l2 = gcry_sexp_nth (list, 1);
      if (! l2)
	err = GPG_ERR_NO_OBJ; /* no cdr for the data object */
    }

  if (! err)
    {
      /* Extract the name of the algorithm.  */
      name = gcry_sexp_nth_data (l2, 0, &n);
      if (! name)
	err = GPG_ERR_INV_OBJ; /* invalid structure of object */
    }

  if ((! err) && (n == 5) && (! memcmp (name, "flags", 5)))
    {
      /* There is a flags element - process it */
      const char *s;
      int i;

      *ret_modern = 1;
      for (i = gcry_sexp_length (l2) - 1; i > 0 && (! err); i--)
        {
          s = gcry_sexp_nth_data (l2, i, &n);
          if (! s)
            ; /* not a data element - ignore */
          else if (n == 3 && ! memcmp (s, "raw", 3))
            ; /* just a dummy because it is the default */
          else if (n == 5 && ! memcmp (s, "pkcs1", 5))
            *ret_want_pkcs1 = 1;
	  else if (n == 11 && ! memcmp (s, "no-blinding", 11))
	    parsed_flags |= PUBKEY_FLAG_NO_BLINDING;
          else
	    err = GPG_ERR_INV_FLAG;
	}
    }

  if (! err)
    {
      /* Get the next which has the actual data */
      gcry_sexp_release (l2);
      l2 = gcry_sexp_nth (list, 2);
      if (! l2)
	err = GPG_ERR_NO_OBJ; /* no cdr for the data object */
    }

  if (! err)
    {
      name = gcry_sexp_nth_data (l2, 0, &n);
      if (! name)
	err = GPG_ERR_INV_OBJ; /* invalid structure of object */
      else
	{
	  gcry_sexp_release (list);
	  list = l2;
	  l2 = NULL;
	}
    }

  if (! err)
    {
      char *name_terminated = gcry_xmalloc (n + 1);
      strncpy (name_terminated, name, n);
      name_terminated[n] = 0;

      ath_mutex_lock (&pubkeys_registered_lock);
      module = gcry_pubkey_lookup_name (name_terminated);
      ath_mutex_unlock (&pubkeys_registered_lock);

      free (name_terminated);

      if (! module)
	err = GPG_ERR_PUBKEY_ALGO; /* unknown algorithm */
      else
	pubkey = (gcry_pubkey_spec_t *) module->spec;
    }

  if (! err)
    {
      elems = pubkey->elements_enc;
      array = gcry_calloc (strlen (elems) + 1, sizeof (*array));
      if (! array)
	err = gpg_err_code_from_errno (errno);
    }

  if (! err)
    err = sexp_elements_extract (list, elems, array);

  if (list)
    gcry_sexp_release (list);
  if (l2)
    gcry_sexp_release (l2);

  if (err)
    {
      ath_mutex_lock (&pubkeys_registered_lock);
      _gcry_module_release (module);
      ath_mutex_unlock (&pubkeys_registered_lock);
      if (array)
	gcry_free (array);
    }
  else
    {
      *retarray = array;
      *retalgo = module;
      *flags = parsed_flags;
    }

  return err;
}

/* Take the hash value and convert into an MPI, suitable for for
   passing to the low level functions.  We currently support the
   old style way of passing just a MPI and the modern interface which
   allows to pass flags so that we can choose between raw and pkcs1
   padding - may be more padding options later. 

   (<mpi>)
   or
   (data
    [(flags [pkcs1])]
    [(hash <algo> <value>)]
    [(value <text>)]
   )
   
   Either the VALUE or the HASH element must be present for use
   with signatures.  VALUE is used for encryption.

   NBITS is the length of the key in bits. 

*/
static gpg_err_code_t
sexp_data_to_mpi (gcry_sexp_t input, unsigned int nbits, gcry_mpi_t *ret_mpi,
                  int for_encryption, int *flags)
{
  gpg_err_code_t rc = 0;
  gcry_sexp_t ldata, lhash, lvalue;
  int i;
  size_t n;
  const char *s;
  int is_raw = 0, is_pkcs1 = 0, unknown_flag=0; 
  int parsed_flags = 0, dummy_flags;

  if (! flags)
    flags = &dummy_flags;
  
  *ret_mpi = NULL;
  ldata = gcry_sexp_find_token (input, "data", 0);
  if (!ldata)
    { /* assume old style */
      *ret_mpi = gcry_sexp_nth_mpi (input, 0, 0);
      return *ret_mpi ? GPG_ERR_NO_ERROR : GPG_ERR_INV_OBJ;
    }

  /* see whether there is a flags object */
  {
    gcry_sexp_t lflags = gcry_sexp_find_token (ldata, "flags", 0);
    if (lflags)
      { /* parse the flags list. */
        for (i=gcry_sexp_length (lflags)-1; i > 0; i--)
          {
            s = gcry_sexp_nth_data (lflags, i, &n);
            if (!s)
              ; /* not a data element*/
            else if ( n == 3 && !memcmp (s, "raw", 3))
              is_raw = 1;
            else if ( n == 5 && !memcmp (s, "pkcs1", 5))
              is_pkcs1 = 1;
	    else if (n == 11 && ! memcmp (s, "no-blinding", 11))
	      parsed_flags |= PUBKEY_FLAG_NO_BLINDING;
            else
              unknown_flag = 1;
          }
        gcry_sexp_release (lflags);
      }
  }

  if (!is_pkcs1 && !is_raw)
    is_raw = 1; /* default to raw */

  /* Get HASH or MPI */
  lhash = gcry_sexp_find_token (ldata, "hash", 0);
  lvalue = lhash? NULL : gcry_sexp_find_token (ldata, "value", 0);

  if (!(!lhash ^ !lvalue))
    rc = GPG_ERR_INV_OBJ; /* none or both given */
  else if (unknown_flag)
    rc = GPG_ERR_INV_FLAG;
  else if (is_raw && is_pkcs1 && !for_encryption)
    rc = GPG_ERR_CONFLICT;
  else if (is_raw && lvalue)
    {
      *ret_mpi = gcry_sexp_nth_mpi (lvalue, 1, 0);
      if (!*ret_mpi)
        rc = GPG_ERR_INV_OBJ;
    }
  else if (is_pkcs1 && lvalue && for_encryption)
    { /* create pkcs#1 block type 2 padding */
      unsigned char *frame = NULL;
      size_t nframe = (nbits+7) / 8;
      const void * value;
      size_t valuelen;
      unsigned char *p;

      if ( !(value=gcry_sexp_nth_data (lvalue, 1, &valuelen)) || !valuelen )
        rc = GPG_ERR_INV_OBJ;
      else if (valuelen + 7 > nframe || !nframe)
        {
          /* Can't encode a VALUELEN value in a NFRAME bytes frame. */
          rc = GPG_ERR_TOO_SHORT; /* the key is too short */
        }
      else if ( !(frame = gcry_malloc_secure (nframe)))
        rc = gpg_err_code_from_errno (errno);
      else
        {
          n = 0;
          frame[n++] = 0;
          frame[n++] = 2; /* block type */
          i = nframe - 3 - valuelen;
          assert (i > 0);
          p = gcry_random_bytes_secure (i, GCRY_STRONG_RANDOM);
          /* replace zero bytes by new values*/
          for (;;)
            {
              int j, k;
              unsigned char *pp;
              
              /* count the zero bytes */
              for (j=k=0; j < i; j++)
                {
                  if (!p[j])
                    k++;
                }
              if (!k)
                break; /* okay: no (more) zero bytes */
              
              k += k/128; /* better get some more */
              pp = gcry_random_bytes_secure (k, GCRY_STRONG_RANDOM);
              for (j=0; j < i && k; j++)
                {
                  if (!p[j])
                    p[j] = pp[--k];
                }
              gcry_free (pp);
            }
          memcpy (frame+n, p, i);
          n += i;
          gcry_free (p);
          
          frame[n++] = 0;
          memcpy (frame+n, value, valuelen);
          n += valuelen;
          assert (n == nframe);

	  /* FIXME, error checking?  */
          gcry_mpi_scan (ret_mpi, GCRYMPI_FMT_USG, frame, &nframe);
        }

      gcry_free(frame);
    }
  else if (is_pkcs1 && lhash && !for_encryption)
    { /* create pkcs#1 block type 1 padding */
      if (gcry_sexp_length (lhash) != 3)
        rc = GPG_ERR_INV_OBJ;
      else if ( !(s=gcry_sexp_nth_data (lhash, 1, &n)) || !n )
        rc = GPG_ERR_INV_OBJ;
      else
        {
          static struct { const char *name; int algo; } hashnames[] = 
          { { "sha1",   GCRY_MD_SHA1 },
            { "md5",    GCRY_MD_MD5 },
            { "rmd160", GCRY_MD_RMD160 },
            { "sha256", GCRY_MD_SHA256 },
            { "sha384", GCRY_MD_SHA384 },
            { "sha512", GCRY_MD_SHA512 },
            { "md2",    GCRY_MD_MD2 },
            { "md4",    GCRY_MD_MD4 },
            { "tiger",  GCRY_MD_TIGER },
            { "haval",  GCRY_MD_HAVAL },
            { NULL }
          };
          int algo;
          byte asn[100];
          byte *frame = NULL;
          size_t nframe = (nbits+7) / 8;
          const void * value;
          size_t valuelen;
          size_t asnlen, dlen;
            
          for (i=0; hashnames[i].name; i++)
            {
              if ( strlen (hashnames[i].name) == n
                   && !memcmp (hashnames[i].name, s, n))
                break;
            }

          algo = hashnames[i].algo;
          asnlen = DIM(asn);
          dlen = gcry_md_get_algo_dlen (algo);

          if (!hashnames[i].name)
            rc = GPG_ERR_DIGEST_ALGO;
          else if ( !(value=gcry_sexp_nth_data (lhash, 2, &valuelen))
                    || !valuelen )
            rc = GPG_ERR_INV_OBJ;
          else if (gcry_md_algo_info (algo, GCRYCTL_GET_ASNOID, asn, &asnlen))
            rc = GPG_ERR_NOT_IMPLEMENTED; /* we don't have all of the above algos */
          else if ( valuelen != dlen )
            {
              /* hash value does not match the length of digest for
                 the given algo */
              rc = GPG_ERR_CONFLICT;
            }
          else if( !dlen || dlen + asnlen + 4 > nframe)
            {
              /* can't encode an DLEN byte digest MD into a NFRAME byte frame */
              rc = GPG_ERR_TOO_SHORT;
            }
          else if ( !(frame = gcry_malloc (nframe)) )
            rc = gpg_err_code_from_errno (errno);
          else
            { /* assemble the pkcs#1 block type 1 */
              n = 0;
              frame[n++] = 0;
              frame[n++] = 1; /* block type */
              i = nframe - valuelen - asnlen - 3 ;
              assert (i > 1);
              memset (frame+n, 0xff, i );
              n += i;
              frame[n++] = 0;
              memcpy (frame+n, asn, asnlen);
              n += asnlen;
              memcpy (frame+n, value, valuelen );
              n += valuelen;
              assert (n == nframe);
      
              /* convert it into an MPI, FIXME: error checking?  */
              gcry_mpi_scan (ret_mpi, GCRYMPI_FMT_USG, frame, &nframe);
            }
          
          gcry_free (frame);
        }
    }
  else
    rc = GPG_ERR_CONFLICT;
   
  gcry_sexp_release (ldata);
  gcry_sexp_release (lhash);
  gcry_sexp_release (lvalue);

  if (! rc)
    *flags = parsed_flags;

  return rc;
}


/*
   Do a PK encrypt operation
  
   Caller has to provide a public key as the SEXP pkey and data as a
   SEXP with just one MPI in it. Alternativly S_DATA might be a
   complex S-Expression, similar to the one used for signature
   verification.  This provides a flag which allows to handle PKCS#1
   block type 2 padding.  The function returns a a sexp which may be
   passed to to pk_decrypt.
  
   Returns: 0 or an errorcode.
  
   s_data = See comment for sexp_data_to_mpi
   s_pkey = <key-as-defined-in-sexp_to_key>
   r_ciph = (enc-val
               (<algo>
                 (<param_name1> <mpi>)
                 ...
                 (<param_namen> <mpi>)
               ))

*/
gpg_error_t
gcry_pk_encrypt (gcry_sexp_t *r_ciph, gcry_sexp_t s_data, gcry_sexp_t s_pkey)
{
  gcry_mpi_t *pkey = NULL, data = NULL, *ciph = NULL;
  const char *algo_name, *algo_elems;
  int flags;
  gpg_err_code_t rc;
  gcry_pubkey_spec_t *pubkey = NULL;
  gcry_module_t *module = NULL;

  REGISTER_DEFAULT_PUBKEYS;

  *r_ciph = NULL;
  /* get the key */
  rc = sexp_to_key (s_pkey, 0, &pkey, &module);
  if (! rc)
    {
      assert (module);
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      algo_name = pubkey->name;
      algo_elems = pubkey->elements_enc;
      
      /* get the stuff we want to encrypt */
      rc = sexp_data_to_mpi (s_data, gcry_pk_get_nbits (s_pkey), &data, 1,
			     &flags);
    }

  if (! rc)
    {
      /* Now we can encrypt data to ciph */
      ciph = gcry_xcalloc (strlen (algo_elems) + 1, sizeof (*ciph));
      rc = pubkey_encrypt (pubkey->id, ciph, data, pkey, flags);
      mpi_free (data);
    }

  /* We did it.  Now build the return list */
  if (! rc)
    {
      char *string, *p;
      int i;
      size_t nelem = strlen (algo_elems);
      size_t needed = 19 + strlen (algo_name) + (nelem * 5);

      if (flags & PUBKEY_FLAG_NO_BLINDING)
	needed += 12;

      /* Build the string.  */
      string = p = gcry_xmalloc (needed);
      p = stpcpy ( p, "(enc-val(flags" );
      if (flags & PUBKEY_FLAG_NO_BLINDING)
	p = stpcpy (p, " no-blinding");
      p = stpcpy (p, ")(");
      p = stpcpy ( p, algo_name );
      for(i=0; algo_elems[i]; i++ ) {
	*p++ = '(';
	*p++ = algo_elems[i];
	p = stpcpy ( p, "%m)" );
      }
      strcpy ( p, "))" );

      /* and now the ugly part:  we don't have a function to
       * pass an array to a format string, so we have to do it this way :-(
       */

      {
	int i;
	void **arg_list = malloc (sizeof (void *) * nelem);
	if (arg_list)
	  {
	    for (i = 0; i < nelem; i++)
	      arg_list[i] = &ciph[i];

	    rc = gcry_sexp_build_array (r_ciph, NULL, string, arg_list);

	    free (arg_list);
	  }
	else
	  rc = gpg_err_code_from_errno (errno);
      }

      if (rc)
	BUG ();
      gcry_free (string);
    }

  if (ciph)
    {
      release_mpi_array (ciph);
      gcry_free (ciph);
    }

  if (module)
    {
      ath_mutex_lock (&pubkeys_registered_lock);
      _gcry_module_release (module);
      ath_mutex_unlock (&pubkeys_registered_lock);
    }

  return gpg_error (rc);
}

/****************
 * Do a PK decrypt operation
 *
 * Caller has to provide a secret key as the SEXP skey and data in a
 * format as created by gcry_pk_encrypt.  For historic reasons the
 * function returns simply an MPI as an S-expression part; this is
 * deprecated and the new method should be used which returns a real
 * S-expressionl this is selected by adding at least an empty flags
 * list to S_DATA.
 * 
 * Returns: 0 or an errorcode.
 *
 * s_data = (enc-val
 *            [(flags)]
 *	      (<algo>
 *		(<param_name1> <mpi>)
 *		...
 *		(<param_namen> <mpi>)
 *	      ))
 * s_skey = <key-as-defined-in-sexp_to_key>
 * r_plain= Either an incomplete S-expression without the parentheses
 *          or if the flags list is used (even if empty) a real S-expression:
 *          (value PLAIN).  */
gpg_error_t
gcry_pk_decrypt (gcry_sexp_t *r_plain, gcry_sexp_t s_data, gcry_sexp_t s_skey)
{
  gcry_mpi_t *skey = NULL, *data = NULL, plain = NULL;
  int modern, want_pkcs1, flags;
  gpg_err_code_t rc;
  gcry_module_t *module_enc = NULL, *module_key = NULL;
  gcry_pubkey_spec_t *pubkey = NULL;

  REGISTER_DEFAULT_PUBKEYS;

  *r_plain = NULL;
  rc = sexp_to_key (s_skey, 1, &skey, &module_key);

  if (! rc)
    rc = sexp_to_enc (s_data, &data, &module_enc, &modern, &want_pkcs1, &flags);

  if (! rc)
    {
      if (((gcry_pubkey_spec_t *) module_key->spec)->id
	  != ((gcry_pubkey_spec_t *) module_enc->spec)->id)
	rc = GPG_ERR_CONFLICT; /* key algo does not match data algo */
      else
	pubkey = (gcry_pubkey_spec_t *) module_key->spec;
    }

  if (! rc)
    rc = pubkey_decrypt (pubkey->id, &plain, data, skey, flags);

  if (! rc)
    {
      if (! modern)
	{
	  if (gcry_sexp_build (r_plain, NULL, "%m", plain))
	    BUG ();
	}
      else
	{
	  if (gcry_sexp_build (r_plain, NULL, "(value %m)", plain))
	    BUG ();
	}
    }

  if (skey)
    {
      release_mpi_array (skey);
      gcry_free (skey);
    }

  if (plain)
    mpi_free (plain);

  if (data)
    {
      release_mpi_array (data);
      gcry_free (data);
    }

  if (module_key || module_enc)
    {
      ath_mutex_lock (&pubkeys_registered_lock);
      if (module_key)
	_gcry_module_release (module_key);
      if (module_enc)
	_gcry_module_release (module_enc);
      ath_mutex_unlock (&pubkeys_registered_lock);
    }

  return gpg_error (rc);
}



/****************
 * Create a signature.
 *
 * Caller has to provide a secret key as the SEXP skey and data
 * expressed as a SEXP list hash with only one element which should
 * instantly be available as a MPI. Alternatively the structure given
 * below may be used for S_HASH, it provides the abiliy to pass flags
 * to the operation; the only flag defined by now is "pkcs1" which
 * does PKCS#1 block type 1 style padding.
 *
 * Returns: 0 or an errorcode.
 *	    In case of 0 the function returns a new SEXP with the
 *	    signature value; the structure of this signature depends on the
 *	    other arguments but is always suitable to be passed to
 *	    gcry_pk_verify
 *
 * s_hash = See comment for sexp_data_to_mpi
 *             
 * s_skey = <key-as-defined-in-sexp_to_key>
 * r_sig  = (sig-val
 *	      (<algo>
 *		(<param_name1> <mpi>)
 *		...
 *		(<param_namen> <mpi>)
 * )) */
gpg_error_t
gcry_pk_sign (gcry_sexp_t *r_sig, gcry_sexp_t s_hash, gcry_sexp_t s_skey)
{
  gcry_mpi_t *skey = NULL, hash = NULL, *result = NULL;
  gcry_pubkey_spec_t *pubkey = NULL;
  gcry_module_t *module = NULL;
  const char *key_algo_name, *algo_name, *algo_elems;
  int i;
  gpg_err_code_t rc;


  REGISTER_DEFAULT_PUBKEYS;

  *r_sig = NULL;
  rc = sexp_to_key (s_skey, 1, &skey, &module);

  if (! rc)
    {
      assert (module);
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      algo_name = key_algo_name = pubkey->name;

      algo_elems = pubkey->elements_sig;

      /* get the stuff we want to sign */
      /* Note that pk_get_nbits does also work on a private key */
      rc = sexp_data_to_mpi (s_hash, gcry_pk_get_nbits (s_skey), &hash, 0, NULL);
    }

  if (! rc)
    {
      result = gcry_xcalloc (strlen (algo_elems) + 1, sizeof (*result));
      rc = pubkey_sign (pubkey->id, result, hash, skey);
    }

  if (! rc)
    {
      char *string, *p;
      size_t nelem, needed = strlen (algo_name) + 20;

      nelem = strlen (algo_elems);

      /* count elements, so that we can allocate enough space */
      needed += 10 * nelem;

      /* build the string */
      string = p = gcry_xmalloc (needed);
      p = stpcpy (p, "(sig-val(");
      p = stpcpy (p, algo_name);
      for(i = 0; algo_elems[i]; i++)
	{
	  *p++ = '(';
	  *p++ = algo_elems[i];
	  p = stpcpy (p, "%m)");
	}
      strcpy (p, "))");

      {
	int i;
	void **arg_list = malloc (sizeof (void *) * nelem);
	if (arg_list)
	  {
	    for (i = 0; i < nelem; i++)
	      arg_list[i] = &result[i];

	    rc = gcry_sexp_build_array (r_sig, NULL, string, arg_list);

	    free (arg_list);
	  }
	else
	  rc = gpg_err_code_from_errno (errno);
      }

      if (rc)
	BUG ();
      gcry_free (string);
    }

  if (skey)
    {
      release_mpi_array (skey);
      gcry_free (skey);
    }

  if (hash)
    mpi_free (hash);

  if (result)
    gcry_free (result);

  return gpg_error (rc);
}


/****************
 * Verify a signature.  Caller has to supply the public key pkey, the
 * signature sig and his hashvalue data.  Public key has to be a
 * standard public key given as an S-Exp, sig is a S-Exp as returned
 * from gcry_pk_sign and data must be an S-Exp like the one in sign
 * too.
 */
gpg_error_t
gcry_pk_verify (gcry_sexp_t s_sig, gcry_sexp_t s_hash, gcry_sexp_t s_pkey)
{
  gcry_module_t *module_key = NULL, *module_sig = NULL;
  gcry_mpi_t *pkey = NULL, hash = NULL, *sig = NULL;
  gpg_err_code_t rc;

  REGISTER_DEFAULT_PUBKEYS;
 
  rc = sexp_to_key (s_pkey, 0, &pkey, &module_key);
  if (! rc)
    rc = sexp_to_sig (s_sig, &sig, &module_sig);

  if ((! rc)
      && (((gcry_pubkey_spec_t *) module_key->spec)->id
	  != ((gcry_pubkey_spec_t *) module_sig->spec)->id))
    rc = GPG_ERR_CONFLICT;

  if (! rc)
    rc = sexp_data_to_mpi (s_hash, gcry_pk_get_nbits (s_pkey), &hash, 0, 0);

  if (! rc)
    rc = pubkey_verify (((gcry_pubkey_spec_t *) module_key->spec)->id,
			hash, sig, pkey, NULL, NULL);

  if (pkey)
    {
      release_mpi_array (pkey);
      gcry_free (pkey);
    }
  if (sig)
    {
      release_mpi_array (sig);
      gcry_free (sig);
    }
  if (hash)
    mpi_free (hash);

  if (module_key || module_sig)
    {
      ath_mutex_lock (&pubkeys_registered_lock);
      if (module_key)
	_gcry_module_release (module_key);
      if (module_sig)
	_gcry_module_release (module_sig);
      ath_mutex_unlock (&pubkeys_registered_lock);
    }

  return gpg_error (rc);
}


/****************
 * Test a key.	This may be used either for a public or a secret key
 * to see whether internal structre is valid.
 *
 * Returns: 0 or an errorcode.
 *
 * s_key = <key-as-defined-in-sexp_to_key>
 */
gpg_error_t
gcry_pk_testkey (gcry_sexp_t s_key)
{
  gcry_module_t *module = NULL;
  gcry_mpi_t *key = NULL;
  gpg_err_code_t rc;
  
  REGISTER_DEFAULT_PUBKEYS;

  /* Note we currently support only secret key checking */
  rc = sexp_to_key (s_key, 1, &key, &module);
  if (! rc)
    {
      rc = pubkey_check_secret_key (((gcry_pubkey_spec_t *) module->spec)->id, key);
      release_mpi_array (key);
      gcry_free (key);
    }
  return gpg_error (rc);
}


/****************
 * Create a public key pair and return it in r_key.
 * How the key is created depends on s_parms:
 * (genkey
 *  (algo
 *    (parameter_name_1 ....)
 *     ....
 *    (parameter_name_n ....)
 * ))
 * The key is returned in a format depending on the
 * algorithm. Both, private and secret keys are returned
 * and optionally some additional informatin.
 * For elgamal we return this structure:
 * (key-data
 *  (public-key
 *    (elg
 *	(p <mpi>)
 *	(g <mpi>)
 *	(y <mpi>)
 *    )
 *  )
 *  (private-key
 *    (elg
 *	(p <mpi>)
 *	(g <mpi>)
 *	(y <mpi>)
 *	(x <mpi>)
 *    )
 *  )
 *  (misc-key-info
 *     (pm1-factors n1 n2 ... nn)
 *  )
 * )
 */
gpg_error_t
gcry_pk_genkey (gcry_sexp_t *r_key, gcry_sexp_t s_parms)
{
  gcry_pubkey_spec_t *pubkey = NULL;
  gcry_module_t *module = NULL;
  gcry_sexp_t list = NULL, l2 = NULL;
  const char *name;
  size_t n;
  gpg_err_code_t rc = GPG_ERR_NO_ERROR;
  int i;
  const char *algo_name = NULL;
  int algo;
  const char *sec_elems = NULL, *pub_elems = NULL;
  gcry_mpi_t skey[10] = { NULL }, *factors = NULL;
  unsigned int nbits = 0;
  unsigned long use_e = 0;

  REGISTER_DEFAULT_PUBKEYS;

  *r_key = NULL;
  list = gcry_sexp_find_token (s_parms, "genkey", 0);
  if (! list)
    rc = GPG_ERR_INV_OBJ; /* Does not contain genkey data */

  if (! rc)
    {
      l2 = gcry_sexp_cadr (list);
      gcry_sexp_release (list);
      list = l2;
      l2 = NULL;
      if (! list)
	rc = GPG_ERR_NO_OBJ; /* no cdr for the genkey */
    }

  if (! rc)
    {
      name = gcry_sexp_nth_data (list, 0, &n);
      if (! name)
	rc = GPG_ERR_INV_OBJ; /* algo string missing */
    }

  if (! rc)
    {
      char *name_terminated = gcry_xmalloc (n + 1);
      strncpy (name_terminated, name, n);
      name_terminated[n] = 0;

      ath_mutex_lock (&pubkeys_registered_lock);
      module = gcry_pubkey_lookup_name (name_terminated);
      ath_mutex_unlock (&pubkeys_registered_lock);

      free (name_terminated);

      if (! module)
	rc = GPG_ERR_PUBKEY_ALGO; /* unknown algorithm */
      else
	{
	  pubkey = (gcry_pubkey_spec_t *) module->spec;
	  algo = pubkey->id;
	  algo_name = pubkey->name;
	  pub_elems = pubkey->elements_pkey;
	  sec_elems = pubkey->elements_skey;
	}
    }

  if (! rc)
    {
      l2 = gcry_sexp_find_token (list, "rsa-use-e", 0);
      if (l2)
	{
	  char buf[50];

	  name = gcry_sexp_nth_data (l2, 1, &n);
	  if ((! name) || (n >= DIM (buf) - 1))
	    rc = GPG_ERR_INV_OBJ; /* no value or value too large */
	  else
	    {
	      memcpy (buf, name, n);
	      buf[n] = 0;
	      use_e = strtoul (buf, NULL, 0);
	    }
	  gcry_sexp_release (l2);
	  l2 = NULL;
	}
      else
	use_e = 65537; /* not given, use the value generated by old versions. */
    }

  if (! rc)
    {
      l2 = gcry_sexp_find_token (list, "nbits", 0);
      gcry_sexp_release (list);
      list = l2;
      l2 = NULL;
      if (! list)
	rc = GPG_ERR_NO_OBJ; /* no nbits parameter */
      else
	{
	  name = gcry_sexp_nth_data (list, 1, &n);
	  if (! name)
	    rc = GPG_ERR_INV_OBJ; /* nbits without a cdr */
	  else
	    {
	      char *p = gcry_xmalloc (n + 1);
	      memcpy (p, name, n);
	      p[n] = 0;
	      nbits = (unsigned int) strtol (p, NULL, 0);
	      gcry_free (p);
	    }
	}
    }

  if (! rc)
    rc = pubkey_generate (pubkey->id, nbits, use_e, skey, &factors);

  if (! rc)
    {
      char *string, *p;
      size_t nelem=0, nelem_cp = 0, needed=0;
      gcry_mpi_t mpis[30];

      nelem = strlen (pub_elems) + strlen (sec_elems);
      for (i = 0; factors[i]; i++)
	nelem++;
      nelem_cp = nelem;

      needed += nelem * 10;
      needed += 2 * strlen (algo_name) + 300;
      if (nelem > DIM (mpis))
	BUG ();

      /* build the string */
      nelem = 0;
      string = p = gcry_xmalloc (needed);
      p = stpcpy (p, "(key-data");
      p = stpcpy (p, "(public-key(");
      p = stpcpy (p, algo_name);
      for(i = 0; pub_elems[i]; i++)
	{
	  *p++ = '(';
	  *p++ = pub_elems[i];
	  p = stpcpy (p, "%m)");
	  mpis[nelem++] = skey[i];
	}
      p = stpcpy (p, "))");
      p = stpcpy (p, "(private-key(");
      p = stpcpy (p, algo_name);
      for (i = 0; sec_elems[i]; i++)
	{
	  *p++ = '(';
	  *p++ = sec_elems[i];
	  p = stpcpy (p, "%m)");
	  mpis[nelem++] = skey[i];
	}
      p = stpcpy (p, "))");

      /* Very ugly hack to make release_mpi_array() work FIXME */
      skey[i] = NULL;

      p = stpcpy (p, "(misc-key-info(pm1-factors");
      for(i = 0; factors[i]; i++)
	{
	  p = stpcpy (p, "%m");
	  mpis[nelem++] = factors[i];
	}
      strcpy (p, ")))");

      while (nelem < DIM (mpis))
	mpis[nelem++] = NULL;

      {
	int elem_n = strlen (pub_elems) + strlen (sec_elems), i;
	void **arg_list = malloc (sizeof (void *) * nelem_cp);
	if (arg_list)
	  {
	    for (i = 0; i < elem_n; i++)
	      arg_list[i] = &mpis[i];
	    for (; i < nelem_cp; i++)
	      arg_list[i] = &factors[i - elem_n];

	    rc = gcry_sexp_build_array (r_key, NULL, string, arg_list);

	    free (arg_list);
	  }
	else
	  rc = gpg_err_code_from_errno (errno);
      }

      if (rc)
	BUG ();
      assert (DIM (mpis) == 30);	/* ? */
      gcry_free (string);
    }

  release_mpi_array (skey);
  /* no free:  skey is a static array */

  if (factors)
    {
      release_mpi_array ( factors );
      gcry_free (factors);
    }

  if (l2)
    gcry_sexp_release (l2);
  if (list)
    gcry_sexp_release (list);

  if (module)
    {
      ath_mutex_lock (&pubkeys_registered_lock);
      _gcry_module_release (module);
      ath_mutex_unlock (&pubkeys_registered_lock);
    }

  return gpg_error (rc);
}

/****************
 * Get the number of nbits from the public key
 * Hmmm: Should we have really this function or is it
 * better to have a more general function to retrieve
 * different propoerties of the key?
 */
unsigned int
gcry_pk_get_nbits (gcry_sexp_t key)
{
  gcry_module_t *module = NULL;
  gcry_pubkey_spec_t *pubkey;
  gcry_mpi_t *keyarr = NULL;
  unsigned int nbits = 0;
  gpg_err_code_t rc;

  REGISTER_DEFAULT_PUBKEYS;

  rc = sexp_to_key (key, 0, &keyarr, &module);
  if (rc == GPG_ERR_INV_OBJ)
    rc = sexp_to_key (key, 1, &keyarr, &module);
  if (rc)
    return 0;
  else
    {
      pubkey = (gcry_pubkey_spec_t *) module->spec;
      nbits = (*pubkey->get_nbits) (pubkey->id, keyarr);

      ath_mutex_lock (&pubkeys_registered_lock);
      _gcry_module_release (module);
      ath_mutex_unlock (&pubkeys_registered_lock);
    }

  release_mpi_array (keyarr);
  gcry_free (keyarr);

  return nbits;
}


/* Return the so called KEYGRIP which is the SHA-1 hash of the public
   key parameters expressed in a way depended on the algorithm.

   ARRAY must either be 20 bytes long or NULL; in the latter case a
   newly allocated array of that size is returned, otherwise ARRAY or
   NULL is returned to indicate an error which is most likely an
   unknown algorithm.  The function accepts public or secret keys. */
unsigned char *
gcry_pk_get_keygrip (gcry_sexp_t key, unsigned char *array)
{
  gcry_sexp_t list = NULL, l2 = NULL;
  gcry_pubkey_spec_t *pubkey = NULL;
  gcry_module_t *module = NULL;
  const char *s, *name;
  size_t n;
  int idx;
  int is_rsa;
  const char *elems;
  gcry_md_hd_t md = NULL;

  REGISTER_DEFAULT_PUBKEYS;

  /* check that the first element is valid */
  list = gcry_sexp_find_token (key, "public-key", 0);
  if (! list)
    list = gcry_sexp_find_token (key, "private-key", 0);
  if (! list)
    list = gcry_sexp_find_token (key, "protected-private-key", 0);
  if (! list)
    return NULL; /* no public- or private-key object */

  l2 = gcry_sexp_cadr (list);
  gcry_sexp_release (list);
  list = l2;
  l2 = NULL;

  name = gcry_sexp_nth_data (list, 0, &n);
  if (! name)
    goto fail; /* invalid structure of object */

  
  {
    char *name_terminated = gcry_xmalloc (n + 1);
    strncpy (name_terminated, name, n);
    name_terminated[n] = 0;

    ath_mutex_lock (&pubkeys_registered_lock);
    module = gcry_pubkey_lookup_name (name_terminated);
    ath_mutex_unlock (&pubkeys_registered_lock);

    free (name_terminated);
  }

  if (! module)
    goto fail; /* unknown algorithm */
  else
    pubkey = (gcry_pubkey_spec_t *) module->spec;

  /* FIXME, special handling should be implemented by the algorithms,
     not by the libgcrypt core.  */
  is_rsa = pubkey->id == GCRY_PK_RSA;
  elems = pubkey->elements_grip;
  if (! elems)
    goto fail; /* no grip parameter */
    
  if (gcry_md_open (&md, GCRY_MD_SHA1, 0))
    goto fail;

  for (idx = 0, s = elems; *s; s++, idx++)
    {
      const char *data;
      size_t datalen;

      l2 = gcry_sexp_find_token (list, s, 1);
      if (! l2)
        goto fail;
      data = gcry_sexp_nth_data (l2, 1, &datalen);
      if (! data)
        goto fail;
      if (!is_rsa)
        {
          char buf[30];

          sprintf (buf, "(1:%c%u:", *s, (unsigned int)datalen);
          gcry_md_write (md, buf, strlen (buf));
        }
      /* pkcs-15 says that for RSA only the modulus should be hashed -
         however, it is not clear wether this is meant to has the raw
         bytes assuming this is an unsigned integer or whether the DER
         required 0 should be prefixed. We hash th raw bytes.  For
         non-RSA we hash S-expressions. */
      gcry_md_write (md, data, datalen);
      gcry_sexp_release (l2);
      if (!is_rsa)
        gcry_md_write (md, ")", 1);
    }

  if (!array)
    {
      array = gcry_malloc (20);
      if (! array)
        goto fail;
    }
  memcpy (array, gcry_md_read (md, GCRY_MD_SHA1), 20);
  gcry_md_close (md);
  gcry_sexp_release (list);
   return array;

 fail:
  if (l2)
    gcry_sexp_release (l2);
  if (md)
    gcry_md_close (md);
  gcry_sexp_release (list);
  return NULL;
}


gpg_error_t
gcry_pk_ctl (int cmd, void *buffer, size_t buflen)
{
  gpg_err_code_t err = GPG_ERR_NO_ERROR;

  REGISTER_DEFAULT_PUBKEYS;

  switch (cmd)
    {
    case GCRYCTL_DISABLE_ALGO:
      /* this one expects a buffer pointing to an integer with the
       * algo number.
       */
      if ((! buffer) || (buflen != sizeof (int)))
	err = GPG_ERR_CIPHER_ALGO;  /* FIXME?  */
      else
	disable_pubkey_algo (*((int *) buffer));
      break;

    default:
      err = GPG_ERR_INV_OP;
    }

  return gpg_error (err);
}


/****************
 * Return information about the given algorithm
 * WHAT select the kind of information returned:
 *  GCRYCTL_TEST_ALGO:
 *	Returns 0 when the specified algorithm is available for use.
 *	Buffer must be NULL, nbytes  may have the address of a variable
 *	with the required usage of the algorithm. It may be 0 for don't
 *	care or a combination of the GCRY_PK_USAGE_xxx flags;
 *  GCRYCTL_GET_ALGO_USAGE:
 *      Return the usage glafs for the give algo.  An invalid alog
 *      does return 0.  Disabled algos are ignored here becuase we
 *      only want to know whether the algo is at all capable of
 *      the usage.
 *
 * Note:  Because this function is in most cases used to return an
 * integer value, we can make it easier for the caller to just look at
 * the return value.  The caller will in all cases consult the value
 * and thereby detecting whether a error occured or not (i.e. while checking
 * the block size)
 */
gpg_error_t
gcry_pk_algo_info (int id, int what, void *buffer, size_t *nbytes)
{
  gpg_err_code_t err = GPG_ERR_NO_ERROR;

  switch (what)
    {
    case GCRYCTL_TEST_ALGO:
      {
	int use = nbytes ? *nbytes : 0;
	if (buffer)
	  err = GPG_ERR_INV_ARG;
	else if (check_pubkey_algo (id, use))
	  err = GPG_ERR_PUBKEY_ALGO;
	break;
      }

    case GCRYCTL_GET_ALGO_USAGE:
      {
	gcry_module_t *pubkey;
	int use = 0;

	REGISTER_DEFAULT_PUBKEYS;

	ath_mutex_lock (&pubkeys_registered_lock);
	pubkey = _gcry_module_lookup_id (pubkeys_registered, id);
	if (pubkey)
	  {
	    use = ((gcry_pubkey_spec_t *) pubkey->spec)->use;
	    _gcry_module_release (pubkey);
	  }
	ath_mutex_unlock (&pubkeys_registered_lock);

	/* FIXME? */
	*nbytes = use;
      }

    case GCRYCTL_GET_ALGO_NPKEY:
      {
	/* FIXME?  */
	int npkey = pubkey_get_npkey (id);
	*nbytes = npkey;
	break;
      }
    case GCRYCTL_GET_ALGO_NSKEY:
      {
	/* FIXME?  */
	int nskey = pubkey_get_nskey (id);
	*nbytes = nskey;
	break;
      }
    case GCRYCTL_GET_ALGO_NSIGN:
      {
	/* FIXME?  */
	int nsign = pubkey_get_nsig (id);
	*nbytes = nsign;
	break;
      }
    case GCRYCTL_GET_ALGO_NENCR:
      {
	/* FIXME?  */
	int nencr = pubkey_get_nenc (id);
	*nbytes = nencr;
	break;
      }

    default:
      err = GPG_ERR_INV_OP;
    }

  return gpg_error (err);
}

gpg_err_code_t
_gcry_pk_init (void)
{
  gpg_err_code_t err = GPG_ERR_NO_ERROR;

  REGISTER_DEFAULT_PUBKEYS;

  return err;
}
