/* wptKeyManager.cpp - Handy functions for the Key Manager dialog
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */
/* x-todo-status: OK */

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <io.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptCommonCtl.h"
#include "wptNLS.h"
#include "wptErrors.h"
#include "wptGPG.h"
#include "wptContext.h"
#include "wptKeylist.h"
#include "wptFileManager.h"
#include "wptDlgs.h"
#include "wptKeyserver.h"
#include "wptKeyManager.h"
#include "wptKeylist.h"
#include "wptHTTP.h"


static void 
km_get_clip_info (const char *uid, char *buf, size_t buflen)
{
    gpgme_key_t pk;
    unsigned long a, algo;

    if (get_pubkey (uid, &pk))
	BUG (0);
    
    a = gpgme_key_get_ulong_attr (pk, GPGME_ATTR_CREATED, NULL, 0);
    algo = gpgme_key_get_ulong_attr (pk, GPGME_ATTR_ALGO, NULL, 0);
    _snprintf (buf, buflen-1, 
	       "pub %04d%s/%s %s %s\r\n"
	       "    Primary key fingerprint: %s\r\n",	
        gpgme_key_get_ulong_attr (pk, GPGME_ATTR_LEN, NULL, 0),
        gpgme_key_expand_attr (GPGME_ATTR_ALGO_SHORT, algo),
        gpgme_key_get_string_attr (pk, GPGME_ATTR_KEYID, NULL, 0) + 8,	
        gpgme_key_expand_attr (GPGME_ATTR_CREATED, a ),
        gpgme_key_get_string_attr (pk, GPGME_ATTR_USERID, NULL, 0),
        get_key_fpr (pk));
}


char*
km_quote_uid (const char * uid)
{    
    char * q = new char[strlen (uid) + 4];
    if (!q)
	BUG (NULL);
    _snprintf (q, strlen (uid) + 3, "\"%s\"", uid);
    return q;
} /* km_quote_uid */


int
km_check_for_seckey (listview_ctrl_t lv, int pos, int * utrust)
{
    char t[128], t2[128];
    int type=0;
    
    listview_get_item_text (lv, pos, 5, t, sizeof t -1);
    listview_get_item_text (lv, pos, 2, t2, sizeof t2  -1);
    if (!strcmp (t2, "pub/sec"))
	type = 1;
    else if (!strcmp (t2, "pub/crd"))
	type = 2;
    if (stristr (t, "ultimate") && utrust)
	*utrust = 1;
    return type;
} /* km_check_for_seckey */


int
km_check_if_protected( listview_ctrl_t lv, int pos )
{
    gpgme_key_t key;
    char keyid[32];    
    
    listview_get_item_text( lv, pos, 1, keyid, sizeof keyid-1 );
    if( get_pubkey( keyid, &key ) )
    	BUG( NULL );    
    return gpgme_key_get_ulong_attr( key, GPGME_ATTR_IS_PROTECTED, NULL, 0 );
} /* km_check_if_protected */


int
km_check_key_status (listview_ctrl_t lv, int pos)
{
    char t[128];
    int i = 1;
    
    listview_get_item_text( lv, pos, 5, t, sizeof t - 1 );
    if( t[0] == '[' && t[1] == ']' )
	return 1;
    for( i = 0; t[i] != ']'; i++ ) {
	if( t[i] == 'E' )
	    msg_box(lv->ctrl, _("This key has expired!\n"
				"Key check failed."), _("Key Manager"), MB_ERR );
	else if( t[i] == 'R' )
	    msg_box(lv->ctrl, _("This key has been revoked by its owner!\n"
				"Key check failed."), _("Key Manager"), MB_ERR );
    }

    return 0;
} /* km_check_key_status */


int
km_get_key_status( listview_ctrl_t lv, int pos )
{
    char t[128];
    int i, flags = 0;

    if( pos == -1 )
	return 0;
    listview_get_item_text( lv, pos, 5, t, sizeof t-1 );
    for( i = 0; t[i] != ']'; i++ ) {
	if( t[i] == 'E' )
	    flags |= KM_FLAG_EXPIRED;
	if( t[i] == 'R' )
	    flags |= KM_FLAG_REVOKED;
	if( t[i] == 'D' )
	    flags |= KM_FLAG_DISABLED;
    }
    return flags;
} /* km_get_key_status */


int
km_enable_disable_key( listview_ctrl_t lv, HWND dlg, int pos, int enable )
{
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    gpgme_error_t err;
    int edit_id;
    char keyid[32];

    listview_get_item_text( lv, pos, 1, keyid, sizeof keyid-1 );
    err = gpgme_editkey_new( &ek );
    if( err )
	BUG( NULL );
    if( enable ) {
	gpgme_editkey_enable_set( ek );
	edit_id = GPGME_EDITKEY_ENABLE;
    }
    else {
	gpgme_editkey_disable_set( ek );
	edit_id = GPGME_EDITKEY_DISABLE;
    }
    err = gpgme_new( &ctx );
    if( err )
	BUG( NULL );
    gpgme_set_edit_ctx( ctx, ek, edit_id );
    err = gpgme_op_editkey( ctx, keyid );
    if( !err ) {
	show_msg( dlg, 1500, _("Key status changed.") );
	keycache_set_reload( 1 );
    }
    else
	msg_box( dlg, gpgme_strerror( err ), _("Key Manager"), MB_ERR );

    gpgme_release( ctx );
    gpgme_editkey_release( ek );
    return err? WPTERR_GENERAL : 0;
} /* km_enable_disable_key */


int
km_clip_export (HWND dlg, listview_ctrl_t lv)
{
    gpgme_error_t err;
    gpgme_recipients_t rset;
    int rc, id = 0, n = 0;
    
    rset = keylist_enum_recipients (lv, KEYLIST_LIST);
    n = gpgme_recipients_count (rset);
    if (!n) {
        msg_box (dlg, _("No key was selected for export."), _("Key Manager"), MB_ERR);
        rc = WPTERR_GENERAL; 
	goto leave;
    }
    
    err = gpgme_op_clip_export (rset);
    if (err) {
        msg_box( dlg, gpgme_strerror (err), _("Key Manager"), MB_ERR);
        rc = WPTERR_GENERAL;
	goto leave;
    }
    if (n == 1) {
	const char *s;
	char buf[256];
	s = gpgme_recipients_get_name (rset, 0);
	km_get_clip_info (s, buf, 255);
	set_clip_text2 (NULL, buf, strlen (buf), 0);
    }

    show_msg (dlg, 1500, _("GnuPG Status: Finished"));
    
leave:
    gpgme_recipients_release (rset);
    return rc;
} /* km_clip_export */


int
km_privkey_export( HWND dlg, listview_ctrl_t lv, const char *fname )
{
    gpgme_recipients_t rset;
    gpgme_data_t keydata;
    gpgme_error_t err;
    gpgme_ctx_t ctx;
    size_t n = 0;

    rset = keylist_enum_recipients( lv, KEYLIST_LIST );
    n = gpgme_recipients_count( rset );
    if( !n ) {
        msg_box( dlg, _("No key was selected for export."), _("Key Manager"), MB_ERR );
        return WPTERR_GENERAL;
    }
    if( n > 1 ) {
	msg_box( dlg, _("Only one secret key can be exported."), _("Key Manager"), MB_ERR );
        return 0; /* we checked this before, so we just quit */
    }
    err = gpgme_data_new( &keydata );
    if( err )
	BUG( dlg );
    err = gpgme_new( &ctx );
    if( err )
	BUG( dlg );
    gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );
    gpgme_control( ctx, GPGME_CTRL_WITH_SECRET_KEY, 1 );

    err = gpgme_op_export( ctx, rset, keydata );
    if( err ) {
        msg_box( dlg, gpgme_strerror( err ), _("Key Manager"), MB_ERR );
        goto leave;
    }

    log_box( _("Key Manager"), MB_OK, 
	     _("Secret key successfully saved in '%s'."), fname );
    
leave:
    err = gpgme_data_release_and_set_file( keydata, fname );
    if( err )
	log_box( _("Key Manager"), MB_OK, 
		 _("Could not save data to '%s'."), fname );
    gpgme_release( ctx );

    return (int)err;
} /* km_privkey_export */


int
km_file_export (HWND dlg, listview_ctrl_t lv, const char * fname)
{
    gpgme_recipients_t rset;
    gpgme_data_t keydata;	
    gpgme_error_t err;
    gpgme_ctx_t ctx;

    rset = keylist_enum_recipients( lv, KEYLIST_LIST );
    if( !gpgme_recipients_count( rset ) ) {
        msg_box( dlg, _("No key was selected for export."), _("Key Manager"), MB_ERR );
        return WPTERR_GENERAL;
    }
    
    err = gpgme_data_new( &keydata );
    if( err )
	BUG( dlg );
    err = gpgme_new( &ctx );
    if( err )
	BUG( dlg );
    gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );
    gpgme_set_comment (ctx, "Generated by WinPT "PGM_VERSION);
    
    err = gpgme_op_export( ctx, rset, keydata );
    if( err ) {
        msg_box( dlg, gpgme_strerror( err ), _("Key Manager"), MB_ERR );
        goto leave;
    }
        
    log_box( _("Key Manager"), MB_OK, 
	     _("Key(s) successfully saved in '%s'."), fname );
    
leave:
    err = gpgme_data_release_and_set_file( keydata, fname );
    if( err )
	log_box( _("Key Manager"), MB_OK,
		 _("Could not save data to '%s'."), fname );    
    gpgme_release( ctx );

    return (int)err;
} /* km_file_export */


static int
extract_dash_escaped_key( void )
{
    gpgme_data_t inp, plain;
    gpgme_error_t err;

    err = gpgme_data_new_from_clipboard (&inp);
    if( err ) {
	msg_box( NULL, gpgme_strerror( err ), _("Key Manager"), MB_ERR );
	return -1;
    }
    gpgme_data_extract_plaintext( inp, &plain );
    gpgme_data_release( inp );
    gpgme_data_release_and_set_clipboard( plain );

    return 0;
} /* extract_dash_escaped_key */


int
km_clip_import( HWND dlg )
{
    gpgme_error_t err;
    gpgme_cliptype_t pgptype;
    int id;
    int has_data = 0;
    
    if( !gpgme_clip_istext_avail( &has_data ) && !has_data ) {
        msg_box( dlg, winpt_strerror( WPTERR_CLIP_ISEMPTY ), _("Key Manager"), MB_ERR );
        return WPTERR_CLIP_ISEMPTY;
    }
    err = gpgme_clip_is_secured( &pgptype, &has_data );
    if( err )
	msg_box( dlg, gpgme_strerror( err ), _("Key Manager"), MB_ERR );
    if( !has_data ) {
        msg_box( dlg, _("No valid OpenPGP data found."), _("Key Manager"), MB_ERR );
        return WPTERR_GENERAL;
    }
    if( !(pgptype & GPGME_CLIP_PUBKEY) && !(pgptype & GPGME_CLIP_SECKEY) ) {
	msg_box( dlg, _("No valid OpenPGP keys found."), _("Key Manager"), MB_ERR );
	return WPTERR_GENERAL;
    }
    if( pgptype & GPGME_CLIP_DASH_ESCAPED ) {
	id = msg_box( dlg, _("The key you want to import is dash escacped.\n"
			     "Do you want to extract the key?"), 
		      _("Key Manager"), MB_YESNO );
	if( id == IDYES )
	    extract_dash_escaped_key( );
	else
	    msg_box( dlg, _("Cannot import dash escaped OpenPGP keys."), _("Key Manager"), MB_INFO );
    }

    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_IMPORT, dlg,
		      clip_import_dlg_proc, NULL,
		      _("Key Import"), IDS_WINPT_IMPORT );

    return 0;
} /* km_clip_import */


int
km_http_import (HWND dlg, const char * url)
{
    FILE * fp;
    char * p;
    char tmpdir[500];
    http_hd_t hd;
    int statcode;
    int rc = 0;

    if (strncmp (url, "http://", 7)) {
	log_box (_("Key Import HTTP"), MB_ERR, _("Invalid HTTP URL: %s"), url);
	return WPTERR_GENERAL;
    }

    GetTempPath (499, tmpdir);
    p = make_filename (tmpdir, "file_http", "tmp");
    if (!p)
	BUG (0);
    fp = fopen (p, "wb");
    if (fp == NULL) {
	free_if_alloc (p);
	log_box (_("Key Import HTTP"), MB_ERR, "%s: %s", p, winpt_strerror (WPTERR_FILE_CREAT));
	return WPTERR_FILE_CREAT;
    }
    /* parse URL */
    rc = http_send_request2 (url, &hd);
    if (!rc)
	rc = http_parse_response (hd, &statcode);
    if (!rc)
	rc = http_parse_data (hd, fp);
    http_hd_free (hd);
    fclose (fp);
    if (rc) {
	free_if_alloc (p);
	msg_box (dlg, http_strerror (rc), _("Key Import HTTP"), MB_ERR);
	return WPTERR_GENERAL;
    }
    km_file_import (dlg, p);
    free_if_alloc (p);
    return 0;
}

int
km_file_import( HWND dlg, const char * fname )
{
    gpgme_data_t keydata = NULL;
    gpgme_ctx_t ctx;
    gpgme_error_t err;
    fm_state_s fm_stat;
    int import_res[14];
    
    memset( &fm_stat, 0, sizeof fm_stat );
    fm_stat.opaque = m_strdup( fname );
    
    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_IMPORT, dlg,
                      file_import_dlg_proc, (LPARAM)&fm_stat,
                      _("File Import"), IDS_WINPT_IMPORT );
    if( fm_stat.cancel == 1 )
        return WPTERR_GENERAL;
    
    err = gpgme_new( &ctx );
    if( err )
	BUG( dlg );
    gpgme_control( ctx, GPGME_CTRL_FORCETRUST, 1 );
    err = gpgme_data_new_from_file (&keydata, fname);
    if( err ) {
        msg_box( dlg, _("Could not read key-data from file."), _("Key Manager"), MB_ERR );
        goto leave;
    }
    
    err = gpgme_op_import( ctx, NULL, keydata );
    if( err ) {
        msg_box( dlg, gpgme_strerror( err ), _("Key Manager"), MB_ERR );
        goto leave;
    }
    
    gpgme_get_import_status( ctx, import_res, NULL );
    print_import_status( import_res, fm_stat.implist_revcert );
    if( import_res[GPGME_IMPSTAT_NOSELFSIG] > 0  ) {
	msg_box( dlg, _("Key without a self signature was dectected!\n"
			"(This key is NOT usable for encryption, etc)\n"
			"\n"	
			"Cannot import these key(s)!"), _("Import"), MB_INFO );
    }
    
leave:        
    gpgme_data_release( keydata );
    gpgme_release( ctx );
    free_if_alloc( fm_stat.opaque );
    return (int)err;
} /* km_file_import */


static void
delete_keys_from_cache (gpgme_recipients_t rset)
{
    gpgme_keycache_t pub = keycache_get_ctx (1);
    void * ctx =NULL;
    const char * s;

    gpgme_recipients_enum_open (rset, &ctx);
    while ((s = gpgme_recipients_enum_read (rset, &ctx)))
	gpgme_keycache_delete_key (pub, s);
    gpgme_recipients_enum_close (rset, &ctx);
} /* delete_keys_from_cache */


int
km_delete_keys (listview_ctrl_t lv, HWND dlg)
{
    gpgme_error_t err;
    gpgme_recipients_t rset;
    char keyid[32], uid[256], date[64], keylen[64];    
    int with_seckey, seckey_type=0;
    int i, rc;
    
    if( listview_get_curr_pos( lv ) == -1 ) {
        msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
        return WPTERR_GENERAL;
    }
    
    err = gpgme_recipients_new (&rset);
    if (err)
	BUG (0);
    
    for( i = 0; i < listview_count_items( lv, 0 ); i++ ) {
        if( listview_get_item_state( lv, i ) ) {
            listview_get_item_text( lv, i, 0, uid, sizeof uid - 1 );
            listview_get_item_text( lv, i, 1, keyid, sizeof keyid - 1 );
	    listview_get_item_text( lv, i, 3, keylen, sizeof keylen - 1 );
	    listview_get_item_text( lv, i, 7, date, sizeof date - 1 );
	    seckey_type = km_check_for_seckey( lv, i, NULL );
            if( !seckey_type ) {
		rc = log_box( _("Key Manager"), MB_YESNO|MB_ICONWARNING,
                              _("Do you really want to delete this key?\n\n"
			        "pub %s %s %s\n"
			        "  \"%s\""), keylen, keyid, date, uid );
                if( rc == IDYES )
                    gpgme_recipients_add_name( rset, keyid );
                with_seckey = 0;
            }
            else {
		rc = log_box( _("Key Manager"), MB_YESNO|MB_ICONWARNING,                
                              _("Do you really want to delete this KEY PAIR?\n\n"
				"Please remember that you are not able to decrypt\n"
				"messages you stored with this key any longer.\n"
				"\n"
			        "pub/sec %s %s %s\n"
			        "  \"%s\""), keylen, keyid, date, uid );
                if( rc == IDYES ) {
		    if( seckey_type == 2 )
			msg_box( dlg, _("The actual secret key is stored on a smartcard.\n"
					"Only the public key and the secret key \n"
					"placeholder will be deleted.\n"), _("Key Manager"), MB_OK );
                    gpgme_recipients_add_name( rset, keyid );
		}
                with_seckey = 1;
            }
        }
    }
    
    if (!gpgme_recipients_count (rset)) {
        gpgme_recipients_release (rset);
        return 0;
    }
    
    err = gpgme_op_delete_keys (rset, with_seckey);
    if (err) {
        if (err == GPGME_Invalid_Key)
            msg_box (dlg, _("No such key."), _("Key Manager"), MB_INFO);
        else if (err == GPGME_Conflict)
            msg_box (dlg, _("Must delete secret key first."), _("Key Manager"), MB_INFO);
        else
            msg_box (dlg, gpgme_strerror (err), _("Key Manager"), MB_ERR);
        return FALSE;
    }
    show_msg (dlg, 1500, _("GnuPG Status: Finished"));
    listview_del_items (lv);
    if (keyring_check_last_access ())
        keycache_set_reload (1);
    delete_keys_from_cache (rset);
    gpgme_recipients_release (rset);

    return (int)err;
} /* km_delete_keys */


int
km_send_to_keyserver (listview_ctrl_t lv, HWND dlg, const char * host, u16 port)
{
    char keyid[32];
    const char *t;
    int id;
    
    id = listview_get_curr_pos( lv );
    if( id == -1 ) {
	msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
	return WPTERR_GENERAL;
    }

    listview_get_item_text( lv, id, 1, keyid, sizeof keyid - 1 );
    id = log_box (_("Key Manager"), MB_YESNO,
		  _("Do you really want to send '%s' to keyserver %s?"),
		    keyid, host);
    if (id == IDYES) {
        t = keyid;
        if (!strncmp (keyid, "0x", 2)) 
	    t += 2;
        hkp_send_key (dlg, host, port, t);
    }

    return 0;
} /* km_send_to_keyserver */


int
km_send_to_mail_recipient( listview_ctrl_t lv, HWND dlg )
{
    gpgme_key_t key;
    gpgme_ctx_t ctx=NULL;
    gpgme_recipients_t rset=NULL;
    gpgme_error_t rc;
    const char * s;
    char keyid[32], tmp[192+256], * p =NULL;
    int pos;

    if( listview_count_items( lv, 1 ) > 1 ) {
	msg_box( dlg, _("Please only select one key."), _("Key Manager"), MB_INFO|MB_OK );
	return WPTERR_GENERAL;
    }
    pos = listview_get_curr_pos( lv );
    if( pos == -1 ) {
	msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
	return WPTERR_GENERAL;
    }
    listview_get_item_text( lv, pos, 1, keyid, sizeof keyid-1 );
    if( get_pubkey( keyid, &key ) )
	BUG( NULL );
    s = gpgme_key_get_string_attr( key, GPGME_ATTR_NAME, NULL, 0 );
    GetTempPath (sizeof tmp-1, tmp);
    strncat (tmp, s, sizeof tmp-200);
    strncat (tmp, ".asc", sizeof tmp-200);
    p = fm_quote_file (tmp);

    rc = gpgme_recipients_new( &rset );
    if( !rc )
	rc = gpgme_recipients_add_name( rset, keyid );
    if( !rc )
	rc = gpgme_new( &ctx );
    if( !rc ) {
	gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );
	rc = gpgme_op_file_export( ctx, rset, p );
    }
    if( rc )
	msg_box( dlg, gpgme_strerror( rc ), _("Key Manager"), MB_ERR );
    else
	mapi_send_pubkey (keyid, tmp);
    free_if_alloc( p );
    gpgme_recipients_release( rset );
    gpgme_release( ctx );    
    return rc;
}


static void
km_refresh_one_key (listview_ctrl_t lv, HWND dlg, int pos)
{
    int idx;
    char keyid[32];
    const char *t;

    if (pos != 0)
	idx = pos;
    else
	idx = listview_get_curr_pos (lv);
    if (idx != -1) 
    {
	listview_get_item_text (lv, idx, 1, keyid, sizeof keyid - 1);
	t = keyid;
	if (!strncmp (keyid, "0x", 2))
	    t += 2;
	hkp_recv_key (dlg, default_keyserver, default_keyserver_port, t, 0, KM_KS_REFRESH);
    }
}


void
km_refresh_from_keyserver (listview_ctrl_t lv, HWND dlg)
{
    int idx, id, i;
    
    if (kserver_check_inet_connection ())
    {
	msg_box (dlg, _("Could not connect to keyserver, abort procedure."),
		 _("Key Manager"), MB_ERR);
	return;
    }
    idx = listview_count_items (lv, 0);
    if (listview_count_items (lv, 1) == idx) {
        id = msg_box (dlg, _("Do you really want to refresh all keys in the keyring?"), _("Key Manager"), MB_YESNO);
        if (id == IDNO)
            return;
        for (i = 0; i < idx; i++)
	    km_refresh_one_key (lv, dlg, i);
    }
    else if (idx == 1)
	km_refresh_one_key (lv, dlg, 0);
    else {
	for (i=0; i < listview_count_items (lv, 0); i++) {
	    if (listview_get_item_state (lv, i))
		km_refresh_one_key (lv, dlg, i);
	}
    }
} /* km_refresh_from_keyserver */


void
km_set_clip_info( const char *uid )
{    
    char buf[256];
    
    km_get_clip_info (uid, buf, 255);    
    set_clip_text( NULL, buf, strlen( buf ) );
} /* km_set_clip_info */



int
km_key_is_v3( listview_ctrl_t lv, int pos )
{
    gpgme_key_t pk;
    const char * fpr;
    unsigned long algo;
    char keyid[32];

    listview_get_item_text( lv, pos, 1, keyid, sizeof keyid-1 );
    if( get_pubkey( keyid, &pk ) )
	BUG( NULL );
    algo = gpgme_key_get_ulong_attr( pk, GPGME_ATTR_ALGO, NULL, 0 );
    fpr = gpgme_key_get_string_attr( pk, GPGME_ATTR_FPR, NULL, 0 );
    return strlen( fpr ) == 32 && algo == GPGME_PK_RSA? 1 : 0;
} /* km_key_is_v3 */


void
km_update_default_key_str (HWND dlg, int * ret_len)
{
    char * keyid, defkeyinf[512];
    const char * fmt;
    
    keyid = get_gnupg_default_key ();
    if (!keyid)
	BUG (0);
    if( (keyid[0] >= 'A' && keyid[0] <= 'Z') || (keyid[0] >= 'a' && keyid[0] <= 'z')
	|| (keyid[0] == '0' && keyid[1] == 'x') )
        fmt = _("Default Key: %s");
    else
        fmt = _("Default Key: 0x%s");
    _snprintf( defkeyinf, sizeof defkeyinf - 1, fmt, keyid );
    SetWindowText( dlg, defkeyinf );
    *ret_len = strlen( defkeyinf );
    free_if_alloc( keyid );
} /* km_return_default_key_str */


void
km_complete_status_bar( HWND sb, listview_ctrl_t lv, int startpos )
{
    char text[384];
    int nkeys = 0, nsec = 0, i;

    GetWindowText( sb, text, sizeof text -1 );
    nkeys = listview_count_items( lv, 0 );
    for( i = 0; i < nkeys; i++ ) {
	if( km_check_for_seckey( lv, i, NULL ) )
	    nsec++;
    }
    _snprintf( text+startpos, sizeof text-1, "                %d keys (%d secret keys)", nkeys, nsec );
    SetWindowText( sb, text );
} /* km_complete_status_bar */


void
km_set_implicit_trust (HWND dlg, listview_ctrl_t lv, int pos)
{
    gpgme_error_t err;
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    char keyid[32];

    listview_get_item_text (lv, pos, 1, keyid, 31);
    err = gpgme_new (&ctx);
    if (err)
	BUG (0);
    err = gpgme_editkey_new (&ek);
    if (err)
	BUG (0);
    gpgme_set_edit_ctx (ctx, ek, GPGME_EDITKEY_TRUST);
    gpgme_editkey_trust_set (ek, 5);

    err = gpgme_op_editkey (ctx, keyid);
    if (err)
	msg_box (dlg, gpgme_strerror (err), _("Key Manager"), MB_ERR);
    else {
	show_msg (dlg, 1500, _("GnuPG Status: Finished"));
	keycache_set_reload (1);
    }

    gpgme_release (ctx);
    gpgme_editkey_release (ek);    
}


gpg_optfile_t
km_groupdb_open( void )
{	
    gpg_optfile_t opt;
    char * optfile;
    int err = 0;
    
    optfile = get_gnupg_cfgfile();
    if( !optfile )
	BUG( NULL );
    if( parse_gpg_options( optfile, &opt ) )
	err = 1;
    free_if_alloc( optfile );
    return err? NULL : opt;
} /* km_groupdb_open */


int
km_groupdb_expand_recipients( const char *name, gpgme_recipients_t rset )
{   
    gpgme_keycache_t kc;
    gpgme_key_t pk;
    gpg_optfile_t opt;
    gpg_group_t grp;    
    gpg_member_t mbr;
    int no_trust = 0, n;
 
    kc = keycache_get_ctx( 1 );
    if( !kc )
	BUG( NULL );

    opt = km_groupdb_open( );
    if( !opt )
	return WPTERR_FILE_OPEN;
    
    grp = find_group( opt, name );
    if( !grp )
	return WPTERR_GENERAL;
    
    /* we are paranoid and check that all group members exist in the
       key cache. there is no need that it is really the real key, but
       an entry should be available. the rest is up to GPG. */
    for( mbr = grp->list; mbr; mbr = mbr->next ) {
	if( gpgme_keycache_find_key( kc, mbr->name, 0, &pk ) )
	    BUG( NULL );
	n = gpgme_key_count_items( pk, GPGME_ATTR_USERID );
	while( n-- ) {
	    const char * s = gpgme_key_get_string_attr( pk, GPGME_ATTR_USERID, NULL, n );
	    if( s && stristr( s, mbr->name )
		&& gpgme_key_get_ulong_attr( pk, GPGME_ATTR_VALIDITY, NULL, n ) < 3 )
		no_trust++;
	}
    }

    gpgme_recipients_add_name( rset, name );
    release_gpg_options( opt );
    
    return no_trust;
} /* km_groupdb_expand_recipients */


static HTREEITEM
km_tv_insert_item( HWND tree, HTREEITEM parent, const char *text )
{
    TVINSERTSTRUCT tvis;
    HTREEITEM node;
    
    memset( &tvis, 0, sizeof tvis );
    tvis.hParent = parent;
    tvis.hInsertAfter = TVI_LAST;
    tvis.item.mask = TVIF_TEXT;
    tvis.item.pszText = (char *)text;
    node = TreeView_InsertItem( tree, &tvis );
    return node;
} /* km_tv_insert_item */


int
km_groups_new( km_group_t *r_gc, HWND ctrl )
{
    km_group_t gc;
    
    gc = new km_group_s;
    if (!gc)
        BUG (NULL);
    gc->tree = ctrl;
    gc->gh = km_groupdb_open ();
    *r_gc = gc;
    return 0;
} /* km_groups_new */


void
km_groups_sync( km_group_t gc )
{
    char * optfile;

    optfile = get_gnupg_cfgfile ();
    if( !optfile )
	BUG( NULL );
    commit_gpg_options( optfile, gc->gh );
    free_if_alloc( optfile );
    gc->need_sync = 0;
} /* km_groups_sync */


void
km_groups_release (km_group_t gc)
{
    if( gc ) {
	/* xxx: this reset the default key (sync=1) */
	gc->need_sync=0;
	if (gc->need_sync)
	    km_groups_sync (gc);
	if (gc->gh)
	    release_gpg_options( gc->gh );
	gc->gh = NULL;
	gc->tree = NULL;
	delete gc;
    }    
} /* km_groups_release */


int
km_groups_load( km_group_t gc )
{    
    HTREEITEM n;
    gpg_group_t grp, g;
    gpg_member_t mbr;
    u32 gid = 0; 
    
    if( !gc->gh )
	return 0;
    grp = gc->gh->grp;
    if( !grp )
	return 0; /* no groups */
        
    for( g = grp; g; g = g->next ) {
	n = km_tv_insert_item( gc->tree, NULL, g->name );
	for( mbr = g->list; mbr; mbr = mbr->next ) {
	    if( mbr->used && mbr->name )
		km_tv_insert_item( gc->tree, n, mbr->name );
	}
    }
    DragAcceptFiles( gc->tree, TRUE );
    gc->need_sync = 0;
    return 0;
} /* km_groups_load */


int
km_groups_add( km_group_t gc, listview_ctrl_t lv, int km_index )
{
    TVITEM tvi;
    char uid[128], valid[64], text[128];
    int i_valid;
    
    memset( &tvi, 0, sizeof tvi );
    tvi.hItem = TreeView_GetSelection( gc->tree );
    tvi.pszText = text;
    tvi.cchTextMax = sizeof text -1;
    tvi.mask = TVIF_TEXT;
    TreeView_GetItem( gc->tree, &tvi );
    
    
    listview_get_item_text( lv, km_index, 0, uid, sizeof uid -1 );
    listview_get_item_text( lv, km_index, 5, valid, sizeof valid -1 );
    
    if( strstr( valid, "Ultimate" ) )
	i_valid = 5;    
    else if( !strstr( valid, "Full" ) )
        i_valid = 4;
    else if( !strstr( valid, "Marginal" ) )
        i_valid = 3;
    else
        i_valid = 0;
    
    /* we can't add the full name. one way would be to use the first
       text until a space appears. 
    group_add_entry(&gc->gh, gid, i_valid, uid);
    treeview_add_item(gc->tree, tvi.hItem, uid);
    */
    gc->need_sync = 1;
    
    return 0;
} /* km_groups_add */


static int
km_groups_del_main( km_group_t gc )
{
    TVITEM tvi;
    char text[128];
    int id;
    
    memset( &tvi, 0, sizeof tvi );
    tvi.hItem = TreeView_GetSelection( gc->tree );
    tvi.pszText = text;
    tvi.cchTextMax = sizeof text -1;
    tvi.mask = TVIF_TEXT;
    TreeView_GetItem( gc->tree, &tvi );
                             
    id = log_box( _("Key Manager"), MB_INFO_ASK, 
		   _("Do you really want to delete this group?\n\n%s"), text);
    if( id == IDNO )
        return 0;
    delete_group( gc->gh, text );    
    TreeView_DeleteItem( gc->tree, &tvi );
    gc->need_sync = 1;
    
    return 0;
} /* km_groups_del */


static int
km_groups_del_entry( km_group_t gc )
{
    TVITEM tvi;
    HTREEITEM root;
    int id;
    char text[128], info[256];
    gpg_group_t	 grp = NULL;
    
    memset( &tvi, 0, sizeof tvi );
    tvi.hItem = TreeView_GetSelection( gc->tree );
    tvi.pszText = text;
    tvi.cchTextMax = sizeof text-1;
    tvi.mask = TVIF_TEXT;
    TreeView_GetItem( gc->tree, &tvi );
    
    _snprintf( info, sizeof info -1,
              _("Do you really want to delete this entry?\n\n%s"), text );
    
    id = msg_box( gc->tree, info, _("Key Manager"), MB_INFO_ASK );
    if( id == IDNO )
        return 0;

    root = TreeView_GetParent( gc->tree, tvi.hItem );
    if( root ) {
    }
    
    delete_member( gc->gh, /*fixme*/NULL, text );
    TreeView_DeleteItem( gc->tree, &tvi );
    gc->need_sync = 1;
    return 0;
} /* km_groups_del_entry */


int
km_groups_del( km_group_t gc )
{
    if ( TreeView_GetParent( gc->tree, TreeView_GetSelection( gc->tree ) ) )
        return km_groups_del_entry( gc );
    else
        return km_groups_del_main( gc );
} /* km_groups_del */
