// SPDX-FileCopyrightText: 2024 g10 code GmbH
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "../editor/composerviewbase.h"
#include "../editor/part/textpart.h"
#include "../editor/recipientseditor.h"
#include "../editor/richtextcomposerng.h"
#include <QObject>
#include <QSignalSpy>
#include <QTest>
#include <TextCustomEditor/RichTextEditorWidget>

using namespace Qt::StringLiterals;

static QByteArray readMailFromFile(const QString &mailFile)
{
    QFile file(QLatin1StringView(DATA_DIR) + QLatin1Char('/') + mailFile);
    file.open(QIODevice::ReadOnly);
    Q_ASSERT(file.isOpen());
    return file.readAll();
}

class ComposerViewBaseTest : public QObject
{
    Q_OBJECT

private Q_SLOTS:
    void initTestCase()
    {
        mComposerBase.setRecipientsEditor(new RecipientsEditor());
        auto richTextComposer = new MessageComposer::RichTextComposerNg(nullptr);
        mComposerBase.setEditor(richTextComposer);
    }

    void testSetMessage()
    {
        KMime::Message::Ptr message(new KMime::Message);
        message->setContent(KMime::CRLFtoLF(readMailFromFile(u"draft.mbox"_s)));
        message->parse();

        mComposerBase.setMessage(message);
        mComposerBase.setFrom(u"carl.from@carlschwan.eu"_s);

        QCOMPARE(mComposerBase.to(), QLatin1StringView("carl.to@carlschwan.eu"));
        QCOMPARE(mComposerBase.cc(), QLatin1StringView("carl.cc@carlschwan.eu"));
        QCOMPARE(mComposerBase.bcc(), QLatin1StringView("carl.bcc@carlschwan.eu"));
        QCOMPARE(mComposerBase.from(), QLatin1StringView("carl.from@carlschwan.eu"));
        QCOMPARE(mComposerBase.subject(), QLatin1StringView("Draft"));

        auto textPart = new MessageComposer::TextPart();
        mComposerBase.editor()->fillComposerTextPart(textPart);
        QCOMPARE(textPart->cleanPlainText(), QLatin1StringView("Draft content"));
    }

    void testSetMessageTooManyRecipient()
    {
        KMime::Message::Ptr message(new KMime::Message);
        message->setContent(KMime::CRLFtoLF(readMailFromFile(u"plaintexttoomanyrecipient.mbox"_s)));
        message->parse();

        QSignalSpy tooManySignal(&mComposerBase, &MessageComposer::ComposerViewBase::tooManyRecipient);
        mComposerBase.setMessage(message);
        QCOMPARE(tooManySignal.count(), 1);
    }

    void testSetMessageHtml()
    {
        KMime::Message::Ptr message(new KMime::Message);
        message->setContent(KMime::CRLFtoLF(readMailFromFile(u"html.mbox"_s)));
        message->parse();

        QSignalSpy enableHtmlSignal(&mComposerBase, &MessageComposer::ComposerViewBase::enableHtml);
        mComposerBase.setMessage(message);
        QCOMPARE(enableHtmlSignal.count(), 1);

        auto textPart = new MessageComposer::TextPart();
        mComposerBase.editor()->fillComposerTextPart(textPart);
        QCOMPARE(textPart->cleanPlainText(), QLatin1StringView("*SOME* HTML text. \n"));
        QVERIFY(textPart->isHtmlUsed());
        QCOMPARE(textPart->cleanHtml(),
                 QLatin1StringView(
                     "<html>\n<head>\n<meta http-equiv=\"content-type\" content=\"text/html; charset=UTF-8\">\n</head>\n<body><p "
                     "style=\"margin-top:0;margin-bottom:0;margin-left:0;margin-right:0;\"><strong>SOME</strong>&nbsp;HTML text. </p>\n</body>\n</html>"));
    }

private:
    MessageComposer::ComposerViewBase mComposerBase;
};

QTEST_MAIN(ComposerViewBaseTest)
#include "composerviewbasetest.moc"