unit Install;

{
  Inno Setup
  Copyright (C) 1998-2001 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Installation progress form & Installation procedures

  $Id: Install.pas,v 1.3 2001/01/07 17:21:21 jr Exp $
}

interface

{$I VERSION.INC}

uses
  WinTypes, WinProcs, SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, NewBevel, NewGauge, NewFCtrl,
  ChildFrm, Undo, ShellAPI, ShlObj;

type
  TInstallForm = class(TSetupChildForm)
    StatusLabel: TLabel;
    FilenameLabel: TNewPathLabel;
    CancelButton: TButton;
    ProgressGauge: TNewGauge;
    procedure CancelButtonClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
  private
    { Private declarations }
    DisableCloseQuery, NeedToAbort: Boolean;
    procedure SetProgress (const AbsoluteValue: Boolean; const AProgress: Longint);
  public
    { Public declarations }
    constructor Create (AOwner: TComponent); override;
    procedure StartInstalling;
  end;

var
  InstallForm: TInstallForm;
  WizardDirValue, WizardGroupValue: String;
  WizardNoIcons: Boolean;

implementation

{$R *.DFM}

uses
  InstFunc, InstFnc2, Msgs, MsgIDs, Main, Wizard, NewDisk,
  CmnFunc, CmnFunc2,
  DDEInt,
  Struct,
  zlib;

type
  TSetupUninstallLog = class(TUninstallLog)
  protected
    procedure ShowException (E: Exception); override;
  end;

var
  SourceF, DestF: File;
  SourceFOpen: Boolean;


{ TSetupUninstallLog }

procedure TSetupUninstallLog.ShowException (E: Exception);
begin
  Application.ShowException (E);
end;


{ TInstallForm }

constructor TInstallForm.Create (AOwner: TComponent);
var
  OldTextHeight, NewTextHeight: Integer;
  NewMaxValue: Longint;
  N: Integer;
begin
  inherited;

  SetFormFont (Self, OldTextHeight, NewTextHeight);
  Caption := SetupMessages[msgSetupAppTitle];

  { Center the form inside the MainForm's client area }
  MainForm.CenterForm (Self);

  CancelButton.Caption := SetupMessages[msgButtonCancel];

  { Calculate the MaxValue for the progress meter }
  NewMaxValue := 1000 * Entries[seIcon].Count;
  for N := 0 to Entries[seFile].Count-1 do
    with PSetupFileEntry(Entries[seFile][N])^ do
      if LocationEntry <> -1 then  { not an "external" file }
        Inc (NewMaxValue, PSetupFileLocationEntry(Entries[seFileLocation][
         LocationEntry])^.CompressedSize)
      else
        Inc (NewMaxValue, ExternalSize);
  if Entries[seIni].Count <> 0 then Inc (NewMaxValue, 1000);
  if Entries[seRegistry].Count <> 0 then Inc (NewMaxValue, 1000);
  ProgressGauge.MaxValue := NewMaxValue;
end;

procedure TInstallForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if not DisableCloseQuery then begin
    CanClose := False;
    if ExitSetupMsgBox then NeedToAbort := True;
  end;
end;

procedure TInstallForm.CancelButtonClick(Sender: TObject);
begin
  { Clicking Cancel will do the same thing as the Close button }
  Close;
end;

procedure TInstallForm.SetProgress (const AbsoluteValue: Boolean;
  const AProgress: Longint);
begin
  with ProgressGauge do begin
    if AbsoluteValue then
      Progress := AProgress
    else
      Progress := Progress + AProgress;
    Update;
  end;
end;

procedure ProcessEvents;
{ Processes any waiting events. Must call this this periodically or else
  events like clicking the Cancel button won't be processed.
  Calls Abort if NeedToAbort is True, which is usually the result of
  the user clicking Cancel and the form closing. }
begin
  if InstallForm.NeedToAbort then Abort;
  Application.ProcessMessages;
  if InstallForm.NeedToAbort then Abort;
end;

procedure SourceIsCorrupted;
begin
  raise Exception.Create(SetupMessages[msgSourceIsCorrupted]);
end;

procedure ReadDiskHeader;
var
  TestDiskID: TDiskID;
  DiskHeader: TDiskHeader;
begin
  BlockRead (SourceF, TestDiskID, SizeOf(TestDiskID));
  if TestDiskID <> DiskID then
    SourceIsCorrupted;
  BlockRead (SourceF, DiskHeader, SizeOf(DiskHeader));
  if FileSize(SourceF) <> DiskHeader.TotalSize then
    SourceIsCorrupted;
end;

var
  LastSourceDir: String;

function FindDisk (const DiskNumber: Integer): String;
var
  Path, F1, F2: String;
begin
  FmtStr (F1, '%s.%d', [SetupHeader.BaseFilename, DiskNumber]);
  FmtStr (F2, '..\DISK%d\', [DiskNumber]);
  F2 := F2 + F1;
  Result := AddBackslash(LastSourceDir) + F1;
  if FileExists(Result) then Exit;
  Result := AddBackslash(SourceDir) + F1;
  if FileExists(Result) then Exit;
  Result := ExpandFilename(AddBackslash(LastSourceDir) + F2);
  if FileExists(Result) then Exit;
  Result := ExpandFilename(AddBackslash(SourceDir) + F2);
  if FileExists(Result) then Exit;
  Path := SourceDir;
  if SelectDisk(DiskNumber, ExtractFilename(Result), Path) then begin
    LastSourceDir := Path;
    Result := AddBackslash(Path) + F1;
  end
  else
    Abort;
end;

type
  PDecompressExtraData = ^TDecompressExtraData;
  TDecompressExtraData = record
    CurDisk, LastDisk: Integer;
    BytesLeft, Size, BytesWritten: Longint;
  end;

function zlibReadProc (var Buf; MaxBytes: Cardinal; ExtraData: Longint): Cardinal; far;
var
  Buffer: Pointer;
  Left, Res: Longint; 
  NewDiskFilename: String;
begin
  Buffer := @Buf;
  with PDecompressExtraData(ExtraData)^ do begin
    Left := BytesLeft;
    if Left > Integer(MaxBytes) then
      Left := MaxBytes;
    Result := Left;
    while Left <> 0 do begin
      BlockRead (SourceF, Buffer^, Left, Res);
      Dec (PDecompressExtraData(ExtraData)^.BytesLeft, Res);
      if Left = Res then
        Break
      else begin
        Dec (Left, Res);
        Inc (Longint(Buffer), Res);
        { Go to next disk }
        if CurDisk = LastDisk then
          { Already on the last disk, so the setup file must be corrupted... }
          SourceIsCorrupted
        else begin
          CloseFile (SourceF);
          SourceFOpen := False;
          Inc (CurDisk);
          NewDiskFilename := FindDisk(CurDisk);
          AssignFile (SourceF, NewDiskFilename);
          FileMode := fmOpenRead or fmShareDenyWrite;  Reset (SourceF, 1);
          SourceFOpen := True;
          ReadDiskHeader;
        end;
      end;
    end;
  end;
  { Increment progress meter }
  InstallForm.SetProgress (False, Integer(Result));
  ProcessEvents;
end;
function zlibWriteProc (var Buf; BufSize: Cardinal; ExtraData: Longint): Cardinal; far;
begin
  with PDecompressExtraData(ExtraData)^ do begin
    if BytesWritten + Longint(BufSize) > Size then
      SourceIsCorrupted;
    BlockWrite (DestF, Buf, BufSize);
    Inc (BytesWritten, BufSize);
  end;
  Result := BufSize;
end;

function AbortRetryIgnoreMsgBox (const Text1, Text2: String): Boolean;
{ Returns True if Ignore was selected, False if Retry was selected, or
  calls Abort if Abort was selected. }
begin
  Result := False;
  case MsgBox(Text1 + SNewLine2 + Text2, '', mbError, MB_ABORTRETRYIGNORE) of
    ID_RETRY: ;
    ID_IGNORE: Result := True;
  else
    Abort;
  end;
end;

procedure CopySourceFileToDestFile;
{ Copies all remaining bytes from SourceF to DestF, incrementing process meter
  as it goes. Assumes both files are opened with a record size of 1. }
type
  PCopyBuffer = ^TCopyBuffer;
  TCopyBuffer = array[0..32767] of Byte;
var
  Buffer: PCopyBuffer;
  NumRead: Cardinal;
begin
  New (Buffer);
  try
    while not Eof(SourceF) do begin
      BlockRead (SourceF, Buffer^, SizeOf(TCopyBuffer), NumRead);
      BlockWrite (DestF, Buffer^, NumRead);
      InstallForm.SetProgress (False, NumRead);
      ProcessEvents;
    end;
  finally
    Dispose (Buffer);
  end;
end;

procedure TInstallForm.StartInstalling;
type
  PRegisterFilesListRec = ^TRegisterFilesListRec;
  TRegisterFilesListRec = record
    Filename: String;
    TypeLib: Boolean;
  end;
var
  UninstLog: TSetupUninstallLog;
  UninstallIconNeeded: Boolean;
  UninstallExeFilename, UninstallTempExeFilename, UninstallDataFilename,
    UninstallMsgFilename, RegSvrExeFilename: String;
  UninstallExeCreated: (ueNone, ueNew, ueReplaced);
  UninstallDataCreated, UninstallMsgCreated, AppendUninstallData: Boolean;
  RegisterFilesList: TList;

  function GetLocalTimeAsStr: String;
  var
    SysTime: TSystemTime;
  begin
    GetLocalTime (SysTime);
    SetString (Result, PChar(@SysTime), SizeOf(SysTime));
  end;

  procedure RecordStartInstall;
  var
    AppDir: String;
  begin
    if shCreateAppDir in SetupHeader.Options then
      AppDir := WizardDirValue;
    UninstLog.Add (utStartInstall, [GetComputerNameString, GetUserNameString,
      AppDir, GetLocalTimeAsStr], 0);
  end;

  procedure RegisterUninstallInfo;
  { Stores uninstall information in the Registry so that the program can be
    uninstalled through the Control Panel Add/Remove Programs applet. }
  var
    RootKey, H, H2: HKEY;
    Disp: DWORD;
    S, S2, Z: String;

    procedure RegError (const Msg: TSetupMessageID; const KeyName: String);
    begin
      raise Exception.Create(FmtSetupMessage(Msg, [RegRootKeyNames[RootKey],
        KeyName]));
    end;

    procedure SetStringValue (const K: HKEY; const ValueName: PChar;
      const Data: String);
    begin
      if RegSetValueEx(K, ValueName, 0, REG_SZ, PChar(Data),
         Length(Data)+1) <> ERROR_SUCCESS then
        RegError (msgErrorRegWriteKey, S2);
    end;

    procedure DeleteOldKeys (const K: HKEY);
    { Pre-1.3.6 versions of Inno Setup supported creation of multiple uninstall
      keys with the same name but different suffixes, such as "_is1",
      "_is2", "_is3", etc. when an application was installed more than once.
      Version 1.08 introduced this "feature" but I and users of Inno Setup
      quickly realized it was very annoying. Version 1.09 made it optional via
      a [Setup] section directive called "OverwriteUninstRegEntries". In version
      1.3.6, this feature has been removed entirely (it didn't coexist with
      UsePreviousAppDir and other new 1.3.x features), but we still check for
      and delete any "old" keys with suffixes of "_is2" and higher. }
    var
      M, I, J, Numbers: Integer;
      Buf: array[0..4095] of Char;
      P: PChar;
      Count: Integer;
      N: String;
    begin
      M := Length(UninstallRegKeyBaseName) + 3;
      { ^ Minimum length of key name to look at. Length of UninstallRegKeyBaseName
        plus length of "_is". }
      I := 0;
      while True do begin
        Count := SizeOf(Buf);
        if RegEnumKeyEx(K, I, Buf, DWORD(Count), nil, nil, nil, nil) <> ERROR_SUCCESS then
          Break;
        if (Count >= M+1) and (Count <= M+3) then begin  { Number in suffix can have 1 to 3 digits }
          { Count the numbers at the end of the key name }
          Numbers := 0;
          P := @Buf[Count];
          for J := 1 to 4 do begin
            Dec (P);  { First pass will compare the last character (Count-1) }
            if P^ in ['0'..'9'] then
              Inc (Numbers)
            else
              Break;
          end;
          if (Numbers >= 1) and (Numbers <= 3) and  { Between 0 and 999? }
             (Count = M + Numbers) then begin  { Make sure the following Dec won't go out of bounds } 
            Dec (P, 2);
            if (P[0] = '_') and (P[1] = 'i') and (P[2] = 's') then begin
              N := StrPas(Buf);
              P[0] := #0;  { Truncate starting at '_' in Buf }
              if UninstallRegKeyBaseName = StrPas(Buf) then begin
                { ^ Does it have the same base name? }
                if RegDeleteKeyIncludingSubkeys(K, PChar(N)) then
                  Dec (I);
                  { ^ Deny the Inc(I) below. We can't have it increment when it
                    successfully deleted a key, or else RegEnumKeyEx will skip
                    a key }
              end;
            end;
          end;
        end;
        Inc (I);
      end;
    end;

  begin
    if IsAdmin then
      RootKey := HKEY_LOCAL_MACHINE
    else
      RootKey := HKEY_CURRENT_USER;
    if RegCreateKeyEx(RootKey, NEWREGSTR_PATH_UNINSTALL, 0, nil,
       REG_OPTION_NON_VOLATILE, KEY_QUERY_VALUE or KEY_ENUMERATE_SUB_KEYS,
       nil, H, @Disp) <> ERROR_SUCCESS then
      RegError (msgErrorRegCreateKey, NEWREGSTR_PATH_UNINSTALL);
    H2 := 0;
    try
      { Delete any uninstall keys left over from previous installs }
      if IsAdmin then begin
        { Delete any keys under HKEY_CURRENT_USER too }
        if RegOpenKeyEx(HKEY_CURRENT_USER, NEWREGSTR_PATH_UNINSTALL, 0,
           KEY_QUERY_VALUE or KEY_ENUMERATE_SUB_KEYS, H2) = ERROR_SUCCESS then begin
          DeleteOldKeys (H2);
          RegCloseKey (H2);
          H2 := 0;
        end
        else
          H2 := 0;
      end;
      DeleteOldKeys (H);

      { Create uninstall key }
      S := UninstallRegKeyBaseName + '_is1';
      S2 := NEWREGSTR_PATH_UNINSTALL + '\' + S;
      if RegCreateKeyEx(H, PChar(S), 0, nil, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE,
         nil, H2, @Disp) <> ERROR_SUCCESS then begin
        H2 := 0;
        RegError (msgErrorRegCreateKey, S2);
      end;
      { do not localize or change any of the following strings }
      SetStringValue (H2, 'Inno Setup: Setup Version', SetupVersion);
      if shCreateAppDir in SetupHeader.Options then
        Z := WizardDirValue
      else
        Z := '';
      SetStringValue (H2, 'Inno Setup: App Path', Z);
      SetStringValue (H2, 'Inno Setup: Icon Group', WizardGroupValue);
      SetStringValue (H2, 'Inno Setup: User', GetUserNameString);
      if SetupHeader.UninstallDisplayName <> '' then
        Z := SetupHeader.UninstallDisplayName
      else
        Z := SetupHeader.AppVerName;
      { Note: DisplayName can't exceed 63 chars on Win9x }
      SetStringValue (H2, 'DisplayName', Copy(Z, 1, 63));
      if SetupHeader.UninstallDisplayIcon <> '' then
        SetStringValue (H2, 'DisplayIcon', ChangeDirConst(SetupHeader.UninstallDisplayIcon));
      SetStringValue (H2, 'UninstallString', AddQuotes(UninstallExeFilename));
      if SetupHeader.AppVersion <> '' then
        SetStringValue (H2, 'DisplayVersion', SetupHeader.AppVersion);
      if SetupHeader.AppPublisher <> '' then
        SetStringValue (H2, 'Publisher', SetupHeader.AppPublisher);
      if SetupHeader.AppPublisherURL <> '' then
        SetStringValue (H2, 'URLInfoAbout', SetupHeader.AppPublisherURL);
      if SetupHeader.AppSupportURL <> '' then
        SetStringValue (H2, 'HelpLink', SetupHeader.AppSupportURL);
      if SetupHeader.AppUpdatesURL <> '' then
        SetStringValue (H2, 'URLUpdateInfo', SetupHeader.AppUpdatesURL);
    finally
      if H2 <> 0 then
        RegCloseKey (H2);
      RegCloseKey (H);
    end;

    UninstLog.Add (utRegDeleteEntireKey, [S2], Integer(RootKey));
  end;

  function GenerateRegSvrExeFilename: String;
  { Generate a name for the REGSVR program used to register OLE servers & type
    libraries on the next reboot }
    function TryDir (const Dir: String): Boolean;
    var
      F: File;
    begin
      RegSvrExeFilename := GenerateUniqueName(Dir, '.exe');
      Result := True;
      try
        { Test if it has write permission to the directory, and also reserve
          the filename }
        AssignFile (F, RegSvrExeFilename);
        FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F, 1);
        CloseFile (F);
      except
        { trap any exceptions }
        Result := False;
      end;
    end;
  begin
    if RegSvrExeFilename = '' then begin
      if not TryDir(WinDir) then
        { In case Windows directory is write protected, try the Temp directory.
          Windows directory is our first choice since some people (ignorantly)
          put things like "DELTREE C:\WINDOWS\TEMP\*.*" in their AUTOEXEC.BAT }          
        TryDir (GetTempDir);
    end;
    Result := RegSvrExeFilename;
  end;

  type
    TMakeDirFlags = set of (mdNoUninstall, mdAlwaysUninstall, mdDeleteAfterInstall,
      mdNotifyChange);
  procedure MakeDir (Dir: String; const Flags: TMakeDirFlags);
  const
    DeleteDirFlags: array[Boolean] of Longint = (0, utDeleteDirOrFiles_CallChangeNotify);
  begin
    Dir := RemoveBackslash(Dir);
    if (Dir = '') or (Dir[Length(Dir)] = ':') or (ExtractFilePath(Dir) = Dir) then
      Exit;
    if DirExists(Dir) then begin
      if not(mdAlwaysUninstall in Flags) then
        Exit;
    end
    else begin
      MakeDir (ExtractFilePath(Dir), Flags - [mdAlwaysUninstall]);
      Dir := ExpandFilename(Dir);
      if not CreateDirectory(PChar(Dir), nil) then
        raise Exception.Create(FmtSetupMessage1(msgErrorCreatingDir, Dir));
      if mdNotifyChange in Flags then begin
        SHChangeNotify (SHCNE_MKDIR, SHCNF_PATH, PChar(Dir), nil);
        SHChangeNotify (SHCNE_UPDATEDIR, SHCNF_PATH or SHCNF_FLUSH,
          PChar(RemoveBackslashUnlessRoot(ExtractFilePath(Dir))), nil);
      end;
    end;
    if mdDeleteAfterInstall in Flags then
      DeleteDirsAfterInstallList.Add (Dir)
    else
      if not(mdNoUninstall in Flags) then
        UninstLog.Add (utDeleteDirOrFiles, [Dir],
          utDeleteDirOrFiles_IsDir or DeleteDirFlags[mdNotifyChange in Flags]);
  end;

  procedure CreateDirs;
  { Creates the application's directories }
  var
    CurDirNumber: Integer;
    Flags: TMakeDirFlags;
  begin
    { Create main application directory }
    MakeDir (WizardDirValue, []);

    { Create the rest of the directories, if any }
    for CurDirNumber := 0 to Entries[seDir].Count-1 do
      with PSetupDirEntry(Entries[seDir][CurDirNumber])^ do begin
        Flags := [];
        if doUninsNeverUninstall in Options then Include (Flags, mdNoUninstall);
        if doDeleteAfterInstall in Options then Include (Flags, mdDeleteAfterInstall);
        if doUninsAlwaysUninstall in Options then Include (Flags, mdAlwaysUninstall);
        MakeDir (ChangeDirConst(DirName), Flags);
      end;
  end;

  procedure ProcessFileEntry (const CurFile: PSetupFileEntry;
    ASourceFile, ADestName: String; const FileLocationFilenames: TStringList);

    function InstallFont (const Filename, FontName: String;
      const FontIsTrueType, AddToFontTableNow: Boolean): String;
    var
      FontInFontDir: Boolean;
      K: HKEY;
      LastError: DWORD;
    begin
      FontInFontDir := ExtractFilePath(Filename) = AddBackslash(FontDir);
      if FontInFontDir then
        Result := ExtractFileName(Filename)
      else
        Result := Filename;
      if IsNT then
        SetIniString ('Fonts', FontName, Result, '')
      else
        if RegOpenKeyEx(HKEY_LOCAL_MACHINE, NEWREGSTR_PATH_SETUP + '\Fonts', 0,
           KEY_SET_VALUE, K) = ERROR_SUCCESS then begin
          RegSetValueEx (K, PChar(FontName), 0, REG_SZ, PChar(Result),
            Length(Filename)+1);
          RegCloseKey (K);
        end;

      if AddToFontTableNow then begin
        LastError := 0;
        repeat
          if AddFontResource(PChar(Result)) <> 0 then begin
            SendNotifyMessage (HWND_BROADCAST, WM_FONTCHANGE, 0, 0);
            Break;
          end;
          LastError := GetLastError;
        until AbortRetryIgnoreMsgBox(FmtSetupMessage(msgErrorFunctionFailed,
          ['AddFontResource', IntToStr(LastError)]),
          SetupMessages[msgEntryAbortRetryIgnore]);
      end;
    end;

    procedure SetFileLocationFilename (const LocationEntry: Integer;
      Filename: String);
    var
      I: Integer;
    begin
      Filename := ExpandFilename(Filename);
      { If Filename was already associated with another LocationEntry,
        disassociate it. If we *don't* do this, then this script won't
        produce the expected result:
          [Files]
          Source: "file1"; DestName: "file2"
          Source: "file2"; DestName: "file2"
          Source: "file1"; DestName: "file1"
        1. It extracts file1 under the name "file2"
        2. It extracts file2 under the name "file2"
        3. It copies file2 to file1 -- oops!
      }
      for I := 0 to FileLocationFilenames.Count-1 do
        if AnsiCompareText(FileLocationFilenames[I], Filename) = 0 then begin
          FileLocationFilenames[I] := '';
          Break;
        end;
      { Expand FileLocationFilenames if needed }
      while FileLocationFilenames.Count <= LocationEntry do
        FileLocationFilenames.Add ('');
      FileLocationFilenames[LocationEntry] := Filename;
    end;

  var
    ProgressUpdated: Boolean;
    PreviousProgress: Longint;
    LastOperation: String;
    CurFileLocation: PSetupFileLocationEntry;
    SourceFile, DestFile, TempFile, ShortFontFilename: String;
    DestFileAlreadyExisted, TempFileLeftOver, SourceFileIsLiteralFile, ReplaceOnRestart: Boolean;
    TestCompID: TCompID;
    ExistingFileAttr: Integer;
    Failed: String;
    CurFileVersionInfoValid: Boolean;
    CurFileVersionInfo, ExistingVersionInfo: TFileVersionNumbers;
    CurFileDateValid: Boolean;
    DecompressData: TDecompressExtraData;
    Adler32: Longint;
    DeleteFlags: Longint;
    CurFileDate, ExistingFileDate, CurFileDate2: TFileTime;
    RegisterRec: PRegisterFilesListRec;
  label Retry, Skip, SkipAll;
  begin
    TempFileLeftOver := False;
    try
      if CurFile^.LocationEntry <> -1 then
        CurFileLocation := PSetupFileLocationEntry(Entries[seFileLocation][CurFile^.LocationEntry])
      else
        CurFileLocation := nil;

    Retry:
      ProgressUpdated := False;
      PreviousProgress := ProgressGauge.Progress;
      try
        ReplaceOnRestart := False;
        DeleteFlags := 0;
        LastOperation := '';
        ShortFontFilename := '';

        if TempFileLeftOver then begin
          { This is usually only called if some error occured }
          DeleteFile (TempFile);
          TempFileLeftOver := False;
          TempFile := '';
        end;

        case CurFile^.FileType of
          ftUninstExe: DestFile := UninstallExeFilename;
          ftRegSvrExe: begin
              if (RegisterFilesList.Count = 0) or not NeedsRestart then
                goto SkipAll;
              DestFile := GenerateRegSvrExeFilename;
            end;
        else
          DestFile := ChangeDirConst(CurFile^.DestName);
          if ADestName <> '' then
            DestFile := ExtractFilePath(DestFile) + ADestName;
        end;
        DestFileAlreadyExisted := FileExists(DestFile);
        if Assigned(CurFileLocation) then begin
          CurFileDate := CurFileLocation^.Date;
          CurFileDateValid := True;
          CurFileVersionInfoValid := foVersionInfoValid in CurFileLocation^.Flags;
          CurFileVersionInfo.MS := CurFileLocation^.FileVersionMS;
          CurFileVersionInfo.LS := CurFileLocation^.FileVersionLS;
        end
        else begin
          CurFileDateValid := GetFileLocalDateTime(ASourceFile, CurFileDate);
          CurFileVersionInfoValid := GetVersionNumbers(ASourceFile, CurFileVersionInfo);
        end;

        { Update the status labels }
        FilenameLabel.Caption := DestFile;  FilenameLabel.Update;

        if DestFileAlreadyExisted then begin
          if CurFile^.CopyMode = cmIfDoesntExist then
            goto Skip;

          LastOperation := SetupMessages[msgErrorReadingExistingDest];

          { If the existing file has version info, make sure it's not
            installing an older version over existing file }
          if (CurFile^.CopyMode <> cmAlwaysOverwrite) and
             GetVersionNumbers(DestFile, ExistingVersionInfo) then begin
            { If the file being installed has no version info, or the existing
              file is a newer version... }
            if not CurFileVersionInfoValid or
               ((ExistingVersionInfo.MS > CurFileVersionInfo.MS) or
                ((ExistingVersionInfo.MS = CurFileVersionInfo.MS) and
                 (ExistingVersionInfo.LS > CurFileVersionInfo.LS))) then begin
              if (CurFile^.CopyMode = cmAlwaysSkipIfSameOrOlder) or
                 (MsgBox(DestFile + SNewLine2 + SetupMessages[msgExistingFileNewer],
                  '', mbError, MB_YESNO) = ID_YES) then
                goto Skip;
            end
            else
            { If the existing file and the file being installed are the same
              version... }
            if (ExistingVersionInfo.MS = CurFileVersionInfo.MS) and
               (ExistingVersionInfo.LS = CurFileVersionInfo.LS) and
               not(foOverwriteSameVersion in CurFile^.Options) then begin
              if not(foCompareTimeStampAlso in CurFile^.Options) then
                goto Skip;
              if CurFileDateValid then begin
                if GetFileLocalDateTime(DestFile, ExistingFileDate) then begin
                  if CompareFileTime(ExistingFileDate, CurFileDate) = 0 then
                    goto Skip
                  else
                  if CompareFileTime(ExistingFileDate, CurFileDate) > 0 then begin
                    if (CurFile^.CopyMode = cmAlwaysSkipIfSameOrOlder) or
                       (MsgBox(DestFile + SNewLine2 + SetupMessages[msgExistingFileNewer],
                        '', mbError, MB_YESNO) = ID_YES) then
                      goto Skip;
                  end;
                end;
              end;
            end;
          end;

          if (foConfirmOverwrite in CurFile^.Options) and
             (MsgBox(DestFile + SNewLine2 + SetupMessages[msgFileExists],
              '', mbConfirmation, MB_YESNO) <> ID_YES) then
             { ^ If file already exists and foConfirmOverwrite is in Options, ask the
               user if it's OK to overwrite }
            goto Skip;

          { Check if existing file is read-only }
          while True do begin
            ExistingFileAttr := GetFileAttributes(PChar(DestFile));
            if (ExistingFileAttr >= 0) and
               (ExistingFileAttr and faReadOnly <> 0) then begin
              if not(foOverwriteReadOnly in CurFile^.Options) and
                 AbortRetryIgnoreMsgBox(DestFile, SetupMessages[msgExistingFileReadOnly]) then
                goto Skip;
              LastOperation := SetupMessages[msgErrorChangingAttr];
              SetFileAttributes (PChar(DestFile), ExistingFileAttr and not faReadOnly);
            end
            else
              Break;
          end;
        end
        else begin
          if foOnlyIfDestFileExists in CurFile^.Options then
            goto Skip;
        end;

        { Locate source file }
        SourceFile := ASourceFile;
        if SourceFile <> '' then
          SourceFileIsLiteralFile := True
        else begin
          { The file is compressed in the setup package... }
          if (CurFile^.LocationEntry < FileLocationFilenames.Count) and
             (FileLocationFilenames[CurFile^.LocationEntry] <> '') and
             FileExists(FileLocationFilenames[CurFile^.LocationEntry]) then begin
            { ^ Has the same file already been copied somewhere else? If so,
              just make a duplicate of that file instead of extracting it over
              again. }
            SourceFileIsLiteralFile := True;
            SourceFile := FileLocationFilenames[CurFile^.LocationEntry];
          end
          else begin
            SourceFileIsLiteralFile := False;
            if SetupLdrOffset1 = 0 then
              SourceFile := FindDisk(CurFileLocation^.FirstDisk)
            else
              SourceFile := SetupLdrOriginalFilename;
          end;
        end;

        { Create a temporary file to copy file to. Create the destination
          file's directory if it didn't already exist. }
        LastOperation := SetupMessages[msgErrorCreatingTemp];
        TempFile := GenerateUniqueName(ExtractFilePath(ExpandFilename(DestFile)), '.tmp');
        MakeDir (ExtractFilePath(TempFile), [mdNoUninstall]);
        AssignFile (DestF, TempFile);
        FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (DestF, 1);
        try
          TempFileLeftOver := True;
          { Open source file }
          LastOperation := SetupMessages[msgErrorReadingSource];
          AssignFile (SourceF, SourceFile);
          FileMode := fmOpenRead or fmShareDenyWrite;  Reset (SourceF, 1);
          SourceFOpen := True;
          try
            { Decompress the file, or simply copy it if it's an "external" file }
            LastOperation := SetupMessages[msgErrorCopying];
            if not SourceFileIsLiteralFile then begin
              if SetupLdrOffset1 = 0 then
                ReadDiskHeader;
              Seek (SourceF, SetupLdrOffset1 + CurFileLocation^.StartOffset);
              BlockRead (SourceF, TestCompID, SizeOf(TestCompID));
              if TestCompID <> ZLIBID then
                SourceIsCorrupted;
              with DecompressData do begin
                CurDisk := CurFileLocation^.FirstDisk;
                LastDisk := CurFileLocation^.LastDisk;
                BytesLeft := CurFileLocation^.CompressedSize;
                Size := CurFileLocation^.OriginalSize;
                BytesWritten := 0;
              end;
            end;
            try
              ProgressUpdated := True;  {zlibReadProc increments progress as it goes}
              if not SourceFileIsLiteralFile then begin
                try
                  CustomInflateData (zlibReadProc, zlibWriteProc, Longint(@DecompressData), Adler32);
                except
                  on EZlibDataError do
                    SourceIsCorrupted;
                end;
                if (DecompressData.BytesWritten <> CurFileLocation^.OriginalSize) or
                   (Adler32 <> CurFileLocation^.Adler) then
                  SourceIsCorrupted;
              end
              else
                CopySourceFileToDestFile;
            except
              { If an exception occurred, put progress meter back to where it was }
              ProgressUpdated := False;
              SetProgress (True, PreviousProgress);
              raise;
            end;
            { Set time/date stamp }
            LocalFileTimeToFileTime (CurFileDate, CurFileDate2);
            SetFileTime (TFileRec(DestF).Handle, @CurFileDate2, nil, @CurFileDate2);
          finally
            if SourceFOpen then
              CloseFile (SourceF);
          end;
        finally
          CloseFile (DestF);
        end;

        { Delete existing version of file, if any. If it can't be deleted
          because it's in use and the "restartreplace" flag was specified
          on the entry, register it to be replaced when the system is
          restarted. }
        if DestFileAlreadyExisted and (CurFile^.FileType <> ftUninstExe) then begin
          LastOperation := SetupMessages[msgErrorReplacingExistingFile];
          AssignFile (DestF, DestFile);
          if not(foRestartReplace in CurFile^.Options) then
            Erase (DestF)  { Erase creates exceptions, DeleteFile does not }
          else
            try
              Erase (DestF);  { Erase creates exceptions, DeleteFile does not }
            except
              on E: EInOutError do begin
                { 5 = ERROR_ACCESS_DENIED, 32 = ERROR_SHARING_VIOLATION }
                if (E.ErrorCode <> 5) and (E.ErrorCode <> 32) then
                  raise;
                LastOperation := SetupMessages[msgErrorRestartReplace];
                NeedsRestart := True;
                RestartReplace (TempFile, DestFile);
                ReplaceOnRestart := True;
              end;
            end;
        end;

        { Set file attributes (on the temp. file) }
        ExistingFileAttr := GetFileAttributes(PChar(TempFile));
        if ExistingFileAttr >= 0 then
          SetFileAttributes (PChar(TempFile), ExistingFileAttr or CurFile^.Attribs);

        { Rename the temporary file to the new name now, unless the file is
          to be replaced when the system is restarted, or if the file is the
          uninstall program and an existing uninstall program already exists. }
        if ReplaceOnRestart or
           ((CurFile^.FileType = ftUninstExe) and DestFileAlreadyExisted) then begin
          if CurFile^.FileType = ftUninstExe then
            UninstallTempExeFilename := TempFile;
          TempFileLeftOver := False;
          if not SourceFileIsLiteralFile then
            SetFileLocationFilename (CurFile^.LocationEntry, TempFile);
        end
        else begin
          LastOperation := SetupMessages[msgErrorRenamingTemp];
          AssignFile (DestF, TempFile);
          Rename (DestF, DestFile);  { Rename creates exceptions, RenameFile does not }
          TempFileLeftOver := False;
          TempFile := '';
          if not SourceFileIsLiteralFile then
            SetFileLocationFilename (CurFile^.LocationEntry, DestFile);
          if foDeleteAfterInstall in CurFile^.Options then
            DeleteFilesAfterInstallList.Add (DestFile);
        end;

        { If it's a font, register it }
        ShortFontFilename := '';
        if (CurFile^.InstallFontName <> '') and
           not(foDeleteAfterInstall in CurFile^.Options) then begin
          ShortFontFilename := InstallFont(DestFile,
            CurFile^.InstallFontName,
            not(foFontIsntTrueType in CurFile^.Options), not ReplaceOnRestart);
          DeleteFlags := DeleteFlags or utDeleteFile_IsFont;
        end;

        { There were no errors ... }
        LastOperation := '';
        if CurFile^.FileType <> ftUninstExe then begin
          { ... so add to undo list, unless it's the uninstall program. But
            don't add foSharedFile files to the undo list yet. }
          if not(foUninsNeverUninstall in CurFile^.Options) and
             not(foSharedFile in CurFile^.Options) then begin
            if DestFileAlreadyExisted then
              DeleteFlags := DeleteFlags or utDeleteFile_ExistedBeforeInstall;
            if foRegisterServer in CurFile^.Options then
              DeleteFlags := DeleteFlags or utDeleteFile_RegisteredServer;
            if foRegisterTypeLib in CurFile^.Options then
              DeleteFlags := DeleteFlags or utDeleteFile_RegisteredTypeLib;
            UninstLog.Add (utDeleteFile, [DestFile, TempFile,
              CurFile^.InstallFontName, ShortFontFilename], DeleteFlags);
          end;
        end
        else begin
          if not DestFileAlreadyExisted then
            UninstallExeCreated := ueNew
          else
            UninstallExeCreated := ueReplaced;
        end;

      Skip:
        { If foRegisterServer or foRegisterTypeLib is in Options, add the
          file to RegisterFilesList for registering later }
        if (foRegisterServer in CurFile^.Options) or
           (foRegisterTypeLib in CurFile^.Options) then begin
          LastOperation := '';
          New (RegisterRec);
          RegisterRec^.Filename := DestFile;
          RegisterRec^.TypeLib := foRegisterTypeLib in CurFile^.Options;
          RegisterFilesList.Add (RegisterRec);
        end;

        { If foSharedFile is in Options, increment the reference count in the
          registry for the file (supported by Win95/NT 4 only) }
        if (foSharedFile in CurFile^.Options) and UsingWindows4 then begin
          LastOperation := '';
          IncrementSharedCount (DestFile, DestFileAlreadyExisted);
          if not(foUninsNeverUninstall in CurFile^.Options) then begin
            DeleteFlags := DeleteFlags or utDeleteFile_SharedFile;
            if foRegisterServer in CurFile^.Options then
              DeleteFlags := DeleteFlags or utDeleteFile_RegisteredServer;
            if foRegisterTypeLib in CurFile^.Options then
              DeleteFlags := DeleteFlags or utDeleteFile_RegisteredTypeLib;
            UninstLog.Add (utDeleteFile, [DestFile, TempFile,
              CurFile^.InstallFontName, ShortFontFilename], DeleteFlags);
          end
          else
            UninstLog.Add (utDecrementSharedCount, [DestFile], 0);
        end;

      SkipAll:
        Failed := '';
      except
        on EAbort do raise;
        on E: Exception do Failed := E.Message;
      end;
      if TempFileLeftOver then begin
        { This is usually called if some error occured }
        DeleteFile (TempFile);
        TempFileLeftOver := False;
        TempFile := '';
      end;
      if Failed <> '' then begin
        if LastOperation <> '' then
          LastOperation := LastOperation + SNewLine;
        if not AbortRetryIgnoreMsgBox(DestFile + SNewLine2 + LastOperation + AddPeriod(Failed),
           SetupMessages[msgFileAbortRetryIgnore]) then begin
          if ProgressUpdated then
            SetProgress (True, PreviousProgress);
          goto Retry;
        end;
      end;

      { Increment progress meter, if not already done so }
      if not ProgressUpdated then begin
        if Assigned(CurFileLocation) then  { not an "external" file }
          SetProgress (False, CurFileLocation^.CompressedSize)
        else
          SetProgress (False, CurFile^.ExternalSize);
      end;

      { Process any events between copying files }
      ProcessEvents;
    finally
      if TempFileLeftOver then
        { This is usually only called if some error occured }
        DeleteFile (TempFile);
    end;
  end;

  procedure CopyFiles;
  { Copies all the application's files }
  var
    FileLocationFilenames: TStringList;
    CurFileNumber: Integer;
    CurFile: PSetupFileEntry;
    SourceWildcard, N, D: String;
    H: THandle;
    FindData: TWin32FindData;
    FoundFiles: Boolean;
  label Retry;
  begin
    LastSourceDir := SourceDir;
    FileLocationFilenames := TStringList.Create;
    try
      for CurFileNumber := 0 to Entries[seFile].Count-1 do begin
        CurFile := PSetupFileEntry(Entries[seFile][CurFileNumber]);
        if CurFile^.LocationEntry <> -1 then
          ProcessFileEntry (CurFile, '', '', FileLocationFilenames)
        else begin
          { File is an 'external' file }
          FoundFiles := False;
          SourceWildcard := ChangeDirConst(CurFile^.SourceFilename);
          Retry:
          H := FindFirstFile(PChar(SourceWildcard), FindData);
          if H <> INVALID_HANDLE_VALUE then begin
            repeat
              if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY = 0 then begin
                FoundFiles := True;
                N := FindData.cFileName;
                if not(foCustomDestName in CurFile^.Options) then
                  D := N
                else
                  D := ExtractFileName(ChangeDirConst(CurFile^.DestName));
                ProcessFileEntry (CurFile, ExtractFilePath(SourceWildcard) + N,
                  D, FileLocationFilenames);
              end;
            until not FindNextFile(H, FindData);
            WinProcs.FindClose (H);
          end;
          if not FoundFiles and
             not(foSkipIfSourceDoesntExist in CurFile^.Options) and
             not AbortRetryIgnoreMsgBox(SetupMessages[msgErrorReadingSource] + SNewLine +
               AddPeriod(FmtSetupMessage(msgSourceDoesntExist, [SourceWildcard])),
               SetupMessages[msgFileAbortRetryIgnore]) then
            goto Retry;
        end;
      end;
    finally
      FileLocationFilenames.Free;
    end;
  end;

  procedure CreateIcons;
  { Creates the program's group and icons in Program Manager using Dynamic Data
    Exchange. But when compiling with Win32 and the "new shell" is detected, it
    creates the group using MkDir and the icons using COM. }
  var
    Conv: TDDEConversation;
    ShellLinks: Boolean;
    CreateGroupCmd: String;

    procedure CreateAnIcon (Name: String; const Description, Path, Parameters,
      WorkingDir, IconFilename: String; const IconIndex, ShowCmd: Integer;
      const NeverUninstall: Boolean; const CloseOnExit: TSetupIconCloseOnExit);
    var
      Cmd: String;

      procedure AppendCmd (const AFormat: PChar; const Args: array of const);
      begin
        Cmd := Cmd + Format(AFormat, Args);
      end;

    var
      BeginsWithGroup: Boolean;
      LinkFilename, PifFilename: String;
      Flags: TMakeDirFlags;
    begin
      BeginsWithGroup := Copy(Name, 1, 8) = '{group}\';
      if not ShellLinks then begin
        if BeginsWithGroup then
          Delete (Name, 1, 8);
        {don't localize these}
        DDE.Execute (Conv, '[ReplaceItem("' + Name + '")]', True);
        Cmd := CreateGroupCmd;
        { On NT 3.x, quotes can ONLY be included around the pathname if it
          contains spaces (otherwise parameters won't show up right). }
        AppendCmd ('[AddItem(%s', [AddQuotes(Path)]);
        if Parameters <> '' then
          AppendCmd (' %s', [Parameters]);
        AppendCmd (',"%s"', [Name]);
        if IconFilename <> '' then
          AppendCmd (',"%s",%d,,,', [IconFilename, IconIndex])
        else
          Cmd := Cmd + ',,,,,';
        if WorkingDir <> '' then
          AppendCmd ('"%s"', [WorkingDir]);
        if ShowCmd = SW_SHOWMINNOACTIVE then
          Cmd := Cmd + ',,1';
        Cmd := Cmd + ')]';
        DDE.Execute (Conv, Cmd, False);
        if not NeverUninstall then
          UninstLog.Add (utDeleteGroupOrItem, [WizardGroupValue, Name], 0);
      end
      else begin
        Name := ChangeDirConst(Name);
        LinkFilename := Name + '.lnk';
        PifFilename := Name + '.pif';
        Flags := [mdNotifyChange];
        if NeverUninstall or not BeginsWithGroup then
          Include (Flags, mdNoUninstall)
        else
          Include (Flags, mdAlwaysUninstall);
        MakeDir (ExtractFilePath(LinkFilename), Flags);
        { Delete any old .lnk and .pif files so that the FileExists checks
          below will work }
        DeleteFile (LinkFilename);
        DeleteFile (PifFilename);
        { Create the shortcut, and apply the "Close on exit" setting, if
          applicable.
          The reason for the FileExists calls below is that the IShellLink
          interface that CreateShellLink uses has an irritating "feature" --
          when creating a shortcut to an MS-DOS application, without warning
          it will change the file extension to .pif, and save the shortcut
          in .pif format. The only reliable way to check if it saved it as
          a .pif is to check if a .pif file exists (IPersistFile::GetCurFile
          unfortunately doesn't work on Win95; it always returns NULL). }
        if CreateShellLink(LinkFilename, Description, Path, Parameters, WorkingDir,
           IconFilename, IconIndex, ShowCmd) and (CloseOnExit <> icNoSetting) and
           not FileExists(LinkFilename) and FileExists(PifFilename) then begin
          try
            ModifyPifFile (PifFilename, CloseOnExit = icYes);
          except
            { Failure isn't important here. Ignore exceptions }
          end;
        end;
        SHChangeNotify (SHCNE_CREATE, SHCNF_PATH, PChar(LinkFilename), nil);
        SHChangeNotify (SHCNE_UPDATEDIR, SHCNF_PATH or SHCNF_FLUSH,
          PChar(RemoveBackslashUnlessRoot(ExtractFilePath(LinkFilename))), nil);
        if not NeverUninstall then begin
          UninstLog.Add (utDeleteFile, [LinkFilename, ''], utDeleteFile_CallChangeNotify);
          { On Win95/98, if a .lnk file points to an MS-DOS program or batch
            file it converts it to a .pif. There doesn't seem to be any good way
            to check if it did this, so just have the uninstaller attempt to
            delete a .pif file of the same name. }
          UninstLog.Add (utDeleteFile, [PifFilename], utDeleteFile_CallChangeNotify);
        end;
      end;

      { Increment progress meter }
      SetProgress (False, 1000);
    end;

    function ExpandAppPath (const Filename: String): String;
    const
      AppPathsBaseKey = NEWREGSTR_PATH_SETUP + '\App Paths\';
    var
      K: HKEY;
      Found: Boolean;
    begin
      if RegOpenKeyEx(HKEY_LOCAL_MACHINE, PChar(AppPathsBaseKey + Filename),
         0, KEY_QUERY_VALUE, K) = ERROR_SUCCESS then begin
        Found := RegQueryStringValue(K, '', Result);
        RegCloseKey (K);
        if Found then
          Exit;
      end;
      Result := Filename;
    end;

  var
    CreateUninstallIcon: Boolean;
    PROGMAN: TDDEString;
    CurIconNumber: Integer;
    Z, FN: String;
  begin
    ShellLinks := UsingWindows4;
    CreateUninstallIcon := (shUninstallable in SetupHeader.Options) and
      ((shAlwaysCreateUninstallIcon in SetupHeader.Options) or UninstallIconNeeded);

    if not ShellLinks then begin
      PROGMAN := 0;
      Conv := 0;
    end;
    try
      try
        if not ShellLinks then begin
          PROGMAN := DDE.CreateString('PROGMAN'); {don't localize}
          Conv := DDE.BeginConnection(PROGMAN, PROGMAN);
          {don't localize these}
          CreateGroupCmd := '[CreateGroup("' + WizardGroupValue;
          Z := '[ShowGroup("' + WizardGroupValue + '",5';
          if (shAlwaysUsePersonalGroup in SetupHeader.Options) and IsNT then begin
            CreateGroupCmd := CreateGroupCmd + '",0)]';
            Z := Z + ',0)]';
          end
          else begin
            CreateGroupCmd := CreateGroupCmd + '")]';
            Z := Z + ')]';
          end;
          DDE.Execute (Conv, CreateGroupCmd, False);
          DDE.Execute (Conv, Z, False);
          UninstLog.Add (utDeleteGroupOrItem, [WizardGroupValue, ''], 0);
        end;

        try
          for CurIconNumber := 0 to Entries[seIcon].Count-1 do
            with PSetupIconEntry(Entries[seIcon][CurIconNumber])^ do begin
              FN := ChangeDirConst(Filename);
              if ioUseAppPaths in Options then
                FN := ExpandAppPath(FN);
              if not(ioCreateOnlyIfFileExists in Options) or FileExists(FN) then
                CreateAnIcon (IconName, ChangeDirConst(Comment), FN,
                  ChangeDirConst(Parameters), ChangeDirConst(WorkingDir),
                  ChangeDirConst(IconFilename), IconIndex, ShowCmd,
                  ioUninsNeverUninstall in Options, CloseOnExit);
            end;
        except
          on E: Exception do begin
            if not(E is EAbort) then
              Application.HandleException (E)
            else
              raise;
          end;
        end;

        { Create the icon for the uninstaller }
        if CreateUninstallIcon then begin
          if SetupHeader.UninstallIconName <> '' then
            Z := SetupHeader.UninstallIconName
          else
            Z := FmtSetupMessage1(msgDefaultUninstallIconName,
              SetupHeader.AppName);
          CreateAnIcon ('{group}\' + Z, '', UninstallExeFilename,
            '', '', '', 0, SW_SHOWNORMAL, False, icNoSetting);
        end;
      except
        on E: Exception do begin
          if not(E is EAbort) then
            Application.HandleException (E)
          else
            raise;
        end;
      end;
    finally
      if not ShellLinks then begin
        if Conv <> 0 then DDE.EndConnection (Conv);
        if PROGMAN <> 0 then DDE.FreeString (PROGMAN);
      end;
    end;
  end;

  procedure CreateIniEntries;
  var
    CurIniNumber: Integer;
    CurIni: PSetupIniEntry;
    IniSection, IniEntry, IniValue, IniFilename: String;
  begin
    for CurIniNumber := 0 to Entries[seIni].Count-1 do begin
      CurIni := PSetupIniEntry(Entries[seIni][CurIniNumber]);
      with CurIni^ do begin
        IniSection := ChangeDirConst(Section);
        IniEntry := ChangeDirConst(Entry);
        IniValue := ChangeDirConst(Value);
        IniFilename := ChangeDirConst(Filename);

        if (IniEntry <> '') and (ioHasValue in Options) and
           (not(ioCreateKeyIfDoesntExist in Options) or
            not IniKeyExists(IniSection, IniEntry, IniFilename)) then
          while not SetIniString(IniSection, IniEntry, IniValue, IniFilename) do begin
            if AbortRetryIgnoreMsgBox(FmtSetupMessage1(msgErrorIniEntry, IniFilename),
               SetupMessages[msgEntryAbortRetryIgnore]) then
              Break;
          end;

        if ioUninsDeleteEntireSection in Options then
          UninstLog.Add (utIniDeleteSection, [IniFilename, IniSection], 0);
        if ioUninsDeleteSectionIfEmpty in Options then
          UninstLog.Add (utIniDeleteSection, [IniFilename, IniSection],
            utIniDeleteSection_OnlyIfEmpty);
        if (ioUninsDeleteEntry in Options) and (IniEntry <> '') then
          UninstLog.Add (utIniDeleteEntry, [IniFilename, IniSection, IniEntry], 0);
        { ^ add utIniDeleteEntry last since we want it done first by the
          uninstaller (in case the entry's also got the
          "uninsdeletesectionifempty" flag) }
      end;
    end;

    { Increment progress meter }
    SetProgress (False, 1000);
  end;

  procedure CreateRegistryEntries;
  var
    K: HKEY;
    Disp: DWORD;
    N, V, ExistingData: String;
    ExistingType, NewType: DWORD;
    S: String;
    CurRegNumber: Integer;
    NeedToRetry: Boolean;
    ErrorCode: Longint;
  begin
    for CurRegNumber := 0 to Entries[seRegistry].Count-1 do begin
      with PSetupRegistryEntry(Entries[seRegistry][CurRegNumber])^ do begin
        S := ChangeDirConst(Subkey);
        N := ChangeDirConst(ValueName);

        repeat
          NeedToRetry := False;

          try
            if roDeleteKey in Options then
              RegDeleteKeyIncludingSubkeys (RootKey, PChar(S));
            if (roDeleteKey in Options) and (Typ = rtNone) then
              { We've deleted the key, and no value is to be created.
                Our work is done. }
            else if (roDeleteValue in Options) and (Typ = rtNone) then begin
              { We're going to delete a value without no intention of creating
                another, so don't create the key if it didn't exist. }
              if RegOpenKeyEx(RootKey, PChar(S), 0, KEY_SET_VALUE, K) = ERROR_SUCCESS then begin
                RegDeleteValue (K, PChar(N));
                RegCloseKey (K);
              end;
            end
            else begin
              if not(roDontCreateKey in Options) then
                ErrorCode := RegCreateKeyEx(RootKey, PChar(S), 0, nil,
                  REG_OPTION_NON_VOLATILE,
                  KEY_QUERY_VALUE or KEY_ENUMERATE_SUB_KEYS or KEY_SET_VALUE,
                  nil, K, @Disp)
              else begin
                if Typ <> rtNone then
                  ErrorCode := RegOpenKeyEx(RootKey, PChar(S), 0,
                    KEY_QUERY_VALUE or KEY_ENUMERATE_SUB_KEYS or KEY_SET_VALUE,
                    K)
                else
                  { We're not creating a value, and we're not deleting a value
                    (this was checked above), so there is no reason to even
                    open the key. Pretend RegOpenKeyEx failed }
                  ErrorCode := ERROR_FILE_NOT_FOUND;
              end;
              { Display error message if it failed to create or open the key.
                Ignore ERROR_FILE_NOT_FOUND (key not found) errors when the
                "dontcreatekey" flag is specified }
              if (ErrorCode <> ERROR_SUCCESS) and not(roNoError in Options) and
                 (not(roDontCreateKey in Options) or (ErrorCode <> ERROR_FILE_NOT_FOUND)) then
                raise Exception.Create(FmtSetupMessage(msgErrorRegCreateKey,
                  [RegRootKeyNames[RootKey], S]));
              { If there was no error opening the key, proceed with deleting
                and/or creating the value }
              if ErrorCode = ERROR_SUCCESS then
              try
                if roDeleteValue in Options then
                  RegDeleteValue (K, PChar(N));
                if (Typ <> rtNone) and
                   (not(roCreateValueIfDoesntExist in Options) or
                    not RegValueExists(K, PChar(N))) then
                  case Typ of
                    rtString, rtExpandString, rtMultiString: begin
                        NewType := REG_SZ;
                        case Typ of
                          rtExpandString: NewType := REG_EXPAND_SZ;
                          rtMultiString: NewType := REG_MULTI_SZ;
                        end;
                        if Typ <> rtMultiString then begin
                          if (Pos('{olddata}', ValueData) <> 0) and
                             RegQueryStringValue(K, PChar(N), ExistingData) and
                             (RegQueryValueEx(K, PChar(N), nil, @ExistingType, nil, nil) = ERROR_SUCCESS) then begin
                            if roPreserveStringType in Options then
                              NewType := ExistingType;
                          end
                          else
                            ExistingData := '';
                          V := ChangeDirConstEx(ValueData, ['olddata', ExistingData])
                        end
                        else begin
                          if (Pos('{olddata}', ValueData) <> 0) and
                             RegQueryMultiStringValue(K, PChar(N), ExistingData) then
                            { successful }
                          else
                            ExistingData := '';
                          V := ChangeDirConstEx(ValueData, ['olddata', ExistingData,
                            'break', #0]);
                          { Multi-string data requires two null terminators at end.
                            Delphi's String type is implicitly null-terminated,
                            so only one null needs to be added to the end. }
                          if (V = '') or (V[Length(V)] <> #0) then
                            V := V + #0;
                        end;
                        if (RegSetValueEx(K, PChar(N), 0, NewType, PChar(V), Length(V)+1) <> ERROR_SUCCESS) and
                           not(roNoError in Options) then
                          raise Exception.Create(FmtSetupMessage(msgErrorRegWriteKey,
                            [RegRootKeyNames[RootKey], S]));
                      end;
                    rtDWord: begin
                        if (RegSetValueEx(K, PChar(N), 0, REG_DWORD, PDWORD(ValueData), SizeOf(DWORD)) <> ERROR_SUCCESS) and
                           not(roNoError in Options) then
                          raise Exception.Create(FmtSetupMessage(msgErrorRegWriteKey,
                            [RegRootKeyNames[RootKey], S]));
                      end;
                    rtBinary: begin
                        if (RegSetValueEx(K, PChar(N), 0, REG_BINARY, PChar(ValueData), Length(ValueData)) <> ERROR_SUCCESS) and
                           not(roNoError in Options) then
                          raise Exception.Create(FmtSetupMessage(msgErrorRegWriteKey,
                            [RegRootKeyNames[RootKey], S]));
                      end;
                  end;
              finally
                RegCloseKey (K);
              end;
            end;
          except
            on E: Exception do begin
              if not AbortRetryIgnoreMsgBox(E.Message + '.', SetupMessages[msgEntryAbortRetryIgnore]) then
                NeedToRetry := True;
            end;
          end;
        until not NeedToRetry;

        if roUninsDeleteEntireKey in Options then
          UninstLog.Add (utRegDeleteEntireKey, [S], RootKey);
        if roUninsDeleteEntireKeyIfEmpty in Options then
          UninstLog.Add (utRegDeleteKeyIfEmpty, [S], RootKey);
        if roUninsDeleteValue in Options then
          UninstLog.Add (utRegDeleteValue, [S, N], RootKey);
          { ^ must add roUninsDeleteValue after roUninstDeleteEntireKey*
            since the entry may have both the roUninsDeleteValue and
            roUninsDeleteEntireKeyIfEmpty options }
        if roUninsClearValue in Options then
          UninstLog.Add (utRegClearValue, [S, N], RootKey);
      end;
    end;

    { Increment progress meter }
    SetProgress (False, 1000);
  end;

  procedure RegisterFiles;
    procedure RegisterServer (const Filename: String);
    var
      NeedToRetry: Boolean;
      LibHandle: THandle;
      LastError: DWORD;
      RegisterServerProc: function: HRESULT; stdcall;
      RegisterStep: (rsLoadLibrary, rsGetProcAddress, rsRegisterServer, rsSuccessful);
      RegisterCode: HResult;
      RegisterMsg: String;
      SaveCursor: HCURSOR;
      SaveErrorMode: UINT;
    begin
      repeat
        NeedToRetry := False;
        RegisterStep := rsLoadLibrary;
        RegisterCode := 0;
        LastError := 0;
        try
          SaveCursor := SetCursor(LoadCursor(0, IDC_WAIT));  { show the 'hourglass' cursor }
          SaveErrorMode := SetErrorMode(SEM_NOOPENFILEERRORBOX);
          try
            LibHandle := LoadLibrary(PChar(Filename));
            if LibHandle = 0 then
              LastError := GetLastError
            else begin
              try
                RegisterStep := rsGetProcAddress;
                @RegisterServerProc := GetProcAddress(LibHandle, 'DllRegisterServer');
                if Assigned(@RegisterServerProc) then begin
                  RegisterStep := rsRegisterServer;
                  RegisterCode := RegisterServerProc;
                  if SUCCEEDED(RegisterCode) then
                    RegisterStep := rsSuccessful;
                end;
              finally
                FreeLibrary (LibHandle);
              end;
            end;
          finally
            SetErrorMode (SaveErrorMode);
            SetCursor (SaveCursor);
          end;
        except
          { ignore any exceptions... }
        end;
        if RegisterStep <> rsSuccessful then begin
          case RegisterStep of
            rsLoadLibrary: RegisterMsg := FmtSetupMessage(msgErrorFunctionFailedWithMessage,
              ['LoadLibrary', IntToStr(LastError), SysErrorMessage(LastError)]);
            rsGetProcAddress: RegisterMsg := SetupMessages[msgErrorRegisterServerMissingExport];
            rsRegisterServer: RegisterMsg := FmtSetupMessage(msgErrorFunctionFailed,
              ['DllRegisterServer', IntToHexStr8(RegisterCode)]);
          end;
          if not AbortRetryIgnoreMsgBox(Filename + SNewLine2 +
             FmtSetupMessage1(msgErrorRegisterServer, AddPeriod(RegisterMsg)),
             SetupMessages[msgFileAbortRetryIgnore2]) then
            NeedToRetry := True;
        end;
      until not NeedToRetry;
    end;
    procedure RegisterServersOnRestart;
    const
      RunOnceKey = NEWREGSTR_PATH_SETUP + '\RunOnce';
    var
      F: TextFile;
      RootKey, H: HKEY;
      I, J: Integer;
      Disp: DWORD;
      ValueName, Data: String;

      procedure RegError (const Msg: TSetupMessageID);
      begin
        raise Exception.Create(FmtSetupMessage(Msg, [RegRootKeyNames[RootKey],
          RunOnceKey]));
      end;
    label 1, 2;
    begin
      if RegSvrExeFilename = '' then
        raise Exception.Create(FmtSetupMessage1(msgErrorInternal, 'R100'));

      AssignFile (F, ChangeFileExt(RegSvrExeFilename, '.lst'));
      FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F);
      try
        Writeln (F, '* List of files to be registered on the next reboot. DO NOT EDIT! *' + SNewLine);
        for I := 0 to RegisterFilesList.Count-1 do
          with PRegisterFilesListRec(RegisterFilesList[I])^ do begin
            if not TypeLib then
              Data := '[s]'
            else
              Data := '[t]';
            Data := Data + Filename;
            Writeln (F, Data);
          end;
      finally
        CloseFile (F);
      end;

      H := 0;
      try
        RootKey := HKEY_LOCAL_MACHINE;
        goto 2;
      1:RootKey := HKEY_CURRENT_USER;
      2:if RegCreateKeyEx(RootKey, RunOnceKey, 0, nil,
           REG_OPTION_NON_VOLATILE, KEY_SET_VALUE or KEY_QUERY_VALUE or
             KEY_ENUMERATE_SUB_KEYS, nil, H, @Disp) <> ERROR_SUCCESS then begin
          H := 0;
          if RootKey = HKEY_LOCAL_MACHINE then
            goto 1
          else
            RegError (msgErrorRegCreateKey);
        end;
        J := 0;
        while True do begin
          Inc (J);
          ValueName := Format('InnoSetupRegFile.%.10d', [J]);  { don't localize }
          { ^ Note: Names of values written to the "RunOnce" key cannot
              exceed 31 characters! Otherwise the original Windows
              Explorer 4.0 will not process them. }
          if not RegValueExists(H, PChar(ValueName)) then begin
            Data := AddQuotes(RegSvrExeFilename) + ' /REG';
            if RegSetValueEx(H, PChar(ValueName), 0, REG_SZ, PChar(Data),
               Length(Data)+1) <> ERROR_SUCCESS then begin
              if RootKey = HKEY_LOCAL_MACHINE then begin
                RegCloseKey (H);
                H := 0;
                goto 1;
              end
              else
                RegError (msgErrorRegWriteKey);
            end;
            Break;
          end;
        end;
      finally
        if H <> 0 then
          RegCloseKey (H);
      end;
    end;
    procedure RegisterTLib (const Filename: String);
    var
      NeedToRetry: Boolean;
    begin
      repeat
        NeedToRetry := False;
        try
          RegisterTypeLibrary (Filename);
        except
          on E: Exception do begin
            if not AbortRetryIgnoreMsgBox(Filename + SNewLine2 +
               FmtSetupMessage1(msgErrorRegisterTypeLib, AddPeriod(E.Message)),
               SetupMessages[msgFileAbortRetryIgnore2]) then
              NeedToRetry := True;
          end;
        end;
      until not NeedToRetry;
    end;
  var
    I: Integer;
  begin
    if not NeedsRestart then
      for I := 0 to RegisterFilesList.Count-1 do begin
        with PRegisterFilesListRec(RegisterFilesList[I])^ do
          if not TypeLib then
            RegisterServer (PRegisterFilesListRec(RegisterFilesList[I])^.Filename)
          else
            RegisterTLib (PRegisterFilesListRec(RegisterFilesList[I])^.Filename);
      end
    else
      { When a restart is needed, all "regserver" & "regtypelib" files will get
        registered on the next logon }
      RegisterServersOnRestart;
  end;

  procedure ProcessInstallDeleteEntries;
  var
    I: Integer;
  begin
    for I := 0 to Entries[seInstallDelete].Count-1 do
      with PSetupDeleteEntry(Entries[seInstallDelete][I])^ do
        case DeleteType of
          dfFiles, dfFilesAndOrSubdirs:
            DelTree (ChangeDirConst(Name), False, True, DeleteType = dfFilesAndOrSubdirs,
              nil, nil);
          dfDirIfEmpty:
            DelTree (ChangeDirConst(Name), True, False, False, nil, nil);
        end;
  end;

  procedure RecordUninstallDeleteEntries;
  var
    I: Integer;
  const
    Flags: array[TSetupDeleteType] of Longint = (
      utDeleteDirOrFiles_Extra or utDeleteDirOrFiles_DeleteFiles,
      utDeleteDirOrFiles_Extra or utDeleteDirOrFiles_DeleteFiles or
        utDeleteDirOrFiles_DeleteSubdirsAlso,
      utDeleteDirOrFiles_Extra or utDeleteDirOrFiles_IsDir);
  begin
    for I := Entries[seUninstallDelete].Count-1 downto 0 do
      { ^ process backwards so the uninstaller will process them in the order
          they appear in the script }
      with PSetupDeleteEntry(Entries[seUninstallDelete][I])^ do
        UninstLog.Add (utDeleteDirOrFiles, [ChangeDirConst(Name)],
          Flags[DeleteType]);
  end;

  procedure RecordUninstallRunEntries;
  var
    I: Integer;
    Flags: Longint;
  begin
    for I := Entries[seUninstallRun].Count-1 downto 0 do
      { ^ process backwards so the uninstaller will process them in the order
          they appear in the script }
      with PSetupRunEntry(Entries[seUninstallRun][I])^ do begin
        Flags := 0;
        case Wait of
          rwNoWait: Flags := Flags or utRun_NoWait;
          rwWaitUntilIdle: Flags := Flags or utRun_WaitUntilIdle;
        end;
        if roShellExec in Options then
          Flags := Flags or utRun_ShellExec;
        case ShowCmd of
          SW_SHOWMINNOACTIVE: Flags := Flags or utRun_RunMinimized;
          SW_SHOWMAXIMIZED: Flags := Flags or utRun_RunMaximized;
        end;
        UninstLog.Add (utRun, [ChangeDirConst(Name),
          ChangeDirConst(Parameters), ChangeDirConst(WorkingDir),
          ChangeDirConst(RunOnceId)], Flags);
      end;
  end;

  procedure GenerateUninstallInfoFilename;
  var
    ExistingFiles: array[0..999] of Boolean;
    BaseDir: String;

    procedure FindFiles;
    var
      H: THandle;
      FindData: TWin32FindData;
      S: String;
    begin
      H := FindFirstFile(PChar(AddBackslash(BaseDir) + 'unins???.*'),
        FindData);
      if H <> INVALID_HANDLE_VALUE then begin
        repeat
          S := FindData.cFilename;
          if (Length(S) >= 9) and (CompareText(Copy(S, 1, 5), 'unins') = 0) and
             (S[6] in ['0'..'9']) and (S[7] in ['0'..'9']) and (S[8] in ['0'..'9']) and
             (S[9] = '.') then
            ExistingFiles[StrToInt(Copy(S, 6, 3))] := True;
        until not FindNextFile(H, FindData);
        WinProcs.FindClose (H);
      end;
    end;

    procedure GenerateFilenames (const I: Integer);
    var
      BaseFilename: String;
    begin
      BaseFilename := AddBackslash(BaseDir) + Format('unins%.3d', [I]);
      UninstallExeFilename := BaseFilename + '.exe';
      UninstallDataFilename := BaseFilename + '.dat';
      UninstallMsgFilename := BaseFilename + '.msg';
    end;

    procedure ReserveDataFile;
    var
      F: File;
    begin
      { Create an empty .dat file to reserve the filename. }
      AssignFile (F, UninstallDataFilename);
      FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F, 1);
      UninstallDataCreated := True;
      CloseFile (F);
    end;

  var
    I: Integer;
  begin
    BaseDir := ChangeDirConst(SetupHeader.UninstallFilesDir);
    MakeDir (BaseDir, []);

    FillChar (ExistingFiles, SizeOf(ExistingFiles), 0);  { set all to False }
    FindFiles;

    { Look for an existing .dat file to append to or overwrite }
    if SetupHeader.UninstallLogMode <> lmNew then
      for I := 0 to 999 do
        if ExistingFiles[I] then begin
          GenerateFilenames (I);
          if FileExists(UninstallDataFilename) and
             UninstLog.Test(UninstallDataFilename, SetupHeader.AppId) then begin
            if SetupHeader.UninstallLogMode = lmAppend then
              AppendUninstallData := True;
            Exit;
          end;
        end;
    { None found; use a new .dat file }
    for I := 0 to 999 do
      if not ExistingFiles[I] then begin
        GenerateFilenames (I);
        ReserveDataFile;
        Exit;
      end;
    raise Exception.Create(FmtSetupMessage1(msgErrorTooManyFilesInDir,
      BaseDir));
  end;

  procedure RenameUninstallExe;

    function LastErrorMessage (const ID: TSetupMessageID): Boolean;
    var
      E: DWORD;
    begin
      E := GetLastError;
      Result := MsgBoxFmt('%s' + SNewLine2 + '%s' + SNewLine2 + '%d - %s',
        [UninstallExeFilename, SetupMessages[ID], E, AddPeriod(SysErrorMessage(E))],
        '', mbError, MB_RETRYCANCEL) = IDRETRY;
    end;

  begin
    { If the uninstall EXE wasn't extracted to a .tmp file because it isn't
      replacing an existing uninstall EXE, exit. }
    if UninstallTempExeFilename = '' then
      Exit;
    while not DeleteFile(UninstallExeFilename) do
      if not LastErrorMessage(msgErrorReplacingExistingFile) then
        Abort;
    while not MoveFile(PChar(UninstallTempExeFilename), PChar(UninstallExeFilename)) do
      if not LastErrorMessage(msgErrorRenamingTemp) then
        Abort;
    UninstallTempExeFilename := '';
  end;

  procedure SaveUninstallMsgData;
  var
    F: File;
    UninstallerMsgTail: TUninstallerMsgTail;
  begin
    { If this installation didn't create or replace an unins???.exe file,
      do nothing }
    if UninstallExeCreated = ueNone then
      Exit;
    if not DetachedUninstMsgFile then begin
      { Bind .msg data in the .exe (default) }
      AssignFile (F, UninstallExeFilename);
      FileMode := fmOpenWrite or fmShareExclusive;  Reset (F, 1);
      try
        UninstallerMsgTail.ID := UninstallerMsgTailID;
        UninstallerMsgTail.Offset := FileSize(F);
        Seek (F, UninstallerMsgTail.Offset);
        BlockWrite (F, SetupMessageFile^, SetupMessageFileSize);
        BlockWrite (F, UninstallerMsgTail, SizeOf(UninstallerMsgTail));
      finally
        CloseFile (F);
      end;
    end
    else begin
      { Save into a separate .msg file }
      AssignFile (F, UninstallMsgFilename);
      FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F, 1);
      try
        UninstallMsgCreated := True;
        BlockWrite (F, SetupMessageFile^, SetupMessageFileSize);
      finally
        CloseFile (F);
      end;
    end;
  end;

  procedure SetStatusLabelText (const ID: TSetupMessageID);
  begin
    StatusLabel.Caption := SetupMessages[ID];
    StatusLabel.Update;
  end;
var
  UninstLogCleared: Boolean;
  I: Integer;
begin
  RegSvrExeFilename := '';
  UninstallIconNeeded := True;
  UninstallExeCreated := ueNone;
  UninstallDataCreated := False;
  UninstallMsgCreated := False;
  AppendUninstallData := False;
  UninstLogCleared := False;
  RegisterFilesList := nil;
  UninstLog := TSetupUninstallLog.Create;
  try
    try
      UninstLog.AppName := SetupHeader.AppName;
      UninstLog.AppId := SetupHeader.AppId;
      if IsAdmin then
        Include (UninstLog.Flags, ufAdminInstalled);
      RecordStartInstall;

      RegisterFilesList := TList.Create;

      { Process InstallDelete entries, if any }
      ProcessInstallDeleteEntries;
      ProcessEvents;

      if SetupHeader.AppMutex <> '' then
        UninstLog.Add (utMutexCheck, [SetupHeader.AppMutex], 0);
      if shChangesAssociations in SetupHeader.Options then
        UninstLog.Add (utRefreshFileAssoc, [''], 0);

      { Record UninstallDelete entries, if any }
      RecordUninstallDeleteEntries;
      ProcessEvents;

      { Create the application directory and extra dirs }
      SetStatusLabelText (msgStatusCreateDirs);
      CreateDirs;
      ProcessEvents;

      if shUninstallable in SetupHeader.Options then begin
        { Generate the filenames for the uninstall info in the application
          directory }
        GenerateUninstallInfoFilename;

        UninstallIconNeeded := not UsingWindows4;
      end;

      { Copy the files }
      SetStatusLabelText (msgStatusExtractFiles);
      CopyFiles;
      FilenameLabel.Caption := '';  FilenameLabel.Update;
      ProcessEvents;

      { Create program icons on Program Manager (or Start Menu), if any }
      if not WizardNoIcons and HasIcons then begin
        SetStatusLabelText (msgStatusCreateIcons);
        CreateIcons;
        ProcessEvents;
      end;

      { Create INI entries, if any }
      if Entries[seIni].Count <> 0 then begin
        SetStatusLabelText (msgStatusCreateIniEntries);
        CreateIniEntries;
        ProcessEvents;
      end;

      { Create registry entries, if any }
      if Entries[seRegistry].Count <> 0 then begin
        SetStatusLabelText (msgStatusCreateRegistryEntries);
        CreateRegistryEntries;
        ProcessEvents;
      end;

      { Register files, if any }
      if RegisterFilesList.Count <> 0 then begin
        SetStatusLabelText (msgStatusRegisterFiles);
        RegisterFiles;
        ProcessEvents;
      end;

      { Save uninstall information. After uninstall info is saved, you cannot
        make any more modifications to the user's system. Any additional
        modifications you want to add must be done before this is called. }
      if shUninstallable in SetupHeader.Options then begin
        SetStatusLabelText (msgStatusSavingUninstall);
        RenameUninstallExe;
        SaveUninstallMsgData;
        { Register uninstall information so the program can be uninstalled
          through the Add/Remove Programs Control Panel applet. This is done
          on NT 3.51 too, so that the uninstall entry for the app will appear
          if the user later upgrades to NT 4.0+. }
        if shCreateUninstallRegKey in SetupHeader.Options then
          RegisterUninstallInfo;
        RecordUninstallRunEntries;
        UninstLog.Add (utEndInstall, [GetLocalTimeAsStr], 0);
        UninstLog.Save (UninstallDataFilename, AppendUninstallData,
          shUpdateUninstallLogAppName in SetupHeader.Options);
      end;
      UninstLogCleared := True;
      UninstLog.Clear;

      if shChangesAssociations in SetupHeader.Options then
        SHChangeNotify (SHCNE_ASSOCCHANGED, SHCNF_IDLIST, nil, nil);

      DisableCloseQuery := True;
      Close;
      PostMessage (MainForm.Handle, WM_SETUPNEXTSTEP, 0, 0);
    except
      on E: Exception do begin
        { Show error message, if any }
        if not(E is EAbort) then begin
          Application.HandleException (E);
          MsgBox (SetupMessages[msgSetupAborted], '', mbCriticalError, MB_OK);
        end;
        { Undo any changes it's made so far }
        if not UninstLogCleared then begin
          if UninstallTempExeFilename <> '' then
            DeleteFile (UninstallTempExeFilename);
          if UninstallExeCreated = ueNew then
            DeleteFile (UninstallExeFilename);
          if UninstallDataCreated then
            DeleteFile (UninstallDataFilename);
          if UninstallMsgCreated then
            DeleteFile (UninstallMsgFilename);
          UninstLog.PerformUninstall (False, nil);
        end;
        Application.Terminate;
        Exit;
      end;
    end;
  finally
    if Assigned(RegisterFilesList) then begin
      for I := RegisterFilesList.Count-1 downto 0 do
        Dispose (PRegisterFilesListRec(RegisterFilesList[I]));
      RegisterFilesList.Free;
    end;
    UninstLog.Free;
  end;
end;

end.
