;;; latin1-disp.el --- display tables for other ISO 8859 on Latin-1 terminals -*- lexical-binding: t; -*-

;; Copyright (C) 2000-2022 Free Software Foundation, Inc.

;; Author: Dave Love <fx@gnu.org>
;; Keywords: i18n

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; This package sets up display of ISO 8859-n for n>1 by substituting
;; Latin-1 characters and sequences of them for characters which can't
;; be displayed, either because we're on a tty or because we don't
;; have the relevant window system fonts available.  For instance,
;; Latin-9 is very similar to Latin-1, so we can display most Latin-9
;; characters using the Latin-1 characters at the same code point and
;; fall back on more-or-less mnemonic ASCII sequences for the rest.

;; For the Latin charsets the ASCII sequences are mostly consistent
;; with the Quail prefix input sequences.  Latin-4 uses the Quail
;; postfix sequences since a prefix method isn't defined for Latin-4.

;; [A different approach is taken in the DOS display tables in
;; term/internal.el, and the relevant ASCII sequences from there are
;; available as an alternative; see `latin1-display-mnemonic'.  Only
;; these sequences are used for Arabic, Cyrillic, Greek and Hebrew.]

;; If you don't even have Latin-1, see iso-ascii.el and use the
;; complete tables from internal.el.  The ASCII sequences used here
;; are mostly in the same style as iso-ascii.

;;; Code:

;; Ensure `standard-display-table' is set up:
(require 'disp-table)

(defconst latin1-display-sets '(latin-2 latin-3 latin-4 latin-5 latin-8
		                latin-9 arabic cyrillic greek hebrew)
  "The ISO8859 character sets with defined Latin-1 display sequences.
These are the nicknames for the sets and correspond to Emacs language
environments.")

(defgroup latin1-display ()
  "Set up display tables for ISO8859 characters using Latin-1."
  :version "21.1"
  :link '(emacs-commentary-link "latin1-disp")
  :group 'i18n)

(defcustom latin1-display-format "{%s}"
  "A format string used to display the ASCII sequences.
The default encloses the sequence in braces, but you could just use
\"%s\" to avoid the braces, maybe with a non-default value of
`latin1-display-face'."
  :group 'latin1-display
  :type 'string)

;;;###autoload
(defcustom latin1-display nil
  "Set up Latin-1/ASCII display for ISO8859 character sets.
This is done for each character set in the list `latin1-display-sets',
if no font is available to display it.  Characters are displayed using
the corresponding Latin-1 characters where they match.  Otherwise
ASCII sequences are used, mostly following the Latin prefix input
methods.  Some different ASCII sequences are used if
`latin1-display-mnemonic' is non-nil.

This option also treats some characters in the `mule-unicode-...'
charsets if you don't have a Unicode font with which to display them.

Setting this variable directly does not take effect;
use either \\[customize] or the function `latin1-display'."
  :group 'latin1-display
  :type 'boolean
  :require 'latin1-disp
  :initialize #'custom-initialize-default
  :set (lambda (_symbol value)
	 (if value
	     (apply #'latin1-display latin1-display-sets)
	   (latin1-display))))

;;;###autoload
(defun latin1-display (&rest sets)
  "Set up Latin-1/ASCII display for the arguments character SETS.
See option `latin1-display' for the method.  The members of the list
must be in `latin1-display-sets'.  With no arguments, reset the
display for all of `latin1-display-sets'.  See also
`latin1-display-setup'."
  (if sets
      (progn
	(mapc #'latin1-display-setup sets)
	(unless (char-displayable-p #x101) ; a with macron
	  ;; Extra stuff for windows-1252, in particular.
	  (mapc
	   (lambda (l)
	     (apply 'latin1-display-char l))
	   '((?\‚ ",") ;; SINGLE LOW-9 QUOTATION MARK
	     (?\„ ",,")	;; DOUBLE LOW-9 QUOTATION MARK
	     (?\… "...") ;; HORIZONTAL ELLIPSIS
	     (?\‰ "o/oo") ;; PER MILLE SIGN
	     (?\‹ "<") ;; SINGLE LEFT-POINTING ANGLE QUOTATION MARK
	     (?\“ "``")	;; LEFT DOUBLE QUOTATION MARK
	     (?\” "''")	;; RIGHT DOUBLE QUOTATION MARK
	     (?\– "-") ;; EN DASH
	     (?\— "--")	;; EM DASH
	     (?\™ "TM")	;; TRADE MARK SIGN
	     (?\› ">") ;; SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
	     (?•  "·")
	     )))
	  (setq latin1-display t))
    (mapc #'latin1-display-reset latin1-display-sets)
    (set-char-table-range standard-display-table '(#x0100 . #x33FF) nil)
    (set-char-table-range standard-display-table '(#xE000 . #xFFFF) nil)
    (setq latin1-display nil)
    (redraw-display)))

(defcustom latin1-display-mnemonic nil
  "Non-nil means to display potentially more mnemonic sequences.
These are taken from the tables in `internal.el' rather than the Quail
input sequences."
  :type 'boolean
  :group 'latin1-display)

(defcustom latin1-display-face 'default
  "Face to use for displaying substituted ASCII sequences."
  :type 'face
  :version "22.1"
  :group 'latin1-display)

(defun latin1-display-char (char display &optional alt-display)
  "Make an entry in `standard-display-table' for CHAR using string DISPLAY.
If ALT-DISPLAY is provided, use that instead if
`latin1-display-mnemonic' is non-nil.  The actual string displayed is
formatted using `latin1-display-format'.

DISPLAY and ALT-DISPLAY are either strings or vectors.  String values
are formatted using `latin1-display-format' and passed to
`standard-display-ascii'; vectors are put into `standard-display-table'
asis."
  (if (and (stringp alt-display)
	   latin1-display-mnemonic)
      (setq display alt-display))
  (if (stringp display)
      (if (eq 'default latin1-display-face)
	  (standard-display-ascii char (format latin1-display-format display))
	(aset standard-display-table char
	      (vconcat (mapcar (lambda (c) (make-glyph-code c latin1-display-face))
			       display))))
    (aset standard-display-table char
	  (make-glyph-code display latin1-display-face))))

(defun latin1-display-identities (charset)
  "Display each character in CHARSET as the corresponding Latin-1 character.
CHARSET is a symbol which is the nickname of a language environment
using an ISO8859 character set."
  (if (eq charset 'cyrillic)
      (setq charset 'cyrillic-iso))
  (let ((i 128)
	(set (car (remq 'ascii (get-language-info charset 'charset)))))
    (while (<= i 255)
      (let ((ch (decode-char set i)))
	(if ch
	    (aset standard-display-table ch (vector i))))
      (setq i (1+ i)))))

(defun latin1-display-reset (language)
  "Set up the default display for each character of LANGUAGE's charset.
LANGUAGE is a symbol naming a language environment using an ISO8859
character set."
  (if (eq language 'cyrillic)
      (setq language 'cyrillic-iso))
  (let ((charset (if (eq language 'arabic)
		     'arabic-iso8859-6
		   (car (remq 'ascii (get-language-info language
							'charset))))))
    (map-charset-chars (lambda (range _arg)
                         (standard-display-default (car range) (cdr range)))
		       charset))
  (sit-for 0))

(defun latin1-display-check-font (language)
  "Return non-nil if we have a font with an encoding for LANGUAGE.
LANGUAGE is a symbol naming a language environment using an ISO8859
character set: `latin-2', `hebrew' etc."
  (if (eq language 'cyrillic)
      (setq language 'cyrillic-iso))
  (let* ((info (get-language-info language 'charset))
	 (char (and info (decode-char (car (remq 'ascii info)) ?\ ))))
    (and char (char-displayable-p char))))

(defun latin1-display-setup (set &optional _force)
  "Set up Latin-1 display for characters in the given SET.
SET must be a member of `latin1-display-sets'.  Normally, check
whether a font for SET is available and don't set the display if it is."
  (cond
   ((eq set 'latin-2)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?Ć "'C" "C'")
	 (?Đ "'D" "/D")
	 (?Ś "'S" "S'")
	 (?ć "'c" "c'")
	 (?đ "'d" "/d")
	 (?Ĺ "'L" "L'")
	 (?ń "'n" "n'")
	 (?Ń "'N" "N'")
	 (?ŕ "'r" "r'")
	 (?Ŕ "'R" "R'")
	 (?ś "'s" "s'")
	 (?ź "'z" "z'")
	 (?Ź "'Z" "Z'")
	 (?Ą "`A" "A;")
	 (?Ę "`E" "E;")
	 (?Ł "`L" "/L")
	 (?Ş "`S" ",S")
	 (?Ţ "`T" ",T")
	 (?Ż "`Z" "Z^.")
	 (?ą "`a" "a;")
	 (?ł "`l" "/l")
	 (?ę "`e" "e;")
	 (?ş "`s" ",s")
	 (?ţ "`t" ",t")
	 (?ż "`z" "z^.")
	 (?˙ "`." "'.")
	 (?Ă "~A" "A(")
	 (?Č "~C" "C<")
	 (?Ď "~D" "D<")
	 (?Ě "~E" "E<")
	 (?ě "~e" "e<")
	 (?Ľ "~L" "L<")
	 (?Ň "~N" "N<")
	 (?Ő "~O" "O''")
	 (?Ř "~R" "R<")
	 (?Š "~S" "S<")
	 (?Ť "~T" "T<")
	 (?Ű "~U" "U''")
	 (?Ž "~Z" "Z<")
	 (?ă "~a" "a(}")
	 (?č "~c" "c<")
	 (?ď "~d" "d<")
	 (?ľ "~l" "l<")
	 (?ň "~n" "n<")
	 (?ő "~o" "o''")
	 (?ř "~r" "r<")
	 (?š "~s" "s<")
	 (?ť "~t" "t<")
	 (?ű "~u" "u''")
	 (?ž "~z" "z<")
	 (?ˇ "~v" "'<")			; ?˘ in latin-pre
	 (?˘ "~~" "'(")
	 (?ů "uu" "u^0")
	 (?Ů "UU" "U^0")
	 (?Ä "\"A")
	 (?ä "\"a")
	 (?Ë "\"E" "E:")
	 (?ë "\"e")
	 (?˝ "''" "'")
	 (?ˇ "'<")			; Lynx's rendering of caron
	 )))

   ((eq set 'latin-3)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?Ħ "/H")
	 (?˘ "~`" "'(")
	 (?Ĥ "^H" "H^")
	 (?ĥ "^h" "h^")
	 (?İ ".I" "I^.")
	 (?Ş ",S")
	 (?Ğ "~G" "G(")
	 (?Ĵ "^J" "J^")
	 (?Ż ".Z" "Z^.")
	 (?ħ "/h")
	 (?ı ".i" "i^.")
	 (?ş ",s")
	 (?ğ "~g" "g(")
	 (?ĵ "^j" "j^")
	 (?ż ".Z" "z^.")
	 (?Ċ ".c" "C^.")
	 (?Ĉ "^C" "C^")
	 (?Ġ ".G" "G^.")
	 (?Ĝ "^G" "G^")
	 (?Ŭ "~U" "U(")
	 (?Ŝ "^S" "S^")
	 (?ċ ".C" "c^.")
	 (?ĉ "^c" "c^")
	 (?ġ ".g" "g^.")
	 (?ĝ "^g" "g^")
	 (?ŭ "~u" "u(")
	 (?ŝ "^s" "s^")
	 (?˙ "/." "^."))))

   ((eq set 'latin-4)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?Ą "A," "A;")
	 (?ĸ "k/" "kk")
	 (?Ŗ "R," ",R")
	 (?Ĩ "I~" "?I")
	 (?Ļ "L," ",L")
	 (?Š "S~" "S<")
	 (?Ē "E-")
	 (?Ģ "G," ",G")
	 (?Ŧ "T/" "/T")
	 (?Ž "Z~" "Z<")
	 (?ą "a," "a;")
	 (?˛ "';")
	 (?ŗ "r," ",r")
	 (?ĩ "i~" "~i")
	 (?ļ "l," ",l")
	 (?ˇ "'<")
	 (?š "s~" "s<")
	 (?ē "e-")
	 (?ģ "g," ",g")
	 (?ŧ "t/" "/t")
	 (?Ŋ "N/" "NG")
	 (?ž "z~" "z<")
	 (?ŋ "n/" "ng")
	 (?Ā "A-")
	 (?Į "I," "I;")
	 (?Č "C~" "C<")
	 (?Ę "E," "E;")
	 (?Ė "E." "E^.")
	 (?Ī "I-")
	 (?Ņ "N," ",N")
	 (?Ō "O-")
	 (?Ķ "K," ",K")
	 (?Ų "U," "U;")
	 (?Ũ "U~" "~U")
	 (?Ū "U-")
	 (?ā "a-")
	 (?į "i," "i;")
	 (?č "c~" "c<")
	 (?ę "e," "e;")
	 (?ė "e." "e^.")
	 (?ī "i-")
	 (?đ "d/" "/d")
	 (?ņ "n," ",n")
	 (?ō "o-")
	 (?ķ "k," ",k")
	 (?ų "u," "u;")
	 (?ũ "u~" "~u")
	 (?ū "u-")
	 (?˙ "^."))))

   ((eq set 'latin-5)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	 (apply 'latin1-display-char l)))
       '((?ğ "~g" "g(")
	 (?Ğ "~G" "G(")
	 (?İ ".I" "I^.")
	 (?ş ",s")
	 (?Ş ",S")
	 (?ê "^e" "e<")			; from latin-post
	 (?ì ".e" "e^.")
	 (?ï "\"i" "i-")		; from latin-post
	 (?ı ".i" "i."))))

   ((eq set 'latin-8)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?Ḃ ".B" "B`")
	 (?ḃ ".b" "b`")
	 (?ċ ".c" "c`")
	 (?Ċ ".C" "C`")
	 (?Ḋ ".D" "D`")
	 (?ḋ ".d" "d`")
	 (?ẁ "`w")
	 (?Ẁ "`W")
	 (?ẃ "'w" "w'")
	 (?Ẃ "'W" "W'")
	 (?ỳ "`y")
	 (?Ỳ "`Y")
	 (?ḟ ".f" "f`")
	 (?Ḟ ".F" "F`")
	 (?ġ ".g" "g`")
	 (?Ġ ".G" "G`")
	 (?ṁ ".m" "m`")
	 (?Ṁ ".M" "M`")
	 (?ṗ ".p" "p`")
	 (?Ṗ ".P" "P`")
	 (?ṡ ".s" "s`")
	 (?Ṡ ".S" "S`")
	 (?ẅ "\"w")
	 (?Ẅ "\"W")
	 (?ŵ "^w" "w^")
	 (?Ŵ "^W" "W^")
	 (?ṫ ".t" "t`")
	 (?Ṫ ".T" "T`")
	 (?ŷ "^y" "y^")
	 (?Ŷ "^Y" "Y^")
	 (?Ÿ "\"Y"))))

   ((eq set 'latin-9)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?š "~s" "s<")
	 (?Š "~S" "S<")
	 (?€ "Euro" "E=")
	 (?ž "~z" "z<")
	 (?Ž "~Z" "Z<")
	 (?Ÿ "\"Y")
	 (?œ "oe")
	 (?Œ "OE"))))

   ((eq set 'greek)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?‘ "9'")
	 (?’ "'9")
	 (?― "-M")
	 (?΅ "'%")
	 (?Ά "'A")
	 (?Έ "'E")
	 (?Ή "'H")
	 (?Ί "'I")
	 (?Ό "'O")
	 (?Ύ "'Y")
	 (?Ώ "W%")
	 (?ΐ "i3")
	 (?Γ "G*")
	 (?Δ "D*")
	 (?Θ "TH")
	 (?Λ "L*")
	 (?Ξ "C*")
	 (?Π "P*")
	 (?Σ "S*")
	 (?Φ "F*")
	 (?Ψ "Q*")
	 (?Ω "W*")
	 (?Ϊ "\"I")
	 (?Ϋ "\"Y")
	 (?ά "a%")
	 (?έ "e%")
	 (?ή "y%")
	 (?ί "i%")
	 (?ΰ "u3")
	 (?α "a*")
	 (?β "b*")
	 (?γ "g*")
	 (?δ "d*")
	 (?ε "e*")
	 (?ζ "z*")
	 (?η "y*")
	 (?θ "h*")
	 (?ι "i*")
	 (?κ "k")
	 (?λ "l*")
	 (?μ "m*")
	 (?ν "n*")
	 (?ξ "c*")
	 (?π "p*")
	 (?ρ "r*")
	 (?ς "*s")
	 (?σ "s*")
	 (?τ "t*")
	 (?υ "u")
	 (?φ "f*")
	 (?χ "x*")
	 (?ψ "q*")
	 (?ω "w*")
	 (?ϊ "\"i")
	 (?ϋ "\"u")
	 (?ό "'o")
	 (?ύ "'u")
	 (?ώ "'w")))
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (aset standard-display-table (car l) (string-to-vector (cadr l)))))
       '((?Α "A")
	 (?Β "B")
	 (?Ε "E")
	 (?Ζ "Z")
	 (?Η "H")
	 (?Ι "I")
	 (?Κ "J")
	 (?Μ "M")
	 (?Ν "N")
	 (?Ο "O")
	 (?Ρ "P")
	 (?Τ "T")
	 (?Υ "Y")
	 (?Χ "X")
	 (?ο "o"))))

   ((eq set 'hebrew)
    ;; Don't start with identities, since we don't have definitions
    ;; for a lot of Hebrew in internal.el.  (Intlfonts is also
    ;; missing some glyphs.)
    (let ((i 34))
      (while (<= i 62)
	(let ((ch (decode-char 'hebrew-iso8859-8 i)))
	  (if ch
	      (aset standard-display-table ch
		    (vector (decode-char 'latin-iso8859-1 i)))))
	(setq i (1+ i))))
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (aset standard-display-table (car l) (string-to-vector (cadr l)))))
       '((?‗ "=2")
	 (?א "A+")
	 (?ב "B+")
	 (?ג "G+")
	 (?ד "D+")
	 (?ה "H+")
	 (?ו "W+")
	 (?ז "Z+")
	 (?ח "X+")
	 (?ט "Tj")
	 (?י "J+")
	 (?ך "K%")
	 (?כ "K+")
	 (?ל "L+")
	 (?ם "M%")
	 (?מ "M+")
	 (?ן "N%")
	 (?נ "N+")
	 (?ס "S+")
	 (?ע "E+")
	 (?ף "P%")
	 (?פ "P+")
	 (?ץ "Zj")
	 (?צ "ZJ")
	 (?ק "Q+")
	 (?ר "R+")
	 (?ש "Sh")
	 (?ת "T+"))))

   ;; Arabic probably isn't so useful in the absence of Arabic
   ;; language support...
   ((eq set 'arabic)
    (setq set 'arabic)
    (or (char-displayable-p ? )
	(aset standard-display-table ?  " "))
    (or (char-displayable-p ?¤)
	(aset standard-display-table ?¤ "¤"))
    (or (char-displayable-p ?­)
	(aset standard-display-table ?­ "­"))
    (mapc (lambda (l)
	    (or (char-displayable-p (car l))
		(apply  'latin1-display-char l)))
	    '((?، ",+")
	      (?؛ ";+")
	      (?؟ "?+")
	      (?ء "H'")
	      (?آ "aM")
	      (?أ "aH")
	      (?ؤ "wH")
	      (?إ "ah")
	      (?ئ "yH")
	      (?ا "a+")
	      (?ب "b+")
	      (?ة "tm")
	      (?ت "t+")
	      (?ث "tk")
	      (?ج "g+")
	      (?ح "hk")
	      (?خ "x+")
	      (?د "d+")
	      (?ذ "dk")
	      (?ر "r+")
	      (?ز "z+")
	      (?س "s+")
	      (?ش "sn")
	      (?ص "c+")
	      (?ض "dd")
	      (?ط "tj")
	      (?ظ "zH")
	      (?ع "e+")
	      (?غ "i+")
	      (?ـ "++")
	      (?ف "f+")
	      (?ق "q+")
	      (?ك "k+")
	      (?ل "l+")
	      (?م "m+")
	      (?ن "n+")
	      (?ه "h+")
	      (?و "w+")
	      (?ى "j+")
	      (?ي "y+")
	      (?ً ":+")
	      (?ٌ "\"+")
	      (?ٍ "=+")
	      (?َ "/+")
	      (?ُ "'+")
	      (?ِ "1+")
	      (?ّ "3+")
	      (?ْ "0+"))))

   ((eq set 'cyrillic)
    (setq set 'cyrillic-iso)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?Ђ "Dj")
	 (?Ѓ "Gj")
	 (?Є "IE")
	 (?Љ "Lj")
	 (?Њ "Nj")
	 (?Ћ "Ts")
	 (?Ќ "Kj")
	 (?Ў "V%")
	 (?Џ "Dzh")
	 (?Б "B=")
	 (?Г "â")
	 (?Д "D")
	 (?Ж "Z%")
	 (?З "3")
	 (?И "U")
	 (?Й "J=")
	 (?Л "L=")
	 (?П "P=")
	 (?У "Y")
	 (?Ф "è")
	 (?Ц "C=")
	 (?Ч "C%")
	 (?Ш "S%")
	 (?Щ "Sc")
	 (?Ъ "=\"")
	 (?Ы "Y=")
	 (?Ь "%\"")
	 (?Э "Ee")
	 (?Ю "Yu")
	 (?Я "Ya")
	 (?б "b")
	 (?в "v=")
	 (?г "g=")
	 (?д "g")
	 (?ж "z%")
	 (?з "z=")
	 (?и "u")
	 (?й "j=")
	 (?к "k")
	 (?л "l=")
	 (?м "m=")
	 (?н "n=")
	 (?п "n")
	 (?р "p")
	 (?т "t=")
	 (?ф "f=")
	 (?ц "c=")
	 (?ч "c%")
	 (?ш "s%")
	 (?щ "sc")
	 (?ъ "='")
	 (?ы "y=")
	 (?ь "%'")
	 (?э "ee")
	 (?ю "yu")
	 (?я "ya")
	 (?№ "N0")
	 (?ђ "dj")
	 (?ѓ "gj")
	 (?є "ie")
	 (?љ "lj")
	 (?њ "nj")
	 (?ћ "ts")
	 (?ќ "kj")
	 (?ў "v%")
	 (?џ "dzh")))
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (aset standard-display-table (car l) (string-to-vector (cadr l)))))
       '((?Ё "Ë")
	 (?Ѕ "S")
	 (?І "I")
	 (?Ї "Ï")
	 (?Ј "J")
	 (?ё "ë")
	 (?§ "§")
	 (?­ "-")
	 (?А "A")
	 (?В "B")
	 (?Е "E")
	 (?К "K")
	 (?М "M")
	 (?Н "H")
	 (?О "O")
	 (?Р "P")
	 (?С "C")
	 (?Т "T")
	 (?Х "X")
	 (?а "a")
	 (?е "e")
	 (?о "o")
	 (?с "c")
	 (?у "y")
	 (?х "x")
	 (?ѕ "s")
	 (?і "i")
	 (?ї "ï")
	 (?ј "j"))))

   (t (error "Unsupported character set: %S" set)))

  (sit-for 0))

;;;###autoload
(defcustom latin1-display-ucs-per-lynx nil ;FIXME: Isn't this a minor mode?
  "Set up Latin-1/ASCII display for Unicode characters.
This uses the transliterations of the Lynx browser.  The display isn't
changed if the display can render Unicode characters.

Setting this variable directly does not take effect;
use either \\[customize] or the function `latin1-display'."
  :group 'latin1-display
  :type 'boolean
  :require 'latin1-disp
  :initialize #'custom-initialize-default
  :set (lambda (_symbol value)
	 (if value
	     (latin1-display-ucs-per-lynx 1)
	   (latin1-display-ucs-per-lynx -1))))

(defun latin1-display-ucs-per-lynx (arg)
  "Set up Latin-1/ASCII display for Unicode characters.
This uses the transliterations of the Lynx browser.

With argument ARG, turn such display on if ARG is positive, otherwise
turn it off and display Unicode characters literally.  The display
isn't changed if the display can render Unicode characters."
  (interactive "p")
  (if (> arg 0)
      (unless (char-displayable-p #x101) ; a with macron
	;; It doesn't look as though we have a Unicode font.
	(let ((latin1-display-format "%s"))
	  (mapc
	   (lambda (l)
	     (apply 'latin1-display-char l))
	   ;; Table derived by running Lynx on a suitable list of
	   ;; characters in a utf-8 file, except for some added by
	   ;; hand at the end.
	   '((?\Ā "A")
	     (?\ā "a")
	     (?\Ă "A")
	     (?\ă "a")
	     (?\Ą "A")
	     (?\ą "a")
	     (?\Ć "C")
	     (?\ć "c")
	     (?\Ĉ "C")
	     (?\ĉ "c")
	     (?\Ċ "C")
	     (?\ċ "c")
	     (?\Č "C")
	     (?\č "c")
	     (?\Ď "D")
	     (?\ď "d")
	     (?\Đ "Ð")
	     (?\đ "d/")
	     (?\Ē "E")
	     (?\ē "e")
	     (?\Ĕ "E")
	     (?\ĕ "e")
	     (?\Ė "E")
	     (?\ė "e")
	     (?\Ę "E")
	     (?\ę "e")
	     (?\Ě "E")
	     (?\ě "e")
	     (?\Ĝ "G")
	     (?\ĝ "g")
	     (?\Ğ "G")
	     (?\ğ "g")
	     (?\Ġ "G")
	     (?\ġ "g")
	     (?\Ģ "G")
	     (?\ģ "g")
	     (?\Ĥ "H")
	     (?\ĥ "h")
	     (?\Ħ "H/")
	     (?\ħ "H")
	     (?\Ĩ "I")
	     (?\ĩ "i")
	     (?\Ī "I")
	     (?\ī "i")
	     (?\Ĭ "I")
	     (?\ĭ "i")
	     (?\Į "I")
	     (?\į "i")
	     (?\İ "I")
	     (?\ı "i")
	     (?\Ĳ "IJ")
	     (?\ĳ "ij")
	     (?\Ĵ "J")
	     (?\ĵ "j")
	     (?\Ķ "K")
	     (?\ķ "k")
	     (?\ĸ "kk")
	     (?\Ĺ "L")
	     (?\ĺ "l")
	     (?\Ļ "L")
	     (?\ļ "l")
	     (?\Ľ "L")
	     (?\ľ "l")
	     (?\Ŀ "L.")
	     (?\ŀ "l.")
	     (?\Ł "L/")
	     (?\ł "l/")
	     (?\Ń "N")
	     (?\ń "n")
	     (?\Ņ "N")
	     (?\ņ "n")
	     (?\Ň "N")
	     (?\ň "n")
	     (?\ŉ "'n")
	     (?\Ŋ "NG")
	     (?\ŋ "N")
	     (?\Ō "O")
	     (?\ō "o")
	     (?\Ŏ "O")
	     (?\ŏ "o")
	     (?\Ő "O\"")
	     (?\ő "o\"")
	     (?\Œ "OE")
	     (?\œ "oe")
	     (?\Ŕ "R")
	     (?\ŕ "r")
	     (?\Ŗ "R")
	     (?\ŗ "r")
	     (?\Ř "R")
	     (?\ř "r")
	     (?\Ś "S")
	     (?\ś "s")
	     (?\Ŝ "S")
	     (?\ŝ "s")
	     (?\Ş "S")
	     (?\ş "s")
	     (?\Š "S")
	     (?\š "s")
	     (?\Ţ "T")
	     (?\ţ "t")
	     (?\Ť "T")
	     (?\ť "t")
	     (?\Ŧ "T/")
	     (?\ŧ "t/")
	     (?\Ũ "U")
	     (?\ũ "u")
	     (?\Ū "U")
	     (?\ū "u")
	     (?\Ŭ "U")
	     (?\ŭ "u")
	     (?\Ů "U")
	     (?\ů "u")
	     (?\Ű "U\"")
	     (?\ű "u\"")
	     (?\Ų "U")
	     (?\ų "u")
	     (?\Ŵ "W")
	     (?\ŵ "w")
	     (?\Ŷ "Y")
	     (?\ŷ "y")
	     (?\Ÿ "Y")
	     (?\Ź "Z")
	     (?\ź "z")
	     (?\Ż "Z")
	     (?\ż "z")
	     (?\Ž "Z")
	     (?\ž "z")
	     (?\ſ "s1")
	     (?\Ƈ "C2")
	     (?\ƈ "c2")
	     (?\Ƒ "F2")
	     (?\ƒ " f")
	     (?\Ƙ "K2")
	     (?\ƙ "k2")
	     (?\Ơ "O9")
	     (?\ơ "o9")
	     (?\Ƣ "OI")
	     (?\ƣ "oi")
	     (?\Ʀ "yr")
	     (?\Ư "U9")
	     (?\ư "u9")
	     (?\Ƶ "Z/")
	     (?\ƶ "z/")
	     (?\Ʒ "ED")
	     (?\Ǎ "A")
	     (?\ǎ "a")
	     (?\Ǐ "I")
	     (?\ǐ "i")
	     (?\Ǒ "O")
	     (?\ǒ "o")
	     (?\Ǔ "U")
	     (?\ǔ "u")
	     (?\Ǖ "U:-")
	     (?\ǖ "u:-")
	     (?\Ǘ "U:'")
	     (?\ǘ "u:'")
	     (?\Ǚ "U:<")
	     (?\ǚ "u:<")
	     (?\Ǜ "U:!")
	     (?\ǜ "u:!")
	     (?\Ǟ "A1")
	     (?\ǟ "a1")
	     (?\Ǡ "A7")
	     (?\ǡ "a7")
	     (?\Ǣ "A3")
	     (?\ǣ "a3")
	     (?\Ǥ "G/")
	     (?\ǥ "g/")
	     (?\Ǧ "G")
	     (?\ǧ "g")
	     (?\Ǩ "K")
	     (?\ǩ "k")
	     (?\Ǫ "O")
	     (?\ǫ "o")
	     (?\Ǭ "O1")
	     (?\ǭ "o1")
	     (?\Ǯ "EZ")
	     (?\ǯ "ez")
	     (?\ǰ "j")
	     (?\Ǵ "G")
	     (?\ǵ "g")
	     (?\Ǻ "AA'")
	     (?\ǻ "aa'")
	     (?\Ǽ "AE'")
	     (?\ǽ "ae'")
	     (?\Ǿ "O/'")
	     (?\ǿ "o/'")
	     (?\Ȁ "A!!")
	     (?\ȁ "a!!")
	     (?\Ȃ "A)")
	     (?\ȃ "a)")
	     (?\Ȅ "E!!")
	     (?\ȅ "e!!")
	     (?\Ȇ "E)")
	     (?\ȇ "e)")
	     (?\Ȉ "I!!")
	     (?\ȉ "i!!")
	     (?\Ȋ "I)")
	     (?\ȋ "i)")
	     (?\Ȍ "O!!")
	     (?\ȍ "o!!")
	     (?\Ȏ "O)")
	     (?\ȏ "o)")
	     (?\Ȑ "R!!")
	     (?\ȑ "r!!")
	     (?\Ȓ "R)")
	     (?\ȓ "r)")
	     (?\Ȕ "U!!")
	     (?\ȕ "u!!")
	     (?\Ȗ "U)")
	     (?\ȗ "u)")
	     (?\ȝ "Z")
	     (?\ɑ "A")
	     (?\ɒ "A.")
	     (?\ɓ "b`")
	     (?\ɔ "O")
	     (?\ɖ "d.")
	     (?\ɗ "d`")
	     (?\ɘ "@<umd>")
	     (?\ə "@")
	     (?\ɚ "R")
	     (?\ɛ "E")
	     (?\ɜ "V\"")
	     (?\ɝ "R<umd>")
	     (?\ɞ "O\"")
	     (?\ɟ "J")
	     (?\ɠ "g`")
	     (?\ɡ "g")
	     (?\ɢ "G")
	     (?\ɣ "Q")
	     (?\ɤ "o-")
	     (?\ɥ "j<rnd>")
	     (?\ɦ "h<?>")
	     (?\ɨ "i\"")
	     (?\ɩ "I")
	     (?\ɪ "I")
	     (?\ɫ "L")
	     (?\ɬ "L")
	     (?\ɭ "l.")
	     (?\ɮ "z<lat>")
	     (?\ɯ "u-")
	     (?\ɰ "j<vel>")
	     (?\ɱ "M")
	     (?\ɳ "n.")
	     (?\ɴ "n\"")
	     (?\ɵ "@.")
	     (?\ɶ "&.")
	     (?\ɷ "U")
	     (?\ɹ "r")
	     (?\ɺ "*<lat>")
	     (?\ɻ "r.")
	     (?\ɽ "*.")
	     (?\ɾ "*")
	     (?\ʀ "R")
	     (?\ʁ "g\"")
	     (?\ʂ "s.")
	     (?\ʃ "S")
	     (?\ʄ "J`")
	     (?\ʇ "t!")
	     (?\ʈ "t.")
	     (?\ʉ "u\"")
	     (?\ʊ "U")
	     (?\ʋ "r<lbd>")
	     (?\ʌ "V")
	     (?\ʍ "w<vls>")
	     (?\ʎ "l^")
	     (?\ʏ "I.")
	     (?\ʐ "z.")
	     (?\ʒ "Z")
	     (?\ʔ "?")
	     (?\ʕ "H<vcd>")
	     (?\ʖ "l!")
	     (?\ʗ "c!")
	     (?\ʘ "p!")
	     (?\ʙ "b<trl>")
	     (?\ʛ "G`")
	     (?\ʝ "j")
	     (?\ʞ "k!")
	     (?\ʟ "L")
	     (?\ʠ "q`")
	     (?\ʤ "d3")
	     (?\ʦ "ts")
	     (?\ʧ "tS")
	     (?\ʰ "<h>")
	     (?\ʱ "<?>")
	     (?\ʲ ";")
	     (?\ʳ "<r>")
	     (?\ʷ "<w>")
	     (?\ʻ ";S")
	     (?\ʼ "`")
	     (?\ˆ "^")
	     (?\ˇ "'<")
	     (?\ˈ "|")
	     (?\ˉ "1-")
	     (?\ˋ "1!")
	     (?\ː ":")
	     (?\ˑ ":\\")
	     (?\˖ "+")
	     (?\˗ "-")
	     (?\˘ "'(")
	     (?\˙ "'.")
	     (?\˚ "'0")
	     (?\˛ "';")
	     (?\˜ "~")
	     (?\˝ "'\"")
	     (?\˥ "_T")
	     (?\˦ "_H")
	     (?\˧ "_M")
	     (?\˨ "_L")
	     (?\˩ "_B")
	     (?\ˬ "_v")
	     (?\ˮ "''")
	     (?\̀ "`")
	     (?\́ "'")
	     (?\̂ "^")
	     (?\̃ "~")
	     (?\̄ "¯")
	     (?\̇ "·")
	     (?\̈ "¨")
	     (?\̊ "°")
	     (?\̋ "''")
	     (?\̍ "|")
	     (?\̎ "||")
	     (?\̏ "``")
	     (?\̡ ";")
	     (?\̢ ".")
	     (?\̣ ".")
	     (?\̤ "<?>")
	     (?\̥ "<o>")
	     (?\̦ ",")
	     (?\̧ "¸")
	     (?\̩ "-")
	     (?\̪ "[")
	     (?\̫ "<w>")
	     (?\̴ "~")
	     (?\̷ "/")
	     (?\̸ "/")
	     (?\̀ "`")
	     (?\́ "'")
	     (?\͂ "~")
	     (?\̈́ "'%")
	     (?\ͅ "j3")
	     (?\͇ "=")
	     (?\͠ "~~")
	     (?\ʹ "'")
	     (?\͵ ",")
	     (?\ͺ "j3")
	     (?\; "?%")
	     (?\΄ "'*")
	     (?\΅ "'%")
	     (?\Ά "A'")
	     (?\· "·")
	     (?\Έ "E'")
	     (?\Ή "Y%")
	     (?\Ί "I'")
	     (?\Ό "O'")
	     (?\Ύ "U%")
	     (?\Ώ "W%")
	     (?\ΐ "i3")
	     (?\Α "A")
	     (?\Β "B")
	     (?\Γ "G")
	     (?\Δ "D")
	     (?\Ε "E")
	     (?\Ζ "Z")
	     (?\Η "Y")
	     (?\Θ "TH")
	     (?\Ι "I")
	     (?\Κ "K")
	     (?\Λ "L")
	     (?\Μ "M")
	     (?\Ν "N")
	     (?\Ξ "C")
	     (?\Ο "O")
	     (?\Π "P")
	     (?\Ρ "R")
	     (?\Σ "S")
	     (?\Τ "T")
	     (?\Υ "U")
	     (?\Φ "F")
	     (?\Χ "X")
	     (?\Ψ "Q")
	     (?\Ω "W*")
	     (?\Ϊ "J")
	     (?\Ϋ "V*")
	     (?\ά "a'")
	     (?\έ "e'")
	     (?\ή "y%")
	     (?\ί "i'")
	     (?\ΰ "u3")
	     (?\α "a")
	     (?\β "b")
	     (?\γ "g")
	     (?\δ "d")
	     (?\ε "e")
	     (?\ζ "z")
	     (?\η "y")
	     (?\θ "th")
	     (?\ι "i")
	     (?\κ "k")
	     (?\λ "l")
	     (?\μ "µ")
	     (?\ν "n")
	     (?\ξ "c")
	     (?\ο "o")
	     (?\π "p")
	     (?\ρ "r")
	     (?\ς "*s")
	     (?\σ "s")
	     (?\τ "t")
	     (?\υ "u")
	     (?\φ "f")
	     (?\χ "x")
	     (?\ψ "q")
	     (?\ω "w")
	     (?\ϊ "j")
	     (?\ϋ "v*")
	     (?\ό "o'")
	     (?\ύ "u%")
	     (?\ώ "w%")
	     (?\ϐ "beta ")
	     (?\ϑ "theta ")
	     (?\ϒ "upsi ")
	     (?\ϕ "phi ")
	     (?\ϖ "pi ")
	     (?\ϗ "k.")
	     (?\Ϛ "T3")
	     (?\ϛ "t3")
	     (?\Ϝ "M3")
	     (?\ϝ "m3")
	     (?\Ϟ "K3")
	     (?\ϟ "k3")
	     (?\Ϡ "P3")
	     (?\ϡ "p3")
	     (?\ϰ "kappa ")
	     (?\ϱ "rho ")
	     (?\ϳ "J")
	     (?\ϴ "'%")
	     (?\ϵ "j3")
	     (?\Ё "IO")
	     (?\Ђ "D%")
	     (?\Ѓ "G%")
	     (?\Є "IE")
	     (?\Ѕ "DS")
	     (?\І "II")
	     (?\Ї "YI")
	     (?\Ј "J%")
	     (?\Љ "LJ")
	     (?\Њ "NJ")
	     (?\Ћ "Ts")
	     (?\Ќ "KJ")
	     (?\Ў "V%")
	     (?\Џ "DZ")
	     (?\А "A")
	     (?\Б "B")
	     (?\В "V")
	     (?\Г "G")
	     (?\Д "D")
	     (?\Е "E")
	     (?\Ж "ZH")
	     (?\З "Z")
	     (?\И "I")
	     (?\Й "J")
	     (?\К "K")
	     (?\Л "L")
	     (?\М "M")
	     (?\Н "N")
	     (?\О "O")
	     (?\П "P")
	     (?\Р "R")
	     (?\С "S")
	     (?\Т "T")
	     (?\У "U")
	     (?\Ф "F")
	     (?\Х "H")
	     (?\Ц "C")
	     (?\Ч "CH")
	     (?\Ш "SH")
	     (?\Щ "SCH")
	     (?\Ъ "\"")
	     (?\Ы "Y")
	     (?\Ь "'")
	     (?\Э "`E")
	     (?\Ю "YU")
	     (?\Я "YA")
	     (?\а "a")
	     (?\б "b")
	     (?\в "v")
	     (?\г "g")
	     (?\д "d")
	     (?\е "e")
	     (?\ж "zh")
	     (?\з "z")
	     (?\и "i")
	     (?\й "j")
	     (?\к "k")
	     (?\л "l")
	     (?\м "m")
	     (?\н "n")
	     (?\о "o")
	     (?\п "p")
	     (?\р "r")
	     (?\с "s")
	     (?\т "t")
	     (?\у "u")
	     (?\ф "f")
	     (?\х "h")
	     (?\ц "c")
	     (?\ч "ch")
	     (?\ш "sh")
	     (?\щ "sch")
	     (?\ъ "\"")
	     (?\ы "y")
	     (?\ь "'")
	     (?\э "`e")
	     (?\ю "yu")
	     (?\я "ya")
	     (?\ё "io")
	     (?\ђ "d%")
	     (?\ѓ "g%")
	     (?\є "ie")
	     (?\ѕ "ds")
	     (?\і "ii")
	     (?\ї "yi")
	     (?\ј "j%")
	     (?\љ "lj")
	     (?\њ "nj")
	     (?\ћ "ts")
	     (?\ќ "kj")
	     (?\ў "v%")
	     (?\џ "dz")
	     (?\Ѣ "Y3")
	     (?\ѣ "y3")
	     (?\Ѫ "O3")
	     (?\ѫ "o3")
	     (?\Ѳ "F3")
	     (?\ѳ "f3")
	     (?\Ѵ "V3")
	     (?\ѵ "v3")
	     (?\Ҁ "C3")
	     (?\ҁ "c3")
	     (?\Ґ "G3")
	     (?\ґ "g3")
	     (?\Ӕ "AE")
	     (?\ӕ "ae")
	     (?\ִ "i")
	     (?\ַ "a")
	     (?\ָ "o")
	     (?\ּ "u")
	     (?\ֿ "h")
	     (?\ׂ ":")
	     (?\א "#")
	     (?\ב "B+")
	     (?\ג "G+")
	     (?\ד "D+")
	     (?\ה "H+")
	     (?\ו "W+")
	     (?\ז "Z+")
	     (?\ח "X+")
	     (?\ט "Tj")
	     (?\י "J+")
	     (?\ך "K%")
	     (?\כ "K+")
	     (?\ל "L+")
	     (?\ם "M%")
	     (?\מ "M+")
	     (?\ן "N%")
	     (?\נ "N+")
	     (?\ס "S+")
	     (?\ע "E+")
	     (?\ף "P%")
	     (?\פ "P+")
	     (?\ץ "Zj")
	     (?\צ "ZJ")
	     (?\ק "Q+")
	     (?\ר "R+")
	     (?\ש "Sh")
	     (?\ת "T+")
	     (?\װ "v")
	     (?\ױ "oy")
	     (?\ײ "ey")
	     (?\، ",+")
	     (?\؛ ";+")
	     (?\؟ "?+")
	     (?\ء "H'")
	     (?\آ "aM")
	     (?\أ "aH")
	     (?\ؤ "wH")
	     (?\إ "ah")
	     (?\ئ "yH")
	     (?\ا "a+")
	     (?\ب "b+")
	     (?\ة "tm")
	     (?\ت "t+")
	     (?\ث "tk")
	     (?\ج "g+")
	     (?\ح "hk")
	     (?\خ "x+")
	     (?\د "d+")
	     (?\ذ "dk")
	     (?\ر "r+")
	     (?\ز "z+")
	     (?\س "s+")
	     (?\ش "sn")
	     (?\ص "c+")
	     (?\ض "dd")
	     (?\ط "tj")
	     (?\ظ "zH")
	     (?\ع "e+")
	     (?\غ "i+")
	     (?\ـ "++")
	     (?\ف "f+")
	     (?\ق "q+")
	     (?\ك "k+")
	     (?\ل "l+")
	     (?\م "m+")
	     (?\ن "n+")
	     (?\ه "h+")
	     (?\و "w+")
	     (?\ى "j+")
	     (?\ي "y+")
	     (?\ً ":+")
	     (?\ٌ "\"+")
	     (?\ٍ "=+")
	     (?\َ "/+")
	     (?\ُ "'+")
	     (?\ِ "1+")
	     (?\ّ "3+")
	     (?\ْ "0+")
	     (?\٠ "0a")
	     (?\١ "1a")
	     (?\٢ "2a")
	     (?\٣ "3a")
	     (?\٤ "4a")
	     (?\٥ "5a")
	     (?\٦ "6a")
	     (?\٧ "7a")
	     (?\٨ "8a")
	     (?\٩ "9a")
	     (?\ٰ "aS")
	     (?\پ "p+")
	     (?\ځ "hH")
	     (?\چ "tc")
	     (?\ژ "zj")
	     (?\ڤ "v+")
	     (?\گ "gf")
	     (?\۰ "0a")
	     (?\۱ "1a")
	     (?\۲ "2a")
	     (?\۳ "3a")
	     (?\۴ "4a")
	     (?\۵ "5a")
	     (?\۶ "6a")
	     (?\۷ "7a")
	     (?\۸ "8a")
	     (?\۹ "9a")
	     (?\ሀ "he")
	     (?\ሁ "hu")
	     (?\ሂ "hi")
	     (?\ሃ "ha")
	     (?\ሄ "hE")
	     (?\ህ "h")
	     (?\ሆ "ho")
	     (?\ለ "le")
	     (?\ሉ "lu")
	     (?\ሊ "li")
	     (?\ላ "la")
	     (?\ሌ "lE")
	     (?\ል "l")
	     (?\ሎ "lo")
	     (?\ሏ "lWa")
	     (?\ሐ "He")
	     (?\ሑ "Hu")
	     (?\ሒ "Hi")
	     (?\ሓ "Ha")
	     (?\ሔ "HE")
	     (?\ሕ "H")
	     (?\ሖ "Ho")
	     (?\ሗ "HWa")
	     (?\መ "me")
	     (?\ሙ "mu")
	     (?\ሚ "mi")
	     (?\ማ "ma")
	     (?\ሜ "mE")
	     (?\ም "m")
	     (?\ሞ "mo")
	     (?\ሟ "mWa")
	     (?\ሠ "`se")
	     (?\ሡ "`su")
	     (?\ሢ "`si")
	     (?\ሣ "`sa")
	     (?\ሤ "`sE")
	     (?\ሥ "`s")
	     (?\ሦ "`so")
	     (?\ሧ "`sWa")
	     (?\ረ "re")
	     (?\ሩ "ru")
	     (?\ሪ "ri")
	     (?\ራ "ra")
	     (?\ሬ "rE")
	     (?\ር "r")
	     (?\ሮ "ro")
	     (?\ሯ "rWa")
	     (?\ሰ "se")
	     (?\ሱ "su")
	     (?\ሲ "si")
	     (?\ሳ "sa")
	     (?\ሴ "sE")
	     (?\ስ "s")
	     (?\ሶ "so")
	     (?\ሷ "sWa")
	     (?\ሸ "xe")
	     (?\ሹ "xu")
	     (?\ሺ "xi")
	     (?\ሻ "xa")
	     (?\ሼ "xE")
	     (?\ሽ "xa")
	     (?\ሾ "xo")
	     (?\ሿ "xWa")
	     (?\ቀ "qe")
	     (?\ቁ "qu")
	     (?\ቂ "qi")
	     (?\ቃ "qa")
	     (?\ቄ "qE")
	     (?\ቅ "q")
	     (?\ቆ "qo")
	     (?\ቈ "qWe")
	     (?\ቊ "qWi")
	     (?\ቋ "qWa")
	     (?\ቌ "qWE")
	     (?\ቍ "qW")
	     (?\ቐ "Qe")
	     (?\ቑ "Qu")
	     (?\ቒ "Qi")
	     (?\ቓ "Qa")
	     (?\ቔ "QE")
	     (?\ቕ "Q")
	     (?\ቖ "Qo")
	     (?\ቘ "QWe")
	     (?\ቚ "QWi")
	     (?\ቛ "QWa")
	     (?\ቜ "QWE")
	     (?\ቝ "QW")
	     (?\በ "be")
	     (?\ቡ "bu")
	     (?\ቢ "bi")
	     (?\ባ "ba")
	     (?\ቤ "bE")
	     (?\ብ "b")
	     (?\ቦ "bo")
	     (?\ቧ "bWa")
	     (?\ቨ "ve")
	     (?\ቩ "vu")
	     (?\ቪ "vi")
	     (?\ቫ "va")
	     (?\ቬ "vE")
	     (?\ቭ "v")
	     (?\ቮ "vo")
	     (?\ቯ "vWa")
	     (?\ተ "te")
	     (?\ቱ "tu")
	     (?\ቲ "ti")
	     (?\ታ "ta")
	     (?\ቴ "tE")
	     (?\ት "t")
	     (?\ቶ "to")
	     (?\ቷ "tWa")
	     (?\ቸ "ce")
	     (?\ቹ "cu")
	     (?\ቺ "ci")
	     (?\ቻ "ca")
	     (?\ቼ "cE")
	     (?\ች "c")
	     (?\ቾ "co")
	     (?\ቿ "cWa")
	     (?\ኀ "`he")
	     (?\ኁ "`hu")
	     (?\ኂ "`hi")
	     (?\ኃ "`ha")
	     (?\ኄ "`hE")
	     (?\ኅ "`h")
	     (?\ኆ "`ho")
	     (?\ኈ "hWe")
	     (?\ኊ "hWi")
	     (?\ኋ "hWa")
	     (?\ኌ "hWE")
	     (?\ኍ "hW")
	     (?\ነ "na")
	     (?\ኑ "nu")
	     (?\ኒ "ni")
	     (?\ና "na")
	     (?\ኔ "nE")
	     (?\ን "n")
	     (?\ኖ "no")
	     (?\ኗ "nWa")
	     (?\ኘ "Ne")
	     (?\ኙ "Nu")
	     (?\ኚ "Ni")
	     (?\ኛ "Na")
	     (?\ኜ "NE")
	     (?\ኝ "N")
	     (?\ኞ "No")
	     (?\ኟ "NWa")
	     (?\አ "e")
	     (?\ኡ "u")
	     (?\ኢ "i")
	     (?\ኣ "a")
	     (?\ኤ "E")
	     (?\እ "I")
	     (?\ኦ "o")
	     (?\ኧ "e3")
	     (?\ከ "ke")
	     (?\ኩ "ku")
	     (?\ኪ "ki")
	     (?\ካ "ka")
	     (?\ኬ "kE")
	     (?\ክ "k")
	     (?\ኮ "ko")
	     (?\ኰ "kWe")
	     (?\ኲ "kWi")
	     (?\ኳ "kWa")
	     (?\ኴ "kWE")
	     (?\ኵ "kW")
	     (?\ኸ "Ke")
	     (?\ኹ "Ku")
	     (?\ኺ "Ki")
	     (?\ኻ "Ka")
	     (?\ኼ "KE")
	     (?\ኽ "K")
	     (?\ኾ "Ko")
	     (?\ዀ "KWe")
	     (?\ዂ "KWi")
	     (?\ዃ "KWa")
	     (?\ዄ "KWE")
	     (?\ዅ "KW")
	     (?\ወ "we")
	     (?\ዉ "wu")
	     (?\ዊ "wi")
	     (?\ዋ "wa")
	     (?\ዌ "wE")
	     (?\ው "w")
	     (?\ዎ "wo")
	     (?\ዐ "`e")
	     (?\ዑ "`u")
	     (?\ዒ "`i")
	     (?\ዓ "`a")
	     (?\ዔ "`E")
	     (?\ዕ "`I")
	     (?\ዖ "`o")
	     (?\ዘ "ze")
	     (?\ዙ "zu")
	     (?\ዚ "zi")
	     (?\ዛ "za")
	     (?\ዜ "zE")
	     (?\ዝ "z")
	     (?\ዞ "zo")
	     (?\ዟ "zWa")
	     (?\ዠ "Ze")
	     (?\ዡ "Zu")
	     (?\ዢ "Zi")
	     (?\ዣ "Za")
	     (?\ዤ "ZE")
	     (?\ዥ "Z")
	     (?\ዦ "Zo")
	     (?\ዧ "ZWa")
	     (?\የ "ye")
	     (?\ዩ "yu")
	     (?\ዪ "yi")
	     (?\ያ "ya")
	     (?\ዬ "yE")
	     (?\ይ "y")
	     (?\ዮ "yo")
	     (?\ዯ "yWa")
	     (?\ደ "de")
	     (?\ዱ "du")
	     (?\ዲ "di")
	     (?\ዳ "da")
	     (?\ዴ "dE")
	     (?\ድ "d")
	     (?\ዶ "do")
	     (?\ዷ "dWa")
	     (?\ዸ "De")
	     (?\ዹ "Du")
	     (?\ዺ "Di")
	     (?\ዻ "Da")
	     (?\ዼ "DE")
	     (?\ዽ "D")
	     (?\ዾ "Do")
	     (?\ዿ "DWa")
	     (?\ጀ "je")
	     (?\ጁ "ju")
	     (?\ጂ "ji")
	     (?\ጃ "ja")
	     (?\ጄ "jE")
	     (?\ጅ "j")
	     (?\ጆ "jo")
	     (?\ጇ "jWa")
	     (?\ገ "ga")
	     (?\ጉ "gu")
	     (?\ጊ "gi")
	     (?\ጋ "ga")
	     (?\ጌ "gE")
	     (?\ግ "g")
	     (?\ጎ "go")
	     (?\ጐ "gWu")
	     (?\ጒ "gWi")
	     (?\ጓ "gWa")
	     (?\ጔ "gWE")
	     (?\ጕ "gW")
	     (?\ጘ "Ge")
	     (?\ጙ "Gu")
	     (?\ጚ "Gi")
	     (?\ጛ "Ga")
	     (?\ጜ "GE")
	     (?\ጝ "G")
	     (?\ጞ "Go")
	     (?\ጟ "GWa")
	     (?\ጠ "Te")
	     (?\ጡ "Tu")
	     (?\ጢ "Ti")
	     (?\ጣ "Ta")
	     (?\ጤ "TE")
	     (?\ጥ "T")
	     (?\ጦ "To")
	     (?\ጧ "TWa")
	     (?\ጨ "Ce")
	     (?\ጩ "Ca")
	     (?\ጪ "Cu")
	     (?\ጫ "Ca")
	     (?\ጬ "CE")
	     (?\ጭ "C")
	     (?\ጮ "Co")
	     (?\ጯ "CWa")
	     (?\ጰ "Pe")
	     (?\ጱ "Pu")
	     (?\ጲ "Pi")
	     (?\ጳ "Pa")
	     (?\ጴ "PE")
	     (?\ጵ "P")
	     (?\ጶ "Po")
	     (?\ጷ "PWa")
	     (?\ጸ "SWe")
	     (?\ጹ "SWu")
	     (?\ጺ "SWi")
	     (?\ጻ "SWa")
	     (?\ጼ "SWE")
	     (?\ጽ "SW")
	     (?\ጾ "SWo")
	     (?\ጿ "SWa")
	     (?\ፀ "`Sa")
	     (?\ፁ "`Su")
	     (?\ፂ "`Si")
	     (?\ፃ "`Sa")
	     (?\ፄ "`SE")
	     (?\ፅ "`S")
	     (?\ፆ "`So")
	     (?\ፈ "fa")
	     (?\ፉ "fu")
	     (?\ፊ "fi")
	     (?\ፋ "fa")
	     (?\ፌ "fE")
	     (?\ፍ "o")
	     (?\ፎ "fo")
	     (?\ፏ "fWa")
	     (?\ፐ "pe")
	     (?\ፑ "pu")
	     (?\ፒ "pi")
	     (?\ፓ "pa")
	     (?\ፔ "pE")
	     (?\ፕ "p")
	     (?\ፖ "po")
	     (?\ፗ "pWa")
	     (?\ፘ "mYa")
	     (?\ፙ "rYa")
	     (?\ፚ "fYa")
	     (?\፠ " ")
	     (?\፡ ":")
	     (?\። "::")
	     (?\፣ ",")
	     (?\፤ ";")
	     (?\፥ "-:")
	     (?\፦ ":-")
	     (?\፧ "`?")
	     (?\፨ ":|:")
	     (?\፩ "`1")
	     (?\፪ "`2")
	     (?\፫ "`3")
	     (?\፬ "`4")
	     (?\፭ "`5")
	     (?\፮ "`6")
	     (?\፯ "`7")
	     (?\፰ "`8")
	     (?\፱ "`9")
	     (?\፲ "`10")
	     (?\፳ "`20")
	     (?\፴ "`30")
	     (?\፵ "`40")
	     (?\፶ "`50")
	     (?\፷ "`60")
	     (?\፸ "`70")
	     (?\፹ "`80")
	     (?\፺ "`90")
	     (?\፻ "`100")
	     (?\፼ "`10000")
	     (?\Ḁ "A-0")
	     (?\ḁ "a-0")
	     (?\Ḃ "B.")
	     (?\ḃ "b.")
	     (?\Ḅ "B-.")
	     (?\ḅ "b-.")
	     (?\Ḇ "B_")
	     (?\ḇ "b_")
	     (?\Ḉ "C,'")
	     (?\ḉ "c,'")
	     (?\Ḋ "D.")
	     (?\ḋ "d.")
	     (?\Ḍ "D-.")
	     (?\ḍ "d-.")
	     (?\Ḏ "D_")
	     (?\ḏ "d_")
	     (?\Ḑ "D,")
	     (?\ḑ "d,")
	     (?\Ḓ "D->")
	     (?\ḓ "d->")
	     (?\Ḕ "E-!")
	     (?\ḕ "e-!")
	     (?\Ḗ "E-'")
	     (?\ḗ "e-'")
	     (?\Ḙ "E->")
	     (?\ḙ "e->")
	     (?\Ḛ "E-?")
	     (?\ḛ "e-?")
	     (?\Ḝ "E,(")
	     (?\ḝ "e,(")
	     (?\Ḟ "F.")
	     (?\ḟ "f.")
	     (?\Ḡ "G-")
	     (?\ḡ "g-")
	     (?\Ḣ "H.")
	     (?\ḣ "h.")
	     (?\Ḥ "H-.")
	     (?\ḥ "h-.")
	     (?\Ḧ "H:")
	     (?\ḧ "h:")
	     (?\Ḩ "H,")
	     (?\ḩ "h,")
	     (?\Ḫ "H-(")
	     (?\ḫ "h-(")
	     (?\Ḭ "I-?")
	     (?\ḭ "i-?")
	     (?\Ḯ "I:'")
	     (?\ḯ "i:'")
	     (?\Ḱ "K'")
	     (?\ḱ "k'")
	     (?\Ḳ "K-.")
	     (?\ḳ "k-.")
	     (?\Ḵ "K_")
	     (?\ḵ "k_")
	     (?\Ḷ "L-.")
	     (?\ḷ "l-.")
	     (?\Ḹ "L--.")
	     (?\ḹ "l--.")
	     (?\Ḻ "L_")
	     (?\ḻ "l_")
	     (?\Ḽ "L->")
	     (?\ḽ "l->")
	     (?\Ḿ "M'")
	     (?\ḿ "m'")
	     (?\Ṁ "M.")
	     (?\ṁ "m.")
	     (?\Ṃ "M-.")
	     (?\ṃ "m-.")
	     (?\Ṅ "N.")
	     (?\ṅ "n.")
	     (?\Ṇ "N-.")
	     (?\ṇ "n-.")
	     (?\Ṉ "N_")
	     (?\ṉ "n_")
	     (?\Ṋ "N->")
	     (?\ṋ "n->")
	     (?\Ṍ "O?'")
	     (?\ṍ "o?'")
	     (?\Ṏ "O?:")
	     (?\ṏ "o?:")
	     (?\Ṑ "O-!")
	     (?\ṑ "o-!")
	     (?\Ṓ "O-'")
	     (?\ṓ "o-'")
	     (?\Ṕ "P'")
	     (?\ṕ "p'")
	     (?\Ṗ "P.")
	     (?\ṗ "p.")
	     (?\Ṙ "R.")
	     (?\ṙ "r.")
	     (?\Ṛ "R-.")
	     (?\ṛ "r-.")
	     (?\Ṝ "R--.")
	     (?\ṝ "r--.")
	     (?\Ṟ "R_")
	     (?\ṟ "r_")
	     (?\Ṡ "S.")
	     (?\ṡ "s.")
	     (?\Ṣ "S-.")
	     (?\ṣ "s-.")
	     (?\Ṥ "S'.")
	     (?\ṥ "s'.")
	     (?\Ṧ "S<.")
	     (?\ṧ "s<.")
	     (?\Ṩ "S.-.")
	     (?\ṩ "s.-.")
	     (?\Ṫ "T.")
	     (?\ṫ "t.")
	     (?\Ṭ "T-.")
	     (?\ṭ "t-.")
	     (?\Ṯ "T_")
	     (?\ṯ "t_")
	     (?\Ṱ "T->")
	     (?\ṱ "t->")
	     (?\Ṳ "U--:")
	     (?\ṳ "u--:")
	     (?\Ṵ "U-?")
	     (?\ṵ "u-?")
	     (?\Ṷ "U->")
	     (?\ṷ "u->")
	     (?\Ṹ "U?'")
	     (?\ṹ "u?'")
	     (?\Ṻ "U-:")
	     (?\ṻ "u-:")
	     (?\Ṽ "V?")
	     (?\ṽ "v?")
	     (?\Ṿ "V-.")
	     (?\ṿ "v-.")
	     (?\Ẁ "W!")
	     (?\ẁ "w!")
	     (?\Ẃ "W'")
	     (?\ẃ "w'")
	     (?\Ẅ "W:")
	     (?\ẅ "w:")
	     (?\Ẇ "W.")
	     (?\ẇ "w.")
	     (?\Ẉ "W-.")
	     (?\ẉ "w-.")
	     (?\Ẋ "X.")
	     (?\ẋ "x.")
	     (?\Ẍ "X:")
	     (?\ẍ "x:")
	     (?\Ẏ "Y.")
	     (?\ẏ "y.")
	     (?\Ẑ "Z>")
	     (?\ẑ "z>")
	     (?\Ẓ "Z-.")
	     (?\ẓ "z-.")
	     (?\Ẕ "Z_")
	     (?\ẕ "z_")
	     (?\ẖ "h_")
	     (?\ẗ "t:")
	     (?\ẘ "w0")
	     (?\ẙ "y0")
	     (?\Ạ "A-.")
	     (?\ạ "a-.")
	     (?\Ả "A2")
	     (?\ả "a2")
	     (?\Ấ "A>'")
	     (?\ấ "a>'")
	     (?\Ầ "A>!")
	     (?\ầ "a>!")
	     (?\Ẩ "A>2")
	     (?\ẩ "a>2")
	     (?\Ẫ "A>?")
	     (?\ẫ "a>?")
	     (?\Ậ "A>-.")
	     (?\ậ "a>-.")
	     (?\Ắ "A('")
	     (?\ắ "a('")
	     (?\Ằ "A(!")
	     (?\ằ "a(!")
	     (?\Ẳ "A(2")
	     (?\ẳ "a(2")
	     (?\Ẵ "A(?")
	     (?\ẵ "a(?")
	     (?\Ặ "A(-.")
	     (?\ặ "a(-.")
	     (?\Ẹ "E-.")
	     (?\ẹ "e-.")
	     (?\Ẻ "E2")
	     (?\ẻ "e2")
	     (?\Ẽ "E?")
	     (?\ẽ "e?")
	     (?\Ế "E>'")
	     (?\ế "e>'")
	     (?\Ề "E>!")
	     (?\ề "e>!")
	     (?\Ể "E>2")
	     (?\ể "e>2")
	     (?\Ễ "E>?")
	     (?\ễ "e>?")
	     (?\Ệ "E>-.")
	     (?\ệ "e>-.")
	     (?\Ỉ "I2")
	     (?\ỉ "i2")
	     (?\Ị "I-.")
	     (?\ị "i-.")
	     (?\Ọ "O-.")
	     (?\ọ "o-.")
	     (?\Ỏ "O2")
	     (?\ỏ "o2")
	     (?\Ố "O>'")
	     (?\ố "o>'")
	     (?\Ồ "O>!")
	     (?\ồ "o>!")
	     (?\Ổ "O>2")
	     (?\ổ "o>2")
	     (?\Ỗ "O>?")
	     (?\ỗ "o>?")
	     (?\Ộ "O>-.")
	     (?\ộ "o>-.")
	     (?\Ớ "O9'")
	     (?\ớ "o9'")
	     (?\Ờ "O9!")
	     (?\ờ "o9!")
	     (?\Ở "O92")
	     (?\ở "o92")
	     (?\Ỡ "O9?")
	     (?\ỡ "o9?")
	     (?\Ợ "O9-.")
	     (?\ợ "o9-.")
	     (?\Ụ "U-.")
	     (?\ụ "u-.")
	     (?\Ủ "U2")
	     (?\ủ "u2")
	     (?\Ứ "U9'")
	     (?\ứ "u9'")
	     (?\Ừ "U9!")
	     (?\ừ "u9!")
	     (?\Ử "U92")
	     (?\ử "u92")
	     (?\Ữ "U9?")
	     (?\ữ "u9?")
	     (?\Ự "U9-.")
	     (?\ự "u9-.")
	     (?\Ỳ "Y!")
	     (?\ỳ "y!")
	     (?\Ỵ "Y-.")
	     (?\ỵ "y-.")
	     (?\Ỷ "Y2")
	     (?\ỷ "y2")
	     (?\Ỹ "Y?")
	     (?\ỹ "y?")
	     (?\ἀ "a")
	     (?\ἁ "ha")
	     (?\ἂ "`a")
	     (?\ἃ "h`a")
	     (?\ἄ "a'")
	     (?\ἅ "ha'")
	     (?\ἆ "a~")
	     (?\ἇ "ha~")
	     (?\Ἀ "A")
	     (?\Ἁ "hA")
	     (?\Ἂ "`A")
	     (?\Ἃ "h`A")
	     (?\Ἄ "A'")
	     (?\Ἅ "hA'")
	     (?\Ἆ "A~")
	     (?\Ἇ "hA~")
	     (?\ἑ "he")
	     (?\Ἑ "hE")
	     (?\ἱ "hi")
	     (?\Ἱ "hI")
	     (?\ὁ "ho")
	     (?\Ὁ "hO")
	     (?\ὑ "hu")
	     (?\Ὑ "hU")
	     (?\᾿ ",,")
	     (?\῀ "?*")
	     (?\῁ "?:")
	     (?\῍ ",!")
	     (?\῎ ",'")
	     (?\῏ "?,")
	     (?\῝ ";!")
	     (?\῞ ";'")
	     (?\῟ "?;")
	     (?\ῥ "rh")
	     (?\Ῥ "Rh")
	     (?\῭ "!:")
	     (?\` "!*")
	     (?\῾ ";;")
	     (?\  " ")
	     (?\  "  ")
	     (?\  " ")
	     (?\  "  ")
	     (?\  " ")
	     (?\  " ")
	     (?\  " ")
	     (?\  " ")
	     (?\  " ")
	     (?\  " ")
	     (?\‐ "-")
	     (?\‑ "-")
	     (?\– "-")
	     (?\— "--")
	     (?\― "-")
	     (?\‖ "||")
	     (?\‗ "=2")
	     (?\‘ "`")
	     (?\’ "'")
	     (?\‚ "'")
	     (?\‛ "'")
	     (?\“ "\"")
	     (?\” "\"")
	     (?\„ "\"")
	     (?\‟ "\"")
	     (?\† "/-")
	     (?\‡ "/=")
	     (?\• " o ")
	     (?\․ ".")
	     (?\‥ "..")
	     (?\… "...")
	     (?\‧ "·")
	     (?\‰ " 0/00")
	     (?\′ "'")
	     (?\″ "''")
	     (?\‴ "'''")
	     (?\‵ "`")
	     (?\‶ "``")
	     (?\‷ "```")
	     (?\‸ "Ca")
	     (?\‹ "<")
	     (?\› ">")
	     (?\※ ":X")
	     (?\‼ "!!")
	     (?\‾ "'-")
	     (?\⁃ "-")
	     (?\⁄ "/")
	     (?\⁈ "?!")
	     (?\⁉ "!?")
	     (?\⁰ "^0")
	     (?\⁴ "^4")
	     (?\⁵ "^5")
	     (?\⁶ "^6")
	     (?\⁷ "^7")
	     (?\⁸ "^8")
	     (?\⁹ "^9")
	     (?\⁺ "^+")
	     (?\⁻ "^-")
	     (?\⁼ "^=")
	     (?\⁽ "^(")
	     (?\⁾ "^)")
	     (?\ⁿ "^n")
	     (?\₀ "_0")
	     (?\₁ "_1")
	     (?\₂ "_2")
	     (?\₃ "_3")
	     (?\₄ "_4")
	     (?\₅ "_5")
	     (?\₆ "_6")
	     (?\₇ "_7")
	     (?\₈ "_8")
	     (?\₉ "_9")
	     (?\₊ "_+")
	     (?\₋ "_-")
	     (?\₌ "_=")
	     (?\₍ "(")
	     (?\₎ ")")
	     (?\₣ "Ff")
	     (?\₤ "Li")
	     (?\₧ "Pt")
	     (?\₩ "W=")
	     (?\€ "EUR")
	     (?\℀ "a/c")
	     (?\℁ "a/s")
	     (?\℃ "oC")
	     (?\℅ "c/o")
	     (?\℆ "c/u")
	     (?\℉ "oF")
	     (?\ℊ "g")
	     (?\ℎ "h")
	     (?\ℏ "\\hbar")
	     (?\ℑ "Im")
	     (?\ℓ "l")
	     (?\№ "No.")
	     (?\℗ "PO")
	     (?\℘ "P")
	     (?\ℜ "Re")
	     (?\℞ "Rx")
	     (?\℠ "(SM)")
	     (?\℡ "TEL")
	     (?\™ "(TM)")
	     (?\Ω "Ohm")
	     (?\K "K")
	     (?\Å "Ang.")
	     (?\℮ "est.")
	     (?\ℴ "o")
	     (?\ℵ "Aleph ")
	     (?\ℶ "Bet ")
	     (?\ℷ "Gimel ")
	     (?\ℸ "Dalet ")
	     (?\⅓ " 1/3")
	     (?\⅔ " 2/3")
	     (?\⅕ " 1/5")
	     (?\⅖ " 2/5")
	     (?\⅗ " 3/5")
	     (?\⅘ " 4/5")
	     (?\⅙ " 1/6")
	     (?\⅚ " 5/6")
	     (?\⅛ " 1/8")
	     (?\⅜ " 3/8")
	     (?\⅝ " 5/8")
	     (?\⅞ " 7/8")
	     (?\⅟ " 1/")
	     (?\Ⅰ "I")
	     (?\Ⅱ "II")
	     (?\Ⅲ "III")
	     (?\Ⅳ "IV")
	     (?\Ⅴ "V")
	     (?\Ⅵ "VI")
	     (?\Ⅶ "VII")
	     (?\Ⅷ "VIII")
	     (?\Ⅸ "IX")
	     (?\Ⅹ "X")
	     (?\Ⅺ "XI")
	     (?\Ⅻ "XII")
	     (?\Ⅼ "L")
	     (?\Ⅽ "C")
	     (?\Ⅾ "D")
	     (?\Ⅿ "M")
	     (?\ⅰ "i")
	     (?\ⅱ "ii")
	     (?\ⅲ "iii")
	     (?\ⅳ "iv")
	     (?\ⅴ "v")
	     (?\ⅵ "vi")
	     (?\ⅶ "vii")
	     (?\ⅷ "viii")
	     (?\ⅸ "ix")
	     (?\ⅹ "x")
	     (?\ⅺ "xi")
	     (?\ⅻ "xii")
	     (?\ⅼ "l")
	     (?\ⅽ "c")
	     (?\ⅾ "d")
	     (?\ⅿ "m")
	     (?\ↀ "1000RCD")
	     (?\ↁ "5000R")
	     (?\ↂ "10000R")
	     (?\← "<-")
	     (?\↑ "-^")
	     (?\→ "->")
	     (?\↓ "-v")
	     (?\↔ "<->")
	     (?\↕ "UD")
	     (?\↖ "<!!")
	     (?\↗ "//>")
	     (?\↘ "!!>")
	     (?\↙ "<//")
	     (?\↨ "UD-")
	     (?\↵ "RET")
	     (?\⇀ ">V")
	     (?\⇐ "<=")
	     (?\⇑ "^^")
	     (?\⇒ "=>")
	     (?\⇓ "vv")
	     (?\⇔ "<=>")
	     (?\∀ "FA")
	     (?\∂ "\\partial")
	     (?\∃ "TE")
	     (?\∅ "{}")
	     (?\∆ "Delta")
	     (?\∇ "Nabla")
	     (?\∈ "(-")
	     (?\∉ "!(-")
	     (?\∊ "(-")
	     (?\∋ "-)")
	     (?\∌ "!-)")
	     (?\∍ "-)")
	     (?\∎ " qed")
	     (?\∏ "\\prod")
	     (?\∑ "\\sum")
	     (?\− " -")
	     (?\∓ "-/+")
	     (?\∔ ".+")
	     (?\∕ "/")
	     (?\∖ " - ")
	     (?\∗ "*")
	     (?\∘ " ° ")
	     (?\∙ "sb")
	     (?\√ " SQRT ")
	     (?\∛ " ROOT³ ")
	     (?\∜ " ROOT4 ")
	     (?\∝ "0(")
	     (?\∞ "infty")
	     (?\∟ "-L")
	     (?\∠ "-V")
	     (?\∥ "PP")
	     (?\∦ " !PP ")
	     (?\∧ "AND")
	     (?\∨ "OR")
	     (?\∩ "(U")
	     (?\∪ ")U")
	     (?\∫ "\\int ")
	     (?\∬ "DI")
	     (?\∮ "Io")
	     (?\∴ ".:")
	     (?\∵ ":.")
	     (?\∶ ":R")
	     (?\∷ "::")
	     (?\∼ "?1")
	     (?\∾ "CG")
	     (?\≃ "?-")
	     (?\≅ "?=")
	     (?\≈ "~=")
	     (?\≉ " !~= ")
	     (?\≌ "=?")
	     (?\≓ "HI")
	     (?\≔ ":=")
	     (?\≕ "=:")
	     (?\≠ "!=")
	     (?\≡ "=3")
	     (?\≢ " !=3 ")
	     (?\≤ "=<")
	     (?\≥ ">=")
	     (?\≦ ".LE.")
	     (?\≧ ".GE.")
	     (?\≨ ".LT.NOT.EQ.")
	     (?\≩ ".GT.NOT.EQ.")
	     (?\≪ "<<")
	     (?\≫ ">>")
	     (?\≮ "!<")
	     (?\≯ "!>")
	     (?\≶ " <> ")
	     (?\≷ " >< ")
	     (?\⊂ "(C")
	     (?\⊃ ")C")
	     (?\⊄ " !(C ")
	     (?\⊅ " !)C ")
	     (?\⊆ "(_")
	     (?\⊇ ")_")
	     (?\⊕ "(+)")
	     (?\⊖ "(-)")
	     (?\⊗ "(×)")
	     (?\⊘ "(/)")
	     (?\⊙ "(·)")
	     (?\⊚ "(°)")
	     (?\⊛ "(*)")
	     (?\⊜ "(=)")
	     (?\⊝ "(-)")
	     (?\⊞ "[+]")
	     (?\⊟ "[-]")
	     (?\⊠ "[×]")
	     (?\⊡ "[·]")
	     (?\⊥ "-T")
	     (?\⊧ " MODELS ")
	     (?\⊨ " TRUE ")
	     (?\⊩ " FORCES ")
	     (?\⊬ " !PROVES ")
	     (?\⊭ " NOT TRUE ")
	     (?\⊮ " !FORCES ")
	     (?\⊲ " NORMAL SUBGROUP OF ")
	     (?\⊳ " CONTAINS AS NORMAL SUBGROUP ")
	     (?\⊴ " NORMAL SUBGROUP OF OR EQUAL TO ")
	     (?\⊵ " CONTAINS AS NORMAL SUBGROUP OR EQUAL TO ")
	     (?\⊸ " MULTIMAP ")
	     (?\⊺ " INTERCALATE ")
	     (?\⊻ " XOR ")
	     (?\⊼ " NAND ")
	     (?\⋅ " · ")
	     (?\⋖ "<.")
	     (?\⋗ ">.")
	     (?\⋘ "<<<")
	     (?\⋙ ">>>")
	     (?\⋮ ":3")
	     (?\⋯ ".3")
	     (?\⌂ "Eh")
	     (?\⌇ "~~")
	     (?\⌈ "<7")
	     (?\⌉ ">7")
	     (?\⌊ "7<")
	     (?\⌋ "7>")
	     (?\⌐ "NI")
	     (?\⌒ "(A")
	     (?\⌕ "TR")
	     (?\⌘ "88")
	     (?\⌠ "Iu")
	     (?\⌡ "Il")
	     (?\⌢ ":(")
	     (?\⌣ ":)")
	     (?\⌤ "|^|")
	     (?\⌧ "[X]")
	     (?\〈 "</")
	     (?\〉 "/>")
	     (?\␣ "Vs")
	     (?\⑀ "1h")
	     (?\⑁ "3h")
	     (?\⑂ "2h")
	     (?\⑃ "4h")
	     (?\⑆ "1j")
	     (?\⑇ "2j")
	     (?\⑈ "3j")
	     (?\⑉ "4j")
	     (?\① "1-o")
	     (?\② "2-o")
	     (?\③ "3-o")
	     (?\④ "4-o")
	     (?\⑤ "5-o")
	     (?\⑥ "6-o")
	     (?\⑦ "7-o")
	     (?\⑧ "8-o")
	     (?\⑨ "9-o")
	     (?\⑩ "10-o")
	     (?\⑪ "11-o")
	     (?\⑫ "12-o")
	     (?\⑬ "13-o")
	     (?\⑭ "14-o")
	     (?\⑮ "15-o")
	     (?\⑯ "16-o")
	     (?\⑰ "17-o")
	     (?\⑱ "18-o")
	     (?\⑲ "19-o")
	     (?\⑳ "20-o")
	     (?\⑴ "(1)")
	     (?\⑵ "(2)")
	     (?\⑶ "(3)")
	     (?\⑷ "(4)")
	     (?\⑸ "(5)")
	     (?\⑹ "(6)")
	     (?\⑺ "(7)")
	     (?\⑻ "(8)")
	     (?\⑼ "(9)")
	     (?\⑽ "(10)")
	     (?\⑾ "(11)")
	     (?\⑿ "(12)")
	     (?\⒀ "(13)")
	     (?\⒁ "(14)")
	     (?\⒂ "(15)")
	     (?\⒃ "(16)")
	     (?\⒄ "(17)")
	     (?\⒅ "(18)")
	     (?\⒆ "(19)")
	     (?\⒇ "(20)")
	     (?\⒈ "1.")
	     (?\⒉ "2.")
	     (?\⒊ "3.")
	     (?\⒋ "4.")
	     (?\⒌ "5.")
	     (?\⒍ "6.")
	     (?\⒎ "7.")
	     (?\⒏ "8.")
	     (?\⒐ "9.")
	     (?\⒑ "10.")
	     (?\⒒ "11.")
	     (?\⒓ "12.")
	     (?\⒔ "13.")
	     (?\⒕ "14.")
	     (?\⒖ "15.")
	     (?\⒗ "16.")
	     (?\⒘ "17.")
	     (?\⒙ "18.")
	     (?\⒚ "19.")
	     (?\⒛ "20.")
	     (?\⒜ "(a)")
	     (?\⒝ "(b)")
	     (?\⒞ "(c)")
	     (?\⒟ "(d)")
	     (?\⒠ "(e)")
	     (?\⒡ "(f)")
	     (?\⒢ "(g)")
	     (?\⒣ "(h)")
	     (?\⒤ "(i)")
	     (?\⒥ "(j)")
	     (?\⒦ "(k)")
	     (?\⒧ "(l)")
	     (?\⒨ "(m)")
	     (?\⒩ "(n)")
	     (?\⒪ "(o)")
	     (?\⒫ "(p)")
	     (?\⒬ "(q)")
	     (?\⒭ "(r)")
	     (?\⒮ "(s)")
	     (?\⒯ "(t)")
	     (?\⒰ "(u)")
	     (?\⒱ "(v)")
	     (?\⒲ "(w)")
	     (?\⒳ "(x)")
	     (?\⒴ "(y)")
	     (?\⒵ "(z)")
	     (?\Ⓐ "A-o")
	     (?\Ⓑ "B-o")
	     (?\Ⓒ "C-o")
	     (?\Ⓓ "D-o")
	     (?\Ⓔ "E-o")
	     (?\Ⓕ "F-o")
	     (?\Ⓖ "G-o")
	     (?\Ⓗ "H-o")
	     (?\Ⓘ "I-o")
	     (?\Ⓙ "J-o")
	     (?\Ⓚ "K-o")
	     (?\Ⓛ "L-o")
	     (?\Ⓜ "M-o")
	     (?\Ⓝ "N-o")
	     (?\Ⓞ "O-o")
	     (?\Ⓟ "P-o")
	     (?\Ⓠ "Q-o")
	     (?\Ⓡ "R-o")
	     (?\Ⓢ "S-o")
	     (?\Ⓣ "T-o")
	     (?\Ⓤ "U-o")
	     (?\Ⓥ "V-o")
	     (?\Ⓦ "W-o")
	     (?\Ⓧ "X-o")
	     (?\Ⓨ "Y-o")
	     (?\Ⓩ "Z-o")
	     (?\ⓐ "a-o")
	     (?\ⓑ "b-o")
	     (?\ⓒ "c-o")
	     (?\ⓓ "d-o")
	     (?\ⓔ "e-o")
	     (?\ⓕ "f-o")
	     (?\ⓖ "g-o")
	     (?\ⓗ "h-o")
	     (?\ⓘ "i-o")
	     (?\ⓙ "j-o")
	     (?\ⓚ "k-o")
	     (?\ⓛ "l-o")
	     (?\ⓜ "m-o")
	     (?\ⓝ "n-o")
	     (?\ⓞ "o-o")
	     (?\ⓟ "p-o")
	     (?\ⓠ "q-o")
	     (?\ⓡ "r-o")
	     (?\ⓢ "s-o")
	     (?\ⓣ "t-o")
	     (?\ⓤ "u-o")
	     (?\ⓥ "v-o")
	     (?\ⓦ "w-o")
	     (?\ⓧ "x-o")
	     (?\ⓨ "y-o")
	     (?\ⓩ "z-o")
	     (?\⓪ "0-o")
	     (?\─ "-")
	     (?\━ "=")
	     (?\│ "|")
	     (?\┃ "|")
	     (?\┄ "-")
	     (?\┅ "=")
	     (?\┆ "|")
	     (?\┇ "|")
	     (?\┈ "-")
	     (?\┉ "=")
	     (?\┊ "|")
	     (?\┋ "|")
	     (?\┌ "+")
	     (?\┍ "+")
	     (?\┎ "+")
	     (?\┏ "+")
	     (?\┐ "+")
	     (?\┑ "+")
	     (?\┒ "+")
	     (?\┓ "+")
	     (?\└ "+")
	     (?\┕ "+")
	     (?\┖ "+")
	     (?\┗ "+")
	     (?\┘ "+")
	     (?\┙ "+")
	     (?\┚ "+")
	     (?\┛ "+")
	     (?\├ "+")
	     (?\┝ "+")
	     (?\┞ "+")
	     (?\┟ "+")
	     (?\┠ "+")
	     (?\┡ "+")
	     (?\┢ "+")
	     (?\┣ "+")
	     (?\┤ "+")
	     (?\┥ "+")
	     (?\┦ "+")
	     (?\┧ "+")
	     (?\┨ "+")
	     (?\┩ "+")
	     (?\┪ "+")
	     (?\┫ "+")
	     (?\┬ "+")
	     (?\┭ "+")
	     (?\┮ "+")
	     (?\┯ "+")
	     (?\┰ "+")
	     (?\┱ "+")
	     (?\┲ "+")
	     (?\┳ "+")
	     (?\┴ "+")
	     (?\┵ "+")
	     (?\┶ "+")
	     (?\┷ "+")
	     (?\┸ "+")
	     (?\┹ "+")
	     (?\┺ "+")
	     (?\┻ "+")
	     (?\┼ "+")
	     (?\┽ "+")
	     (?\┾ "+")
	     (?\┿ "+")
	     (?\╀ "+")
	     (?\╁ "+")
	     (?\╂ "+")
	     (?\╃ "+")
	     (?\╄ "+")
	     (?\╅ "+")
	     (?\╆ "+")
	     (?\╇ "+")
	     (?\╈ "+")
	     (?\╉ "+")
	     (?\╊ "+")
	     (?\╋ "+")
	     (?\╌ "+")
	     (?\╍ "+")
	     (?\╎ "+")
	     (?\╏ "+")
	     (?\═ "+")
	     (?\║ "+")
	     (?\╒ "+")
	     (?\╓ "+")
	     (?\╔ "+")
	     (?\╕ "+")
	     (?\╖ "+")
	     (?\╗ "+")
	     (?\╘ "+")
	     (?\╙ "+")
	     (?\╚ "+")
	     (?\╛ "+")
	     (?\╜ "+")
	     (?\╝ "+")
	     (?\╞ "+")
	     (?\╟ "+")
	     (?\╠ "+")
	     (?\╡ "+")
	     (?\╢ "+")
	     (?\╣ "+")
	     (?\╤ "+")
	     (?\╥ "+")
	     (?\╦ "+")
	     (?\╧ "+")
	     (?\╨ "+")
	     (?\╩ "+")
	     (?\╪ "+")
	     (?\╫ "+")
	     (?\╬ "+")
	     (?\╱ "/")
	     (?\╲ "\\")
	     (?\▀ "TB")
	     (?\▄ "LB")
	     (?\█ "FB")
	     (?\▌ "lB")
	     (?\▐ "RB")
	     (?\░ ".S")
	     (?\▒ ":S")
	     (?\▓ "?S")
	     (?\■ "fS")
	     (?\□ "OS")
	     (?\▢ "RO")
	     (?\▣ "Rr")
	     (?\▤ "RF")
	     (?\▥ "RY")
	     (?\▦ "RH")
	     (?\▧ "RZ")
	     (?\▨ "RK")
	     (?\▩ "RX")
	     (?\▪ "sB")
	     (?\▬ "SR")
	     (?\▭ "Or")
	     (?\▲ "^")
	     (?\△ "uT")
	     (?\▶ "|>")
	     (?\▷ "Tr")
	     (?\► "|>")
	     (?\▼ "v")
	     (?\▽ "dT")
	     (?\◀ "<|")
	     (?\◁ "Tl")
	     (?\◄ "<|")
	     (?\◆ "Db")
	     (?\◇ "Dw")
	     (?\◊ "LZ")
	     (?\○ "0m")
	     (?\◎ "0o")
	     (?\● "0M")
	     (?\◐ "0L")
	     (?\◑ "0R")
	     (?\◘ "Sn")
	     (?\◙ "Ic")
	     (?\◢ "Fd")
	     (?\◣ "Bd")
	     (?\◯ "Ci")
	     (?\★ "*2")
	     (?\☆ "*1")
	     (?\☎ "TEL")
	     (?\☏ "tel")
	     (?\☜ "<--")
	     (?\☞ "-->")
	     (?\☡ "CAUTION ")
	     (?\☧ "XP")
	     (?\☹ ":-(")
	     (?\☺ ":-)")
	     (?\☻ "(-:")
	     (?\☼ "SU")
	     (?\♀ "f.")
	     (?\♂ "m.")
	     (?\♠ "cS")
	     (?\♡ "cH")
	     (?\♢ "cD")
	     (?\♣ "cC")
	     (?\♤ "cS-")
	     (?\♥ "cH-")
	     (?\♦ "cD-")
	     (?\♧ "cC-")
	     (?\♩ "Md")
	     (?\♪ "M8")
	     (?\♫ "M2")
	     (?\♬ "M16")
	     (?\♭ "b")
	     (?\♮ "Mx")
	     (?\♯ "#")
	     (?\✓ "X")
	     (?\✗ "X")
	     (?\✠ "-X")
	     (?\　 " ")
	     (?\、 ",_")
	     (?\。 "._")
	     (?\〃 "+\"")
	     (?\〄 "JIS")
	     (?\々 "*_")
	     (?\〆 ";_")
	     (?\〇 "0_")
	     (?\《 "<+")
	     (?\》 ">+")
	     (?\「 "<'")
	     (?\」 ">'")
	     (?\『 "<\"")
	     (?\』 ">\"")
	     (?\【 "(\"")
	     (?\】 ")\"")
	     (?\〒 "=T")
	     (?\〓 "=_")
	     (?\〔 "('")
	     (?\〕 ")'")
	     (?\〖 "(I")
	     (?\〗 ")I")
	     (?\〚 "[[")
	     (?\〛 "]]")
	     (?\〜 "-?")
	     (?\〠 "=T:)")
	     (?\〿 " ")
	     (?\ぁ "A5")
	     (?\あ "a5")
	     (?\ぃ "I5")
	     (?\い "i5")
	     (?\ぅ "U5")
	     (?\う "u5")
	     (?\ぇ "E5")
	     (?\え "e5")
	     (?\ぉ "O5")
	     (?\お "o5")
	     (?\か "ka")
	     (?\が "ga")
	     (?\き "ki")
	     (?\ぎ "gi")
	     (?\く "ku")
	     (?\ぐ "gu")
	     (?\け "ke")
	     (?\げ "ge")
	     (?\こ "ko")
	     (?\ご "go")
	     (?\さ "sa")
	     (?\ざ "za")
	     (?\し "si")
	     (?\じ "zi")
	     (?\す "su")
	     (?\ず "zu")
	     (?\せ "se")
	     (?\ぜ "ze")
	     (?\そ "so")
	     (?\ぞ "zo")
	     (?\た "ta")
	     (?\だ "da")
	     (?\ち "ti")
	     (?\ぢ "di")
	     (?\っ "tU")
	     (?\つ "tu")
	     (?\づ "du")
	     (?\て "te")
	     (?\で "de")
	     (?\と "to")
	     (?\ど "do")
	     (?\な "na")
	     (?\に "ni")
	     (?\ぬ "nu")
	     (?\ね "ne")
	     (?\の "no")
	     (?\は "ha")
	     (?\ば "ba")
	     (?\ぱ "pa")
	     (?\ひ "hi")
	     (?\び "bi")
	     (?\ぴ "pi")
	     (?\ふ "hu")
	     (?\ぶ "bu")
	     (?\ぷ "pu")
	     (?\へ "he")
	     (?\べ "be")
	     (?\ぺ "pe")
	     (?\ほ "ho")
	     (?\ぼ "bo")
	     (?\ぽ "po")
	     (?\ま "ma")
	     (?\み "mi")
	     (?\む "mu")
	     (?\め "me")
	     (?\も "mo")
	     (?\ゃ "yA")
	     (?\や "ya")
	     (?\ゅ "yU")
	     (?\ゆ "yu")
	     (?\ょ "yO")
	     (?\よ "yo")
	     (?\ら "ra")
	     (?\り "ri")
	     (?\る "ru")
	     (?\れ "re")
	     (?\ろ "ro")
	     (?\ゎ "wA")
	     (?\わ "wa")
	     (?\ゐ "wi")
	     (?\ゑ "we")
	     (?\を "wo")
	     (?\ん "n5")
	     (?\ゔ "vu")
	     (?\゛ "\"5")
	     (?\゜ "05")
	     (?\ゝ "*5")
	     (?\ゞ "+5")
	     (?\ァ "a6")
	     (?\ア "A6")
	     (?\ィ "i6")
	     (?\イ "I6")
	     (?\ゥ "u6")
	     (?\ウ "U6")
	     (?\ェ "e6")
	     (?\エ "E6")
	     (?\ォ "o6")
	     (?\オ "O6")
	     (?\カ "Ka")
	     (?\ガ "Ga")
	     (?\キ "Ki")
	     (?\ギ "Gi")
	     (?\ク "Ku")
	     (?\グ "Gu")
	     (?\ケ "Ke")
	     (?\ゲ "Ge")
	     (?\コ "Ko")
	     (?\ゴ "Go")
	     (?\サ "Sa")
	     (?\ザ "Za")
	     (?\シ "Si")
	     (?\ジ "Zi")
	     (?\ス "Su")
	     (?\ズ "Zu")
	     (?\セ "Se")
	     (?\ゼ "Ze")
	     (?\ソ "So")
	     (?\ゾ "Zo")
	     (?\タ "Ta")
	     (?\ダ "Da")
	     (?\チ "Ti")
	     (?\ヂ "Di")
	     (?\ッ "TU")
	     (?\ツ "Tu")
	     (?\ヅ "Du")
	     (?\テ "Te")
	     (?\デ "De")
	     (?\ト "To")
	     (?\ド "Do")
	     (?\ナ "Na")
	     (?\ニ "Ni")
	     (?\ヌ "Nu")
	     (?\ネ "Ne")
	     (?\ノ "No")
	     (?\ハ "Ha")
	     (?\バ "Ba")
	     (?\パ "Pa")
	     (?\ヒ "Hi")
	     (?\ビ "Bi")
	     (?\ピ "Pi")
	     (?\フ "Hu")
	     (?\ブ "Bu")
	     (?\プ "Pu")
	     (?\ヘ "He")
	     (?\ベ "Be")
	     (?\ペ "Pe")
	     (?\ホ "Ho")
	     (?\ボ "Bo")
	     (?\ポ "Po")
	     (?\マ "Ma")
	     (?\ミ "Mi")
	     (?\ム "Mu")
	     (?\メ "Me")
	     (?\モ "Mo")
	     (?\ャ "YA")
	     (?\ヤ "Ya")
	     (?\ュ "YU")
	     (?\ユ "Yu")
	     (?\ョ "YO")
	     (?\ヨ "Yo")
	     (?\ラ "Ra")
	     (?\リ "Ri")
	     (?\ル "Ru")
	     (?\レ "Re")
	     (?\ロ "Ro")
	     (?\ヮ "WA")
	     (?\ワ "Wa")
	     (?\ヰ "Wi")
	     (?\ヱ "We")
	     (?\ヲ "Wo")
	     (?\ン "N6")
	     (?\ヴ "Vu")
	     (?\ヵ "KA")
	     (?\ヶ "KE")
	     (?\ヷ "Va")
	     (?\ヸ "Vi")
	     (?\ヹ "Ve")
	     (?\ヺ "Vo")
	     (?\・ ".6")
	     (?\ー "-6")
	     (?\ヽ "*6")
	     (?\ヾ "+6")
	     (?\ㄅ "b4")
	     (?\ㄆ "p4")
	     (?\ㄇ "m4")
	     (?\ㄈ "f4")
	     (?\ㄉ "d4")
	     (?\ㄊ "t4")
	     (?\ㄋ "n4")
	     (?\ㄌ "l4")
	     (?\ㄍ "g4")
	     (?\ㄎ "k4")
	     (?\ㄏ "h4")
	     (?\ㄐ "j4")
	     (?\ㄑ "q4")
	     (?\ㄒ "x4")
	     (?\ㄓ "zh")
	     (?\ㄔ "ch")
	     (?\ㄕ "sh")
	     (?\ㄖ "r4")
	     (?\ㄗ "z4")
	     (?\ㄘ "c4")
	     (?\ㄙ "s4")
	     (?\ㄚ "a4")
	     (?\ㄛ "o4")
	     (?\ㄜ "e4")
	     (?\ㄝ "eh4")
	     (?\ㄞ "ai")
	     (?\ㄟ "ei")
	     (?\ㄠ "au")
	     (?\ㄡ "ou")
	     (?\ㄢ "an")
	     (?\ㄣ "en")
	     (?\ㄤ "aN")
	     (?\ㄥ "eN")
	     (?\ㄦ "er")
	     (?\ㄧ "i4")
	     (?\ㄨ "u4")
	     (?\ㄩ "iu")
	     (?\ㄪ "v4")
	     (?\ㄫ "nG")
	     (?\ㄬ "gn")
	     (?\㈜ "(JU)")
	     (?\㈠ "1c")
	     (?\㈡ "2c")
	     (?\㈢ "3c")
	     (?\㈣ "4c")
	     (?\㈤ "5c")
	     (?\㈥ "6c")
	     (?\㈦ "7c")
	     (?\㈧ "8c")
	     (?\㈨ "9c")
	     (?\㈩ "10c")
	     (?\㉿ "KSC")
	     (?\㏂ "am")
	     (?\㏘ "pm")
	     (?\ﬀ "ff")
	     (?\ﬁ "fi")
	     (?\ﬂ "fl")
	     (?\ﬃ "ffi")
	     (?\ﬄ "ffl")
	     (?\ﬅ "St")
	     (?\ﬆ "st")
	     (?\ﹽ "3+;")
	     (?\ﺂ "aM.")
	     (?\ﺄ "aH.")
	     (?\ﺈ "ah.")
	     (?\ﺍ "a+-")
	     (?\ﺎ "a+.")
	     (?\ﺏ "b+-")
	     (?\ﺐ "b+.")
	     (?\ﺑ "b+,")
	     (?\ﺒ "b+;")
	     (?\ﺓ "tm-")
	     (?\ﺔ "tm.")
	     (?\ﺕ "t+-")
	     (?\ﺖ "t+.")
	     (?\ﺗ "t+,")
	     (?\ﺘ "t+;")
	     (?\ﺙ "tk-")
	     (?\ﺚ "tk.")
	     (?\ﺛ "tk,")
	     (?\ﺜ "tk;")
	     (?\ﺝ "g+-")
	     (?\ﺞ "g+.")
	     (?\ﺟ "g+,")
	     (?\ﺠ "g+;")
	     (?\ﺡ "hk-")
	     (?\ﺢ "hk.")
	     (?\ﺣ "hk,")
	     (?\ﺤ "hk;")
	     (?\ﺥ "x+-")
	     (?\ﺦ "x+.")
	     (?\ﺧ "x+,")
	     (?\ﺨ "x+;")
	     (?\ﺩ "d+-")
	     (?\ﺪ "d+.")
	     (?\ﺫ "dk-")
	     (?\ﺬ "dk.")
	     (?\ﺭ "r+-")
	     (?\ﺮ "r+.")
	     (?\ﺯ "z+-")
	     (?\ﺰ "z+.")
	     (?\ﺱ "s+-")
	     (?\ﺲ "s+.")
	     (?\ﺳ "s+,")
	     (?\ﺴ "s+;")
	     (?\ﺵ "sn-")
	     (?\ﺶ "sn.")
	     (?\ﺷ "sn,")
	     (?\ﺸ "sn;")
	     (?\ﺹ "c+-")
	     (?\ﺺ "c+.")
	     (?\ﺻ "c+,")
	     (?\ﺼ "c+;")
	     (?\ﺽ "dd-")
	     (?\ﺾ "dd.")
	     (?\ﺿ "dd,")
	     (?\ﻀ "dd;")
	     (?\ﻁ "tj-")
	     (?\ﻂ "tj.")
	     (?\ﻃ "tj,")
	     (?\ﻄ "tj;")
	     (?\ﻅ "zH-")
	     (?\ﻆ "zH.")
	     (?\ﻇ "zH,")
	     (?\ﻈ "zH;")
	     (?\ﻉ "e+-")
	     (?\ﻊ "e+.")
	     (?\ﻋ "e+,")
	     (?\ﻌ "e+;")
	     (?\ﻍ "i+-")
	     (?\ﻎ "i+.")
	     (?\ﻏ "i+,")
	     (?\ﻐ "i+;")
	     (?\ﻑ "f+-")
	     (?\ﻒ "f+.")
	     (?\ﻓ "f+,")
	     (?\ﻔ "f+;")
	     (?\ﻕ "q+-")
	     (?\ﻖ "q+.")
	     (?\ﻗ "q+,")
	     (?\ﻘ "q+;")
	     (?\ﻙ "k+-")
	     (?\ﻚ "k+.")
	     (?\ﻛ "k+,")
	     (?\ﻜ "k+;")
	     (?\ﻝ "l+-")
	     (?\ﻞ "l+.")
	     (?\ﻟ "l+,")
	     (?\ﻠ "l+;")
	     (?\ﻡ "m+-")
	     (?\ﻢ "m+.")
	     (?\ﻣ "m+,")
	     (?\ﻤ "m+;")
	     (?\ﻥ "n+-")
	     (?\ﻦ "n+.")
	     (?\ﻧ "n+,")
	     (?\ﻨ "n+;")
	     (?\ﻩ "h+-")
	     (?\ﻪ "h+.")
	     (?\ﻫ "h+,")
	     (?\ﻬ "h+;")
	     (?\ﻭ "w+-")
	     (?\ﻮ "w+.")
	     (?\ﻯ "j+-")
	     (?\ﻰ "j+.")
	     (?\ﻱ "y+-")
	     (?\ﻲ "y+.")
	     (?\ﻳ "y+,")
	     (?\ﻴ "y+;")
	     (?\ﻵ "lM-")
	     (?\ﻶ "lM.")
	     (?\ﻷ "lH-")
	     (?\ﻸ "lH.")
	     (?\ﻹ "lh-")
	     (?\ﻺ "lh.")
	     (?\ﻻ "la-")
	     (?\ﻼ "la.")
	     (?\！ "!")
	     (?\＂ "\"")
	     (?\＃ "#")
	     (?\＄ "$")
	     (?\％ "%")
	     (?\＆ "&")
	     (?\＇ "'")
	     (?\（ "(")
	     (?\） ")")
	     (?\＊ "*")
	     (?\＋ "+")
	     (?\， ",")
	     (?\－ "-")
	     (?\． ".")
	     (?\／ "/")
	     (?\０ "0")
	     (?\１ "1")
	     (?\２ "2")
	     (?\３ "3")
	     (?\４ "4")
	     (?\５ "5")
	     (?\６ "6")
	     (?\７ "7")
	     (?\８ "8")
	     (?\９ "9")
	     (?\： ":")
	     (?\； ";")
	     (?\＜ "<")
	     (?\＝ "=")
	     (?\＞ ">")
	     (?\？ "?")
	     (?\＠ "@")
	     (?\Ａ "A")
	     (?\Ｂ "B")
	     (?\Ｃ "C")
	     (?\Ｄ "D")
	     (?\Ｅ "E")
	     (?\Ｆ "F")
	     (?\Ｇ "G")
	     (?\Ｈ "H")
	     (?\Ｉ "I")
	     (?\Ｊ "J")
	     (?\Ｋ "K")
	     (?\Ｌ "L")
	     (?\Ｍ "M")
	     (?\Ｎ "N")
	     (?\Ｏ "O")
	     (?\Ｐ "P")
	     (?\Ｑ "Q")
	     (?\Ｒ "R")
	     (?\Ｓ "S")
	     (?\Ｔ "T")
	     (?\Ｕ "U")
	     (?\Ｖ "V")
	     (?\Ｗ "W")
	     (?\Ｘ "X")
	     (?\Ｙ "Y")
	     (?\Ｚ "Z")
	     (?\［ "[")
	     (?\＼ "\\")
	     (?\］ "]")
	     (?\＾ "^")
	     (?\＿ "_")
	     (?\｀ "`")
	     (?\ａ "a")
	     (?\ｂ "b")
	     (?\ｃ "c")
	     (?\ｄ "d")
	     (?\ｅ "e")
	     (?\ｆ "f")
	     (?\ｇ "g")
	     (?\ｈ "h")
	     (?\ｉ "i")
	     (?\ｊ "j")
	     (?\ｋ "k")
	     (?\ｌ "l")
	     (?\ｍ "m")
	     (?\ｎ "n")
	     (?\ｏ "o")
	     (?\ｐ "p")
	     (?\ｑ "q")
	     (?\ｒ "r")
	     (?\ｓ "s")
	     (?\ｔ "t")
	     (?\ｕ "u")
	     (?\ｖ "v")
	     (?\ｗ "w")
	     (?\ｘ "x")
	     (?\ｙ "y")
	     (?\ｚ "z")
	     (?\｛ "{")
	     (?\｜ "|")
	     (?\｝ "}")
	     (?\～ "~")
	     (?\｡ ".")
	     (?\｢ "\"")
	     (?\｣ "\"")
	     (?\､ ",")
	     ;; Not from Lynx
	     (?﻿ "")
	     (?� "?")))))
    (aset standard-display-table
	  (make-char 'mule-unicode-0100-24ff) nil)
    (aset standard-display-table
	  (make-char 'mule-unicode-2500-33ff) nil)
    (aset standard-display-table
	  (make-char 'mule-unicode-e000-ffff) nil))
  (setq latin1-display-ucs-per-lynx (not latin1-display-ucs-per-lynx))
  (redraw-display))

(provide 'latin1-disp)

;;; latin1-disp.el ends here
